/*
 * backup.c
 *
 * Copyright 2004 rmenhal
 *
 * Licensed under GNU General Public License version 2. See the file COPYING
 * for details.
 */

#include "printf.h"
#include "xboxkrnl.h"
#include "strh.h"
#include "cfgparser.h"
#include "ntfile.h"
#include "xbe-header.h"
#include "virtualcdrom.h"


unsigned char DebugFlag;

extern char attach_xbe[], attach_xbe_end[];

static void mainthread(PVOID parm1, PVOID parm2);




/* Error sector report file name
 */

#define ERROR_SECTOR_FILE_NAME		"errorsectors.bin"


/* Require some extra free space in the free space check in addition
 * to the ISO data.
 */

#define EXTRA_SPACE_REQ			(2*1024*1024)


/* Slice the ISO into parts this big. Keep it divisible by the sector size 2048.
 */

#define DEFAULT_ISO_SLICE_SIZE		0xffe00000


/* Do the actual data copy in sizes of this big. In case the disk is damaged
 * or dirty, reading may fail so several levels are provided. This will also
 * provide a retry functionality. Keep the last level the sector size 2048
 * and other levels divisible by 2048.
 */

static unsigned long copy_level_size[] = { 4*1024*1024, 131072, 2048 };

#define MAX_COPY_LEVEL	(sizeof(copy_level_size) / sizeof(unsigned long) - 1)



void die() {
    /* red-orange-green blink */
    HalWriteSMBusValue(0x20, 0x08, 0, 0x63);
    HalWriteSMBusValue(0x20, 0x07, 0, 0x01);
    while (1);
}



/* Main thread setup code from Phoenix Bios Loader */

void boot(void)
{
    HANDLE hThread = 0;
    ULONG Id = 0;
    LARGE_INTEGER Timeout;
    ULONG Status = 0;
    
    Timeout.QuadPart = 0;
    
    
    if(!NT_SUCCESS(PsCreateSystemThreadEx(&hThread,
					  0,
					  65536,
					  0,
					  &Id,
					  NULL,
					  NULL,
					  FALSE,
					  FALSE, 
					  (PVOID)&mainthread))) {
	
	HalReturnToFirmware(2);
    }
    while(1) {
	Status = NtWaitForSingleObjectEx(hThread, 1 /* UserMode */ ,
					 FALSE, &Timeout);
	if (Status == STATUS_SUCCESS) {
	    NtClose(hThread);
	    HalReturnToFirmware(2);
	}
    }
}




#define DIRNAME_TITLE_CHARS	20

struct rip_info {
    unsigned long title_id;
    short title_name[40];
    char title_dirname[44];
    LARGE_INTEGER title_size;
    HANDLE dev_handle;
    HANDLE target_dir_handle;
    HANDLE errsec_handle;
    char errsec_file_opened : 1;
};



static void report_error_sector(struct rip_info *ri, PLARGE_INTEGER ofs)
{
    LARGE_INTEGER sector_num;
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    NTSTATUS status;

    if (!ri->errsec_file_opened) {
	
	RtlInitAnsiString(&ansi_str, ERROR_SECTOR_FILE_NAME);
	
	obj_attr.RootDirectory = ri->target_dir_handle;
	obj_attr.ObjectName = &ansi_str;
	obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
	status = NtCreateFile(&ri->errsec_handle, GENERIC_WRITE | SYNCHRONIZE,
			      &obj_attr, &io_status, NULL, 0,
			      FILE_SHARE_READ | FILE_SHARE_WRITE |
			      FILE_SHARE_DELETE,
			      FILE_SUPERSEDE,
			      FILE_SYNCHRONOUS_IO_NONALERT |
			      FILE_NON_DIRECTORY_FILE);

	if (!NT_SUCCESS(status))
	    /* shrug */
	    return;
	
	ri->errsec_file_opened = 1;
    }

    sector_num.QuadPart = ofs->QuadPart >> 11;
    
    NtWriteFile(ri->errsec_handle, NULL, NULL, NULL, &io_status,
		&sector_num, sizeof(LARGE_INTEGER), NULL);
}


static void close_error_sector_file(struct rip_info *ri)
{
    if (!ri->errsec_file_opened)
	return;
    
    ri->errsec_file_opened = 0;
    NtClose(ri->errsec_handle);
}



static void mangle_name_for_fatx(char *s)
{
    int i;
    static char allowed_marks[] = { '!', '#', '$', '%', '&',
				    '\'', '(', ')', '-', '.',
				    '@', '[', ']', '^', '_',
				    '`', '{', '}', '~' };

    for (; *s; s++) {
	if ((*s >= 'a' && *s <= 'z') || (*s >= 'A' && *s <= 'Z') ||
	    (*s >= '0' && *s <= '9'))
	    continue;
	
	for (i = 0; i < sizeof(allowed_marks); i++)
	    if (*s == allowed_marks[i])
		break;
	if (i < sizeof(allowed_marks))
	    continue;

	*s = '_';
    }
}


static NTSTATUS open_query_dvd(struct rip_info *ri, char *name)
{
    NTSTATUS status;
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    FILE_FS_SIZE_INFORMATION szinfo;
    DISK_GEOMETRY geom;
    
    RtlInitAnsiString(&ansi_str, name);
    
    obj_attr.RootDirectory = NULL;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
    status = NtOpenFile(&ri->dev_handle, GENERIC_READ | SYNCHRONIZE,
			&obj_attr, &io_status,
			FILE_SHARE_READ,
			FILE_NO_INTERMEDIATE_BUFFERING |
			FILE_SYNCHRONOUS_IO_NONALERT);
    
    if (!NT_SUCCESS(status))
	return status;
    
    status = NtQueryVolumeInformationFile(ri->dev_handle, &io_status,
					  &szinfo, sizeof(szinfo), 3);

    if (NT_SUCCESS(status)) {

	ri->title_size.QuadPart = szinfo.TotalAllocationUnits.QuadPart
	    *szinfo.SectorsPerAllocationUnit*szinfo.BytesPerSector;

    } else {

	dprintf("Volume query failed in open_dvd_query, status: %08x\n", status);
	
	status = NtDeviceIoControlFile(ri->dev_handle, NULL, NULL, NULL, &io_status,
				       IOCTL_CDROM_GET_DRIVE_GEOMETRY,
				       NULL, 0, &geom, sizeof(DISK_GEOMETRY));

	if (!NT_SUCCESS(status)) {
	    NtClose(ri->dev_handle);
	    return status;
	}

	ri->title_size.QuadPart = geom.Cylinders.QuadPart
	    *geom.TracksPerCylinder*geom.SectorsPerTrack*geom.BytesPerSector;
    }

    ri->errsec_file_opened = 0;

    return STATUS_SUCCESS;
}


static void close_dvd(struct rip_info *ri)
{
    close_error_sector_file(ri);
    NtClose(ri->dev_handle);
}


static NTSTATUS get_variable_title_info(struct rip_info *ri, char *name)
{
    LARGE_INTEGER time;
    TIME_FIELDS fields;
    ANSI_STRING ansi_str;
    UNICODE_STRING uc_str;

    KeQuerySystemTime(&time);
    if (!RtlTimeToTimeFields(&time, &fields)) {
	fields.Year = 2004;
	fields.Month = 12;
	fields.Day = 6;
	fields.Hour = 12;
	fields.Minute = 0;
	fields.Second = 0;
	fields.Milliseconds = 0;
	fields.Weekday = 1;
    }

    sprintf(ri->title_dirname, "Backup-%04d-%02d-%02d-%02d%02d%02d-UTC",
	    fields.Year, fields.Month, fields.Day,
	    fields.Hour, fields.Minute, fields.Second);
    
    ri->title_id = 0;
    
    uc_str.Length = 0;
    uc_str.MaximumLength = sizeof(ri->title_name);
    uc_str.Buffer = ri->title_name;

    RtlInitAnsiString(&ansi_str, ri->title_dirname);

    RtlAnsiStringToUnicodeString(&uc_str, &ansi_str, FALSE);

    return STATUS_SUCCESS;
}


static NTSTATUS get_title_info(struct rip_info *ri, char *name)
{
    HANDLE h;
    NTSTATUS status;
    ANSI_STRING ansi_str;
    UNICODE_STRING uc_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    char buf[0x200];
    PXBE_HEADER hdr;
    PXBE_CERTIFICATE cert;
    LARGE_INTEGER ofs;

    status = open_query_dvd(ri, name);

    if (!NT_SUCCESS(status))
	return status;

    strh_dnzcpy(buf, name, sizeof(buf));
    strh_dnzcat(buf, "\\default.xbe", sizeof(buf));
    
    RtlInitAnsiString(&ansi_str, buf);
    
    obj_attr.RootDirectory = NULL;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;

    status = NtOpenFile(&h, GENERIC_READ | SYNCHRONIZE,
			&obj_attr, &io_status,
			FILE_SHARE_READ,
			FILE_NON_DIRECTORY_FILE |
			FILE_SYNCHRONOUS_IO_NONALERT);

    if (!NT_SUCCESS(status)) {
	dprintf("Unable to open default.xbe, status: %08x\n", status);
	return get_variable_title_info(ri, name);
    }
    
    status = NtReadFile(h, NULL, NULL, NULL, &io_status,
			buf, sizeof(XBE_HEADER), NULL);

    if (!NT_SUCCESS(status) || io_status.Information != sizeof(XBE_HEADER))
	goto read_failed;

    hdr = (PXBE_HEADER)buf;

    if (*(int *)hdr->Magic != 0x48454258) {
	NtClose(h);
	close_dvd(ri);
	return STATUS_DATA_ERROR;
    }

    ofs.QuadPart = (int)hdr->Certificate - (int)hdr->BaseAddress;
    
    status = NtReadFile(h, NULL, NULL, NULL, &io_status,
			buf, sizeof(XBE_CERTIFICATE), &ofs);

    if (!NT_SUCCESS(status) || io_status.Information != sizeof(XBE_CERTIFICATE))
	goto read_failed;

    NtClose(h);

    cert = (PXBE_CERTIFICATE)buf;
    ri->title_id = cert->TitleId;
    memcpy(ri->title_name, cert->TitleName, sizeof(ri->title_name));

    ansi_str.Length = 0;
    ansi_str.MaximumLength = sizeof(ri->title_dirname);
    ansi_str.Buffer = ri->title_dirname;

    uc_str.Length = sizeof(ri->title_name);
    uc_str.MaximumLength = uc_str.Length;
    uc_str.Buffer = ri->title_name;

    RtlUnicodeStringToAnsiString(&ansi_str, &uc_str, FALSE);

    ri->title_dirname[DIRNAME_TITLE_CHARS] = '\0';
    mangle_name_for_fatx(ri->title_dirname);

    sprintf(buf, "-%08x", (int)ri->title_id);
    strh_dnzcat(ri->title_dirname, buf, sizeof(ri->title_dirname));
    
    return STATUS_SUCCESS;

 read_failed:
    dprintf("Reading default.xbe failed, status: %08x, Information: %08x\n",
	    status, io_status.Information);
    NtClose(h);
    close_dvd(ri);
    return status;
}


static NTSTATUS get_free_space(PLARGE_INTEGER free_space, char *name)
{
    NTSTATUS status;
    HANDLE h;
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    FILE_FS_SIZE_INFORMATION szinfo;

    RtlInitAnsiString(&ansi_str, name);

    obj_attr.RootDirectory = NULL;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
    status = NtOpenFile(&h, GENERIC_READ | SYNCHRONIZE,
			&obj_attr, &io_status,
			FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
			FILE_SYNCHRONOUS_IO_NONALERT);

    if (!NT_SUCCESS(status)) {
	dprintf("Open failed in get_free_space, status: %08x\n", status);
	return status;
    }

    status = NtQueryVolumeInformationFile(h, &io_status, &szinfo, sizeof(szinfo), 3);

    NtClose(h);

    if (!NT_SUCCESS(status)) {
	dprintf("Query failed in get_free_space, status: %08x\n", status);
	return status;
    }

    free_space->QuadPart = szinfo.AvailableAllocationUnits.QuadPart
	*szinfo.SectorsPerAllocationUnit*szinfo.BytesPerSector;

    return STATUS_SUCCESS;
}


static void boot_dashboard(void)
{
    PVOID ld = *LaunchDataPage;

    if (ld == NULL)
	*LaunchDataPage = ld = MmAllocateContiguousMemory(4096);

    if (ld) {
	MmPersistContiguousMemory(ld, 4096, TRUE);
	memset(ld, 0, 4096);
	*(int *)ld = -1;
    }

    HalReturnToFirmware(2);
    die();
}


static NTSTATUS create_target_directory(struct rip_info *ri, char *backup_dir)
{
    char path_name[MAX_PATHNAME];
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    NTSTATUS status;

    strh_dnzcpy(path_name, backup_dir, MAX_PATHNAME);
    strh_dnzcat(path_name, "\\", MAX_PATHNAME);
    strh_dnzcat(path_name, ri->title_dirname, MAX_PATHNAME);

    RtlInitAnsiString(&ansi_str, path_name);
    
    obj_attr.RootDirectory = NULL;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
    status = NtCreateFile(&ri->target_dir_handle, GENERIC_ALL | SYNCHRONIZE,
			  &obj_attr, &io_status, NULL, 0,
			  FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
			  FILE_CREATE,
			  FILE_SYNCHRONOUS_IO_NONALERT | FILE_DIRECTORY_FILE);

    return status;
}


static NTSTATUS create_slice_file(struct rip_info *ri, int num, PHANDLE handle)
{
    char file_name[43];
    char extension[sizeof(".partXY.iso")];
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    NTSTATUS status;

    sprintf(extension, ".part%d.iso", num);

    strh_dnzcpy(file_name, ri->title_dirname, sizeof(file_name));
    strh_dnzcat(file_name, extension, sizeof(file_name));

    RtlInitAnsiString(&ansi_str, file_name);
    
    obj_attr.RootDirectory = ri->target_dir_handle;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
    status = NtCreateFile(handle, GENERIC_WRITE | SYNCHRONIZE,
			  &obj_attr, &io_status, NULL, 0,
			  FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
			  FILE_CREATE,
			  FILE_SYNCHRONOUS_IO_NONALERT | FILE_NON_DIRECTORY_FILE);

    return status;
}



static NTSTATUS copy_slice(struct rip_info *ri, int level,
			   HANDLE handle, ULONG slice_size,
			   PLARGE_INTEGER ofs,
			   void *membuf)
{
    ULONG data_left = slice_size;
    ULONG chunk_size;
    IO_STATUS_BLOCK io_status;
    NTSTATUS status;
    LARGE_INTEGER our_ofs = *ofs;

    while (data_left > 0) {
	
	chunk_size = data_left < copy_level_size[level] ?
	    data_left : copy_level_size[level];
	
	status = NtReadFile(ri->dev_handle, NULL, NULL, NULL, &io_status,
			    membuf, chunk_size, &our_ofs);
	
	if (NT_SUCCESS(status) && io_status.Information == chunk_size) {
	    
	write_data:
	    status = NtWriteFile(handle, NULL, NULL, NULL, &io_status,
				 membuf, chunk_size, NULL);
	    
	    if (NT_SUCCESS(status) && io_status.Information != chunk_size)
		status = STATUS_DATA_ERROR;
	    
	} else {

	    /* Increase copy level and retry. If we're already at the highest level
	     * then write the sector number to errorsectors.bin file.
	     */
	    
	    if (level < MAX_COPY_LEVEL) {
		
		status = copy_slice(ri, level + 1,
				    handle, chunk_size,
				    &our_ofs, membuf);
		
	    } else {
		
		report_error_sector(ri, &our_ofs);
		
		/* Write zeroes to the output file */
		
		memset(membuf, 0, chunk_size);
		
		goto write_data;
	    }
	}
			     
	if (!NT_SUCCESS(status))
	    return status;

	our_ofs.QuadPart += chunk_size;
	data_left -= chunk_size;
    }

    return STATUS_SUCCESS;
}


static NTSTATUS create_attach_file(struct rip_info *ri)
{
    HANDLE h;
    ANSI_STRING ansi_str;
    OBJECT_ATTRIBUTES obj_attr;
    IO_STATUS_BLOCK io_status;
    NTSTATUS status;
    PXBE_HEADER hdr;
    PXBE_CERTIFICATE cert;

    RtlInitAnsiString(&ansi_str, "default.xbe");
    
    obj_attr.RootDirectory = ri->target_dir_handle;
    obj_attr.ObjectName = &ansi_str;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;
    
    status = NtCreateFile(&h, GENERIC_WRITE | SYNCHRONIZE,
			  &obj_attr, &io_status, NULL, 0,
			  FILE_SHARE_READ | FILE_SHARE_WRITE |
			  FILE_SHARE_DELETE,
			  FILE_CREATE,
			  FILE_SYNCHRONOUS_IO_NONALERT |
			  FILE_NON_DIRECTORY_FILE);
    
    if (!NT_SUCCESS(status))
	return status;

    hdr = (PXBE_HEADER)attach_xbe;
    cert = (PXBE_CERTIFICATE)(attach_xbe + ((char *)hdr->Certificate -
					    (char *)hdr->BaseAddress));
    
    cert->TitleId = ri->title_id;
    memcpy(cert->TitleName, ri->title_name, sizeof(cert->TitleName));
    
    status = NtWriteFile(h, NULL, NULL, NULL, &io_status,
			 attach_xbe, attach_xbe_end - attach_xbe, NULL);

    NtClose(h);

    return status;
}


static void detach_virtual_disc(void)
{
    NTSTATUS status;
    OBJECT_ATTRIBUTES obj_attr;
    HANDLE h;
    IO_STATUS_BLOCK io_status;
    ANSI_STRING dev_name;

    RtlInitAnsiString(&dev_name, "\\Device\\CdRom1");

    obj_attr.RootDirectory = NULL;
    obj_attr.ObjectName = &dev_name;
    obj_attr.Attributes = OBJ_CASE_INSENSITIVE;

    status = NtOpenFile(&h, GENERIC_READ | SYNCHRONIZE, &obj_attr, &io_status,
			FILE_SHARE_READ,
			FILE_NO_INTERMEDIATE_BUFFERING |
			FILE_SYNCHRONOUS_IO_NONALERT);

    if (!NT_SUCCESS(status)) {
	return;
    }

    NtDeviceIoControlFile(h, NULL, NULL, NULL, &io_status,
			  IOCTL_VIRTUAL_CDROM_DETACH,
			  NULL, 0, NULL, 0);

    NtClose(h);

    IoDismountVolumeByName(&dev_name);
}


static void mainthread(PVOID parm1, PVOID parm2)
{
    LARGE_INTEGER time_start;
    LARGE_INTEGER time_end;
    LARGE_INTEGER time_duration;
    NTSTATUS status;
    HANDLE h;
    void *membuf;
    unsigned long membuf_size;
    LARGE_INTEGER ofs;
    ULONG slice_size;
    LARGE_INTEGER data_left;
    LARGE_INTEGER free_space;
    struct rip_info ri = { 0, };
    CONFIGENTRY cfg_entry;
    char *backup_dir;
    unsigned long iso_slice_size;
    int i;


    status = cfgparser_get_config(&cfg_entry);
    
    if (!NT_SUCCESS(status))
	boot_dashboard();
    
    printf_init(cfg_entry.log, cfg_entry.path);

    detach_virtual_disc();	/* Detach possible virtual disc, nobody
				 * wants to rip that. :)
				 */
    
    dprintf("---- Xbox ISO Ripper v1.0 ----\n");

    KeQuerySystemTime(&time_start);


    status = get_title_info(&ri, "\\Device\\CdRom0");
    
    if (!NT_SUCCESS(status)) {
	dprintf("Can't get title information, aborting, status: %08x\n", status);
	boot_dashboard();
    }

    dprintf("Title ID: %08x\n"
	    "Title directory: %s\n"
	    "Total size: 0x%08x%08x bytes\n",
	    ri.title_id,
	    ri.title_dirname,
	    *((int *)&ri.title_size + 1), *(int *)&ri.title_size
	);


    for (i = 0, backup_dir = cfg_entry.primary_backup_dir;
	 i < 2;
	 i++, backup_dir = cfg_entry.secondary_backup_dir) {
	
	if (backup_dir[0] == '\0')
	    continue;
	
	status = get_free_space(&free_space, backup_dir);
	if (!NT_SUCCESS(status))
	    continue;

	if (free_space.QuadPart < ri.title_size.QuadPart + EXTRA_SPACE_REQ)
	    continue;

	status = create_target_directory(&ri, backup_dir);
	if (NT_SUCCESS(status))
	    break;
    }
    
    if (i == 2) {
	dprintf("No free space on either backup directory, aborting.\n");
	boot_dashboard();
    }

    dprintf("Backup directory: %s\n", backup_dir);


    membuf = NULL;
    membuf_size = copy_level_size[0];
    
    status = NtAllocateVirtualMemory(&membuf, 0, &membuf_size,
				     MEM_COMMIT | MEM_NOZERO, PAGE_READWRITE);

    if (!NT_SUCCESS(status)) {
	dprintf("virt mem alloc failed with status: 0x%08x\n", status);
	boot_dashboard();
    }


    /* Zero slice size low bits to make it a multiple of sector size 2048 bytes */

    iso_slice_size = cfg_entry.iso_slice_size & ~0x7ff;
    if (iso_slice_size == 0)
	iso_slice_size = DEFAULT_ISO_SLICE_SIZE;
    
    ofs.QuadPart = 0;
    data_left = ri.title_size;

    i = 1;

    while (data_left.QuadPart > 0) {
	slice_size = data_left.QuadPart < iso_slice_size ?
	    data_left.QuadPart : iso_slice_size;
	
	status = create_slice_file(&ri, i, &h);
	
	if (NT_SUCCESS(status))
	    status = copy_slice(&ri, 0, h, slice_size, &ofs, membuf);
	
	if (!NT_SUCCESS(status)) {
	    dprintf("Creating slice file failed, status: %08x\n", status);
	    break;
	}

	NtClose(h);

	ofs.QuadPart += slice_size;
	data_left.QuadPart -= slice_size;
	i++;
    }


    close_dvd(&ri);


    status = create_attach_file(&ri);

    if (!NT_SUCCESS(status)) {
	dprintf("Failed creating disc attacher default.xbe, status: %08x\n",
		status);
    }

    
    KeQuerySystemTime(&time_end);
    time_duration.QuadPart = time_end.QuadPart - time_start.QuadPart;

    {
	unsigned long seconds =
	    (unsigned long)(time_duration.QuadPart >> 7) / 78125;
	
	dprintf("Ripping time: %d minutes %d seconds\n",
		seconds / 60, seconds % 60);
    }


    boot_dashboard();
}
