/*
   Xbox XBE bootloader

    by Michael Steil & anonymous
    VESA Framebuffer code by Milosch Meriac

*/

#include "printf.c"
#include "consts.h"
#include "xboxkrnl.h"
#include "xbox.h"
#include "boot.h"
#include "BootString.h"
#include "BootParser.h"
#include "config.h"

int NewFramebuffer;
long KernelSize;
PHYSICAL_ADDRESS PhysKernelPos, PhysEscapeCodePos;
PVOID EscapeCodePos;
int xbox_ram = 64;

static int ReadFile(HANDLE Handle, PVOID Buffer, ULONG Size);
int WriteFile(HANDLE Handle, PVOID Buffer, ULONG Size);
int SaveFile(char *szFileName,PBYTE Buffer,ULONG Size);
void DismountFileSystems(void);
int RemapDrive(char *szDrive);
HANDLE OpenFile(HANDLE Root, LPCSTR Filename, LONG Length, ULONG Mode);
BOOL GetFileSize(HANDLE File, LONGLONG *Size);

NTSTATUS GetConfig(CONFIGENTRY *entry);
NTSTATUS GetConfigXBE(CONFIGENTRY *entry);



unsigned char * volatile MemoryMappedStartAddress = (unsigned char*)0xff000000;

void wait_ms(unsigned int ticks) {
        
	/*
	  	32 Bit range = 1200 sec ! => 20 min
		1. sec = 0x369E99
		1 ms =  3579,545
					
	*/
	
	DWORD COUNT_start;
	DWORD temp;
	DWORD COUNT_TO;
	DWORD HH;
	
	// Maximum Input range
	if (ticks>(1200*1000)) ticks = 1200*1000;
	
	COUNT_TO = (DWORD) ((float)(ticks*3579.545));
	COUNT_start = IoInputDword(0x8008);	

	while(1) {

		// Reads out the System timer
		HH = IoInputDword(0x8008);		
		temp = HH-COUNT_start;
		// We reached the counter
		if (temp>COUNT_TO) break;
	
	};
	

}


unsigned int XFlashGetID(void) {


	unsigned char dwVal_Manu;
	unsigned char dwVal_Dev;
	unsigned int dwFlashDeviceDescriptor;
	
	MemoryMappedStartAddress[0xAAA] = 0xAA;
	MemoryMappedStartAddress[0x555] = 0x55;
	MemoryMappedStartAddress[0xAAA] = 0x90;

	// Read Manufactor ID
        dwVal_Manu =  MemoryMappedStartAddress[0x0];
	// Read Device ID
	dwVal_Dev =  MemoryMappedStartAddress[0x2];

	dwFlashDeviceDescriptor = (dwVal_Manu << 8) + dwVal_Dev;
	
	MemoryMappedStartAddress[0x0] = 0xf0;
	dwVal_Manu = MemoryMappedStartAddress[0x0];

	return  dwFlashDeviceDescriptor;
}




void die() {
	while(1);
}


void boot() {


	CONFIGENTRY entry;
	int counter=0;

	framebuffer = (unsigned int*)(0xF0000000+*(unsigned int*)0xFD600800);
	memset(framebuffer,0,SCREEN_WIDTH*SCREEN_HEIGHT*4);

	memset(&entry,0,sizeof(CONFIGENTRY));
	cx = 0;
	cy = 0;

	
	// Unlock the Xenium Flash to get Write Access

	IoOutputByte(0xef,3);
	IoOutputByte(0xee,1);
	
	dprintf("\n");
	dprintf("\n");
	dprintf("\n");
	dprintf("\n");
	dprintf("\n");
	
	//dprintf("    Flash ID: %04X",XFlashGetID());
	
	if (XFlashGetID() == 0x01C4) {
		dprintf("       You have an Original Xenium ");
	} else {
		dprintf("       You have a Cloned Fake Xenium");
	}
	
	while(1) {
		counter++;
		counter = counter % 8;	
		IoOutputByte(0xee,counter);
		wait_ms(1000);
	}
	

}


int WriteFile(HANDLE Handle, PVOID Buffer, ULONG Size)
{
        IO_STATUS_BLOCK IoStatus;

        // Try to write the buffer
        if (!NT_SUCCESS(NtWriteFile(Handle, NULL, NULL, NULL, &IoStatus,
                Buffer, Size, NULL)))
                return 0;

        // Verify that the amount written is the correct size
        if (IoStatus.Information != Size)
                return 0;

        return 1;
}

int ReadFile(HANDLE Handle, PVOID Buffer, ULONG Size)
{
        IO_STATUS_BLOCK IoStatus;

        // Try to write the buffer
        if (!NT_SUCCESS(NtReadFile(Handle, NULL, NULL, NULL, &IoStatus,
                Buffer, Size, NULL)))
                return 0;

        // Verify that the amount read is the correct size
        if (IoStatus.Information != Size)
                return 0;

        return 1;
}

int SaveFile(char *szFileName,PBYTE Buffer,ULONG Size) {

	ANSI_STRING DestFileName;
        IO_STATUS_BLOCK IoStatus;
        OBJECT_ATTRIBUTES Attributes;
        HANDLE DestHandle = NULL;

	RtlInitAnsiString(&DestFileName,szFileName);
        Attributes.RootDirectory = NULL;
        Attributes.ObjectName = &DestFileName;
        Attributes.Attributes = OBJ_CASE_INSENSITIVE;

	if (!NT_SUCCESS(NtCreateFile(&DestHandle,
		GENERIC_WRITE  | GENERIC_READ | SYNCHRONIZE,
		&Attributes, &IoStatus,
		NULL, FILE_RANDOM_ACCESS,
		FILE_SHARE_READ, FILE_CREATE,
		FILE_SYNCHRONOUS_IO_NONALERT | FILE_NON_DIRECTORY_FILE))) {
			dprintf("Error saving File\n");
			return 0;
	}

	if(!WriteFile(DestHandle, Buffer, Size)) {
		dprintf("Error saving File\n");
		return 0;
	}

	NtClose(DestHandle);

	return 1;
}

// Dismount all file systems
void DismountFileSystems(void) {

        ANSI_STRING String;

        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition1");
        IoDismountVolumeByName(&String);
        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition2");
        IoDismountVolumeByName(&String);
        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition3");
        IoDismountVolumeByName(&String);
        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition4");
        IoDismountVolumeByName(&String);
        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition5");
        IoDismountVolumeByName(&String);
        RtlInitAnsiString(&String, "\\Device\\Harddisk0\\Partition6");
        IoDismountVolumeByName(&String);
}

NTSTATUS GetConfig(CONFIGENTRY *entry) {

	ULONGLONG FileSize;
	PBYTE Buffer;
	HANDLE hFile;

	memset(entry,0,sizeof(CONFIGENTRY));

        if (!(hFile = OpenFile(NULL, "\\??\\D:\\linuxboot.cfg", -1, FILE_NON_DIRECTORY_FILE)))
                return 1;

	if(!GetFileSize(hFile,&FileSize)) {
		dprintf("Error getting file size!\n");
		die();
	}

	Buffer = MmAllocateContiguousMemoryEx(FileSize,
			MIN_KERNEL, MAX_KERNEL, 0, PAGE_READWRITE);

	if (!Buffer) {
		dprintf("Error alloc memory for File\n");
		return 1;
	}

	if (!ReadFile(hFile, Buffer, FileSize)) {
		dprintf("Error loading file\n");
		return 1;
	}

	ParseConfig("\\??\\D:\\",Buffer,entry);

	NtClose(hFile);

	return STATUS_SUCCESS;
}

NTSTATUS GetConfigXBE(CONFIGENTRY *entry) {
	PBYTE Buffer;
        unsigned int TempConfigStart;
        unsigned int TempConfigSize;

	// This is the Real kernel Size
	memcpy(&TempConfigStart,(void*)0x011080+0x14,4);
	// this is the kernel Size we pass to the Kernel loader
	memcpy(&TempConfigSize, (void*)0x011080+0x18,4);

	Buffer = MmAllocateContiguousMemoryEx(CONFIG_BUFFERSIZE,MIN_KERNEL,
	                        MAX_KERNEL, 0, PAGE_READWRITE);

       	memset(Buffer,0x00,CONFIG_BUFFERSIZE);
       	memcpy(Buffer,(void*)0x010000+TempConfigStart,TempConfigSize);

	ParseConfig("\\??\\D:\\",Buffer,entry);

	return STATUS_SUCCESS;
}

// Remap the drive to the XBE directory, even if the drive already mapped
int RemapDrive(char *szDrive)
{
	ANSI_STRING LinkName, TargetName;
	char *DDrivePath = 0;
	char *temp = 0;

	// Allocate room for the drive path
	DDrivePath = (char *)MmAllocateContiguousMemoryEx(
				XeImageFileName->Length + 1,MIN_KERNEL,
	                        MAX_KERNEL, 0, PAGE_READWRITE);
	if (!DDrivePath)
		return 0;

	// Copy the XBE filename for now
	memcpy(DDrivePath, XeImageFileName->Buffer, XeImageFileName->Length);
	DDrivePath[XeImageFileName->Length] = 0;

	// Delete the trailing backslash, chopping off the XBE name, and make it
	// into an ANSI_STRING
	if (!(temp = HelpStrrchr(DDrivePath, '\\')))
		return 0;
	*temp = 0;

	RtlInitAnsiString(&TargetName, DDrivePath);

	// Set up the link
	RtlInitAnsiString(&LinkName, szDrive);
	IoDeleteSymbolicLink(&LinkName);

	if(!NT_SUCCESS(IoCreateSymbolicLink(&LinkName, &TargetName))) {
		dprintf("Error IoCreateSymbolicLink\n");
		return 0;
	}

	// Delete the filename memory
	MmFreeContiguousMemory(DDrivePath);

	return 1;
}

// Opens a file or directory for read-only access
// Length parameter is negative means use strlen()
// This was originally designed to open directories, but it turned out to be
// too much of a hassle and was scrapped.  Use only for files with the
// FILE_NON_DIRECTORY_FILE mode.
HANDLE OpenFile(HANDLE Root, LPCSTR Filename, LONG Length, ULONG Mode)
{
        ANSI_STRING FilenameString;
        OBJECT_ATTRIBUTES Attributes;
        IO_STATUS_BLOCK IoStatus;
        HANDLE Handle;

        // Initialize the filename string
        // If a length is specified, set up the string manually
        if (Length >= 0)
        {
                FilenameString.Length = (USHORT) Length;
                FilenameString.MaxLength = (USHORT) Length;
                FilenameString.Buffer = (PSTR) Filename;
        }
        // Use RtlInitAnsiString to do it for us
        else
                RtlInitAnsiString(&FilenameString, Filename);

        // Initialize the object attributes
        Attributes.Attributes = OBJ_CASE_INSENSITIVE;
        Attributes.RootDirectory = Root;
        Attributes.ObjectName = &FilenameString;

        // Try to open the file or directory
        if (!NT_SUCCESS(NtCreateFile(&Handle, GENERIC_READ | SYNCHRONIZE,
                &Attributes, &IoStatus, NULL, 0, FILE_SHARE_READ | FILE_SHARE_WRITE
                | FILE_SHARE_DELETE, FILE_OPEN, FILE_SYNCHRONOUS_IO_NONALERT |
                Mode)))
                return NULL;

        return Handle;
}

// Gets the size of a file
BOOL GetFileSize(HANDLE File, LONGLONG *Size)
{
        FILE_NETWORK_OPEN_INFORMATION SizeInformation;
        IO_STATUS_BLOCK IoStatus;

        // Try to retrieve the file size
        if (!NT_SUCCESS(NtQueryInformationFile(File, &IoStatus,
                &SizeInformation, sizeof(SizeInformation),
                FileNetworkOpenInformation)))
                return FALSE;

        *Size = SizeInformation.EndOfFile.QuadPart;
        return TRUE;
}

