

#ifndef MENU_H
#define MENU_H

#include <stdio.h>
#include <stdarg.h>
#include <xtl.h>
#include "debug.h"

#define		DIRENT_MAX_COUNT	512

const SHORT MENU_DEADZONE = (SHORT)( 0.24f * FLOAT(0x7FFF) );

#define		MENU_DEADZONE	0.15f
#define		MENU_OVERDRIVE	0.9f

#define		MENU_NORM_MULT	0.5f


typedef struct MenuEntry
{
	float	xpos;
	float	ypos;
	WCHAR	*text;
	bool	active;
	bool	selected;
	int		id;
}MENUENTRY;


class CXBoxMenu
{
public:

	CXBoxMenu(CXBoxDebug *debug,LPDIRECT3DTEXTURE8 backgroundtexture,CXBFont &font)
		:mFont(font)
	{
		int twiddle=0;

		mpMenu=NULL;
		mpDebug=debug;
		mpBackgroundTexture=backgroundtexture;
		mMenuSelect=0;

		// Default colours
		mSelectedColour=0xffff0000;
		mHighlightColour=0xffffffff;
		mLowlightColour=0x80ffffff;
		mDisabledColour=0x30ffffff;

		if( FAILED( D3DXCreateSprite(g_pd3dDevice,&mpSpriteInterface ) ) )
		{
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxMenu() - Failed to create sprite interface. Xbox Halted (Press OFF)");
			while(1) twiddle++;
		}

		// Texture size correction
		if(mpBackgroundTexture)
		{
			D3DSURFACE_DESC desc;
			mpBackgroundTexture->GetLevelDesc(0, &desc);
			mpDebug->Message(L"CXBoxMenu() - Size=%d, Width=%d, Height=%d",desc.Size,desc.Width,desc.Height);
			mBackgroundScaling.x=640.0f/desc.Width;
			mBackgroundScaling.y=480.0f/desc.Height;
		}
	}

	~CXBoxMenu()
	{
	}

	void SetColours(DWORD selected, DWORD highlight, DWORD lowlight, DWORD disabled)
	{
		mSelectedColour=selected;
		mHighlightColour=highlight;
		mLowlightColour=lowlight;
		mDisabledColour=disabled;
		if(mpDebug) mpDebug->Message(L"CXBoxMenu() - SetColours accepted %08x %08x %08x",highlight,lowlight,disabled);
	}

	void SetMenuStruct(MENUENTRY *menustruct)
	{
		mpMenu=menustruct;
		if(mpDebug) mpDebug->Message(L"CXBoxMenu() - SetMenuStruct accepted a new menu");
	}

	void ProcessMenu(XBGAMEPAD &Gamepad)
	{
		static DWORD rampstart=0;
		float velocity,t1,t2,scale;

		scale=0.0;
		t1 = (float)(Gamepad.sThumbLY)/32768.0f;
		t2 = (float)(Gamepad.sThumbRY)/32768.0f;

		velocity=t1+t2;
		// Of both are out of the deadzone then we must rescale
		if(fabs(t1)>MENU_DEADZONE && fabs(t2)>MENU_DEADZONE ) velocity/=2.0;

		// For our purposes the axis are reversed
		velocity=-velocity;

		// do dead zone and gamma curve
		if(fabs(velocity) < MENU_DEADZONE)
		{
			// zero dead zone (for spring slack & noise)
			velocity = 0.0f;
			rampstart=0;
		}
		else
		{
			//start at 0.0f after deadzone
			if( velocity < 0.0f ) velocity+=MENU_DEADZONE; else velocity-=MENU_DEADZONE;

			//rescale to correct range bacause of deadzone removal
			velocity *= (1.0f/(1.0f-MENU_DEADZONE));
			velocity *= MENU_NORM_MULT;
		}

		// Up/Down DPAD overides
		if(Gamepad.wPressedButtons & XINPUT_GAMEPAD_DPAD_UP) velocity=-1;
		if(Gamepad.wPressedButtons & XINPUT_GAMEPAD_DPAD_DOWN) velocity=1;

		// Select the new entry
		mMenuSelect+=velocity;

		// We must clamp the selector within the list bounds
		if(mMenuSelect<0) mMenuSelect=0;

		int loop=0;
		while(mpMenu[loop].text!=NULL) loop++;

		if((int)mMenuSelect>loop) mMenuSelect=(float)(loop-1);

	}

	void RenderMenu(void)
	{
		// The Backdrop, its allowed to be NULL
		if(mpBackgroundTexture)
		{
			mpSpriteInterface->Begin();
			mpSpriteInterface->Draw(mpBackgroundTexture,NULL,&mBackgroundScaling,NULL,0.0f ,NULL,D3DCOLOR_ARGB(0xff, 0xff, 0xff, 0xff));
			mpSpriteInterface->End();
		}

		// Not allowed to have a null struct
		if(mpMenu==NULL)
		{
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxMenu() - RenderMenu aborted, no menu struct.");
			return;
		}

		// Now the file list
		mFont.Begin();
		int loop=0;
		while(mpMenu[loop].text!=NULL)
		{
			// Special case if this is the selected entry
			if(loop==(int)mMenuSelect)
			{
				if(mpMenu[loop].active)
				{
					// Print 100% alpha
					mFont.DrawText(mpMenu[loop].xpos,mpMenu[loop].ypos,mHighlightColour,mpMenu[loop].text);
				}
				else
				{
					// Disabled menu item
					mFont.DrawText(mpMenu[loop].xpos,mpMenu[loop].ypos,mLowlightColour,mpMenu[loop].text);
				}
			}
			else
			{
				if(mpMenu[loop].selected)
				{
					// The one selected but not under the curson
					mFont.DrawText(mpMenu[loop].xpos,mpMenu[loop].ypos,mSelectedColour,mpMenu[loop].text);
				}
				else if(mpMenu[loop].active)
				{
					// Highlight as the one selected
					mFont.DrawText(mpMenu[loop].xpos,mpMenu[loop].ypos,mLowlightColour,mpMenu[loop].text);
				}
				else
				{
					// Disabled menu item
					mFont.DrawText(mpMenu[loop].xpos,mpMenu[loop].ypos,mDisabledColour,mpMenu[loop].text);
				}
			}
			// Next Menu item
			loop++;
		}
		mFont.End();	
	}

	int GetSelected(void)
	{
		return (int)mMenuSelect;
	}
	
	float				mMenuSelect;

	LPD3DXSPRITE		mpSpriteInterface;
	CXBoxDebug			*mpDebug;
	CXBFont				mFont;
	LPDIRECT3DTEXTURE8	mpBackgroundTexture;
	D3DXVECTOR2			mBackgroundScaling;
	MENUENTRY			*mpMenu;

	DWORD				mSelectedColour;
	DWORD				mHighlightColour;
	DWORD				mLowlightColour;
	DWORD				mDisabledColour;

};

#endif