/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Minimon.h"

void MinxCore::ExecCode()
{
	unsigned char code ;

	if ( m_Halted )
	{
		int cycles = PredictIRQ( FRAME_CYCLES );
		Clock( cycles );
		m_Clock += cycles;
		return ;
	}

	Fetch( code );

	switch( code )
	{
    case 0x00: // addb A, A
		Add( m_CpuReg.byte.A, m_CpuReg.byte.A, false );
		break ;
    case 0x01: // addb A, B
		Add( m_CpuReg.byte.A, m_CpuReg.byte.B, false );
		break ;
    case 0x02: // addb A, %(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x03: // addb A, [HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x04: // addb A, [NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x05: // addb A, [%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;
			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x06: // addb A,[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x07: // addb A,[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Add( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x08: // addcb A,A
		Add( m_CpuReg.byte.A, m_CpuReg.byte.A, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		break ;
    case 0x09: // addcb A,B
		Add( m_CpuReg.byte.B, m_CpuReg.byte.A, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		break ;
    case 0x0A: // addcb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x0B: // addcb A,[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x0C: // addcb A,[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( m_CpuReg.byte.I << 16 | m_CpuReg.word.N, byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x0D: // addcb A,[%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;
			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x0E: // addcb A,[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x0F: // addcb A,[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Add( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x10: // subb A, A
		Sub( m_CpuReg.byte.A, m_CpuReg.byte.A, false );
		break ;
    case 0x11: // subb A, B
		Sub( m_CpuReg.byte.A, m_CpuReg.byte.B, false );
		break ;
    case 0x12: // subb A, %(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x13: // subb A, [HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x14: // subb A, [NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x15: // subb A, [%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;
			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x16: // subb A,[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x17: // subb A,[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Sub( m_CpuReg.byte.A, byte, false );
		}
		break ;
    case 0x18: // subcb A,A
		Sub( m_CpuReg.byte.A, m_CpuReg.byte.A, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		break ;
    case 0x19: // subcb A,B
		Sub( m_CpuReg.byte.A, m_CpuReg.byte.B, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		break ;
    case 0x1A: // subcb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x1B: // subcb A,[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x1C: // subcb A,[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x1D: // subcb A,[%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;
			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x1E: // subcb A,[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x1F: // subcb A,[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Sub( m_CpuReg.byte.A, byte, (m_CpuReg.byte.F & FLAG_CARRY) != 0 );
		}
		break ;
    case 0x20: // andb A,A
		{
			m_CpuReg.byte.A &= m_CpuReg.byte.A;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x21: // andb A,B
		{
			m_CpuReg.byte.A &= m_CpuReg.byte.B;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x22: // andb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			m_CpuReg.byte.A &= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x23: // andb A,[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			m_CpuReg.byte.A &= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x24: // andb A,[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.addr.N, byte );
			m_CpuReg.byte.A &= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x25: // andb A,[%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;

			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );

			m_CpuReg.byte.A &= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x26: // andb A,[X1]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.XA, byte );
			m_CpuReg.byte.A &= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x27: // andb A,[X2]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.YA, byte );
			m_CpuReg.byte.A &= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x28: // orb A,A
		{
			m_CpuReg.byte.A |= m_CpuReg.byte.A;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x29: // orb A,B
		{
			m_CpuReg.byte.A |= m_CpuReg.byte.B;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2A: // orb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			m_CpuReg.byte.A |= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2B: // orb A,[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			m_CpuReg.byte.A |= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2C: // orb A,[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.addr.N, byte );
			m_CpuReg.byte.A |= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2D: // orb A,[%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;

			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );

			m_CpuReg.byte.A |= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2E: // orb A,[X1]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.XA, byte );
			m_CpuReg.byte.A |= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x2F: // orb A,[X2]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.YA, byte );
			m_CpuReg.byte.A |= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x30: // cmpb A,A
		{
			unsigned char left = m_CpuReg.byte.A;
			Sub( left, m_CpuReg.byte.A, false );
		}
		break ;
    case 0x31: // cmpb A,B
		{
			unsigned char left = m_CpuReg.byte.A;
			Sub( left, m_CpuReg.byte.B, false );
		}
		break ;
    case 0x32: // cmpb A,%(i8)s
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned char byte;

			Fetch( byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x33: // cmpb A,[HL]
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned char byte;

			Peek( m_CpuReg.addr.HLA, byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x34: // cmpb A,[NN+%(i8)s]
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned char byte;

			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x35: // cmpb A,[%(i16)s]
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned short addr;
			unsigned char byte;

			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x36: // cmpb A,[X1]
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned char byte;

			Peek( m_CpuReg.addr.XA, byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x37: // cmpb A,[X2]
		{
			unsigned char left = m_CpuReg.byte.A;
			unsigned char byte;

			Peek( m_CpuReg.addr.YA, byte );
			Sub( left, byte, false );
		}
		break ;
    case 0x38: // xorb A,A
		{
			m_CpuReg.byte.A ^= m_CpuReg.byte.A;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x39: // xorb A,B
		{
			m_CpuReg.byte.A ^= m_CpuReg.byte.B;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3A: // xorb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			m_CpuReg.byte.A ^= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3B: // xorb A,[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			m_CpuReg.byte.A ^= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3C: // xorb A,[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.addr.N, byte );
			m_CpuReg.byte.A ^= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3D: // xorb A,[%(i16)s]
		{
			unsigned short addr;
			unsigned char byte;

			Fetch( addr );
			Peek( (m_CpuReg.byte.I << 16) | addr, byte );

			m_CpuReg.byte.A ^= byte;
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3E: // xorb A,[X1]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.XA, byte );
			m_CpuReg.byte.A ^= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x3F: // xorb A,[X2]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.YA, byte );
			m_CpuReg.byte.A ^= byte;

			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A );
		}
		break ;
    case 0x40: // movb A,A
		m_CpuReg.byte.A = m_CpuReg.byte.A;
		break ;
    case 0x41: // movb A,B
		m_CpuReg.byte.A = m_CpuReg.byte.B;
		break ;
    case 0x42: // movb A,L
		m_CpuReg.byte.A = m_CpuReg.byte.L;
		break ;
    case 0x43: // movb A,H
		m_CpuReg.byte.A = m_CpuReg.byte.H;
		break ;
    case 0x44: // movb A,[NN+%(i8)s]
		Fetch( m_CpuReg.byte.IndexN );
		Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, m_CpuReg.byte.A );
		break ;
    case 0x45: // movb A,[HL]
		Peek( m_CpuReg.addr.HLA, m_CpuReg.byte.A );
		break ;
    case 0x46: // movb A,[X1]
		Peek( m_CpuReg.addr.XA, m_CpuReg.byte.A );
		break ;
    case 0x47: // movb A,[X2]
		Peek( m_CpuReg.addr.YA, m_CpuReg.byte.A );
		break ;
    case 0x48: // movb B,A
		m_CpuReg.byte.B = m_CpuReg.byte.A;
		break;
    case 0x49: // movb B,B
		m_CpuReg.byte.B = m_CpuReg.byte.B;
		break;
    case 0x4A: // movb B,L
		m_CpuReg.byte.B = m_CpuReg.byte.L;
		break;
    case 0x4B: // movb B,H
		m_CpuReg.byte.B = m_CpuReg.byte.H;
		break;
    case 0x4C: // movb B,[NN+%(i8)s]
		{
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | (m_CpuReg.word.N), m_CpuReg.byte.B );
		}
		break ;
    case 0x4D: // movb B,[HL]
		Peek( m_CpuReg.addr.HLA, m_CpuReg.byte.B );
		break ;
    case 0x4E: // movb B,[X1]
		Peek( m_CpuReg.addr.XA, m_CpuReg.byte.B );
		break ;
    case 0x4F: // movb B,[X2]
		Peek( m_CpuReg.addr.YA, m_CpuReg.byte.B );
		break ;
    case 0x50: // movb L,A
		m_CpuReg.byte.L = m_CpuReg.byte.A;
		break;
    case 0x51: // movb L,B
		m_CpuReg.byte.L = m_CpuReg.byte.B;
		break;
    case 0x52: // movb L,L
		m_CpuReg.byte.L = m_CpuReg.byte.L;
		break;
    case 0x53: // movb L,H
		m_CpuReg.byte.L = m_CpuReg.byte.H;
		break;
    case 0x54: // movb L,[NN+%(i8)s]
		{
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | (m_CpuReg.word.N), m_CpuReg.byte.L );
		}
		break ;
    case 0x55: // movb L,[HL]
		Peek( m_CpuReg.addr.HLA, m_CpuReg.byte.L );
		break ;
    case 0x56: // movb L,[X1]
		Peek( m_CpuReg.addr.XA, m_CpuReg.byte.L );
		break ;
    case 0x57: // movb L,[X2]
		Peek( m_CpuReg.addr.YA, m_CpuReg.byte.L );
		break ;
    case 0x58: // movb H,A
		m_CpuReg.byte.H = m_CpuReg.byte.A;
		break;
    case 0x59: // movb H,B
		m_CpuReg.byte.H = m_CpuReg.byte.B;
		break;
    case 0x5A: // movb H,L
		m_CpuReg.byte.H = m_CpuReg.byte.L;
		break;
    case 0x5B: // movb H,H
		m_CpuReg.byte.H = m_CpuReg.byte.H;
		break;
    case 0x5C: // movb H,[NN+%(i8)s]
		{
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | (m_CpuReg.word.N), m_CpuReg.byte.H );
		}
		break ;
    case 0x5D: // movb H,[HL]
		Peek( m_CpuReg.addr.HLA, m_CpuReg.byte.H );
		break ;
    case 0x5E: // movb H,[X1]
		Peek( m_CpuReg.addr.XA, m_CpuReg.byte.H );
		break ;
    case 0x5F: // movb H,[X2]
		Peek( m_CpuReg.addr.YA, m_CpuReg.byte.H );
		break ;
    case 0x60: // movb [X1],A
		Poke( m_CpuReg.addr.XA, m_CpuReg.byte.A );
		break ;
    case 0x61: // movb [X1],B
		Poke( m_CpuReg.addr.XA, m_CpuReg.byte.B );
		break ;
    case 0x62: // movb [X1],L
		Poke( m_CpuReg.addr.XA, m_CpuReg.byte.L );
		break ;
    case 0x63: // movb [X1],H
		Poke( m_CpuReg.addr.XA, m_CpuReg.byte.H );
		break ;
    case 0x64: // movb [X1],[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Poke( m_CpuReg.addr.XA, byte );
		}
		break ;
    case 0x65: // movb [X1],[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Poke( m_CpuReg.addr.XA, byte );
		}
		break ;
    case 0x66: // movb [X1],[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Poke( m_CpuReg.addr.XA, byte );
		}
		break ;
    case 0x67: // movb [X1],[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Poke( m_CpuReg.addr.XA, byte );
		}
		break ;
    case 0x68: // movb [HL],A
		Poke( m_CpuReg.addr.HLA, m_CpuReg.byte.A );
		break ;
    case 0x69: // movb [HL],B
		Poke( m_CpuReg.addr.HLA, m_CpuReg.byte.B );
		break ;
    case 0x6A: // movb [HL],L
		Poke( m_CpuReg.addr.HLA, m_CpuReg.byte.L );
		break ;
    case 0x6B: // movb [HL],H
		Poke( m_CpuReg.addr.HLA, m_CpuReg.byte.H );
		break ;
    case 0x6C: // movb [HL],[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x6D: // movb [HL],[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x6E: // movb [HL],[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x6F: // movb [HL],[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x70: // movb [X2],A
		Poke( m_CpuReg.addr.YA, m_CpuReg.byte.A );
		break ;
    case 0x71: // movb [X2],B
		Poke( m_CpuReg.addr.YA, m_CpuReg.byte.B );
		break ;
    case 0x72: // movb [X2],L
		Poke( m_CpuReg.addr.YA, m_CpuReg.byte.L );
		break ;
    case 0x73: // movb [X2],H
		Poke( m_CpuReg.addr.YA, m_CpuReg.byte.H );
		break ;
    case 0x74: // movb [X2],[NN+%(i8)s]
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Poke( m_CpuReg.addr.YA, byte );
		}
		break ;
    case 0x75: // movb [X2],[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Poke( m_CpuReg.addr.YA, byte );
		}
		break ;
    case 0x76: // movb [X2],[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Poke( m_CpuReg.addr.YA, byte );
		}
		break ;
    case 0x77: // movb [X2],[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Poke( m_CpuReg.addr.YA, byte );
		}
		break ;
    case 0x78: // movb [NN+%(i8)s],A
		Fetch( m_CpuReg.byte.IndexN );
		Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, m_CpuReg.byte.A );
		break ;
    case 0x79: // movb [NN+%(i8)s],B
		Fetch( m_CpuReg.byte.IndexN );
		Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, m_CpuReg.byte.B );
		break ;
    case 0x7A: // movb [NN+%(i8)s],L
		Fetch( m_CpuReg.byte.IndexN );
		Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, m_CpuReg.byte.L );
		break ;
    case 0x7B: // movb [NN+%(i8)s],H
		Fetch( m_CpuReg.byte.IndexN );
		Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, m_CpuReg.byte.H );
		break ;
    case 0x7D: // movb [NN+%(i8)s],[HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			Fetch( m_CpuReg.byte.IndexN );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
		}
		break ;
    case 0x7E: // movb [NN+%(i8)s],[X1]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.XA, byte );
			Fetch( m_CpuReg.byte.IndexN );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
		}
		break ;
    case 0x7F: // movb [NN+%(i8)s],[X2]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.YA, byte );
			Fetch( m_CpuReg.byte.IndexN );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
		}
		break ;
    case 0x80: // incb A
		Flags8( FLAG_ZERO, ++m_CpuReg.byte.A );
		break ;
    case 0x81: // incb B
		Flags8( FLAG_ZERO, ++m_CpuReg.byte.B );
		break ;
    case 0x82: // incb L
		Flags8( FLAG_ZERO, ++m_CpuReg.byte.L );
		break ;
    case 0x83: // incb H
		Flags8( FLAG_ZERO, ++m_CpuReg.byte.H );
		break ;
    case 0x84: // incb N
		Flags8( FLAG_ZERO, ++m_CpuReg.byte.N );
		break ;
    case 0x85: // incb [NN+%(i8)s]
		{
			unsigned char byte;
			
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Flags8( FLAG_ZERO, ++byte );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
		}
		break ;
    case 0x86: // incb [HL]
		{
			unsigned char byte;
			
			Peek( m_CpuReg.addr.HLA, byte );
			Flags8( FLAG_ZERO, ++byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x87: // incw SP
		Flags16( FLAG_ZERO, ++m_CpuReg.word.SP );
		break ;
    case 0x88: // decb A
		Flags8( FLAG_ZERO, --m_CpuReg.byte.A );
		break ;
    case 0x89: // decb B
		Flags8( FLAG_ZERO, --m_CpuReg.byte.B );
		break ;
    case 0x8A: // decb L
		Flags8( FLAG_ZERO, --m_CpuReg.byte.L );
		break ;
    case 0x8B: // decb H
		Flags8( FLAG_ZERO, --m_CpuReg.byte.H );
		break ;
    case 0x8C: // decb N
		Flags8( FLAG_ZERO, --m_CpuReg.byte.N );
		break ;
    case 0x8D: // decb [NN+%(i8)s]
		{
			unsigned char byte;
			
			Fetch( m_CpuReg.byte.IndexN );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
			Flags8( FLAG_ZERO, --byte );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );
		}
		break ;
    case 0x8E: // decb [HL]
		{
			unsigned char byte;
			
			Peek( m_CpuReg.addr.HLA, byte );
			Flags8( FLAG_ZERO, --byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0x8F: // decw SP
		Flags16( FLAG_ZERO, --m_CpuReg.word.SP );
		break ;
    case 0x90: // incw BA
		Flags16( FLAG_ZERO, ++m_CpuReg.word.BA );
		break ;
    case 0x91: // incw HL
		Flags16( FLAG_ZERO, ++m_CpuReg.word.HL );
		break ;
    case 0x92: // incw X1
		Flags16( FLAG_ZERO, ++m_CpuReg.word.X );
		break ;
    case 0x93: // incw X2
		Flags16( FLAG_ZERO, ++m_CpuReg.word.Y );
		break ;
    case 0x94: // testb A,B
		Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A & m_CpuReg.byte.B );
		break ;
    case 0x95: // testb [HL],%(i8)s
		{
			unsigned char byteA, byteB;
			Peek( m_CpuReg.addr.HLA, byteA );
			Fetch( byteB );
			Flags8( FLAG_SIGN | FLAG_ZERO, byteA & byteB );
		}
		break ;
    case 0x96: // testb A,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.A & byte );
		}
		break ;
    case 0x97: // testb B,%(i8)s
		{
			unsigned char byte;
			Fetch( byte );
			Flags8( FLAG_SIGN | FLAG_ZERO, m_CpuReg.byte.B & byte );
		}
		break ;
    case 0x98: // decw BA
		Flags16( FLAG_ZERO, --m_CpuReg.word.BA );
		break ;
    case 0x99: // decw HL
		Flags16( FLAG_ZERO, --m_CpuReg.word.HL );
		break ;
    case 0x9A: // decw X1
		Flags16( FLAG_ZERO, --m_CpuReg.word.X );
		break ;
    case 0x9B: // decw X2
		Flags16( FLAG_ZERO, --m_CpuReg.word.Y );
		break ;
    case 0x9C: // andb FLAGS,%(i8)s
		{
			unsigned char flag;
			Fetch( flag );
			m_CpuReg.byte.F &= flag;
		}
		break;
    case 0x9D: // orb FLAGS,%(i8)s
		{
			unsigned char flag;
			Fetch( flag );
			m_CpuReg.byte.F |= flag;
		}
		break;
    case 0x9E: // xorb FLAGS,%(i8)s
		{
			unsigned char flag;
			Fetch( flag );
			m_CpuReg.byte.F ^= flag;
		}
		break;
    case 0x9F: // movb FLAGS,%(i8)s
		{
			unsigned char flag;
			Fetch( flag );
			m_CpuReg.byte.F = flag;
		}
		break;
    case 0xA0: // pushw BA
		Push( m_CpuReg.word.BA );
		break;
    case 0xA1: // pushw HL
		Push( m_CpuReg.word.HL );
		break;
    case 0xA2: // pushw X
		Push( m_CpuReg.word.X );
		break;
    case 0xA3: // pushw Y
		Push( m_CpuReg.word.Y );
		break;
    case 0xA4: // pushb N
		Push( m_CpuReg.byte.N );
		break ;
    case 0xA5: // pushx HL
		Push( m_CpuReg.byte.I );
		break ;
    case 0xA6: // pushxxx
		Push( m_CpuReg.byte.XI );
		Push( m_CpuReg.byte.YI );
		break ;
    case 0xA7: // pushb FLAGS
		Push( m_CpuReg.byte.F );
		break ;
    case 0xA8: // popw BA
		Pop( m_CpuReg.word.BA );
		break ;
    case 0xA9: // popw HL
		Pop( m_CpuReg.word.HL );
		break ;
    case 0xAA: // popw X1
		Pop( m_CpuReg.word.X );
		break ;
    case 0xAB: // popw X2
		Pop( m_CpuReg.word.Y );
		break ;
    case 0xAC: // popb N
		Pop( m_CpuReg.byte.N );
		break ;
    case 0xAD: // popx HL
		Pop( m_CpuReg.byte.I );
		break ;
    case 0xAE: // popxxx
		Pop( m_CpuReg.byte.YI );
		Pop( m_CpuReg.byte.XI );
		break ;
    case 0xAF: // popb FLAGS
		Pop( m_CpuReg.byte.F );
		break ;
    case 0xB0: // movb A,%(i8)s
		Fetch( m_CpuReg.byte.A );
		break ;
    case 0xB1: // movb B,%(i8)s
		Fetch( m_CpuReg.byte.B );
		break ;
    case 0xB2: // movb L,%(i8)s
		Fetch( m_CpuReg.byte.L );
		break ;
    case 0xB3: // movb H,%(i8)s
		Fetch( m_CpuReg.byte.H );
		break ;
    case 0xB4: // movw NN,%(i8)s00
		Fetch( m_CpuReg.byte.N );
		break ;
    case 0xB5: // movb [HL],%(i8)s
		{
			unsigned char byte;

			Fetch( byte );
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0xB6: // movb [X1],%(i8)s
		{
			unsigned char byte;

			Fetch( byte );
			Poke( m_CpuReg.addr.XA, byte );
		}
		break ;
    case 0xB7: // movb [X2],%(i8)s
		{
			unsigned char byte;

			Fetch( byte );
			Poke( m_CpuReg.addr.YA, byte );
		}
		break ;
    case 0xB8: // movw BA,[%(i16)s]
		{
			unsigned short addr;
			Fetch( addr );
			Peek( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.BA );
		}
		break ;
    case 0xB9: // movw HL,[%(i16)s]
		{
			unsigned short addr;
			Fetch( addr );
			Peek( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.HL );
		}
		break ;
    case 0xBA: // movw X1,[%(i16)s]
		{
			unsigned short addr;
			Fetch( addr );
			Peek( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.X );
		}
		break ;
    case 0xBB: // movw X2,[%(i16)s]
		{
			unsigned short addr;
			Fetch( addr );
			Peek( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.Y );
		}
		break ;
    case 0xBC: // movw [%(i16)s],BA
    	{
			unsigned short addr;
			Fetch( addr );
			Poke( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.BA );
		}
		break ;
	case 0xBD: // movw [%(i16)s],HL
    	{
			unsigned short addr;
			Fetch( addr );
			Poke( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.HL );
		}
		break ;
    case 0xBE: // movw [%(i16)s],X1
    	{
			unsigned short addr;
			Fetch( addr );
			Poke( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.X );
		}
		break ;
    case 0xBF: // movw [%(i16)s],X2
    	{
			unsigned short addr;
			Fetch( addr );
			Poke( (long)addr | (m_CpuReg.byte.I << 16), m_CpuReg.word.Y );
		}
		break ;
    case 0xC0: // addw BA,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Add( m_CpuReg.word.BA, word, false );
		}		
		break ;
    case 0xC1: // addw HL,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Add( m_CpuReg.word.HL, word, false );
		}		
		break ;
    case 0xC2: // addw X1,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Add( m_CpuReg.word.X, word, false );
		}		
		break ;
    case 0xC3: // addw X2,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Add( m_CpuReg.word.Y, word, false );
		}		
		break ;
    case 0xC4: // movw BA,%(i16)s
		Fetch( m_CpuReg.word.BA );
		break ;
    case 0xC5: // movw HL,%(i16)s
		Fetch( m_CpuReg.word.HL );
		break ;
    case 0xC6: // movw X1,%(i16)s
		Fetch( m_CpuReg.word.X );
		break ;
    case 0xC7: // movw X2,%(i16)s
		Fetch( m_CpuReg.word.Y );
		break ;
    case 0xC8: // xchgw BA,HL
		{
			unsigned short temp;

			temp = m_CpuReg.word.BA;
			m_CpuReg.word.BA = m_CpuReg.word.HL;
			m_CpuReg.word.HL = temp;
		}
		break ;
    case 0xC9: // xchgw BA,X1
		{
			unsigned short temp;

			temp = m_CpuReg.word.BA;
			m_CpuReg.word.BA = m_CpuReg.word.X;
			m_CpuReg.word.X = temp;
		}
		break ;
    case 0xCA: // xchgw BA,X2
		{
			unsigned short temp;

			temp = m_CpuReg.word.BA;
			m_CpuReg.word.BA = m_CpuReg.word.Y;
			m_CpuReg.word.Y = temp;
		}
		break ;
    case 0xCB: // xchgw BA,SP
		{
			unsigned short temp;

			temp = m_CpuReg.word.BA;
			m_CpuReg.word.BA = m_CpuReg.word.SP;
			m_CpuReg.word.SP = temp;
		}
		break ;
    case 0xCC: // xchgb A,B
		{
			unsigned char temp;

			temp = m_CpuReg.byte.B;
			m_CpuReg.byte.B = m_CpuReg.byte.A;
			m_CpuReg.byte.A = temp;
		}
		break ;
    case 0xCD: // xchgb A,[HL]
		{
			unsigned char byte;

			Peek( m_CpuReg.addr.HLA, byte );
			Poke( m_CpuReg.addr.HLA, m_CpuReg.byte.A );
			m_CpuReg.byte.A = byte;
		}
		break ;
	case 0xCE:
		ExecCodeCE();
		return ;
	case 0xCF:
		ExecCodeCF();
		return ;
    case 0xD0: // subw BA,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Sub( m_CpuReg.word.BA, word, false );
		}		
		break ;
    case 0xD1: // subw HL,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Sub( m_CpuReg.word.HL, word, false );
		}		
		break ;
    case 0xD2: // subw X1,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Sub( m_CpuReg.word.X, word, false );
		}		
		break ;
    case 0xD3: // subw X2,%(i16)s
    	{
			unsigned short word;
			Fetch( word );
			Sub( m_CpuReg.word.Y, word, false );
		}		
		break ;
    case 0xD4: // cmpw BA,%(i16)s
    	{
			unsigned short left = m_CpuReg.word.BA;
			unsigned short right;
			Fetch( right );
			Sub( left, right, false );
		}		
		break ;
    case 0xD5: // cmpw HL,%(i16)s
    	{
			unsigned short left = m_CpuReg.word.HL;
			unsigned short right;
			Fetch( right );
			Sub( left, right, false );
		}		
		break ;
    case 0xD6: // cmpw X1,%(i16)s
    	{
			unsigned short left = m_CpuReg.word.X;
			unsigned short right;
			Fetch( right );
			Sub( left, right, false );
		}		
		break ;
    case 0xD7: // cmpw X2,%(i16)s
    	{
			unsigned short left = m_CpuReg.word.Y;
			unsigned short right;
			Fetch( right );
			Sub( left, right, false );
		}		
		break ;
    case 0xD8: // andb [NN+%(i8)s],%(b8)s
		{
			unsigned char byteA, byteB;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byteB );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
			
			byteA &= byteB;
			Flags8( FLAG_ZERO | FLAG_SIGN, byteA );

			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
		}
		break ;
    case 0xD9: // orb [NN+%(i8)s],%(b8)s
		{
			unsigned char byteA, byteB;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byteB );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
			
			byteA |= byteB;
			Flags8( FLAG_ZERO | FLAG_SIGN, byteA );

			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
		}
		break ;
    case 0xDA: // xorb [NN+%(i8)s],%(b8)s
		{
			unsigned char byteA, byteB;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byteB );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
			
			byteA ^= byteB;
			Flags8( FLAG_ZERO | FLAG_SIGN, byteA );

			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
		}
		break ;
    case 0xDB: // cmpb [NN+%(i8)s],%(b8)s
		{
			unsigned char byteA, byteB;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byteB );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );

			Sub( byteA, byteB, false );
		}
		break ;
    case 0xDC: // testb [NN+%(i8)s],%(b8)s
		{
			unsigned char byteA, byteB;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byteB );
			Peek( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byteA );
			Flags8( FLAG_ZERO | FLAG_SIGN, byteA & byteB );
		}
		break ;
    case 0xDD: // movb [NN+%(i8)s],%(b8)s
		{
			unsigned char byte;
			Fetch( m_CpuReg.byte.IndexN );
			Fetch( byte );
			Poke( (m_CpuReg.byte.I << 16) | m_CpuReg.word.N, byte );			
		}
		break ;
    case 0xDE: // bcde
		{
			m_CpuReg.byte.A = (m_CpuReg.byte.A & 0x0F) | (m_CpuReg.byte.B << 4);
		}
		break ;
    case 0xDF: // bcdd
		{
			m_CpuReg.byte.B = m_CpuReg.byte.A >> 4;
			m_CpuReg.byte.A = m_CpuReg.byte.A & 0x0F;
		}
		break ;
    case 0xE0: // callc %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_CARRY )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed char) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE1: // callnc %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_CARRY) )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed char) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE2: // callz %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_ZERO )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed char) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE3: // callnz %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_ZERO) )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed char) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE4: // jc %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_CARRY )
			{
				m_CpuReg.word.PC += (signed char) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE5: // jnc %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_CARRY) )
			{
				m_CpuReg.word.PC += (signed char) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE6: // jz %(d8)s
		{
			unsigned char offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_ZERO )
			{
				m_CpuReg.word.PC += (signed char) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE7: // jnz %(addr)s
		{
			unsigned char offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_ZERO) )
			{
				m_CpuReg.word.PC += (signed char) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE8: // callcw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_CARRY )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed short) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xE9: // callncw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_CARRY) )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed short) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xEA: // callzw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_ZERO )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed short) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xEB: // callnzw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_ZERO) )
			{
				Push( m_CpuReg.byte.V );
				Push( m_CpuReg.word.PC );

				m_CpuReg.word.PC += (signed short) offset - 1;

				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xEC: // jcw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_CARRY )
			{
				m_CpuReg.word.PC += (signed short) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xED: // jncw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_CARRY) )
			{
				m_CpuReg.word.PC += (signed short) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xEE: // jzw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( m_CpuReg.byte.F & FLAG_ZERO )
			{
				m_CpuReg.word.PC += (signed short) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xEF: // jnzw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );

			if( !(m_CpuReg.byte.F & FLAG_ZERO) )
			{
				m_CpuReg.word.PC += (signed short) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xF0: // callb %(addr)s
		{
			unsigned char offset;
			Fetch( offset );
			Push( m_CpuReg.byte.V );
			Push( m_CpuReg.word.PC );
			m_CpuReg.word.PC += (signed char) offset - 1;
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			LogBranch( m_CpuReg.byte.PC );
		}
		break ;
    case 0xF1: // jmp %(addr)s
		{
			unsigned char offset;
			Fetch( offset );
			m_CpuReg.word.PC += (signed char) offset - 1;
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			LogBranch( m_CpuReg.byte.PC );
		}
		break ;
    case 0xF2: // callw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );
			Push( m_CpuReg.byte.V );
			Push( m_CpuReg.word.PC );
			m_CpuReg.word.PC += (signed short) offset - 1;
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			LogBranch( m_CpuReg.byte.PC );
		}
		break ;
    case 0xF3: // jmpw %(addr)s
		{
			unsigned short offset;
			Fetch( offset );
			m_CpuReg.word.PC += (signed short) offset - 1;
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			LogBranch( m_CpuReg.byte.PC );
		}
		break ;
    case 0xF4: // jmp HL
		m_CpuReg.word.PC = m_CpuReg.word.HL;
		m_CpuReg.byte.V = m_CpuReg.byte.U;
		LogBranch( m_CpuReg.byte.PC );
		break ;
    case 0xF5: // jdbnz %(addr)s
		{
			unsigned char offset;
			Fetch( offset );
			Flags8( FLAG_ZERO, --m_CpuReg.byte.B );
			if( m_CpuReg.byte.B != 0 )
			{
				m_CpuReg.word.PC += (signed char) offset - 1;
				m_CpuReg.byte.V = m_CpuReg.byte.U;
				LogBranch( m_CpuReg.byte.PC );
			}
		}
		break ;
    case 0xF6: // bcdx A
		m_CpuReg.byte.A = (m_CpuReg.byte.A >> 4 & 0x0F) | (m_CpuReg.byte.A << 4 & 0xF0);
		break ;
    case 0xF7: // bcdx [HL]
		{
			unsigned char byte;
			Peek( m_CpuReg.addr.HLA, byte );
			byte = (byte >> 4 & 0x0F) | (byte << 4 & 0xF0);
			Poke( m_CpuReg.addr.HLA, byte );
		}
		break ;
    case 0xF8: // ret
		Pop( m_CpuReg.word.PC );
		Pop( m_CpuReg.byte.V );
		m_CpuReg.byte.U = m_CpuReg.byte.V;
		break ;
    case 0xF9: // reti
		Pop( m_CpuReg.byte.F );
		Pop( m_CpuReg.word.PC );
		Pop( m_CpuReg.byte.V );
		m_CpuReg.byte.U = m_CpuReg.byte.V;
		LogBranch( m_CpuReg.byte.PC );
		break ;
    case 0xFA: // retskip
		Pop( m_CpuReg.word.PC );
		Pop( m_CpuReg.byte.V );
		m_CpuReg.byte.U = m_CpuReg.byte.V;
		m_CpuReg.word.PC += 2;
		LogBranch( m_CpuReg.byte.PC );
		break ;
    case 0xFB: // call [%(i16)s]
		{
			unsigned short addr;
			
			Fetch( addr );
			Push( m_CpuReg.byte.V );
			Push( m_CpuReg.byte.PC );
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			Peek( (long)(addr) | (m_CpuReg.byte.I << 16), m_CpuReg.word.PC );
			LogBranch( m_CpuReg.byte.PC );
		}
		break;

    case 0xFC: // int %(irq)s
		{
			unsigned char IRQ;
			
			Fetch( IRQ );
			ForceIRQ( IRQ );
			LogBranch( m_CpuReg.byte.PC );

			int errorIRQ = IRQ / 2;
			this->Error( "IRQ Request", 1, &errorIRQ );
		}
		break;
    case 0xFD: // jint %(irq)s
		{
			unsigned char IRQ;

			Fetch( IRQ );
			Peek( IRQ & 0xFE, m_CpuReg.word.PC );
			m_CpuReg.byte.V = m_CpuReg.byte.U;
			LogBranch( m_CpuReg.byte.PC );

			int errorIRQ = IRQ / 2;
		}

		// ????  Should do some other stuff here...
		break ;
    case 0xFF: // nop
		break ;
	default:
		{
			int errorCode[] = {code};
			this->Error( "Unknown opcode", 1, errorCode );
		}
		break ;
	}

	m_Clock += CycleTable[ code ];
	Clock( CycleTable[ code ] );
}
