/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __DISASSEMBLY_H
#define __DISASSEMBLY_H

#define SPLIT_HEIGHT (9 * 16)

enum ValueMode
{
	VALUE_MEMORY,
	VALUE_ADDRESS,
	VALUE_LINE_ADDRESS,
	VALUE_OP,
	VALUE_NONE,
	VALUE_TERM
};

enum
{
	MEMPROC_WATCH = 1,
	MEMPROC_EDIT  = 2
};

enum Immediate
{
	NO_IMMEDIATE,

	// Immediate values
	IMM_8,		// 0x00
	IMM_16,		// 0x0000

	// 16 bit register immediate	
	REG_PC,	
	REG_SP,	
	REG_BA,	
	REG_HL,	
	REG_X,
	REG_Y,

	// 8 bit register immediate
	REG_A,	REG_B,	REG_H,	REG_L,	REG_U,	REG_V,
	REG_F,	REG_E,	REG_XI,	REG_YI,	REG_N,	REG_I,

	// Addressing modes
	ADDR_I16,		// [0x0000]
	ADDR_I_I16,		// [I+0x0000]
	ADDR_N_I8,		// [I+N+0xFF]
	ADDR_HL,		// [HL]
	ADDR_X,			// [X]
	ADDR_Y,			// [Y]
	ADDR_PC_I8,		// 0x0000	(calculated by PC + signed 8 bit)
	ADDR_PC_I16,	// 0x0000	(calculated by PC + signed 16 bit)
	ADDR_SP_I8,		// [SP + 0x00]
	ADDR_X_I8,		// [X + 0x00]
	ADDR_Y_I8,		// [Y + 0x00]
	ADDR_X_L,		// [X + L]
	ADDR_Y_L,		// [Y + L]
};

struct DasmValue
{
	int m_Mode;
	int m_Value;

	RECT rect;
};

struct DisasmData
{
	ColorSet *m_Colors;

	Mini *m_Emulator;
	Executor *m_Executor;

	HBITMAP m_Font;

	DasmValue m_ValueBlock[1024];

	long m_LastBranched;
	int  m_DasmViewMode;
	int  m_LastView;
	long m_FollowAddress;
	unsigned short m_UserAddress;
};

struct Operation
{
	char *m_Mnemonic;	// "mov"
	char *m_Extender;	// "byte"  ex  "mov byte [X],[Y]"  "short"  "jmp short 0x0000"
	
	char m_Op[2];		// Left and right operands
	char m_Length;
};

#ifdef EXPORT

Operation Mnemonic[0x300] = {
	/* 00 */ {	"add",		NULL,			{REG_A,			REG_A},				1},
	/* 01 */ {	"add",		NULL,			{REG_A,			REG_B},				1},
	/* 02 */ {	"add",		NULL,			{REG_A,			IMM_8},				2},
	/* 03 */ {	"add",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 04 */ {	"add",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 05 */ {	"add",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 06 */ {	"add",		NULL,			{REG_A,			ADDR_X},			1},
	/* 07 */ {	"add",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 08 */ {	"addc",		NULL,			{REG_A,			REG_A},				1},
	/* 09 */ {	"addc",		NULL,			{REG_A,			REG_B},				1},
	/* 0A */ {	"addc",		NULL,			{REG_A,			IMM_8},				2},
	/* 0B */ {	"addc",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 0C */ {	"addc",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 0D */ {	"addc",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 0E */ {	"addc",		NULL,			{REG_A,			ADDR_X},			1},
	/* 0F */ {	"addc",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 10 */ {	"sub",		NULL,			{REG_A,			REG_A},				1},
	/* 11 */ {	"sub",		NULL,			{REG_A,			REG_B},				1},
	/* 12 */ {	"sub",		NULL,			{REG_A,			IMM_8},				2},
	/* 13 */ {	"sub",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 14 */ {	"sub",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 15 */ {	"sub",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 16 */ {	"sub",		NULL,			{REG_A,			ADDR_X},			1},
	/* 17 */ {	"sub",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 18 */ {	"subc",		NULL,			{REG_A,			REG_A},				1},
	/* 19 */ {	"subc",		NULL,			{REG_A,			REG_B},				1},
	/* 1A */ {	"subc",		NULL,			{REG_A,			IMM_8},				2},
	/* 1B */ {	"subc",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 1C */ {	"subc",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 1D */ {	"subc",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 1E */ {	"subc",		NULL,			{REG_A,			ADDR_X},			1},
	/* 1F */ {	"subc",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 20 */ {	"and",		NULL,			{REG_A,			REG_A},				1},
	/* 21 */ {	"and",		NULL,			{REG_A,			REG_B},				1},
	/* 22 */ {	"and",		NULL,			{REG_A,			IMM_8},				2},
	/* 23 */ {	"and",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 24 */ {	"and",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 25 */ {	"and",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 26 */ {	"and",		NULL,			{REG_A,			ADDR_X},			1},
	/* 27 */ {	"and",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 28 */ {	"or",		NULL,			{REG_A,			REG_A},				1},
	/* 29 */ {	"or",		NULL,			{REG_A,			REG_B},				1},
	/* 2A */ {	"or",		NULL,			{REG_A,			IMM_8},				2},
	/* 2B */ {	"or",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 2C */ {	"or",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 2D */ {	"or",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 2E */ {	"or",		NULL,			{REG_A,			ADDR_X},			1},
	/* 2F */ {	"or",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 30 */ {	"cmp",		NULL,			{REG_A,			REG_A},				1},
	/* 31 */ {	"cmp",		NULL,			{REG_A,			REG_B},				1},
	/* 32 */ {	"cmp",		NULL,			{REG_A,			IMM_8},				2},
	/* 33 */ {	"cmp",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 34 */ {	"cmp",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 35 */ {	"cmp",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 36 */ {	"cmp",		NULL,			{REG_A,			ADDR_X},			1},
	/* 37 */ {	"cmp",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 38 */ {	"xor",		NULL,			{REG_A,			REG_A},				1},
	/* 39 */ {	"xor",		NULL,			{REG_A,			REG_B},				1},
	/* 3A */ {	"xor",		NULL,			{REG_A,			IMM_8},				2},
	/* 3B */ {	"xor",		NULL,			{REG_A,			ADDR_HL},			1},
	/* 3C */ {	"xor",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 3D */ {	"xor",		NULL,			{REG_A,			ADDR_I_I16},		3},
	/* 3E */ {	"xor",		NULL,			{REG_A,			ADDR_X},			1},
	/* 3F */ {	"xor",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 40 */ {	"mov",		NULL,			{REG_A,			REG_A},				1},
	/* 41 */ {	"mov",		NULL,			{REG_A,			REG_B},				1},
	/* 42 */ {	"mov",		NULL,			{REG_A,			REG_L},				2},
	/* 43 */ {	"mov",		NULL,			{REG_A,			REG_H},				1},
	/* 44 */ {	"mov",		NULL,			{REG_A,			ADDR_N_I8},			2},
	/* 45 */ {	"mov",		NULL,			{REG_A,			ADDR_HL},			3},
	/* 46 */ {	"mov",		NULL,			{REG_A,			ADDR_X},			1},
	/* 47 */ {	"mov",		NULL,			{REG_A,			ADDR_Y},			1},
	/* 48 */ {	"mov",		NULL,			{REG_B,			REG_A},				1},
	/* 49 */ {	"mov",		NULL,			{REG_B,			REG_B},				1},
	/* 4A */ {	"mov",		NULL,			{REG_B,			REG_L},				1},
	/* 4B */ {	"mov",		NULL,			{REG_B,			REG_H},				1},
	/* 4C */ {	"mov",		NULL,			{REG_B,			ADDR_N_I8},			2},
	/* 4D */ {	"mov",		NULL,			{REG_B,			ADDR_HL},			1},
	/* 4E */ {	"mov",		NULL,			{REG_B,			ADDR_X},			1},
	/* 4F */ {	"mov",		NULL,			{REG_B,			ADDR_Y},			1},
	/* 50 */ {	"mov",		NULL,			{REG_L,			REG_A},				1},
	/* 51 */ {	"mov",		NULL,			{REG_L,			REG_B},				1},
	/* 52 */ {	"mov",		NULL,			{REG_L,			REG_L},				1},
	/* 53 */ {	"mov",		NULL,			{REG_L,			REG_H},				1},
	/* 54 */ {	"mov",		NULL,			{REG_L,			ADDR_N_I8},			2},
	/* 55 */ {	"mov",		NULL,			{REG_L,			ADDR_HL},			1},
	/* 56 */ {	"mov",		NULL,			{REG_L,			ADDR_X},			1},
	/* 57 */ {	"mov",		NULL,			{REG_L,			ADDR_Y},			1},
	/* 58 */ {	"mov",		NULL,			{REG_H,			REG_A},				1},
	/* 59 */ {	"mov",		NULL,			{REG_H,			REG_B},				1},
	/* 5A */ {	"mov",		NULL,			{REG_H,			REG_L},				1},
	/* 5B */ {	"mov",		NULL,			{REG_H,			REG_H},				1},
	/* 5C */ {	"mov",		NULL,			{REG_H,			ADDR_N_I8},			2},
	/* 5D */ {	"mov",		NULL,			{REG_H,			ADDR_HL},			1},
	/* 5E */ {	"mov",		NULL,			{REG_H,			ADDR_X},			1},
	/* 5F */ {	"mov",		NULL,			{REG_H,			ADDR_Y},			1},
	/* 60 */ {	"mov",		NULL,			{ADDR_X,		REG_A},				1},
	/* 61 */ {	"mov",		NULL,			{ADDR_X,		REG_B},				1},
	/* 62 */ {	"mov",		NULL,			{ADDR_X,		REG_L},				1},
	/* 63 */ {	"mov",		NULL,			{ADDR_X,		REG_H},				1},
	/* 64 */ {	"mov",		NULL,			{ADDR_X,		ADDR_N_I8},			2},
	/* 65 */ {	"mov",		NULL,			{ADDR_X,		ADDR_HL},			1},
	/* 66 */ {	"mov",		NULL,			{ADDR_X,		ADDR_X},			1},
	/* 67 */ {	"mov",		NULL,			{ADDR_X,		ADDR_Y},			1},
	/* 68 */ {	"mov",		NULL,			{ADDR_HL,		REG_A},				1},
	/* 69 */ {	"mov",		NULL,			{ADDR_HL,		REG_B},				1},
	/* 6A */ {	"mov",		NULL,			{ADDR_HL,		REG_L},				1},
	/* 6B */ {	"mov",		NULL,			{ADDR_HL,		REG_H},				1},
	/* 6C */ {	"mov",		NULL,			{ADDR_HL,		ADDR_N_I8},			2},
	/* 6D */ {	"mov",		NULL,			{ADDR_HL,		ADDR_HL},			1},
	/* 6E */ {	"mov",		NULL,			{ADDR_HL,		ADDR_X},			1},
	/* 6F */ {	"mov",		NULL,			{ADDR_HL,		ADDR_Y},			1},
	/* 70 */ {	"mov",		NULL,			{ADDR_Y,		REG_A},				1},
	/* 71 */ {	"mov",		NULL,			{ADDR_Y,		REG_B},				1},
	/* 72 */ {	"mov",		NULL,			{ADDR_Y,		REG_L},				1},
	/* 73 */ {	"mov",		NULL,			{ADDR_Y,		REG_H},				1},
	/* 74 */ {	"mov",		NULL,			{ADDR_Y,		ADDR_N_I8},			2},
	/* 75 */ {	"mov",		NULL,			{ADDR_Y,		ADDR_HL},			1},
	/* 76 */ {	"mov",		NULL,			{ADDR_Y,		ADDR_X},			1},
	/* 77 */ {	"mov",		NULL,			{ADDR_Y,		ADDR_Y},			1},
	/* 78 */ {	"mov",		NULL,			{ADDR_N_I8,		REG_A},				1},
	/* 79 */ {	"mov",		NULL,			{ADDR_N_I8,		REG_B},				1},
	/* 7A */ {	"mov",		NULL,			{ADDR_N_I8,		REG_L},				1},
	/* 7B */ {	"mov",		NULL,			{ADDR_N_I8,		REG_H},				1},
	/* 7C */ {	"????",		NULL,			{ADDR_N_I8,		ADDR_N_I8},			2},
	/* 7D */ {	"mov",		NULL,			{ADDR_N_I8,		ADDR_HL},			1},
	/* 7E */ {	"mov",		NULL,			{ADDR_N_I8,		ADDR_X},			1},
	/* 7F */ {	"mov",		NULL,			{ADDR_N_I8,		ADDR_Y},			1},
	/* 80 */ {	"inc",		NULL,			{REG_A,			NO_IMMEDIATE},		1},
	/* 81 */ {	"inc",		NULL,			{REG_B,			NO_IMMEDIATE},		1},
	/* 82 */ {	"inc",		NULL,			{REG_L,			NO_IMMEDIATE},		1},
	/* 83 */ {	"inc",		NULL,			{REG_H,			NO_IMMEDIATE},		1},
	/* 84 */ {	"inc",		NULL,			{REG_N,			NO_IMMEDIATE},		1},
	/* 85 */ {	"inc",		NULL,			{ADDR_N_I8,		NO_IMMEDIATE},		2},
	/* 86 */ {	"inc",		NULL,			{ADDR_HL,		NO_IMMEDIATE},		1},
	/* 87 */ {	"inc",		NULL,			{REG_SP,		NO_IMMEDIATE},		1},
	/* 88 */ {	"dec",		NULL,			{REG_A,			NO_IMMEDIATE},		1},
	/* 89 */ {	"dec",		NULL,			{REG_B,			NO_IMMEDIATE},		1},
	/* 8A */ {	"dec",		NULL,			{REG_L,			NO_IMMEDIATE},		1},
	/* 8B */ {	"dec",		NULL,			{REG_H,			NO_IMMEDIATE},		1},
	/* 8C */ {	"dec",		NULL,			{REG_N,			NO_IMMEDIATE},		1},
	/* 8D */ {	"dec",		NULL,			{ADDR_N_I8,		NO_IMMEDIATE},		2},
	/* 8E */ {	"dec",		NULL,			{ADDR_HL,		NO_IMMEDIATE},		1},
	/* 8F */ {	"dec",		NULL,			{REG_SP,		NO_IMMEDIATE},		1},
	/* 90 */ {	"inc",		NULL,			{REG_BA,		NO_IMMEDIATE},		1},
	/* 91 */ {	"inc",		NULL,			{REG_HL,		NO_IMMEDIATE},		1},
	/* 92 */ {	"inc",		NULL,			{REG_X,			NO_IMMEDIATE},		1},
	/* 93 */ {	"inc",		NULL,			{REG_Y,			NO_IMMEDIATE},		1},
	/* 94 */ {	"test",		NULL,			{REG_A,			REG_B},				1},
	/* 95 */ {	"test",		"byte",			{ADDR_HL,		IMM_8},				2},
	/* 96 */ {	"test",		NULL,			{REG_A,			IMM_8},				2},
	/* 97 */ {	"test",		NULL,			{REG_B,			IMM_8},				2},
	/* 98 */ {	"dec",		NULL,			{REG_BA,		NO_IMMEDIATE},		1},
	/* 99 */ {	"dec",		NULL,			{REG_HL,		NO_IMMEDIATE},		1},
	/* 9A */ {	"dec",		NULL,			{REG_X,			NO_IMMEDIATE},		1},
	/* 9B */ {	"dec",		NULL,			{REG_Y,			NO_IMMEDIATE},		1},
	/* 9C */ {	"and",		NULL,			{REG_F,			IMM_8},				2},
	/* 9D */ {	"or",		NULL,			{REG_F,			IMM_8},				2},
	/* 9E */ {	"xor",		NULL,			{REG_F,			IMM_8},				2},
	/* 9F */ {	"mov",		NULL,			{REG_F,			IMM_8},				2},
	/* A0 */ {	"push",		NULL,			{REG_BA,		NO_IMMEDIATE},		1},
	/* A1 */ {	"push",		NULL,			{REG_HL,		NO_IMMEDIATE},		1},
	/* A2 */ {	"push",		NULL,			{REG_X,			NO_IMMEDIATE},		1},
	/* A3 */ {	"push",		NULL,			{REG_Y,			NO_IMMEDIATE},		1},
	/* A4 */ {	"push",		NULL,			{REG_N,			NO_IMMEDIATE},		1},
	/* A5 */ {	"push",		NULL,			{REG_I,			NO_IMMEDIATE},		1},
	/* A6 */ {	"pushx",	NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* A7 */ {	"push",		NULL,			{REG_F,			NO_IMMEDIATE},		1},
	/* A8 */ {	"pop",		NULL,			{REG_BA,		NO_IMMEDIATE},		1},
	/* A9 */ {	"pop",		NULL,			{REG_HL,		NO_IMMEDIATE},		1},
	/* AA */ {	"pop",		NULL,			{REG_X,			NO_IMMEDIATE},		1},
	/* AB */ {	"pop",		NULL,			{REG_Y,			NO_IMMEDIATE},		1},
	/* AC */ {	"pop",		NULL,			{REG_N,			NO_IMMEDIATE},		1},
	/* AD */ {	"pop",		NULL,			{REG_I,			NO_IMMEDIATE},		1},
	/* AE */ {	"popx",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* AF */ {	"pop",		NULL,			{REG_F,			NO_IMMEDIATE},		1},
	/* B0 */ {	"mov",		NULL,			{REG_A,			IMM_8},				1},
	/* B1 */ {	"mov",		NULL,			{REG_B,			IMM_8},				1},
	/* B2 */ {	"mov",		NULL,			{REG_L,			IMM_8},				1},
	/* B3 */ {	"mov",		NULL,			{REG_H,			IMM_8},				1},
	/* B4 */ {	"mov",		NULL,			{REG_N,			IMM_8},				1},
	/* B5 */ {	"mov",		NULL,			{ADDR_HL,		IMM_8},				1},
	/* B6 */ {	"mov",		NULL,			{ADDR_X,		IMM_8},				1},
	/* B7 */ {	"mov",		NULL,			{ADDR_Y,		IMM_8},				1},
	/* B8 */ {	"mov",		NULL,			{REG_BA,		ADDR_I_I16},		1},
	/* B9 */ {	"mov",		NULL,			{REG_HL,		ADDR_I_I16},		1},
	/* BA */ {	"mov",		NULL,			{REG_X,			ADDR_I_I16},		1},
	/* BB */ {	"mov",		NULL,			{REG_Y,			ADDR_I_I16},		1},
	/* BC */ {	"mov",		NULL,			{ADDR_I_I16,	REG_BA},			3},
	/* BD */ {	"mov",		NULL,			{ADDR_I_I16,	REG_HL},			3},
	/* BE */ {	"mov",		NULL,			{ADDR_I_I16,	REG_X},				3},
	/* BF */ {	"mov",		NULL,			{ADDR_I_I16,	REG_Y},				3},
	/* C0 */ {	"add",		NULL,			{REG_BA,		IMM_16},			3},
	/* C1 */ {	"add",		NULL,			{REG_HL,		IMM_16},			3},
	/* C2 */ {	"add",		NULL,			{REG_X,			IMM_16},			3},
	/* C3 */ {	"add",		NULL,			{REG_Y,			IMM_16},			3},
	/* C4 */ {	"mov",		NULL,			{REG_BA,		IMM_16},			3},
	/* C5 */ {	"mov",		NULL,			{REG_HL,		IMM_16},			3},
	/* C6 */ {	"mov",		NULL,			{REG_X,			IMM_16},			3},
	/* C7 */ {	"mov",		NULL,			{REG_Y,			IMM_16},			3},
	/* C8 */ {	"xchg",		NULL,			{REG_BA,		REG_HL},			1},
	/* C9 */ {	"xchg",		NULL,			{REG_BA,		REG_X},				1},
	/* CA */ {	"xchg",		NULL,			{REG_BA,		REG_Y},				1},
	/* CB */ {	"xchg",		NULL,			{REG_BA,		REG_SP},			1},
	/* CC */ {	"xchg",		NULL,			{REG_A,			REG_B},				1},
	/* CD */ {	"xchg",		NULL,			{REG_A,			ADDR_HL},			1},
	/* CE */ {	NULL,		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		0}, // Shift codes do not disassemble
	/* CF */ {	NULL,		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		0}, // Shift codes do not disassemble
	/* D0 */ {	"sub",		NULL,			{REG_BA,		IMM_16},			3},
	/* D1 */ {	"sub",		NULL,			{REG_HL,		IMM_16},			3},
	/* D2 */ {	"sub",		NULL,			{REG_X,			IMM_16},			3},
	/* D3 */ {	"sub",		NULL,			{REG_Y,			IMM_16},			3},
	/* D4 */ {	"cmp",		NULL,			{REG_BA,		IMM_16},			3},
	/* D5 */ {	"cmp",		NULL,			{REG_HL,		IMM_16},			3},
	/* D6 */ {	"cmp",		NULL,			{REG_X,			IMM_16},			3},
	/* D7 */ {	"cmp",		NULL,			{REG_Y,			IMM_16},			3},
	/* D8 */ {	"and",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* D9 */ {	"or",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* DA */ {	"xor",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* DB */ {	"cmp",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* DC */ {	"test",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* DD */ {	"mov",		"byte",			{ADDR_N_I8,		IMM_8},				3},
	/* DE */ {	"bcde",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* DF */ {	"bcdd",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* E0 */ {	"callc",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E1 */ {	"callnc",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E2 */ {	"callz",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E3 */ {	"callnz",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E4 */ {	"jc",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E5 */ {	"jnc",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E6 */ {	"jz",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E7 */ {	"jnz",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* E8 */ {	"callc",	"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* E9 */ {	"callnc",	"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* EA */ {	"callz",	"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* EB */ {	"callnz",	"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* EC */ {	"jc",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* ED */ {	"jnc",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* EE */ {	"jz",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* EF */ {	"jnz",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* F0 */ {	"call",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* F1 */ {	"jmp",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* F2 */ {	"call",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		3},
	/* F3 */ {	"jmp",		"long",			{ADDR_PC_I16,	NO_IMMEDIATE},		2},
	/* F4 */ {	"jmp",		NULL,			{REG_HL,		NO_IMMEDIATE},		1},
	/* F5 */ {	"jdbnz",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE},		2},
	/* F6 */ {	"bcdx",		NULL,			{REG_A,			NO_IMMEDIATE},		1},
	/* F7 */ {	"bcdx",		"byte",			{ADDR_HL,		NO_IMMEDIATE},		1},
	/* F8 */ {	"ret",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* F9 */ {	"reti",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* FA */ {	"retskip",	NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* FB */ {	"call",		"long",			{ADDR_I_I16,	NO_IMMEDIATE},		3},
	/* FC */ {	"int",		NULL,			{IMM_8,			NO_IMMEDIATE},		1},
	/* FD */ {	"jint",		NULL,			{IMM_8,			NO_IMMEDIATE},		1},
	/* FE */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
	/* FF */ {	"nop",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE},		1},
// ==== CE BLOCK ===================================================
	/* 00 */ {	"add",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 01 */ {	"add",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 02 */ {	"add",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 03 */ {	"add",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 04 */ {	"add",		NULL,			{ADDR_HL,		REG_A}},	
	/* 05 */ {	"add",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 06 */ {	"add",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 07 */ {	"add",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 08 */ {	"addc",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 09 */ {	"addc",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 0A */ {	"addc",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 0B */ {	"addc",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 0C */ {	"addc",		NULL,			{ADDR_HL,		REG_A}},	
	/* 0D */ {	"addc",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 0E */ {	"addc",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 0F */ {	"addc",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 10 */ {	"sub",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 11 */ {	"sub",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 12 */ {	"sub",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 13 */ {	"sub",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 14 */ {	"sub",		NULL,			{ADDR_HL,		REG_A}},	
	/* 15 */ {	"sub",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 16 */ {	"sub",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 17 */ {	"sub",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 18 */ {	"subc",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 19 */ {	"subc",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 1A */ {	"subc",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 1B */ {	"subc",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 1C */ {	"subc",		NULL,			{ADDR_HL,		REG_A}},	
	/* 1D */ {	"subc",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 1E */ {	"subc",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 1F */ {	"subc",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 20 */ {	"and",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 21 */ {	"and",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 22 */ {	"and",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 23 */ {	"and",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 24 */ {	"and",		NULL,			{ADDR_HL,		REG_A}},	
	/* 25 */ {	"and",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 26 */ {	"and",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 27 */ {	"and",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 28 */ {	"or",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 29 */ {	"or",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 2A */ {	"or",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 2B */ {	"or",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 2C */ {	"or",		NULL,			{ADDR_HL,		REG_A}},	
	/* 2D */ {	"or",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 2E */ {	"or",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 2F */ {	"or",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 30 */ {	"cmp",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 31 */ {	"cmp",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 32 */ {	"cmp",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 33 */ {	"cmp",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 34 */ {	"cmp",		NULL,			{ADDR_HL,		REG_A}},	
	/* 35 */ {	"cmp",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 36 */ {	"cmp",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 37 */ {	"cmp",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 38 */ {	"xor",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 39 */ {	"xor",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 3A */ {	"xor",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 3B */ {	"xor",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 3C */ {	"xor",		NULL,			{ADDR_HL,		REG_A}},	
	/* 3D */ {	"xor",		NULL,			{ADDR_HL,		IMM_8}},	
	/* 3E */ {	"xor",		NULL,			{ADDR_HL,		ADDR_X}},	
	/* 3F */ {	"xor",		NULL,			{ADDR_HL,		ADDR_Y}},	
	/* 40 */ {	"mov",		NULL,			{REG_A,			ADDR_X_I8}},	
	/* 41 */ {	"mov",		NULL,			{REG_A,			ADDR_Y_I8}},	
	/* 42 */ {	"mov",		NULL,			{REG_A,			ADDR_X_L}},	
	/* 43 */ {	"mov",		NULL,			{REG_A,			ADDR_Y_L}},	
	/* 44 */ {	"mov",		NULL,			{ADDR_X_I8,		REG_A}},	
	/* 45 */ {	"mov",		NULL,			{ADDR_Y_I8,		REG_A}},	
	/* 46 */ {	"mov",		NULL,			{ADDR_X,		REG_A}},	
	/* 47 */ {	"mov",		NULL,			{ADDR_Y,		REG_A}},	
	/* 48 */ {	"mov",		NULL,			{REG_B,			ADDR_X_I8}},	
	/* 49 */ {	"mov",		NULL,			{REG_B,			ADDR_Y_I8}},	
	/* 4A */ {	"mov",		NULL,			{REG_B,			ADDR_X_L}},	
	/* 4B */ {	"mov",		NULL,			{REG_B,			ADDR_Y_L}},	
	/* 4C */ {	"mov",		NULL,			{ADDR_X_I8,		REG_B}},	
	/* 4D */ {	"mov",		NULL,			{ADDR_Y_I8,		REG_B}},	
	/* 4E */ {	"mov",		NULL,			{ADDR_X,		REG_B}},	
	/* 4F */ {	"mov",		NULL,			{ADDR_Y,		REG_B}},	
	/* 50 */ {	"mov",		NULL,			{REG_L,			ADDR_X_I8}},	
	/* 51 */ {	"mov",		NULL,			{REG_L,			ADDR_Y_I8}},	
	/* 52 */ {	"mov",		NULL,			{REG_L,			ADDR_X_L}},	
	/* 53 */ {	"mov",		NULL,			{REG_L,			ADDR_Y_L}},	
	/* 54 */ {	"mov",		NULL,			{ADDR_X_I8,		REG_L}},	
	/* 55 */ {	"mov",		NULL,			{ADDR_Y_I8,		REG_L}},	
	/* 56 */ {	"mov",		NULL,			{ADDR_X,		REG_L}},	
	/* 57 */ {	"mov",		NULL,			{ADDR_Y,		REG_L}},	
	/* 58 */ {	"mov",		NULL,			{REG_H,			ADDR_X_I8}},	
	/* 59 */ {	"mov",		NULL,			{REG_H,			ADDR_Y_I8}},	
	/* 5A */ {	"mov",		NULL,			{REG_H,			ADDR_X_L}},	
	/* 5B */ {	"mov",		NULL,			{REG_H,			ADDR_Y_L}},	
	/* 5C */ {	"mov",		NULL,			{ADDR_X_I8,		REG_H}},	
	/* 5D */ {	"mov",		NULL,			{ADDR_Y_I8,		REG_H}},	
	/* 5E */ {	"mov",		NULL,			{ADDR_X,		REG_H}},	
	/* 5F */ {	"mov",		NULL,			{ADDR_Y,		REG_H}},	
	/* 60 */ {	"mov",		"byte",			{ADDR_HL,		ADDR_X_I8}},	
	/* 61 */ {	"mov",		"byte",			{ADDR_HL,		ADDR_Y_I8}},	
	/* 62 */ {	"mov",		"byte",			{ADDR_HL,		ADDR_X_L}},	
	/* 63 */ {	"mov",		"byte",			{ADDR_HL,		ADDR_Y_L}},	
	/* 64 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 65 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 66 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 67 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 68 */ {	"mov",		"byte",			{ADDR_X,		ADDR_X_I8}},	
	/* 69 */ {	"mov",		"byte",			{ADDR_X,		ADDR_Y_I8}},	
	/* 6A */ {	"mov",		"byte",			{ADDR_X,		ADDR_X_L}},	
	/* 6B */ {	"mov",		"byte",			{ADDR_X,		ADDR_Y_L}},	
	/* 6C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 70 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 71 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 72 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 73 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 74 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 75 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 76 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 77 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 78 */ {	"mov",		"byte",			{ADDR_Y,		ADDR_X_I8}},	
	/* 79 */ {	"mov",		"byte",			{ADDR_Y,		ADDR_Y_I8}},	
	/* 7A */ {	"mov",		"byte",			{ADDR_Y,		ADDR_X_L}},	
	/* 7B */ {	"mov",		"byte",			{ADDR_Y,		ADDR_Y_L}},	
	/* 7C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 80 */ {	"sal",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 81 */ {	"sal",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 82 */ {	"sal",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 83 */ {	"sal",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 84 */ {	"shl",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 85 */ {	"shl",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 86 */ {	"shl",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 87 */ {	"shl",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 88 */ {	"sar",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 89 */ {	"sar",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 8A */ {	"sar",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 8B */ {	"sar",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 8C */ {	"shr",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 8D */ {	"shr",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 8E */ {	"shr",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 8F */ {	"shr",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 90 */ {	"rolc",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 91 */ {	"rolc",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 92 */ {	"rolc",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 93 */ {	"rolc",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 94 */ {	"rol",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 95 */ {	"rol",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 96 */ {	"rol",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 97 */ {	"rol",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 98 */ {	"rorc",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 99 */ {	"rorc",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 9A */ {	"rorc",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 9B */ {	"rorc",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* 9C */ {	"ror",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* 9D */ {	"ror",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* 9E */ {	"ror",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* 9F */ {	"ror",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* A0 */ {	"not",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* A1 */ {	"not",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* A2 */ {	"not",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* A3 */ {	"not",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* A4 */ {	"neg",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* A5 */ {	"neg",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* A6 */ {	"neg",		"byte",			{ADDR_N_I8,		NO_IMMEDIATE}},	
	/* A7 */ {	"neg",		"byte",			{ADDR_HL,		NO_IMMEDIATE}},	
	/* A8 */ {	"ext",		NULL,			{REG_BA,		REG_A}},
	/* A9 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AA */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AC */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AD */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AE */ {	"halt",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AF */ {	"nop",		"word",			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* B0 */ {	"and",		NULL,			{REG_B,			IMM_8}},	
	/* B1 */ {	"and",		NULL,			{REG_L,			IMM_8}},	
	/* B2 */ {	"and",		NULL,			{REG_H,			IMM_8}},	
	/* B3 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* B4 */ {	"or",		NULL,			{REG_B,			IMM_8}},	
	/* B5 */ {	"or",		NULL,			{REG_L,			IMM_8}},	
	/* B6 */ {	"or",		NULL,			{REG_H,			IMM_8}},	
	/* B7 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* B8 */ {	"xor",		NULL,			{REG_B,			IMM_8}},	
	/* B9 */ {	"xor",		NULL,			{REG_L,			IMM_8}},	
	/* BA */ {	"xor",		NULL,			{REG_H,			IMM_8}},	
	/* BB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BC */ {	"cmp",		NULL,			{REG_B,			IMM_8}},	
	/* BD */ {	"cmp",		NULL,			{REG_L,			IMM_8}},	
	/* BE */ {	"cmp",		NULL,			{REG_H,			IMM_8}},	
	/* BF */ {	"cmp",		NULL,			{REG_N,			IMM_8}},
	/* C0 */ {	"mov",		NULL,			{REG_A,			REG_N}},	
	/* C1 */ {	"mov",		NULL,			{REG_A,			REG_F}},	
	/* C2 */ {	"mov",		NULL,			{REG_N,			REG_A}},	
	/* C3 */ {	"mov",		NULL,			{REG_F,			REG_A}},
	/* C4 */ {	"mov",		NULL,			{REG_U,			IMM_8}},	
	/* C5 */ {	"mov",		NULL,			{REG_I,			IMM_8}},	
	/* C6 */ {	"mov",		NULL,			{REG_XI,		IMM_8}},	
	/* C7 */ {	"mov",		NULL,			{REG_YI,		IMM_8}},	
	/* C8 */ {	"mov",		NULL,			{REG_A,			REG_V}},	
	/* C9 */ {	"mov",		NULL,			{REG_A,			REG_I}},	
	/* CA */ {	"mov",		NULL,			{REG_A,			REG_XI}},	
	/* CB */ {	"mov",		NULL,			{REG_A,			REG_YI}},	
	/* CC */ {	"mov",		NULL,			{REG_U,			REG_A}},	
	/* CD */ {	"mov",		NULL,			{REG_I,			REG_A}},	
	/* CE */ {	"mov",		NULL,			{REG_XI,		REG_A}},	
	/* CF */ {	"mov",		NULL,			{REG_YI,		REG_A}},	
	/* D0 */ {	"mov",		NULL,			{REG_A,			ADDR_I16}},	
	/* D1 */ {	"mov",		NULL,			{REG_B,			ADDR_I16}},	
	/* D2 */ {	"mov",		NULL,			{REG_L,			ADDR_I16}},	
	/* D3 */ {	"mov",		NULL,			{REG_H,			ADDR_I16}},	
	/* D4 */ {	"mov",		NULL,			{ADDR_I16,		REG_A}},	
	/* D5 */ {	"mov",		NULL,			{ADDR_I16,		REG_B}},	
	/* D6 */ {	"mov",		NULL,			{ADDR_I16,		REG_L}},	
	/* D7 */ {	"mov",		NULL,			{ADDR_I16,		REG_H}},	
	/* D8 */ {	"mul",		NULL,			{REG_HL,		REG_A}},	
	/* D9 */ {	"div",		NULL,			{REG_HL,		REG_A}},	
	/* DA */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* DB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* DC */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* DD */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* DE */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* DF */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* E0 */ {	"jl",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E1 */ {	"jle",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E2 */ {	"jg",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E3 */ {	"jge",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E4 */ {	"jo",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E5 */ {	"jno",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E6 */ {	"jp",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E7 */ {	"jn",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E8 */ {	"jnx0",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* E9 */ {	"jnx1",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* EA */ {	"jnx2",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* EB */ {	"jnx3",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* EC */ {	"jx0",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* ED */ {	"jx1",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* EE */ {	"jx2",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* EF */ {	"jx3",		"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F0 */ {	"calll",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F1 */ {	"callle",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F2 */ {	"callg",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F3 */ {	"callge",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F4 */ {	"callo",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F5 */ {	"callno",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F6 */ {	"callp",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F7 */ {	"calln",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F8 */ {	"callnx0",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* F9 */ {	"callnx1",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FA */ {	"callnx2",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FB */ {	"callnx3",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FC */ {	"callx0",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FD */ {	"callx1",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FE */ {	"callx2",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
	/* FF */ {	"callx3",	"short",		{ADDR_PC_I8,	NO_IMMEDIATE}},	
// ==== CF BLOCK ===================================================
	/* 00 */ {	"add",		NULL,			{REG_BA,		REG_BA}},	
	/* 01 */ {	"add",		NULL,			{REG_BA,		REG_HL}},	
	/* 02 */ {	"add",		NULL,			{REG_BA,		REG_X}},	
	/* 03 */ {	"add",		NULL,			{REG_BA,		REG_Y}},	
	/* 04 */ {	"addc",		NULL,			{REG_BA,		REG_BA}},	
	/* 05 */ {	"addc",		NULL,			{REG_BA,		REG_HL}},	
	/* 06 */ {	"addc",		NULL,			{REG_BA,		REG_X}},	
	/* 07 */ {	"addc",		NULL,			{REG_BA,		REG_Y}},	
	/* 08 */ {	"sub",		NULL,			{REG_BA,		REG_BA}},	
	/* 09 */ {	"sub",		NULL,			{REG_BA,		REG_HL}},	
	/* 0A */ {	"sub",		NULL,			{REG_BA,		REG_X}},	
	/* 0B */ {	"sub",		NULL,			{REG_BA,		REG_Y}},	
	/* 0C */ {	"subc",		NULL,			{REG_BA,		REG_BA}},	
	/* 0D */ {	"subc",		NULL,			{REG_BA,		REG_HL}},	
	/* 0E */ {	"subc",		NULL,			{REG_BA,		REG_X}},	
	/* 0F */ {	"subc",		NULL,			{REG_BA,		REG_Y}},	
	/* 10 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 11 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 12 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 13 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 14 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 15 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 16 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 17 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 18 */ {	"cmp",		NULL,			{REG_BA,		REG_BA}},	
	/* 19 */ {	"cmp",		NULL,			{REG_BA,		REG_HL}},	
	/* 1A */ {	"cmp",		NULL,			{REG_BA,		REG_X}},	
	/* 1B */ {	"cmp",		NULL,			{REG_BA,		REG_Y}},	
	/* 1C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 1D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 1E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 1F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 20 */ {	"add",		NULL,			{REG_HL,		REG_BA}},	
	/* 21 */ {	"add",		NULL,			{REG_HL,		REG_HL}},	
	/* 22 */ {	"add",		NULL,			{REG_HL,		REG_X}},	
	/* 23 */ {	"add",		NULL,			{REG_HL,		REG_Y}},	
	/* 24 */ {	"addc",		NULL,			{REG_HL,		REG_BA}},	
	/* 25 */ {	"addc",		NULL,			{REG_HL,		REG_HL}},	
	/* 26 */ {	"addc",		NULL,			{REG_HL,		REG_X}},	
	/* 27 */ {	"addc",		NULL,			{REG_HL,		REG_Y}},	
	/* 28 */ {	"sub",		NULL,			{REG_HL,		REG_BA}},	
	/* 29 */ {	"sub",		NULL,			{REG_HL,		REG_HL}},	
	/* 2A */ {	"sub",		NULL,			{REG_HL,		REG_X}},	
	/* 2B */ {	"sub",		NULL,			{REG_HL,		REG_Y}},	
	/* 2C */ {	"subc",		NULL,			{REG_HL,		REG_BA}},	
	/* 2D */ {	"subc",		NULL,			{REG_HL,		REG_HL}},	
	/* 2E */ {	"subc",		NULL,			{REG_HL,		REG_X}},	
	/* 2F */ {	"subc",		NULL,			{REG_HL,		REG_Y}},	
	/* 30 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 31 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 32 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 33 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 34 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 35 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 36 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 37 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 38 */ {	"cmp",		NULL,			{REG_HL,		REG_BA}},	
	/* 39 */ {	"cmp",		NULL,			{REG_HL,		REG_HL}},	
	/* 3A */ {	"cmp",		NULL,			{REG_HL,		REG_X}},	
	/* 3B */ {	"cmp",		NULL,			{REG_HL,		REG_Y}},	
	/* 3C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 3D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 3E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 3F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 40 */ {	"add",		NULL,			{REG_X,			REG_BA}},	
	/* 41 */ {	"add",		NULL,			{REG_X,			REG_HL}},	
	/* 42 */ {	"add",		NULL,			{REG_Y,			REG_BA}},	
	/* 43 */ {	"add",		NULL,			{REG_Y,			REG_HL}},	
	/* 44 */ {	"add",		NULL,			{REG_SP,		REG_BA}},	
	/* 45 */ {	"add",		NULL,			{REG_SP,		REG_HL}},	
	/* 46 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 47 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 48 */ {	"sub",		NULL,			{REG_X,			REG_BA}},	
	/* 49 */ {	"sub",		NULL,			{REG_X,			REG_HL}},	
	/* 4A */ {	"sub",		NULL,			{REG_Y,			REG_BA}},	
	/* 4B */ {	"sub",		NULL,			{REG_Y,			REG_HL}},	
	/* 4C */ {	"sub",		NULL,			{REG_SP,		REG_BA}},	
	/* 4D */ {	"sub",		NULL,			{REG_SP,		REG_HL}},	
	/* 4E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 4F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 50 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 51 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 52 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 53 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 54 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 55 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 56 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 57 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 58 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 59 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 5A */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 5B */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 5C */ {	"cmp",		NULL,			{REG_SP,		REG_BA}},	
	/* 5D */ {	"cmp",		NULL,			{REG_SP,		REG_HL}},	
	/* 5E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 5F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 60 */ {	"cmpn",		NULL,			{REG_BA,		IMM_16}},	
	/* 61 */ {	"cmpn",		NULL,			{REG_HL,		IMM_16}},	
	/* 62 */ {	"cmpn",		NULL,			{REG_X,			IMM_16}},	
	/* 63 */ {	"cmpn",		NULL,			{REG_Y,			IMM_16}},	
	/* 64 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 65 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 66 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 67 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 68 */ {	"add",		NULL,			{REG_SP,		IMM_16}},	
	/* 69 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6A */ {	"sub",		NULL,			{REG_SP,		IMM_16}},	
	/* 6B */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6C */ {	"cmp",		NULL,			{REG_SP,		IMM_16}},	
	/* 6D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 6E */ {	"mov",		NULL,			{REG_SP,		IMM_16}},	
	/* 6F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 70 */ {	"mov",		NULL,			{REG_BA,		ADDR_SP_I8}},	
	/* 71 */ {	"mov",		NULL,			{REG_HL,		ADDR_SP_I8}},	
	/* 72 */ {	"mov",		NULL,			{REG_X,			ADDR_SP_I8}},	
	/* 73 */ {	"mov",		NULL,			{REG_Y,			ADDR_SP_I8}},	
	/* 74 */ {	"mov",		NULL,			{ADDR_SP_I8,	REG_BA}},	
	/* 75 */ {	"mov",		NULL,			{ADDR_SP_I8,	REG_HL}},	
	/* 76 */ {	"mov",		NULL,			{ADDR_SP_I8,	REG_X}},	
	/* 77 */ {	"mov",		NULL,			{ADDR_SP_I8,	REG_Y}},	
	/* 78 */ {	"mov",		NULL,			{REG_SP,		ADDR_I_I16}},	
	/* 79 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7A */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7B */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7C */ {	"mov",		NULL,			{ADDR_I_I16,	REG_SP}},	
	/* 7D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 7F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 80 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 81 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 82 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 83 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 84 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 85 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 86 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 87 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 88 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 89 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8A */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8B */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 8F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 90 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 91 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 92 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 93 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 94 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 95 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 96 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 97 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 98 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 99 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9A */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9B */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9C */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9D */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9E */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* 9F */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A0 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A1 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A2 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A3 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A4 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A5 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A6 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A7 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A8 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* A9 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AA */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AC */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AD */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AE */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* AF */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* B0 */ {	"push",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* B1 */ {	"push",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* B2 */ {	"push",		NULL,			{REG_L,			NO_IMMEDIATE}},	
	/* B3 */ {	"push",		NULL,			{REG_H,			NO_IMMEDIATE}},	
	/* B4 */ {	"pop",		NULL,			{REG_A,			NO_IMMEDIATE}},	
	/* B5 */ {	"pop",		NULL,			{REG_B,			NO_IMMEDIATE}},	
	/* B6 */ {	"pop",		NULL,			{REG_L,			NO_IMMEDIATE}},	
	/* B7 */ {	"pop",		NULL,			{REG_H,			NO_IMMEDIATE}},	
	/* B8 */ {	"pusha",	NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* B9 */ {	"pushax",	NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BA */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BC */ {	"popa",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BD */ {	"popax",	NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BE */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* BF */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* C0 */ {	"mov",		NULL,			{REG_BA,		ADDR_HL}},	
	/* C1 */ {	"mov",		NULL,			{REG_HL,		ADDR_HL}},	
	/* C2 */ {	"mov",		NULL,			{REG_X,			ADDR_HL}},	
	/* C3 */ {	"mov",		NULL,			{REG_Y,			ADDR_HL}},	
	/* C4 */ {	"mov",		NULL,			{ADDR_HL,		REG_BA}},	
	/* C5 */ {	"mov",		NULL,			{ADDR_HL,		REG_HL}},	
	/* C6 */ {	"mov",		NULL,			{ADDR_HL,		REG_X}},	
	/* C7 */ {	"mov",		NULL,			{ADDR_HL,		REG_Y}},	
	/* C8 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* C9 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CA */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CC */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CD */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CE */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* CF */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* D0 */ {	"mov",		NULL,			{REG_BA,		ADDR_X}},	
	/* D1 */ {	"mov",		NULL,			{REG_HL,		ADDR_X}},	
	/* D2 */ {	"mov",		NULL,			{REG_X,			ADDR_X}},	
	/* D3 */ {	"mov",		NULL,			{REG_Y,			ADDR_X}},	
	/* D4 */ {	"mov",		NULL,			{ADDR_X,		REG_BA}},	
	/* D5 */ {	"mov",		NULL,			{ADDR_X,		REG_HL}},	
	/* D6 */ {	"mov",		NULL,			{ADDR_X,		REG_X}},	
	/* D7 */ {	"mov",		NULL,			{ADDR_X,		REG_Y}},	
	/* D8 */ {	"mov",		NULL,			{REG_BA,		ADDR_Y}},	
	/* D9 */ {	"mov",		NULL,			{REG_HL,		ADDR_Y}},	
	/* DA */ {	"mov",		NULL,			{REG_X,			ADDR_Y}},	
	/* DB */ {	"mov",		NULL,			{REG_Y,			ADDR_Y}},	
	/* DC */ {	"mov",		NULL,			{ADDR_Y,		REG_BA}},	
	/* DD */ {	"mov",		NULL,			{ADDR_Y,		REG_HL}},	
	/* DE */ {	"mov",		NULL,			{ADDR_Y,		REG_X}},	
	/* DF */ {	"mov",		NULL,			{ADDR_Y,		REG_Y}},	
	/* E0 */ {	"mov",		NULL,			{REG_BA,		REG_BA}},	
	/* E1 */ {	"mov",		NULL,			{REG_BA,		REG_HL}},	
	/* E2 */ {	"mov",		NULL,			{REG_BA,		REG_X}},	
	/* E3 */ {	"mov",		NULL,			{REG_BA,		REG_Y}},	
	/* E4 */ {	"mov",		NULL,			{REG_HL,		REG_BA}},	
	/* E5 */ {	"mov",		NULL,			{REG_HL,		REG_HL}},	
	/* E6 */ {	"mov",		NULL,			{REG_HL,		REG_X}},	
	/* E7 */ {	"mov",		NULL,			{REG_HL,		REG_Y}},	
	/* E8 */ {	"mov",		NULL,			{REG_X,			REG_BA}},	
	/* E9 */ {	"mov",		NULL,			{REG_X,			REG_HL}},	
	/* EA */ {	"mov",		NULL,			{REG_X,			REG_X}},	
	/* EB */ {	"mov",		NULL,			{REG_X,			REG_Y}},	
	/* EC */ {	"mov",		NULL,			{REG_Y,			REG_BA}},	
	/* ED */ {	"mov",		NULL,			{REG_Y,			REG_HL}},	
	/* EE */ {	"mov",		NULL,			{REG_Y,			REG_X}},	
	/* EF */ {	"mov",		NULL,			{REG_Y,			REG_Y}},	
	/* F0 */ {	"mov",		NULL,			{REG_SP,		REG_BA}},	
	/* F1 */ {	"mov",		NULL,			{REG_SP,		REG_HL}},	
	/* F2 */ {	"mov",		NULL,			{REG_SP,		REG_X}},	
	/* F3 */ {	"mov",		NULL,			{REG_SP,		REG_Y}},	
	/* F4 */ {	"mov",		NULL,			{REG_HL,		REG_SP}},	
	/* F5 */ {	"mov",		NULL,			{REG_HL,		REG_PC}},	
	/* F6 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* F7 */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* F8 */ {	"mov",		NULL,			{REG_BA,		REG_SP}},	
	/* F9 */ {	"mov",		NULL,			{REG_BA,		REG_PC}},	
	/* FA */ {	"mov",		NULL,			{REG_X,			REG_SP}},	
	/* FB */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* FC */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* FD */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
	/* FE */ {	"mov",		NULL,			{REG_Y,			REG_SP}},	
	/* FF */ {	"????",		NULL,			{NO_IMMEDIATE,	NO_IMMEDIATE}},	
};

#else
extern Operation Mnemonic[0x300];
#endif

LRESULT CALLBACK AssemblyViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );
LRESULT CALLBACK MemoryViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );
LRESULT CALLBACK CodePointViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK MemoryProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK OpcodeProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
char *disassemble( Mini *mini, long address, char *buffer );

#endif