/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Minimon.h"

Configuration::Configuration()
{
	m_Cfg = CreateFile( 
		"minimon.cfg", 
		GENERIC_WRITE | GENERIC_READ,
		FILE_SHARE_READ,
		NULL,
		OPEN_ALWAYS,
		FILE_ATTRIBUTE_NORMAL,
		NULL );

	m_Root = NULL;

	if( m_Cfg == INVALID_HANDLE_VALUE )
	{
		MessageBox( NULL, "Unable to open a configuration file\n\rYou will not be able to preserve your settings", g_AppTitle, MB_OK );
		return ;
	}

	ReadConfig();
}

Configuration::~Configuration()
{	
	if( m_Cfg != INVALID_HANDLE_VALUE )
	{
		WriteConfig();
		CloseHandle( m_Cfg );
	}

	DeleteNode( m_Root );
}

void Configuration::DeleteNode( XMLNode *node )
{
	if( node == NULL )
	{
		return ;
	}
	
	DeleteNode( node->m_Next );
	DeleteNode( node->m_Child );

	if( node->m_Name )
	{
		delete node->m_Name;
	}


	if( node->m_Prop != NULL )
	{
		for( int i = 0; i < node->m_Properties; i++ )
		{
			if( node->m_Prop[i].m_Name != NULL )
			{
				delete node->m_Prop[i].m_Name;
			}

			if( node->m_Prop[i].m_Text != NULL )
			{
				delete node->m_Prop[i].m_Text;
			}
		}

		delete node->m_Prop;
	}

	delete node;
}

void Configuration::Preformat( char *proto )
{
	char *seek;

	seek = proto;
	// Convert whitespace for speed
	while( *seek )
	{
		switch( *seek )
		{
		case '\n':
		case '\r':
		case '\t':
			*seek = ' ';

		}
		seek++;
	}

	seek = proto;

	bool leading = false;

	while( *seek )
	{
		if( *seek == ' ' )
		{			
			while( *seek && *seek == ' ' )
				seek++;

			if( *seek == '=' )
			{
				leading = false;
			}
			else if( leading )
			{
				proto++;
			}
			else
			{
				leading = true;
			}

			if( seek != proto )
			{
				while( *seek != ' ' && *seek )
				{
					*(proto++) = *(seek);	
					*(seek++) = ' ';
				}
			}
		}
		else
		{
			proto++;
			seek++;
			leading = true;
		}
	}

	*proto = 0;
}

bool Configuration::CreateNode( XMLNode **node, char *proto )
{	
	XMLNode *xml = *node = new XMLNode;
	ZeroMemory( xml, sizeof(XMLNode) );

	bool terminate = false;

	Preformat( proto );	

	char *seek = proto;

	while( *seek )
	{
		while( *seek && *seek != ' ' )
		{
			if( *seek == '"' )
			{
				seek++; while( *seek && *seek != '"' ) seek ++;
			}
			else if( *seek == '\'' )
			{
				seek++; while( *seek && *seek != '\'' ) seek ++;
			}
			seek++;
		}

		xml->m_Properties++;

		while( *seek && *seek == ' ' )
			seek++;
	}
	
	seek = proto;

	while( *seek && *seek != ' ' )
		seek++;

	*seek = 0;

	if( strlen( proto ) > 0 )
	{
		if( *proto == '/' )
		{
			terminate = true;
			proto++;
		}

		xml->m_Name = new char[ seek - proto + 1 ];
		strcpy( xml->m_Name, proto );
		proto = seek+1;
	}

	xml->m_Properties--;

	if( xml->m_Properties > 0 )
	{
		xml->m_Prop = new Property[xml->m_Properties];
		ZeroMemory( xml->m_Prop, sizeof( Property )*xml->m_Properties );

		for( int i = 0; i < xml->m_Properties; i++ )
		{
			while( *proto && *proto == ' ' )
				proto++;
			seek = proto;
			while( *seek && *seek != ' ' )
			{
				if( *seek == '"' )
				{
					seek++; while( *seek && *seek != '"' ) seek ++;
				}
				else if( *seek == '\'' )
				{
					seek++; while( *seek && *seek != '\'' ) seek ++;
				}
				seek++;
			}
			*seek = 0;

			char *data = proto;

			while( *data && *data != '=' )
				data++;

			if( *data == '=' )
			{
				*(data++) = 0;

				switch( *data )
				{
				case 0:
					break ;
				case '#':
					xml->m_Prop[i].m_Format = FORMAT_HEX;
					xml->m_Prop[i].m_Value = strtol( data+1, NULL, 16 );
					break ;
				case '0':case '1':case '2':case '3':case '4':
				case '5':case '6':case '7':case '8':case '9':
					xml->m_Prop[i].m_Format = FORMAT_DECIMAL;
					xml->m_Prop[i].m_Value = strtol( data, NULL, 10 );
					break ;
				case '\'':
					{
						char *locate = data+1;

						while( *locate && *locate != '\'' )
							locate++;
						*locate = 0;

						xml->m_Prop[i].m_Format = FORMAT_SINGLEQUOTE;
						xml->m_Prop[i].m_Text = new char[ locate - data ];
						strcpy( xml->m_Prop[i].m_Text, data+1 );
					}
					break ;
				case '"':
					{
						char *locate = data+1;

						while( *locate && *locate != '"' )
							locate++;
						*locate = 0;

						xml->m_Prop[i].m_Format = FORMAT_DOUBLEQUOTE;
						xml->m_Prop[i].m_Text = new char[ locate - data ];
						strcpy( xml->m_Prop[i].m_Text, data+1 );
					}
					break ;
				default:
					xml->m_Prop[i].m_Format = FORMAT_NOQUOTE;
					xml->m_Prop[i].m_Text = new char[ strlen(data)+1 ];
					strcpy( xml->m_Prop[i].m_Text, data );
				}
			}
			else
			{
				*data = 0;
			}

			xml->m_Prop[i].m_Name = new char[data-proto+1];
			strcpy( xml->m_Prop[i].m_Name, proto );

			proto = seek+1;
		}
	}

	return terminate;
}

void Configuration::ReadConfig()
{
	long size;
	
	size = GetFileSize( m_Cfg, NULL );
	SetFilePointer( m_Cfg, 0, NULL, FILE_BEGIN );

	char *cfgBuffer = new char[size+2];
	char *token;
	DWORD bytesRead;

	cfgBuffer[0] = ' ';
	cfgBuffer[size+1] = 0;

	ReadFile( m_Cfg, cfgBuffer+1, size, &bytesRead, NULL );

	token = strtok( cfgBuffer, "<" );

	if( token == NULL )
	{
		return ;
	}

	XMLNode **xml, *cursor = NULL, *parent = NULL;

	xml = &m_Root;	

	while( token = strtok( NULL, "<" ) )
	{
		char *seek = token;
		bool terminate;

		while( *seek && *seek != '>' )
			seek++;

		if( *seek != '>' )
			continue;

		*seek = 0;
		
		terminate = CreateNode(xml, token);

		(*xml)->m_Parent = parent;

		if( cursor == NULL )
		{
			cursor = *xml;
		}

		XMLNode *newcursor = *xml;

		if( terminate )
		{			
			if( cursor->m_Parent != NULL && strcmp(cursor->m_Parent->m_Name, (*xml)->m_Name) == 0 )
			{
				cursor = cursor->m_Parent;
				parent = cursor->m_Parent;
				DeleteNode( *xml );
				*xml = NULL;
			}
			else
			{
				cursor = newcursor;
			}

			xml = &cursor->m_Next;
		}
		else
		{
			cursor = newcursor;
			parent = cursor;
			xml = &cursor->m_Child;
		}
	}

	delete cfgBuffer;
}

void Configuration::DumpNode( HANDLE fo, XMLNode *node, int tabs )
{
	char buffer[1024], *data = buffer;
	DWORD readBytes;

	if( node == NULL )
		return ;

	for( int i = 0; i < tabs; i++ )
		*(data++) = '\t';

	if( node->m_Child == NULL )
	{
		sprintf( data, "</%s", node->m_Name );
	}
	else
	{		
		sprintf( data, "<%s", node->m_Name );
	}

	while(*(++data)) ;

	for( int i = 0; i < node->m_Properties; i++ )
	{
		switch( node->m_Prop[i].m_Format )
		{
		case FORMAT_HEX:
			{
				char buffer[256];
				sprintf(data, " %s=#%s", node->m_Prop[i].m_Name, itoa(node->m_Prop[i].m_Value,buffer,16) );
			}
			break ;
		case FORMAT_DECIMAL:
			sprintf(data, " %s=%li", node->m_Prop[i].m_Name, node->m_Prop[i].m_Value );
			break ;
		case FORMAT_NOQUOTE:
			sprintf(data, " %s=%s", node->m_Prop[i].m_Name, node->m_Prop[i].m_Text );
			break ;
		case FORMAT_SINGLEQUOTE:
			sprintf(data, " %s='%s'", node->m_Prop[i].m_Name, node->m_Prop[i].m_Text );
			break ;
		case FORMAT_DOUBLEQUOTE:
			sprintf(data, " %s=\"%s\"", node->m_Prop[i].m_Name, node->m_Prop[i].m_Text );
			break ;
		}

		while(*(++data)) ;
	}

	sprintf( data, ">\n" );

	WriteFile( m_Cfg, buffer, strlen(buffer), &readBytes, NULL );

	if( node->m_Child )
	{
		DumpNode( fo, node->m_Child, tabs+1 );
		for(data=buffer,i=0;i<tabs;i++,data++) *data = '\t';
		sprintf(data,"</%s>\n", node->m_Name);
		WriteFile( m_Cfg, buffer, strlen(buffer), &readBytes, NULL );
	}

	DumpNode( fo, node->m_Next, tabs );

}

int Configuration::Read( char *var, int &val, char *text )
{
	if( m_SeekNode == NULL )
	{
		return NOT_FOUND;
	}

	for( int i = 0; i < m_SeekNode->m_Properties; i++ )
	{
		if( strcmp(m_SeekNode->m_Prop[i].m_Name, var) != 0 )
			continue ;

		switch( m_SeekNode->m_Prop[i].m_Format )
		{
		case FORMAT_HEX:
		case FORMAT_DECIMAL:
			val = m_SeekNode->m_Prop[i].m_Value;
			return VALUE_NUMBER;
		case FORMAT_NOQUOTE:
		case FORMAT_SINGLEQUOTE:
		case FORMAT_DOUBLEQUOTE:
			if( text == NULL )
				return NO_VARIABLE;

			strcpy( text, m_SeekNode->m_Prop[i].m_Text );
			return VALUE_TEXT;
		}
	}

	return NO_VARIABLE;
}

int Configuration::Write( char *var, int val )
{
	if( m_SeekNode == NULL )
	{
		return NOT_FOUND;
	}

	for( int i = 0; i < m_SeekNode->m_Properties; i++ )
	{
		if( strcmp(m_SeekNode->m_Prop[i].m_Name, var) != 0 )
			continue ;

		switch( m_SeekNode->m_Prop[i].m_Format )
		{
		case FORMAT_NOQUOTE:
		case FORMAT_SINGLEQUOTE:
		case FORMAT_DOUBLEQUOTE:
			delete m_SeekNode->m_Prop[i].m_Text;
			m_SeekNode->m_Prop[i].m_Format = FORMAT_DECIMAL;
		case FORMAT_HEX:
		case FORMAT_DECIMAL:
			m_SeekNode->m_Prop[i].m_Value = val;
			return VALUE_NUMBER;
		}
	}

	Property *prop = new Property[ m_SeekNode->m_Properties+1 ];
	memcpy( prop, m_SeekNode->m_Prop, sizeof(Property) * m_SeekNode->m_Properties );
	delete m_SeekNode->m_Prop;
	m_SeekNode->m_Prop = prop;
	
	prop = &m_SeekNode->m_Prop[ m_SeekNode->m_Properties++ ];
	prop->m_Format = FORMAT_DECIMAL;
	prop->m_Name = new char[strlen(var)+1];
	strcpy( prop->m_Name, var );
	prop->m_Value = val;
	prop->m_Text = NULL;

	return NO_VARIABLE;
}

int Configuration::Write( char *var, char *text )
{
	if( m_SeekNode == NULL )
	{
		return NOT_FOUND;
	}

	for( int i = 0; i < m_SeekNode->m_Properties; i++ )
	{
		if( strcmp(m_SeekNode->m_Prop[i].m_Name, var) != 0 )
			continue ;

		switch( m_SeekNode->m_Prop[i].m_Format )
		{
		case FORMAT_HEX:
		case FORMAT_DECIMAL:
			m_SeekNode->m_Prop[i].m_Format = FORMAT_DOUBLEQUOTE;
			m_SeekNode->m_Prop[i].m_Text = new char[strlen(text)+1];
			strcpy( m_SeekNode->m_Prop[i].m_Text, text );
			return VALUE_TEXT;
		case FORMAT_NOQUOTE:
		case FORMAT_SINGLEQUOTE:
		case FORMAT_DOUBLEQUOTE:
			delete m_SeekNode->m_Prop[i].m_Text;
			m_SeekNode->m_Prop[i].m_Text = new char[strlen(text)+1];
			strcpy( m_SeekNode->m_Prop[i].m_Text, text );
			return VALUE_TEXT;
		}
	}

	Property *prop = new Property[ m_SeekNode->m_Properties+1 ];
	memcpy( prop, m_SeekNode->m_Prop, sizeof(Property) * m_SeekNode->m_Properties );
	delete m_SeekNode->m_Prop;
	m_SeekNode->m_Prop = prop;
	
	prop = &m_SeekNode->m_Prop[ m_SeekNode->m_Properties++ ];
	prop->m_Format = FORMAT_DOUBLEQUOTE;
	prop->m_Name = new char[strlen(var)+1];
	strcpy( prop->m_Name, var );
	prop->m_Text = new char[strlen(text)+1];
	strcpy( prop->m_Text, text );

	return NO_VARIABLE;
}

bool Configuration::Create( char *address )
{
	if( address == NULL )
		return false;

	char *parsed = new char[strlen(address)+1];
	strcpy(parsed, address);

	m_SeekName = strtok( parsed, "." );
	m_SeekNode = m_Root;

	while( m_SeekNode != NULL )
	{
		XMLNode *seek = m_SeekNode;

		while( seek != NULL && strcmp( seek->m_Name, m_SeekName ) != 0 )
			seek = seek->m_Next;

		if( seek == NULL )
		{
			CreateNode( &m_SeekNode->m_Parent->m_Child, m_SeekName );
			m_SeekNode->m_Parent->m_Child->m_Next = m_SeekNode;
			m_SeekNode->m_Parent->m_Child->m_Parent = m_SeekNode->m_Parent;
			m_SeekNode = m_SeekNode->m_Parent->m_Child;			

			m_SeekName = strtok( NULL, "." );
			while( m_SeekName != NULL )
			{
				CreateNode( &m_SeekNode->m_Child, m_SeekName );
				m_SeekNode->m_Child->m_Parent = m_SeekNode;
				m_SeekNode = m_SeekNode->m_Child;

				m_SeekName = strtok( NULL, "." );
			}

			delete parsed;
			return true;
		}

		m_SeekNode = seek;

		m_SeekName = strtok( NULL, "." );

		if( m_SeekName == NULL )
		{
			m_SeekName = m_SeekNode->m_Name;

			while( m_SeekNode->m_Next != NULL )
				m_SeekNode = m_SeekNode->m_Next;

			CreateNode( &m_SeekNode->m_Next, m_SeekName );
			m_SeekNode = m_SeekNode->m_Next;

			delete parsed;
			return true;
		}
		else
		{
			if( m_SeekNode->m_Child == NULL )
			{
				while( m_SeekName != NULL )
				{
					CreateNode( &m_SeekNode->m_Child, m_SeekName );
					m_SeekNode->m_Child->m_Parent = m_SeekNode;
					m_SeekNode = m_SeekNode->m_Child;

					m_SeekName = strtok( NULL, "." );
				}

				delete parsed;
				return true;
			}

			m_SeekNode = m_SeekNode->m_Child;
		}
	}

	return false;
}

bool Configuration::Seek( char *address )
{
	if( address != NULL )
	{
		char *parsed = new char[strlen(address)+1];
		strcpy(parsed, address);

		m_SeekName = strtok( parsed, "." );
		m_SeekNode = m_Root;

		while( m_SeekName != NULL )
		{
			while( m_SeekNode != NULL && strcmp( m_SeekNode->m_Name, m_SeekName ) != 0 )
				m_SeekNode = m_SeekNode->m_Next;

			if( m_SeekNode == NULL )
			{
				delete parsed;
				return false;
			}

			m_SeekName = strtok( NULL, "." );

			if( m_SeekName != NULL )
			{
				m_SeekNode = m_SeekNode->m_Child;
			}
		}

		m_SeekName = m_SeekNode->m_Name;
		delete parsed;
	}
	else
	{
		if( m_SeekNode != NULL )
		{
			m_SeekNode = m_SeekNode->m_Next;
		}

		while( m_SeekNode != NULL && strcmp( m_SeekNode->m_Name, m_SeekName ) != 0 )
			m_SeekNode = m_SeekNode->m_Next;
	}

	if( m_SeekNode == NULL )
	{
		return false;
	}

	return true;
}

void Configuration::WriteConfig()
{
	SetFilePointer( m_Cfg, 0, NULL, FILE_BEGIN );
	DumpNode( m_Cfg, m_Root );
	SetEndOfFile( m_Cfg );

	// TODO: WRITE XML NODES TO FILE HERE
}
