/* libspectrum.h: the library for dealing with ZX Spectrum emulator files
   Copyright (c) 2001-2007 Philip Kendall, Darren Salt, Fredrick Meunier

   $Id: libspectrum.h.in,v 1.141.2.2 2007/05/09 20:08:07 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: philip-fuse@shadowmagic.org.uk

*/

/* NB: This file is autogenerated from libspectrum.h.in. Do not edit
   unless you know what you're doing */

#ifndef LIBSPECTRUM_LIBSPECTRUM_H
#define LIBSPECTRUM_LIBSPECTRUM_H

#ifdef __cplusplus
extern "C" {
#endif				/* #ifdef __cplusplus */

#include <stdarg.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef _WIN322

#define WIN32_LEAN_AND_MEAN	/* Exclude rarely used stuff from Windows
				   headers */
#include <windows.h>

#ifdef LIBSPECTRUM_EXPORTS
#define WIN32_DLL __declspec( dllexport )
#else				/* #ifdef LIBSPECTRUM_EXPORTS */
#define WIN32_DLL __declspec( dllimport )
#endif				/* #ifdef LIBSPECTRUM_EXPORTS */

#else				/* #ifdef _WIN32 */

#define WIN32_DLL

#endif				/* #ifdef _WIN32 */

#ifdef __GNUC__
#define DEPRECATED __attribute__((deprecated))
#else                           /* #ifdef __GNUC__ */
#define DEPRECATED 
#endif                          /* #ifdef __GNUC__ */

//LIBSPECTRUM_INCLUDE_GCRYPT

/* Standard typedefs */
//LIBSPECTRUM_DEFINE_TYPES

typedef unsigned char uint8_t ;
typedef char int8_t ;
typedef unsigned short uint16_t ;
typedef short int16_t ;
typedef unsigned int uint32_t ;
typedef int int32_t ;
typedef unsigned __int64 uint64_t ;
typedef __int64 int64_t ;


typedef int gint;
typedef gint gboolean;
typedef unsigned int guint;
typedef const void * gconstpointer;
typedef void * gpointer;

typedef struct _GSList GSList;
#ifndef FALSE
#define FALSE (0)
#endif
#ifndef TRUE
#define TRUE (1)
#endif

struct _GSList {
    gpointer data;
    GSList *next;
};

typedef void		(*GFunc)		(gpointer	data,
						 gpointer	user_data);

typedef gint		(*GCompareFunc)		(gconstpointer	a,
						 gconstpointer	b);


GSList* g_slist_insert_sorted	(GSList		*list,
				 gpointer	 data,
				 GCompareFunc	 func);

GSList* g_slist_append		(GSList		*list,
				 gpointer	 data);

GSList* g_slist_prepend	(GSList		*list,
				 gpointer	 data);

GSList* g_slist_remove		(GSList		*list,
				 gpointer	 data);

void	g_slist_foreach		(GSList		*list,
				 GFunc		 func,
				 gpointer	 user_data);

void	g_slist_free		(GSList		*list);

GSList* g_slist_nth		(GSList		*list,
				 guint		n);

GSList* g_slist_find_custom	(GSList		*list,
				 gpointer	data,
				 GCompareFunc	func );

gint	g_slist_position	(GSList		*list,
				 GSList		*llink);

typedef struct _GHashTable	GHashTable;

typedef guint		(*GHashFunc)		(gconstpointer	key);

typedef gboolean	(*GHRFunc)		(gpointer	key,
						 gpointer	value,
						 gpointer	user_data);

gint  g_int_equal (gconstpointer   v,
		   gconstpointer   v2);
guint g_int_hash  (gconstpointer   v);

GHashTable* g_hash_table_new		(GHashFunc	 hash_func,
					 GCompareFunc	 key_compare_func);

void	    g_hash_table_destroy	(GHashTable	*hash_table);

void	    g_hash_table_insert		(GHashTable	*hash_table,
					 gpointer	 key,
					 gpointer	 value);

gpointer    g_hash_table_lookup		(GHashTable	*hash_table,
					 gconstpointer	 key);

guint	    g_hash_table_foreach_remove	(GHashTable	*hash_table,
					 GHRFunc	 func,
					 gpointer	 user_data);

#define GINT_TO_POINTER(i)      ((gpointer)  (i))
#define GPOINTER_TO_INT(p)      ((gint)   (p))

typedef  uint8_t libspectrum_byte;
typedef   int8_t libspectrum_signed_byte;
typedef uint16_t libspectrum_word;
typedef  int16_t libspectrum_signed_word;
typedef uint32_t libspectrum_dword;
typedef  int32_t libspectrum_signed_dword;
typedef uint64_t libspectrum_qword;
typedef  int64_t libspectrum_signed_qword;


/* glib replacement (if necessary) */
//LIBSPECTRUM_GLIB_REPLACEMENT

/* Library capabilities */
//LIBSPECTRUM_CAPABILITIES

/*
 * General libspectrum routines
 */

/* Error handling */

/* The various errors which can occur */
typedef enum libspectrum_error {

  LIBSPECTRUM_ERROR_NONE = 0,

  LIBSPECTRUM_ERROR_WARNING,

  LIBSPECTRUM_ERROR_MEMORY,
  LIBSPECTRUM_ERROR_UNKNOWN,
  LIBSPECTRUM_ERROR_CORRUPT,
  LIBSPECTRUM_ERROR_SIGNATURE,
  LIBSPECTRUM_ERROR_SLT,	/* .slt data found at end of a .z80 file */
  LIBSPECTRUM_ERROR_INVALID,	/* Invalid parameter supplied */

  LIBSPECTRUM_ERROR_LOGIC = -1,

} libspectrum_error;

/* Initialisation */

libspectrum_error WIN32_DLL libspectrum_init( void );

/* Version checking */

int WIN32_DLL libspectrum_check_version( const char *version );
const char WIN32_DLL *libspectrum_version( void );
const char WIN32_DLL *libspectrum_gcrypt_version( void );

/* Error handling */
typedef libspectrum_error
(*libspectrum_error_function_t)( libspectrum_error error,
				 const char *format, va_list ap );

extern libspectrum_error_function_t WIN32_DLL libspectrum_error_function;

libspectrum_error WIN32_DLL
libspectrum_default_error_function( libspectrum_error error,
				    const char *format, va_list ap );

/* Attempt to identify a given file */

/* Various types of file we might manage to identify */
typedef enum libspectrum_id_t {

  /* These types present in all versions of libspectrum */

  LIBSPECTRUM_ID_UNKNOWN = 0,		/* Unidentified file */
  LIBSPECTRUM_ID_RECORDING_RZX,		/* RZX input recording */
  LIBSPECTRUM_ID_SNAPSHOT_SNA,		/* .sna snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_Z80,		/* .z80 snapshot */
  LIBSPECTRUM_ID_TAPE_TAP,		/* Z80-style .tap tape image */
  LIBSPECTRUM_ID_TAPE_TZX,		/* TZX tape image */

  /* Below here, present only in 0.1.1 and later */

  LIBSPECTRUM_ID_DISK_DSK,		/* .dsk +3 disk image */
  LIBSPECTRUM_ID_DISK_SCL,		/* .scl TR-DOS disk image */
  LIBSPECTRUM_ID_DISK_TRD,		/* .trd TR-DOS disk image */
  LIBSPECTRUM_ID_CARTRIDGE_DCK,		/* .dck Timex cartridge image */

  /* Below here, present only in 0.2.0 and later */

  LIBSPECTRUM_ID_TAPE_WARAJEVO,		/* Warajevo-style .tap tape image */ 

  LIBSPECTRUM_ID_SNAPSHOT_PLUSD,	/* +D snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_SP,		/* .sp snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_SNP,		/* .snp snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_ZXS,		/* .zxs snapshot (zx32) */
  LIBSPECTRUM_ID_SNAPSHOT_SZX,		/* .szx snapshot (Spectaculator) */

  /* Below here, present only in 0.2.1 and later */

  LIBSPECTRUM_ID_COMPRESSED_BZ2,	/* bzip2 compressed file */
  LIBSPECTRUM_ID_COMPRESSED_GZ,		/* gzip compressed file */

  /* Below here, present only in 0.2.2 and later */

  LIBSPECTRUM_ID_HARDDISK_HDF,		/* .hdf hard disk image */
  LIBSPECTRUM_ID_CARTRIDGE_IF2,		/* .rom Interface II cartridge image */

  /* Below here, present only in 0.3.0 and later */

  LIBSPECTRUM_ID_MICRODRIVE_MDR,	/* .mdr microdrive cartridge */
  LIBSPECTRUM_ID_TAPE_CSW,		/* .csw tape image */
  LIBSPECTRUM_ID_TAPE_Z80EM		/* Z80Em tape image */

} libspectrum_id_t;

/* And 'classes' of file */
typedef enum libspectrum_class_t {

  LIBSPECTRUM_CLASS_UNKNOWN,

  LIBSPECTRUM_CLASS_CARTRIDGE_TIMEX,	/* Timex cartridges */
  LIBSPECTRUM_CLASS_DISK_PLUS3,		/* +3 disk */
  LIBSPECTRUM_CLASS_DISK_TRDOS,		/* TR-DOS disk */
  LIBSPECTRUM_CLASS_RECORDING,		/* Input recording */
  LIBSPECTRUM_CLASS_SNAPSHOT,		/* Snapshot */
  LIBSPECTRUM_CLASS_TAPE,		/* Tape */

  /* Below here, present only in 0.2.1 and later */

  LIBSPECTRUM_CLASS_COMPRESSED,		/* A compressed file */

  /* Below here, present only in 0.2.2 and later */

  LIBSPECTRUM_CLASS_HARDDISK,		/* A hard disk image */
  LIBSPECTRUM_CLASS_CARTRIDGE_IF2,	/* Interface II cartridges */

  /* Below here, present only in 0.3.0 and later */

  LIBSPECTRUM_CLASS_MICRODRIVE,		/* Microdrive cartridges */

} libspectrum_class_t;

libspectrum_error WIN32_DLL
libspectrum_identify_file( libspectrum_id_t *type, const char *filename,
                           const unsigned char *buffer, size_t length );

libspectrum_error WIN32_DLL
libspectrum_identify_file_with_class(
  libspectrum_id_t *type, libspectrum_class_t *libspectrum_class,
  const char *filename, const unsigned char *buffer, size_t length );

libspectrum_error WIN32_DLL
libspectrum_identify_file_raw( libspectrum_id_t *type, const char *filename,
			       const unsigned char *buffer, size_t length );

libspectrum_error WIN32_DLL
libspectrum_identify_class( libspectrum_class_t *libspectrum_class,
                            libspectrum_id_t type );

/* Different Spectrum variants and their capabilities */

/* The machine types we can handle */
typedef enum libspectrum_machine {

  LIBSPECTRUM_MACHINE_48,
  LIBSPECTRUM_MACHINE_TC2048,
  LIBSPECTRUM_MACHINE_128,
  LIBSPECTRUM_MACHINE_PLUS2,
  LIBSPECTRUM_MACHINE_PENT,
  LIBSPECTRUM_MACHINE_PLUS2A,
  LIBSPECTRUM_MACHINE_PLUS3,

  /* Used by libspectrum_tape_guess_hardware if we can't work out what
     hardware should be used */
  LIBSPECTRUM_MACHINE_UNKNOWN,

  LIBSPECTRUM_MACHINE_16,
  LIBSPECTRUM_MACHINE_TC2068,

  LIBSPECTRUM_MACHINE_SCORP,
  LIBSPECTRUM_MACHINE_PLUS3E,
  LIBSPECTRUM_MACHINE_SE,

  LIBSPECTRUM_MACHINE_TS2068,

} libspectrum_machine;

const char* WIN32_DLL libspectrum_machine_name( libspectrum_machine type );

/* The various capabilities of the different machines */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_AY;                              /* AY-3-8192 */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_128_MEMORY;        /* 128-style memory paging */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_MEMORY;       /* +3-style memory paging */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_DISK;            /* +3-style disk drive */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_MEMORY;    /* Timex-style memory paging */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_VIDEO;       /* Timex-style video modes */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_TRDOS_DISK;         /* TRDOS-style disk drive */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_DOCK; /* T[SC]2068-style cartridge port */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_SINCLAIR_JOYSTICK;
                                            /* Sinclair-style joystick ports */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_KEMPSTON_JOYSTICK;
                                            /* Kempston-style joystick ports */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_SCORP_MEMORY; /* Scorpion-style memory paging */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_EVEN_M1;
                             /* M1 cycles always start on even tstate counts */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_SE_MEMORY;          /* SE-style memory paging */
extern const int WIN32_DLL
LIBSPECTRUM_MACHINE_CAPABILITY_NTSC;                         /* NTSC display */

/* Get the capabilities of a machine */
int WIN32_DLL libspectrum_machine_capabilities( libspectrum_machine type );

/* Get the timings of a machine */
libspectrum_dword WIN32_DLL
libspectrum_timings_processor_speed( libspectrum_machine machine );
libspectrum_dword WIN32_DLL
libspectrum_timings_ay_speed( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_left_border( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_horizontal_screen( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_right_border( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_horizontal_retrace( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_top_border( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_vertical_screen( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_bottom_border( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_vertical_retrace( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_interrupt_length( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_top_left_pixel( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_tstates_per_line( libspectrum_machine machine );
libspectrum_word WIN32_DLL
libspectrum_timings_lines_per_frame( libspectrum_machine machine );
libspectrum_dword WIN32_DLL
libspectrum_timings_tstates_per_frame( libspectrum_machine machine );

/* Creator information */

typedef struct libspectrum_creator libspectrum_creator;

libspectrum_error WIN32_DLL
libspectrum_creator_alloc( libspectrum_creator **creator );
libspectrum_error WIN32_DLL
libspectrum_creator_free( libspectrum_creator *creator );

libspectrum_error WIN32_DLL
libspectrum_creator_set_program( libspectrum_creator *creator,
				 const char *program );
const char WIN32_DLL *
libspectrum_creator_program( libspectrum_creator *creator );

libspectrum_error WIN32_DLL
libspectrum_creator_set_major( libspectrum_creator *creator,
			       libspectrum_word major );
libspectrum_word WIN32_DLL
libspectrum_creator_major( libspectrum_creator *creator );

libspectrum_error WIN32_DLL
libspectrum_creator_set_minor( libspectrum_creator *creator,
			       libspectrum_word minor );
libspectrum_word WIN32_DLL
libspectrum_creator_minor( libspectrum_creator *creator );

libspectrum_error WIN32_DLL
libspectrum_creator_set_competition_code( libspectrum_creator *creator,
					  libspectrum_dword competition_code );
libspectrum_dword WIN32_DLL
libspectrum_creator_competition_code( libspectrum_creator *creator );

libspectrum_error WIN32_DLL
libspectrum_creator_set_custom( libspectrum_creator *creator,
				libspectrum_byte *data, size_t length );

libspectrum_byte WIN32_DLL *
libspectrum_creator_custom( libspectrum_creator *creator );

size_t WIN32_DLL
libspectrum_creator_custom_length( libspectrum_creator *creator );

/*
 * Snap handling routines
 */

typedef struct libspectrum_snap libspectrum_snap;

libspectrum_error WIN32_DLL libspectrum_snap_alloc( libspectrum_snap **snap );
libspectrum_error WIN32_DLL libspectrum_snap_free( libspectrum_snap *snap );

/* Read in a snapshot, optionally guessing what type it is */
libspectrum_error WIN32_DLL
libspectrum_snap_read( libspectrum_snap *snap, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* Write a snapshot */
libspectrum_error WIN32_DLL
libspectrum_snap_write( libspectrum_byte **buffer, size_t *length,
			int *out_flags, libspectrum_snap *snap,
	 		libspectrum_id_t type, libspectrum_creator *creator,
			int in_flags );

/* The flags that can be given to libspectrum_snap_write() */
extern const int WIN32_DLL LIBSPECTRUM_FLAG_SNAPSHOT_NO_COMPRESSION;
extern const int WIN32_DLL LIBSPECTRUM_FLAG_SNAPSHOT_ALWAYS_COMPRESS;

/* The flags that may be returned from libspectrum_snap_write() */
extern const int WIN32_DLL LIBSPECTRUM_FLAG_SNAPSHOT_MINOR_INFO_LOSS;
extern const int WIN32_DLL LIBSPECTRUM_FLAG_SNAPSHOT_MAJOR_INFO_LOSS;

/* The joystick types we can handle */
typedef enum libspectrum_joystick {

  LIBSPECTRUM_JOYSTICK_NONE,

  LIBSPECTRUM_JOYSTICK_CURSOR,
  LIBSPECTRUM_JOYSTICK_KEMPSTON,
  LIBSPECTRUM_JOYSTICK_SINCLAIR_1,
  LIBSPECTRUM_JOYSTICK_SINCLAIR_2,
  LIBSPECTRUM_JOYSTICK_TIMEX_1,
  LIBSPECTRUM_JOYSTICK_TIMEX_2,
  LIBSPECTRUM_JOYSTICK_FULLER,

} libspectrum_joystick;

const char* WIN32_DLL libspectrum_joystick_name( libspectrum_joystick type );

extern const int WIN32_DLL LIBSPECTRUM_JOYSTICK_INPUT_NONE;
extern const int WIN32_DLL LIBSPECTRUM_JOYSTICK_INPUT_KEYBOARD;
extern const int WIN32_DLL LIBSPECTRUM_JOYSTICK_INPUT_JOYSTICK_1;
extern const int WIN32_DLL LIBSPECTRUM_JOYSTICK_INPUT_JOYSTICK_2;

/* Accessor functions */
libspectrum_machine WIN32_DLL libspectrum_snap_machine( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_machine( libspectrum_snap *snap, libspectrum_machine machine );
libspectrum_byte WIN32_DLL libspectrum_snap_a( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_a( libspectrum_snap *snap, libspectrum_byte a );
libspectrum_byte WIN32_DLL libspectrum_snap_f( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_f( libspectrum_snap *snap, libspectrum_byte f );
libspectrum_word WIN32_DLL libspectrum_snap_bc( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_bc( libspectrum_snap *snap, libspectrum_word bc );
libspectrum_word WIN32_DLL libspectrum_snap_de( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_de( libspectrum_snap *snap, libspectrum_word de );
libspectrum_word WIN32_DLL libspectrum_snap_hl( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_hl( libspectrum_snap *snap, libspectrum_word hl );
libspectrum_byte WIN32_DLL libspectrum_snap_a_( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_a_( libspectrum_snap *snap, libspectrum_byte a_ );
libspectrum_byte WIN32_DLL libspectrum_snap_f_( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_f_( libspectrum_snap *snap, libspectrum_byte f_ );
libspectrum_word WIN32_DLL libspectrum_snap_bc_( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_bc_( libspectrum_snap *snap, libspectrum_word bc_ );
libspectrum_word WIN32_DLL libspectrum_snap_de_( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_de_( libspectrum_snap *snap, libspectrum_word de_ );
libspectrum_word WIN32_DLL libspectrum_snap_hl_( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_hl_( libspectrum_snap *snap, libspectrum_word hl_ );
libspectrum_word WIN32_DLL libspectrum_snap_ix( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_ix( libspectrum_snap *snap, libspectrum_word ix );
libspectrum_word WIN32_DLL libspectrum_snap_iy( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_iy( libspectrum_snap *snap, libspectrum_word iy );
libspectrum_byte WIN32_DLL libspectrum_snap_i( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_i( libspectrum_snap *snap, libspectrum_byte i );
libspectrum_byte WIN32_DLL libspectrum_snap_r( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_r( libspectrum_snap *snap, libspectrum_byte r );
libspectrum_word WIN32_DLL libspectrum_snap_sp( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_sp( libspectrum_snap *snap, libspectrum_word sp );
libspectrum_word WIN32_DLL libspectrum_snap_pc( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_pc( libspectrum_snap *snap, libspectrum_word pc );
libspectrum_byte WIN32_DLL libspectrum_snap_iff1( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_iff1( libspectrum_snap *snap, libspectrum_byte iff1 );
libspectrum_byte WIN32_DLL libspectrum_snap_iff2( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_iff2( libspectrum_snap *snap, libspectrum_byte iff2 );
libspectrum_byte WIN32_DLL libspectrum_snap_im( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_im( libspectrum_snap *snap, libspectrum_byte im );
libspectrum_dword WIN32_DLL libspectrum_snap_tstates( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_tstates( libspectrum_snap *snap, libspectrum_dword tstates );
int WIN32_DLL libspectrum_snap_halted( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_halted( libspectrum_snap *snap, int halted );
int WIN32_DLL libspectrum_snap_last_instruction_ei( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_last_instruction_ei( libspectrum_snap *snap, int last_instruction_ei );
libspectrum_byte WIN32_DLL libspectrum_snap_out_ula( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_ula( libspectrum_snap *snap, libspectrum_byte out_ula );
libspectrum_byte WIN32_DLL libspectrum_snap_out_128_memoryport( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_128_memoryport( libspectrum_snap *snap, libspectrum_byte out_128_memoryport );
libspectrum_byte WIN32_DLL libspectrum_snap_out_plus3_memoryport( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_plus3_memoryport( libspectrum_snap *snap, libspectrum_byte out_plus3_memoryport );
libspectrum_byte WIN32_DLL libspectrum_snap_out_ay_registerport( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_ay_registerport( libspectrum_snap *snap, libspectrum_byte out_ay_registerport );
libspectrum_byte WIN32_DLL libspectrum_snap_ay_registers( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_ay_registers( libspectrum_snap *snap, int idx, libspectrum_byte ay_registers );
libspectrum_byte WIN32_DLL libspectrum_snap_out_scld_hsr( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_scld_hsr( libspectrum_snap *snap, libspectrum_byte out_scld_hsr );
libspectrum_byte WIN32_DLL libspectrum_snap_out_scld_dec( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_out_scld_dec( libspectrum_snap *snap, libspectrum_byte out_scld_dec );
int WIN32_DLL libspectrum_snap_beta_paged( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_paged( libspectrum_snap *snap, int beta_paged );
int WIN32_DLL libspectrum_snap_beta_direction( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_direction( libspectrum_snap *snap, int beta_direction );
libspectrum_byte WIN32_DLL libspectrum_snap_beta_system( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_system( libspectrum_snap *snap, libspectrum_byte beta_system );
libspectrum_byte WIN32_DLL libspectrum_snap_beta_track( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_track( libspectrum_snap *snap, libspectrum_byte beta_track );
libspectrum_byte WIN32_DLL libspectrum_snap_beta_sector( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_sector( libspectrum_snap *snap, libspectrum_byte beta_sector );
libspectrum_byte WIN32_DLL libspectrum_snap_beta_data( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_data( libspectrum_snap *snap, libspectrum_byte beta_data );
libspectrum_byte WIN32_DLL libspectrum_snap_beta_status( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_beta_status( libspectrum_snap *snap, libspectrum_byte beta_status );
libspectrum_byte WIN32_DLL * libspectrum_snap_pages( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_pages( libspectrum_snap *snap, int idx, libspectrum_byte* pages );
libspectrum_byte WIN32_DLL * libspectrum_snap_slt( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_slt( libspectrum_snap *snap, int idx, libspectrum_byte* slt );
size_t WIN32_DLL libspectrum_snap_slt_length( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_slt_length( libspectrum_snap *snap, int idx, size_t slt_length );
libspectrum_byte WIN32_DLL * libspectrum_snap_slt_screen( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_slt_screen( libspectrum_snap *snap, libspectrum_byte* slt_screen );
int WIN32_DLL libspectrum_snap_slt_screen_level( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_slt_screen_level( libspectrum_snap *snap, int slt_screen_level );
int WIN32_DLL libspectrum_snap_zxatasp_active( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_active( libspectrum_snap *snap, int zxatasp_active );
int WIN32_DLL libspectrum_snap_zxatasp_upload( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_upload( libspectrum_snap *snap, int zxatasp_upload );
int WIN32_DLL libspectrum_snap_zxatasp_writeprotect( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_writeprotect( libspectrum_snap *snap, int zxatasp_writeprotect );
libspectrum_byte WIN32_DLL libspectrum_snap_zxatasp_port_a( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_port_a( libspectrum_snap *snap, libspectrum_byte zxatasp_port_a );
libspectrum_byte WIN32_DLL libspectrum_snap_zxatasp_port_b( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_port_b( libspectrum_snap *snap, libspectrum_byte zxatasp_port_b );
libspectrum_byte WIN32_DLL libspectrum_snap_zxatasp_port_c( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_port_c( libspectrum_snap *snap, libspectrum_byte zxatasp_port_c );
libspectrum_byte WIN32_DLL libspectrum_snap_zxatasp_control( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_control( libspectrum_snap *snap, libspectrum_byte zxatasp_control );
size_t WIN32_DLL libspectrum_snap_zxatasp_pages( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_pages( libspectrum_snap *snap, size_t zxatasp_pages );
size_t WIN32_DLL libspectrum_snap_zxatasp_current_page( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxatasp_current_page( libspectrum_snap *snap, size_t zxatasp_current_page );
libspectrum_byte WIN32_DLL * libspectrum_snap_zxatasp_ram( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_zxatasp_ram( libspectrum_snap *snap, int idx, libspectrum_byte* zxatasp_ram );
int WIN32_DLL libspectrum_snap_zxcf_active( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxcf_active( libspectrum_snap *snap, int zxcf_active );
int WIN32_DLL libspectrum_snap_zxcf_upload( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxcf_upload( libspectrum_snap *snap, int zxcf_upload );
libspectrum_byte WIN32_DLL libspectrum_snap_zxcf_memctl( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxcf_memctl( libspectrum_snap *snap, libspectrum_byte zxcf_memctl );
size_t WIN32_DLL libspectrum_snap_zxcf_pages( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_zxcf_pages( libspectrum_snap *snap, size_t zxcf_pages );
libspectrum_byte WIN32_DLL * libspectrum_snap_zxcf_ram( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_zxcf_ram( libspectrum_snap *snap, int idx, libspectrum_byte* zxcf_ram );
int WIN32_DLL libspectrum_snap_interface2_active( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_interface2_active( libspectrum_snap *snap, int interface2_active );
libspectrum_byte WIN32_DLL * libspectrum_snap_interface2_rom( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_interface2_rom( libspectrum_snap *snap, int idx, libspectrum_byte* interface2_rom );
int WIN32_DLL libspectrum_snap_dock_active( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_dock_active( libspectrum_snap *snap, int dock_active );
libspectrum_byte WIN32_DLL libspectrum_snap_exrom_ram( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_exrom_ram( libspectrum_snap *snap, int idx, libspectrum_byte exrom_ram );
libspectrum_byte WIN32_DLL * libspectrum_snap_exrom_cart( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_exrom_cart( libspectrum_snap *snap, int idx, libspectrum_byte* exrom_cart );
libspectrum_byte WIN32_DLL libspectrum_snap_dock_ram( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_dock_ram( libspectrum_snap *snap, int idx, libspectrum_byte dock_ram );
libspectrum_byte WIN32_DLL * libspectrum_snap_dock_cart( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_dock_cart( libspectrum_snap *snap, int idx, libspectrum_byte* dock_cart );
int WIN32_DLL libspectrum_snap_issue2( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_issue2( libspectrum_snap *snap, int issue2 );
size_t WIN32_DLL libspectrum_snap_joystick_active_count( libspectrum_snap *snap );
void WIN32_DLL libspectrum_snap_set_joystick_active_count( libspectrum_snap *snap, size_t joystick_active_count );
libspectrum_joystick WIN32_DLL libspectrum_snap_joystick_list( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_joystick_list( libspectrum_snap *snap, int idx, libspectrum_joystick joystick_list );
int WIN32_DLL libspectrum_snap_joystick_inputs( libspectrum_snap *snap, int idx );
void WIN32_DLL libspectrum_snap_set_joystick_inputs( libspectrum_snap *snap, int idx, int joystick_inputs );

/* DEPRECATED: use libspectrum_snap_read() instead */
DEPRECATED
libspectrum_error WIN32_DLL
libspectrum_sna_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

/* DEPRECATED: use libspectrum_snap_read() instead */
DEPRECATED
libspectrum_error WIN32_DLL
libspectrum_z80_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

/* DEPRECATED: use libspectrum_snap_write() instead */
DEPRECATED
libspectrum_error WIN32_DLL
libspectrum_z80_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_snap *snap );

/*
 * Tape handling routines
 */

/* The various types of block available */
typedef enum libspectrum_tape_type {

  /* Values must be the same as used in the .tzx format */

  LIBSPECTRUM_TAPE_BLOCK_ROM = 0x10,
  LIBSPECTRUM_TAPE_BLOCK_TURBO,
  LIBSPECTRUM_TAPE_BLOCK_PURE_TONE,
  LIBSPECTRUM_TAPE_BLOCK_PULSES,
  LIBSPECTRUM_TAPE_BLOCK_PURE_DATA,
  LIBSPECTRUM_TAPE_BLOCK_RAW_DATA,

  LIBSPECTRUM_TAPE_BLOCK_GENERALISED_DATA = 0x19,

  LIBSPECTRUM_TAPE_BLOCK_PAUSE = 0x20,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_START,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_END,
  LIBSPECTRUM_TAPE_BLOCK_JUMP,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_START,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_END,

  LIBSPECTRUM_TAPE_BLOCK_SELECT = 0x28,

  LIBSPECTRUM_TAPE_BLOCK_STOP48 = 0x2a,

  LIBSPECTRUM_TAPE_BLOCK_COMMENT = 0x30,
  LIBSPECTRUM_TAPE_BLOCK_MESSAGE,
  LIBSPECTRUM_TAPE_BLOCK_ARCHIVE_INFO,
  LIBSPECTRUM_TAPE_BLOCK_HARDWARE,

  LIBSPECTRUM_TAPE_BLOCK_CUSTOM = 0x35,

  LIBSPECTRUM_TAPE_BLOCK_CONCAT = 0x5a,

  /* Past here are block types not in the .tzx format */

  LIBSPECTRUM_TAPE_BLOCK_RLE_PULSE = 0x100,

} libspectrum_tape_type;

typedef struct libspectrum_tape_block libspectrum_tape_block;
typedef struct libspectrum_tape_generalised_data_symbol_table libspectrum_tape_generalised_data_symbol_table;

/* Something to step through all the blocks in a tape */
typedef struct _GSList *libspectrum_tape_iterator;

/* Some flags */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_BLOCK;  /* End of block */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_STOP;	  /* Stop tape */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_STOP48; /* Stop tape if in 48K mode */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_NO_EDGE; /* Edge isn't really an edge */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_LEVEL_LOW; /* Set level low */
extern const int WIN32_DLL LIBSPECTRUM_TAPE_FLAGS_LEVEL_HIGH; /* Set level high */

/* The states which a block can be in */
typedef enum libspectrum_tape_state_type {

  LIBSPECTRUM_TAPE_STATE_INVALID = 0,

  LIBSPECTRUM_TAPE_STATE_PILOT,	/* Pilot pulses */
  LIBSPECTRUM_TAPE_STATE_SYNC1,	/* First sync pulse */
  LIBSPECTRUM_TAPE_STATE_SYNC2,	/* Second sync pulse */
  LIBSPECTRUM_TAPE_STATE_DATA1,	/* First edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_DATA2,	/* Second edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_PAUSE,	/* The pause at the end of a block */

} libspectrum_tape_state_type;

/* Routines to manipulate tape blocks */
libspectrum_error WIN32_DLL
libspectrum_tape_block_alloc( libspectrum_tape_block **block,
			      libspectrum_tape_type type );
libspectrum_error WIN32_DLL
libspectrum_tape_block_free( libspectrum_tape_block *block );
libspectrum_tape_type WIN32_DLL
libspectrum_tape_block_type( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL
libspectrum_tape_block_set_type( libspectrum_tape_block *block,
				 libspectrum_tape_type type );

libspectrum_error WIN32_DLL
libspectrum_tape_block_init( libspectrum_tape_block *block );

libspectrum_error WIN32_DLL
libspectrum_tape_block_description( char *buffer, size_t length,
	                            libspectrum_tape_block *block );

int WIN32_DLL
libspectrum_tape_block_metadata( libspectrum_tape_block *block );

/* Accessor functions */
libspectrum_dword WIN32_DLL libspectrum_tape_block_bit_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_bit_length( libspectrum_tape_block *block, libspectrum_dword bit_length );
libspectrum_dword WIN32_DLL libspectrum_tape_block_bit0_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_bit0_length( libspectrum_tape_block *block, libspectrum_dword bit0_length );
libspectrum_dword WIN32_DLL libspectrum_tape_block_bit1_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_bit1_length( libspectrum_tape_block *block, libspectrum_dword bit1_length );
size_t WIN32_DLL libspectrum_tape_block_bits_in_last_byte( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_bits_in_last_byte( libspectrum_tape_block *block, size_t bits_in_last_byte );
size_t WIN32_DLL libspectrum_tape_block_bits_per_data_symbol( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_bits_per_data_symbol( libspectrum_tape_block *block, size_t bits_per_data_symbol );
size_t WIN32_DLL libspectrum_tape_block_count( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_count( libspectrum_tape_block *block, size_t count );
libspectrum_byte WIN32_DLL * libspectrum_tape_block_data( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_data( libspectrum_tape_block *block, libspectrum_byte* data );
size_t WIN32_DLL libspectrum_tape_block_data_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_data_length( libspectrum_tape_block *block, size_t data_length );
libspectrum_tape_generalised_data_symbol_table WIN32_DLL * libspectrum_tape_block_data_table( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_data_table( libspectrum_tape_block *block, libspectrum_tape_generalised_data_symbol_table* data_table );
int WIN32_DLL libspectrum_tape_block_ids( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_ids( libspectrum_tape_block *block, int *ids );
int WIN32_DLL libspectrum_tape_block_offset( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_offset( libspectrum_tape_block *block, int offset );
int WIN32_DLL libspectrum_tape_block_offsets( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_offsets( libspectrum_tape_block *block, int *offsets );
libspectrum_dword WIN32_DLL libspectrum_tape_block_pause( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pause( libspectrum_tape_block *block, libspectrum_dword pause );
libspectrum_dword WIN32_DLL libspectrum_tape_block_pilot_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pilot_length( libspectrum_tape_block *block, libspectrum_dword pilot_length );
size_t WIN32_DLL libspectrum_tape_block_pilot_pulses( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pilot_pulses( libspectrum_tape_block *block, size_t pilot_pulses );
libspectrum_word WIN32_DLL libspectrum_tape_block_pilot_repeats( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pilot_repeats( libspectrum_tape_block *block, libspectrum_word *pilot_repeats );
libspectrum_byte WIN32_DLL libspectrum_tape_block_pilot_symbols( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pilot_symbols( libspectrum_tape_block *block, libspectrum_byte *pilot_symbols );
libspectrum_tape_generalised_data_symbol_table WIN32_DLL * libspectrum_tape_block_pilot_table( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pilot_table( libspectrum_tape_block *block, libspectrum_tape_generalised_data_symbol_table* pilot_table );
libspectrum_dword WIN32_DLL libspectrum_tape_block_pulse_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pulse_length( libspectrum_tape_block *block, libspectrum_dword pulse_length );
libspectrum_dword WIN32_DLL libspectrum_tape_block_pulse_lengths( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_pulse_lengths( libspectrum_tape_block *block, libspectrum_dword *pulse_lengths );
libspectrum_tape_state_type WIN32_DLL libspectrum_tape_block_state( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_state( libspectrum_tape_block *block, libspectrum_tape_state_type state );
libspectrum_dword WIN32_DLL libspectrum_tape_block_sync1_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_sync1_length( libspectrum_tape_block *block, libspectrum_dword sync1_length );
libspectrum_dword WIN32_DLL libspectrum_tape_block_sync2_length( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_sync2_length( libspectrum_tape_block *block, libspectrum_dword sync2_length );
char WIN32_DLL * libspectrum_tape_block_text( libspectrum_tape_block *block );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_text( libspectrum_tape_block *block, char* text );
char WIN32_DLL * libspectrum_tape_block_texts( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_texts( libspectrum_tape_block *block, char* *texts );
int WIN32_DLL libspectrum_tape_block_types( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_types( libspectrum_tape_block *block, int *types );
int WIN32_DLL libspectrum_tape_block_values( libspectrum_tape_block *block, size_t idx );
libspectrum_error WIN32_DLL libspectrum_tape_block_set_values( libspectrum_tape_block *block, int *values );

/* A linked list of tape blocks */
typedef struct libspectrum_tape libspectrum_tape;

libspectrum_error WIN32_DLL libspectrum_tape_alloc( libspectrum_tape **tape );
libspectrum_error WIN32_DLL libspectrum_tape_clear( libspectrum_tape *tape );
libspectrum_error WIN32_DLL libspectrum_tape_free( libspectrum_tape *tape );

/* Read in a tape file, optionally guessing what sort of file it is */
libspectrum_error WIN32_DLL
libspectrum_tape_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* Write a tape file */
libspectrum_error WIN32_DLL
libspectrum_tape_write( libspectrum_byte **buffer, size_t *length,
			libspectrum_tape *tape, libspectrum_id_t type );

/* Does this tape structure actually contain a tape? */
int WIN32_DLL libspectrum_tape_present( const libspectrum_tape *tape );

libspectrum_error WIN32_DLL
libspectrum_tape_get_next_edge( libspectrum_dword *tstates, int *flags,
	                        libspectrum_tape *tape );

/* Get the current block from the tape */
libspectrum_tape_block WIN32_DLL *
libspectrum_tape_current_block( libspectrum_tape *tape );

/* Peek at the next block on the tape */
libspectrum_tape_block WIN32_DLL *
libspectrum_tape_peek_next_block( libspectrum_tape *tape );

/* Cause the next block on the tape to be active, initialise it
   and return it */
libspectrum_tape_block WIN32_DLL *
libspectrum_tape_select_next_block( libspectrum_tape *tape );

/* Get the position on the tape of the current block */
libspectrum_error WIN32_DLL
libspectrum_tape_position( int *n, libspectrum_tape *tape );

/* Select the nth block on the tape */
libspectrum_error WIN32_DLL
libspectrum_tape_nth_block( libspectrum_tape *tape, int n );

/* Append a block to the current tape */
libspectrum_error WIN32_DLL
libspectrum_tape_append_block( libspectrum_tape *tape,
			       libspectrum_tape_block *block );

libspectrum_error
libspectrum_tape_remove_block( libspectrum_tape *tape,
			       libspectrum_tape_iterator it );

libspectrum_error
libspectrum_tape_insert_block( libspectrum_tape *tape,
			       libspectrum_tape_block *block,
			       size_t position );

/*** Routines for .tap format files ***/

libspectrum_error WIN32_DLL
libspectrum_tap_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );
libspectrum_error WIN32_DLL
libspectrum_tap_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

/*** Routines for .tzx format files ***/

libspectrum_error WIN32_DLL
libspectrum_tzx_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );

libspectrum_error WIN32_DLL
libspectrum_tzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

/*** Routines for Warajevo .tap format files ***/

libspectrum_error WIN32_DLL
libspectrum_warajevo_read( libspectrum_tape *tape,
			   const libspectrum_byte *buffer, size_t length );

/*** Routines for iterating through a tape ***/

libspectrum_tape_block WIN32_DLL *
libspectrum_tape_iterator_init( libspectrum_tape_iterator *iterator,
				libspectrum_tape *tape );

libspectrum_tape_block WIN32_DLL *
libspectrum_tape_iterator_next( libspectrum_tape_iterator *iterator );

/*** Routines for handling the TZX generalised data block symbol table
     structure ***/

typedef enum libspectrum_tape_generalised_data_symbol_edge_type {

  LIBSPECTRUM_TAPE_GENERALISED_DATA_SYMBOL_EDGE = 0x00,
  LIBSPECTRUM_TAPE_GENERALISED_DATA_SYMBOL_NO_EDGE,
  LIBSPECTRUM_TAPE_GENERALISED_DATA_SYMBOL_LOW,
  LIBSPECTRUM_TAPE_GENERALISED_DATA_SYMBOL_HIGH,

} libspectrum_tape_generalised_data_symbol_edge_type;

typedef struct libspectrum_tape_generalised_data_symbol libspectrum_tape_generalised_data_symbol;

libspectrum_dword WIN32_DLL libspectrum_tape_generalised_data_symbol_table_symbols_in_block( const libspectrum_tape_generalised_data_symbol_table *table );
libspectrum_byte WIN32_DLL libspectrum_tape_generalised_data_symbol_table_max_pulses( const libspectrum_tape_generalised_data_symbol_table *table );
libspectrum_word WIN32_DLL libspectrum_tape_generalised_data_symbol_table_symbols_in_table( const libspectrum_tape_generalised_data_symbol_table *table );
libspectrum_tape_generalised_data_symbol* WIN32_DLL libspectrum_tape_generalised_data_symbol_table_symbol( const libspectrum_tape_generalised_data_symbol_table *table, size_t which );	

libspectrum_tape_generalised_data_symbol_edge_type WIN32_DLL libspectrum_tape_generalised_data_symbol_type( const libspectrum_tape_generalised_data_symbol *symbol );
libspectrum_word WIN32_DLL libspectrum_tape_generalised_data_symbol_pulse( const libspectrum_tape_generalised_data_symbol *symbol, size_t which );


/*
 * Input recording routines
 */

typedef struct libspectrum_rzx libspectrum_rzx;

libspectrum_error WIN32_DLL libspectrum_rzx_alloc( libspectrum_rzx **rzx );
libspectrum_error WIN32_DLL libspectrum_rzx_free( libspectrum_rzx *rzx );

libspectrum_error WIN32_DLL
libspectrum_rzx_start_input( libspectrum_rzx *rzx, libspectrum_dword tstates );
libspectrum_error WIN32_DLL
libspectrum_rzx_stop_input( libspectrum_rzx *rzx );
libspectrum_error WIN32_DLL
libspectrum_rzx_add_snap( libspectrum_rzx *rzx, libspectrum_snap *snap );
libspectrum_error WIN32_DLL
libspectrum_rzx_rollback( libspectrum_rzx *rzx, libspectrum_snap **snap );
libspectrum_error WIN32_DLL
libspectrum_rzx_rollback_to( libspectrum_rzx *rzx, libspectrum_snap **snap,
			     size_t which );

libspectrum_error WIN32_DLL
libspectrum_rzx_store_frame( libspectrum_rzx *rzx, size_t instructions, 
			     size_t count, libspectrum_byte *in_bytes );

libspectrum_error WIN32_DLL
libspectrum_rzx_start_playback( libspectrum_rzx *rzx, int which,
				libspectrum_snap **snap );
libspectrum_error WIN32_DLL
libspectrum_rzx_playback_frame( libspectrum_rzx *rzx, int *finished, libspectrum_snap **snap );
libspectrum_error WIN32_DLL
libspectrum_rzx_playback( libspectrum_rzx *rzx, libspectrum_byte *byte );

/* Get and set the tstate counter */
size_t WIN32_DLL libspectrum_rzx_tstates( libspectrum_rzx *rzx );

/* Get the current frame's instruction count */
size_t WIN32_DLL libspectrum_rzx_instructions( libspectrum_rzx *rzx );

libspectrum_dword WIN32_DLL libspectrum_rzx_get_keyid( libspectrum_rzx *rzx );

typedef struct libspectrum_signature {

  /* Where in the buffer is the signed data? */
  const libspectrum_byte *start; ptrdiff_t length;

//LIBSPECTRUM_SIGNATURE_PARAMETERS

} libspectrum_signature;

libspectrum_error WIN32_DLL
libspectrum_rzx_get_signature( libspectrum_rzx *rzx,
			       libspectrum_signature *signature );

typedef struct libspectrum_rzx_dsa_key {

  /* The standard DSA parameters, represented as hex strings with MSB first.
     Set x = NULL for a public key */
  const char *p, *q, *g, *y, *x;

} libspectrum_rzx_dsa_key;

libspectrum_error WIN32_DLL
libspectrum_rzx_read( libspectrum_rzx *rzx, const libspectrum_byte *buffer,
		      size_t length );

libspectrum_error WIN32_DLL
libspectrum_rzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_rzx *rzx, libspectrum_id_t snap_format,
		       libspectrum_creator *creator, int compress,
		       libspectrum_rzx_dsa_key *key );

libspectrum_error WIN32_DLL libspectrum_rzx_alloc( libspectrum_rzx **rzx );

/* Something to step through all the blocks in an input recording */
typedef struct _GSList *libspectrum_rzx_iterator;

/* The block types which can appear in RZX files */
typedef enum libspectrum_rzx_block_id {

  LIBSPECTRUM_RZX_CREATOR_BLOCK = 0x10,

  LIBSPECTRUM_RZX_SIGN_START_BLOCK = 0x20,
  LIBSPECTRUM_RZX_SIGN_END_BLOCK = 0x21,

  LIBSPECTRUM_RZX_SNAPSHOT_BLOCK = 0x30,

  LIBSPECTRUM_RZX_INPUT_BLOCK = 0x80,

} libspectrum_rzx_block_id;

libspectrum_rzx_iterator WIN32_DLL
libspectrum_rzx_iterator_begin( libspectrum_rzx *rzx );
libspectrum_rzx_iterator WIN32_DLL
libspectrum_rzx_iterator_next( libspectrum_rzx_iterator it );

libspectrum_rzx_block_id WIN32_DLL
libspectrum_rzx_iterator_get_type( libspectrum_rzx_iterator it );
size_t WIN32_DLL
libspectrum_rzx_iterator_get_frames( libspectrum_rzx_iterator it );

/*
 * Microdrive image handling routines
 */

typedef struct libspectrum_microdrive libspectrum_microdrive;

#define LIBSPECTRUM_MICRODRIVE_BLOCK_MAX 254
#define LIBSPECTRUM_MICRODRIVE_HEAD_LEN 15
#define LIBSPECTRUM_MICRODRIVE_DATA_LEN 512
#define LIBSPECTRUM_MICRODRIVE_BLOCK_LEN ( LIBSPECTRUM_MICRODRIVE_HEAD_LEN + \
				    LIBSPECTRUM_MICRODRIVE_HEAD_LEN +  \
				    LIBSPECTRUM_MICRODRIVE_DATA_LEN + 1 )
#define LIBSPECTRUM_MICRODRIVE_CARTRIDGE_LENGTH \
	( LIBSPECTRUM_MICRODRIVE_BLOCK_MAX * LIBSPECTRUM_MICRODRIVE_BLOCK_LEN )

/* Constructor/destructor */

libspectrum_error WIN32_DLL
libspectrum_microdrive_alloc( libspectrum_microdrive **microdrive );
libspectrum_error WIN32_DLL
libspectrum_microdrive_free( libspectrum_microdrive *microdrive );

/* Accessors */

libspectrum_byte WIN32_DLL
libspectrum_microdrive_data( const libspectrum_microdrive *microdrive,
			     size_t which );
void WIN32_DLL
libspectrum_microdrive_set_data( libspectrum_microdrive *microdrive,
				 size_t which, libspectrum_byte data );

int WIN32_DLL
libspectrum_microdrive_write_protect( const libspectrum_microdrive *microdrive );
void WIN32_DLL
libspectrum_microdrive_set_write_protect( libspectrum_microdrive *microdrive,
					  int write_protect );

libspectrum_byte WIN32_DLL
libspectrum_microdrive_cartridge_len( const libspectrum_microdrive *microdrive );
void WIN32_DLL
libspectrum_microdrive_set_cartridge_len( libspectrum_microdrive *microdrive,
					  libspectrum_byte len );

/* .mdr format routines */

int WIN32_DLL
libspectrum_microdrive_checksum( libspectrum_microdrive *microdrive,
				 libspectrum_byte what );
libspectrum_error WIN32_DLL
libspectrum_microdrive_mdr_read( libspectrum_microdrive *microdrive,
				 libspectrum_byte *buffer, size_t length );
libspectrum_error WIN32_DLL
libspectrum_microdrive_mdr_write( const libspectrum_microdrive *microdrive,
				  libspectrum_byte **buffer, size_t *length );

/*
 * Timex DOCK/EXROM handling routines
 */

/* The types of memory which this bank will page into */
typedef enum libspectrum_dck_bank {
  LIBSPECTRUM_DCK_BANK_DOCK = 0,
  LIBSPECTRUM_DCK_BANK_EXROM = 254,
  LIBSPECTRUM_DCK_BANK_HOME = 255
} libspectrum_dck_bank; 

/* The type of one 8Kb page */
typedef enum libspectrum_dck_page_type {
  LIBSPECTRUM_DCK_PAGE_NULL = 0,
  LIBSPECTRUM_DCK_PAGE_RAM_EMPTY,
  LIBSPECTRUM_DCK_PAGE_ROM,
  LIBSPECTRUM_DCK_PAGE_RAM
} libspectrum_dck_page_type;

/* One set of 8 x 8Kb pages */
typedef struct libspectrum_dck_block {
  libspectrum_dck_bank bank;
  libspectrum_dck_page_type access[8];
  libspectrum_byte *pages[8];
} libspectrum_dck_block;

/* A number of 8 x 8Kb page sets */
/* FIXME: Remove the arbitrary limit on the number of banks */
typedef struct libspectrum_dck {
  libspectrum_dck_block *dck[256];	/* dck block data */
} libspectrum_dck;

libspectrum_error WIN32_DLL libspectrum_dck_alloc( libspectrum_dck **dck );
libspectrum_error WIN32_DLL
libspectrum_dck_free( libspectrum_dck *dck, int keep_pages );

/* Read in a DCK file */

libspectrum_error WIN32_DLL
libspectrum_dck_read( libspectrum_dck *dck, const libspectrum_byte *buffer,
                      size_t length );

libspectrum_error WIN32_DLL
libspectrum_dck_read2( libspectrum_dck *dck, const libspectrum_byte *buffer,
                       size_t length, const char *filename );

/*
 * Crypto functions
 */

libspectrum_error WIN32_DLL
libspectrum_verify_signature( libspectrum_signature *signature,
			      libspectrum_rzx_dsa_key *key );
libspectrum_error WIN32_DLL
libspectrum_signature_free( libspectrum_signature *signature );

/*
 * IDE hard disk handling routines
 */

typedef enum libspectrum_ide_databus {
  
  LIBSPECTRUM_IDE_DATA8,
  LIBSPECTRUM_IDE_DATA16,
  LIBSPECTRUM_IDE_DATA16_BYTESWAP,
  LIBSPECTRUM_IDE_DATA16_DATA2,

} libspectrum_ide_databus;

typedef enum libspectrum_ide_unit {
  
  LIBSPECTRUM_IDE_MASTER = 0,
  LIBSPECTRUM_IDE_SLAVE = 1,

} libspectrum_ide_unit;

typedef enum libspectrum_ide_register {

  LIBSPECTRUM_IDE_REGISTER_DATA = 0,
  LIBSPECTRUM_IDE_REGISTER_ERROR_FEATURE = 1,
  LIBSPECTRUM_IDE_REGISTER_SECTOR_COUNT = 2,
  LIBSPECTRUM_IDE_REGISTER_SECTOR = 3,
  LIBSPECTRUM_IDE_REGISTER_CYLINDER_LOW = 4,
  LIBSPECTRUM_IDE_REGISTER_CYLINDER_HIGH = 5,
  LIBSPECTRUM_IDE_REGISTER_HEAD_DRIVE = 6,
  LIBSPECTRUM_IDE_REGISTER_COMMAND_STATUS = 7,
  LIBSPECTRUM_IDE_REGISTER_DATA2 = 8,

} libspectrum_ide_register;

typedef struct libspectrum_ide_channel libspectrum_ide_channel;

libspectrum_error WIN32_DLL
libspectrum_ide_alloc( libspectrum_ide_channel **chn,
		       libspectrum_ide_databus databus );
libspectrum_error WIN32_DLL
libspectrum_ide_free( libspectrum_ide_channel *chn );

libspectrum_error WIN32_DLL
libspectrum_ide_insert( libspectrum_ide_channel *chn,
                        libspectrum_ide_unit unit,
                        const char *filename );
libspectrum_error WIN32_DLL
libspectrum_ide_commit( libspectrum_ide_channel *chn,
			libspectrum_ide_unit unit );
int WIN32_DLL
libspectrum_ide_dirty( libspectrum_ide_channel *chn,
		       libspectrum_ide_unit unit );
libspectrum_error WIN32_DLL
libspectrum_ide_eject( libspectrum_ide_channel *chn,
		       libspectrum_ide_unit unit );

libspectrum_error WIN32_DLL
libspectrum_ide_reset( libspectrum_ide_channel *chn );

libspectrum_byte WIN32_DLL
libspectrum_ide_read( libspectrum_ide_channel *chn,
		      libspectrum_ide_register reg );

void WIN32_DLL
libspectrum_ide_write( libspectrum_ide_channel *chn,
		       libspectrum_ide_register reg,
		       libspectrum_byte data );

#ifdef __cplusplus
};
#endif				/* #ifdef __cplusplus */

#endif				/* #ifndef LIBSPECTRUM_LIBSPECTRUM_H */
