/*****************************************************************************
** File:        y8950.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the YM2413 sound chip. 
**              Wraps the c++ implementation taken from openMSX
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "Y8950.h"
#include "fmopl.h"
#include "IoPort.h"
#include "romMapper.h"
#include "SlotManager.h"
#include <stdlib.h>

#define FREQUENCY        3579545
#define SAMPLERATE       44100
#define BUFFER_SIZE      10000

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))


struct Y8950 {
    Mixer* mixer;
    FM_OPL* opl;
    int    slotHandle;
    UInt8  address;
    UInt32 clock;
    UInt32 timer1;
    UInt32 counter1;
    UInt32 timer2;
    UInt32 counter2;
    Int32  buffer[BUFFER_SIZE];
    Int32  defaultBuffer[BUFFER_SIZE];
};

extern INT32 outd;
extern INT32 ams;
extern INT32 vib;
extern INT32 feedback2;

static Y8950* theY8950 = NULL;

static UInt8 y8950ReadAddress(Y8950* y8950, UInt16 ioPort, UInt32 cpuClock)
{
    return (UInt8)OPLRead(y8950->opl, 0);
}

static UInt8 y8950ReadData(Y8950* y8950, UInt16 ioPort, UInt32 cpuClock)
{
    return (UInt8)OPLRead(y8950->opl, 1);
}

static void y8950WriteAddress(Y8950* y8950, UInt16 ioPort, UInt8 address, UInt32 cpuClock)
{
    OPLWrite(y8950->opl, 0, address);
}

static void y8950WriteData(Y8950* y8950, UInt16 ioPort, UInt8 data, UInt32 cpuClock)
{
    mixerSync(y8950->mixer, cpuClock);

    OPLWrite(y8950->opl, 1, data);
}

static Int32* y8950Sync(void* ref, UInt32 count) 
{
    Y8950* y8950 = (Y8950*)ref;
    UInt32 i;

    for (i = 0; i < count; i++) {
        y8950->buffer[i] = Y8950UpdateOne(y8950->opl);
    }

    return y8950->buffer;
}


int y8950GetState(Y8950* y8950, UInt8* buffer, UInt32 systemTime)
{
    int offset  = OFFSETOF(FM_OPL, firstMember);
    int oplSize = sizeof(FM_OPL) + 9 * sizeof(OPL_CH) + sizeof(YM_DELTAT) - offset;
    void* oplMem = y8950->opl->deltat->memory;
    int size = 0;

    memcpy(buffer + size, (char*)y8950->opl + offset, oplSize);
    size += oplSize;

    memcpy(buffer + size, oplMem, 256 * 1024);
    size += 256 * 1024;

    memcpy(buffer + size, &outd, sizeof(outd));
    size += sizeof(outd);

    memcpy(buffer + size, &ams, sizeof(ams));
    size += sizeof(ams);

    memcpy(buffer + size, &vib, sizeof(vib));
    size += sizeof(vib);

    memcpy(buffer + size, &feedback2, sizeof(feedback2));
    size += sizeof(feedback2);

    return size;
}

int y8950SetState(Y8950* y8950, UInt8* buffer, UInt32 systemTime)
{
    int offset  = OFFSETOF(FM_OPL, firstMember);
    int oplSize = sizeof(FM_OPL) + 9 * sizeof(OPL_CH) + sizeof(YM_DELTAT) - offset;
    void* oplMem = y8950->opl->deltat->memory;
    int size = 0;

    OPLResetChip(y8950->opl);

    memcpy((char*)y8950->opl + offset, buffer + size, oplSize);
    size += oplSize;
    y8950->opl->deltat->memory = oplMem;

    memcpy(oplMem, buffer + size, 256 * 1024);
    size += 256 * 1024;

    memcpy(&outd, buffer + size, sizeof(outd));
    size += sizeof(outd);

    memcpy(&ams, buffer + size, sizeof(ams));
    size += sizeof(ams);

    memcpy(&vib, buffer + size, sizeof(vib));
    size += sizeof(vib);

    memcpy(&feedback2, buffer + size, sizeof(feedback2));
    size += sizeof(feedback2);

    return size;
}

static void y8950Destroy(Y8950* y8950) {
    slotUnregisterUnslotted(y8950->slotHandle);

    ioPortUnregister(0xc0);
    ioPortUnregister(0xc1);

    mixerUnregisterChannel(y8950->mixer, MIXER_CHANNEL_MSX_AUDIO);
    OPLDestroy(y8950->opl);

    theY8950 = NULL;
}

int y8950Create(Mixer* mixer, UInt32 cpuClock)
{
    SlotCallbacks callbacks = { y8950Destroy, NULL, NULL, y8950GetState, y8950SetState };
    Y8950* y8950 = (Y8950*)calloc(1, sizeof(Y8950));

    theY8950 = y8950;

    y8950->slotHandle = slotRegisterUnslotted(AUDIO_Y8950, &callbacks, y8950);

    y8950->mixer = mixer;
    y8950->clock = cpuClock;
    y8950->counter1 = -1;
    y8950->counter2 = -1;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_MSX_AUDIO, 0, y8950Sync, y8950);

    y8950->opl = OPLCreate(OPL_TYPE_Y8950, FREQUENCY, SAMPLERATE, 256);
    OPLResetChip(y8950->opl);

    ioPortRegister(0xc0, y8950ReadAddress, y8950WriteAddress, y8950);
    ioPortRegister(0xc1, y8950ReadData,    y8950WriteData,    y8950);

    return 1;
}

void y8950TimerSet(int timer, int count)
{
    if (timer == 0) {
        theY8950->timer1 = count;
    }
    else {
        theY8950->timer2 = count;
    }
}

void y8950TimerStart(int timer, int start, UInt8 ref)
{
    if (timer == 0) {
        if ((start != 0) == (theY8950->counter1 == -1)) {
            theY8950->counter1  = start ? theY8950->timer1 : -1;
        }
    }
    else {
        if ((start != 0) == (theY8950->counter2 == -1)) {
            theY8950->counter2  = start ? theY8950->timer2 : -1;
        }
    }
}

void y8950Tick(UInt32 cpuClock) 
{
    if (theY8950 != NULL) {
        while (cpuClock - theY8950->clock >= 286) {
            theY8950->clock += 286;
     
            if (theY8950->counter1 != -1) {
                if (theY8950->counter1-- == 0) {
                    if (OPLTimerOver(theY8950->opl, 0)) {
                        theY8950->counter1 = theY8950->timer1;
                    }
                }
            }

            if (theY8950->counter2 != -1) {
                if (theY8950->counter2-- == 0) {
                    if (OPLTimerOver(theY8950->opl, 1)) {
                        theY8950->counter2 = theY8950->timer2;
                    }
                }
            }
        }
    }
}


