/*****************************************************************************
** File:        Win32.c
**
** Author:      Daniel Vik
**
** Description: The central file that ties everything together. Contains
**              the program entry point and the main window.
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#define DIRECTINPUT_VERSION     0x0500

#include <windows.h>
#include <tchar.h>
#include <direct.h>
#include <math.h>
#include <CommCtrl.h>
#include <Mmsystem.h>
 
#include "MSX.h"
#include "Machine.h"
#include "audioMixer.h"
#include "videoRender.h"
#include "Language.h"   
#include "resource.h"
#include "ziphelper.h"
#include "MegaromCartridge.h"
#include "Casette.h"
#include "JoystickIO.h"
#include "Disk.h"

#include "Win32Timer.h"
#include "Win32Sound.h"
#include "Win32Properties.h"
#include "Win32joystick.h"
#include "Win32keyboard.h"
#include "Win32directx.h"
#include "Win32FileHistory.h"
#include "Win32file.h"
#include "Win32Help.h"
#include "Win32Menu.h"
#include "Win32StatusBar.h"
#include "Win32ToolBar.h"
#include "Win32ScreenShot.h"
#include "Win32WaveCapture.h"
#include "Win32MouseEmu.h"
#include "Win32machineConfig.h"


typedef enum { EMU_RUNNING, EMU_PAUSED, EMU_STOPPED, EMU_SUSPENDED } EmuState;


void emulatorSetFrequency(int logFrequency, int* syncPeriod, int* frequency);
void emulatorRestartSound(int stereo);
void emulatorSuspend();
void emulatorResume();
void emulatorReset();
void emulatorStart(int hard);
void emulatorStop();

typedef void (*KbdLockFun)(); 

KbdLockFun kbdLockEnable = NULL;
KbdLockFun kbdLockDisable = NULL;

#define WINKEYS_DISABLE() 

static Properties* pProperties;

typedef struct {
    HWND hwnd;
    HMENU hMenu;
    int showMenu;
    int showDialog;
    HBITMAP hBmp;
    HBITMAP hLogo;
    HBITMAP hLogoSmall;
    BITMAPINFO bmInfo;
    HWND dskWnd;
    
    // Frame buffer
    void* frameBuffer[2];
    int*  frameLine[2];
    int   frameBufferIndex;
    void* bmBits;
    int*  srcArr;

    //
    void* bmBitsGDI;
    UInt8 keyMap[16];
    EmuState emuState;
    int updatePending;
    char pStoredState[MAX_PATH];
    char pCurDir[MAX_PATH];
    Video* pVideo;
    int logSound;
    Mixer* mixer;
    int enteringFullscreen;
    DeviceInfo deviceInfo;
    Machine* machine;
    int screenMode;
    int evenOdd;
    int interlace;
    int autostart;
    int noautostart;
    UInt32 ledState;
    int maxSpeed;
    int maxSpeedCount;
    int mouseLock;
    int X;
    int Y;
    int DX;
    int DY;
    int DDY;
    int DTY;
} Port;

#define WM_UPDATE            (WM_USER + 1245)
#define WM_PRINTSCREEN       (WM_USER + 1246)
#define WM_SHOWDSKWIN        (WM_USER + 1247)

#define WIDTH  320
#define HEIGHT 240

static Port Prt;


static BOOL registerFileType(char* extension, char* appName, char* description, int iconIndex) {
    char path[MAX_PATH];
    char fileName[MAX_PATH];
    char buffer[MAX_PATH + 32];
    HKEY hKey;
    DWORD exist;
    DWORD rv;

    GetModuleFileName(GetModuleHandle(NULL), fileName, MAX_PATH);

    rv = RegCreateKeyEx(HKEY_CLASSES_ROOT, appName, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    rv = RegSetValueEx(hKey, "", 0, REG_SZ, description, strlen(description) + 1);
    RegCloseKey(hKey);

    rv = RegCreateKeyEx(HKEY_CLASSES_ROOT, extension, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    rv = RegSetValueEx(hKey, "", 0, REG_SZ, appName, strlen(appName) + 1);

    sprintf(buffer, "%s /onearg %%1", fileName);
    sprintf(path, "%s\\Shell\\Open\\command", appName);
    rv = RegCreateKeyEx(HKEY_CLASSES_ROOT, path, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    rv = RegSetValueEx(hKey, "", 0, REG_SZ, buffer, strlen(buffer) + 1);
    RegCloseKey(hKey);

    sprintf(path, "%s\\DefaultIcon", appName);
    sprintf(buffer, "%s,%d", fileName, iconIndex);
    rv = RegCreateKeyEx(HKEY_CLASSES_ROOT, path, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    rv = RegSetValueEx(hKey, "", 0, REG_SZ, buffer, strlen(buffer) + 1);
    RegCloseKey(hKey);

    return TRUE;
}

static void registerFileTypes() {
    registerFileType(".dsk", "blueMSXdsk", "DSK Image", 1);
    registerFileType(".di1", "blueMSXdsk", "DSK Image", 1);
    registerFileType(".di2", "blueMSXdsk", "DSK Image", 1);
    registerFileType(".rom", "blueMSXrom", "ROM Image", 2);
    registerFileType(".mx1", "blueMSXrom", "ROM Image", 2);
    registerFileType(".mx2", "blueMSXrom", "ROM Image", 2);
    registerFileType(".cas", "blueMSXcas", "CAS Image", 3);
    registerFileType(".sta", "blueMSXsta", "blueMSX State", 4);
}

void centerDialog(HWND hDlg) {
    RECT r1;
    RECT r2;
    int x;
    int y;

//    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        GetWindowRect(GetParent(hDlg), &r1);
        GetWindowRect(hDlg, &r2);

        x = r1.left + (r1.right - r1.left - r2.right + r2.left) / 2;
        y = r1.top  + (r1.bottom - r1.top - r2.bottom + r2.top) / 2;
        SetWindowPos(hDlg, NULL, x, y, 0, 0, SWP_NOZORDER | SWP_NOSIZE);
//    }
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

static int diskFileExist(char* fileName, char* zipFile) {
    if (zipFile == NULL || *zipFile == 0) {
        FILE* file = fopen(fileName, "r");
        if (file != NULL) {
            fclose(file);
            return 1;
        }
        return 0;
    }
    else {
        int size;
        char* buf = zipLoadFile(zipFile, fileName, &size);
        if (buf != NULL) {
            free(buf);
            return 1;
        }
        return 0;
    }

    return 0;
}

char* diskGetNext(char* diskName, char* zipFile) {
    static char name[512];
    static int pos = -1;
    int c;
    int j;

    strcpy(name, diskName);

    pos = strlen(name) - 5;

    if (pos < 0) {
        return name;
    }

    while (pos >= 0) {
        c = name[pos];
    
        if (c >= '0' && c <= '9') {
            if (c < '9') {
                name[pos] = c + 1;
                if (diskFileExist(name, zipFile)) {
                    return name;
                }
            }

            for (j = 0; j < c; j++) {
                name[pos] = j;
                if (diskFileExist(name, zipFile)) {
                    return name;
                }
            }
            name[pos] = c;
        }
        pos--;
    }

    pos = strlen(name) - 5;
    c = name[pos];

    if (c >= 'A' && c <= 'Z') {
        if (c < 'Z') {
            name[pos] = c + 1;
            if (diskFileExist(name, zipFile)) {
                pos = -1;
                return name;
            }
        }

        for (j = 'A'; j <= c; j++) {
            name[pos] = j;
            if (diskFileExist(name, zipFile)) {
                pos = -1;
                return name;
            }
        }
    }

    if (c >= 'a' && c <= 'z') {
        if (c < 'z') {
            name[pos] = c + 1;
            if (diskFileExist(name, zipFile)) {
                pos = -1;
                return name;
            }
        }

        for (j = 'a'; j <= c; j++) {
            name[pos] = j;
            if (diskFileExist(name, zipFile)) {
                pos = -1;
                return name;
            }
        }
    }

    return name;
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

static int fileIndex = 0;
static char baseDir[512];
static char baseFileName[512];

static char* quickSaveCreateFileBase()
{
    static char fileBase[128];

    if (*pProperties->cartridge.slotA) {
        if (strcmp(pProperties->cartridge.slotA, CARTNAME_SCCPLUS) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_FMPAC) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_PAC) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_MEGARAM128) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_MEGARAM256) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_MEGARAM512) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_MEGARAM768) &&
            strcmp(pProperties->cartridge.slotA, CARTNAME_MEGARAM2M)) 
        {
            if (*pProperties->cartridge.slotAZip) {
                strcpy(fileBase, stripPathExt(pProperties->cartridge.slotAZip));
                return fileBase;
            }
            strcpy(fileBase, stripPathExt(pProperties->cartridge.slotA));
            return fileBase;
        }
    }

    if (*pProperties->cartridge.slotB) {
        if (strcmp(pProperties->cartridge.slotB, CARTNAME_SCCPLUS) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_FMPAC) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_PAC) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_MEGARAM128) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_MEGARAM256) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_MEGARAM512) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_MEGARAM768) &&
            strcmp(pProperties->cartridge.slotB, CARTNAME_MEGARAM2M)) 
        {
            if (*pProperties->cartridge.slotBZip) {
                strcpy(fileBase, stripPathExt(pProperties->cartridge.slotBZip));
                return fileBase;
            }
            strcpy(fileBase, stripPathExt(pProperties->cartridge.slotB));
            return fileBase;
        }
    }
    
    if (*pProperties->diskdrive.slotA) {
        if (*pProperties->diskdrive.slotAZip) {
            strcpy(fileBase, pProperties->diskdrive.slotAZip);
            return fileBase;
        }
        strcpy(fileBase, stripPathExt(pProperties->diskdrive.slotA));
        return fileBase;
    }
    
    if (*pProperties->diskdrive.slotB) {
        if (*pProperties->diskdrive.slotBZip) {
            strcpy(fileBase, pProperties->diskdrive.slotBZip);
            return fileBase;
        }
        strcpy(fileBase, stripPathExt(pProperties->diskdrive.slotB));
        return fileBase;
    }
    
    if (*pProperties->cassette.tape) {
        if (*pProperties->cassette.tapeZip) {
            strcpy(fileBase, pProperties->cassette.tapeZip);
            return fileBase;
        }
        strcpy(fileBase, stripPathExt(pProperties->cassette.tape));
        return fileBase;
    }

    return "empty";
}

void quickSaveInit(char* directory)
{
	WIN32_FIND_DATA fileData;
    FILETIME writeTime;
    HANDLE hFile;
	char lastfile[512];
    char filename[512];
	char filenum[12];

    // save directory
    strcpy(baseDir,      directory);
    strcpy(baseFileName, quickSaveCreateFileBase());

	// screenshot ammount
	fileIndex = 0;

	mkdir(baseDir);

    sprintf(filename, "%s\\%s??.sta", baseDir, baseFileName);

    hFile = FindFirstFile(filename,&fileData);
    if (hFile == INVALID_HANDLE_VALUE) {
       fileIndex = 0;
    }
	else {
        writeTime = fileData.ftLastWriteTime;
		strcpy(lastfile, fileData.cFileName);
	    while (FindNextFile(hFile, &fileData ) != 0) {
            if (CompareFileTime(&fileData.ftLastWriteTime, &writeTime) < 0) {
                break;
            }
            writeTime = fileData.ftLastWriteTime;
            strcpy(lastfile,fileData.cFileName);
		}
		// now make some sense out of it
		filenum[0]=lastfile[strlen(lastfile) - 6];
		filenum[1]=lastfile[strlen(lastfile) - 5];
		filenum[2]='\0';
		fileIndex=atoi(filenum);
    }
}

char* quickSaveCreateFileName()
{
	static char fname[512];
    
    sprintf(fname, "%s\\QuickSave", Prt.pCurDir);
    quickSaveInit(fname);

    fileIndex++;
    if (fileIndex > 99) fileIndex = 0;
	sprintf(fname, "%s\\%s%02i.sta", baseDir, baseFileName, fileIndex);

    return fname;
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

static int getZoom() {
    switch (pProperties->video.size) {
    case P_VIDEO_SIZEX1:
        return 1;
    case P_VIDEO_SIZEX2:
        return 2;
    case P_VIDEO_SIZEFULLSCREEN:
        switch (pProperties->video.fullRes) {
        case P_VIDEO_FRES320X240_16:
        case P_VIDEO_FRES320X240_32:
            return 1;
        case P_VIDEO_FRES640X480_16:
        case P_VIDEO_FRES640X480_32:
            return 2;
        }
    }
    return 1;
}

static int tempStateExists() {
    if (*pProperties->filehistory.quicksave) {
        FILE* file;

        file = fopen(pProperties->filehistory.quicksave, "r");
        if (file != NULL) {
            fclose(file);
            return 1;
        }
    }

    return 0;
}

static void updateMenu(int show) {
    int doDelay = show;

    if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
        show = 1;
    }

    if (pProperties->video.driver != P_VIDEO_DRVGDI) {
         if (doDelay) { 
             Sleep(150);
             DirectXSetGDISurface();
             Sleep(150);
         }
    }

    Prt.showMenu = menuShow(Prt.hwnd,
                            pProperties, 
                            Prt.emuState == EMU_RUNNING, 
                            Prt.emuState == EMU_STOPPED, 
                            Prt.logSound,
                            tempStateExists(),
                            show);

    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        mouseEmuActivate(!show);
    }

    InvalidateRect(Prt.hwnd, NULL, 0);
}

static void enterDialogShow() {
    if (pProperties->video.driver != P_VIDEO_DRVGDI) {
        Sleep(150);
        DirectXSetGDISurface();
        Sleep(150);
    }
    Prt.showDialog++;
}

static void exitDialogShow() {
    Prt.showDialog--;
}

static void ScaleWindow(HWND hwnd) {
    RECT r;
    int x, y, w, h;
    int zoom = getZoom();

    Prt.enteringFullscreen = 1;

    DirectXExitFullscreenMode(hwnd);

    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        statusBarShow(FALSE);
        toolBarShow(FALSE);
    
        SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_POPUP | WS_THICKFRAME | WS_DLGFRAME);

        if (pProperties->video.driver != P_VIDEO_DRVGDI) {
            int rv;
            int depth = 32;
            switch (pProperties->video.fullRes) {
            case P_VIDEO_FRES320X240_16:
            case P_VIDEO_FRES640X480_16:
                depth = 16;
                break;
            case P_VIDEO_FRES320X240_32:
            case P_VIDEO_FRES640X480_32:
                depth = 32;
                break;
            }
            rv = DirectXEnterFullscreenMode(hwnd, zoom * WIDTH, zoom * HEIGHT, depth, 
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEO ||
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM, 
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM);
            if (rv != DXE_OK) {
                DirectXExitFullscreenMode(hwnd);
                rv = DirectXEnterFullscreenMode(hwnd, 2 * WIDTH, 2 * HEIGHT, 16, 
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEO ||
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM, 
                                            pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM);
            }
            if (rv != DXE_OK) {
                MessageBox(NULL, langErrorEnterFullscreen(), langErrorTitle(), MB_OK);
                DirectXExitFullscreenMode(hwnd);
                pProperties->video.size = P_VIDEO_SIZEX2;
                mouseEmuActivate(1);
                SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                                WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX);
                SetWindowPos(hwnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_SHOWWINDOW); 

                statusBarShow(TRUE);
                toolbarUpdate(zoom == 2, Prt.emuState == EMU_RUNNING);
                toolBarShow(TRUE);

                w = WIDTH * zoom + Prt.DX;
                h = HEIGHT * zoom + Prt.DY + Prt.DTY + Prt.DDY;
                SetWindowPos(hwnd, NULL, Prt.X, Prt.Y, w, h, SWP_NOZORDER);
                
                rv = DirectXEnterWindowedMode(Prt.hwnd, zoom * WIDTH, zoom * HEIGHT, 
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEO ||
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM, 
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM);
                if (rv != DXE_OK) {
                    pProperties->video.driver = P_VIDEO_DRVGDI;
                }
                Prt.enteringFullscreen = 0;

                return;
            }
        }

        x = -1 * GetSystemMetrics(SM_CXFIXEDFRAME) - 1;
        y = -1 * GetSystemMetrics(SM_CYFIXEDFRAME) - 1;
        w = GetSystemMetrics(SM_CXSCREEN) - 2 * x;
        h = GetSystemMetrics(SM_CYSCREEN) - 2 * y;

        if (pProperties->video.driver != P_VIDEO_DRVGDI) {
            SetWindowPos(hwnd, 0, x, y, w, h, SWP_SHOWWINDOW | SWP_NOZORDER);
        }
        else {
            SetWindowPos(hwnd, HWND_TOPMOST, x, y, w, h, SWP_SHOWWINDOW); 
        }
    }
    else {
        toolbarUpdate(zoom == 2, Prt.emuState == EMU_RUNNING);
        if (GetWindowLong(hwnd, GWL_STYLE) & WS_POPUP) {
            mouseEmuActivate(1);
            SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX);
            SetWindowPos(hwnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_SHOWWINDOW); 
        }

        w = WIDTH * zoom + Prt.DX;
        h = HEIGHT * zoom + Prt.DY + Prt.DTY + Prt.DDY;
        SetWindowPos(hwnd, NULL, Prt.X, Prt.Y, w, h, SWP_NOZORDER);
        
        statusBarShow(TRUE);
        toolBarShow(TRUE);

        if (pProperties->video.driver != P_VIDEO_DRVGDI) {
            int rv = DirectXEnterWindowedMode(Prt.hwnd, zoom * WIDTH, zoom * HEIGHT, 
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEO ||
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM, 
                                              pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM);
            if (rv != DXE_OK) {
                MessageBox(NULL, langErrorDirectXFailed(), langErrorTitle(), MB_OK);
                pProperties->video.driver = P_VIDEO_DRVGDI;
            }
        }
    }

    Prt.enteringFullscreen = 0;

    GetClientRect(Prt.hwnd, &r);
    r.left += 5;
    r.top += 5;
    r.right -= 5;
    r.bottom -= 5;

    if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
        r.top += Prt.DTY;
    }

    updateMenu(0);
    mouseEmuSetCaptureInfo(&r);
}

void updateVideoRender() {
    switch (pProperties->video.monType) {
    case P_VIDEO_COLOR:
        videoSetColorMode(Prt.pVideo, VIDEO_COLOR);
        break;
    case P_VIDEO_BW:
        videoSetColorMode(Prt.pVideo, VIDEO_BLACKWHITE);
        break;
    case P_VIDEO_GREEN:
        videoSetColorMode(Prt.pVideo, VIDEO_GREEN);
        break;
    }

    switch (pProperties->video.palEmu) {
    case P_VIDEO_PALNONE:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_FAST);
        break;
    case P_VIDEO_PALYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP);
        break;
    case P_VIDEO_PALNYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP_NOISE);
        break;
    case P_VIDEO_PALCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR);
        break;
    case P_VIDEO_PALNCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR_NOISE);
        break;
	case P_VIDEO_PALSCALE2X:
		videoSetPalMode(Prt.pVideo, VIDEO_PAL_SCALE2X);
		break;
	case P_VIDEO_PAL_STRETCHED:
		videoSetPalMode(Prt.pVideo, VIDEO_PAL_STRETCHED);
		break;
    }

    videoSetFrameSkip(Prt.pVideo, pProperties->video.frameSkip);
}

void updateJoystick() {    
    switch (pProperties->joy1.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(1, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(1, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(1, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(1, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(1, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(1, JOY_HW2);
        break;
    }

    JoystickSetKeyStateKey(1, JOY_UP,    pProperties->joy1.keyUp);
    JoystickSetKeyStateKey(1, JOY_DOWN,  pProperties->joy1.keyDown);
    JoystickSetKeyStateKey(1, JOY_LEFT,  pProperties->joy1.keyLeft);
    JoystickSetKeyStateKey(1, JOY_RIGHT, pProperties->joy1.keyRight);
    JoystickSetKeyStateKey(1, JOY_BT1,   pProperties->joy1.button1);
    JoystickSetKeyStateKey(1, JOY_BT2,   pProperties->joy1.button2);

    JoystickSetKeyStateKey(2, JOY_UP,    pProperties->joy2.keyUp);
    JoystickSetKeyStateKey(2, JOY_DOWN,  pProperties->joy2.keyDown);
    JoystickSetKeyStateKey(2, JOY_LEFT,  pProperties->joy2.keyLeft);
    JoystickSetKeyStateKey(2, JOY_RIGHT, pProperties->joy2.keyRight);
    JoystickSetKeyStateKey(2, JOY_BT1,   pProperties->joy2.button1);
    JoystickSetKeyStateKey(2, JOY_BT2,   pProperties->joy2.button2);

    switch (pProperties->joy1.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(1, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(1, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(1, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(1, JOY_AF_FAST);
        break;
    }

    switch (pProperties->joy2.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(2, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(2, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(2, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(2, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(2, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(2, JOY_HW2);
        break;
    }

    switch (pProperties->joy2.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(2, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(2, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(2, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(2, JOY_AF_FAST);
        break;
    }
}

void showPropertiesDialog(int startPane) {
    Properties oldProp = *pProperties;
    int restart = 0;
    int changed;
    int i;

    changed = showProperties(pProperties, Prt.hwnd, startPane, Prt.mixer);
    exitDialogShow();
    if (!changed) {
        return;
    }

    /* Save properties */
    propSave(pProperties);

    /* Always update video render */
    updateVideoRender();

    /* Always update joystick controls */
    updateJoystick();

    /* Update window size only if changed */
    if (pProperties->video.driver != oldProp.video.driver ||
        pProperties->video.fullRes != oldProp.video.fullRes ||
        pProperties->video.size != oldProp.video.size ||
        pProperties->video.horizontalStretch != oldProp.video.horizontalStretch ||
        pProperties->video.verticalStretch != oldProp.video.verticalStretch)
    {
        ScaleWindow(Prt.hwnd);
    }

    joystickIoSetType(0, pProperties->joy1.type == P_JOY_NONE  ? 0 : pProperties->joy1.type == P_JOY_MOUSE ? 2 : 1, pProperties->joy1.type);
    joystickIoSetType(1, pProperties->joy2.type == P_JOY_NONE  ? 0 : pProperties->joy2.type == P_JOY_MOUSE ? 2 : 1, pProperties->joy2.type);

    mouseEmuEnable(pProperties->joy1.type == P_JOY_MOUSE || pProperties->joy2.type == P_JOY_MOUSE);

    /* Must restart MSX if CPU or ram configuration changed */
    if (memcmp(&oldProp.emulation, &pProperties->emulation, sizeof(oldProp.emulation))) {
        restart = 1;
    }

    /* Update sound only if changed, Must restart if changed */
    if (oldProp.sound.bufSize              != pProperties->sound.bufSize ||
        oldProp.sound.driver               != pProperties->sound.driver  ||
        oldProp.sound.frequency            != pProperties->sound.frequency ||
        oldProp.sound.chip.enableY8950     != pProperties->sound.chip.enableY8950 ||
        oldProp.sound.chip.enableYM2413    != pProperties->sound.chip.enableYM2413 ||
        oldProp.sound.chip.enableMoonsound != pProperties->sound.chip.enableMoonsound ||
        oldProp.sound.stereo               != pProperties->sound.stereo) 
    {
        emulatorRestartSound(pProperties->sound.stereo);
    }

    if (oldProp.sound.chip.enableY8950     != pProperties->sound.chip.enableY8950 ||
        oldProp.sound.chip.enableYM2413    != pProperties->sound.chip.enableYM2413 ||
        oldProp.sound.chip.enableMoonsound != pProperties->sound.chip.enableMoonsound)
    {
        restart = 1;
    }

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }
    
    if (pProperties->emulation.registerFileTypes && !oldProp.emulation.registerFileTypes) {
        registerFileTypes();
    }
    
    if (pProperties->emulation.disableWinKeys && !oldProp.emulation.disableWinKeys) {
        if (kbdLockEnable && Prt.emuState == EMU_RUNNING && pProperties->emulation.disableWinKeys) {
            kbdLockEnable();
        }
        else {
            kbdLockDisable();
        }
    }

    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    if (restart) {
        emulatorReset();
    }

    if (oldProp.settings.disableScreensaver != pProperties->settings.disableScreensaver) {
        POINT pt;
        SystemParametersInfo(SPI_SETSCREENSAVEACTIVE, !pProperties->settings.disableScreensaver, 0, SPIF_SENDWININICHANGE); 
        GetCursorPos(&pt);
        SetCursorPos(pt.x + 1, pt.y);
    }

    updateMenu(0);

    InvalidateRect(Prt.hwnd, NULL, TRUE);
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
typedef struct {
    _TCHAR title[128];
    _TCHAR description[128];
    char* fileList;
    int   fileListCount;
    int   autoReset;
    char  selectFile[512];
    int   selectFileIndex;
} ZipFileDlgInfo;

static BOOL CALLBACK dskZipDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static ZipFileDlgInfo* dlgInfo;

    switch (iMsg) {
    case WM_INITDIALOG:
        {
            char* fileList;
            int sel = 0;
            int i;
            
            centerDialog(hDlg);
            dlgInfo = (ZipFileDlgInfo*)lParam;

            SetWindowText(hDlg, dlgInfo->title);

            SendMessage(GetDlgItem(hDlg, IDC_DSKLOADTXT), WM_SETTEXT, 0, (LPARAM)dlgInfo->description);
            SetWindowText(GetDlgItem(hDlg, IDC_DSKRESET), langDlgZipReset());
            SetWindowText(GetDlgItem(hDlg, IDOK), langDlgOK());
            SetWindowText(GetDlgItem(hDlg, IDCANCEL), langDlgCancel());

            fileList = dlgInfo->fileList;

            if (dlgInfo->selectFileIndex != -1) {
                sel = dlgInfo->selectFileIndex;
            }

            for (i = 0; i < dlgInfo->fileListCount; i++) {
                if (dlgInfo->selectFileIndex != -1 && 0 == strcmp(dlgInfo->selectFile, fileList)) {
                    sel = i;
                }
                SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_ADDSTRING, 0, (LPARAM)fileList);
                fileList += strlen(fileList) + 1;
            }

            if (dlgInfo->autoReset == -1) {
                ShowWindow(GetDlgItem(hDlg, IDC_DSKRESET), FALSE);
            }
            else {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, dlgInfo->autoReset ? BST_CHECKED : BST_UNCHECKED, 0);
            }
            SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_SETCURSEL, sel, 0);

            return FALSE;
        }

    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDC_DSKRESET:
            if (dlgInfo->autoReset == 1) {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, BST_UNCHECKED, 0);
                dlgInfo->autoReset = 0;
            }
            else if (dlgInfo->autoReset == 0) {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, BST_CHECKED, 1);
                dlgInfo->autoReset = 1;
            }
            break;
        case IDC_DSKLIST:
            if (HIWORD(wParam) != 2) {
                break;
            }
            // else, fall through
        case IDOK:
            dlgInfo->selectFileIndex = SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_GETCURSEL, 0, 0);
            SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_GETTEXT, dlgInfo->selectFileIndex, (LPARAM)dlgInfo->selectFile);
            EndDialog(hDlg, TRUE);
            return TRUE;
        case IDCANCEL:
            dlgInfo->selectFileIndex = -1;
            dlgInfo->selectFile[0] = '\0';
            EndDialog(hDlg, FALSE);
            return TRUE;
        }
        break;
    case WM_CLOSE:
        dlgInfo->selectFileIndex = -1;
        dlgInfo->selectFile[0] = '\0';
        EndDialog(hDlg, FALSE);
        return TRUE;
    }

    return FALSE;
}

static char* convertTapePos(int tapePos)
{
    static char str[64];
    int pos = tapePos / 128;

    _stprintf(str, "%dh %02dm %02ds", pos / 3600, (pos / 60) % 60, pos % 60);
    return str;
}

static BOOL CALLBACK tapePosDlg(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) 
{
    static int currIndex;
    static HWND hwnd;
    static TapeContent* tc;
    static int tcCount;

    switch (iMsg) {
    case WM_INITDIALOG:
        {
            char buffer[32];
            LV_COLUMN lvc = {0};
         
            centerDialog(hDlg);

            currIndex = -1;

            SetWindowText(hDlg, langDlgTapeTitle());

            SendMessage(GetDlgItem(hDlg, IDC_SETTAPEPOSTXT), WM_SETTEXT, 0, (LPARAM)langDlgTapeSetPosText());
            SetWindowText(GetDlgItem(hDlg, IDC_SETTAPECUSTOM), langDlgTapeCustom());
            SetWindowText(GetDlgItem(hDlg, IDOK), langDlgOK());
            SetWindowText(GetDlgItem(hDlg, IDCANCEL), langDlgCancel());

            SendMessage(GetDlgItem(hDlg, IDC_SETTAPECUSTOM), BM_SETCHECK, pProperties->cassette.showCustomFiles ? BST_CHECKED : BST_UNCHECKED, 0);

            tc = tapeGetContent(&tcCount);
    
            hwnd = GetDlgItem(hDlg, IDC_SETTAPELIST);

            EnableWindow(GetDlgItem(hDlg, IDOK), FALSE);

            ListView_SetExtendedListViewStyle(hwnd, LVS_EX_FULLROWSELECT);

            lvc.mask       = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
            lvc.fmt        = LVCFMT_LEFT;
            lvc.cx         = 100;
            lvc.pszText    = buffer;
	        lvc.cchTextMax = 32;

            sprintf(buffer, langDlgTabPosition());
            lvc.cx = 95;
            ListView_InsertColumn(hwnd, 0, &lvc);
            sprintf(buffer, langDlgTabType());
            lvc.cx = 65;
            ListView_InsertColumn(hwnd, 1, &lvc);
            sprintf(buffer, langDlgTabFilename());
            lvc.cx = 105;
            ListView_InsertColumn(hwnd, 2, &lvc);
        }

        SendMessage(hDlg, WM_UPDATE, 0, 0);
        return FALSE;

    case WM_UPDATE:
        {
            static char* typeNames[] = { "ASCII", "BIN", "BAS", "Custom" };
            int curPos;
            int i;

            ListView_DeleteAllItems(hwnd);

            curPos = tapeGetCurrentPos();

            for (i = 0; i < tcCount; i++) {
                char buffer[64] = {0};
                LV_ITEM lvi = {0};

                if (pProperties->cassette.showCustomFiles || tc[i].type != TAPE_CUSTOM) {
                    lvi.mask       = LVIF_TEXT;
                    lvi.iItem      = i;
                    lvi.pszText    = buffer;
	                lvi.cchTextMax = 64;
                    
                    sprintf(buffer, convertTapePos(tc[i].pos));
                    ListView_InsertItem(hwnd, &lvi);
                    lvi.iSubItem++;
                    
                    sprintf(buffer, typeNames[tc[i].type]);
                    ListView_SetItem(hwnd, &lvi);
                    lvi.iSubItem++;
                    
                    sprintf(buffer, tc[i].fileName);
                    ListView_SetItem(hwnd, &lvi);

                    if (tc[i].pos <= curPos) {
                        SetFocus(hwnd);
                        ListView_SetItemState(hwnd, i, LVIS_SELECTED, LVIS_SELECTED);
                    }
                }
            }
        }
        return TRUE;

    case WM_NOTIFY:
        switch (wParam) {
        case IDC_SETTAPELIST:
            if ((((NMHDR FAR *)lParam)->code) == LVN_ITEMCHANGED) {
                if (ListView_GetSelectedCount(hwnd)) {
                    int index = ListView_GetNextItem(hwnd, -1, LVNI_SELECTED);

                    if (currIndex == -1 && index != -1) {
                        EnableWindow(GetDlgItem(hDlg, IDOK), TRUE);
                    }
                    currIndex = index;
                }
                else {
                    if (currIndex != -1) {
                        EnableWindow(GetDlgItem(hDlg, IDOK), FALSE);
                    }
                    currIndex = -1;
                }
            }
            if ((((NMHDR FAR *)lParam)->code) == LVN_ITEMACTIVATE) {
                if (ListView_GetSelectedCount(hwnd)) {
                    int index = ListView_GetNextItem(hwnd, -1, LVNI_SELECTED);
                    SendMessage(hDlg, WM_COMMAND, IDOK, 0);
                }
                return TRUE;
            }
        }
        break;

    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDC_SETTAPECUSTOM:
            pProperties->cassette.showCustomFiles = BST_CHECKED == SendMessage(GetDlgItem(hDlg, IDC_SETTAPECUSTOM), BM_GETCHECK, 0, 0);
            SendMessage(hDlg, WM_UPDATE, 0, 0);
            return TRUE;

        case IDCANCEL:
            EndDialog(hDlg, FALSE);
            return TRUE;

        case IDOK:
            {
                int index = 0;
                int i;

                if (ListView_GetSelectedCount(hwnd)) {
                    currIndex = ListView_GetNextItem(hwnd, -1, LVNI_SELECTED);
                }

                for (i = 0; i < tcCount; i++) {
                    if (pProperties->cassette.showCustomFiles || tc[i].type != TAPE_CUSTOM) {
                        if (currIndex == index) {
                            tapeSetCurrentPos(tc[i].pos);
                        }
                        index++;
                    }
                }
             
                EndDialog(hDlg, TRUE);
            }
            return TRUE;
        }
        return FALSE;

    case WM_CLOSE:
        EndDialog(hDlg, FALSE);
        return TRUE;
    }

    return FALSE;
}


void setTapePosition() {
    if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
    }
    else {
        tapeSetReadOnly(1);
        msxChangeCassette(strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL, 
                          strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL);
    }

    DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_SETTAPEPOS), Prt.hwnd, tapePosDlg);

    if (Prt.emuState != EMU_STOPPED) {
        emulatorResume();
    }
    else {
        msxChangeCassette(NULL, NULL);
        tapeSetReadOnly(pProperties->cassette.readOnly);
    }
}


static int insertCartridge(int drive, char* fname, char* inZipFile) {
    int autostart = Prt.autostart | pProperties->cartridge.autoReset;
    char romName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(romName, inZipFile);
        }
        else {
            // This is really ugly and should be done nicer...
            // The idea is to get files of three types and merge them into
            // one list. Maybe it should be done in zipGetFileList?
            int i;
            char* fileList;
            int countRom;
            int countMx1;
            int countMx2;
            char* fileListRom = zipGetFileList(filename, ".rom", &countRom);
            char* fileListMx1 = zipGetFileList(filename, ".mx1", &countMx1);
            char* fileListMx2 = zipGetFileList(filename, ".mx2", &countMx2);
            int count = countRom + countMx1 + countMx2;
            int sizeRom = 0;
            int sizeMx1 = 0;
            int sizeMx2 = 0;

            for (i = 0; i < countRom; i++) {
                sizeRom += strlen(fileListRom + sizeRom) + 1;
            }
            for (i = 0; i < countMx1; i++) {
                sizeMx1 += strlen(fileListMx1 + sizeMx1) + 1;
            }
            for (i = 0; i < countMx2; i++) {
                sizeMx2 += strlen(fileListMx2 + sizeMx2) + 1;
            }

            fileList = malloc(sizeRom + sizeMx1 + sizeMx2);
            memcpy(fileList, fileListRom, sizeRom);
            memcpy(fileList + sizeRom, fileListMx1, sizeMx1);
            memcpy(fileList + sizeRom + sizeMx1, fileListMx2, sizeMx2);

            if (count == 0) {
                MessageBox(NULL, langErrorNoRomInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(romName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadRom());
                _stprintf(dlgInfo.description, "%s", langDlgLoadRomDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                dlgInfo.selectFileIndex = -1;
                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->cartridge.slotAZip : pProperties->cartridge.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                autostart = dlgInfo.autoReset;
                strcpy(romName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }

    if (drive == 0) {
        strcpy(pProperties->cartridge.slotA, filename);
        strcpy(pProperties->cartridge.slotAZip, romName);
        updateFileHistory(*pProperties->filehistory.cartridgeA, filename);
    }
    else {
        strcpy(pProperties->cartridge.slotB, filename);
        strcpy(pProperties->cartridge.slotBZip, romName);
        updateFileHistory(*pProperties->filehistory.cartridgeB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        msxChangeCartridge(drive, 0 == strcmp(CARTNAME_SCCPLUS, filename) ? ROM_SCCPLUS :
                                  0 == strcmp(CARTNAME_FMPAC, filename) ? ROM_FMPAC :
                                  0 == strcmp(CARTNAME_PAC, filename) ? ROM_PAC :
                                  0 == strcmp(CARTNAME_MEGARAM128, filename) ? ROM_MEGARAM128 :
                                  0 == strcmp(CARTNAME_MEGARAM256, filename) ? ROM_MEGARAM256 :
                                  0 == strcmp(CARTNAME_MEGARAM512, filename) ? ROM_MEGARAM512 :
                                  0 == strcmp(CARTNAME_MEGARAM768, filename) ? ROM_MEGARAM768 :
                                  0 == strcmp(CARTNAME_MEGARAM2M,  filename) ? ROM_MEGARAM2M  :
                                  ROM_UNKNOWN, 
                                  filename, isZip ? romName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertDiskette(int drive, char* fname, char* inZipFile) {
    char diskName[512] = "";
    char filename[512] = "";
    int autostart = Prt.autostart | (drive == 0 ? pProperties->diskdrive.autostartA : 0);
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(diskName, inZipFile);
        }
        else {
            // This is really ugly and should be done nicer...
            // The idea is to get files of three types and merge them into
            // one list. Maybe it should be done in zipGetFileList?
            int i;
            char* fileList;
            int countDsk;
            int countDi1;
            int countDi2;
            char* fileListDsk = zipGetFileList(filename, ".dsk", &countDsk);
            char* fileListDi1 = zipGetFileList(filename, ".di1", &countDi1);
            char* fileListDi2 = zipGetFileList(filename, ".di2", &countDi2);
            int count = countDsk + countDi1 + countDi2;
            int sizeDsk = 0;
            int sizeDi1 = 0;
            int sizeDi2 = 0;

            for (i = 0; i < countDsk; i++) {
                sizeDsk += strlen(fileListDsk + sizeDsk) + 1;
            }
            for (i = 0; i < countDi1; i++) {
                sizeDi1 += strlen(fileListDi1 + sizeDi1) + 1;
            }
            for (i = 0; i < countDi2; i++) {
                sizeDi2 += strlen(fileListDi2 + sizeDi2) + 1;
            }

            fileList = malloc(sizeDsk + sizeDi1 + sizeDi2);
            memcpy(fileList, fileListDsk, sizeDsk);
            memcpy(fileList + sizeDsk, fileListDi1, sizeDi1);
            memcpy(fileList + sizeDsk + sizeDi1, fileListDi2, sizeDi2);

            if (count == 0) {
                MessageBox(NULL, langErrorNoDskInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(diskName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadDsk());
                _stprintf(dlgInfo.description, "%s", langDlgLoadDskDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                dlgInfo.selectFileIndex = -1;
                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->diskdrive.slotAZip : pProperties->diskdrive.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(diskName, dlgInfo.selectFile);
                autostart = dlgInfo.autoReset;
            }
            free(fileList);
        }
    }

    if (drive == 0) {
        strcpy(pProperties->diskdrive.slotA, filename);
        strcpy(pProperties->diskdrive.slotAZip, diskName);
        updateFileHistory(*pProperties->filehistory.diskdriveA, filename);
    }
    else {
        strcpy(pProperties->diskdrive.slotB, filename);
        strcpy(pProperties->diskdrive.slotBZip, diskName);
        updateFileHistory(*pProperties->filehistory.diskdriveB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        msxChangeDiskette(drive, filename, isZip ? diskName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertCassette(char* fname, char* inZipFile) {
    int autostart = Prt.autostart;
    char tapeName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;

    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(tapeName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".cas", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoCasInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(tapeName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadCas());
                _stprintf(dlgInfo.description, "%s", langDlgLoadCasDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                dlgInfo.selectFileIndex = -1;
                strcpy(dlgInfo.selectFile, pProperties->cassette.tapeZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                autostart = dlgInfo.autoReset;
                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(tapeName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }

    strcpy(pProperties->cassette.tape, filename);
    strcpy(pProperties->cassette.tapeZip, tapeName);
    updateFileHistory(*pProperties->filehistory.cassette, filename);

    if (autostart && !Prt.noautostart) {
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        msxChangeCassette(filename, isZip ? tapeName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertDisketteOrCartridge(int drive, char* filename) {
    ZipFileDlgInfo dlgInfo;
    int countDsx;
    int countDi1;
    int countDi2;
    int countRox;
    int countMx1;
    int countMx2;
    int countCas;
    char* fileListDsk = NULL;
    char* fileListRom = NULL;
    char* fileListDsx = zipGetFileList(filename, ".dsk", &countDsx);
    char* fileListDi1 = zipGetFileList(filename, ".di1", &countDi1);
    char* fileListDi2 = zipGetFileList(filename, ".di2", &countDi2);
    char* fileListRox = zipGetFileList(filename, ".rom", &countRox);
    char* fileListMx1 = zipGetFileList(filename, ".mx1", &countMx1);
    char* fileListMx2 = zipGetFileList(filename, ".mx2", &countMx2);
    char* fileListCas = zipGetFileList(filename, ".cas", &countCas);
    int countRom = countRox + countMx1 + countMx2;
    int countDsk = countDsx + countDi1 + countDi2;
    char* fileList;
    int sizeDsk = 0;
    int sizeDsx = 0;
    int sizeDi1 = 0;
    int sizeDi2 = 0;
    int sizeRox = 0;
    int sizeRom = 0;
    int sizeMx1 = 0;
    int sizeMx2 = 0;
    int sizeCas = 0;
    int success = 0;
    int i;

    // First merge different dsk formats into one list
    for (i = 0; i < countDsx; i++) {
        sizeDsx += strlen(fileListDsx + sizeDsx) + 1;
    }
    for (i = 0; i < countDi1; i++) {
        sizeDi1 += strlen(fileListDi1 + sizeDi1) + 1;
    }
    for (i = 0; i < countDi2; i++) {
        sizeDi2 += strlen(fileListDi2 + sizeDi2) + 1;
    }

    if (countDsk > 0) {
        fileListDsk = malloc(sizeDsx + sizeDi1 + sizeDi2);
        memcpy(fileListDsk, fileListDsx, sizeDsx);
        memcpy(fileListDsk + sizeDsx, fileListDi1, sizeDi1);
        memcpy(fileListDsk + sizeDsx + sizeDi1, fileListDi2, sizeDi2);
    }

    // Then merge different dsk formats into one list
    for (i = 0; i < countRox; i++) {
        sizeRox += strlen(fileListRox + sizeRox) + 1;
    }
    for (i = 0; i < countMx1; i++) {
        sizeMx1 += strlen(fileListMx1 + sizeMx1) + 1;
    }
    for (i = 0; i < countMx2; i++) {
        sizeMx2 += strlen(fileListMx2 + sizeMx2) + 1;
    }

    if (countRom > 0) {
        fileListRom = malloc(sizeRox + sizeMx1 + sizeMx2);
        memcpy(fileListRom, fileListRox, sizeRox);
        memcpy(fileListRom + sizeRox, fileListMx1, sizeMx1);
        memcpy(fileListRom + sizeRox + sizeMx1, fileListMx2, sizeMx2);
    }

    // Finally check different types...
    if (fileListDsk == NULL && fileListRom == NULL && fileListCas == NULL) {
        return 0;
    }

    if (fileListDsk == NULL && fileListCas == NULL) {
        free(fileListRom);
        return insertCartridge(drive, filename, NULL);
    }

    if (fileListRom == NULL && fileListCas == NULL) {
        free(fileListDsk);
        return insertDiskette(drive, filename, NULL);
    }

    if (fileListRom == NULL && fileListDsk == NULL) {
        free(fileListCas);
        return insertCassette(filename, NULL);
    }

    for (i = 0; i < countRom; i++) {
        sizeRom += strlen(fileListRom + sizeRom) + 1;
    }

    for (i = 0; i < countDsk; i++) {
        sizeDsk += strlen(fileListDsk + sizeDsk) + 1;
    }

    for (i = 0; i < countCas; i++) {
        sizeCas += strlen(fileListCas + sizeCas) + 1;
    }

    fileList = malloc(sizeDsk + sizeRom + sizeCas);
    memcpy(fileList, fileListRom, sizeRom);
    memcpy(fileList + sizeRom, fileListDsk, sizeDsk);
    memcpy(fileList + sizeRom + sizeDsk, fileListCas, sizeCas);

    _stprintf(dlgInfo.title, "%s", langDlgLoadRomDskCas());
    _stprintf(dlgInfo.description, "%s", langDlgLoadRomDskCasDesc());
    dlgInfo.fileList = fileList;
    dlgInfo.fileListCount = countDsk + countRom + countCas;
    dlgInfo.autoReset = pProperties->diskdrive.autostartA || pProperties->cartridge.autoReset || Prt.autostart;
    dlgInfo.selectFileIndex = -1;
    dlgInfo.selectFile[0] = 0;

    DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);
    Prt.autostart = dlgInfo.autoReset;

    if (strlen(dlgInfo.selectFile) > 4) {
        if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'R' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'O' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'M')
        {
            success = insertCartridge(0, filename, dlgInfo.selectFile);
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'M' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'X' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == '1')
        {
            success = insertCartridge(0, filename, dlgInfo.selectFile);
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'M' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'X' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == '2')
        {
            success = insertCartridge(0, filename, dlgInfo.selectFile);
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'D' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'S' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'K')
        {
            success = insertDiskette(0, filename, dlgInfo.selectFile);    
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'D' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'I' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == '1')
        {
            success = insertDiskette(0, filename, dlgInfo.selectFile);    
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'D' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'I' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == '2')
        {
            success = insertDiskette(0, filename, dlgInfo.selectFile);    
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'C' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'A' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'S')
        {
            success = insertCassette(filename, dlgInfo.selectFile);    
        }
    }
    
    free(fileListDsk);
    free(fileListRom);
    free(fileList);

    return success;
}

static int tryLaunchUnknownFile(char* fname) 
{
    int success = 0;
    int len = strlen(fname);
    
    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'S' &&
            toupper(fname[len - 2]) == 'T' &&
            toupper(fname[len - 1]) == 'A')
        {
            strcpy(Prt.pStoredState, fname);
            emulatorStart(1);
            success = 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'R' &&
            toupper(fname[len - 2]) == 'O' &&
            toupper(fname[len - 1]) == 'M')
        {
            success = insertCartridge(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'M' &&
            toupper(fname[len - 2]) == 'X' &&
            toupper(fname[len - 1]) == '1')
        {
            success = insertCartridge(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'M' &&
            toupper(fname[len - 2]) == 'X' &&
            toupper(fname[len - 1]) == '2')
        {
            success = insertCartridge(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'S' &&
            toupper(fname[len - 1]) == 'K')
        {
            success = insertDiskette(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == '1')
        {
            success = insertDiskette(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == '2')
        {
            success = insertDiskette(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'C' &&
            toupper(fname[len - 2]) == 'A' &&
            toupper(fname[len - 1]) == 'S')
        {
            success = insertCassette(fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            success = insertDisketteOrCartridge(0, fname);
        }
    }

    return success;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

int getModifiers() {
    return (GetAsyncKeyState(VK_LMENU)   > 1UL ? MOD_ALT     : 0) |
           (GetAsyncKeyState(VK_MENU)    > 1UL ? MOD_ALT     : 0) |
           (GetAsyncKeyState(VK_SHIFT)   > 1UL ? MOD_SHIFT   : 0) |
           (GetAsyncKeyState(VK_CONTROL) > 1UL ? MOD_CONTROL : 0);
}

static LRESULT CALLBACK wndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam) 
{
    PAINTSTRUCT ps;
    HDC hdc;
    RECT r;
    int i;
    char* pFname;

    switch (iMsg) {
    case WM_CREATE:
        keyboardInit(hwnd);
        SetTimer(hwnd, 10, 100, NULL);
        SetTimer(hwnd, 11, 20, NULL);
        DragAcceptFiles(hwnd, TRUE);
        return 0;

    case WM_DROPFILES:
        {
            char fname[MAX_PATH];
            HDROP hDrop;
            hDrop = (HDROP)wParam;
            DragQueryFile(hDrop, 0, fname, 512);
            DragFinish (hDrop);

            tryLaunchUnknownFile(fname);
        }
        return 0;

    case WM_COMMAND:
        i = LOWORD(wParam) - ID_CARTRIDGEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            Sleep(333);
            insertCartridge(0, pProperties->filehistory.cartridgeA[i], NULL);
        }
        i = LOWORD(wParam) - ID_CARTRIDGEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            Sleep(333);
            insertCartridge(1, pProperties->filehistory.cartridgeB[i], NULL);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            Sleep(333);
            insertDiskette(0, pProperties->filehistory.diskdriveA[i], NULL);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            Sleep(333);
            insertDiskette(1, pProperties->filehistory.diskdriveB[i], NULL);
        }
        i = LOWORD(wParam) - ID_CASSETTE_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            Sleep(333);
            insertCassette(pProperties->filehistory.cassette[i], NULL);
        }
        
        switch (LOWORD(wParam)) {
        case ID_LOG_WAV:
            if (Prt.logSound) {
                mixerStopLog(Prt.mixer);
                Prt.logSound = 0;
            }
            else {
                Prt.logSound = mixerStartLog(Prt.mixer, wavCaptureCreateFileName());
            }
            break;
        case ID_PRT_SCR:
            SendMessage(hwnd, WM_PRINTSCREEN, 0, 0);
            break;
        case ID_FILE_READONLY_CASSETTE:
            pProperties->cassette.readOnly ^= 1;
            break;
        case ID_FILE_AUTOREWNIND_CASSETTE:
            pProperties->cassette.autoRewind ^= 1;
            break;

        case ID_FILE_SAVE_CASSETTE:
            if (*pProperties->cassette.tape) {
                int type;

                if (Prt.emuState == EMU_STOPPED) {
                    tapeSetReadOnly(1);
                    msxChangeCassette(strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL, 
                                    strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL);
                }
                else {
                    emulatorSuspend();
                }
                
                type = tapeGetFormat();

                enterDialogShow();
                pFname = saveFile(hwnd, langDlgSaveCassette(), "Tape Image - fMSX-DOS     (*.cas)\0*.cas\0Tape Image - fMSX98/AT   (*.cas)\0*.cas\0", &type, pProperties->cassette.defDir);
                exitDialogShow();
                SetCurrentDirectory(Prt.pCurDir);
                if (pFname != NULL && strlen(pFname) != 0) {
                    if (type == 1 || type == 2) {
                        tapeSave(pFname, type);
                    }
                }

                if (Prt.emuState == EMU_STOPPED) {
                    msxChangeCassette(NULL, NULL);
                    tapeSetReadOnly(pProperties->cassette.readOnly);
                }
                else {
                    emulatorResume();
                }
            }
            break;
        case ID_FILE_SAVE:
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                enterDialogShow();
                pFname = saveFile(hwnd, langDlgSaveState(), "CPU state   (*.sta)\0*.sta\0", NULL, pProperties->emulation.statsDefDir);
                exitDialogShow(); 
                SetCurrentDirectory(Prt.pCurDir);
                if (pFname != NULL && strlen(pFname) != 0) {
                    char *ptr = pFname + strlen(pFname) - 1;
                    while(*ptr != '.' && ptr > pFname) {
                        ptr--;
                    }
                    if (ptr == pFname) {
                        ptr = pFname + strlen(pFname);
                    }

                    strcpy(ptr, ".sta");
                    msxSaveState(pFname);
                }
                emulatorResume();
            }
            break;
        case ID_FILE_LOAD:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgLoadState(), "CPU state   (*.sta)\0*.sta\0", pProperties->emulation.statsDefDir, 1, NULL);
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                emulatorStop();
                strcpy(Prt.pStoredState, pFname);
                emulatorStart(1);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_FILE_QSAVE:
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                strcpy(pProperties->filehistory.quicksave, quickSaveCreateFileName());
                msxSaveState(pProperties->filehistory.quicksave);
                emulatorResume();
            }
            break;
        case ID_FILE_QLOAD:
            if (tempStateExists()) {
                emulatorStop();
                sprintf(Prt.pStoredState, pProperties->filehistory.quicksave);
                emulatorStart(1);
            }
            break;
        case ID_TB_CARTA:
        case ID_FILE_INSERT_CARTRIDGEA:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgInsertRom1(), "ROM cartridge   (*.rom, *.mx1, *.mx2, *.zip)\0*.rom; *.mx1; *.mx2; *.zip\0", pProperties->cartridge.defDir, 1, NULL);
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                Sleep(333);
                insertCartridge(0, pFname, NULL);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_TB_CARTB:
        case ID_FILE_INSERT_CARTRIDGEB:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgInsertRom2(), "ROM cartridge   (*.rom, *.mx1, *.mx2, *.zip)\0*.rom; *.mx1; *.mx2; *.zip\0", pProperties->cartridge.defDir, 1, NULL);
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                Sleep(333);
                insertCartridge(1, pFname, NULL);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_FILE_CARTA_MEGARAM128:
            Sleep(333);
            insertCartridge(0, CARTNAME_MEGARAM128, NULL);
            break;
        case ID_FILE_CARTB_MEGARAM128:
            Sleep(333);
            insertCartridge(1, CARTNAME_MEGARAM128, NULL);
            break;
        case ID_FILE_CARTA_MEGARAM256:
            Sleep(333);
            insertCartridge(0, CARTNAME_MEGARAM256, NULL);
            break;
        case ID_FILE_CARTB_MEGARAM256:
            Sleep(333);
            insertCartridge(1, CARTNAME_MEGARAM256, NULL);
            break;
        case ID_FILE_CARTA_MEGARAM512:
            Sleep(333);
            insertCartridge(0, CARTNAME_MEGARAM512, NULL);
            break;
        case ID_FILE_CARTB_MEGARAM512:
            Sleep(333);
            insertCartridge(1, CARTNAME_MEGARAM512, NULL);
            break;
        case ID_FILE_CARTA_MEGARAM768:
            Sleep(333);
            insertCartridge(0, CARTNAME_MEGARAM768, NULL);
            break;
        case ID_FILE_CARTB_MEGARAM768:
            Sleep(333);
            insertCartridge(1, CARTNAME_MEGARAM768, NULL);
            break;
        case ID_FILE_CARTA_MEGARAM2M:
            Sleep(333);
            insertCartridge(0, CARTNAME_MEGARAM2M, NULL);
            break;
        case ID_FILE_CARTB_MEGARAM2M:
            Sleep(333);
            insertCartridge(1, CARTNAME_MEGARAM2M, NULL);
            break;
        case ID_FILE_CARTA_SCCPLUS:
            Sleep(333);
            insertCartridge(0, CARTNAME_SCCPLUS, NULL);
            break;
        case ID_FILE_CARTB_SCCPLUS:
            Sleep(333);
            insertCartridge(1, CARTNAME_SCCPLUS, NULL);
            break;
        case ID_FILE_CARTA_FMPAC:
            Sleep(333);
            insertCartridge(0, CARTNAME_FMPAC, NULL);
            break;
        case ID_FILE_CARTA_PAC:
            Sleep(333);
            insertCartridge(0, CARTNAME_PAC, NULL);
            break;
        case ID_FILE_CARTB_FMPAC:
            Sleep(333);
            insertCartridge(1, CARTNAME_FMPAC, NULL);
            break;
        case ID_FILE_CARTB_PAC:
            Sleep(333);
            insertCartridge(1, CARTNAME_PAC, NULL);
            break;
        case ID_FILE_CARTRIDGE_AUTORESET:
            pProperties->cartridge.autoReset ^= 1;
            break;
        case ID_FILE_INSERT_DISKETTEA_RESET:
            pProperties->diskdrive.autostartA ^= 1;
            break;
        case ID_TB_DISKA:
        case ID_FILE_INSERT_DISKETTEA:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgInsertDiskA(), "Disk image   (*.dsk, *.di1, *.di2, *.zip)\0*.dsk; *.di1; *.di2; *.zip\0", pProperties->diskdrive.defDir, 0, ".dsk");
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                Sleep(333);
                insertDiskette(0, pFname, NULL);
            }
            emulatorResume();
            break;
        case ID_TB_DISKB:
        case ID_FILE_INSERT_DISKETTEB:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgInsertDiskB(), "Disk image   (*.dsk, *.di1, *.di2, *.zip)\0*.dsk; *.di1; *.di2; *.zip\0", pProperties->diskdrive.defDir, 0, ".dsk");
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                Sleep(333);
                insertDiskette(1, pFname, NULL);
            }
            emulatorResume();
            break;
        case ID_TB_CAS:
        case ID_FILE_INSERT_CASSETTE:
            emulatorSuspend();
            enterDialogShow();
            pFname = openFile(hwnd, langDlgInsertCas(), "Tape image   (*.cas, *.zip)\0*.cas; *.zip\0", pProperties->cassette.defDir, 0, ".cas");
            exitDialogShow();
            SetCurrentDirectory(Prt.pCurDir);
            if (pFname != NULL) {
                Sleep(333);
                insertCassette(pFname, NULL);
                if (pProperties->cassette.autoRewind) {
                    tapeSetCurrentPos(0);
                }
            }
            emulatorResume();
            break;
            
        case ID_FILE_REMOVE_CARTRIDGEA:
            pProperties->cartridge.slotA[0] = 0;
            pProperties->cartridge.slotAZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                if (pProperties->cartridge.autoReset) {
                    emulatorStop();
                    Sleep(333);
                    emulatorStart(1);
                }
                else {
                    emulatorSuspend();
                    msxChangeCartridge(0, ROM_UNKNOWN, NULL, NULL);
                    emulatorResume();
                }
            }
            break;
        case ID_FILE_REMOVE_CARTRIDGEB:
            pProperties->cartridge.slotB[0] = 0;
            pProperties->cartridge.slotBZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                if (pProperties->cartridge.autoReset) {
                    emulatorStop();
                    Sleep(333);
                    emulatorStart(1);
                }
                else {
                    emulatorSuspend();
                    msxChangeCartridge(1, ROM_UNKNOWN, NULL, NULL);
                    emulatorResume();
                }
            }
            break;
        case ID_FILE_REMOVE_DISKETTEA:
            pProperties->diskdrive.slotA[0] = 0;
            pProperties->diskdrive.slotAZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                msxChangeDiskette(0, NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_REMOVE_DISKETTEB:
            pProperties->diskdrive.slotB[0] = 0;
            pProperties->diskdrive.slotBZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                msxChangeDiskette(1, NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_REMOVE_CASSETTE:
            pProperties->cassette.tape[0] = 0;
            pProperties->cassette.tapeZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                msxChangeCassette(NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_POSITION_CASSETTE:
                enterDialogShow();
                setTapePosition();
                exitDialogShow();
            break;
        case ID_FILE_REWIND_CASSETTE:
            if (Prt.emuState != EMU_STOPPED) {
                    emulatorSuspend();
                }
                else {
                    tapeSetReadOnly(1);
                    msxChangeCassette(strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL, 
                                    strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL);
                }
                tapeSetCurrentPos(0);
            if (Prt.emuState != EMU_STOPPED) {
                emulatorResume();
            }
            else {
                msxChangeCassette(NULL, NULL);
                tapeSetReadOnly(pProperties->cassette.readOnly);
            }
            break;
        case ID_FILE_EXIT:
            DestroyWindow(hwnd);
            break;
        case ID_SIZE_NORMAL:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX1)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_X2:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX2)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_FULLSCREEN:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_TB_PLAYPAUSE:
        case ID_RUN_RUN:
            if (Prt.emuState == EMU_STOPPED) {
                Sleep(333);
                emulatorStart(1);
                toolbarUpdate(getZoom() == 2, 1);
            }
            else if (Prt.emuState == EMU_PAUSED) {
                soundResume();
                toolbarUpdate(getZoom() == 2, 1);
                Prt.emuState = EMU_RUNNING;
                mouseEmuUpdateRunState(1);
            }
            else {
                soundSuspend();
                toolbarUpdate(getZoom() == 2, 0);
                Prt.emuState = EMU_PAUSED;
                mouseEmuUpdateRunState(0);
            }
            break;
        case ID_RUN_STOP:
            if (Prt.emuState != EMU_STOPPED) {
                toolbarUpdate(getZoom() == 2, 0);
                emulatorStop();
                InvalidateRect(Prt.hwnd, NULL, TRUE);
            }
            break;
        case ID_TB_RESET:
        case ID_RUN_RESET:
            emulatorStop();
            toolbarUpdate(getZoom() == 2, 1);
            Sleep(333);
            emulatorStart(1);
            break;
        case ID_RUN_SOFTRESET:
            emulatorStop();
            toolbarUpdate(getZoom() == 2, 1);
            Sleep(333);
            emulatorStart(0);
            break;
        case ID_RUN_CLEANRESET:
            emulatorStop();
            pProperties->diskdrive.slotA[0] = 0;
            pProperties->diskdrive.slotAZip[0] = 0;
            pProperties->diskdrive.slotB[0] = 0;
            pProperties->diskdrive.slotBZip[0] = 0;
            pProperties->cartridge.slotA[0] = 0;
            pProperties->cartridge.slotAZip[0] = 0;
            pProperties->cartridge.slotB[0] = 0;
            pProperties->cartridge.slotBZip[0] = 0;
            toolbarUpdate(getZoom() == 2, 1);
            Sleep(333);
            emulatorStart(1);
            break;
        case ID_OPTIONS_EMULATION:
            showPropertiesDialog(PROP_EMULATION);
            break;
        case ID_OPTIONS_VIDEO:
            showPropertiesDialog(PROP_VIDEO);
            break;
        case ID_OPTIONS_AUDIO:
            showPropertiesDialog(PROP_SOUND);
            break;
        case ID_TB_PROPERTIES:
        case ID_OPTIONS_CONTROLS:
            showPropertiesDialog(PROP_CONTROLS);
            break;
        case ID_OPTIONS_PERFORMANCE:
            showPropertiesDialog(PROP_PERFORMANCE);
            break;
        case ID_OPTIONS_SETTINGS:
            showPropertiesDialog(PROP_SETTINGS);
            break;
        case ID_OPTIONS_LANGUAGE:
            {
                int lang;
                int success;
                enterDialogShow();
                lang = langShowDlg(Prt.hwnd, pProperties->language);
                exitDialogShow();
                success = langSetLanguage(lang);
                if (success) {
                    pProperties->language = lang;
                }
            }
            break;
        case ID_TOOLS_MACHINEEDITOR:
            {
                int apply;
                enterDialogShow();
                apply = confShowDialog(Prt.hwnd, pProperties->emulation.machineName);
                exitDialogShow();
                if (apply) {
                    Machine* machine = createMachine(pProperties->emulation.machineName);
                    if (machine != NULL) {
                        pProperties->sound.chip.enableYM2413    = machine->audio.enableYM2413;
                        pProperties->sound.chip.enableY8950     = machine->audio.enableY8950;
                        pProperties->sound.chip.enableMoonsound = machine->audio.enableMoonsound;
                        pProperties->sound.chip.moonsoundSRAM   = machine->audio.moonsoundSRAM;
                        destroyMachine(machine);
                    }
                    SendMessage(hwnd, WM_COMMAND, ID_RUN_RESET, 0);
                }
            }
            break;
        case ID_TB_HELP:
        case ID_HELP_HELP:
            {
                HINSTANCE rv;

                rv = ShellExecute(Prt.hwnd, "open", "blueMSX.chm", NULL, NULL, SW_SHOWNORMAL);
                if (rv <= (HINSTANCE)32) {
                    MessageBox(NULL, langErrorNoHelp(), langErrorTitle(), MB_OK);
                }
            }
            break;
        case ID_HELP_ABOUT:
            helpShowAbout(hwnd);
            break;
        }

        updateMenu(0);
        break;

    case WM_NOTIFY:
        switch(((LPNMHDR)lParam)->code){
        case TBN_DROPDOWN:
            {
                RECT      rc;
                TPMPARAMS tpm;
                HMENU     hMenu = NULL;
                LPNMTOOLBAR ntb = (LPNMTOOLBAR)lParam;

                SendMessage(ntb->hdr.hwndFrom, TB_GETRECT, (WPARAM)ntb->iItem, (LPARAM)&rc);

                MapWindowPoints(ntb->hdr.hwndFrom, HWND_DESKTOP, (LPPOINT)&rc, 2);                         

                tpm.cbSize = sizeof(TPMPARAMS);
                tpm.rcExclude = rc;

                switch (ntb->iItem) {
                case ID_TB_RESET:
                    hMenu = menuCreateReset(pProperties);
                    break;
                case ID_TB_CARTA:
                    hMenu = menuCreateCartA(pProperties);
                    break;
                case ID_TB_CARTB:
                    hMenu = menuCreateCartB(pProperties);
                    break;
                case ID_TB_DISKA:
                    hMenu = menuCreateDiskA(pProperties);
                    break;
                case ID_TB_DISKB:
                    hMenu = menuCreateDiskB(pProperties);
                    break;
                case ID_TB_CAS:
                    hMenu = menuCreateCassette(pProperties);
                    break;
                case ID_TB_ZOOM:
                    hMenu = menuCreateZoom(pProperties);
                    break;
                case ID_TB_PROPERTIES:
                    hMenu = menuCreateOptions(pProperties);
                    break;
                }
                if (hMenu != NULL) {
                    TrackPopupMenuEx(hMenu, TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_VERTICAL,               
                                    rc.left, rc.bottom, Prt.hwnd, &tpm); 
                    DestroyMenu(hMenu);
                }
            }
            break;

        case TTN_GETDISPINFO: 
            { 
                LPTOOLTIPTEXT lpttt = (LPTOOLTIPTEXT)lParam; 
                lpttt->hinst = GetModuleHandle(NULL); 

                switch (lpttt->hdr.idFrom) { 
                case ID_TB_RESET:
                    _stprintf(lpttt->szText, "%s", langTooltipReset());
                    break; 
                case ID_TB_CARTA:
                    _stprintf(lpttt->szText, "%s", langTooltipCart1());
                    break;
                case ID_TB_CARTB:
                    _stprintf(lpttt->szText, "%s", langTooltipCart2());
                    break;
                case ID_TB_DISKA:
                    _stprintf(lpttt->szText, "%s", langTooltipDiskA());
                    break;
                case ID_TB_DISKB:
                    _stprintf(lpttt->szText, "%s", langTooltipDiskB());
                    break;
                case ID_TB_CAS:
                    _stprintf(lpttt->szText, "%s", langTooltipCas());
                    break;
                case ID_TB_PLAYPAUSE:
                    _stprintf(lpttt->szText, "%s", Prt.emuState == EMU_RUNNING ? 
                              langTooltipPause() : Prt.emuState == EMU_PAUSED ? 
                              langTooltipResume() : langTooltipStart());
                    break;
                case ID_TB_ZOOM:
                    _stprintf(lpttt->szText, "%s", langTooltipWindowSize());
                    break;
                case ID_TB_PROPERTIES:
                    _stprintf(lpttt->szText, "%s", langTooltipProperties());
                    break;
                case ID_TB_HELP:
                    _stprintf(lpttt->szText, "%s", langTooltipHelp());
                    break;
                }
            }
            break; 
        }
        break;

    case WM_SYSKEYDOWN:
    case WM_KEYDOWN:
        switch (wParam) {
        case VK_SCROLL:
            i = getModifiers();
            if (i == MOD_ALT) {
                pProperties->emulation.audioSwitch = !pProperties->emulation.audioSwitch;
                msxSetAudioSwitch(pProperties->emulation.audioSwitch);
            }
            else {
                pProperties->emulation.frontSwitch = !pProperties->emulation.frontSwitch;
                msxSetFrontSwitch(pProperties->emulation.frontSwitch);
            }
            return 0;
        case VK_CANCEL:
            SendMessage(hwnd, WM_COMMAND, ID_FILE_EXIT, 0);
            return 0;
        case VK_F6:
            i = getModifiers();
            if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_LOG_WAV, 0);
            }
            return 0;
        case VK_F7:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_LOAD, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_CARTRIDGEA, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QLOAD, 0);
            }
            return 0;
        case VK_F8:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_SAVE, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_CARTRIDGEB, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QSAVE, 0);
            }
            else if (i == MOD_SHIFT) {
                Prt.mouseLock ^= 1;
                mouseSetForceLock(Prt.mouseLock);
            }
            return 0;
        case VK_F9:
            i = getModifiers();
            if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RUN, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_DISKETTEA, 0);
            }
            else if (i == MOD_SHIFT) {
                Prt.maxSpeed = 1;
            }
            else if (i == MOD_ALT) {
                if (*pProperties->diskdrive.slotA) {
                    if (*pProperties->diskdrive.slotAZip) {
                        strcpy(pProperties->diskdrive.slotAZip, diskGetNext(pProperties->diskdrive.slotAZip, pProperties->diskdrive.slotA));
                        msxChangeDiskette(0, pProperties->diskdrive.slotA, pProperties->diskdrive.slotAZip);
                    }
                    else {
                        strcpy(pProperties->diskdrive.slotA, diskGetNext(pProperties->diskdrive.slotA, NULL));
                        msxChangeDiskette(0, pProperties->diskdrive.slotA, NULL);
                    }
                    SendMessage(Prt.dskWnd, WM_SHOWDSKWIN, 0, 0);
                }
            }
            return 0;
        case VK_F10:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_NORMAL, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_DISKETTEB, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_STOP, 0);
            }
            else if (i == MOD_SHIFT) {
                pProperties->emulation.speed = 50;
                emulatorSetFrequency(pProperties->emulation.speed, NULL, NULL);
            }
            return 0;
        case VK_F11:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_X2, 0);
            }
            else if (i == MOD_SHIFT) {
                if (pProperties->emulation.speed > 0) {
                    pProperties->emulation.speed--;
                    emulatorSetFrequency(pProperties->emulation.speed, NULL, NULL);
                }
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_CASSETTE, 0);
            }
            else if (i == (MOD_CONTROL | MOD_SHIFT)) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_REWIND_CASSETTE, 0);
            }
            else if (i == (MOD_CONTROL | MOD_ALT)) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_POSITION_CASSETTE, 0);
            }
            return 0;
        case VK_F12:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_FULLSCREEN, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_SOFTRESET, 0);
            }
            else if (i == (MOD_CONTROL | MOD_ALT)) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_CLEANRESET, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RESET, 0);
            }
            else if (i == MOD_SHIFT) {
                if (pProperties->emulation.speed < 100) {
                    pProperties->emulation.speed++;
                    emulatorSetFrequency(pProperties->emulation.speed, NULL, NULL);
                }
            }
            return 0;
        }

        return 0;

    case WM_SYSKEYUP:
    case WM_KEYUP:
        switch (wParam) {
		case VK_SNAPSHOT:
            if (getModifiers() == 0) {
                SendMessage(hwnd, WM_PRINTSCREEN, 0, 0);
            }
            return 0;

        case VK_F9:
            Prt.maxSpeed = 0;
            return 0;
        }
            
        return 0;

    case WM_CHAR:
    case WM_SYSCHAR:
        return 0;

    case WM_PRINTSCREEN:
        SetTimer(hwnd, 12, 50, NULL);
        return 0;

    case WM_SYSCOMMAND:
        switch(wParam) {
        case SC_MAXIMIZE:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            return 0;
        case SC_MINIMIZE:
            emulatorSuspend();
            break;
        case SC_RESTORE:
            emulatorResume();
            break;
        }
        break;

    case WM_NCLBUTTONDBLCLK:
        if (wParam == HTCAPTION) {
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            return 0;
        }
        break;

    case WM_ENTERSIZEMOVE:
        emulatorSuspend();
        break;

    case WM_EXITSIZEMOVE:
        emulatorResume();
        break;

    case WM_ENTERMENULOOP:
        emulatorSuspend();
        return 0;

    case WM_EXITMENULOOP:
        emulatorResume();
        updateMenu(0);
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            PostMessage(hwnd, WM_LBUTTONDOWN, 0, 0);
            PostMessage(hwnd, WM_LBUTTONUP, 0, 0);
        }
        return 0;

    case WM_MOVE:
        if (!Prt.enteringFullscreen && pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }

    case WM_DISPLAYCHANGE:
        if (pProperties->video.driver != P_VIDEO_DRVGDI) {
            int zoom = getZoom();
            if (Prt.enteringFullscreen) {
                DirectXUpdateWindowedMode(hwnd, zoom * WIDTH, zoom * HEIGHT,
                                          pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEO ||
                                          pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM, 
                                          pProperties->video.driver == P_VIDEO_DRVDIRECTX_VIDEOSYSMEM);
            }
        }
        break;

    case WM_SIZE:
        statusBarUpdatePos();
        InvalidateRect(hwnd, NULL, FALSE);
        break;
        
    case WM_ACTIVATE:
        if (wParam == WA_INACTIVE) {
            if (kbdLockDisable != NULL) {
                kbdLockDisable(0);
            }
            mouseEmuActivate(0);
        }
        else {
            if (kbdLockEnable != NULL && Prt.emuState == EMU_RUNNING && pProperties->emulation.disableWinKeys) {
                kbdLockEnable();
            }
            mouseEmuActivate(1);
        }
        break;

    case WM_MOUSEMOVE:
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            if (HIWORD(lParam) < 2) {
                if (!Prt.showMenu) {
                    updateMenu(1);
                }
            }
        }
        break;

    case WM_LBUTTONDOWN:
        updateMenu(0);
        break;

    case WM_ERASEBKGND:
        return 1;

    case WM_TIMER:
        if (wParam == 10) {
            statusBarUpdate(Prt.emuState == EMU_STOPPED, 
                            Prt.emuState == EMU_PAUSED, 
                            Prt.screenMode, 
                            pProperties->emulation.speed,
                            Prt.ledState,
                            pProperties->emulation.frontSwitch,
                            pProperties->emulation.audioSwitch); 
        }
        if (wParam == 11) {
            keyboardUpdate(Prt.keyMap);
        }
        if (wParam == 12) {
            int yOffset = 0;
            int xOffset = 0;
            int width = WIDTH * getZoom();
            int height = (HEIGHT - 1) * getZoom();

            if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
                yOffset += Prt.DTY;
            }
            if (!pProperties->video.horizontalStretch) {
//                xOffset += 24 * getZoom();
//                width   -= 48 * getZoom();
            }

			ScreenShot(hwnd, width, height, xOffset, yOffset);
            KillTimer(hwnd, 12);
        }
        break;

    case WM_UPDATE:
        if (Prt.emuState != EMU_STOPPED && pProperties->video.driver != P_VIDEO_DRVGDI) {
            int dy = 0;
            int dsy = 0;
            int yOffset = 0;
            int zoom = getZoom();

            GetClientRect(hwnd, &r);
            if (Prt.showMenu && pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
                int cy = GetSystemMetrics(SM_CYSCREEN);
                dy = r.bottom - cy;
                r.bottom = cy;
            }
            else if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
                yOffset = Prt.DTY;
                dsy = Prt.DY;
            }

            DirectXUpdateSurface(Prt.pVideo, Prt.bmBits, WIDTH, HEIGHT, Prt.srcArr, Prt.showMenu | Prt.showDialog, dy, yOffset, zoom, pProperties->video.horizontalStretch, pProperties->video.verticalStretch, Prt.evenOdd, Prt.interlace);
            Prt.updatePending = 0;
        }
        else {
            InvalidateRect(hwnd, NULL, FALSE);
        }
        return 0;

    case WM_INPUTLANGCHANGE:
        break;

    case WM_PAINT:
        {
            int dy = 0;
            int dsy = 0;
            int yOffset = 0;
            BITMAP bm;
            HDC hMemDC;
            HBITMAP hBitmap;
            int zoom = getZoom();

            hdc = BeginPaint(hwnd, &ps);   

            GetClientRect(hwnd, &r);
            if (Prt.showMenu && pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
                int cy = GetSystemMetrics(SM_CYSCREEN);
                dy = r.bottom - cy;
                r.bottom = cy;
            }
            else if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
                yOffset = Prt.DTY;
                dsy = Prt.DY;
            }

            if (Prt.emuState == EMU_STOPPED) {
                hMemDC = CreateCompatibleDC(hdc);
                if (zoom == 1) {
                    hBitmap = (HBITMAP)SelectObject(hMemDC, Prt.hLogoSmall);
                    GetObject(Prt.hLogoSmall, sizeof(BITMAP), (PSTR)&bm);
                }
                else {
                    hBitmap = (HBITMAP)SelectObject(hMemDC, Prt.hLogo);
                    GetObject(Prt.hLogo, sizeof(BITMAP), (PSTR)&bm);
                }

                {
                    int x = ((r.right - r.left) - bm.bmWidth) / 2;
                    int y = ((r.bottom - r.top - dsy) - bm.bmHeight) / 2 + dy;

                    if (x > 0 || y > 0) {
                        StretchBlt(hdc, 0, dy + yOffset, r.right - r.left, r.bottom - r.top - dsy, hMemDC, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
                    }
                    else {
                        BitBlt(hdc, x, y + yOffset, bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
                    }
                }
                SelectObject(hMemDC, hBitmap);
                DeleteDC(hMemDC);
            }
            else {
                if (pProperties->video.driver != P_VIDEO_DRVGDI) {
                    DirectXUpdateSurface(Prt.pVideo, Prt.bmBits, WIDTH, HEIGHT, Prt.srcArr, Prt.showMenu | Prt.showDialog, dy, yOffset, zoom, pProperties->video.horizontalStretch, pProperties->video.verticalStretch, Prt.evenOdd, Prt.interlace);
                }
                else {
                    int width = r.right - r.left;
                    int height = r.bottom - r.top - yOffset - (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? Prt.DDY : 0);
                    DWORD* bmBitsSrc = (DWORD*)Prt.bmBits + WIDTH * (HEIGHT - 1) * 2;
                    
                    videoRender(Prt.pVideo, 32, zoom, Prt.evenOdd, Prt.interlace, bmBitsSrc, WIDTH, HEIGHT, Prt.srcArr, Prt.bmBitsGDI, 
                                -1 * (int)sizeof(DWORD) * WIDTH, zoom * WIDTH * sizeof(DWORD));
                    Prt.bmInfo.bmiHeader.biWidth    = zoom * WIDTH;
                    Prt.bmInfo.bmiHeader.biHeight   = zoom * HEIGHT;
                    Prt.bmInfo.bmiHeader.biBitCount = 32;
                    StretchDIBits(hdc, 0, dy + yOffset + (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? 1 : 0), 
                                  width, height, 0, 0, zoom * WIDTH, zoom * HEIGHT, Prt.bmBitsGDI, 
                                  &Prt.bmInfo, DIB_RGB_COLORS, SRCCOPY);
                }
            }
            EndPaint(hwnd, &ps);
            Prt.updatePending = 0;
        }
        return 0;
        
    case WM_DESTROY:         
        emulatorStop();
        Sleep(400);
        if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }
        Prt.enteringFullscreen = 1;
        DirectXExitFullscreenMode(hwnd);
        PostQuitMessage(0);
        keyboardDestroy();
        return 0;
    }

    return DefWindowProc(hwnd, iMsg, wParam, lParam);
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////

char* getCmdArgument(char* szLine, int argNo) {
    static char argBuf[512];
    int i;

    for (i = 0; i <= argNo; i++) {
        char* arg = argBuf;

        while (*szLine == ' ') szLine++;

        if (*szLine == 0) return NULL;

        if (*szLine == '\"') {
            szLine++;
            while (*szLine != '\"' && *szLine != 0) {
                *arg++ = *szLine++;
            }
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
        else {
            do {
                *arg++ = *szLine++;
            } while (*szLine != ' ' && *szLine != 0);
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
    }
    return argBuf;
}

static RomType romNameToType(char* name) {
    RomType romType = ROM_UNKNOWN;

    if (name == NULL) {
        return ROM_UNKNOWN;
    }

    romType = romMapperTypeFromString(name);

    if (romType == ROM_UNKNOWN) {
        romType = atoi(name);
        if (romType < ROM_STANDARD || romType > ROM_MAXROMID) {
            romType = ROM_UNKNOWN;
        }
    }

    return romType;
}

int isRomFileType(char* fname) {
    int len = strlen(fname);

//    if (fname[len-1] == '\"') len--;

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'R' &&
            toupper(fname[len - 2]) == 'O' &&
            toupper(fname[len - 1]) == 'M')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

int isDskFileType(char* fname) {
    int len = strlen(fname);

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'S' &&
            toupper(fname[len - 1]) == 'K')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

int emuCheckResetArgument(PSTR szLine) {
    int i;
    char*   argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/reset") == 0) {
            return 1;
        }
    }

    return 0;
}

int emuCheckIniFileArgument(PSTR szLine) {
    int i;
    char*   argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/inifile") == 0) {
            return 1;
        }
    }

    return 0;
}

int emuCheckLanguageArgument(PSTR szLine, int defaultLang) {
    int i;
    int lang;
    char* argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/language") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return defaultLang;
            lang = langFromName(argument);
            return lang == EMU_LANG_UNKNOWN ? defaultLang : lang;
        }
    }

    return defaultLang;
}

int emuStartWithArguments(PSTR szLine) {
    int i;
    char*   argument;
    char    rom1[512] = "";
    char    rom2[512] = "";
    RomType romType1  = ROM_UNKNOWN;
    RomType romType2  = ROM_UNKNOWN;
    char    machineName[64] = "";
    char    diskA[512] = "";
    char    diskB[512] = "";
    int     fullscreen = 0;
    int     startEmu = 0;

    // If one argument, assume it is a rom or disk to run
    if (!getCmdArgument(szLine, 1)) {
        argument = getCmdArgument(szLine, 0);
        
        if (*argument != '/') {
            if (*argument == '\"') argument++;

            if (*argument) {
                Prt.autostart = 1;
                return tryLaunchUnknownFile(argument);
            }
            return 0;
        }
    }

    // If more than one argument, check arguments,
    // set configuration and then run

    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/rom1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom1, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType1 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/rom2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom2, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType2 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskA") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskA, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskB") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskB, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/family") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            strcpy(machineName, argument); // FIXME verify arg
            if (!machineIsValid(machineName)) return 0;
            startEmu = 1;
        }
        if (strcmp(argument, "/machine") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            strcpy(machineName, argument);
            if (!machineIsValid(machineName)) return 0;
            startEmu = 1;
        }
        if (strcmp(argument, "/fullscreen") == 0) {
            fullscreen = 1;
        }
    }

    if (!startEmu) {
        return 1;
    }

    pProperties->cartridge.slotA[0] = 0;
    pProperties->cartridge.slotAZip[0] = 0;
    pProperties->cartridge.slotB[0] = 0;
    pProperties->cartridge.slotBZip[0] = 0;

    pProperties->diskdrive.slotA[0] = 0;
    pProperties->diskdrive.slotAZip[0] = 0;
    pProperties->diskdrive.slotB[0] = 0;
    pProperties->diskdrive.slotBZip[0] = 0;
    
    Prt.noautostart = 1;

    if (strlen(rom1)  && !insertCartridge(0, rom1, NULL)) return 0;
    if (strlen(rom2)  && !insertCartridge(1, rom2, NULL)) return 0;
    if (strlen(diskA) && !insertDiskette(0, diskA, NULL)) return 0;
    if (strlen(diskB) && !insertDiskette(1, diskB, NULL)) return 0;

    if (strlen(machineName)) strcpy(pProperties->emulation.machineName, machineName);

    propSetVideoSize(pProperties, fullscreen ? P_VIDEO_SIZEFULLSCREEN : P_VIDEO_SIZEX2);

    Prt.noautostart = 0;

    emulatorStop();
    emulatorStart(1);

    return 1;
}

static BOOL CALLBACK dskProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    switch (iMsg) {
    case WM_INITDIALOG:
        centerDialog(hDlg);
        return FALSE;

    case WM_SHOWDSKWIN:
        {
            RECT r1;
            RECT r2;
            int x;
            int y;

            enterDialogShow();

            GetWindowRect(GetParent(hDlg), &r1);
            GetWindowRect(hDlg, &r2);

            x = r1.left + (r1.right - r1.left - r2.right + r2.left) / 2;
            y = r1.top  + (r1.bottom - r1.top - r2.bottom + r2.top) / 2;

            SetWindowText(GetDlgItem(hDlg, IDC_DISKIMAGE), 
                stripPath(*pProperties->diskdrive.slotAZip ? 
                pProperties->diskdrive.slotAZip : pProperties->diskdrive.slotA));
            SetWindowPos(hDlg, NULL, x, y, 0, 0, SWP_NOZORDER | SWP_NOSIZE | SWP_SHOWWINDOW);
            SetTimer(hDlg, 1, 1000, NULL);
            SetFocus(GetParent(hDlg));
        }
        return TRUE;

    case WM_TIMER:
        ShowWindow(hDlg, FALSE);
        KillTimer(hDlg, 1);
    
        exitDialogShow();
        return TRUE;
    }
    return FALSE;
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////


#ifdef _CONSOLE
int main(int argc, char **argv)
#else
WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, PSTR szLine, int iShow)
#endif
{
#ifdef _CONSOLE
    char szLine[8192] = "";
#endif
    char buffer[512];
    static WNDCLASSEX wndClass;
    HINSTANCE hInstance = (HINSTANCE)GetModuleHandle(NULL);
    BOOL screensaverActive;
    HWND hwndStatus;
    HWND hwndToolbar;
    char* ptr;
    RECT wr;
    RECT cr;
    RECT sr;
    RECT tr;
    MSG msg;
    int i;
    HINSTANCE kbdLockInst;

#ifdef _CONSOLE
    for (i = 1; i < argc; i++) {
        strcat(szLine, argv[i]);
        strcat(szLine, " ");
    }
#endif

    GetModuleFileName(hInstance, buffer, 512);
    ptr = stripPath(buffer);
    *ptr = 0;
    chdir(buffer);

	mkdir("SRAM");
    mkdir("Casinfo");

    kbdLockInst = LoadLibrary("kbdlock.dll");

    if (kbdLockInst != NULL) {
        kbdLockEnable  = (KbdLockFun)GetProcAddress(kbdLockInst, (LPCSTR)2);
        kbdLockDisable = (KbdLockFun)GetProcAddress(kbdLockInst, (LPCSTR)3);
    }

    InitCommonControls(); 

    wndClass.cbSize         = sizeof(wndClass);
    wndClass.style          = CS_OWNDC;
    wndClass.lpfnWndProc    = wndProc;
    wndClass.cbClsExtra     = 0;
    wndClass.cbWndExtra     = 0;
    wndClass.hInstance      = hInstance;
    wndClass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_BLUEMSX));
    wndClass.hIconSm        = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_BLUEMSX));
    wndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground  = NULL;
    wndClass.lpszMenuName   = NULL;
    wndClass.lpszClassName  = "blueMSX";

    RegisterClassEx(&wndClass);

    pProperties = propCreate(emuCheckResetArgument(szLine), emuCheckIniFileArgument(szLine));

    memset(Prt.keyMap, 0xff, 16);

    Prt.showDialog = 0;
    Prt.ledState = 0;
    Prt.autostart = 0;
    Prt.noautostart = 0;
    Prt.screenMode = 0;
    Prt.enteringFullscreen = 1;
    Prt.emuState = EMU_STOPPED;
    Prt.hLogo = LoadBitmap(hInstance, "LOGO");
    Prt.hLogoSmall = LoadBitmap(hInstance, "LOGOSMALL");
    Prt.updatePending = 0;
    Prt.pVideo = videoCreate();
    Prt.mixer  = mixerCreate();
    
    /* Set SCREEN8 colors */

    Prt.X = CW_USEDEFAULT;
    Prt.Y = CW_USEDEFAULT;

    GetCurrentDirectory(MAX_PATH - 1, Prt.pCurDir);

    if (!SetCurrentDirectory("Config")) {
        mkdir("Config");
        initDefaultMachines();
    }

    SetCurrentDirectory(Prt.pCurDir);

    sprintf(buffer, "%s\\Screenshots", Prt.pCurDir);
    InitScreenShot(buffer);
    
    sprintf(buffer, "%s\\Audio Capture", Prt.pCurDir);
    wavCaptureInit(buffer);

    langInit();

    SystemParametersInfo(SPI_GETSCREENSAVEACTIVE, 0, &screensaverActive, SPIF_SENDWININICHANGE); 
    SystemParametersInfo(SPI_SETSCREENSAVEACTIVE, !pProperties->settings.disableScreensaver, 0, SPIF_SENDWININICHANGE); 

    if (pProperties->emulation.registerFileTypes) {
        registerFileTypes();
    }

    pProperties->language = emuCheckLanguageArgument(szLine, pProperties->language);
    langSetLanguage(pProperties->language);

    tapeSetReadOnly(pProperties->cassette.readOnly);

    joystickIoSetType(0, pProperties->joy1.type == P_JOY_NONE  ? 0 : pProperties->joy1.type == P_JOY_MOUSE ? 2 : 1, pProperties->joy1.type);
    joystickIoSetType(1, pProperties->joy2.type == P_JOY_NONE  ? 0 : pProperties->joy2.type == P_JOY_MOUSE ? 2 : 1, pProperties->joy2.type);

    Prt.hwnd = CreateWindow("blueMSX", langDlgMainWindow(), 
                            WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX, 
                            CW_USEDEFAULT, CW_USEDEFAULT, 800, 200, NULL, 
                            menuCreate(pProperties, 0, 1, 0, tempStateExists()), hInstance, NULL);

    Prt.dskWnd = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_DISKIMAGE), Prt.hwnd, dskProc);

    hwndStatus = statusBarInit(Prt.hwnd);
    hwndToolbar = toolBarInit(Prt.hwnd);

    GetWindowRect(Prt.hwnd, &wr);
    GetClientRect(Prt.hwnd, &cr);
    GetWindowRect(hwndStatus, &sr);
    GetWindowRect(hwndToolbar, &tr);

    Prt.X  = wr.left;
    Prt.Y  = wr.top;
    Prt.DY = (wr.bottom - wr.top) - (cr.bottom - cr.top);
    Prt.DDY = (sr.bottom - sr.top) - 1;
    Prt.DTY = (tr.bottom - tr.top) - 1;
    Prt.DX = (wr.right - wr.left) - (cr.right - cr.left);

    mouseEmuInit(Prt.hwnd, 1);
    mouseEmuEnable(pProperties->joy1.type == P_JOY_MOUSE || pProperties->joy2.type == P_JOY_MOUSE);

    SetWindowPos(Prt.hwnd, NULL, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOZORDER);

    Prt.enteringFullscreen = 0;

    Prt.bmInfo.bmiHeader.biSize           = sizeof(BITMAPINFOHEADER);
    Prt.bmInfo.bmiHeader.biWidth          = WIDTH;
    Prt.bmInfo.bmiHeader.biHeight         = HEIGHT;
    Prt.bmInfo.bmiHeader.biPlanes         = 1;
    Prt.bmInfo.bmiHeader.biBitCount       = 32;
    Prt.bmInfo.bmiHeader.biCompression    = BI_RGB;
    Prt.bmInfo.bmiHeader.biClrUsed        = 0;
    Prt.bmInfo.bmiHeader.biClrImportant   = 0;

    Prt.frameBuffer[0] = calloc(4 * WIDTH * HEIGHT, sizeof(UInt32));
    Prt.frameBuffer[1] = calloc(4 * WIDTH * HEIGHT, sizeof(UInt32));
    Prt.frameLine[0] = calloc(256, sizeof(int));
    Prt.frameLine[1] = calloc(256, sizeof(int));

    Prt.frameBufferIndex = 0;

    Prt.bmBits = Prt.frameBuffer[0];
    Prt.srcArr = Prt.frameLine[0];

    Prt.bmBitsGDI = malloc(4096 * 4096 * sizeof(UInt32));
    Prt.logSound      = 0;
    Prt.maxSpeed      = 0;
    Prt.maxSpeedCount = 0;

    soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, pProperties->sound.bufSize, pProperties->sound.stereo ? 2 : 1);

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }
    
    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    updateVideoRender();
    updateJoystick();

    if (*szLine) {
        int success;
        if (0 == strncmp(szLine, "/onearg ", 8)) {
            char args[2048];
            char* ptr;
            sprintf(args, "\"%s", szLine + 8);
            ptr = args + strlen(args);
            while(*--ptr == ' ') {
                *ptr = 0; 
            }
            strcat(args, "\"");
            success = emuStartWithArguments(args);
        }
        else {
            success = emuStartWithArguments(szLine);
        }
        if (!success) {            
            exit(0);
            return 0;
        }
    }
    
    ScaleWindow(Prt.hwnd);

    ShowWindow(Prt.hwnd, TRUE);
    UpdateWindow(Prt.hwnd);

    while (GetMessage(&msg, NULL, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
    
    Sleep(500);
    videoDestroy(Prt.pVideo);
    propDestroy(pProperties);

    soundDestroy();
    Sleep(500);
    mixerDestroy(Prt.mixer);

    SystemParametersInfo(SPI_SETSCREENSAVEACTIVE, screensaverActive, 0, SPIF_SENDWININICHANGE); 

    if (kbdLockDisable) {
        kbdLockDisable();
    }
    FreeLibrary(kbdLockInst);

    exit(0);

    return 0;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

UInt32  emuFixedPalette[256];
UInt32  emuPalette0;
UInt32  emuPalette[300];
UInt32* emuFrameBuffer;
int*    emuLineWidth;

static HANDLE       emuThread;
static HANDLE       emuSyncSem;
static HANDLE       emuStartSem;
static void*        emuTimer;
static volatile int emuSuspendFlag;
static int          emuExitFlag;

int emulatorGetSyncPeriod() {
    return pProperties->emulation.syncMethod == P_EMU_SYNC1MS ? 1 : 10;
}

static void getDeviceInfo(DeviceInfo* deviceInfo) 
{
    strcpy(pProperties->cartridge.slotA,    deviceInfo->cartridge[0].name);
    strcpy(pProperties->cartridge.slotAZip, deviceInfo->cartridge[0].inZipName);

    strcpy(pProperties->cartridge.slotB,    deviceInfo->cartridge[1].name);
    strcpy(pProperties->cartridge.slotBZip, deviceInfo->cartridge[1].inZipName);

    strcpy(pProperties->diskdrive.slotA,    deviceInfo->diskette[0].name);
    strcpy(pProperties->diskdrive.slotAZip, deviceInfo->diskette[0].inZipName);

    strcpy(pProperties->diskdrive.slotB,    deviceInfo->diskette[1].name);
    strcpy(pProperties->diskdrive.slotBZip, deviceInfo->diskette[1].inZipName);

    strcpy(pProperties->cassette.tape,      deviceInfo->cassette.name);
    strcpy(pProperties->cassette.tapeZip,   deviceInfo->cassette.inZipName);
    
    pProperties->sound.chip.enableYM2413    = deviceInfo->audio.enableYM2413;
    pProperties->sound.chip.enableY8950     = deviceInfo->audio.enableY8950;
    pProperties->sound.chip.enableMoonsound = deviceInfo->audio.enableMoonsound;
    pProperties->sound.chip.moonsoundSRAM   = deviceInfo->audio.moonsoundSRAM;
}

static void setDeviceInfo(DeviceInfo* deviceInfo) 
{
    /* Set cart A */
    deviceInfo->cartridge[0].inserted =  strlen(pProperties->cartridge.slotA);
    deviceInfo->cartridge[0].type = ROM_UNKNOWN;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_SCCPLUS;
    }
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_FMPAC;
    }
    if (0 == strcmp(CARTNAME_PAC, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_PAC;
    }
    if (0 == strcmp(CARTNAME_MEGARAM128, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_MEGARAM128;
    }
    if (0 == strcmp(CARTNAME_MEGARAM256, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_MEGARAM256;
    }
    if (0 == strcmp(CARTNAME_MEGARAM512, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_MEGARAM512;
    }
    if (0 == strcmp(CARTNAME_MEGARAM768, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_MEGARAM768;
    }
    if (0 == strcmp(CARTNAME_MEGARAM2M, pProperties->cartridge.slotA)) {
        deviceInfo->cartridge[0].type = ROM_MEGARAM2M;
    }

    strcpy(deviceInfo->cartridge[0].name,      pProperties->cartridge.slotA);
    strcpy(deviceInfo->cartridge[0].inZipName, pProperties->cartridge.slotAZip);

    /* Set cart B */
    deviceInfo->cartridge[1].inserted =  strlen(pProperties->cartridge.slotB);
    deviceInfo->cartridge[1].type = ROM_UNKNOWN;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_SCCPLUS;
    }
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_FMPAC;
    }
    if (0 == strcmp(CARTNAME_PAC, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_PAC;
    }
    if (0 == strcmp(CARTNAME_MEGARAM128, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_MEGARAM128;
    }
    if (0 == strcmp(CARTNAME_MEGARAM256, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_MEGARAM256;
    }
    if (0 == strcmp(CARTNAME_MEGARAM512, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_MEGARAM512;
    }
    if (0 == strcmp(CARTNAME_MEGARAM768, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_MEGARAM768;
    }
    if (0 == strcmp(CARTNAME_MEGARAM2M, pProperties->cartridge.slotB)) {
        deviceInfo->cartridge[1].type = ROM_MEGARAM2M;
    }

    strcpy(deviceInfo->cartridge[1].name,      pProperties->cartridge.slotB);
    strcpy(deviceInfo->cartridge[1].inZipName, pProperties->cartridge.slotBZip);

    /* Set disk A */
    deviceInfo->diskette[0].inserted =  strlen(pProperties->diskdrive.slotA);
    strcpy(deviceInfo->diskette[0].name,      pProperties->diskdrive.slotA);
    strcpy(deviceInfo->diskette[0].inZipName, pProperties->diskdrive.slotAZip);

    /* Set disk B */
    deviceInfo->diskette[1].inserted =  strlen(pProperties->diskdrive.slotB);
    strcpy(deviceInfo->diskette[1].name,      pProperties->diskdrive.slotB);
    strcpy(deviceInfo->diskette[1].inZipName, pProperties->diskdrive.slotBZip);

    /* Set tape */
    deviceInfo->cassette.inserted =  strlen(pProperties->cassette.tape);
    strcpy(deviceInfo->cassette.name,      pProperties->cassette.tape);
    strcpy(deviceInfo->cassette.inZipName, pProperties->cassette.tapeZip);
    
    /* Set audio config */
    deviceInfo->audio.enableYM2413    = pProperties->sound.chip.enableYM2413;
    deviceInfo->audio.enableY8950     = pProperties->sound.chip.enableY8950;
    deviceInfo->audio.enableMoonsound = pProperties->sound.chip.enableMoonsound;
    deviceInfo->audio.moonsoundSRAM   = pProperties->sound.chip.moonsoundSRAM;
}

static DWORD WINAPI emulatorThread(void* param) {
    int syncPeriod;
    int frequency;
    int success = 0;

    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_ABOVE_NORMAL);

    emulatorSetFrequency(pProperties->emulation.speed, &syncPeriod, &frequency);

    msxSetFrontSwitch(pProperties->emulation.frontSwitch);
    msxSetAudioSwitch(pProperties->emulation.audioSwitch);

    success = msxRun(Prt.machine,
                     &Prt.deviceInfo,
                     Prt.mixer,
                     strlen(Prt.pStoredState) ? Prt.pStoredState : NULL,
                     syncPeriod,
                     frequency,
                     (int)param);

    ReleaseSemaphore(emuStartSem, 1, NULL);

    Prt.ledState = 0;
    StopTimer(emuTimer);

    if (!success) {
        MessageBox(NULL, langErrorStartEmu(), langErrorTitle(), MB_ICONHAND | MB_OK);
        PostMessage(Prt.hwnd, WM_COMMAND, ID_RUN_STOP, 0);
    }

    ExitThread(0);
    
    return 0;
}

void emulatorStart(int hard) {
    DWORD id;
    int i;

    emuExitFlag = 0;

    for(i=0; i<256; i++) {
        emuFixedPalette[i] = videoGetColor(Prt.pVideo, (i & 0x1c) << 3, (i & 0xe0), ((i & 3) == 3 ? 7 : 2 * (i & 3)) << 5);
    }

    Prt.machine = createMachine(pProperties->emulation.machineName);

    if (Prt.machine == NULL) {  
        MessageBox(NULL, langErrorStartEmu(), langErrorTitle(), MB_ICONHAND | MB_OK);
        return;
    }

    emuFrameBuffer = (UInt32*)Prt.bmBits;
    emuLineWidth   = (int*)Prt.srcArr;

    emuSyncSem  = CreateSemaphore(NULL, 0, 10, NULL);
    emuStartSem = CreateSemaphore(NULL, 0, 10, NULL);
    emuTimer   = StartTimer(Prt.hwnd, emuSyncSem, emulatorGetSyncPeriod(), pProperties->emulation.syncMethod == P_EMU_SYNC1MS);
    
    setDeviceInfo(&Prt.deviceInfo);    

    soundResume();

    if (kbdLockEnable != NULL && pProperties->emulation.disableWinKeys) {
        kbdLockEnable();
    }

    emuThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)emulatorThread, (void*)hard, 0, &id);
    
    WaitForSingleObject(emuStartSem, INFINITE);
    
    Prt.emuState = EMU_RUNNING;
    mouseEmuUpdateRunState(1);

    getDeviceInfo(&Prt.deviceInfo);

    joystickIoGetType(0, &pProperties->joy1.type);
    joystickIoGetType(1, &pProperties->joy2.type);

    strcpy(pProperties->emulation.machineName, Prt.machine->name);
} 

void emulatorStop() {
    if (Prt.emuState == EMU_STOPPED) {
        return;
    }

    if (kbdLockDisable != NULL) {
        kbdLockDisable();
    }

    Prt.emuState = EMU_STOPPED;

    mouseEmuUpdateRunState(0);
    emuExitFlag = 1;
    ReleaseSemaphore(emuSyncSem, 1, NULL);
    WaitForSingleObject(emuThread, INFINITE);
    Prt.pStoredState[0] = 0;
    soundSuspend();
    destroyMachine(Prt.machine);
    CloseHandle(emuThread);
    CloseHandle(emuSyncSem);
    CloseHandle(emuStartSem);
}

void emulatorSetFrequency(int logFrequency, int* syncPeriod, int* frequency) {
    int sync = (int)(3580 * emulatorGetSyncPeriod() * pow(2.0, (logFrequency - 50) / 15.0515));
    int freq = (int)(3579545 * pow(2.0, (logFrequency - 50) / 15.0515));

    if (syncPeriod != NULL && frequency != NULL) {
        *syncPeriod = sync;
        *frequency  = freq;
    }
    else {
        msxSetFrequency(sync, freq);
    }
}

void emulatorSuspend() {
    if (Prt.emuState == EMU_RUNNING) {
        if (kbdLockDisable != NULL) {
            kbdLockDisable();
        }
        Prt.emuState = EMU_SUSPENDED;
        mouseEmuUpdateRunState(0);
        do {
            Sleep(10);
        } while (!emuSuspendFlag);
        soundSuspend();
    }
}

void emulatorResume() {
    if (Prt.emuState == EMU_SUSPENDED) {
        if (kbdLockEnable != NULL && pProperties->emulation.disableWinKeys) {
            kbdLockEnable();
        }
        soundResume();
        Prt.emuState = EMU_RUNNING;
        mouseEmuUpdateRunState(1);
    }
}

void emulatorReset() {
    EmuState emuState = Prt.emuState;
    mouseEmuUpdateRunState(0);

    if (emuState == EMU_RUNNING) {
        emulatorStop();
        emulatorStart(1);
        mouseEmuUpdateRunState(1);
    }
}

void emulatorRestartSound(int stereo) {
    emulatorSuspend();
    soundDestroy();
    soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, pProperties->sound.bufSize, stereo ? 2 : 1);
    emulatorResume();
}

void SetColor(int palEntry, UInt32 rgbColor) {
    UInt32 color = videoGetColor(Prt.pVideo, ((rgbColor >> 16) & 0xff), ((rgbColor >> 8) & 0xff), rgbColor & 0xff);
    if (palEntry == 0) {
        emuPalette0 = color;
    }
    else {
        emuPalette[palEntry] = color;
    }
}

UInt8 Joystick(register UInt8 N) {
    return JoystickGetState(N + 1);
}

void Keyboard(UInt8* keybardMap) {
    memcpy(keybardMap, Prt.keyMap, 16);
}

void SetCapslockLed(int enable) {
    if (enable) Prt.ledState |= 1;
    else        Prt.ledState &= ~1;
}

void SetKanaLed(int enable) {
    if (enable) Prt.ledState |= 2;
    else        Prt.ledState &= ~2;
}

int WaitForSync(void) {
    ReleaseSemaphore(emuStartSem, 1, NULL);
    do {
        emuSuspendFlag = 1;
        if (Prt.maxSpeed == 0) {
            WaitForSingleObject(emuSyncSem, INFINITE);
        }
        else if (Prt.maxSpeedCount == 0) {
            Sleep(10);
            Prt.maxSpeedCount = 9 * 10 / emulatorGetSyncPeriod();
        }
        else {
            Prt.maxSpeedCount--;
        }
    } while (!emuExitFlag && Prt.emuState != EMU_RUNNING);
    emuSuspendFlag = 0;

    return emuExitFlag;
}

void RefreshScreen(int screenMode, int evenOdd, int interlace) {
    static int frameSkip = 0;
    if (frameSkip++ < pProperties->video.frameSkip) {
        return;
    }
    frameSkip = 0;

    Prt.updatePending++;
    if (Prt.updatePending <= 1) {
        Prt.bmBits = emuFrameBuffer;
        Prt.srcArr = emuLineWidth;
        Prt.screenMode = screenMode;
        Prt.evenOdd = evenOdd;
        Prt.interlace = interlace;

        if (!interlace) {
            Prt.frameBufferIndex ^= 1;
            emuFrameBuffer = (UInt32*)Prt.frameBuffer[Prt.frameBufferIndex];
            emuLineWidth   = (int*)Prt.frameLine[Prt.frameBufferIndex];
        }

        PostMessage(Prt.hwnd, WM_UPDATE, 0, 0);
    }
}
