/*****************************************************************************
** File:        Win32.c
**
** Author:      Daniel Vik
**
** Description: The central file that ties everything together. Contains
**              the program entry point and the main window.
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
******************************************************************************
*/
#define DIRECTINPUT_VERSION     0x0500

#include <windows.h>
#include <tchar.h>
#include <direct.h>
#include <math.h>
#include <CommCtrl.h>
#include <Mmsystem.h>
 
#include "MSX.h"
#include "audioMixer.h"
#include "videoRender.h"
#include "Language.h"   
#include "resource.h"
#include "ziphelper.h"
#include "Casette.h"
#include "Disk.h"

#include "Win32Timer.h"
#include "Win32Sound.h"
#include "Win32Properties.h"
#include "Win32joystick.h"
#include "Win32keyboard.h"
#include "Win32directx.h"
#include "Win32FileHistory.h"
#include "Win32file.h"
#include "Win32Help.h"
#include "Win32Menu.h"
#include "Win32StatusBar.h"
#include "Win32ToolBar.h"
#include "Win32ScreenShot.h"
#include "Win32WaveCapture.h"
#include "Win32MouseEmu.h"

#include "romMSX.h"
#include "romMSXBR.h"
#include "romMSXJ.h"
#include "romMSX2.h"
#include "romMSX2EXT.h"
#include "romMSX2J.h"
#include "romMSX2EXTJ.h"
#include "romMSX2P.h"
#include "romMSX2PEXT.h"
#include "romMSXKR.h"
#include "romMSXHAN.h"
#include "romMSX2KR.h"
#include "romMSX2HAN.h"
#include "romMSX2EXTKR.h"


typedef enum { EMU_RUNNING, EMU_PAUSED, EMU_STOPPED, EMU_SUSPENDED } EmuState;

void emulatorSetFrequency(int logFrequency);
void emulatorSuspend();
void emulatorResume();
void emulatorReset();
void emulatorStart(int hard);
void emulatorStop();

static Properties* pProperties;

typedef struct {
    HWND hwnd;
    HMENU hMenu;
    int showMenu;
    HBITMAP hBmp;
    HBITMAP hLogo;
    BITMAPINFO bmInfo;
    void* bmBits;
    int* srcArr;
    void* bmBitsGDI;
    UInt8 keyMap[16];
    EmuState emuState;
    int updatePending;
    char pStoredState[MAX_PATH];
    char pCurDir[MAX_PATH];
    Video* pVideo;
    int logSound;
    Mixer* mixer;
    int enteringFullscreen;
    int screenMode;
    int evenOdd;
    int interlace;
    int autostart;
    int noautostart;
    UInt32 ledState;
    int tapeCurPos;
    int X;
    int Y;
    int DX;
    int DY;
    int DDY;
    int DTY;
} Port;

#define WM_UPDATE            (WM_USER + 0)

#define WIDTH  320
#define HEIGHT 240

static Port Prt;

static int getZoom() {
    switch (pProperties->video.size) {
    case P_VIDEO_SIZEX1:
        return 1;
    case P_VIDEO_SIZEX2:
        return 2;
    case P_VIDEO_SIZEFULLSCREEN:
        switch (pProperties->video.fullRes) {
        case P_VIDEO_FRES320X240_16:
        case P_VIDEO_FRES320X240_32:
            return 1;
        case P_VIDEO_FRES640X480_16:
        case P_VIDEO_FRES640X480_32:
            return 2;
        }
    }
    return 1;
}

static int tempStateExists() {
    char fileName[512];
    FILE* file;

    sprintf(fileName, "%s\\%s", Prt.pCurDir, "quicksave.sta");

    file = fopen(fileName, "r");
    if (file != NULL) {
        fclose(file);
        return 1;
    }

    return 0;
}

static void updateMenu(int show) {
    if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
        show = 1;
    }

    Prt.showMenu = menuShow(Prt.hwnd,
                            pProperties, 
                            Prt.emuState == EMU_RUNNING, 
                            Prt.emuState == EMU_STOPPED, 
                            Prt.logSound,
                            tempStateExists(),
                            show);

    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        mouseEmuActivate(!show);
    }

    InvalidateRect(Prt.hwnd, NULL, 0);
}

static void ScaleWindow(HWND hwnd) {
    RECT r;
    int x, y, w, h;
    int zoom = getZoom();

    Prt.enteringFullscreen = 1;

    DirectXExitFullscreenMode(hwnd);

    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        statusBarShow(FALSE);
        toolBarShow(FALSE);
    
        SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_POPUP | WS_THICKFRAME | WS_DLGFRAME);

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            int rv;
            int depth = 32;
            switch (pProperties->video.fullRes) {
            case P_VIDEO_FRES320X240_16:
            case P_VIDEO_FRES640X480_16:
                depth = 16;
                break;
            case P_VIDEO_FRES320X240_32:
            case P_VIDEO_FRES640X480_32:
                depth = 32;
                break;
            }
            rv = DirectXEnterFullscreenMode(hwnd, zoom * WIDTH, zoom * HEIGHT, depth);
            if (rv != DXE_OK) {
                DirectXExitFullscreenMode(hwnd);
                rv = DirectXEnterFullscreenMode(hwnd, 2 * WIDTH, 2 * HEIGHT, 16);
            }
            if (rv != DXE_OK) {
                MessageBox(NULL, langErrorEnterFullscreen(), langErrorTitle(), MB_OK);
                DirectXExitFullscreenMode(hwnd);
                pProperties->video.size = P_VIDEO_SIZEX2;
                mouseEmuActivate(1);
                SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                                WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX);
                SetWindowPos(hwnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_SHOWWINDOW); 

                statusBarShow(TRUE);
                toolbarUpdate(zoom == 2, Prt.emuState == EMU_RUNNING);
                toolBarShow(TRUE);

                w = WIDTH * zoom + Prt.DX;
                h = HEIGHT * zoom + Prt.DY + Prt.DTY + Prt.DDY;
                SetWindowPos(hwnd, NULL, Prt.X, Prt.Y, w, h, SWP_NOZORDER);
                
                rv = DirectXEnterWindowedMode(Prt.hwnd, zoom * WIDTH, zoom * HEIGHT);
                if (rv != DXE_OK) {
                    pProperties->video.driver = P_VIDEO_DRVGDI;
                }
                Prt.enteringFullscreen = 0;

                return;
            }
        }

        x = -1 * GetSystemMetrics(SM_CXFIXEDFRAME) - 1;
        y = -1 * GetSystemMetrics(SM_CYFIXEDFRAME) - 1;
        w = GetSystemMetrics(SM_CXSCREEN) - 2 * x;
        h = GetSystemMetrics(SM_CYSCREEN) - 2 * y;

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            SetWindowPos(hwnd, 0, x, y, w, h, SWP_SHOWWINDOW | SWP_NOZORDER);
        }
        else {
            SetWindowPos(hwnd, HWND_TOPMOST, x, y, w, h, SWP_SHOWWINDOW); 
        }
    }
    else {
        toolbarUpdate(zoom == 2, Prt.emuState == EMU_RUNNING);
        if (GetWindowLong(hwnd, GWL_STYLE) & WS_POPUP) {
            mouseEmuActivate(1);
            SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX);
            SetWindowPos(hwnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_SHOWWINDOW); 
        }

        w = WIDTH * zoom + Prt.DX;
        h = HEIGHT * zoom + Prt.DY + Prt.DTY + Prt.DDY;
        SetWindowPos(hwnd, NULL, Prt.X, Prt.Y, w, h, SWP_NOZORDER);
        
        statusBarShow(TRUE);
        toolBarShow(TRUE);

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            int rv = DirectXEnterWindowedMode(Prt.hwnd, zoom * WIDTH, zoom * HEIGHT);
            if (rv != DXE_OK) {
                MessageBox(NULL, langErrorDirectXFailed(), langErrorTitle(), MB_OK);
                pProperties->video.driver = P_VIDEO_DRVGDI;
            }
        }
    }

    Prt.enteringFullscreen = 0;

    GetClientRect(Prt.hwnd, &r);
    r.left += 5;
    r.top += 5;
    r.right -= 5;
    r.bottom -= 5;

    if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
        r.top += Prt.DTY;
    }

    updateMenu(0);
    mouseEmuSetCaptureInfo(&r);
}

void updateVideoRender() {
    switch (pProperties->video.monType) {
    case P_VIDEO_COLOR:
        videoSetColorMode(Prt.pVideo, VIDEO_COLOR);
        break;
    case P_VIDEO_BW:
        videoSetColorMode(Prt.pVideo, VIDEO_BLACKWHITE);
        break;
    case P_VIDEO_GREEN:
        videoSetColorMode(Prt.pVideo, VIDEO_GREEN);
        break;
    }

    switch (pProperties->video.palEmu) {
    case P_VIDEO_PALNONE:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_FAST);
        break;
    case P_VIDEO_PALYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP);
        break;
    case P_VIDEO_PALNYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP_NOISE);
        break;
    case P_VIDEO_PALCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR);
        break;
    case P_VIDEO_PALNCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR_NOISE);
        break;
	case P_VIDEO_PALSCALE2X:
		videoSetPalMode(Prt.pVideo, VIDEO_PAL_SCALE2X);
		break;
    }

    videoSetFrameSkip(Prt.pVideo, pProperties->video.frameSkip);
}

void updateJoystick() {    
    switch (pProperties->joy1.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(1, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(1, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(1, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(1, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(1, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(1, JOY_HW2);
        break;
    }

    JoystickSetKeyStateKey(1, JOY_UP,    pProperties->joy1.keyUp);
    JoystickSetKeyStateKey(1, JOY_DOWN,  pProperties->joy1.keyDown);
    JoystickSetKeyStateKey(1, JOY_LEFT,  pProperties->joy1.keyLeft);
    JoystickSetKeyStateKey(1, JOY_RIGHT, pProperties->joy1.keyRight);
    JoystickSetKeyStateKey(1, JOY_BT1,   pProperties->joy1.button1);
    JoystickSetKeyStateKey(1, JOY_BT2,   pProperties->joy1.button2);

    JoystickSetKeyStateKey(2, JOY_UP,    pProperties->joy2.keyUp);
    JoystickSetKeyStateKey(2, JOY_DOWN,  pProperties->joy2.keyDown);
    JoystickSetKeyStateKey(2, JOY_LEFT,  pProperties->joy2.keyLeft);
    JoystickSetKeyStateKey(2, JOY_RIGHT, pProperties->joy2.keyRight);
    JoystickSetKeyStateKey(2, JOY_BT1,   pProperties->joy2.button1);
    JoystickSetKeyStateKey(2, JOY_BT2,   pProperties->joy2.button2);

    switch (pProperties->joy1.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(1, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(1, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(1, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(1, JOY_AF_FAST);
        break;
    }

    switch (pProperties->joy2.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(2, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(2, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(2, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(2, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(2, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(2, JOY_HW2);
        break;
    }

    switch (pProperties->joy2.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(2, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(2, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(2, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(2, JOY_AF_FAST);
        break;
    }
}

void showPropertiesDialog(int startPane) {
    Properties oldProp = *pProperties;
    int restart = 0;
    int i;

    int changed = showProperties(pProperties, Prt.hwnd, startPane, Prt.mixer);
    if (!changed) {
        return;
    }

    /* Save properties */
    propSave(pProperties);

    /* Always update video render */
    updateVideoRender();

    /* Always update joystick controls */
    updateJoystick();

    /* Update window size only if changed */
    if (pProperties->video.driver != oldProp.video.driver ||
        pProperties->video.fullRes != oldProp.video.fullRes ||
        pProperties->video.size != oldProp.video.size)
    {
        ScaleWindow(Prt.hwnd);
    }

    /* Must restart if video type changed */
    restart |= pProperties->video.videoType != oldProp.video.videoType;

    /* Must restart MSX if joystick type changed between joystick, mouse or none */
    restart |= oldProp.joy1.type == P_JOY_NONE  && pProperties->joy1.type != P_JOY_NONE;
    restart |= oldProp.joy1.type != P_JOY_NONE  && pProperties->joy1.type == P_JOY_NONE;
    restart |= oldProp.joy1.type == P_JOY_MOUSE && pProperties->joy1.type != P_JOY_MOUSE;
    restart |= oldProp.joy1.type != P_JOY_MOUSE && pProperties->joy1.type == P_JOY_MOUSE;
    restart |= oldProp.joy2.type == P_JOY_NONE  && pProperties->joy2.type != P_JOY_NONE;
    restart |= oldProp.joy2.type != P_JOY_NONE  && pProperties->joy2.type == P_JOY_NONE;
    restart |= oldProp.joy2.type == P_JOY_MOUSE && pProperties->joy2.type != P_JOY_MOUSE;
    restart |= oldProp.joy2.type != P_JOY_MOUSE && pProperties->joy2.type == P_JOY_MOUSE;

    mouseEmuEnable(pProperties->joy1.type == P_JOY_MOUSE || pProperties->joy2.type == P_JOY_MOUSE);

    /* Must restart MSX if CPU or ram configuration changed */
    if (memcmp(&oldProp.emulation, &pProperties->emulation, sizeof(oldProp.emulation))) {
        restart = 1;
    }

    /* Update sound only if changed, Must restart if changed */
    if (oldProp.sound.bufSize       != pProperties->sound.bufSize ||
        oldProp.sound.driver        != pProperties->sound.driver  ||
        oldProp.sound.frequency     != pProperties->sound.frequency ||
        oldProp.sound.chip.msxaudio != pProperties->sound.chip.msxaudio ||
        oldProp.sound.chip.msxmusic != pProperties->sound.chip.msxmusic ||
        oldProp.sound.stereo        != pProperties->sound.stereo) 
    {
        soundDestroy();
        soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), pProperties->sound.stereo ? 2 : 1);
        restart = 1;
    }

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }

    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    if (restart) {
        emulatorReset();
    }

    updateMenu(0);

    InvalidateRect(Prt.hwnd, NULL, TRUE);
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

typedef struct {
    int curPos;
    int maxPos;
    int apply;
} TapePosDlgInfo;

static char* convertTapePos(int tapePos)
{
    static char str[64];
    int pos = tapePos / 128;

    _stprintf(str, "%dh %02dm %02ds", pos / 3600, (pos / 60) % 60, pos % 60);
    return str;
}

static BOOL CALLBACK tapePosDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) 
{
    static TapePosDlgInfo* dlgInfo;

    switch (iMsg) {
    case WM_INITDIALOG:
        dlgInfo = (TapePosDlgInfo*)lParam;

        SetWindowText(hDlg, langDlgTapeTitle());

        SendMessage(GetDlgItem(hDlg, IDC_TAPEGROUPBOX), WM_SETTEXT, 0, (LPARAM)langDlgTapeFrameText());
        SendMessage(GetDlgItem(hDlg, IDC_TAPECURPOS), WM_SETTEXT, 0, (LPARAM)langDlgTapeCurrentPos());
        SendMessage(GetDlgItem(hDlg, IDC_TAPETOTALTIME), WM_SETTEXT, 0, (LPARAM)langDlgTapeTotalTime());
        SetWindowText(GetDlgItem(hDlg, IDOK), langDlgOK());
        SetWindowText(GetDlgItem(hDlg, IDCANCEL), langDlgCancel());

        SendMessage(GetDlgItem(hDlg, IDC_TAPECUR), WM_SETTEXT, 0, (LPARAM)convertTapePos(dlgInfo->curPos));
        SendMessage(GetDlgItem(hDlg, IDC_TAPEMAX), WM_SETTEXT, 0, (LPARAM)convertTapePos(dlgInfo->maxPos));

        SendMessage(GetDlgItem(hDlg, IDC_TAPEPOS), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, dlgInfo->maxPos/128));
        SendMessage(GetDlgItem(hDlg, IDC_TAPEPOS), TBM_SETPOS,   1, (LPARAM)(dlgInfo->curPos/128));

        return FALSE;

    case WM_NOTIFY:
        if (wParam == IDC_TAPEPOS) {
            dlgInfo->curPos = 128 * SendMessage(GetDlgItem(hDlg, IDC_TAPEPOS), TBM_GETPOS, 0, 0);
            if (dlgInfo->curPos > dlgInfo->maxPos) {
                dlgInfo->curPos = dlgInfo->maxPos;
            }
            SendMessage(GetDlgItem(hDlg, IDC_TAPECUR), WM_SETTEXT, 0, (LPARAM)convertTapePos(dlgInfo->curPos));
        }
        break;

    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDOK:
            dlgInfo->apply = 1;
            EndDialog(hDlg, TRUE);
            return TRUE;
        case IDCANCEL:
            EndDialog(hDlg, FALSE);
            return TRUE;
        }
        break;

    case WM_CLOSE:
        EndDialog(hDlg, FALSE);
        return TRUE;
    }

    return FALSE;
}

void setTapePosition()
{
    TapePosDlgInfo dlgInfo;

    if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
    }
    else {
        tapeInsert(strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL, 
                   strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL);
        tapeSetCurrentPos(Prt.tapeCurPos);
    }

    dlgInfo.curPos = tapeGetCurrentPos();
    dlgInfo.maxPos = tapeGetLength();
    dlgInfo.apply = 0;

    DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_TAPEPOS), Prt.hwnd, tapePosDlgProc, (LPARAM)&dlgInfo);
    if (dlgInfo.apply) {
        tapeSetCurrentPos(dlgInfo.curPos);
        Prt.tapeCurPos = dlgInfo.curPos;
    }
    if (Prt.emuState != EMU_STOPPED) {
        emulatorResume();
    }
    else {
        tapeInsert(NULL, NULL);
    }
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
typedef struct {
    _TCHAR title[128];
    _TCHAR description[128];
    char* fileList;
    int   fileListCount;
    int   autoReset;
    char  selectFile[512];
} ZipFileDlgInfo;

static BOOL CALLBACK dskZipDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static ZipFileDlgInfo* dlgInfo;

    switch (iMsg) {
    case WM_INITDIALOG:
        {
            char* fileList;
            int sel = 0;
            int i;
            
            dlgInfo = (ZipFileDlgInfo*)lParam;

            SetWindowText(hDlg, dlgInfo->title);

            SendMessage(GetDlgItem(hDlg, IDC_DSKLOADTXT), WM_SETTEXT, 0, (LPARAM)dlgInfo->description);
            SetWindowText(GetDlgItem(hDlg, IDC_DSKRESET), langDlgZipReset());
            SetWindowText(GetDlgItem(hDlg, IDOK), langDlgOK());
            SetWindowText(GetDlgItem(hDlg, IDCANCEL), langDlgCancel());

            fileList = dlgInfo->fileList;

            for (i = 0; i < dlgInfo->fileListCount; i++) {
                if (0 == strcmp(dlgInfo->selectFile, fileList)) {
                    sel = i;
                }
                SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_ADDSTRING, 0, (LPARAM)fileList);
                fileList += strlen(fileList) + 1;
            }

            if (dlgInfo->autoReset == -1) {
                ShowWindow(GetDlgItem(hDlg, IDC_DSKRESET), FALSE);
            }
            else {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, dlgInfo->autoReset ? BST_CHECKED : BST_UNCHECKED, 0);
            }
            SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_SETCURSEL, sel, 0);

            return FALSE;
        }

    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDC_DSKRESET:
            if (dlgInfo->autoReset == 1) {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, BST_UNCHECKED, 0);
                dlgInfo->autoReset = 0;
            }
            else if (dlgInfo->autoReset == 0) {
                SendMessage(GetDlgItem(hDlg, IDC_DSKRESET), BM_SETCHECK, BST_CHECKED, 1);
                dlgInfo->autoReset = 1;
            }
            break;
        case IDC_DSKLIST:
            if (HIWORD(wParam) != 2) {
                break;
            }
            // else, fall through
        case IDOK:
            SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_GETTEXT, SendMessage(GetDlgItem(hDlg, IDC_DSKLIST), LB_GETCURSEL, 0, 0), (LPARAM)dlgInfo->selectFile);
            EndDialog(hDlg, TRUE);
            return TRUE;
        case IDCANCEL:
            dlgInfo->selectFile[0] = '\0';
            EndDialog(hDlg, FALSE);
            return TRUE;
        }
        break;
    case WM_CLOSE:
        dlgInfo->selectFile[0] = '\0';
        EndDialog(hDlg, FALSE);
        return TRUE;
    }

    return FALSE;
}

static int insertCartridge(int drive, char* fname, char* inZipFile) {
    int autostart = Prt.autostart | pProperties->cartridge.autoReset;
    char romName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(romName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".rom", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoRomInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(romName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadRom());
                _stprintf(dlgInfo.description, "%s", langDlgLoadRomDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->cartridge.slotAZip : pProperties->cartridge.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                autostart = dlgInfo.autoReset;
                strcpy(romName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }

    if (drive == 0) {
        strcpy(pProperties->cartridge.slotA, filename);
        strcpy(pProperties->cartridge.slotAZip, romName);
        updateFileHistory(*pProperties->filehistory.cartridgeA, filename);
    }
    else {
        strcpy(pProperties->cartridge.slotB, filename);
        strcpy(pProperties->cartridge.slotBZip, romName);
        updateFileHistory(*pProperties->filehistory.cartridgeB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        ChangeCartridge(drive, strcmp(CARTNAME_SCCPLUS, filename) ? strcmp(CARTNAME_FMPAC, filename) ? ROM_UNKNOWN : ROM_FMPAC : ROM_SCCPLUS, filename, isZip ? romName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertDiskette(int drive, char* fname, char* inZipFile) {
    char diskName[512] = "";
    char filename[512] = "";
    int autostart = Prt.autostart | (drive == 0 ? pProperties->diskdrive.autostartA : 0);
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(diskName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".dsk", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoDskInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(diskName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadDsk());
                _stprintf(dlgInfo.description, "%s", langDlgLoadDskDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->diskdrive.slotAZip : pProperties->diskdrive.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(diskName, dlgInfo.selectFile);
                autostart = dlgInfo.autoReset;
            }
            free(fileList);
        }
    }

    if (drive == 0) {
        strcpy(pProperties->diskdrive.slotA, filename);
        strcpy(pProperties->diskdrive.slotAZip, diskName);
        updateFileHistory(*pProperties->filehistory.diskdriveA, filename);
    }
    else {
        strcpy(pProperties->diskdrive.slotB, filename);
        strcpy(pProperties->diskdrive.slotBZip, diskName);
        updateFileHistory(*pProperties->filehistory.diskdriveB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        ChangeDisk(drive, filename, isZip ? diskName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertCassette(char* fname, char* inZipFile) {
    int autostart = Prt.autostart;
    char tapeName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;

    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(tapeName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".cas", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoCasInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(tapeName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadCas());
                _stprintf(dlgInfo.description, "%s", langDlgLoadCasDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, pProperties->cassette.tapeZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                autostart = dlgInfo.autoReset;
                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(tapeName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }

    strcpy(pProperties->cassette.tape, filename);
    strcpy(pProperties->cassette.tapeZip, tapeName);
    updateFileHistory(*pProperties->filehistory.cassette, filename);

    if (autostart && !Prt.noautostart) {
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        emulatorSuspend();
        tapeInsert(filename, isZip ? tapeName : NULL);
        emulatorResume();
    }

    return 1;
}

static int insertDisketteOrCartridge(int drive, char* filename) {
    ZipFileDlgInfo dlgInfo;
    int countDsk;
    int countRom;
    int countCas;
    char* fileListDsk = zipGetFileList(filename, ".dsk", &countDsk);
    char* fileListRom = zipGetFileList(filename, ".rom", &countRom);
    char* fileListCas = zipGetFileList(filename, ".cas", &countCas);
    char* fileList;
    int sizeDsk = 0;
    int sizeRom = 0;
    int sizeCas = 0;
    int success = 0;
    int i;

    if (fileListDsk == NULL && fileListRom == NULL && fileListCas == NULL) {
        return 0;
    }

    if (fileListDsk == NULL && fileListCas == NULL) {
        free(fileListRom);
        return insertCartridge(drive, filename, NULL);
    }

    if (fileListRom == NULL && fileListCas == NULL) {
        free(fileListDsk);
        return insertDiskette(drive, filename, NULL);
    }

    if (fileListRom == NULL && fileListDsk == NULL) {
        free(fileListCas);
        return insertCassette(filename, NULL);
    }

    for (i = 0; i < countRom; i++) {
        sizeRom += strlen(fileListRom + sizeRom) + 1;
    }

    for (i = 0; i < countDsk; i++) {
        sizeDsk += strlen(fileListDsk + sizeDsk) + 1;
    }

    for (i = 0; i < countCas; i++) {
        sizeCas += strlen(fileListCas + sizeCas) + 1;
    }

    fileList = malloc(sizeDsk + sizeRom + sizeCas);
    memcpy(fileList, fileListRom, sizeRom);
    memcpy(fileList + sizeRom, fileListDsk, sizeDsk);
    memcpy(fileList + sizeRom + sizeDsk, fileListCas, sizeCas);

    _stprintf(dlgInfo.title, "%s", langDlgLoadRomDskCas());
    _stprintf(dlgInfo.description, "%s", langDlgLoadRomDskCasDesc());
    dlgInfo.fileList = fileList;
    dlgInfo.fileListCount = countDsk + countRom + countCas;
    dlgInfo.autoReset = pProperties->diskdrive.autostartA || pProperties->cartridge.autoReset || Prt.autostart;
    dlgInfo.selectFile[0] = 0;

    DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);
    Prt.autostart = dlgInfo.autoReset;

    if (strlen(dlgInfo.selectFile) > 4) {
        if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'R' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'O' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'M')
        {
            success = insertCartridge(0, filename, dlgInfo.selectFile);
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'D' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'S' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'K')
        {
            success = insertDiskette(0, filename, dlgInfo.selectFile);    
        }
        else if (toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 4]) == '.' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 3]) == 'C' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 2]) == 'A' &&
            toupper(dlgInfo.selectFile[strlen(dlgInfo.selectFile) - 1]) == 'S')
        {
            success = insertCassette(filename, dlgInfo.selectFile);    
        }
    }
    
    free(fileListDsk);
    free(fileListRom);
    free(fileList);

    return success;
}

static int tryLaunchUnknownFile(char* fname) 
{
    int success = 0;
    int len = strlen(fname);
    
    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'R' &&
            toupper(fname[len - 2]) == 'O' &&
            toupper(fname[len - 1]) == 'M')
        {
            success = insertCartridge(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'S' &&
            toupper(fname[len - 1]) == 'K')
        {
            success = insertDiskette(0, fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'C' &&
            toupper(fname[len - 2]) == 'A' &&
            toupper(fname[len - 1]) == 'S')
        {
            success = insertCassette(fname, NULL);
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            success = insertDisketteOrCartridge(0, fname);
        }
    }

    return success;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

int getModifiers() {
    return (GetAsyncKeyState(VK_LMENU)   > 1UL ? MOD_ALT     : 0) |
           (GetAsyncKeyState(VK_MENU)    > 1UL ? MOD_ALT     : 0) |
           (GetAsyncKeyState(VK_SHIFT)   > 1UL ? MOD_SHIFT   : 0) |
           (GetAsyncKeyState(VK_CONTROL) > 1UL ? MOD_CONTROL : 0);
}

static LRESULT CALLBACK wndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam) 
{
    PAINTSTRUCT ps;
    HDC hdc;
    RECT r;
    int i;
    char* pFname;

    switch (iMsg) {
    case WM_CREATE:
        keyboardInit(hwnd);
        SetTimer(hwnd, 10, 100, NULL);
        SetTimer(hwnd, 11, 20, NULL);
        DragAcceptFiles(hwnd, TRUE);
        return 0;

    case WM_DROPFILES:
        {
            char fname[MAX_PATH];
            HDROP hDrop;
            hDrop = (HDROP)wParam;
            DragQueryFile(hDrop, 0, fname, 512);
            DragFinish (hDrop);

            tryLaunchUnknownFile(fname);
        }
        return 0;

    case WM_COMMAND:
        i = LOWORD(wParam) - ID_CARTRIDGEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertCartridge(0, pProperties->filehistory.cartridgeA[i], NULL);
        }
        i = LOWORD(wParam) - ID_CARTRIDGEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertCartridge(1, pProperties->filehistory.cartridgeB[i], NULL);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertDiskette(0, pProperties->filehistory.diskdriveA[i], NULL);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertDiskette(1, pProperties->filehistory.diskdriveB[i], NULL);
        }
        i = LOWORD(wParam) - ID_CASSETTE_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertCassette(pProperties->filehistory.cassette[i], NULL);
        }
        
        switch (LOWORD(wParam)) {
        case ID_LOG_WAV:
            if (Prt.logSound) {
                mixerStopLog(Prt.mixer);
                Prt.logSound = 0;
            }
            else {
                Prt.logSound = mixerStartLog(Prt.mixer, wavCaptureCreateFileName());
            }
            break;
        case ID_PRT_SCR:
            {
                int yOffset = 0;
                if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
                    yOffset = Prt.DTY;
                }
			    ScreenShot(hwnd, WIDTH * getZoom(), (HEIGHT - 1) * getZoom(), yOffset);
            }
            break;
        case ID_FILE_SAVE:
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                pFname = saveFile(hwnd, langDlgSaveState(), "CPU state   (*.sta)\0*.sta\0", pProperties->emulation.statsDefDir);
                if (pFname != NULL && strlen(pFname) != 0) {
                    char *ptr = pFname + strlen(pFname) - 1;
                    while(*ptr != '.' && ptr > pFname) {
                        ptr--;
                    }
                    if (ptr == pFname) {
                        ptr = pFname + strlen(pFname);
                    }

                    strcpy(ptr, ".sta");
                    SaveState(pFname);
                }
                emulatorResume();
            }
            break;
        case ID_FILE_LOAD:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgLoadState(), "CPU state   (*.sta)\0*.sta\0", pProperties->emulation.statsDefDir, 1, NULL);
            if (pFname != NULL) {
                emulatorStop();
                strcpy(Prt.pStoredState, pFname);
                emulatorStart(1);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_FILE_QSAVE:
            if (Prt.emuState != EMU_STOPPED) {
                char fileName[512];
                emulatorSuspend();
                sprintf(fileName, "%s\\%s", Prt.pCurDir, "quicksave.sta");
                SaveState(fileName);
                emulatorResume();
            }
            break;
        case ID_FILE_QLOAD:
            if (tempStateExists()) {
                emulatorStop();
                sprintf(Prt.pStoredState, "%s\\%s", Prt.pCurDir, "quicksave.sta");
                emulatorStart(1);
            }
            break;
        case ID_TB_CARTA:
        case ID_FILE_INSERT_CARTRIDGEA:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgInsertRom1(), "ROM cartridge   (*.rom, *.zip)\0*.rom; *.zip\0", pProperties->cartridge.defDir, 1, NULL);
            if (pFname != NULL) {
                insertCartridge(0, pFname, NULL);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_TB_CARTB:
        case ID_FILE_INSERT_CARTRIDGEB:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgInsertRom2(), "ROM cartridge   (*.rom, *.zip)\0*.rom; *.zip\0", pProperties->cartridge.defDir, 1, NULL);
            if (pFname != NULL) {
                insertCartridge(1, pFname, NULL);
            }
            else {
                emulatorResume();
            }
            break;
        case ID_FILE_CARTA_SCCPLUS:
            insertCartridge(0, CARTNAME_SCCPLUS, NULL);
            break;
        case ID_FILE_CARTB_SCCPLUS:
            insertCartridge(1, CARTNAME_SCCPLUS, NULL);
            break;
        case ID_FILE_CARTA_FMPAC:
            insertCartridge(0, CARTNAME_FMPAC, NULL);
            break;
        case ID_FILE_CARTB_FMPAC:
            insertCartridge(1, CARTNAME_FMPAC, NULL);
            break;
        case ID_FILE_CARTRIDGE_AUTORESET:
            pProperties->cartridge.autoReset ^= 1;
            break;
        case ID_FILE_INSERT_DISKETTEA_RESET:
            pProperties->diskdrive.autostartA ^= 1;
            break;
        case ID_TB_DISKA:
        case ID_FILE_INSERT_DISKETTEA:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgInsertDiskA(), "Disk image   (*.dsk, *.zip)\0*.dsk; *.zip\0", pProperties->diskdrive.defDir, 0, ".dsk");
            if (pFname != NULL) {
                insertDiskette(0, pFname, NULL);
            }
            emulatorResume();
            break;
        case ID_TB_DISKB:
        case ID_FILE_INSERT_DISKETTEB:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgInsertDiskB(), "Disk image   (*.dsk, *.zip)\0*.dsk; *.zip\0", pProperties->diskdrive.defDir, 0, ".dsk");
            if (pFname != NULL) {
                insertDiskette(1, pFname, NULL);
            }
            emulatorResume();
            break;
        case ID_TB_CAS:
        case ID_FILE_INSERT_CASSETTE:
            emulatorSuspend();
            pFname = openFile(hwnd, langDlgInsertCas(), "Tape image   (*.cas, *.zip)\0*.cas; *.zip\0", pProperties->cassette.defDir, 0, ".cas");
            if (pFname != NULL) {
                insertCassette(pFname, NULL);
            }
            emulatorResume();
            break;
            
        case ID_FILE_REMOVE_CARTRIDGEA:
            pProperties->cartridge.slotA[0] = 0;
            pProperties->cartridge.slotAZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                if (pProperties->cartridge.autoReset) {
                    emulatorStop();
                    emulatorStart(1);
                }
                else {
                    emulatorSuspend();
                    ChangeCartridge(0, ROM_UNKNOWN, "CARTA.ROM", NULL);
                    emulatorResume();
                }
            }
            break;
        case ID_FILE_REMOVE_CARTRIDGEB:
            pProperties->cartridge.slotB[0] = 0;
            pProperties->cartridge.slotBZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                if (pProperties->cartridge.autoReset) {
                    emulatorStop();
                    emulatorStart(1);
                }
                else {
                    emulatorSuspend();
                    ChangeCartridge(1, ROM_UNKNOWN, "CARTB.ROM", NULL);
                    emulatorResume();
                }
            }
            break;
        case ID_FILE_REMOVE_DISKETTEA:
            pProperties->diskdrive.slotA[0] = 0;
            pProperties->diskdrive.slotAZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                ChangeDisk(0, NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_REMOVE_DISKETTEB:
            pProperties->diskdrive.slotB[0] = 0;
            pProperties->diskdrive.slotBZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                ChangeDisk(1, NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_REMOVE_CASSETTE:
            pProperties->cassette.tape[0] = 0;
            pProperties->cassette.tapeZip[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                tapeInsert(NULL, NULL);
                emulatorResume();
            }
            break;
        case ID_FILE_POSITION_CASSETTE:
                setTapePosition();
            break;
        case ID_FILE_REWIND_CASSETTE:
            if (Prt.emuState != EMU_STOPPED) {
                emulatorSuspend();
                tapeSetCurrentPos(0);
                emulatorResume();
            }
            Prt.tapeCurPos = 0;
            break;
        case ID_FILE_EXIT:
            DestroyWindow(hwnd);
            break;
        case ID_SIZE_NORMAL:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX1)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_X2:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX2)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_FULLSCREEN:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_TB_PLAYPAUSE:
        case ID_RUN_RUN:
            if (Prt.emuState == EMU_STOPPED) {
                emulatorStart(1);
                toolbarUpdate(getZoom() == 2, 1);
            }
            else if (Prt.emuState == EMU_PAUSED) {
                soundResume();
                toolbarUpdate(getZoom() == 2, 1);
                Prt.emuState = EMU_RUNNING;
                mouseEmuUpdateRunState(1);
            }
            else {
                soundSuspend();
                toolbarUpdate(getZoom() == 2, 0);
                Prt.emuState = EMU_PAUSED;
                mouseEmuUpdateRunState(0);
            }
            break;
        case ID_RUN_STOP:
            if (Prt.emuState != EMU_STOPPED) {
                toolbarUpdate(getZoom() == 2, 0);
                emulatorStop();
                InvalidateRect(Prt.hwnd, NULL, TRUE);
            }
            break;
        case ID_TB_RESET:
        case ID_RUN_RESET:
            emulatorStop();
            toolbarUpdate(getZoom() == 2, 1);
            emulatorStart(1);
            break;
        case ID_RUN_SOFTRESET:
            emulatorStop();
            toolbarUpdate(getZoom() == 2, 1);
            emulatorStart(0);
            break;
        case ID_RUN_CLEANRESET:
            emulatorStop();
            pProperties->diskdrive.slotA[0] = 0;
            pProperties->diskdrive.slotAZip[0] = 0;
            pProperties->diskdrive.slotB[0] = 0;
            pProperties->diskdrive.slotBZip[0] = 0;
            pProperties->cartridge.slotA[0] = 0;
            pProperties->cartridge.slotAZip[0] = 0;
            pProperties->cartridge.slotB[0] = 0;
            pProperties->cartridge.slotBZip[0] = 0;
            toolbarUpdate(getZoom() == 2, 1);
            emulatorStart(1);
            break;
        case ID_OPTIONS_EMULATION:
            showPropertiesDialog(PROP_EMULATION);
            break;
        case ID_OPTIONS_VIDEO:
            showPropertiesDialog(PROP_VIDEO);
            break;
        case ID_OPTIONS_AUDIO:
            showPropertiesDialog(PROP_SOUND);
            break;
        case ID_TB_PROPERTIES:
        case ID_OPTIONS_CONTROLS:
            showPropertiesDialog(PROP_CONTROLS);
            break;
        case ID_OPTIONS_PERFORMANCE:
            showPropertiesDialog(PROP_PERFORMANCE);
            break;
        case ID_OPTIONS_LANGUAGE:
            {
                int lang = langShowDlg(Prt.hwnd, pProperties->language);
                int success = langSetLanguage(lang);
                if (success) {
                    pProperties->language = lang;
                }
            }
            break;
        case ID_TB_HELP:
        case ID_HELP_HELP:
            {
                char curDir[MAX_PATH];
                HINSTANCE rv;

                GetCurrentDirectory(MAX_PATH - 1, curDir);
                SetCurrentDirectory(Prt.pCurDir);
                rv = ShellExecute(Prt.hwnd, "open", "blueMSX.chm", NULL, NULL, SW_SHOWNORMAL);
                if (rv <= (HINSTANCE)32) {
                    MessageBox(NULL, langErrorNoHelp(), langErrorTitle(), MB_OK);
                }
                SetCurrentDirectory(curDir);
            }
            break;
        case ID_HELP_ABOUT:
            helpShowAbout(hwnd);
            break;
        }

        updateMenu(0);
        break;

    case WM_NOTIFY:
        switch(((LPNMHDR)lParam)->code){
        case TBN_DROPDOWN:
            {
                RECT      rc;
                TPMPARAMS tpm;
                HMENU     hMenu = NULL;
                LPNMTOOLBAR ntb = (LPNMTOOLBAR)lParam;

                SendMessage(ntb->hdr.hwndFrom, TB_GETRECT, (WPARAM)ntb->iItem, (LPARAM)&rc);

                MapWindowPoints(ntb->hdr.hwndFrom, HWND_DESKTOP, (LPPOINT)&rc, 2);                         

                tpm.cbSize = sizeof(TPMPARAMS);
                tpm.rcExclude = rc;

                switch (ntb->iItem) {
                case ID_TB_RESET:
                    hMenu = menuCreateReset(pProperties);
                    break;
                case ID_TB_CARTA:
                    hMenu = menuCreateCartA(pProperties);
                    break;
                case ID_TB_CARTB:
                    hMenu = menuCreateCartB(pProperties);
                    break;
                case ID_TB_DISKA:
                    hMenu = menuCreateDiskA(pProperties);
                    break;
                case ID_TB_DISKB:
                    hMenu = menuCreateDiskB(pProperties);
                    break;
                case ID_TB_CAS:
                    hMenu = menuCreateCassette(pProperties);
                    break;
                case ID_TB_ZOOM:
                    hMenu = menuCreateZoom(pProperties);
                    break;
                case ID_TB_PROPERTIES:
                    hMenu = menuCreateOptions(pProperties);
                    break;
                }
                if (hMenu != NULL) {
                    TrackPopupMenuEx(hMenu, TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_VERTICAL,               
                                    rc.left, rc.bottom, Prt.hwnd, &tpm); 
                    DestroyMenu(hMenu);
                }
            }
            break;

        case TTN_GETDISPINFO: 
            { 
                LPTOOLTIPTEXT lpttt = (LPTOOLTIPTEXT)lParam; 
                lpttt->hinst = GetModuleHandle(NULL); 

                switch (lpttt->hdr.idFrom) { 
                case ID_TB_RESET:
                    _stprintf(lpttt->szText, "%s", langTooltipReset());
                    break; 
                case ID_TB_CARTA:
                    _stprintf(lpttt->szText, "%s", langTooltipCart1());
                    break;
                case ID_TB_CARTB:
                    _stprintf(lpttt->szText, "%s", langTooltipCart2());
                    break;
                case ID_TB_DISKA:
                    _stprintf(lpttt->szText, "%s", langTooltipDiskA());
                    break;
                case ID_TB_DISKB:
                    _stprintf(lpttt->szText, "%s", langTooltipDiskB());
                    break;
                case ID_TB_CAS:
                    _stprintf(lpttt->szText, "%s", langTooltipCas());
                    break;
                case ID_TB_PLAYPAUSE:
                    _stprintf(lpttt->szText, "%s", Prt.emuState == EMU_RUNNING ? 
                              langTooltipPause() : Prt.emuState == EMU_PAUSED ? 
                              langTooltipResume() : langTooltipStart());
                    break;
                case ID_TB_ZOOM:
                    _stprintf(lpttt->szText, "%s", langTooltipWindowSize());
                    break;
                case ID_TB_PROPERTIES:
                    _stprintf(lpttt->szText, "%s", langTooltipProperties());
                    break;
                case ID_TB_HELP:
                    _stprintf(lpttt->szText, "%s", langTooltipHelp());
                    break;
                }
            }
            break; 
        }
        break;

    case WM_SYSKEYDOWN:
    case WM_KEYDOWN:
        switch (wParam) {
        case VK_CANCEL:
            SendMessage(hwnd, WM_COMMAND, ID_FILE_EXIT, 0);
            return 0;
        case VK_F6:
            i = getModifiers();
            if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_LOG_WAV, 0);
            }
            return 0;
        case VK_F7:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_LOAD, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_CARTRIDGEA, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QLOAD, 0);
            }
            return 0;
        case VK_F8:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_SAVE, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_CARTRIDGEB, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QSAVE, 0);
            }
            return 0;
        case VK_F9:
            i = getModifiers();
            if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RUN, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_DISKETTEA, 0);
            }
            return 0;
        case VK_F10:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_NORMAL, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_INSERT_DISKETTEB, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_STOP, 0);
            }
            else if (i == MOD_SHIFT) {
                pProperties->emulation.speed = 50;
                emulatorSetFrequency(pProperties->emulation.speed);
            }
            return 0;
        case VK_F11:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_X2, 0);
            }
            else if (i == MOD_SHIFT) {
                if (pProperties->emulation.speed > 0) {
                    pProperties->emulation.speed--;
                    emulatorSetFrequency(pProperties->emulation.speed);
                }
            }
            return 0;
        case VK_F12:
            i = getModifiers();
            if (i == MOD_ALT) {
                SendMessage(hwnd, WM_COMMAND, ID_SIZE_FULLSCREEN, 0);
            }
            else if (i == MOD_CONTROL) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_SOFTRESET, 0);
            }
            else if (i == (MOD_CONTROL | MOD_ALT)) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_CLEANRESET, 0);
            }
            else if (i == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RESET, 0);
            }
            else if (i == MOD_SHIFT) {
                if (pProperties->emulation.speed < 100) {
                    pProperties->emulation.speed++;
                    emulatorSetFrequency(pProperties->emulation.speed);
                }
            }
            return 0;
        }

        return 0;

    case WM_SYSKEYUP:
    case WM_KEYUP:
        switch (wParam) {
		case VK_SNAPSHOT:
            if (getModifiers() == 0) {
                SendMessage(hwnd, WM_COMMAND, ID_PRT_SCR, 0);
            }
            return 0;
        }
            
        return 0;

    case WM_CHAR:
    case WM_SYSCHAR:
        return 0;

    case WM_SYSCOMMAND:
        switch(wParam) {
        case SC_MAXIMIZE:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            return 0;
        case SC_MINIMIZE:
            emulatorSuspend();
            break;
        case SC_RESTORE:
            emulatorResume();
            break;
        }
        break;

    case WM_ENTERSIZEMOVE:
        emulatorSuspend();
        break;

    case WM_EXITSIZEMOVE:
        emulatorResume();
        break;

    case WM_ENTERMENULOOP:
        emulatorSuspend();
        return 0;

    case WM_EXITMENULOOP:
        emulatorResume();
        updateMenu(0);
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            PostMessage(hwnd, WM_LBUTTONDOWN, 0, 0);
            PostMessage(hwnd, WM_LBUTTONUP, 0, 0);
        }
        return 0;

    case WM_MOVE:
        if (!Prt.enteringFullscreen && pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }

    case WM_DISPLAYCHANGE:
        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            int zoom = getZoom();
            if (Prt.enteringFullscreen) {
                DirectXUpdateWindowedMode(hwnd, zoom * WIDTH, zoom * HEIGHT);
            }
        }
        break;

    case WM_SIZE:
        statusBarUpdatePos();
        InvalidateRect(hwnd, NULL, FALSE);
        break;
        
    case WM_ACTIVATE:
        if (wParam == WA_INACTIVE) {
            mouseEmuActivate(0);
        }
        else {
            mouseEmuActivate(1);
        }
        break;

    case WM_MOUSEMOVE:
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            if (HIWORD(lParam) < 2) {
                if (!Prt.showMenu) {
                    Sleep(200);
                }
                updateMenu(1);
            }
        }
        break;

    case WM_LBUTTONDOWN:
        updateMenu(0);
        break;

    case WM_ERASEBKGND:
        return 1;

    case WM_TIMER:
        if (wParam == 10) {
            statusBarUpdate(Prt.emuState == EMU_STOPPED, 
                            Prt.emuState == EMU_PAUSED, 
                            Prt.screenMode, 
                            pProperties->emulation.speed,
                            Prt.ledState); 
        }
        if (wParam == 11) {
            keyboardUpdate(Prt.keyMap);
        }
        break;

    case WM_UPDATE:
        InvalidateRect(hwnd, NULL, FALSE);
        return 0;

    case WM_INPUTLANGCHANGE:
        break;

    case WM_PAINT:
        {
            int dy = 0;
            int dsy = 0;
            int yOffset = 0;
            BITMAP bm;
            HDC hMemDC;
            HBITMAP hBitmap;

            hdc = BeginPaint(hwnd, &ps);   

            GetClientRect(hwnd, &r);
            if (Prt.showMenu && pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
                int cy = GetSystemMetrics(SM_CYSCREEN);
                dy = r.bottom - cy;
                r.bottom = cy;
            }
            else if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
                yOffset = Prt.DTY;
                dsy = Prt.DY;
            }

            if (Prt.emuState == EMU_STOPPED) {
                hMemDC = CreateCompatibleDC(hdc);
                hBitmap = (HBITMAP)SelectObject(hMemDC, Prt.hLogo);
                GetObject(Prt.hLogo, sizeof(BITMAP), (PSTR)&bm);
                {
                    int x = ((r.right - r.left) - bm.bmWidth) / 2;
                    int y = ((r.bottom - r.top - dsy) - bm.bmHeight) / 2 + dy;

                    if (x > 0 || y > 0) {
                        StretchBlt(hdc, 0, dy, r.right - r.left, r.bottom - r.top, hMemDC, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
                    }
                    else {
                        BitBlt(hdc, x, y + yOffset, bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
                    }
                }
                SelectObject(hMemDC, hBitmap);
                DeleteDC(hMemDC);
            }
            else {
                int zoom = getZoom();

                if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
                    if (Prt.emuState != EMU_STOPPED) {
                        DirectXUpdateSurface(Prt.pVideo, Prt.bmBits, WIDTH, HEIGHT, Prt.srcArr, dy, yOffset, zoom, Prt.evenOdd, Prt.interlace);
                    }
                }
                else {
                    int width = r.right - r.left;
                    int height = r.bottom - r.top - yOffset - (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? Prt.DDY : 0);
                    DWORD* bmBitsSrc = (DWORD*)Prt.bmBits + WIDTH * (HEIGHT - 1) * 2;
                    
                    videoRender(Prt.pVideo, 32, zoom, Prt.evenOdd, Prt.interlace, bmBitsSrc, WIDTH, HEIGHT, Prt.srcArr, Prt.bmBitsGDI, 
                                -1 * (int)sizeof(DWORD) * WIDTH, zoom * WIDTH * sizeof(DWORD));
                    Prt.bmInfo.bmiHeader.biWidth    = zoom * WIDTH;
                    Prt.bmInfo.bmiHeader.biHeight   = zoom * HEIGHT;
                    Prt.bmInfo.bmiHeader.biBitCount = 32;
                    StretchDIBits(hdc, 0, dy + yOffset + (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? 1 : 0), 
                                  width, height, 0, 0, zoom * WIDTH, zoom * HEIGHT, Prt.bmBitsGDI, 
                                  &Prt.bmInfo, DIB_RGB_COLORS, SRCCOPY);
                }
            }
            EndPaint(hwnd, &ps);
            Prt.updatePending = 0;
        }
        return 0;
        
    case WM_DESTROY:         
        emulatorStop();
        Sleep(400);
        if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }
        Prt.enteringFullscreen = 1;
        DirectXExitFullscreenMode(hwnd);
        PostQuitMessage(0);
        keyboardDestroy();
        return 0;
    }

    return DefWindowProc(hwnd, iMsg, wParam, lParam);
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////

char* getCmdArgument(char* szLine, int argNo) {
    static char argBuf[512];
    int i;

    for (i = 0; i <= argNo; i++) {
        char* arg = argBuf;

        while (*szLine == ' ') szLine++;

        if (*szLine == 0) return NULL;

        if (*szLine == '\"') {
            szLine++;
            while (*szLine != '\"' && *szLine != 0) {
                *arg++ = *szLine++;
            }
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
        else {
            do {
                *arg++ = *szLine++;
            } while (*szLine != ' ' && *szLine != 0);
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
    }
    return argBuf;
}

static RomType romNameToType(char* name) {
    RomType romType = ROM_UNKNOWN;

    if (name == NULL) {
        return ROM_UNKNOWN;
    }

    if (0 == strcmp(name, "standard"))    romType = ROM_STANDARD;
    if (0 == strcmp(name, "msxdos2"))     romType = ROM_MSXDOS2;
    if (0 == strcmp(name, "konami5"))     romType = ROM_KONAMI5;
    if (0 == strcmp(name, "konami4"))     romType = ROM_KONAMI4;
    if (0 == strcmp(name, "ascii8"))      romType = ROM_ASCII8;
    if (0 == strcmp(name, "ascii16"))     romType = ROM_ASCII16;
    if (0 == strcmp(name, "gamemaster2")) romType = ROM_GAMEMASTER2;
    if (0 == strcmp(name, "ascii8sram"))  romType = ROM_ASCII8SRAM;
    if (0 == strcmp(name, "ascii16sram")) romType = ROM_ASCII16SRAM;
    if (0 == strcmp(name, "konami4nf"))   romType = ROM_KONAMI4NF;
    if (0 == strcmp(name, "ascii16nf"))   romType = ROM_ASCII16NF;
    if (0 == strcmp(name, "sccplus"))     romType = ROM_SCCPLUS;
    if (0 == strcmp(name, "fmpac"))       romType = ROM_FMPAC;
    if (0 == strcmp(name, "rtype"))       romType = ROM_RTYPE;
    if (0 == strcmp(name, "crossblaim"))  romType = ROM_CROSSBLAIM;
    if (0 == strcmp(name, "harryfox"))    romType = ROM_HARRYFOX;
    if (0 == strcmp(name, "korean80"))    romType = ROM_KOREAN80;
    if (0 == strcmp(name, "korean126"))   romType = ROM_KOREAN126;

    if (romType == ROM_UNKNOWN) {
        romType = atoi(name);
        if (romType < ROM_STANDARD || romType > ROM_KOREAN126) {
            romType = ROM_UNKNOWN;
        }
    }

    return romType;
}

static PropEmuFamily famNameToType(char* name) {
    PropEmuFamily family = -1;

    if (name == NULL) {
        return family;
    }

    if (0 == strcmp(name, "msx"))    family = P_EMU_MSX;
    if (0 == strcmp(name, "msxbr"))  family = P_EMU_MSXBR;
    if (0 == strcmp(name, "msxj"))   family = P_EMU_MSXJ;
    if (0 == strcmp(name, "msxkr"))  family = P_EMU_MSXKR;
    if (0 == strcmp(name, "msx2"))   family = P_EMU_MSX2;
    if (0 == strcmp(name, "msx2j"))  family = P_EMU_MSX2J;
    if (0 == strcmp(name, "msx2kr")) family = P_EMU_MSX2KR;
    if (0 == strcmp(name, "msx2p"))  family = P_EMU_MSX2P;

    return family;
}

int isRomFileType(char* fname) {
    int len = strlen(fname);

//    if (fname[len-1] == '\"') len--;

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'R' &&
            toupper(fname[len - 2]) == 'O' &&
            toupper(fname[len - 1]) == 'M')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

int isDskFileType(char* fname) {
    int len = strlen(fname);

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'S' &&
            toupper(fname[len - 1]) == 'K')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

int emuCheckResetArgument(PSTR szLine) {
    int i;
    char*   argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/reset") == 0) {
            return 1;
        }
    }

    return 0;
}

int emuCheckIniFileArgument(PSTR szLine) {
    int i;
    char*   argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/inifile") == 0) {
            return 1;
        }
    }

    return 0;
}

int emuCheckLanguageArgument(PSTR szLine, int defaultLang) {
    int i;
    int lang;
    char* argument;
    
    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/language") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return defaultLang;
            lang = langFromName(argument);
            return lang == EMU_LANG_UNKNOWN ? defaultLang : lang;
        }
    }

    return defaultLang;
}

int emuStartWithArguments(PSTR szLine) {
    int i;
    char*   argument;
    char    rom1[512] = "";
    char    rom2[512] = "";
    RomType romType1  = ROM_UNKNOWN;
    RomType romType2  = ROM_UNKNOWN;
    PropEmuFamily family = -1;
    char    diskA[512] = "";
    char    diskB[512] = "";
    int     fullscreen = 0;
    int     startEmu = 0;

    // If one argument, assume it is a rom or disk to run
    if (!getCmdArgument(szLine, 1)) {
        argument = getCmdArgument(szLine, 0);
        
        if (*argument != '/') {
            if (*argument == '\"') argument++;

            if (*argument) {
                Prt.autostart = 1;
                return tryLaunchUnknownFile(argument);
            }
            return 0;
        }
    }

    // If more than one argument, check arguments,
    // set configuration and then run

    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/rom1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom1, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType1 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/rom2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom2, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType2 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskA") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskA, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskB") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskB, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/family") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            family = famNameToType(argument);
            if (family == -1) return 0;
            startEmu = 1;
        }
        if (strcmp(argument, "/fullscreen") == 0) {
            fullscreen = 1;
        }
    }

    if (!startEmu) {
        return 1;
    }

    pProperties->cartridge.slotA[0] = 0;
    pProperties->cartridge.slotAZip[0] = 0;
    pProperties->cartridge.slotB[0] = 0;
    pProperties->cartridge.slotBZip[0] = 0;

    pProperties->diskdrive.slotA[0] = 0;
    pProperties->diskdrive.slotAZip[0] = 0;
    pProperties->diskdrive.slotB[0] = 0;
    pProperties->diskdrive.slotBZip[0] = 0;
    
    Prt.noautostart = 1;

    if (strlen(rom1)  && !insertCartridge(0, rom1, NULL)) return 0;
    if (strlen(rom2)  && !insertCartridge(1, rom2, NULL)) return 0;
    if (strlen(diskA) && !insertDiskette(0, diskA, NULL)) return 0;
    if (strlen(diskB) && !insertDiskette(1, diskB, NULL)) return 0;

    if (family != -1) pProperties->emulation.family = family;

    propSetVideoSize(pProperties, fullscreen ? P_VIDEO_SIZEFULLSCREEN : P_VIDEO_SIZEX2);

    Prt.noautostart = 0;

    emulatorStop();
    emulatorStart(1);

    return 1;
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////

#ifdef _CONSOLE
int main(int argc, char **argv)
#else
WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, PSTR szLine, int iShow)
#endif
{
#ifdef _CONSOLE
    char szLine[8192] = "";
#endif
    char buffer[512];
    static WNDCLASSEX wndClass;
    HINSTANCE hInstance = (HINSTANCE)GetModuleHandle(NULL);
    HWND hwndStatus;
    HWND hwndToolbar;
    RECT wr;
    RECT cr;
    RECT sr;
    RECT tr;
    MSG msg;
    int i;

#ifdef _CONSOLE
    for (i = 1; i < argc; i++) {
        strcat(szLine, argv[i]);
        strcat(szLine, " ");
    }
#endif

	mkdir("SRAM");

    InitCommonControls(); 

    wndClass.cbSize         = sizeof(wndClass);
    wndClass.style          = CS_OWNDC;
    wndClass.lpfnWndProc    = wndProc;
    wndClass.cbClsExtra     = 0;
    wndClass.cbWndExtra     = 0;
    wndClass.hInstance      = hInstance;
    wndClass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON9));
    wndClass.hIconSm        = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON9));
    wndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground  = NULL;
    wndClass.lpszMenuName   = NULL;
    wndClass.lpszClassName  = "blueMSX";

    RegisterClassEx(&wndClass);

    pProperties = propCreate(emuCheckResetArgument(szLine), emuCheckIniFileArgument(szLine));

    memset(Prt.keyMap, 0xff, 16);

    Prt.ledState = 0;
    Prt.tapeCurPos = 0;
    Prt.autostart = 0;
    Prt.noautostart = 0;
    Prt.screenMode = 0;
    Prt.enteringFullscreen = 1;
    Prt.emuState = EMU_STOPPED;
    Prt.hLogo = LoadBitmap(hInstance, "LOGO");
    Prt.updatePending = 0;
    Prt.pVideo = videoCreate();
    Prt.mixer  = mixerCreate();
    
    /* Set SCREEN8 colors */

    Prt.X = CW_USEDEFAULT;
    Prt.Y = CW_USEDEFAULT;

    GetCurrentDirectory(MAX_PATH - 1, Prt.pCurDir);

    sprintf(buffer, "%s\\Screenshots", Prt.pCurDir);
    InitScreenShot(buffer);
    
    sprintf(buffer, "%s\\Audio Capture", Prt.pCurDir);
    wavCaptureInit(buffer);

    langInit();

    pProperties->language = emuCheckLanguageArgument(szLine, pProperties->language);

    langSetLanguage(pProperties->language);

    Prt.hwnd = CreateWindow("blueMSX", langDlgMainWindow(), 
                            WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX, 
                            CW_USEDEFAULT, CW_USEDEFAULT, 800, 200, NULL, 
                            menuCreate(pProperties, 0, 1, 0, tempStateExists()), hInstance, NULL);


    hwndStatus = statusBarInit(Prt.hwnd);
    hwndToolbar = toolBarInit(Prt.hwnd);

    GetWindowRect(Prt.hwnd, &wr);
    GetClientRect(Prt.hwnd, &cr);
    GetWindowRect(hwndStatus, &sr);
    GetWindowRect(hwndToolbar, &tr);

    Prt.X  = wr.left;
    Prt.Y  = wr.top;
    Prt.DY = (wr.bottom - wr.top) - (cr.bottom - cr.top);
    Prt.DDY = (sr.bottom - sr.top) - 1;
    Prt.DTY = (tr.bottom - tr.top) - 1;
    Prt.DX = (wr.right - wr.left) - (cr.right - cr.left);

    mouseEmuInit(Prt.hwnd, 1);
    mouseEmuEnable(pProperties->joy1.type == P_JOY_MOUSE || pProperties->joy2.type == P_JOY_MOUSE);

    SetWindowPos(Prt.hwnd, NULL, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOZORDER);

    Prt.enteringFullscreen = 0;

    Prt.bmInfo.bmiHeader.biSize           = sizeof(BITMAPINFOHEADER);
    Prt.bmInfo.bmiHeader.biWidth          = WIDTH;
    Prt.bmInfo.bmiHeader.biHeight         = HEIGHT;
    Prt.bmInfo.bmiHeader.biPlanes         = 1;
    Prt.bmInfo.bmiHeader.biBitCount       = 32;
    Prt.bmInfo.bmiHeader.biCompression    = BI_RGB;
    Prt.bmInfo.bmiHeader.biClrUsed        = 0;
    Prt.bmInfo.bmiHeader.biClrImportant   = 0;

    Prt.bmBits = calloc(4 * WIDTH * HEIGHT, sizeof(UInt32));
    Prt.srcArr = calloc(256, sizeof(int));
    Prt.bmBitsGDI = malloc(4096 * 4096 * sizeof(UInt32));
    Prt.logSound = 0;

    soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), pProperties->sound.stereo ? 2 : 1);

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }
    
    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    updateVideoRender();
    updateJoystick();

    if (*szLine) {
        int success = emuStartWithArguments(szLine);
        if (!success) {
            exit(0);
            return 0;
        }
    }
    
    ScaleWindow(Prt.hwnd);

    ShowWindow(Prt.hwnd, TRUE);
    UpdateWindow(Prt.hwnd);

    while (GetMessage(&msg, NULL, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
    
    Sleep(500);
    videoDestroy(Prt.pVideo);
    propDestroy(pProperties);

    soundDestroy();
    Sleep(500);
    mixerDestroy(Prt.mixer);
    
    exit(0);

    return 0;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

UInt32  emuFixedPalette[256];
UInt32  emuPalette0;
UInt32  emuPalette[300];
UInt32* emuFrameBuffer;
int*    emuLineWidth;

static HANDLE       emuThread;
static HANDLE       emuSyncSem;
static void*        emuTimer;
static volatile int emuSuspendFlag;
static int          emuExitFlag;

extern int  SyncPeriod;
extern int MsxFrequency;
extern Mixer* TheMixer;


static DWORD WINAPI emulatorThread(void* param) {
    int success;

    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_ABOVE_NORMAL);

    success = StartMSX();
    TrashMSX();

    Prt.ledState = 0;
    StopTimer(emuTimer);

    if (!success) {
        MessageBox(NULL, langErrorStartEmu(), "blueMSX - Error", MB_ICONHAND | MB_OK);
        PostMessage(Prt.hwnd, WM_COMMAND, ID_RUN_STOP, 0);
    }

    ExitThread(0);
    
    return 0;
}

int emulatorGetSyncPeriod() {
    return pProperties->emulation.syncMethod == P_EMU_SYNC1MS ? 1 : 10;
}

void emulatorSetFrequency(int logFrequency) {
    SyncPeriod = (int)(3580 * emulatorGetSyncPeriod() * pow(2.0, (logFrequency - 50) / 15.0515));
    MsxFrequency = (int)(3579545 * pow(2.0, (logFrequency - 50) / 15.0515));
}

void emulatorSuspend() {
    if (Prt.emuState == EMU_RUNNING) {
        Prt.emuState = EMU_SUSPENDED;
        mouseEmuUpdateRunState(0);
        do {
            Sleep(10);
        } while (!emuSuspendFlag);
        soundSuspend();
    }
}

void emulatorResume() {
    if (Prt.emuState == EMU_SUSPENDED) {
        Prt.emuState = EMU_RUNNING;
        mouseEmuUpdateRunState(1);
        soundResume();
    }
}

void emulatorReset() {
    EmuState emuState = Prt.emuState;
    mouseEmuUpdateRunState(0);

    if (emuState == EMU_RUNNING) {
        Prt.emuState = EMU_PAUSED;
        soundSuspend();
    }
    Sleep(500);
    if (emuState != EMU_STOPPED) {
        char fileName[512];
        sprintf(fileName, "%s\\%s", Prt.pCurDir, "temp.sta");
        SaveState(fileName);
    }
    emulatorStop();
    if (emuState != EMU_STOPPED) {
        sprintf(Prt.pStoredState, "%s\\%s", Prt.pCurDir, "temp.sta");
    }

    Prt.emuState = emuState;
    if (emuState == EMU_RUNNING) {
        emulatorStart(1);
        mouseEmuUpdateRunState(1);
    }
    else {
        soundSuspend();
    }
}

void emulatorRestartSound(int stereo) {
    emulatorSuspend();
    soundDestroy();
    soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), stereo ? 2 : 1);
    emulatorResume();
}

void emulatorStart(int hard) {
    DWORD id;
    int i;

    emuExitFlag = 0;

    ResetMSX(hard);

    for(i=0; i<256; i++) {
        emuFixedPalette[i] = videoGetColor(Prt.pVideo, (i & 0x1c) << 3, (i & 0xe0), ((i & 3) == 3 ? 7 : 2 * (i & 3)) << 5);
    }

    ProgDir        = Prt.pCurDir;
    emuFrameBuffer = (UInt32*)Prt.bmBits;
    emuLineWidth           = (int*)Prt.srcArr;

    emulatorSetFrequency(pProperties->emulation.speed);

    enableYM2413 = pProperties->sound.chip.msxmusic;
    enableY8950  = pProperties->sound.chip.msxaudio;
    VPeriod    = pProperties->video.videoType == P_VIDEO_PAL ? CPU_VPERIOD_PAL : CPU_VPERIOD_NTSC;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotA)) ROMTypeA = ROM_SCCPLUS;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotB)) ROMTypeB = ROM_SCCPLUS;
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotA)) ROMTypeA = ROM_FMPAC;
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotB)) ROMTypeB = ROM_FMPAC;
    CartA      = strlen(pProperties->cartridge.slotA) ? pProperties->cartridge.slotA : NULL;
    CartB      = strlen(pProperties->cartridge.slotB) ? pProperties->cartridge.slotB : NULL;
    CartAZip   = strlen(pProperties->cartridge.slotAZip) ? pProperties->cartridge.slotAZip : NULL;
    CartBZip   = strlen(pProperties->cartridge.slotBZip) ? pProperties->cartridge.slotBZip : NULL;
    DiskA      = strlen(pProperties->diskdrive.slotA) ? pProperties->diskdrive.slotA : NULL;
    DiskB      = strlen(pProperties->diskdrive.slotB) ? pProperties->diskdrive.slotB : NULL;
    DiskAZip   = strlen(pProperties->diskdrive.slotAZip) ? pProperties->diskdrive.slotAZip : NULL;
    DiskBZip   = strlen(pProperties->diskdrive.slotBZip) ? pProperties->diskdrive.slotBZip : NULL;
    CasName    = strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL;
    CasNameZip = strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL;
    CasPos     = Prt.tapeCurPos;
    JoyTypeA   = pProperties->joy1.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy1.type == P_JOY_MOUSE ? 3 : 1;
    JoyTypeB   = pProperties->joy2.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy2.type == P_JOY_MOUSE ? 3 : 1;
    StateName  = strlen(Prt.pStoredState) ? Prt.pStoredState : NULL;
    switch (pProperties->emulation.family) {
    case P_EMU_MSX:
        MSXVersion   = 0;
        MSXSysRom    = romMSX;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX.cmos";
        break;
    case P_EMU_MSXBR:
        MSXVersion   = 0;
        MSXSysRom    = romMSXBR;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXBR.cmos";
        break;
    case P_EMU_MSXJ:
        MSXVersion   = 0;
        MSXSysRom    = romMSXJ;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXJ.cmos";
        break;
    case P_EMU_MSXKR:
        MSXVersion   = 0;
        MSXSysRom    = romMSXKR;
        KoreaMSX1    = romMSXHAN;
        MSXSysRomExt = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXKR.cmos";
        break;
    case P_EMU_MSX2:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2;
        MSXSysRomExt = romMSX2EXT;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2.cmos";
        break;
    case P_EMU_MSX2J:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2J;
        MSXSysRomExt = romMSX2EXTJ;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2J.cmos";
        break;
    case P_EMU_MSX2KR:
        MSXVersion   = 0;
        MSXSysRom    = romMSX2KR;
        KoreaMSX2    = romMSX2HAN;
        MSXSysRomExt = romMSX2EXTKR;
        KoreaMSX1    = NULL;
        CMOSname     = "MSX2KR.cmos";
        break;
    case P_EMU_MSX2P:
        MSXVersion   = 2;
        MSXSysRom    = romMSX2P;
        MSXSysRomExt = romMSX2PEXT;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2P.cmos";
        break;
    }
    RAMPages   = 1 << pProperties->emulation.RAMsize;
    VRAMPages  = 1 << pProperties->emulation.VRAMsize;
    ComName    = NULL;
    PrnName    = NULL;
    TheMixer   = Prt.mixer;

    emuSyncSem = CreateSemaphore(NULL, 0, 1000, NULL);
    emuTimer   = StartTimer(Prt.hwnd, emuSyncSem, emulatorGetSyncPeriod(), pProperties->emulation.syncMethod == P_EMU_SYNC1MS);

    SetCapslockLed(0);
    SetKanaLed(0);

    emuThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)emulatorThread, 0, 0, &id);
    
    soundResume();

    Prt.emuState = EMU_RUNNING;
    mouseEmuUpdateRunState(1);
}

void emulatorStop() {
    if (Prt.emuState == EMU_STOPPED) {
        return;
    }

    Prt.emuState = EMU_STOPPED;
    mouseEmuUpdateRunState(0);
    emuExitFlag = 1;
    ReleaseSemaphore(emuSyncSem, 100, NULL);
    WaitForSingleObject(emuThread, INFINITE);
    Prt.pStoredState[0] = 0;
    soundSuspend();
    CloseHandle(emuThread);
    CloseHandle(emuSyncSem);
}

void SetColor(int palEntry, UInt32 rgbColor) {
    UInt32 color = videoGetColor(Prt.pVideo, ((rgbColor >> 16) & 0xff), ((rgbColor >> 8) & 0xff), rgbColor & 0xff);
    if (palEntry == 0) {
        emuPalette0 = color;
    }
    else {
        emuPalette[palEntry] = color;
    }
}

UInt8 Joystick(register UInt8 N) {
    return JoystickGetState(N + 1);
}

void Mouse(int* dx, int* dy, int* buttons) {
    mouseEmuGetState(dx, dy, buttons);
}

void Keyboard(UInt8* keybardMap) {
    memcpy(keybardMap, Prt.keyMap, 16);
}

void SetCapslockLed(int enable) {
    if (enable) Prt.ledState |= 1;
    else        Prt.ledState &= ~1;
}

void SetKanaLed(int enable) {
    if (enable) Prt.ledState |= 2;
    else        Prt.ledState &= ~2;
}

int WaitForSync(void) {
    do {
        emuSuspendFlag = 1;
        WaitForSingleObject(emuSyncSem, INFINITE);
    } while (!emuExitFlag && Prt.emuState != EMU_RUNNING);
    emuSuspendFlag = 0;

    return emuExitFlag;
}

void RefreshScreen(int screenMode, int evenOdd, int interlace) {
    static int frameSkip = 0;
    if (frameSkip++ < pProperties->video.frameSkip) {
        return;
    }
    frameSkip = 0;

    Prt.updatePending++;
    if (Prt.updatePending <= 1) {
        Prt.screenMode = screenMode;
        Prt.evenOdd = evenOdd;
        Prt.interlace = interlace;
        PostMessage(Prt.hwnd, WM_UPDATE, 0, 0);
    }
}
