/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/***************************************************************************/
/*                                                                         */
/*                         HARDware PCEngine                               */
/*                                                                         */
/* This source file implements all functions relatives to pc engine inner  */
/* hardware (memory access e.g.)                                           */
/*                                                                         */
/***************************************************************************/

#include "hard_pce.h"

/**
  * Variables declaration
  * cf explanations in the header file
  **/

struct_hard_pce *hard_pce;

UChar *RAM;

// Video
UInt16 *SPRAM;
UInt32 *VRAM2;
UInt32 *VRAMS;
UChar  *Pal;
UChar  *vchange;
UChar  *vchanges;
UChar  *WRAM;
UChar  *VRAM;
UInt32 *p_scanline;

// Audio
UChar *PCM;

// I/O
IO io;

// CD
UChar cd_port_1800 = 0;
/**/ UChar cd_port_1801 = 0;
/**/ UChar cd_port_1802 = 0;
/**/ UChar cd_port_1804 = 0;
/**/ UChar * cd_read_buffer;
UChar *cd_sector_buffer;
UChar *cd_extra_mem;
UChar *cd_extra_super_mem;
UChar *ac_extra_mem;

UInt32 pce_cd_read_datacnt;
/**/ UChar cd_sectorcnt;
UChar pce_cd_curcmd;
/**/
// Memory
UChar * zp_base;
UChar *sp_base;
UChar *mmr;
UChar *IOAREA;

// Interruption
UChar *p_irequest;
UChar *p_aftercli;
UInt32 *p_cyclecount;
UInt32 *p_cyclecountold;
UInt32 *p_ibackup;

const UInt32 TimerPeriod = 1097;

// registers

#if defined(SHARED_MEMORY)

UInt16 *p_reg_pc;
UChar *p_reg_a;
UChar *p_reg_x;
UChar *p_reg_y;
UChar *p_reg_p;
UChar *p_reg_s;

#else

UInt16 reg_pc;
UChar reg_a;
UChar reg_x;
UChar reg_y;
UChar reg_p;
UChar reg_s;

#endif

// Miscellaneous
UInt32 *p_cycles;
UInt32 *p_frames;
SInt32 *p_external_control_cpu;

// Shared memory
static int shm_handle;

/**
  * Initialize the hardware
  **/
void
hard_init ()
{
#if defined(SHARED_MEMORY)
	shm_handle =
		shmget ((key_t) SHM_HANDLE, sizeof (struct_hard_pce),
			IPC_CREAT | IPC_EXCL | 0666);
	if (shm_handle == -1)
		fprintf (stderr, "Couldn't get shared memory\n");
	else
	{
		hard_pce = (struct_hard_pce *) shmat (shm_handle, NULL, 0);
		if (hard_pce == NULL)
			fprintf (stderr, "Couldn't attach shared memory\n");

		p_reg_pc = &hard_pce->s_reg_pc;
		p_reg_a = &hard_pce->s_reg_a;
		p_reg_x = &hard_pce->s_reg_x;
		p_reg_y = &hard_pce->s_reg_y;
		p_reg_p = &hard_pce->s_reg_p;
		p_reg_s = &hard_pce->s_reg_s;
		p_external_control_cpu = &hard_pce->s_external_control_cpu;
	}
#else
	hard_pce = (struct_hard_pce *) malloc(sizeof(struct_hard_pce));
#endif 
	
	RAM = hard_pce->RAM;
	PCM = hard_pce->PCM;
	WRAM = hard_pce->WRAM;
	VRAM = hard_pce->VRAM;
	VRAM2 = hard_pce->VRAM2;
	VRAMS = hard_pce->VRAMS;
	vchange = hard_pce->vchange;
	vchanges = hard_pce->vchanges;

	SPRAM = hard_pce->SPRAM;
	Pal = hard_pce->Pal;
	
	p_scanline = &hard_pce->s_scanline;
	p_irequest = &hard_pce->s_irequest;
	p_aftercli = &hard_pce->s_aftercli;
	
	p_cyclecount = &hard_pce->s_cyclecount;
	p_cyclecountold = &hard_pce->s_cyclecountold;
	p_ibackup = &hard_pce->s_ibackup;
	
	p_cycles = &hard_pce->s_cycles;
	p_frames = &hard_pce->s_frames;
	
	mmr = hard_pce->mmr;
	
	#if defined(SHARED_MEMORY)
	/* Add debug on beginning option by setting 0 here */
	external_control_cpu = -1;
	#endif
}

/**
  *  Terminate the hardware
  **/
void
hard_term ()
{
#if defined(SHARED_MEMORY)
	if (shmctl (shm_handle, IPC_RMID, NULL) == -1)
		fprintf (stderr, "Couldn't destroy shared memory\n");
#else
	free(hard_pce);
#endif
}

/**
  * Functions to access PCE hardware
  **/

int return_value_mask_tab_0002[32] =
	{
		0xFF,
		0xFF,
		0xFF,
		0xFF, /* unused */
		0xFF, /* unused */
		0xFF,
		0xFF,
		0xFF,
		0xFF, /* 8 */	
		0xFF,
		0x1F, /* A */
		0x7F,
		0x1F, /* C */
		0xFF,
		0xFF,	/* E */
		0x1F,
		0xFF, /* 10 */
		0xFF,
	/* No data for remaining reg, assuming 0xFF */
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF	
	};

int return_value_mask_tab_0003[32] =
	{
		0xFF,
		0xFF,
		0xFF,
		0xFF, /* unused */
		0xFF, /* unused */
		0x1F,
		0x03,
		0x03,
		0x01, /* 8 */	/* ?? */
		0x00,
		0x7F, /* A */
		0x7F,
		0xFF, /* C */
		0x01,	
		0x00, /* E */
		0x00,
		0xFF, /* 10 */
		0xFF,
	/* No data for remaining reg, assuming 0xFF */
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF,	
		0xFF	
	
	};


int return_value_mask_tab_0400[8] = 
  {
		0xFF,
		0x00,
		0xFF,
	  0x01,
		0xFF,
		0x01,
		0xFF, /* unused */
		0xFF  /* unused */
	};

int return_value_mask_tab_0800[16] = 
	{
		0x03,
		0xFF,
		0xFF,
		0x0F,
		0xDF,
		0xFF,
		0x1F,
		0x9F,
		0xFF,
		0x83,
	/* No data for remainig reg, assuming 0xFF */
		0xFF,
		0xFF,
		0xFF,
		0xFF,
		0xFF,
		0xFF	
	};

int return_value_mask_tab_0c00[2] = 
	{
		0x7F,
		0x01
	};

int return_value_mask_tab_1400[4] = 
	{
		0xFF,
		0xFF,
		0x03,
		0x03
	};


//! Returns the useful value mask depending on port value
int
return_value_mask(UInt16 A)
{

	if (A < 0x400) // VDC
		{
			if (A & 0x3 == 0x02) 
				{
					return return_value_mask_tab_0002[io.vdc_reg];
				}
			else
			if (A & 0x3 == 0x02) 
				{
					return return_value_mask_tab_0003[io.vdc_reg];
				}
			else
				return 0xFF;
			
		}
	
	if (A < 0x800) // VCE
	  return return_value_mask_tab_0400[A & 0x07];
	
	if (A < 0xC00) /* PSG */
		return return_value_mask_tab_0800[A & 0x0F];
	
	if (A < 0x1000) /* Timer */
		return return_value_mask_tab_0c00[A & 0x01];
	
	if (A < 0x1400) /* Joystick / IO port */
		return 0xFF;
	
	if (A < 0x1800) /* Interruption acknowledgement */
		return return_value_mask_tab_1400[A & 0x03];
	
	/* We don't know for higher ports */
	return 0xFF;
	
}

//! Adds the io_buffer feature
UChar IO_read (UInt16 A)
{
	int mask;
	UChar temporary_return_value;

	if ((A < 0x800) || (A >= 0x1800)) // latch isn't affected out of the 0x800 - 0x1800 range
		return IO_read_raw(A);

	mask = return_value_mask(A);

	temporary_return_value = IO_read_raw(A);
	
	io.io_buffer = temporary_return_value | (io.io_buffer & ~mask);

	return io.io_buffer;
}

/* read */
UChar
IO_read_raw (UInt16 A)
{
	UChar ret;

#ifndef FINAL_RELEASE
	if ((A & 0x1F00) == 0x1A00)
		Log ("AC Read at %04x\n", A);
#endif

	switch (A & 0x1FC0)
	{
	case 0x0000:		/* VDC */
		switch (A & 3)
		{
		case 0:
			ret = io.vdc_status;
			io.vdc_status = 0;	//&=VDC_InVBlank;//&=~VDC_BSY;
			return ret;
		case 1:
			return 0;
		case 2:
			if (io.vdc_reg == VRR)
				return VRAM[io.VDC[MARR].W * 2];
			else
				return io.VDC[io.vdc_reg].B.l;
		case 3:
			if (io.vdc_reg == VRR)
			{
				ret = VRAM[io.VDC[MARR].W * 2 + 1];
				io.VDC[MARR].W += io.vdc_inc;
				return ret;
			}
			else
				return io.VDC[io.vdc_reg].B.h;
		}
		break;

	case 0x0400:		/* VCE */
		switch (A & 7)
		{
		case 4:
			return io.VCE[io.vce_reg.W].B.l;
		case 5:
			return io.VCE[io.vce_reg.W++].B.h;
		}
		break;
	case 0x0800:		/* PSG */
		switch (A & 15)
		{
		case 0:
			return io.psg_ch;
		case 1:
			return io.psg_volume;
		case 2:
			return io.PSG[io.psg_ch][2];
		case 3:
			return io.PSG[io.psg_ch][3];
		case 4:
			return io.PSG[io.psg_ch][4];
		case 5:
			return io.PSG[io.psg_ch][5];
		case 6:
		{
			int ofs = io.PSG[io.psg_ch][PSG_DATA_INDEX_REG];
			io.PSG[io.psg_ch][PSG_DATA_INDEX_REG] = (io.PSG[io.psg_ch][PSG_DATA_INDEX_REG] + 1) & 31;
			return io.wave[io.psg_ch][ofs];
		}
		case 7:
			return io.PSG[io.psg_ch][7];

		case 8:
			return io.psg_lfo_freq;
		case 9:
			return io.psg_lfo_ctrl;
		default:
			return NODATA;
		}
		break;
	case 0x0c00:		/* timer */
		return io.timer_counter;

	case 0x1000:		/* joypad */
		ret = io.JOY[io.joy_counter] ^ 0xff;
		if (io.joy_select & 1)
			ret >>= 4;
		else
		{
			ret &= 15;
			io.joy_counter = (io.joy_counter + 1) % 5;
		}

/* return ret | Country; *//* country 0:JPN 1<<6=US */
		return ret | 0x30; // those 2 bits are always on, bit 6 = 0 (Jap), bit 7 = 0 (Attached cd)

	case 0x1400:		/* IRQ */
		switch (A & 15)
		{
		case 2:
			return io.irq_mask;
		case 3:
			ret = io.irq_status;
			io.irq_status = 0;
			return ret;
		}
		break;


	case 0x18C0:		// Memory management ?
		switch (A & 15)
		{
		case 5:
		case 1:
			return 0xAA;
		case 2:
		case 6:
			return 0x55;
		case 3:
		case 7:
			return 0x03;
		}
		break;

	case 0x1AC0:
		switch (A & 15)
		{
		case 0:
			return (UChar) (io.ac_shift);
		case 1:
			return (UChar) (io.ac_shift >> 8);
		case 2:
			return (UChar) (io.ac_shift >> 16);
		case 3:
			return (UChar) (io.ac_shift >> 24);
		case 4:
			return io.ac_shiftbits;
		case 5:
			return io.ac_unknown4;
		case 14:
			return 0x10;
		case 15:
			return 0x51;
		default:
			Log ("Unknown Arcade card port access : 0x%04X\n", A);
		}
		break;

	case 0x1A00:
	{
		UChar ac_port = (A >> 4) & 3;
		switch (A & 15)
		{
		case 0:
		case 1:
			/*
			 * switch (io.ac_control[ac_port] & (AC_USE_OFFSET | AC_USE_BASE))
			 * {
			 * case 0:
			 * return ac_extra_mem[0];
			 * case AC_USE_OFFSET:
			 * ret = ac_extra_mem[io.ac_offset[ac_port]];
			 * if (!(io.ac_control[ac_port] & AC_INCREMENT_BASE))
			 * io.ac_offset[ac_port]+=io.ac_incr[ac_port];
			 * return ret;
			 * case AC_USE_BASE:
			 * ret = ac_extra_mem[io.ac_base[ac_port]];
			 * if (io.ac_control[ac_port] & AC_INCREMENT_BASE)
			 * io.ac_base[ac_port]+=io.ac_incr[ac_port];
			 * return ret;
			 * default:
			 * ret = ac_extra_mem[io.ac_base[ac_port] + io.ac_offset[ac_port]];
			 * if (io.ac_control[ac_port] & AC_INCREMENT_BASE)
			 * io.ac_base[ac_port]+=io.ac_incr[ac_port];
			 * else
			 * io.ac_offset[ac_port]+=io.ac_incr[ac_port];
			 * return ret;
			 * }
			 * return 0;
			 */
			 			 
			if (io.ac_control[ac_port] & AC_USE_OFFSET)
				ret = ac_extra_mem[((io.ac_base[ac_port] +
						     io.
						     ac_offset[ac_port]) &
						    0x1fffff)];
			else
				ret = ac_extra_mem[((io.
						     ac_base[ac_port]) &
						    0x1fffff)];

			if (io.ac_control[ac_port] & AC_ENABLE_INC)
			{
				if (io.
				    ac_control[ac_port] & AC_INCREMENT_BASE)
					io.ac_base[ac_port] =
						(io.ac_base[ac_port] +
						 io.
						 ac_incr[ac_port]) & 0xffffff;
				else
					io.ac_offset[ac_port] =
						(io.ac_offset[ac_port] +
						 io.
						 ac_incr[ac_port]) & 0xffff;
			}

			return ret;


		case 2:
			return (UChar) (io.ac_base[ac_port]);
		case 3:
			return (UChar) (io.ac_base[ac_port] >> 8);
		case 4:
			return (UChar) (io.ac_base[ac_port] >> 16);
		case 5:
			return (UChar) (io.ac_offset[ac_port]);
		case 6:
			return (UChar) (io.ac_offset[ac_port] >> 8);
		case 7:
			return (UChar) (io.ac_incr[ac_port]);
		case 8:
			return (UChar) (io.ac_incr[ac_port] >> 8);
		case 9:
			return io.ac_control[ac_port];
		default:
			Log ("Unknown Arcade card port access : 0x%04X\n", A);
		}
		break;
	}
	case 0x1800:		// CD-ROM extention
		return pce_cd_handle_read_1800(A);
	}
#ifndef FINAL_RELEASE
	#if !defined(KERNEL_DS)
    fprintf (stderr, "ignore I/O read %04X\nat PC = %04X\n", A, M.PC.W);
	#endif
#endif
	return NODATA;
}

/**
  * Change bank setting
  **/
void
bank_set (UChar P, UChar V)
{

#if defined(CD_DEBUG)
  if (V >= 0x40 && V <= 0x43)
		printf("AC pseudo bank switching !!! (mmr[%d] = %d)\n", P, V);
#endif
	
	mmr[P] = V;
	if (ROMMap[V] == IOAREA)
		Page[P] = IOAREA;
	else
		Page[P] = ROMMap[V] - P * 0x2000;
}
