/*
	WinSTon

	Tool Bar icons
*/

#include "..\includes\winston.h"
#include "..\includes\dialog.h"
#include "..\includes\file.h"
#include "..\includes\floppy.h"
#include "..\includes\reset.h"
#include "..\includes\memory.h"
#include "..\includes\misc.h"
#include "..\includes\screen.h"
#include "..\includes\statusbar.h"
#include "..\includes\toolbar.h"
#include "..\includes\view.h"

#define TOOLBAR_BUTTON_WIDTH	23		// Width/Height of each button
#define TOOLBAR_BUTTON_HEIGHT	21
#define TOOLBAR_X				0		// XY of tool bar
#define TOOLBAR_Y				3

// State and position of tool bar buttons
static int ToolBarButtonStates[NUM_TOOLBAR_BUTTONS] = { BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP,BUTTON_STATE_UP };
static int ToolBarButtonsX[NUM_TOOLBAR_BUTTONS] = { 8,30,47,69,80,102,119,141,152,180 };
static int ToolBarButtonsSrcX[NUM_TOOLBAR_BUTTONS] = { 0,22,33,55,66,88,99,121,132,154 };
static int ToolBarButtonsWidths[NUM_TOOLBAR_BUTTONS] = { 22,11,22,11,22,11,22,11,22,22 };
static int ToolBarButtonsTwins[NUM_TOOLBAR_BUTTONS] = { 1,0,3,2,5,4,7,6,-1,-1 };
// Menus
HMENU hToolBarMenus[MAX_POPUP_MENU][2];					// Handles for pop-up menus
int ToolBarMenusX[MAX_POPUP_MENU] = { 0,39,72,110 };	// X coordinates of menus
// Text to show in status bar when move over button
static char *pszToolBarTexts[NUM_TOOLBAR_BUTTONS] = {
	"Reset ST.",						// TOOLBAR_BUTTON_RESET
	"Reset ST.",						// TOOLBAR_BUTTON_RESET_MENU
	"Load floppy image into Drive A.",	// TOOLBAR_BUTTON_FLOPPYA
	"Load floppy image into Drive A.",	// TOOLBAR_BUTTON_FLOPPYA_MENU
	"Load floppy image into Drive B.",	// TOOLBAR_BUTTON_FLOPPYB
	"Load floppy image into Drive B.",	// TOOLBAR_BUTTON_FLOPPYB_MENU
	"Modify the WinSTon options.",		// TOOLBAR_BUTTON_OPTIONS
	"Modify the WinSTon options.",		// TOOLBAR_BUTTON_OPTIONS_MENU
	"Run Favourite disc images.",		// TOOLBAR_BUTTON_FAVOURITES
	"Help topics."						// TOOLBAR_BUTTON_HELP
};

//-----------------------------------------------------------------------
/*
	Create 'pop-up' menus for ToolBar
*/
void ToolBar_CreateMenus(void)
{
	// Create our menus
	hToolBarMenus[POPUP_MENU_RESET][0] = LoadMenu(hInst,MAKEINTRESOURCE(IDR_MENU_RESET));
	hToolBarMenus[POPUP_MENU_RESET][1] = GetSubMenu(hToolBarMenus[POPUP_MENU_RESET][0],0);
	hToolBarMenus[POPUP_MENU_FLOPPY_A][0] = LoadMenu(hInst,MAKEINTRESOURCE(IDR_MENU_FLOPPYA));
	hToolBarMenus[POPUP_MENU_FLOPPY_A][1] = GetSubMenu(hToolBarMenus[POPUP_MENU_FLOPPY_A][0],0);
	hToolBarMenus[POPUP_MENU_FLOPPY_B][0] = LoadMenu(hInst,MAKEINTRESOURCE(IDR_MENU_FLOPPYB));
	hToolBarMenus[POPUP_MENU_FLOPPY_B][1] = GetSubMenu(hToolBarMenus[POPUP_MENU_FLOPPY_B][0],0);
	hToolBarMenus[POPUP_MENU_OPTIONS][0] = LoadMenu(hInst,MAKEINTRESOURCE(IDR_MENU_OPTIONS));
	hToolBarMenus[POPUP_MENU_OPTIONS][1] = GetSubMenu(hToolBarMenus[POPUP_MENU_OPTIONS][0],0);
}

//-----------------------------------------------------------------------
/*
	Free 'pop-up' menus
*/
void ToolBar_FreeMenus(void)
{
	// Free all menus
	DestroyMenu(hToolBarMenus[POPUP_MENU_RESET][0]);
	DestroyMenu(hToolBarMenus[POPUP_MENU_FLOPPY_A][0]);
	DestroyMenu(hToolBarMenus[POPUP_MENU_FLOPPY_B][0]);
	DestroyMenu(hToolBarMenus[POPUP_MENU_OPTIONS][0]);
}

//-----------------------------------------------------------------------
/*
	Pop-up menu
*/
void ToolBar_PopUpMenu(HWND hWnd,int Menu)
{
	POINT Point;

	// Open 'menu' under button
	Point.x = ToolBarMenusX[Menu];
	Point.y = TOOLBAR_Y+TOOLBAR_BUTTON_HEIGHT;
	ClientToScreen(hWnd,&Point);
	View_SetMenuChecks();
	// Set filenames on menu Floppy A,B
	View_SetMenuFileNames(hToolBarMenus[POPUP_MENU_FLOPPY_A][1],NUM_FLOPPYA_MENU_ITEMS,0);
	View_SetMenuFileNames(hToolBarMenus[POPUP_MENU_FLOPPY_B][1],NUM_FLOPPYB_MENU_ITEMS,1);
	// Show th emenu
	TrackPopupMenu(hToolBarMenus[Menu][1],0,Point.x,Point.y,0,hWnd,NULL);
}

//-----------------------------------------------------------------------
/*
	Return TOOLBAR_BUTTON_xxxx or -1(not found) from XY
*/
int ToolBar_FindChosenButton(int x, int y)
{
	int i;

	// Is in Y range?
	if ( (y>=TOOLBAR_Y) && (y<=(TOOLBAR_Y+TOOLBAR_BUTTON_HEIGHT)) ) {
		// Check each button
		for(i=0; i<NUM_TOOLBAR_BUTTONS; i++) {
			// Is in X range?
			if ( (x>=ToolBarButtonsX[i]) && (x<(ToolBarButtonsX[i]+ToolBarButtonsWidths[i])) )
				return(i);	// eg TOOLBAR_BUTTON_xxxx
		}
	}

	// Not found matching button
	return(-1);
}

//-----------------------------------------------------------------------
/*
	Draw a single tool bar button
*/
void ToolBar_DrawButton(HDC hDC, HDC MemDC,int ButtonID)
{
	HDC DrawDC,DrawMemDC;
	int BitmapX,BitmapY;

	if (!bInFullScreen) {	// Only draw when in Window, else DirectX may draw! Doh!
		DrawDC = hDC;
		if (hDC==NULL)
			DrawDC = GetDC(hWnd);
		DrawMemDC = MemDC;
		if (MemDC==NULL) {
			DrawMemDC = CreateCompatibleDC(DrawDC);
			SelectObject(DrawMemDC,Bitmaps[BITMAP_TOOLBAR_ICONS]);
		}

		// Find button in bitmap array
		BitmapX = ToolBarButtonsSrcX[ButtonID];									// X of bitmap for this button
		BitmapY = TOOLBAR_BUTTON_HEIGHT*ToolBarButtonStates[ButtonID];			// Line in bitmap of Up,Highlight and Down
		// And draw
		BitBlt(DrawDC,ToolBarButtonsX[ButtonID],TOOLBAR_Y,ToolBarButtonsWidths[ButtonID],TOOLBAR_BUTTON_HEIGHT,DrawMemDC,BitmapX,BitmapY,SRCCOPY);

		if (hDC==NULL)
			ReleaseDC(hWnd,DrawDC);
		if (MemDC==NULL)
			DeleteDC(DrawMemDC);
	}
}

//-----------------------------------------------------------------------
/*
	Draw all buttons on the tool bar
*/
void ToolBar_DrawButtons(HDC hDC)
{
	HDC DrawDC,MemDC;
	int i;

	if (!bInFullScreen) {	// Only draw when in Window, else DirectX may draw! Doh!
		DrawDC = hDC;
		if (hDC==NULL)
			DrawDC = GetDC(hWnd);
		MemDC = CreateCompatibleDC(DrawDC);

		SelectObject(MemDC,Bitmaps[BITMAP_TOOLBAR_ICONS]);
		for(i=0; i<NUM_TOOLBAR_BUTTONS; i++)
			ToolBar_DrawButton(DrawDC,MemDC,i);

		if (hDC==NULL)
			ReleaseDC(hWnd,DrawDC);
		DeleteDC(MemDC);
	}
}

//-----------------------------------------------------------------------
/*
	Return TRUE if button state has changed
*/
BOOL ToolBar_SetButtonHighlight(int ChosenButton)
{
	BOOL bNoNeedForRedraw = FALSE;

	// First check if was already highlighted
	if (ChosenButton>=0) {
		if (ToolBarButtonStates[ChosenButton]!=BUTTON_STATE_HIGHLIGHT)
			bNoNeedForRedraw = TRUE;
	}
	else
		bNoNeedForRedraw = TRUE;

	// Clear out, to make sure we only ever have one button highlighted
	Memory_Clear(ToolBarButtonStates,sizeof(ToolBarButtonStates));

	// And set button
	if (ChosenButton>=0) {
		ToolBarButtonStates[ChosenButton] = BUTTON_STATE_HIGHLIGHT;
		// Highlight 'twin', ie sub-menu
		if (ToolBarButtonsTwins[ChosenButton]!=-1)
			ToolBarButtonStates[ToolBarButtonsTwins[ChosenButton]] = BUTTON_STATE_HIGHLIGHT;
	}

	return(bNoNeedForRedraw);
}

//-----------------------------------------------------------------------
/*
	When we move the mouse we update the toolbar and the status bar text
*/
void ToolBar_UpdateOnMoveMove(int x,int y)
{
	BOOL bNeedRedraw;
	int ChosenButton;

	// Is cursor over icons??
	ChosenButton = ToolBar_FindChosenButton(x,y);
	if (ChosenButton!=-1)
		StatusBar_SetText(pszToolBarTexts[ChosenButton]);	// Set text in status bar
	else
		StatusBar_SetText(pszStatusBarHelpText);			// We are not over an icon

	// Do need to redraw button, ie changed state?
	bNeedRedraw = ToolBar_SetButtonHighlight(ChosenButton);
	if (bNeedRedraw)
		ToolBar_DrawButtons(NULL);
}

//-----------------------------------------------------------------------
/*
	Return TRUE if either SHIFT key is being held down
*/
BOOL ToolBar_IsShiftKeyHeld(void)
{
	// Holding SHIFT?
	if (GetAsyncKeyState(VK_SHIFT)&0x8000)
		return(TRUE);									// Yes

	// No
	return(FALSE);
}

//-----------------------------------------------------------------------
/*
	Ask user if need to reset
*/
BOOL ToolBar_QueryUserReset(BOOL bShiftKeyHeld)
{
	// If hold down shift then don't reset, otherwise reset every time
	if (bShiftKeyHeld)
		return(FALSE);									// Don't reset

	return(TRUE);										// Reset
}

//-----------------------------------------------------------------------
/*
	Activate Reset Button
*/
void ToolBar_Activate_Reset(BOOL bShiftKeyHeld,int ResetQuery,BOOL bColdReset)
{
	int bRet;

	// Check if user is holding down 'Shift' and reset/don't reset accordingly(on RESETQUERY_ASKUSER)
	if (ResetQuery==RESETQUERY_ASKUSER)
		bRet = ToolBar_QueryUserReset(bShiftKeyHeld);
	else
		bRet = (ResetQuery==RESETQUERY_RESET) ? TRUE:FALSE;

	if (bRet) {
		// Reset ST, put mouse back into ST mode
		Dialog_CopyDetailsFromConfiguration(TRUE);
		if (bColdReset)
			Reset_Cold();
		else
			Reset_Warm();
		Screen_Draw();
	}

	// Go back to ST mode if user said OK or we've not asked
	if ( (bRet) || (ResetQuery!=RESETQUERY_ASKUSER) ) {
		WinSTon_UnPauseEmulation();
		View_ToggleWindowsMouse(MOUSE_ST);
	}
}

//-----------------------------------------------------------------------
/*
	Open disc image selection and dialog and reset emulator
	Pass drive 0 or 1
*/
void ToolBar_OpenFloppyDialog(int Drive,BOOL bOverrideShiftKey,int ResetQuery)
{
	char szTmpFileName[MAX_FILENAME_LENGTH];
	BOOL bShiftKeyHeld = FALSE, bQueryReset = FALSE;

	// Check SHIFT key status
	if (!bOverrideShiftKey)
		bShiftKeyHeld = ToolBar_IsShiftKeyHeld();		// Store if SHIFT is held
	// Show 'file open' dialog so can choose '.ST' or '.MSA' floppy image
	strcpy(szTmpFileName,EmulationDrives[Drive].szFileName);
	if (File_OpenDlg(hWnd,szTmpFileName,Drive)) {
		// Did we give a valid name, and are we Drive A? If so ask user is wish to reset
		if ( (strlen(szTmpFileName)>0) && (Drive==0) )
			bQueryReset = TRUE;
		// Insert disc into drive
		if (Floppy_InsertDiscIntoDrive(Drive,szTmpFileName)) {
			// Add filename to menu
			View_AddMenuFileName(Drive,szTmpFileName);
		}
	}

	// Ask user if wish to 'cold' reset now?
	if (bQueryReset)
		ToolBar_Activate_Reset(bShiftKeyHeld,ResetQuery,TRUE);
}

//-----------------------------------------------------------------------
/*
	Activate Floppy A Button
*/
void ToolBar_Activate_FloppyA(BOOL bOverrideShiftKey)
{
	ToolBar_OpenFloppyDialog(0,bOverrideShiftKey,RESETQUERY_ASKUSER);		// Drive A
}

//-----------------------------------------------------------------------
/*
	Activate Floppy B Button
*/
void ToolBar_Activate_FloppyB(BOOL bOverrideShiftKey)
{
	ToolBar_OpenFloppyDialog(1,bOverrideShiftKey,RESETQUERY_ASKUSER);		// Drive B
}

//-----------------------------------------------------------------------
/*
	Activate Options Button
*/
void ToolBar_Activate_Options(void)
{
	Dialog_DoProperty(nLastOpenPage,FALSE);
}

//-----------------------------------------------------------------------
/*
	Activate Favourites Button
*/
void ToolBar_Activate_Favourites(void)
{
	Dialog_DoProperty(DIALOG_PAGE_FAVOURITES,FALSE);
}

//-----------------------------------------------------------------------
/*
	Activate Help Button
*/
void ToolBar_Activate_Help(void)
{
	WinSTon_GoWeb("help\\contents.html",FALSE);
}

//-----------------------------------------------------------------------
/*
	Activate Tool Bar button
*/
void ToolBar_ActivateOnButtonDown(int ChosenButton)
{
	if (ChosenButton!=-1) {
		// Press button
		ToolBarButtonStates[ChosenButton] = BUTTON_STATE_DOWN;
		InvalidateRect(hWnd,NULL,FALSE);
		UpdateWindow(hWnd);

		// Handle click on each button
		switch(ChosenButton) {
			case TOOLBAR_BUTTON_RESET:
				ToolBar_Activate_Reset(ToolBar_IsShiftKeyHeld(),RESETQUERY_ASKUSER,TRUE);
				break;
			case TOOLBAR_BUTTON_RESET_MENU:
				ToolBar_PopUpMenu(hWnd,POPUP_MENU_RESET);
				break;
			case TOOLBAR_BUTTON_FLOPPYA:
				ToolBar_Activate_FloppyA(FALSE);
				break;
			case TOOLBAR_BUTTON_FLOPPYA_MENU:
				ToolBar_PopUpMenu(hWnd,POPUP_MENU_FLOPPY_A);
				break;
			case TOOLBAR_BUTTON_FLOPPYB:
				ToolBar_Activate_FloppyB(FALSE);
				break;
			case TOOLBAR_BUTTON_FLOPPYB_MENU:
				ToolBar_PopUpMenu(hWnd,POPUP_MENU_FLOPPY_B);
				break;
			case TOOLBAR_BUTTON_OPTIONS:
				ToolBar_Activate_Options();
				break;
			case TOOLBAR_BUTTON_OPTIONS_MENU:
				ToolBar_PopUpMenu(hWnd,POPUP_MENU_OPTIONS);
				break;
			case TOOLBAR_BUTTON_FAVOURITES:
				ToolBar_Activate_Favourites();
				break;
			case TOOLBAR_BUTTON_HELP:
				ToolBar_Activate_Help();
				break;
		}

		// Release button
		ToolBarButtonStates[ChosenButton] = BUTTON_STATE_UP;
		// Release 'twin', ie sub-menu
		if (ToolBarButtonsTwins[ChosenButton]!=-1)
			ToolBarButtonStates[ToolBarButtonsTwins[ChosenButton]] = BUTTON_STATE_UP;
		InvalidateRect(hWnd,NULL,FALSE);
		UpdateWindow(hWnd);
		StatusBar_SetText("");
	}
}

//-----------------------------------------------------------------------
/*
	Activate Tool Bar buttons(left-click)
*/
void ToolBar_ActivateOnLeftButtonDown(int x,int y)
{
	int ChosenButton;

	// Is cursor over icons??
	ChosenButton = ToolBar_FindChosenButton(x,y);
	// And choose!
	ToolBar_ActivateOnButtonDown(ChosenButton);
}

//-----------------------------------------------------------------------
/*
	Activate Tool Bar buttons(right-click to drop-down menus)
*/
void ToolBar_ActivateOnRightButtonDown(int x,int y)
{
	int ChosenButton;

	// Is cursor over icons??
	ChosenButton = ToolBar_FindChosenButton(x,y);
	// Remap button to drop-down(if have one)
	switch(ChosenButton) {
		case TOOLBAR_BUTTON_RESET:
		case TOOLBAR_BUTTON_RESET_MENU:
			ChosenButton = TOOLBAR_BUTTON_RESET_MENU;
			break;
		case TOOLBAR_BUTTON_FLOPPYA:
		case TOOLBAR_BUTTON_FLOPPYA_MENU:
			ChosenButton = TOOLBAR_BUTTON_FLOPPYA_MENU;
			break;
		case TOOLBAR_BUTTON_FLOPPYB:
		case TOOLBAR_BUTTON_FLOPPYB_MENU:
			ChosenButton = TOOLBAR_BUTTON_FLOPPYB_MENU;
			break;
		case TOOLBAR_BUTTON_OPTIONS:
		case TOOLBAR_BUTTON_OPTIONS_MENU:
			ChosenButton = TOOLBAR_BUTTON_OPTIONS_MENU;
			break;
		default:
			ChosenButton = -1;
			break;
	}

	// And choose!
	ToolBar_ActivateOnButtonDown(ChosenButton);
}
