//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/************************************************************************
 *                                                                      *
 *	Portions, but not all of this source file are based on MAME v0.60	*
 *	File "sn76496.c". All copyright goes to the original author.		*
 *	The remaining parts, including DAC processing, by neopop_uk			*
 *                                                                      *
 ************************************************************************/

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include "mem.h"
#include "sound.h"

//=============================================================================

SoundChip leftChip;
SoundChip rightChip;

_u16 sound_frequency;
bool stereo;

//DAC
#define DAC_BUFFERSIZE		44100

_u32 dacBufferRead, dacBufferWrite;//, dacBufferCount;

int dacFreqDup;	// How many bytes to duplicate, versus 11025khz
_u16 dacBufferL[DAC_BUFFERSIZE];
_u16 dacBufferR[DAC_BUFFERSIZE];

//=============================================================================

#define SOUNDCHIPCLOCK	(3072000)	//Unverified / seems okay

#define MAX_OUTPUT 0x7fff
#define STEP 0x10000		//Fixed point adjuster

static _u32 VolTable[16];
static _u32 UpdateStep;	//Number of steps during one sample.

/* Formulas for noise generator */
/* bit0 = output */

/* noise feedback for white noise mode (verified on real SN76489 by John Kortink) */
#define FB_WNOISE 0x14002	/* (16bits) bit16 = bit0(out) ^ bit2 ^ bit15 */

/* noise feedback for periodic noise mode */
#define FB_PNOISE 0x08000	/* 15bit rotate */

/* noise generator start preset (for periodic noise) */
#define NG_PRESET 0x0f35

//=============================================================================

static _s16 sound_getSample(SoundChip* chip)
{
	int i;

	int vol[4];
	unsigned int out;
	int left;

	/* vol[] keeps track of how long each square wave stays */
	/* in the 1 position during the sample period. */
	vol[0] = vol[1] = vol[2] = vol[3] = 0;

	for (i = 0; i < 3; i++)
	{
		if (chip->Output[i]) vol[i] += chip->Count[i];
		chip->Count[i] -= STEP;

		/* Period[i] is the half period of the square wave. Here, in each */
		/* loop I add Period[i] twice, so that at the end of the loop the */
		/* square wave is in the same status (0 or 1) it was at the start. */
		/* vol[i] is also incremented by Period[i], since the wave has been 1 */
		/* exactly half of the time, regardless of the initial position. */
		/* If we exit the loop in the middle, Output[i] has to be inverted */
		/* and vol[i] incremented only if the exit status of the square */
		/* wave is 1. */

		while (chip->Count[i] <= 0)
		{
			chip->Count[i] += chip->Period[i];
			if (chip->Count[i] > 0)
			{
				chip->Output[i] ^= 1;
				if (chip->Output[i]) vol[i] += chip->Period[i];
				break;
			}
			chip->Count[i] += chip->Period[i];
			vol[i] += chip->Period[i];
		}
		if (chip->Output[i]) vol[i] -= chip->Count[i];
	}

	left = STEP;
	do
	{
		int nextevent;

		if (chip->Count[3] < left) nextevent = chip->Count[3];
		else nextevent = left;

		if (chip->Output[3]) vol[3] += chip->Count[3];
		chip->Count[3] -= nextevent;
		if (chip->Count[3] <= 0)
		{
			if (chip->RNG & 1) chip->RNG ^= chip->NoiseFB;
			chip->RNG >>= 1;
			chip->Output[3] = chip->RNG & 1;
			chip->Count[3] += chip->Period[3];
			if (chip->Output[3]) vol[3] += chip->Period[3];
		}
		if (chip->Output[3]) vol[3] -= chip->Count[3];

		left -= nextevent;
	} while (left > 0);

	out = vol[0] * chip->Volume[0] + vol[1] * chip->Volume[1] +
		vol[2] * chip->Volume[2] + vol[3] * chip->Volume[3];

	if (out > MAX_OUTPUT * STEP) out = MAX_OUTPUT * STEP;

	return out / STEP;
}

//=============================================================================

void dac_write(_u8 left, _u8 right)
{
	int i;

	for (i = 0; i < dacFreqDup; i++)
	{
		dacBufferL[dacBufferWrite] = left << 8;
		dacBufferR[dacBufferWrite] = right << 8;
		
		dacBufferWrite++;

		if (dacBufferWrite >= DAC_BUFFERSIZE)
			dacBufferWrite -= DAC_BUFFERSIZE;

//		dacBufferCount++;
/*		if (dacBufferCount >= DAC_BUFFERSIZE)
		{
			//system_message("DAC Buffer Overflow");

			//Overflow
			dacBufferRead = 0; 
			dacBufferWrite = 512;	//Offset writing to stop immediate underflow
			dacBufferCount = 0;
		}*/
	}
	
}
	
//=============================================================================

void sound_update(_s16* buffer, int length_samples)
{
	int i;
	_u16 chip, dac;

	/* If the volume is 0, increase the counter */
	for (i = 0;i < 4;i++)
	{
		/* note that I do count += length, NOT count = length + 1. You might think */
		/* it's the same since the volume is 0, but doing the latter could cause */
		/* interferencies when the program is rapidly modulating the volume. */
		if (leftChip.Volume[i] == 0)
			if (leftChip.Count[i] <= length_samples*STEP) 
				leftChip.Count[i] += length_samples*STEP;

		if (rightChip.Volume[i] == 0)
			if (rightChip.Count[i] <= length_samples*STEP) 
				rightChip.Count[i] += length_samples*STEP;
	}

	while (length_samples > 0)
	{
		if (stereo)
		{
			//Left Channel
			chip = sound_getSample(&leftChip);
		//	if (dacBufferCount > 0)
				dac = dacBufferL[dacBufferRead];
		//	else dac = 0;
			
			*(buffer++) = (chip + dac) >> 1;

			//Right Channel
			chip = sound_getSample(&rightChip);
		//	if (dacBufferCount > 0)
				dac = dacBufferR[dacBufferRead];
		//	else dac = 0;

			*(buffer++) = (chip + dac) >> 1;

			length_samples -= 2;
		}
		else
		{
			//Mix a mono track out of: (L + R) >> 1
			chip = (sound_getSample(&leftChip) + sound_getSample(&rightChip)) >> 1;

			//Mix in the DAC
		//	if (dacBufferCount > 0)
				dac = (dacBufferL[dacBufferRead] + dacBufferR[dacBufferRead]) >> 1;
		//	else dac = 0;
		
			*(buffer++) = (chip + dac) >> 1;
			
			length_samples -= 1;
		}

		//Advance the DAC read
	//	if (dacBufferCount > 0)
		{
	//		dacBufferCount --;

			//Don't play the same sample twice!
			dacBufferL[dacBufferRead] = 0;
			dacBufferR[dacBufferRead] = 0;

			dacBufferRead++;
			if (dacBufferRead >= DAC_BUFFERSIZE)
				dacBufferRead -= DAC_BUFFERSIZE;
		}
	}
}

//=============================================================================

void WriteSoundChip(SoundChip* chip, _u8 data)
{
	//Command
	if (data & 0x80)
	{
		int r = (data & 0x70) >> 4;
		int c = r/2;

		chip->LastRegister = r;
		chip->Register[r] = (chip->Register[r] & 0x3f0) | (data & 0x0f);
	
		switch(r)
		{
		case 0:	/* tone 0 : frequency */
		case 2:	/* tone 1 : frequency */
		case 4:	/* tone 2 : frequency */
			chip->Period[c] = UpdateStep * chip->Register[r];
			if (chip->Period[c] == 0) chip->Period[c] = UpdateStep;
			if (r == 4)
			{
				/* update noise shift frequency */
				if ((chip->Register[6] & 0x03) == 0x03)
					chip->Period[3] = 2 * chip->Period[2];
			}
			break;

		case 1:	/* tone 0 : volume */
		case 3:	/* tone 1 : volume */
		case 5:	/* tone 2 : volume */
		case 7:	/* noise  : volume */
#ifdef NEOPOP_DEBUG
			if (filter_sound)
			{
				if (chip == &leftChip)
					system_debug_message("sound (L): Set Tone %d Volume to %d", c, 15 - (data & 0xF));
				else
					system_debug_message("sound (R): Set Tone %d Volume to %d", c, 15 - (data & 0xF));
			}
#endif
			chip->Volume[c] = VolTable[data & 0xF];
			break;

		case 6:	/* noise  : frequency, mode */
			{
				int n = chip->Register[6];
#ifdef NEOPOP_DEBUG
	if (filter_sound)
	{
		char *pm, *nm = "White";
		if ((n & 4)) nm = "Periodic";

		switch(n & 3)
		{
		case 0: pm = "N/512"; break;
		case 1: pm = "N/1024"; break;
		case 2: pm = "N/2048"; break;
		case 3: pm = "Tone#2"; break;
		}
			
		if (chip == &leftChip)
			system_debug_message("sound (L): Set Noise Mode to %s, Period = %s", nm, pm);
		else
			system_debug_message("sound (R): Set Noise Mode to %s, Period = %s", nm, pm);
	}
#endif
				chip->NoiseFB = (n & 4) ? FB_WNOISE : FB_PNOISE;
				n &= 3;
				/* N/512,N/1024,N/2048,Tone #2 output */
				chip->Period[3] = (n == 3) ? 2 * chip->Period[2] : (UpdateStep << (5+n));

				/* reset noise shifter */
				chip->RNG = NG_PRESET;
				chip->Output[3] = chip->RNG & 1;

			}
			break;
		}
	}
	else
	{
		int r = chip->LastRegister;
		int c = r/2;

		switch (r)
		{
			case 0:	/* tone 0 : frequency */
			case 2:	/* tone 1 : frequency */
			case 4:	/* tone 2 : frequency */
				chip->Register[r] = (chip->Register[r] & 0x0f) | ((data & 0x3f) << 4);
				chip->Period[c] = UpdateStep * chip->Register[r];
				if (chip->Period[c] == 0) chip->Period[c] = UpdateStep;
				if (r == 4)
				{
					/* update noise shift frequency */
					if ((chip->Register[6] & 0x03) == 0x03)
						chip->Period[3] = 2 * chip->Period[2];
				}
#ifdef NEOPOP_DEBUG
	if (filter_sound)
	{
		if (chip == &leftChip)
			system_debug_message("sound (L): Set Tone %d Frequency to %d", c, chip->Register[r]);
		else
			system_debug_message("sound (R): Set Tone %d Frequency to %d", c, chip->Register[r]);
	}
#endif
				break;
		}
	}
}

//=============================================================================

//Resets the sound chips, also used whenever sound options are changed
void sound_init(int SampleRate)
{
	int i;
	double out;

	/* the base clock for the tone generators is the chip clock divided by 16; */
	/* for the noise generator, it is clock / 256. */
	/* Here we calculate the number of steps which happen during one sample */
	/* at the given sample rate. No. of events = sample rate / (clock/16). */
	/* STEP is a multiplier used to turn the fraction into a fixed point */
	/* number. */
	UpdateStep = (_u32)(((double)STEP * SampleRate * 16) / SOUNDCHIPCLOCK);

	//Initialise Left Chip
	memset(&leftChip, 0, sizeof(SoundChip));

	//Initialise Right Chip
	memset(&rightChip, 0, sizeof(SoundChip));

	//Default register settings
	for (i = 0;i < 8;i+=2)
	{
		leftChip.Register[i] = 0;
		leftChip.Register[i + 1] = 0x0f;	/* volume = 0 */
		rightChip.Register[i] = 0;
		rightChip.Register[i + 1] = 0x0f;	/* volume = 0 */
	}

	for (i = 0;i < 4;i++)
	{
		leftChip.Output[i] = 0;
		leftChip.Period[i] = leftChip.Count[i] = UpdateStep;
		rightChip.Output[i] = 0;
		rightChip.Period[i] = rightChip.Count[i] = UpdateStep;
	}

	//Build the volume table
	out = MAX_OUTPUT / 3;

	/* build volume table (2dB per step) */
	for (i = 0;i < 15;i++)
	{
		VolTable[i] = (_u32)out;
		out /= 1.258925412;	/* = 10 ^ (2/20) = 2dB */
	}
	VolTable[15] = 0;


	//Clear the DAC buffer
	dacBufferRead = 0; 
	dacBufferWrite = 512;	//Offset writing to stop immediate underflow
//	dacBufferCount = 0;
	for (i = 0; i < DAC_BUFFERSIZE; i++)
	{
		dacBufferL[i] = 0;
		dacBufferR[i] = 0;
	}

	dacFreqDup = (SampleRate / 11025);
}

//=============================================================================
