//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// History of changes:
// ===================
//
// Version 0.50 / 20 JUL 2002 - neopop_uk
// ======================================
//	- Cleaned and tidied up for the source release
//
//---------------------------------------------------------------------------

#include "neopop.h"
#include "TLCS900H_registers.h"
#include "mem.h"
#include "gfx.h"
#include "interrupt.h"
#include "TLCS900H_interpret.h"
#include "Z80_interface.h"
#include "bios.h"

//=============================================================================

_u32 timer_hint;

_u32 timer_clock0, timer_clock1, timer_clock2, timer_clock3;

_u8 timer[4];	//Up-counters

//=============================================================================

void interrupt(_u8 index)
{
	push32(pc);
	push16(sr);

	//Up the IFF
	setStatusIFF(((sr & 0x7000) >> 12) + 1);

	//Access the interrupt vector table to find the jump destination
	pc = *(_u32*)(ram + 0x6FB8 + (index * 4));

	halted = false;

#ifdef NEOPOP_DEBUG
	//system_debug_message("interrupt %d: pc -> %06X", index, pc);
#endif
}

//=============================================================================

void updateTimers(_u8 cputicks)
{
	//Graphics timer / H-INT 
	timer_hint += cputicks;

	//Graphics update
	if (timer_hint >= TIMER_HINT_RATE)
	{
		timer_hint = 0;//-= TIMER_HINT_RATE;

		//Draw Scanlines
		if (ram[0x8009] < SCREEN_HEIGHT)
		{
			if (system_colour)	gfx_draw_scanline_colour();
			else				gfx_draw_scanline_mono();
		}

		//VBlank
		if (ram[0x8009] == SCREEN_HEIGHT)
		{
			//Character Over / Vblank Status
			ram[0x8010] = 0x40;

			//Update the screen
			//system_VBL();
			doneFrame = 1 ;

#ifndef BIOSHLE
			update_rtc();	//Periodically update the RTC
#endif

			//Vertical Interrupt? (Confirmed IRQ Level)
			if (statusIFF() <= 4)
				interrupt(5); // VBL
		}

		ram[0x8009]++;	//Next scanline
		if (ram[0x8009] == SCREEN_HEIGHT + 47)	//199th scanline
		{
			ram[0x8009] = 0;
			ram[0x8010] = 0;	//Character Over / Vblank Status
		}
	}

	//=============================================

	//Tick the Clock Generator
	timer_clock0 += cputicks;
	timer_clock1 += cputicks;
	timer_clock2 += cputicks;
	timer_clock3 += cputicks;

	//Run Timer 0 (TRUN)?
	if ((ram[0x20] & 0x01))
	{
		//T01MOD
		switch(ram[0x24] & 0x03)
		{
		case 0:	if (timer_clock0 >= TIMER_HINT_RATE)
				{
					if (ram[0x8009] < SCREEN_HEIGHT-1 || ram[0x8009] == 198)
					{
						timer[0]++;
						timer_clock0 = 0;//-= TIMER_HINT_RATE;
					}
				}
				break;

		case 1:	if (timer_clock0 >= TIMER_T1_RATE)
				{
					timer[0]++;
					timer_clock0 = 0;//-= TIMER_T1_RATE;
				}
				break;

		case 2:	if (timer_clock0 >= TIMER_T4_RATE)
				{
					timer[0]++;
					timer_clock0 = 0;//-= TIMER_T4_RATE;
				}
				break;

		case 3:	if (timer_clock0 >= TIMER_T16_RATE)
				{
					timer[0]++;
					timer_clock0 = 0;//-= TIMER_T16_RATE;
				}
				break;
		}

		//Threshold check
		if (ram[0x22] && timer[0] >= ram[0x22])
		{
			timer[0] = 0;

			if (statusIFF() <= (ram[0x73] & 0x7))
			{
				interrupt(7); // Timer 0 Int.
				return;
			}
		}
	}

	//=======================

	//Run Timer 1 (TRUN)?
	if ((ram[0x20] & 0x02))
	{
		//T23MOD
		switch((ram[0x24] & 0x0C) >> 2)
		{
		case 0:	//TODO: TO0TRG
				break;

		case 1:	if (timer_clock1 >= TIMER_T1_RATE)
				{
					timer[1]++;
					timer_clock1 = 0;//-= TIMER_T1_RATE;
				}
				break;

		case 2:	if (timer_clock1 >= TIMER_T16_RATE)
				{
					timer[1]++;
					timer_clock1 = 0;//-= TIMER_T16_RATE;
				}
				break;

		case 3:	if (timer_clock1 >= TIMER_T256_RATE)
				{
					timer[1]++;
					timer_clock1 = 0;//-= TIMER_T256_RATE;
				}
				break;
		}

		//Threshold check
		if (ram[0x23] && timer[1] >= ram[0x23])
		{
			timer[1] = 0;

			if (statusIFF() <= ((ram[0x73] & 0x70) >> 4))
			{
				interrupt(8); // Timer 1 Int.
				return;
			}
		}
	}

	//=======================

	//Run Timer 2 (TRUN)?
	if ((ram[0x20] & 0x04))
	{
		//T23MOD
		switch(ram[0x28] & 0x03)
		{
		case 0:	// -
				break;

		case 1:	if (timer_clock2 >= TIMER_T1_RATE)
				{
					timer[2]++;
					timer_clock2 = 0;//-= TIMER_T1_RATE;
				}
				break;

		case 2:	if (timer_clock2 >= TIMER_T4_RATE)
				{
					timer[2]++;
					timer_clock2 = 0;//-= TIMER_T4_RATE;
				}
				break;

		case 3:	if (timer_clock2 >= TIMER_T16_RATE)
				{
					timer[2]++;
					timer_clock2 = 0;//-= TIMER_T16_RATE;
				}
				break;
		}

		//Threshold check
		if (ram[0x26] && timer[2] >= ram[0x26])
		{
			timer[2] = 0;

			if (statusIFF() <= ((ram[0x74] & 0x07)))
			{
				interrupt(9);	// Timer 2 Int.
				return;
			}
		}
	}

	//=======================

	//Run Timer 3 (TRUN)?
	if ((ram[0x20] & 0x08))
	{
		//T23MOD
		switch((ram[0x28] & 0x0C) >> 2)
		{
		case 0:	//TODO: TO2TRG
				break;

		case 1:	if (timer_clock3 >= TIMER_T1_RATE)
				{
					timer[3]++;
					timer_clock3 = 0;//-= TIMER_T1_RATE;
				}
				break;

		case 2:	if (timer_clock3 >= TIMER_T16_RATE)
				{
					timer[3]++;
					timer_clock3 = 0;//-= TIMER_T16_RATE;
				}
				break;

		case 3:	if (timer_clock3 >= TIMER_T256_RATE)
				{
					timer[3]++;
					timer_clock3 = 0;//-= TIMER_T256_RATE;
				}
				break;
		}

		//Threshold check
		if (ram[0x27] && timer[3] >= ram[0x27])
		{
			timer[3] = 0;

			Z80_irq();
			if (statusIFF() <= ((ram[0x74] & 0x70) >> 4))
			{
				interrupt(10); // Timer 3 Int.
				return;
			}
		}
	}
}

//=============================================================================

void reset_timers(void)
{
	timer_hint = 0;

	timer[0] = 0;
	timer[1] = 0;
	timer[2] = 0;
	timer[3] = 0;
	
	timer_clock0 = 0;
	timer_clock1 = 0;
	timer_clock2 = 0;
	timer_clock3 = 0;
}

//=============================================================================

