//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

21 JUL 2002 - neopop_uk
=======================================
- Created this file to remove some of the mess from 'system_main.c' and
to make rom loading more abstracted, primarily for ports to systems with
alternate methods of loading roms.
- Simplified 'LoadRomFile' using the 'system_io_read' function. It's now
more memory efficient too.
- Simplified 'LoadRomZip', also more efficient and the error messages
should be more obvious.
- For simplicity, I have made ZIP support optional

//---------------------------------------------------------------------------
*/

//=============================================================================

#define ZIPSUPPORT	//Comment this line to remove zip support

//=============================================================================

#include "neopop.h"
#include <windows.h>
#include <sys/stat.h>

#include "system_main.h"
#include "system_rom.h"

#ifdef NEOPOP_DEBUG
#include "Debugger/system_debug.h"
#endif

#include "rom.h"

//=============================================================================

#ifdef ZIPSUPPORT
#include "zLIB\unzip.h"
static bool LoadRomZip(char* filename);
#endif

static bool LoadRomFile(char* filename);

//=============================================================================

//-----------------------------------------------------------------------------
// system_load_rom()
//-----------------------------------------------------------------------------
bool system_load_rom(char* filename)
{
	bool ret;
	int i;

#ifdef NEOPOP_DEBUG
	system_debug_clear();
	system_debug_stop();
#endif

	//Kill the old rom
	system_unload_rom();

	//Store File Name
	memset(rom.filename, 0, _MAX_PATH);
	for (i = strlen(filename); i > 0; i--)
		if (filename[i] == '/' || filename[i] == '\\')
			break;
	strncpy(rom.filename, filename + i + 1, strlen(filename) - i - 5);

	//Load the file

#ifdef ZIPSUPPORT
	_strlwr(filename);
	if (strstr(filename, ".zip"))
		ret = LoadRomZip(filename);		// Load Zip
	else
#endif

	ret = LoadRomFile(filename);	// Load raw file

	//Success?
	if (ret)
	{
		rom_loaded();			//Perform independent actions
		system_changed_rom();	//Adjust window heading
		return true;
	}
	else
	{
		//Tidy up
		system_unload_rom();
		return false;
	}
}

//-----------------------------------------------------------------------------
// system_unload_rom()
//-----------------------------------------------------------------------------
void system_unload_rom(void)
{
	if (rom.data)
	{
		int i;

		free(rom.data);
		rom.data = NULL;
		rom.length = 0;
		rom_header = 0;

		for (i = 0; i < 16; i++)
			rom.name[i] = 0;

		for (i = 0; i < _MAX_PATH; i++)
			rom.filename[i] = 0;
	}
}

//=============================================================================

static bool LoadRomFile(char* filename)
{
	struct _stat statbuffer;

	//Get file length
	if (_stat(filename, &statbuffer) == -1)
	{
		system_message("LoadRomFile: Cannot find rom file \"%s\"", filename);
		return false;
	}

	rom.length = statbuffer.st_size;
	rom.data = (_u8*)calloc(rom.length, sizeof(_u8));
		
	if (system_io_rom_read(filename, rom.data, rom.length))
	{
		//Success!
		return true;
	}
	else
	{
		//Failed.
		system_message("LoadRomFile: Cannot open rom file \"%s\"", filename);
		return false;
	}
}

//=============================================================================

#ifdef ZIPSUPPORT
static bool LoadRomZip(char* filename)
{
	unzFile zip = NULL;
	char currentZipFileName[_MAX_PATH];
	unz_file_info fileInfo;

	if ((zip = unzOpen(filename)) == NULL)
	{
		system_message("LoadRomZip: Cannot find zip file \"%s\"", filename);
		return false;
	}

	//Scan for the file list
	if (unzGoToFirstFile(zip) != UNZ_OK)
	{
		unzClose(zip);
		system_message("LoadRomZip: Corrupted zip file \"%s\"", filename);
		return false;
	}
		
	while (unzGetCurrentFileInfo(zip, &fileInfo, currentZipFileName, 
			_MAX_PATH, NULL, 0, NULL, 0) == UNZ_OK)
	{
		_strlwr(currentZipFileName);

		//Find a rom with the correct extension
		if (strstr(currentZipFileName, ".ngp") == NULL &&
			strstr(currentZipFileName, ".ngc") == NULL &&
			strstr(currentZipFileName, ".npc") == NULL)
		{
			//Last one?
			if (unzGoToNextFile(zip) == UNZ_END_OF_LIST_OF_FILE)
				break;
			else
				continue;	//Try the next...
		}

		//Extract It
		rom.length = fileInfo.uncompressed_size;

		//Open the file
		if(unzOpenCurrentFile(zip) == UNZ_OK)
		{
			rom.length = fileInfo.uncompressed_size;

			//Reserve the space required
			rom.data = (_u8*)calloc(rom.length, sizeof(_u8));
									
			//Read the File
			if(unzReadCurrentFile(zip, rom.data, rom.length) >= 0)
			{
				//Load complete
				unzCloseCurrentFile(zip);
				unzClose(zip);
				return true;	//success!
			}
			else
			{
				system_message("LoadRomZip: unzReadCurrentFile: \"%s\" could not be read from zip", currentZipFileName);
				unzCloseCurrentFile(zip);
				unzClose(zip);
				return false;
			}
		}
		else
		{
			system_message("LoadRomZip: \"%s\" cannot be opened", currentZipFileName);
			unzClose(zip);
			return false;
		}
	}

	unzClose(zip);
	system_message("LoadRomZip: No roms found in \"%s\"", filename);
	return false;
}
#endif

//=============================================================================

