//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

21 JUL 2002 - neopop_uk
=======================================
- Moved a lot of functions out of this file, including get/put config.
- Re-added the performance percentage, by popular demand!

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include <direct.h>
#include <tchar.h>
#include <windows.h>
#include <windowsx.h>
#include "resource.h"

#include "system_main.h"
#include "system_graphics.h"
#include "system_input.h"
#include "system_sound.h"
#include "system_rom.h"
#include "system_config.h"

#ifdef NEOPOP_DEBUG
#include "Debugger/system_debug.h"
#endif

#include "state.h"
#include "bios.h"
#include "gfx.h"
#include "rom.h"

//=============================================================================

HWND g_hWnd;
HINSTANCE g_hInstance;
RECT g_ScreenRect;

//=============================================================================

static HMENU g_menu;
#define FPSTHROTTLERATE 6
static DWORD fpsTick, fpsDiff, fpsPercent, fpsFrameCount = 0, fpsLastTime = 0;
static bool paused = false;
static bool auto_pause = false;

//=============================================================================

//-----------------------------------------------------------------------------
// system_VBL()
//-----------------------------------------------------------------------------
void system_VBL(void)
{
	//Throttle speed, calculate performance percentage
	fpsFrameCount++;
	if (fpsFrameCount == FPSTHROTTLERATE)
	{
		do
		{
			fpsTick = timeGetTime();
			fpsDiff = fpsTick - fpsLastTime;
			fpsPercent = (int)((1668.3 * FPSTHROTTLERATE) / (float)fpsDiff);
		}
		while (fpsPercent > 100);
		fpsFrameCount = 0;
		fpsLastTime = fpsTick;

		if (rom.data)
		{
			char title[128];
			sprintf(title, PROGRAM_NAME" - %s - %3d%%", rom.name, fpsPercent);
			SetWindowText(g_hWnd, title);
		}
	}

	//Update!
	if (mute == false) system_sound_update();
	system_input_update();
	system_graphics_update();
}

//-----------------------------------------------------------------------------
// system_message()
//-----------------------------------------------------------------------------
void __cdecl system_message(char* vaMessage,...)
{
	char message[1001];
    va_list vl;

    va_start(vl, vaMessage);
    vsprintf(message, vaMessage, vl);
	va_end(vl);

	MessageBox(GetDesktopWindow(), message, PROGRAM_NAME, MB_OK | MB_ICONSTOP);
}

//-----------------------------------------------------------------------------
// system_changed_rom()
//-----------------------------------------------------------------------------
void system_changed_rom(void)
{
	char title[128];
	sprintf(title, PROGRAM_NAME" - %s", rom.name);
	SetWindowText(g_hWnd, title);
}

//=============================================================================

//-----------------------------------------------------------------------------
// setZoom()
//-----------------------------------------------------------------------------
static void setZoom(int z)
{
	UINT m;
	zoom = z;

	//Check the menu entries
	if (z == 1) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX1, m);
	if (z == 2) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX2, m);
	if (z == 3) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX3, m);
	if (z == 4) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX4, m);

	//Adjust the window
	SetWindowPos(g_hWnd, NULL, 0,0, 
		(SCREEN_WIDTH * zoom) + GetSystemMetrics(SM_CXFIXEDFRAME)*2 + (BORDER_WIDTH * 2), 
		(SCREEN_HEIGHT * zoom) + GetSystemMetrics(SM_CYFIXEDFRAME)*2 + 
			GetSystemMetrics(SM_CYCAPTION) + GetSystemMetrics(SM_CYMENU) + (BORDER_HEIGHT * 2), 
		SWP_NOZORDER | SWP_NOMOVE);

	//Update the display rectangle
	GetClientRect(g_hWnd, &g_ScreenRect);
	ClientToScreen(g_hWnd, (POINT*)&g_ScreenRect.left);
	ClientToScreen(g_hWnd, (POINT*)&g_ScreenRect.right);

	system_graphics_update();
}

//=============================================================================

//-----------------------------------------------------------------------------
// AboutProc()
//-----------------------------------------------------------------------------
BOOL CALLBACK AboutProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{
	case WM_INITDIALOG:
		{
			//Apply version info
			HWND ver = GetDlgItem(hDlg, IDC_VERSION);
			Static_SetText(ver, NEOPOP_VERSION "\nby neopop_uk");
			return 1;
		}

	case WM_COMMAND:
		if(LOWORD(wParam) == IDCANCEL)
		{
			EndDialog(hDlg, 0);
			return 1;
		}
	}

	return 0;
}

//-----------------------------------------------------------------------------
// WindowProc()
//-----------------------------------------------------------------------------
LRESULT CALLBACK WindowProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{
	case WM_CLOSE:
		//Get Window position
		if (IsIconic(g_hWnd) == 0)
		{
			RECT r;
			GetWindowRect(g_hWnd, &r);
			mainview_x = r.left;
			mainview_y = r.top;
		}

		PostQuitMessage(0);
		return 0;

		//Avoid nasty sound stuttering when interacting with the
		//Windows interface. Sound will be restarted on the next
		//screen update
	case WM_MENUSELECT:
	case WM_ENTERSIZEMOVE:
		system_sound_start();
		return 0;

	case WM_COMMAND:

		switch(LOWORD(wParam))
		{
		//Game Menu
		case ID_GAME_LOADROM:
			{
				char Selection[_MAX_PATH] = "\0";
				char File[_MAX_FNAME] = "\0";

				OPENFILENAME OpenFileName;
				ZeroMemory(&OpenFileName, sizeof(OPENFILENAME));
				OpenFileName.lStructSize	= sizeof(OPENFILENAME);
				OpenFileName.hwndOwner		= g_hWnd;
				OpenFileName.hInstance		= g_hInstance;
				OpenFileName.lpstrFilter	= "Rom Files (*.ngp,*.ngc,*.npc,*.zip)\0*.ngp;*.ngc;*.npc;*.zip\0\0";
				OpenFileName.lpstrFile		= Selection;
				OpenFileName.lpstrFileTitle = File;
				OpenFileName.nMaxFileTitle  = _MAX_FNAME;
				OpenFileName.nMaxFile		= _MAX_PATH;
				OpenFileName.lpstrInitialDir = RomDirectory;
				OpenFileName.lpstrTitle		= "Open Rom...";
				OpenFileName.Flags			= OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;

				if (GetOpenFileName(&OpenFileName) != 0)
				{
					if (system_load_rom(Selection))	//Load the rom
					{
						//Enable the unload
						EnableMenuItem(g_menu, ID_GAME_UNLOADROM, MF_ENABLED);
						EnableMenuItem(g_menu, ID_GAME_LOADSTATE, MF_ENABLED);
						EnableMenuItem(g_menu, ID_GAME_SAVESTATE, MF_ENABLED);

						reset();
#ifndef NEOPOP_DEBUG
						EnableMenuItem(g_menu, ID_GAME_PAUSE, MF_ENABLED);
						CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_UNCHECKED);
						paused = false;
						system_sound_start();
#endif

						//Only change path when load is a success.
						_getcwd(RomDirectory, _MAX_PATH);
					}
				}

				return 0;
			}
	
		case ID_GAME_UNLOADROM:
			if (rom.data)
			{
				system_unload_rom();
				reset();

#ifdef NEOPOP_DEBUG
				system_debug_clear();
				system_debug_message("Game removed. Returning to bios...");
#endif
				
				//Restore old name
				SetWindowText(g_hWnd, PROGRAM_NAME);
				
				//Disable the unload + pause
				EnableMenuItem(g_menu, ID_GAME_UNLOADROM, MF_GRAYED);
				EnableMenuItem(g_menu, ID_GAME_LOADSTATE, MF_GRAYED);
				EnableMenuItem(g_menu, ID_GAME_SAVESTATE, MF_GRAYED);
				EnableMenuItem(g_menu, ID_GAME_PAUSE, MF_GRAYED);
				CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_UNCHECKED);
			}

			return 0;

		case ID_GAME_RESET:
			paused = false;
			CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_UNCHECKED);
			reset();
			return 0;

		case ID_GAME_PAUSE:
			paused = !paused;
			if (paused)
			{
				system_sound_stop();
				CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_CHECKED);
			}
			else
			{
				system_sound_start();
				CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_UNCHECKED);
			}
			return 0;

		//Load State
		case ID_GAME_LOADSTATE:
			{
				TCHAR Selection[_MAX_PATH] = "\0";
				TCHAR File[_MAX_FNAME] = "\0";
				OPENFILENAME OpenFileName;

				if (rom.data)
					wsprintf(Selection, _T("%s.ngs"), rom.filename);

				ZeroMemory(&OpenFileName, sizeof(OPENFILENAME));
				OpenFileName.lStructSize	= sizeof(OPENFILENAME);
				OpenFileName.hwndOwner		= g_hWnd;
				OpenFileName.hInstance		= g_hInstance;
				OpenFileName.lpstrFilter	= "Saved States (*.ngs)\0*.ngs\0\0";
				OpenFileName.lpstrFile		= Selection;
				OpenFileName.lpstrFileTitle = File;
				OpenFileName.nMaxFileTitle  = _MAX_FNAME;
				OpenFileName.nMaxFile		= _MAX_PATH;
				OpenFileName.lpstrInitialDir = StateDirectory;
				OpenFileName.lpstrTitle		= "Open Saved State...";
				OpenFileName.Flags			= OFN_HIDEREADONLY | OFN_FILEMUSTEXIST |
												OFN_PATHMUSTEXIST | OFN_EXTENSIONDIFFERENT;

				if (GetOpenFileName(&OpenFileName) != 0)
				{
					state_restore(Selection);
					_getcwd(StateDirectory, _MAX_PATH);
				}

				return 0;
			}

			break;

		//Save State
		case ID_GAME_SAVESTATE:
			{
				TCHAR Selection[_MAX_PATH] = "\0";
				TCHAR File[_MAX_FNAME] = "\0";
				OPENFILENAME OpenFileName;

				if (rom.data)
					wsprintf(Selection, _T("%s.ngs"), rom.filename);

				ZeroMemory(&OpenFileName, sizeof(OPENFILENAME));
				OpenFileName.lStructSize	= sizeof(OPENFILENAME);
				OpenFileName.hwndOwner		= g_hWnd;
				OpenFileName.hInstance		= g_hInstance;
				OpenFileName.lpstrFilter	= "Saved States (*.ngs)\0*.ngs\0\0";
				OpenFileName.lpstrFile		= Selection;
				OpenFileName.lpstrFileTitle = File;
				OpenFileName.nMaxFileTitle  = _MAX_FNAME;
				OpenFileName.nMaxFile		= _MAX_PATH;
				OpenFileName.lpstrInitialDir = StateDirectory;
				OpenFileName.lpstrTitle		= "Choose Saved State...";
				OpenFileName.Flags			= OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT |
												OFN_PATHMUSTEXIST | OFN_EXTENSIONDIFFERENT;
				OpenFileName.lpstrDefExt	= "ngs";

				if (GetSaveFileName(&OpenFileName) != 0)
				{
					state_store(Selection);
					_getcwd(StateDirectory, _MAX_PATH);
				}

				return 0;
			}
			break;

		case ID_GAME_EXIT:
			//Get Window position
			if (IsIconic(g_hWnd) == 0)
			{
				RECT r;
				GetWindowRect(g_hWnd, &r);
				mainview_x = r.left;
				mainview_y = r.top;
			}

			PostQuitMessage(0);
			return 0;

		//Options Menu

		case ID_OPTIONS_STEREO:
			system_sound_shutdown();
			stereo = true;
			CheckMenuItem(g_menu, ID_OPTIONS_STEREO, MF_CHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_MONO, MF_UNCHECKED);
			system_sound_init();
			system_sound_start();
			break;

		case ID_OPTIONS_MONO:
			system_sound_shutdown();
			stereo = false;
			CheckMenuItem(g_menu, ID_OPTIONS_STEREO, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_MONO, MF_CHECKED);
			system_sound_init();
			system_sound_start();
			break;

		case ID_OPTIONS_MUTE:
			mute = !mute;
			system_sound_start();
			if (mute)
				CheckMenuItem(g_menu, ID_OPTIONS_MUTE, MF_CHECKED);
			else
				CheckMenuItem(g_menu, ID_OPTIONS_MUTE, MF_UNCHECKED);
			return 0;
			
		case ID_OPTIONS_SOUND_11KHZ:
			system_sound_shutdown();
			sound_frequency = 11025;
			system_sound_init();
			system_sound_start();

			//Check the Sound menu entries
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_11KHZ, MF_CHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_22KHZ, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_44KHZ, MF_UNCHECKED);
			break;
		
		case ID_OPTIONS_SOUND_22KHZ:
			system_sound_shutdown();
			sound_frequency = 22050;
			system_sound_init();
			system_sound_start();
			//Check the Sound menu entries
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_11KHZ, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_22KHZ, MF_CHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_44KHZ, MF_UNCHECKED);
			break;

		case ID_OPTIONS_SOUND_44KHZ:
			system_sound_shutdown();
			sound_frequency = 44100;
			system_sound_init();
			system_sound_start();
			//Check the Sound menu entries
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_11KHZ, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_22KHZ, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_SOUND_44KHZ, MF_CHECKED);
			break;

		case ID_OPTIONS_CONTROLS:
			DialogBox(g_hInstance, MAKEINTRESOURCE(IDD_CONTROLS), g_hWnd, InputDlg);
 			return 0;

		case ID_OPTIONS_LANGUAGE_ENGLISH:
			language_english = true;
			CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_ENGLISH, MF_CHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_JAPANESE, MF_UNCHECKED);
			break;

		case ID_OPTIONS_LANGUAGE_JAPANESE:
			language_english = false;
			CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_ENGLISH, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_JAPANESE, MF_CHECKED);
			break;

		case ID_OPTIONS_DISPLAYTYPE_COLOUR:
			system_colour = true;
			CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_COLOUR, MF_CHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_GREYSCALE, MF_UNCHECKED);
			break;

		case ID_OPTIONS_DISPLAYTYPE_GREYSCALE:
			system_colour = false;
			CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_COLOUR, MF_UNCHECKED);
			CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_GREYSCALE, MF_CHECKED);
			break;

		case ID_OPTIONS_ZOOMX1: setZoom(1);	return 0;
		case ID_OPTIONS_ZOOMX2: setZoom(2);	return 0;
		case ID_OPTIONS_ZOOMX3: setZoom(3);	return 0;
		case ID_OPTIONS_ZOOMX4: setZoom(4);	return 0;

		//Help Menu
		case ID_HELP_ABOUTNEOPOP:
			DialogBox(g_hInstance, MAKEINTRESOURCE(IDD_ABOUT), g_hWnd, AboutProc);
 			return 0;
		}
		break;

	case WM_SIZE:
		if (wParam == SIZE_MINIMIZED)
		{
			system_sound_stop();
			auto_pause = true;
			break;
		}
		if (wParam == SIZE_RESTORED)
		{
			system_sound_start();
			auto_pause = false;
			break;
		}

	case WM_MOVE:
		GetClientRect(hWnd, &g_ScreenRect);
		ClientToScreen(hWnd, (POINT*)&g_ScreenRect.left);
		ClientToScreen(hWnd, (POINT*)&g_ScreenRect.right);
		return 0;

	case WM_PAINT:
		system_graphics_update();
		break;

#ifndef NEOPOP_DEBUG
	case WM_ACTIVATE:
		auto_pause = ! ((LOWORD(wParam) == WA_ACTIVE) || 
						(LOWORD(wParam) == WA_CLICKACTIVE));

		if (auto_pause)
		{
			system_sound_stop();
		}
		else
		{
			system_sound_start();
		}
		return 0;
#endif
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


//=============================================================================

//-----------------------------------------------------------------------------
// WinInit()
//-----------------------------------------------------------------------------
bool WinInit(void)
{
    WNDCLASSEX wc;
	UINT m;
	int width = (SCREEN_WIDTH * zoom) + GetSystemMetrics(SM_CXFIXEDFRAME)*2;
	int height = (SCREEN_HEIGHT * zoom) + GetSystemMetrics(SM_CYFIXEDFRAME)*2 + 
			GetSystemMetrics(SM_CYCAPTION) + GetSystemMetrics(SM_CYMENU);

	// Set up and register window class
	wc.cbSize = sizeof(WNDCLASSEX);
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = WindowProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;					
	wc.hInstance = g_hInstance;
	wc.hIcon = LoadIcon(g_hInstance, MAKEINTRESOURCE( IDI_ICON ));
	wc.hCursor = NULL; // No mouse pointer required
	wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wc.lpszMenuName = NULL;
	wc.lpszClassName = "NeoPop";
	wc.hIconSm = NULL;
	RegisterClassEx(&wc);	// Register the class
	
	g_menu = LoadMenu(g_hInstance, MAKEINTRESOURCE(IDR_MENU));

	//Check the Zoom menu entries
	if (zoom == 1) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX1, m);
	if (zoom == 2) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX2, m);
	if (zoom == 3) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX3, m);
	if (zoom == 4) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_ZOOMX4, m);

	//Check the pause menu item
	CheckMenuItem(g_menu, ID_GAME_PAUSE, MF_UNCHECKED);
	
	//Set the mute state
	if (mute)
		CheckMenuItem(g_menu, ID_OPTIONS_MUTE, MF_CHECKED);
	else
		CheckMenuItem(g_menu, ID_OPTIONS_MUTE, MF_UNCHECKED);

	//Set the speaker options
	if (stereo)
	{
		CheckMenuItem(g_menu, ID_OPTIONS_STEREO, MF_CHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_MONO, MF_UNCHECKED);
	}
	else
	{
		CheckMenuItem(g_menu, ID_OPTIONS_STEREO, MF_UNCHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_MONO, MF_CHECKED);
	}

	//Check the Sound menu entries
	if (sound_frequency == 11025) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_SOUND_11KHZ, m);
	if (sound_frequency == 22050) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_SOUND_22KHZ, m);
	if (sound_frequency == 44100) m = MF_CHECKED; else m = MF_UNCHECKED;
	CheckMenuItem(g_menu, ID_OPTIONS_SOUND_44KHZ, m);

	//Check the language
	if (language_english)
	{
		CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_ENGLISH, MF_CHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_JAPANESE, MF_UNCHECKED);
	}
	else
	{
		CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_ENGLISH, MF_UNCHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_LANGUAGE_JAPANESE, MF_CHECKED);
	}

	//Check the display
	if (system_colour)
	{
		CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_COLOUR, MF_CHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_GREYSCALE, MF_UNCHECKED);
	}
	else
	{
		CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_COLOUR, MF_CHECKED);
		CheckMenuItem(g_menu, ID_OPTIONS_DISPLAYTYPE_GREYSCALE, MF_UNCHECKED);
	}
	
	//Disable the unload + pause
	EnableMenuItem(g_menu, ID_GAME_UNLOADROM, MF_GRAYED);
	
	//Can't pause the bios!
	EnableMenuItem(g_menu, ID_GAME_PAUSE, MF_GRAYED);

	EnableMenuItem(g_menu, ID_GAME_LOADSTATE, MF_GRAYED);
	EnableMenuItem(g_menu, ID_GAME_SAVESTATE, MF_GRAYED);

	//Create the application window
	g_hWnd = CreateWindow("NeoPop", PROGRAM_NAME,
		WS_BORDER | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,
		mainview_x,
		mainview_y,

		width, height,
		NULL, g_menu, g_hInstance, NULL);
	if (g_hWnd == NULL)	//Do an error check
		return false;

	//Make the window appear on top
	setZoom(zoom);
	ShowWindow(g_hWnd, SW_SHOWNORMAL); 
  	SetFocus(g_hWnd);

	//Successful initialisation
	return true;
}

//=============================================================================

//-----------------------------------------------------------------------------
// WinMain()
//-----------------------------------------------------------------------------
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE a, LPSTR cmdline, int c)
{
	MSG msg;
	g_hInstance = hInstance; // Copy instance for use by DirectX

	//Fill the bios buffer however it needs to be...
	if (biosInstall() == false)
		return 0;

	system_get_config();	//Read configuration settings

	//Create the window and setup the menus
	if (WinInit() == false)
	{
		system_message("An error has occured creating the application window");
		return 0;
	}

	//Initialise Draw
	if (system_graphics_init() == false)
	{
		system_message("An error has occured initialising DirectDraw");
		return 0;
	}

	//Initialise Input
	if (system_input_init() == false)
	{
		system_message("An error has occured initialising DirectInput");
		system_graphics_shutdown();
		return 0;
	}

	//Initialise Sound
	if (system_sound_init() == false)
	{
		//Don't care too much, just disable the mute option to
		//give the user some small indication.
		EnableMenuItem(g_menu, ID_OPTIONS_MUTE, MF_GRAYED);
		CheckMenuItem(g_menu, ID_OPTIONS_MUTE, MF_CHECKED);
		mute = true;
	}

#ifdef NEOPOP_DEBUG
	system_debug_init();	//Open the debugger
#else
	system_sound_start();	//Start up sound straight away!
#endif

	biosInstall();		// Build a bios

	//Attempt to load from the command line
	if (strlen(cmdline) > 4)
		system_load_rom(cmdline);

	reset();

	//=========

	do
	{
#ifdef NEOPOP_DEBUG
		//======================================
		// DEBUGGER BUILD - MAIN LOOP
		//======================================
		if (system_debug_running())
		{
			//Stop BEFORE the breakpoint
			if (breakpoint_reached() == false)
				emulate_debug(running_dis_TLCS900h, running_dis_Z80);
		}
		//======================================
		//======================================
#else
		//======================================
		// MAIN BUILD - MAIN LOOP
		//======================================
		if (paused || auto_pause)
			Sleep(100);
		else
			emulate();
		//======================================
		//======================================
#endif

		//Message Pump
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	while(msg.message != WM_QUIT);

	//=========

	//Shutdown
#ifdef NEOPOP_DEBUG
	system_debug_shutdown();
#endif

	system_unload_rom();

	system_graphics_shutdown();
	system_input_shutdown();
	system_sound_shutdown();

	system_put_config();	//Write configuration settings

	//exit.
	UnregisterClass("NeoPop", g_hInstance);
	return msg.wParam;
}

//=============================================================================
