//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

21 JUL 2002 - neopop_uk
=======================================
- Promoted 'system_graphics_clear' with a comment heading to make it
more prominent as an emulator interface function.

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include <windows.h>
#include <ddraw.h>

#include "system_main.h"
#include "system_graphics.h"

#include "mem.h"
#include "gfx.h"

//=============================================================================

static DWORD dwRMask32, dwRShiftL32 = 8, dwRShiftR32 = 0;
static DWORD dwGMask32, dwGShiftL32 = 8, dwGShiftR32 = 0;
static DWORD dwBMask32, dwBShiftL32 = 8, dwBShiftR32 = 0;
static DWORD dwAMask32, dwAShiftL32 = 8, dwAShiftR32 = 0;

static WORD dwRMask16, dwRShiftL16 = 8, dwRShiftR16 = 0;
static WORD dwGMask16, dwGShiftL16 = 8, dwGShiftR16 = 0;
static WORD dwBMask16, dwBShiftL16 = 8, dwBShiftR16 = 0;
static WORD dwAMask16, dwAShiftL16 = 8, dwAShiftR16 = 0;

static LPDIRECTDRAW            lpDD = NULL;           
static LPDIRECTDRAWSURFACE     lpDDSPrimary = NULL;   
static LPDIRECTDRAWSURFACE     lpDDSOne = NULL;
static LPDIRECTDRAWCLIPPER     lpClipper = NULL;      

static DDSURFACEDESC ddsd;

//=============================================================================

//-----------------------------------------------------------------------------
// system_graphics_clear()
//-----------------------------------------------------------------------------
void system_graphics_clear(void)
{
	memset(cfb, 0xFFFFFFFF, sizeof(cfb));
	system_graphics_update();
}

//=============================================================================

bool system_graphics_init(void)
{
    HRESULT	ddrval;

	WORD	dwMask16;	
	DWORD	dwMask32;

	ddrval = DirectDrawCreate(NULL,&lpDD,NULL);

    if(ddrval!=DD_OK)
    {
        return false;
    }

	ddrval = IDirectDraw_SetCooperativeLevel(lpDD, g_hWnd, DDSCL_NORMAL);

    if(ddrval != DD_OK)
        return false;

 	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

    ddrval = IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSPrimary, NULL);

    if(ddrval != DD_OK)
        return false;

	ddrval = IDirectDraw_CreateClipper(lpDD, 0, &lpClipper, NULL);

    if(ddrval != DD_OK)
        return false;
	
	IDirectDrawSurface_SetClipper(lpDDSPrimary, lpClipper);
	IDirectDrawClipper_SetHWnd(lpClipper, 0, g_hWnd);
    IDirectDraw_Release(lpClipper);
		
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS|DDSD_HEIGHT|DDSD_WIDTH;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddsd.dwWidth = SCREEN_WIDTH + (BORDER_WIDTH * 2);
    ddsd.dwHeight = SCREEN_HEIGHT + (BORDER_HEIGHT * 2);

	IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSOne, NULL);    

    if(lpDDSOne == NULL)
        return false;

	//GET COLOUR BIT MASKS

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	if (IDirectDrawSurface_Lock(lpDDSPrimary, NULL, &ddsd, 
			DDLOCK_WAIT, NULL) != DD_OK)
		return false;

    dwRMask32 = ddsd.ddpfPixelFormat.dwRBitMask;
    dwGMask32 = ddsd.ddpfPixelFormat.dwGBitMask;
    dwBMask32 = ddsd.ddpfPixelFormat.dwBBitMask;
    dwAMask32 = ddsd.ddpfPixelFormat.dwRGBAlphaBitMask;
    
	dwRMask16 = (_u16)dwRMask32;
	dwGMask16 = (_u16)dwGMask32;
	dwBMask16 = (_u16)dwBMask32;
	dwAMask16 = (_u16)dwAMask32;

	//16 bit masks
	for( dwMask16=dwRMask16; dwMask16 && !(dwMask16&1); dwMask16>>=1 ) dwRShiftR16++;
    for( ; dwMask16; dwMask16>>=1 ) dwRShiftL16--;
    for( dwMask16=dwGMask16; dwMask16 && !(dwMask16&1); dwMask16>>=1 ) dwGShiftR16++;
    for( ; dwMask16; dwMask16>>=1 ) dwGShiftL16--;
    for( dwMask16=dwBMask16; dwMask16 && !(dwMask16&1); dwMask16>>=1 ) dwBShiftR16++;
    for( ; dwMask16; dwMask16>>=1 ) dwBShiftL16--;
    for( dwMask16=dwAMask16; dwMask16 && !(dwMask16&1); dwMask16>>=1 ) dwAShiftR16++;
    for( ; dwMask16; dwMask16>>=1 ) dwAShiftL16--;

	//32 bit masks
	for( dwMask32=dwRMask32; dwMask32 && !(dwMask32&1); dwMask32>>=1 ) dwRShiftR32++;
    for( ; dwMask32; dwMask32>>=1 ) dwRShiftL32--;
    for( dwMask32=dwGMask32; dwMask32 && !(dwMask32&1); dwMask32>>=1 ) dwGShiftR32++;
    for( ; dwMask32; dwMask32>>=1 ) dwGShiftL32--;
    for( dwMask32=dwBMask32; dwMask32 && !(dwMask32&1); dwMask32>>=1 ) dwBShiftR32++;
    for( ; dwMask32; dwMask32>>=1 ) dwBShiftL32--;
    for( dwMask32=dwAMask32; dwMask32 && !(dwMask32&1); dwMask32>>=1 ) dwAShiftR32++;
    for( ; dwMask32; dwMask32>>=1 ) dwAShiftL32--;

	IDirectDrawSurface_Unlock(lpDDSPrimary, NULL);	

	//Only 16 or 32 bit renderer
	if (ddsd.ddpfPixelFormat.dwRGBBitCount != 16 &&
		ddsd.ddpfPixelFormat.dwRGBBitCount != 32)
	{
		MessageBox(g_hWnd, "Only 16 or 32 bit colour modes are supported.", 
			PROGRAM_NAME, MB_OK | MB_ICONSTOP);
		return false;
	}

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);

	//Okay!
	return true;
}

void system_graphics_update(void)
{
	HRESULT ddrval;

	ddsd.dwSize = sizeof(ddsd);

	//Update the render target
	if (lpDDSOne && IDirectDrawSurface_Lock(lpDDSOne, NULL, &ddsd, 
		DDLOCK_WRITEONLY | DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL) == DD_OK)
	{
		int x, y, row;

		_u32 w = ddsd.lPitch >> 1;

		//Clear entire screen
		memset(ddsd.lpSurface, 0, ddsd.lPitch * (SCREEN_HEIGHT + (BORDER_HEIGHT*2)));

		//16 BIT DISPLAY
		if (ddsd.ddpfPixelFormat.dwRGBBitCount == 16)
		{
			for (y = 0; y < SCREEN_HEIGHT; y++)
			{
				row = ((y + BORDER_HEIGHT) * w);

				for (x = 0; x < SCREEN_WIDTH; x++)
				{
					_u8 r = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0xFF0000) >> 0x10);
					_u8 g = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0x00FF00) >> 0x08);
					_u8 b = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0x0000FF) >> 0x00);

					_u16 dr = ((r>>(dwRShiftL16))<<dwRShiftR16)&dwRMask16;
					_u16 dg = ((g>>(dwGShiftL16))<<dwGShiftR16)&dwGMask16;
					_u16 db = ((b>>(dwBShiftL16))<<dwBShiftR16)&dwBMask16;

					*((_u16*)(ddsd.lpSurface) + x + BORDER_WIDTH + row) = dr | dg | db;
				}
			}
		}
		else	//32 BIT DISPLAY
		{
			w >>= 1;

			for (y = 0; y < SCREEN_HEIGHT; y++)
			{
				row = ((y + BORDER_HEIGHT) * w);

				for (x = 0; x < SCREEN_WIDTH; x++)
				{
					_u8 r = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0xFF0000) >> 0x10);
					_u8 g = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0x00FF00) >> 0x08);
					_u8 b = (_u8)((cfb[x + (y * SCREEN_WIDTH)] & 0x0000FF) >> 0x00);

					_u32 dr = ((r>>(dwRShiftL32))<<dwRShiftR32)&dwRMask32;
					_u32 dg = ((g>>(dwGShiftL32))<<dwGShiftR32)&dwGMask32;
					_u32 db = ((b>>(dwBShiftL32))<<dwBShiftR32)&dwBMask32;

					*((_u32*)(ddsd.lpSurface) + x + BORDER_WIDTH + row) = dr | dg | db;
				}
			}
		}
	
		IDirectDrawSurface_Unlock(lpDDSOne, NULL);	
		
		//Update the primary surface
		ddrval = IDirectDrawSurface_Blt(lpDDSPrimary, &g_ScreenRect, 
			lpDDSOne, NULL, DDBLT_WAIT, NULL);

		if (ddrval == DDERR_SURFACELOST)
		{
			IDirectDrawSurface_Restore(lpDDSPrimary);
			IDirectDrawSurface_Restore(lpDDSOne);
		}
	}
}

void system_graphics_shutdown(void)
{
    if (lpDD)
    {
        if (lpDDSPrimary)	IDirectDraw_Release(lpDDSPrimary);

        if (lpDDSOne)		IDirectDraw_Release(lpDDSOne);

        IDirectDraw_Release(lpDD);
        lpDD = NULL;
    }
}

//=============================================================================
