/*
	WinSTon

	Screen Property Page
*/

#include "..\includes\winston.h"
#include "..\includes\dialog.h"
#include "..\includes\dsurface.h"
#include "..\includes\misc.h"
#include "..\includes\screensnapshot.h"

// Button IDs
static int Screen_DialogItems[] = {
	IDC_STATIC_FRAMESPERSECOND,
	IDC_EDIT_FRAMESPERSECOND,
	IDC_BUTTON_SCREEN_RECORD,
	IDC_BUTTON_SCREEN_STOP,
	IDC_BUTTON_SCREEN_CAPTURE,
	0,	//term
};

static int Screen_FullScreen_DialogItems[] = {
	IDC_RADIO_FULLSCREEN,
	IDC_STATIC_DISPLAYMODES,
	IDC_COMBO_DISPLAYMODES,
	0,	//term
};

static int Screen_AdvancedFullScreen_DialogItems[] = {
	IDC_RADIO_INTERLACED,
	IDC_RADIO_SYNCRETRACE,
	0,	//term
};

static int ScreenFramesPerSecond_DialogItems[] = {
	IDC_STATIC_FRAMESPERSECOND,
	IDC_EDIT_FRAMESPERSECOND,
	0,	//term
};

char szFramesPerSecond[256];
HWND ScreenhDlg=NULL;
BOOL bCapturedScreen;

//-----------------------------------------------------------------------
/*
	Set defaults for Screen
*/
void Dialog_Screen_SetDefaults(void)
{
	// Set defaults for Screen
	ConfigureParams.Screen.bFullScreen = FALSE;
	ConfigureParams.Screen.Advanced.bDoubleSizeWindow = FALSE;
	ConfigureParams.Screen.Advanced.bAllowOverscan = TRUE;
	ConfigureParams.Screen.Advanced.bInterlacedFullScreen = TRUE;
	ConfigureParams.Screen.Advanced.bSyncToRetrace = FALSE;
	ConfigureParams.Screen.Advanced.bFrameSkip = FALSE;	
	ConfigureParams.Screen.ChosenDisplayMode = DISPLAYMODE_HICOL_HIGHRES;
	ConfigureParams.Screen.bCaptureChange = FALSE;
	ConfigureParams.Screen.nFramesPerSecond = 1;
	ConfigureParams.Screen.bUseHighRes = FALSE;
}

//-----------------------------------------------------------------------
/*
	Store/Clear hDlg handle so can change options on this page while viewing another
*/
void Dialog_Screen_ClearHandle(void)
{
	ScreenhDlg = NULL;
}

//-----------------------------------------------------------------------
/*
	Get hDlg handle
*/
HWND Dialog_Screen_GetHandle(void)
{
	return(ScreenhDlg);
}

//-----------------------------------------------------------------------
/*
	Enable item according to options states
*/
void Dialog_Screen_EnableItems(HWND hDlg)
{
	// Turn on all by default
	Dialog_EnableItems(hDlg,Screen_DialogItems,TRUE);

	// Use DirectX, if not disable full-screen options
	Dialog_EnableItems(hDlg,Screen_FullScreen_DialogItems,bDirectDrawWorking);

	// Have already captured screen?
	if (bCapturedScreen)
		Dialog_EnableItem(hDlg,IDC_BUTTON_SCREEN_CAPTURE,FALSE);
	// Are we recording?
	if (ScreenSnapShot_AreWeRecording()) {
		Dialog_EnableItem(hDlg,IDC_BUTTON_SCREEN_RECORD,FALSE);
		Dialog_EnableItem(hDlg,IDC_BUTTON_SCREEN_STOP,TRUE);
		Dialog_EnableItem(hDlg,IDC_EDIT_FRAMESPERSECOND,FALSE);
		Dialog_EnableItem(hDlg,IDC_STATIC_FRAMESPERSECOND,FALSE);
		Dialog_EnableItem(hDlg,IDC_RADIO_CAPTURECHANGE,FALSE);
	}
	else {
		Dialog_EnableItem(hDlg,IDC_BUTTON_SCREEN_RECORD,TRUE);
		Dialog_EnableItem(hDlg,IDC_BUTTON_SCREEN_STOP,FALSE);
		Dialog_EnableItem(hDlg,IDC_EDIT_FRAMESPERSECOND,TRUE);
		Dialog_EnableItem(hDlg,IDC_STATIC_FRAMESPERSECOND,TRUE);
		Dialog_EnableItem(hDlg,IDC_RADIO_CAPTURECHANGE,TRUE);
	}
	// Are we to record on changes?
	if (Dialog_ReadButton(hDlg,IDC_RADIO_CAPTURECHANGE))
		Dialog_EnableItems(hDlg,ScreenFramesPerSecond_DialogItems,FALSE);
}

//-----------------------------------------------------------------------
/*
	Keep range between 0...50 fps
*/
int Dialog_Screen_LimitFramesPerSecond(int nFramesPerSecond)
{
	return(Misc_LimitInt(nFramesPerSecond, 1,50));
}

//-----------------------------------------------------------------------
/*
	Write settings into dialog page
*/
void Dialog_Screen_WritePopulate(HWND hDlg)
{
	// Set global dialog handle
	ScreenhDlg = hDlg;
	// Allow one screen capture
	bCapturedScreen = FALSE;

	// Set items
	Dialog_SetButton(hDlg,IDC_RADIO_FULLSCREEN,DialogParams.Screen.bFullScreen);

	Dialog_SetComboBoxItems(hDlg,IDC_COMBO_DISPLAYMODES,pszAvailableDisplayModeTextNames,0);
	Dialog_ComboBoxSelectString(hDlg,IDC_COMBO_DISPLAYMODES,szDisplayModeTextNames[DialogParams.Screen.ChosenDisplayMode]);
	Dialog_SetButton(hDlg,IDC_RADIO_CAPTURECHANGE,DialogParams.Screen.bCaptureChange);
	sprintf(szFramesPerSecond,"%d",Dialog_Screen_LimitFramesPerSecond(DialogParams.Screen.nFramesPerSecond));
	Dialog_SetText(hDlg,IDC_EDIT_FRAMESPERSECOND,szFramesPerSecond);
	Dialog_SetRadioButtons(hDlg,IDC_RADIO_COLOUR,IDC_RADIO_MONO,DialogParams.Screen.bUseHighRes);
	
	Dialog_Screen_EnableItems(hDlg);

	bOKDialog = TRUE;
}

//-----------------------------------------------------------------------
/*
	Read settings back from dialog page
*/
void Dialog_Screen_ReadPopulate(HWND hDlg)
{
	// Read items
	DialogParams.Screen.bFullScreen = Dialog_ReadButton(hDlg,IDC_RADIO_FULLSCREEN);
	DialogParams.Screen.ChosenDisplayMode = DisplayModeOptions[Dialog_GetSelectedComboBoxItem(hDlg,IDC_COMBO_DISPLAYMODES)];
	DialogParams.Screen.bCaptureChange = Dialog_ReadButton(hDlg,IDC_RADIO_CAPTURECHANGE);
	Dialog_ReadText(hDlg,IDC_EDIT_FRAMESPERSECOND,szFramesPerSecond);
	DialogParams.Screen.nFramesPerSecond = Dialog_Screen_LimitFramesPerSecond(atoi(szFramesPerSecond));
	DialogParams.Screen.bUseHighRes = Dialog_ReadRadioButtons(hDlg,IDC_RADIO_COLOUR,IDC_RADIO_MONO);
}

//-----------------------------------------------------------------------
/*
	Capture single screen image
*/
void Dialog_Screen_Capture(HWND hDlg)
{
	ScreenSnapShot_SaveScreen();
	bCapturedScreen = TRUE;								// Grey out button so cannot choose again in this session!
	Dialog_Screen_EnableItems(hDlg);
}

//-----------------------------------------------------------------------
/*
	Start animation recording
*/
void Dialog_Screen_RecordAnimation(HWND hDlg)
{
	// Read details back so 'DialogParams' is up-to-date
	Dialog_Screen_ReadPopulate(hDlg);
	// Begin recording from here, inform user
	ScreenSnapShot_BeginRecording(hDlg,DialogParams.Screen.bCaptureChange,DialogParams.Screen.nFramesPerSecond);
	Dialog_Screen_EnableItems(hDlg);
}

//-----------------------------------------------------------------------
/*
	Stop animation recording
*/
void Dialog_Screen_StopAnimation(HWND hDlg)
{
	// Stop recording now, inform user
	ScreenSnapShot_EndRecording(hDlg);
	Dialog_Screen_EnableItems(hDlg);
}

//-----------------------------------------------------------------------
/*
	Screen Property page handler
*/
BOOL FAR PASCAL Dialog_Screen_DlgProc(HWND hDlg, UINT Message, UINT wParam, LONG lParam)
{
	switch(Message) {
		case WM_INITDIALOG:
			Dialog_Screen_WritePopulate(hDlg);
			nLastOpenPage = DIALOG_PAGE_SCREEN;
			return(TRUE);

		case WM_NOTIFY:
			switch(((NMHDR *)lParam)->code) {
				case PSN_APPLY:
					Dialog_Screen_ReadPopulate(hDlg);
					return(TRUE);
				case PSN_HELP:
					WinSTon_GoWeb("help\\options\\screen.html",FALSE);
					return(TRUE);
			}
			break;

		case WM_HELP:
			// Open help
			WinSTon_GoWeb("help\\options\\screen.html",FALSE);
			return(TRUE);

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
				case IDC_RADIO_FULLSCREEN:
					DialogParams.Screen.bFullScreen = Dialog_ReadButton(hDlg,IDC_RADIO_FULLSCREEN);
					Dialog_SetButton(Dialog_Configure_GetHandle(),IDC_RADIO_CONFIG_FULLSCREEN,DialogParams.Screen.bFullScreen);
					return(TRUE);
				case IDC_BUTTON_ADVANCED:
					Dialog_Screen_CreateAdvancedDialog(hDlg);
					return(TRUE);					
				case IDC_RADIO_CAPTURECHANGE:
					Dialog_Screen_EnableItems(hDlg);
					return(TRUE);
				case IDC_BUTTON_SCREEN_CAPTURE:
					Dialog_Screen_Capture(hDlg);
					return(TRUE);
				case IDC_BUTTON_SCREEN_RECORD:
					Dialog_Screen_RecordAnimation(hDlg);
					return(TRUE);
				case IDC_BUTTON_SCREEN_STOP:
					Dialog_Screen_StopAnimation(hDlg);
					return(TRUE);
			}
			break;
	}

	return(FALSE);	// Not Processed!
}

//-----------------------------------------------------------------------
/*
	Write settings into dialog page
*/
void Dialog_Screen_WriteAdvancePopulate(HWND hDlg)
{
	// Write details
	Dialog_SetButton(hDlg,IDC_RADIO_DOUBLESIZE,DialogParams.Screen.Advanced.bDoubleSizeWindow);
	Dialog_SetButton(hDlg,IDC_RADIO_ALLOWOVERSCAN,DialogParams.Screen.Advanced.bAllowOverscan);
	Dialog_SetButton(hDlg,IDC_RADIO_INTERLACED,DialogParams.Screen.Advanced.bInterlacedFullScreen);
	Dialog_SetButton(hDlg,IDC_RADIO_SYNCRETRACE,DialogParams.Screen.Advanced.bSyncToRetrace);
	Dialog_SetButton(hDlg,IDC_RADIO_FRAMESKIP,DialogParams.Screen.Advanced.bFrameSkip);

	// Disable items if no DirectX
	Dialog_EnableItems(hDlg,Screen_AdvancedFullScreen_DialogItems,bDirectDrawWorking);
}

//-----------------------------------------------------------------------
/*
	Read settings back from dialog page
*/
void Dialog_Screen_ReadAdvancePopulate(HWND hDlg)
{
	// Read items
	DialogParams.Screen.Advanced.bDoubleSizeWindow = Dialog_ReadButton(hDlg,IDC_RADIO_DOUBLESIZE);
	DialogParams.Screen.Advanced.bAllowOverscan = Dialog_ReadButton(hDlg,IDC_RADIO_ALLOWOVERSCAN);
	DialogParams.Screen.Advanced.bInterlacedFullScreen = Dialog_ReadButton(hDlg,IDC_RADIO_INTERLACED);
	DialogParams.Screen.Advanced.bSyncToRetrace = Dialog_ReadButton(hDlg,IDC_RADIO_SYNCRETRACE);
	DialogParams.Screen.Advanced.bFrameSkip = Dialog_ReadButton(hDlg,IDC_RADIO_FRAMESKIP);
}

//-----------------------------------------------------------------------
/*
	Screen Advanced dialog
*/
BOOL PASCAL ScreenAdvanceDlgProc(HWND hWnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
	switch(Message) {
		case WM_INITDIALOG:
			Dialog_Screen_WriteAdvancePopulate(hWnd);
			return(FALSE);

		case WM_HELP:
			// Open help
			WinSTon_GoWeb("help\\options\\screen.html",FALSE);
			return(TRUE);

		case WM_COMMAND:
			switch(wParam) {
				case IDOK:
					Dialog_Screen_ReadAdvancePopulate(hWnd);
					EndDialog(hWnd,0);
					return(TRUE);

				case IDCANCEL:
					EndDialog(hWnd,0);
					return(TRUE);
				}
			break;
	}

	return(FALSE);
}

//-----------------------------------------------------------------------
/*
	Create dialog for Screen Advanced
*/
void Dialog_Screen_CreateAdvancedDialog(HWND hWnd)
{
	PROC lpfnDlgProc;

	// Open dialog
	lpfnDlgProc = MakeProcInstance((PROC)ScreenAdvanceDlgProc,hInst);
	DialogBoxParam(hInst,MAKEINTRESOURCE(IDD_DIALOG_SCREENADVANCED),hWnd,(DLGPROC)lpfnDlgProc,NULL);
}
