
#ifndef MEMORYBUS_H
#define MEMORYBUS_H

#include <stdio.h>
#include <string.h>
#include "Memory.h"

#define MAX_MAPPED_MEMORIES 50
#define MAX_OVERLAPPED_MEMORIES 4

/**
 * Emulates a memory bus which may be composed of memories of the same
 * size or narrower.
 *
 * @author Kyle Davis
 */
template <typename A, typename W>
class MemoryBus
{

    public:
        MemoryBus();
        ~MemoryBus();
        void reset();
        void addMemory(Memory<A, W>* m);
        void removeMemory(Memory<A, W>* m);
        W peek(A location);
        void poke(A location, W value);

    private:
        Memory<A, W>*   mappedMemories[MAX_MAPPED_MEMORIES];
        W               mappedMemoryCount;
        W*              writeableMemoryCounts;
        Memory<A, W>*** writeableMemorySpace;
        W*              readableMemoryCounts;
        Memory<A, W>*** readableMemorySpace;
        W               unmappedPeek;

};

template <typename A, typename W>
MemoryBus<A, W>::MemoryBus()
{
    UINT32 size = 1 << (sizeof(A) << 3);
    UINT64 i;
    writeableMemoryCounts = new W[size];
    memset(writeableMemoryCounts, 0, sizeof(W) * size);
    writeableMemorySpace = new Memory<A, W>**[size];
    for (i = 0; i < size; i++)
        writeableMemorySpace[i] = new Memory<A, W>*[MAX_OVERLAPPED_MEMORIES];
    readableMemoryCounts = new W[size];
    memset(readableMemoryCounts, 0, sizeof(W) * size);
    readableMemorySpace = new Memory<A, W>**[size];
    for (i = 0; i < size; i++)
        readableMemorySpace[i] = new Memory<A, W>*[MAX_OVERLAPPED_MEMORIES];
    mappedMemoryCount = 0;
    unmappedPeek = (1 << (sizeof(W) << 3)) - 1;
}

template <typename A, typename W>
MemoryBus<A, W>::~MemoryBus()
{
    UINT64 size = 1 << (sizeof(A) << 3);
    UINT64 i;
    delete[] writeableMemoryCounts;
    for (i = 0; i < size; i++)
        delete[] writeableMemorySpace[i];
    delete[] writeableMemorySpace;
    delete[] readableMemoryCounts;
    for (i = 0; i < size; i++)
        delete[] readableMemorySpace[i];
    delete[] readableMemorySpace;
}

template <typename A, typename W>
void MemoryBus<A, W>::reset()
{
    for (A i = 0; i < mappedMemoryCount; i++)
        mappedMemories[i]->reset();
}

template <typename A, typename W>
void MemoryBus<A, W>::addMemory(Memory<A, W>* m)
{
    //get the important info
    A size = m->getSize();

    A numLocations = m->getNumLocations();
    for (A i = 0; i < numLocations; i++) {
        A nextLocation = m->getLocation(i);
        A nextEnd = nextLocation + size - 1;
//fprintf(stderr, "adding memory to range: %04X  %04X  %04X\n", nextLocation, size, nextEnd);

        if (m->isLocationReadable(i)) {
            for (UINT64 j = nextLocation; j <= nextEnd; j++) {
                A memCount = readableMemoryCounts[j];
                readableMemorySpace[j][memCount] = m;
                readableMemoryCounts[j]++;
            }
        }

        if (m->isLocationWriteable(i)) {
            for (UINT64 j = nextLocation; j <= nextEnd; j++) {
                A memCount = writeableMemoryCounts[j];
                writeableMemorySpace[j][memCount] = m;
                writeableMemoryCounts[j]++;
                
            }
        }
    }
}

template <typename A, typename W>
void MemoryBus<A, W>::removeMemory(Memory<A, W>* m)
{
    //get the important info
    A size = m->getSize();
    A i,l;
    UINT64 j;

    A numLocations = m->getNumLocations();
    for (i = 0; i < numLocations; i++) {
        A nextLocation = m->getLocation(i);
        A nextEnd = nextLocation + size - 1;

        if (m->isLocationReadable(i)) {
            for (j = nextLocation; j <= nextEnd; j++) {
                A memCount = readableMemoryCounts[j];
                for (A k = 0; k < memCount; k++) {
                    if (readableMemorySpace[j][k] == m) {
                        for (l = k; l < (memCount-1); l++) {
                            readableMemorySpace[j][l] = 
                                readableMemorySpace[j][l+1];
                        }
                        readableMemorySpace[j][memCount-1] = NULL;
                        readableMemoryCounts[j]--;
                        break;
                    }
                }
            }
        }

        if (m->isLocationWriteable(i)) {
            for (j = nextLocation; j <= nextEnd; j++) {
                A memCount = writeableMemoryCounts[j];
                for (A k = 0; k < memCount; k++) {
                    if (writeableMemorySpace[j][k] == m) {
                        for (l = k; l < (memCount-1); l++) {
                            writeableMemorySpace[j][l] = 
                                writeableMemorySpace[j][l+1];
                        }
                        writeableMemorySpace[j][memCount-1] = NULL;
                        writeableMemoryCounts[j]--;
                        break;
                    }
                }
            }
        }
    }

    //remove it from our list of memories
    for (i = 0; i < mappedMemoryCount; i++) {
        if (mappedMemories[i] == m) {
            for (j = i; j < (mappedMemoryCount-1); j++)
                mappedMemories[j] = mappedMemories[j+1];
            mappedMemoryCount--;
        }
    }
}

template <typename A, typename W>
W MemoryBus<A, W>::peek(A location)
{
    A numMemories = readableMemoryCounts[location];

    W value = unmappedPeek;
    for (A i = 0; i < numMemories; i++)
        value &= readableMemorySpace[location][i]->peek(location);

    return value;
}

template <typename A, typename W>
void MemoryBus<A, W>::poke(A location, W value)
{
    A numMemories = readableMemoryCounts[location];

    for (A i = 0; i < numMemories; i++)
        readableMemorySpace[location][i]->poke(location, value);
}

typedef MemoryBus<UINT16, UINT16> MemoryBus16Bit;
typedef MemoryBus<UINT16, UINT8> MemoryBus8Bit;

#endif
