/*
NEStopia / Linux
Port by R. Belmont

main.cpp - main file
*/

#include <xtl.h>
#include <iostream>
#include <fstream>
#include <strstream>
#include <sstream>
#include <iomanip>
#include <string.h>
#include <cassert>
#include <sys\stat.h>
#include <sys\types.h>
#include <errno.h>
#include <vector>
#include <direct.h>
#include <stdlib.h>
#include <time.h>

#include "..\core\api\NstApiEmulator.hpp"
#include "..\core\api\NstApiVideo.hpp"
#include "..\core\api\NstApiSound.hpp"
#include "..\core\api\NstApiInput.hpp"
#include "..\core\api\NstApiMachine.hpp"
#include "..\core\api\NstApiUser.hpp"
#include "..\core\api\NstApiNsf.hpp"
#include "..\core\api\NstApiMovie.hpp"
#include "..\core\api\NstApiFds.hpp"
#include "..\core\api\NstApiRewinder.hpp"
#include "..\core\api\NstApiCartridge.hpp"
#include "..\core\api\NstApiCheats.hpp"
#include "..\core\NstCrc32.hpp"
#include "..\core\NstChecksum.hpp"
#include "..\core\NstXml.hpp"
#include "oss.h"
#include "settings.h"
#include "auxio.h"
#include "input.h"
#include "controlconfig.h"
#include "cheats.h"
#include "seffect.h"
#include "main.h"
#include "GlobalExtern.h"


#define NST_VERSION "1.4.0 [release H]"

using namespace Nes::Api;
using namespace LinuxNst;

// base class, all interfaces derives from this
Emulator emulator;

// forward declaration
void SetupVideo();
void SetupSound();
void SetupInput();
void SetEmulationVolume(int vol);

static short lbuf[48000];
long exholding[48000*2];

static unsigned short keys[65536];

int updateok, playing = 0, cur_width, cur_Rwidth, cur_height, cur_Rheight, loaded = 0, framerate;
int nst_quit = 1, nsf_mode = 0, state_save = 0, state_load = 0, movie_save = 0, movie_load = 0, movie_stop = 0;
static int schedule_stop = 0;
short soundBuffer[0x8000];

static char savename[2048], capname[2048], gamebasename[2048];
static char caption[2048];
char rootname[2048], lastarchname[2048];
static InputDefT *ctl_defs;
static Video::Output *cNstVideo;
static Sound::Output *cNstSound;
static Input::Controllers *cNstPads;
static Cartridge::Database::Entry dbentry;
static bool         using_opengl  = false;
static bool         linear_filter  = false;
static int          gl_w, gl_h;

static void         *intbuffer;	// intermediate buffer: the NST engine draws into this, and we may blit it
// either as-is or in other ways
int bufFrameSize = 0;
int	bufSize      = 0;
int	bufInPos     = 0;
int	bufOutPos    = 0;
int	bufUsed      = 0;
int sampleRate = 44100;
extern IDirect3DTexture8 *TexScreen;
int nFramesEmulated = 0;
int nCurrentFrame = 0;
int nFramesRendered = 0;
unsigned int nDoFPS = 0;
float fps = 0;
extern int DxSoundCheck();
extern int DxSoundInit();
extern int Region;
extern int UnlimitedSprites;
int ZapperX = 1;
int ZapperY = 1;
bool ZapperEnabled = false;
extern int CursorSpeed;
extern void DeleteCheats();
extern int DxSoundNESExit();
extern void PrintFreeMemory(char* n);
extern void InvalidFile(char* msg);
bool ErrorFile = false;
extern long nDSoundVol;
extern void DxSoundSetVolume();

// get the favored system selected by the user
static Machine::FavoredSystem get_favored_system(void)
{
    return Machine::FAVORED_NES_NTSC;
}

// convert a number into the next highest power of 2
static int powerOfTwo( const int value )
{
    int result = 1;
    while ( result < value )
        result <<= 1;
    return result;	
}

// *******************
// emulation callbacks
// *******************
unsigned char *nintendo;
int nesPitch = 512;
D3DLOCKED_RECT lock;
D3DSURFACE_DESC desc;
DWORD dwTexWidth  = 0;
DWORD dwTexHeight = 0;

extern int ColorMode;
extern vector<string> PaletteList;
extern int PaletteSelected;

void SetCustom(char* palette) {
    unsigned char colors[64][3];
    unsigned char custom[64][3];

    FILE *f = fopen(palette, "rb");

    if(f) {
        for(int i = 0; i < 64; i++) {
            fread(&colors[i][0], 1, 1, f);
            fread(&colors[i][1], 1, 1, f);
            fread(&colors[i][2], 1, 1, f);
        }

        fclose(f);
        std::memcpy(custom, colors, 64 * 3);
        Video video(emulator);
        if(NES_FAILED(video.GetPalette().SetCustom( custom )))
            video.GetPalette().SetMode(Nes::Api::Video::Palette::MODE_YUV);
        else
            video.GetPalette().SetMode(Nes::Api::Video::Palette::MODE_CUSTOM);
    }	
}

void SetYUV() {
    Video video(emulator);
    video.GetPalette().SetMode(Nes::Api::Video::Palette::MODE_YUV);
}

void SetRGB() {
    Video video(emulator);
    video.GetPalette().SetMode(Nes::Api::Video::Palette::MODE_RGB);
}

void UpdatePalette() {
    if(ColorMode == 0)
        SetYUV();
    if(ColorMode == 1)
        SetRGB();
    if(ColorMode == 2) {
        if(PaletteSelected == -1) {
            SetYUV();
        }
        else {
            char c[2048];
            sprintf(c, "D:\\palettes\\%s.pal", PaletteList[PaletteSelected].c_str());
            SetCustom(c);
        }
    }
}

// called right before Nestopia is about to write pixels
static bool NST_CALLBACK VideoLock(void* userData, Video::Output& video)
{
    if(SoftwareFilter < 8) {
        TexScreen->GetLevelDesc( 0, &desc );
        dwTexWidth  = desc.Width;
        dwTexHeight = desc.Height;
        TexScreen->LockRect( 0, &lock, 0, 0L );
        video.pixels = (unsigned int*)lock.pBits;
        video.pitch = lock.Pitch;
    }
    else {
        video.pixels = (unsigned char*)nintendo;
        video.pitch = nesPitch;
    }

    return true; // true=lock success, false=lock failed (Nestopia will carry on but skip video)
}

extern int scalefactor;

// called right after Nestopia has finished writing pixels (not called if previous lock failed)
static void NST_CALLBACK VideoUnlock(void* userData, Video::Output& video)
{
    if(SoftwareFilter < 8)
        TexScreen->UnlockRect(0);
    DirectScreen();
}

// do a "partial" shutdown
static void nst_unload2(void)
{
    Machine machine(emulator);

    // power down the emulated NES
    machine.Power(false);

    // unload the cart
    machine.Unload();

    // erase any cheats
    Cheats cheats(emulator);
    cheats.ClearCodes();
}

static void nst_unload(void)
{
    nst_unload2();
}

// returns if we're currently playing a game or NSF
bool NstIsPlaying()
{
    return playing;
}

extern void SoundExit();

void StopGame() {
    Machine machine(emulator);



    auxio_do_movie_stop();

    if (nintendo)
    {
        free(nintendo);
        nintendo = NULL;
    }

    memset(soundBuffer, 0, sizeof(soundBuffer));

    playing = 0;

    machine.Power(false);

    machine.Unload();

    auxio_shutdown();
    DxSoundNESExit();
}

// shuts everything down
void NstStopPlaying() {
    Nes::Api::Cartridge::Database database( emulator );
    database.Unload();

    auxio_do_movie_stop();

    if (nintendo) {
        free(nintendo);
        nintendo = NULL;
    }

    Machine machine(emulator);

    // power down the emulated NES
    machine.Power(false);
    machine.Unload();

    memset(soundBuffer, 0, sizeof(soundBuffer));

    Cheats cheats(emulator);
    cheats.ClearCodes();
    playing = 0;
    auxio_shutdown();
    DxSoundNESExit();
}

#define CRg(rg) (sizeof(rg) / sizeof(rg[0]))
std::string svst[2];

// generate the filename for quicksave files
std::string StrQuickSaveFile(int isvst)
{
    std::ostringstream ossFile;
    ossFile << "config\\qsave\\Punchout.nst";
    mkdir("config\\qsave");

    return ossFile.str();
}

// save state to memory slot
static void QuickSave(int isvst)
{
    std::string strFile = StrQuickSaveFile(isvst);
    if (strFile.empty())
        return;

    Machine machine( emulator );
    std::ofstream os(strFile.c_str());
    // use "NO_COMPRESSION" to make it easier to hack save states
    Nes::Result res = machine.SaveState(os, Nes::Api::Machine::USE_COMPRESSION);
}


// restore state from memory slot
static void QuickLoad(int isvst)
{
    std::string strFile = StrQuickSaveFile(isvst);
    if (strFile.empty())
        return;

    Machine machine( emulator );
    std::ifstream is(strFile.c_str());
    Nes::Result res = machine.LoadState(is);
}


// start playing
void NstPlayGame(void)
{
    // apply any cheats into the engine
    //sCheatMgr->Enable();
    updateok = 0;
}

// start playing an NSF file
void NstPlayNsf(void)
{
    Nsf nsf( emulator );

    nsf.PlaySong();
}

// stop playing an NSF file
void NstStopNsf(void)
{
    Nsf nsf( emulator );

    nsf.StopSong();

    // clear the audio buffer
    memset(lbuf, 0, sizeof(lbuf));
}

// schedule a NEStopia quit
void NstScheduleQuit(void)
{
    nst_quit = 1;
}

void SetSprites() {
    Video video(emulator);
    if(UnlimitedSprites)
        video.EnableUnlimSprites(true);
    else
        video.EnableUnlimSprites(false);
}

void ResetSystem() {
    Machine machine( emulator );
    Fds fds( emulator );

    if(ResetType) {
        machine.Reset(false);
    }
    else {
        machine.Reset(true);
    }
    // put the disk system back to disk 0 side 0
    fds.EjectDisk();
    fds.InsertDisk(0, 0);
}

void FlipFDSDisk() {
    Fds fds( emulator );

    if (fds.CanChangeDiskSide()) {
        fds.ChangeSide();
    }
}

void SetDisk(int disk, int side) {
    Fds fds( emulator );
    fds.InsertDisk(disk, side);
}

void SetRewindSound() {
    if(RewindSound) {
        Rewinder(emulator).EnableSound(true);
    }
    else {
        Rewinder(emulator).EnableSound(false);
    }
}

void EnableRewind() {
    if (NES_SUCCEEDED(Rewinder(emulator).Enable(true))) {
        if(RewindSound) {
            Rewinder(emulator).EnableSound(true);
        }
        else {
            Rewinder(emulator).EnableSound(false);
        }
    }
}

void DisableRewind() {
    Rewinder(emulator).Enable(false);
    Rewinder(emulator).EnableSound(false);
}

void SetRewinderBackwards() {
    Rewinder(emulator).SetDirection(Rewinder::BACKWARD);
}

void SetRewinderForwards() {
    Rewinder(emulator).SetDirection(Rewinder::FORWARD);
}

void VSSystemAddCoin1() {
    cNstPads->vsSystem.insertCoin |= Input::Controllers::VsSystem::COIN_1;
}

void VSSystemAddCoin2() {
    cNstPads->vsSystem.insertCoin |= Input::Controllers::VsSystem::COIN_2;	
}

void SaveState(char* file) {
    auxio_do_state_save(file);
}

void LoadState(char* file) {
    auxio_do_state_load(file);
}

void MovieLoad() {
    auxio_do_movie_load();
}

void MovieSave() {
    auxio_do_movie_save();
}

void MovieStop() {	
    auxio_do_movie_stop();
}

static void NST_CALLBACK DoLog(void *userData, const char *string, Nes::ulong length)
{
    dprintf("%s\n", string);
}

// for various file operations, usually called during image file load, power on/off and reset
static void NST_CALLBACK DoFileIO(void *userData, User::File& file)
{
    unsigned char *compbuffer;
    int compsize, compoffset;
    char mbstr[512];

    switch (file.GetAction())
    {
    case User::File::LOAD_ROM:
        wcstombs(mbstr, file.GetName(), 511);

        if (auxio_load_archive(lastarchname, &compbuffer, &compsize, &compoffset, (const char *)mbstr))
        {
            file.SetContent((const void*)&compbuffer[compoffset], (Nes::ulong)compsize);

            free(compbuffer);
        }				
        break;

    case User::File::LOAD_SAMPLE:
    case User::File::LOAD_SAMPLE_MOERO_PRO_YAKYUU:
    case User::File::LOAD_SAMPLE_MOERO_PRO_YAKYUU_88:
    case User::File::LOAD_SAMPLE_MOERO_PRO_TENNIS:
    case User::File::LOAD_SAMPLE_TERAO_NO_DOSUKOI_OOZUMOU:
    case User::File::LOAD_SAMPLE_AEROBICS_STUDIO:
        wcstombs(mbstr, file.GetName(), 511);

        if (auxio_load_archive(lastarchname, &compbuffer, &compsize, &compoffset, (const char *)mbstr))
        {
            int chan, bits, rate;

            if (!strncmp((const char *)compbuffer, "RIFF", 4))
            {
                chan = compbuffer[20] | compbuffer[21]<<8;
                rate = compbuffer[24] | compbuffer[25]<<8 | compbuffer[26]<<16 | compbuffer[27]<<24;
                bits = compbuffer[34] | compbuffer[35]<<8; 

                //					std::cout << "WAV has " << chan << " chan, " << bits << " bits per sample, rate = " << rate << "\n";

                file.SetSampleContent((const void*)&compbuffer[compoffset], (Nes::ulong)compsize, (chan == 2) ? true : false, bits, rate);
            }

            free(compbuffer);
        }				
        break;

    case User::File::LOAD_BATTERY: // load in battery data from a file
    case User::File::LOAD_EEPROM: // used by some Bandai games, can be treated the same as battery files
    case User::File::LOAD_TAPE: // for loading Famicom cassette tapes
    case User::File::LOAD_TURBOFILE: // for loading turbofile data
        {
            int size;
            FILE *f;

            f = fopen(savename, "rb");
            if (!f)
            {
                return;
            }
            fseek(f, 0, SEEK_END);
            size = ftell(f);
            fclose(f);

            std::ifstream batteryFile( savename, std::ifstream::in|std::ifstream::binary );

            if (batteryFile.is_open())
            {
                file.SetContent( batteryFile );
            }
            break;
        }

    case User::File::SAVE_BATTERY: // save battery data to a file
    case User::File::SAVE_EEPROM: // can be treated the same as battery files
    case User::File::SAVE_TAPE: // for saving Famicom cassette tapes
    case User::File::SAVE_TURBOFILE: // for saving turbofile data
        {
            std::ofstream batteryFile( savename, std::ifstream::out|std::ifstream::binary );
            const void* savedata;
            unsigned long savedatasize;

            file.GetContent( savedata, savedatasize );

            if (batteryFile.is_open())
                batteryFile.write( (const char*) savedata, savedatasize );

            break;
        }

    case User::File::LOAD_FDS: // for loading modified Famicom Disk System files
        {
            char fdsname[2048];

            sprintf(fdsname, "%s\\%s.ups", SRAMPath.c_str(), globalGameName.c_str());

            std::ifstream batteryFile( fdsname, std::ifstream::in|std::ifstream::binary );

            // no ups, look for ips
            if (!batteryFile.is_open())
            {
                sprintf(fdsname, "%s\\%s.ips", SRAMPath.c_str(), globalGameName.c_str());

                std::ifstream batteryFile( fdsname, std::ifstream::in|std::ifstream::binary );

                if (!batteryFile.is_open())
                {
                    return;
                }

                file.SetPatchContent(batteryFile);
                return;
            }

            file.SetPatchContent(batteryFile);
            break;
        }

    case User::File::SAVE_FDS: // for saving modified Famicom Disk System files
        {
            char fdsname[2048];

            sprintf(fdsname, "%s\\%s.ups", SRAMPath.c_str(), globalGameName.c_str());

            std::ofstream fdsFile( fdsname, std::ifstream::out|std::ifstream::binary );

            if (fdsFile.is_open())
                file.GetPatchContent( User::File::PATCH_UPS, fdsFile );

            break;
        }
    }
}

// called right before Nestopia is about to write sound samples
static bool NST_CALLBACK SoundLock(void* userData,Nes::Api::Sound::Output& sound)
{
    return true;
}

static void NST_CALLBACK SoundUnlock(void* userData,Nes::Api::Sound::Output& sound)
{
    cNstSound->samples[0] = &soundBuffer[0];
}
unsigned int GetNESCRC() {
    unsigned int crc = Nes::Api::Cartridge(emulator).GetProfile()->hash.GetCrc32();
    return crc;
}

unsigned int turboATemp = 1;
unsigned int turboBTemp = 1;
unsigned int turboSelectTemp = 1;
unsigned int turboStartTemp = 1;
int turboACount = 0;
int turboBCount = 0;
int turboSelectCount = 0;
int turboStartCount = 0;
bool frameskip = false;

void CheckControllerDefault() {
    if (g_Gamepads[DefaultController].fY1 > 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_UP) 
        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::UP;
    if (g_Gamepads[DefaultController].fY1 < 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_DOWN)
        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::DOWN;
    if (g_Gamepads[DefaultController].fX1 < 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_LEFT)
        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::LEFT;
    if (g_Gamepads[DefaultController].fX1 > 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_RIGHT)
        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::RIGHT;

    if(NESControllers[0].Select == 8 || NESControllers[0].Select == 9) {
        if(NESControllers[0].Select == 8) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[0].turboSelect) {
                    turboSelectCount++;
                    if(turboSelectCount > 100)
                        turboSelectCount = 0;
                    if((turboSelectCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
        if(NESControllers[0].Select == 9) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[0].turboSelect) {
                    turboSelectCount++;
                    if(turboSelectCount > 100)
                        turboSelectCount = 0;
                    if((turboSelectCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
    }
    else {
        if(g_Gamepads[DefaultController].bAnalogButtons[NESControllers[0].Select]) {
            if(tButtons[0].turboSelect) {
                turboSelectCount++;
                if(turboSelectCount > 100)
                    turboSelectCount = 0;
                if((turboSelectCount % turboSpeed) == 0) {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
            else {
                cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
            }
        }
    }		

    if(NESControllers[0].Start == 8 || NESControllers[0].Start == 9) {
        if(NESControllers[0].Start == 8) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[0].turboStart) {
                    turboStartCount++;
                    if(turboStartCount > 100)
                        turboStartCount = 0;
                    if((turboStartCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
        if(NESControllers[0].Start == 9) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[0].turboStart) {
                    turboStartCount++;
                    if(turboStartCount > 100)
                        turboStartCount = 0;
                    if((turboStartCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::START;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::START;
                }
            }
        }
    }
    else {
        if(g_Gamepads[DefaultController].bAnalogButtons[NESControllers[0].Start]) {
            if(tButtons[0].turboStart) {
                turboStartCount++;
                if(turboStartCount > 100)
                    turboStartCount = 0;
                if((turboStartCount % turboSpeed) == 0) {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::START;
                }
            }
            else {
                cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::START;
            }
        }
    }		

    if(NESControllers[0].A == 8 || NESControllers[0].A == 9) {
        if(NESControllers[0].A == 8) {	
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[0].turboA) {
                    turboACount++;
                    if(turboACount > 100)
                        turboACount = 0;
                    if((turboACount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
        }
        if(NESControllers[0].A == 9) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[0].turboA) {
                    turboACount++;
                    if(turboACount > 100)
                        turboACount = 0;
                    if((turboACount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
        }
    }
    else {
        if(g_Gamepads[DefaultController].bAnalogButtons[NESControllers[0].A]) {
            if(tButtons[0].turboA) {
                turboACount++;
                if(turboACount > 100)
                    turboACount = 0;
                if((turboACount % turboSpeed) == 0) {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
            else {
                cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::A;
            }
        }
    }

    if(NESControllers[0].B == 8 || NESControllers[0].B == 9) {
        if(NESControllers[0].B == 8) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[0].turboB) {
                    turboBCount++;
                    if(turboBCount > 100)
                        turboBCount = 0;
                    if((turboBCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
        }
        if(NESControllers[0].B == 9) {
            if(g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[0].turboB) {
                    turboBCount++;
                    if(turboBCount > 100)
                        turboBCount = 0;
                    if((turboBCount % turboSpeed) == 0) {
                        cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
                    }
                }
                else {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
        }
    }
    else {
        if(g_Gamepads[DefaultController].bAnalogButtons[NESControllers[0].B]) {
            if(tButtons[0].turboB) {
                turboBCount++;
                if(turboBCount > 100)
                    turboBCount = 0;
                if((turboBCount % turboSpeed) == 0) {
                    cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
            else {
                cNstPads->pad[0].buttons |= Nes::Core::Input::Controllers::Pad::B;
            }
        }
    }		
}

void CheckController(int i) {
    if (g_Gamepads[i].fY1 > 0 || g_Gamepads[i].wButtons & XINPUT_GAMEPAD_DPAD_UP) 
        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::UP;
    if (g_Gamepads[i].fY1 < 0 || g_Gamepads[i].wButtons & XINPUT_GAMEPAD_DPAD_DOWN)
        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::DOWN;
    if (g_Gamepads[i].fX1 < 0 || g_Gamepads[i].wButtons & XINPUT_GAMEPAD_DPAD_LEFT)
        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::LEFT;
    if (g_Gamepads[i].fX1 > 0 || g_Gamepads[i].wButtons & XINPUT_GAMEPAD_DPAD_RIGHT)
        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::RIGHT;

    if(NESControllers[i].Select == 8 || NESControllers[i].Select == 9) {
        if(NESControllers[i].Select == 8) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[i].turboSelect) {
                    turboSelectCount++;
                    if(turboSelectCount > 100)
                        turboSelectCount = 0;
                    if((turboSelectCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
        if(NESControllers[i].Select == 9) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[i].turboSelect) {
                    turboSelectCount++;
                    if(turboSelectCount > 100)
                        turboSelectCount = 0;
                    if((turboSelectCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
    }
    else {
        if(g_Gamepads[i].bAnalogButtons[NESControllers[i].Select]) {
            if(tButtons[i].turboSelect) {
                turboSelectCount++;
                if(turboSelectCount > 100)
                    turboSelectCount = 0;
                if((turboSelectCount % turboSpeed) == 0) {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
            else {
                cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
            }
        }
    }		

    if(NESControllers[i].Start == 8 || NESControllers[i].Start == 9) {
        if(NESControllers[i].Start == 8) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[i].turboStart) {
                    turboStartCount++;
                    if(turboStartCount > 100)
                        turboStartCount = 0;
                    if((turboStartCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::SELECT;
                }
            }
        }
        if(NESControllers[i].Start == 9) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[i].turboStart) {
                    turboStartCount++;
                    if(turboStartCount > 100)
                        turboStartCount = 0;
                    if((turboStartCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::START;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::START;
                }
            }
        }
    }
    else {
        if(g_Gamepads[i].bAnalogButtons[NESControllers[i].Start]) {
            if(tButtons[i].turboStart) {
                turboStartCount++;
                if(turboStartCount > 100)
                    turboStartCount = 0;
                if((turboStartCount % turboSpeed) == 0) {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::START;
                }
            }
            else {
                cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::START;
            }
        }
    }		

    if(NESControllers[i].A == 8 || NESControllers[i].A == 9) {
        if(NESControllers[i].A == 8) {	
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[i].turboA) {
                    turboACount++;
                    if(turboACount > 100)
                        turboACount = 0;
                    if((turboACount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
        }
        if(NESControllers[i].A == 9) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[i].turboA) {
                    turboACount++;
                    if(turboACount > 100)
                        turboACount = 0;
                    if((turboACount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
        }
    }
    else {
        if(g_Gamepads[i].bAnalogButtons[NESControllers[i].A]) {
            if(tButtons[i].turboA) {
                turboACount++;
                if(turboACount > 100)
                    turboACount = 0;
                if((turboACount % turboSpeed) == 0) {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
                }
            }
            else {
                cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::A;
            }
        }
    }

    if(NESControllers[i].B == 8 || NESControllers[i].B == 9) {
        if(NESControllers[i].B == 8) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_BACK) {
                if(tButtons[i].turboB) {
                    turboBCount++;
                    if(turboBCount > 100)
                        turboBCount = 0;
                    if((turboBCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
        }
        if(NESControllers[i].B == 9) {
            if(g_Gamepads[i].wButtons & XINPUT_GAMEPAD_START) {
                if(tButtons[i].turboB) {
                    turboBCount++;
                    if(turboBCount > 100)
                        turboBCount = 0;
                    if((turboBCount % turboSpeed) == 0) {
                        cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
                    }
                }
                else {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
        }
    }
    else {
        if(g_Gamepads[i].bAnalogButtons[NESControllers[i].B]) {
            if(tButtons[i].turboB) {
                turboBCount++;
                if(turboBCount > 100)
                    turboBCount = 0;
                if((turboBCount % turboSpeed) == 0) {
                    cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
                }
            }
            else {
                cNstPads->pad[i].buttons |= Nes::Core::Input::Controllers::Pad::B;
            }
        }
    }		
}
void CheckInput() {
    ReadInput();

    if(PauseGame == 0) {
        if (g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_RIGHT_THUMB) {
            exitGameMenu = false;
            DxSoundStop();
            GameOptionMenu();
        }
    }

    if(PauseGame == 1) {
        if (g_Gamepads[DefaultController].bAnalogButtons[XINPUT_GAMEPAD_LEFT_TRIGGER] &&
            g_Gamepads[DefaultController].bAnalogButtons[XINPUT_GAMEPAD_RIGHT_TRIGGER]) {
                exitGameMenu = false;
                DxSoundStop();
                GameOptionMenu();
        }
    }

    if(PauseGame == 2) {
        if (g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_START &&
            g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                exitGameMenu = false;
                DxSoundStop();
                GameOptionMenu();
        }
    }

    if(PauseGame == 3) {
        if (g_Gamepads[DefaultController].bAnalogButtons[XINPUT_GAMEPAD_LEFT_TRIGGER] &&
            g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                exitGameMenu = false;
                DxSoundStop();
                GameOptionMenu();
        }
    }

    if(PauseGame == 4) {
        if (g_Gamepads[DefaultController].bAnalogButtons[XINPUT_GAMEPAD_RIGHT_TRIGGER] &&
            g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_BACK) {
                exitGameMenu = false;
                DxSoundStop();
                GameOptionMenu();
        }
    }

    if (g_Gamepads[DefaultController].fY2 > 0)  {
        nDSoundVol+=10;
        DxSoundSetVolume();
    }  

    if (g_Gamepads[DefaultController].fY2 < 0)  {
        nDSoundVol-=10;
        DxSoundSetVolume();
    }  

    if(Input(emulator).GetConnectedController(0) == Input::PAD1) {
        CheckControllerDefault();
    }
    if(Input(emulator).GetConnectedController(1) == Input::PAD2) {
        CheckController(1);
    }
    if(Input(emulator).GetConnectedController(2) == Input::PAD3) {
        CheckController(2);
    }
    if(Input(emulator).GetConnectedController(3) == Input::PAD4) {
        CheckController(3);
    }

    if(Input(emulator).GetConnectedController(0) == Input::ZAPPER || Input(emulator).GetConnectedController(1) == Input::ZAPPER) {
        int w = Video::Output::WIDTH;
        if (g_Gamepads[DefaultController].fY1 > 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_UP) {
            ZapperY-=CursorSpeed;
            if(ZapperY < 1)
                ZapperY = 1;
            cNstPads->zapper.y = ZapperY;			
        }
        if (g_Gamepads[DefaultController].fY1 < 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_DOWN) {
            ZapperY+=CursorSpeed;
            if(ZapperY > 240)
                ZapperY = 240;
            cNstPads->zapper.y = ZapperY;	
        }
        if (g_Gamepads[DefaultController].fX1 < 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_LEFT) {
            ZapperX-=CursorSpeed;
            if(ZapperX < 1)
                ZapperX = 1;
            cNstPads->zapper.x = ZapperX;	
        }
        if (g_Gamepads[DefaultController].fX1 > 0 || g_Gamepads[DefaultController].wButtons & XINPUT_GAMEPAD_DPAD_RIGHT) {
            ZapperX+=CursorSpeed;
            if(ZapperX > 256)
                ZapperX = 256;
            cNstPads->zapper.x = ZapperX;	
        }

        if(g_Gamepads[DefaultController].bAnalogButtons[NESControllers[0].A]) {
            cNstPads->zapper.fire = 1;
        }
    }

    if(FastForward != -1) {
        if(g_Gamepads[DefaultController].bAnalogButtons[FastForward]) { 
            Nes::Api::Sound sound( emulator );
            emulator.Execute(NULL, NULL, NULL);
            emulator.Execute(NULL, NULL, NULL);
        }
    }

    if(Rewind != -1) {
        if(g_Gamepads[DefaultController].bAnalogButtons[Rewind]) { 
            Rewinder(emulator).SetDirection(Rewinder::BACKWARD);
        }
        else {
            Rewinder(emulator).SetDirection(Rewinder::FORWARD);
        }
    }

    if(VSCoin1 != -1) {
        if(g_Gamepads[DefaultController].bAnalogButtons[VSCoin1]) { 
            cNstPads->vsSystem.insertCoin |= Input::Controllers::VsSystem::COIN_1;
        }
    }

    if(VSCoin2 != -1) {
        if(g_Gamepads[DefaultController].bAnalogButtons[VSCoin2]) { 
            cNstPads->vsSystem.insertCoin = Input::Controllers::VsSystem::COIN_2;
        }
    }
}

void ClearInput() {
    cNstPads->pad[0].buttons = 0;
    cNstPads->pad[1].buttons = 0;
    cNstPads->pad[2].buttons = 0;
    cNstPads->pad[3].buttons = 0;
    cNstPads->zapper.fire = 0;
    cNstPads->vsSystem.insertCoin = 0;
}

LARGE_INTEGER emutickDelay;
float emulastTimeDelay = 0;
LARGE_INTEGER emuticksPerSecondDelay;

int emuDelayValid() {
    QueryPerformanceCounter(&emutickDelay);
    if((((FLOAT)emutickDelay.QuadPart) - emulastTimeDelay) >=
        (((FLOAT)emuticksPerSecondDelay.QuadPart) / 60.0f)) {
            emulastTimeDelay = (FLOAT)emutickDelay.QuadPart;
            return 1;
    }
    else {
        return 0;
    }
}

static void CalculateFPS() {
    static time_t fpstimer;
    static unsigned int nPreviousFrames;
    time_t temptime = clock();
    fps = (float)(nFramesRendered - nPreviousFrames) * CLOCKS_PER_SEC / (temptime - fpstimer);
    fpstimer = temptime;
    nPreviousFrames = nFramesRendered;
    //std::cout << "FPS: " << fps << std::endl;
}

void NesRun(bool sound) {
    nFramesEmulated++;
    nCurrentFrame++;
    CheckInput();
    if(!nst_quit) {
        if(sound) {
            nFramesRendered++;
            emulator.Execute(cNstVideo, cNstSound, cNstPads);
        }
        else {
            emulator.Execute(NULL, cNstSound, cNstPads);
        }

        ClearInput();
        if (nDoFPS < nFramesRendered) {
            CalculateFPS();
            nDoFPS = nFramesRendered + 60;
        }
    }
}

void StartNesGame() {
    QueryPerformanceFrequency(&ticksPerSecondDelay);
    QueryPerformanceCounter(&tickDelay);
    lastTimeDelay = (FLOAT)tickDelay.QuadPart;

    nFramesEmulated = 0;
    nCurrentFrame = 0;
    nFramesRendered = 0;
    nDoFPS = 0;
    EnableRewind();
    updateok = 0;
    nst_quit = 0;
    nFramesRendered = 0;
    nFramesEmulated = 0;
    nCurrentFrame = 0;
    // acquire the video interface
    Video video( emulator );

    while (!nst_quit) {
        DxSoundCheck();
    }
}

extern void GameDelay();

bool GameStarted = false;
int NestopiaMain(char* game) {
    void* userData = (void*) 0xDEADC0DE;
    playing = 0;
    nintendo = NULL;
    auxio_init();
    // setup video lock/unlock callbacks
    Video::Output::lockCallback.Set( VideoLock, userData );
    Video::Output::unlockCallback.Set( VideoUnlock, userData );
    Sound::Output::lockCallback.Set( SoundLock, userData );
    Sound::Output::unlockCallback.Set( SoundUnlock, userData );
    // misc callbacks (for others, see NstApuUser.hpp)
    User::fileIoCallback.Set( DoFileIO, userData );
    User::logCallback.Set( DoLog, userData );
    // try to load the FDS BIOS
    auxio_set_fds_bios();
    // and the NST database
    auxio_load_db();
    cNstVideo = new Video::Output;
    cNstSound = new Sound::Output;
    cNstPads  = new Input::Controllers;
    NstLoadGame(game);
    if(ErrorFile) {
        Nes::Api::Cartridge::Database database( emulator );
        database.Unload();
        auxio_shutdown();
        delete cNstVideo;
        delete cNstSound;
        delete cNstPads;
        ReturnToGUI();
        exitGameMenu = true;
        GameDelay();
        SetDelay();
        return 0;
    }

    SetupVideo();
    SetupSound();
    SetupInput();
    UpdatePalette();
    StartNesGame();
    NstStopPlaying();
    delete cNstVideo;
    delete cNstSound;
    delete cNstPads;
    SaveGameIni(globalGameName.c_str());
    ReturnToGUI();
    exitGameMenu = true;
    GameDelay();
    SetDelay();
    return 0;
}

int scaleVideo = 1;
int scalefactor = 1;
int NTSCMode = 0;

void SetupVideo()
{
    // renderstate structure
    Video::RenderState renderState;
    Machine machine( emulator );
    Cartridge::Database database( emulator );
    Video::RenderState::Filter filter;
    int i;
    if(Region == 0) {
        // figure out the region
        Machine::Mode n = machine.GetDesiredMode();
        if(n != Machine::NTSC) {
            machine.SetMode(Machine::PAL);
            framerate = 50;
        }
        else {
            machine.SetMode(Machine::NTSC);
            framerate = 60;
        }
    }

    if(Region == 1) {
        machine.SetMode(Machine::NTSC);
        framerate = 60;
    }

    if(Region == 2) {
		machine.SetMode(Machine::PAL);
        framerate = 50;
    }

    /*if(Region == 3) {
        machine.SetMode(Machine::FAMICOM);
        framerate = 60;
    }

    if(Region == 4) {
		machine.SetMode(Profile::System::DENDY);
        framerate = 60;
    }*/

    // we don't create a window in NSF mode
    if (nsf_mode) 
    {
        return;
    }

    // compute the major video parameters from the scaler type and scale factor
    switch (scaleVideo)
    {
    case 0:	// None (no scaling unless OpenGL)
        cur_width = Video::Output::WIDTH;
        cur_height = Video::Output::HEIGHT;
        cur_Rwidth = cur_width * scalefactor;
        cur_Rheight = cur_height * scalefactor;
        filter = Video::RenderState::FILTER_NONE;
        break;

    case 1: // NTSC
        scalefactor = 1;
        cur_width = Video::Output::NTSC_WIDTH;
        cur_Rwidth = cur_width * scalefactor;
        cur_height = Video::Output::HEIGHT;
        cur_Rheight = cur_height * 2 * scalefactor;
        filter = Video::RenderState::FILTER_NTSC;
        break;

    case 2: // scale x
        cur_width = cur_Rwidth = Video::Output::WIDTH * scalefactor;
        cur_height = cur_Rheight = Video::Output::HEIGHT * scalefactor;

        switch (scalefactor)
        {
        case 2:
            filter = Video::RenderState::FILTER_SCALE2X;
            break;

        case 3:
            filter = Video::RenderState::FILTER_SCALE3X;
            break;

        default:
            filter = Video::RenderState::FILTER_NONE;
            break;
        }
        break;

    case 3: // scale HQx
        cur_width = cur_Rwidth = Video::Output::WIDTH * scalefactor;
        cur_height = cur_Rheight = Video::Output::HEIGHT * scalefactor;

        switch (scalefactor)
        {
        case 2:
            filter = Video::RenderState::FILTER_HQ2X;
            break;

        case 3:
            filter = Video::RenderState::FILTER_HQ3X;
            break;

        case 4:
            filter = Video::RenderState::FILTER_HQ4X;
            break;

        default:
            filter = Video::RenderState::FILTER_NONE;
            break;
        }
        break;
    }

    //linear_filter = (sSettings->GetRenderType() == 2);
    renderState.filter = filter;
    renderState.width = cur_width;
    renderState.height = cur_height;
    renderState.bits.count = 16;//screen->format->BitsPerPixel;
    renderState.bits.mask.r = 0xf800;
    renderState.bits.mask.g = 0x07E0;
    renderState.bits.mask.b = 0x001F;

    // allocate the intermediate render buffer
    if(nintendo) {
        free(nintendo);
        nintendo = 0;
    }
    nintendo = (unsigned char*)osd_malloc(256 * 240 * 4);

    // acquire the video interface
    Video video( emulator );

    // set the sprite limit
    if(UnlimitedSprites)
        video.EnableUnlimSprites(true);
    else
        video.EnableUnlimSprites(false);

    switch (NTSCMode)
    {
    case 0:	// composite
        video.SetSharpness(Video::DEFAULT_SHARPNESS_COMP);
        video.SetColorResolution(Video::DEFAULT_COLOR_RESOLUTION_COMP);
        video.SetColorBleed(Video::DEFAULT_COLOR_BLEED_COMP);
        video.SetColorArtifacts(Video::DEFAULT_COLOR_ARTIFACTS_COMP);
        video.SetColorFringing(Video::DEFAULT_COLOR_FRINGING_COMP);
        break;

    case 1:	// S-Video
        video.SetSharpness(Video::DEFAULT_SHARPNESS_SVIDEO);
        video.SetColorResolution(Video::DEFAULT_COLOR_RESOLUTION_SVIDEO);
        video.SetColorBleed(Video::DEFAULT_COLOR_BLEED_SVIDEO);
        video.SetColorArtifacts(Video::DEFAULT_COLOR_ARTIFACTS_SVIDEO);
        video.SetColorFringing(Video::DEFAULT_COLOR_FRINGING_SVIDEO);
        break;

    case 2:	// RGB
        video.SetSharpness(Video::DEFAULT_SHARPNESS_RGB);
        video.SetColorResolution(Video::DEFAULT_COLOR_RESOLUTION_RGB);
        video.SetColorBleed(Video::DEFAULT_COLOR_BLEED_RGB);
        video.SetColorArtifacts(Video::DEFAULT_COLOR_ARTIFACTS_RGB);
        video.SetColorFringing(Video::DEFAULT_COLOR_FRINGING_RGB);
        break;

    case 3:
        video.SetSharpness(NTSCSharpness);
        video.SetColorResolution(NTSCResolution);
        video.SetColorBleed(NTSCColorBleed);
        video.SetColorArtifacts(NTSCArtifacts);
        video.SetColorFringing(NTSCFringing);
        break;
    }

    // set the render state
    if (NES_FAILED(video.SetRenderState(renderState)))
    {
        InvalidFile("Video Error");
    }
}

void SetEmulationVolume(int vol) {
    Nes::Api::Sound sound( emulator );
    sound.SetVolume(Nes::Api::Sound::ALL_CHANNELS, vol);
}

// initialize sound going into the game
void SetupSound()
{

    DxSoundInit();
    Nes::Api::Sound sound( emulator );
    sound.SetSampleBits( 16 );
    sound.SetSampleRate(44100);
    sound.SetVolume(Nes::Api::Sound::ALL_CHANNELS, 100);
    sound.SetSpeaker( Nes::Api::Sound::SPEAKER_STEREO);
    sampleRate   = 44100;

    memset(soundBuffer, 0, sizeof(soundBuffer));
    sampleRate   = 44100;
    bufFrameSize = (sampleRate / framerate);
    bufSize      = bufFrameSize*2;
    bufInPos     = 0;
    bufOutPos    = 0;
    bufUsed      = 0;

    cNstSound->samples[0] = soundBuffer;
    cNstSound->length[0] = bufFrameSize;
    cNstSound->samples[1] = NULL;
    cNstSound->length[1] = 0;
}

// initialize input going into the game
void SetupInput()
{
    Input(emulator).AutoSelectControllers();
    Input(emulator).AutoSelectAdapter();

    if(Controller1Type == 0)
        Input(emulator).ConnectController(0, Input::UNCONNECTED);
    if(Controller1Type == 1)
        Input(emulator).ConnectController(0, Input::PAD1);
    if(Controller1Type == 2)
        Input(emulator).ConnectController(0, Input::ZAPPER);

    if(Controller2Type == 0)
        Input(emulator).ConnectController(1, Input::UNCONNECTED);
    if(Controller2Type == 1)
        Input(emulator).ConnectController(1, Input::PAD2);
    if(Controller2Type == 2)
        Input(emulator).ConnectController(1, Input::ZAPPER);


    if(Controller3Type == 0)
        Input(emulator).ConnectController(2, Input::UNCONNECTED);
    if(Controller3Type == 1)
        Input(emulator).ConnectController(2, Input::PAD3);

    if(Controller4Type == 0)
        Input(emulator).ConnectController(3, Input::UNCONNECTED);
    if(Controller4Type == 1)
        Input(emulator).ConnectController(3, Input::PAD4);


    if(Controller1Type == 2 || Controller2Type == 2) {
        cNstPads->zapper.x = 256/2;
        cNstPads->zapper.y = 240/2;
        ZapperX = cNstPads->zapper.x;
        ZapperY = cNstPads->zapper.y;
        ZapperEnabled = true;
    }
    else {
        ZapperEnabled = false;
    }
}

void configure_savename( const char* filename )
{
    sprintf(savename, "%s\\%s%s", SRAMPath.c_str(), globalGameName.c_str(), ".sav");
    strcpy(rootname, savename);
    strcpy(gamebasename, savename);
}

// try and find a patch for the game being loaded
static int find_patch(char *patchname)
{
    FILE *f;

    sprintf(patchname, "D:\\patches\\%s.ips", globalGameName.c_str());
    if ((f = fopen(patchname, "rb")) != NULL)
    {
        fclose(f);
        return 1;
    }
    else
    {
        sprintf(patchname, "D:\\patches\\%s.ups", globalGameName.c_str());
        if ((f = fopen(patchname, "rb")) != NULL)
        {
            fclose(f);
            return 1;
        }
    }

    return 0;
}

// load a game or NES music file
void NstLoadGame(const char* filename)
{
    // acquire interface to machine
    Cartridge::Database database( emulator );
    Machine machine( emulator );
    Nsf nsf( emulator );
    Nes::Result result;
    unsigned char *compbuffer;
    int compsize, wascomp, compoffset;
    char gamename[512], patchname[512];

    if (nsf_mode)
    {
        Nsf nsf( emulator );

        nsf.StopSong();

        // clear the audio buffer
        memset(soundBuffer, 0, sizeof(soundBuffer));

        playing = 0;
    }

    // unload if necessary
    nst_unload();

    // (re)configure savename
    configure_savename(filename);

    // check if it's an archive
    wascomp = 0;
    if (auxio_load_archive(filename, &compbuffer, &compsize, &compoffset, NULL, gamename))
    {
        std::istrstream file((char *)compbuffer+compoffset, compsize);
        wascomp = 1;

        strncpy(lastarchname, filename, 511);

        configure_savename(gamename);

        if (database.IsLoaded())
        {
            dbentry = database.FindEntry((void *)&compbuffer[compoffset], compsize, get_favored_system());
        }

        if (find_patch(patchname))
        {
            std::ifstream pfile(patchname, std::ios::in|std::ios::binary);

            Machine::Patch patch(pfile, false);

            // load game and softpatch
            result = machine.Load( file, get_favored_system(), patch );
        }
        else
        {
            // load game
            result = machine.Load( file, get_favored_system() );
        }
    }
    else
    {
        FILE *f;
        int length;
        unsigned char *buffer;

        // this is a little ugly
        if (database.IsLoaded())
        {
            f = fopen(filename, "rb");
            if (!f)
            {
                loaded = 0;
                return;
            }

            fseek(f, 0, SEEK_END);
            length = ftell(f);
            fseek(f, 0, SEEK_SET);

            buffer = (unsigned char *)osd_malloc(length);
            fread(buffer, length, 1, f);
            fclose(f);

            dbentry = database.FindEntry(buffer, length, get_favored_system());

            free(buffer);
        }

        configure_savename(filename);

        // C++ file stream
        std::ifstream file(filename, std::ios::in|std::ios::binary);

        if (find_patch(patchname))
        {
            std::ifstream pfile(patchname, std::ios::in|std::ios::binary);

            Machine::Patch patch(pfile, false);

            // load game and softpatch
            result = machine.Load( file, get_favored_system(), patch );
        }
        else
        {
            // load game
            result = machine.Load( file, get_favored_system() );
        }
    }

    ErrorFile = false;
    // failed?
    if (NES_FAILED(result))
    {
        switch (result)
        {
        case Nes::RESULT_ERR_INVALID_FILE:
            InvalidFile("Invalid file");
            break;

        case Nes::RESULT_ERR_OUT_OF_MEMORY:
            InvalidFile("Out of memory");
            break;

        case Nes::RESULT_ERR_CORRUPT_FILE:
            InvalidFile("Corrupt or missing file");
            break;

        case Nes::RESULT_ERR_UNSUPPORTED_MAPPER:
            InvalidFile("Unsupported mapper");
            break;

        case Nes::RESULT_ERR_MISSING_BIOS:
            InvalidFile("Can't find D:\\bios\\disksys.rom for FDS game");
            break;

        default:
            InvalidFile("Unknown error");
            break;
        }
        ErrorFile = true;
        return;
    }

    // free the buffer if necessary
    if (wascomp)
    {
        free(compbuffer);
    }

    // is this an NSF?
    nsf_mode = (machine.Is(Machine::SOUND)) ? 1 : 0;

    if (nsf_mode)
    {
        // initialization
        cNstSound->samples[0] = soundBuffer;
        cNstSound->length[0] = 44100/framerate;
        cNstSound->samples[1] = NULL;
        cNstSound->length[1] = 0;

        updateok = 0;
        playing = 1;
        schedule_stop = 0;
    }
    else
    {
        if (machine.Is(Machine::DISK))
        {
            Fds fds( emulator );

            fds.InsertDisk(0, 0);
        }
    }

    // note that something is loaded
    loaded = 1;

    // power on
    machine.Power( true ); // false = power off
}


