////////////////////////////////////////////////////////////////////////////////////////
//
// Nestopia - NES/Famicom emulator written in C++
//
// Copyright (C) 2003-2008 Martin Freij
//
// This file is part of Nestopia.
//
// Nestopia is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Nestopia is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Nestopia; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
////////////////////////////////////////////////////////////////////////////////////////

#include "NstInpDevice.hpp"
#include "../NstPpu.hpp"
#include "NstInpZapper.hpp"

namespace Nes
{
	namespace Core
	{
		namespace Input
		{
			const byte Zapper::lightMap[Video::Screen::PALETTE] =
			{
				0x66, 0x28, 0x23, 0x24, 0x29, 0x28, 0x25, 0x2B,
				0x2F, 0x2E, 0x30, 0x2F, 0x2E, 0x00, 0x00, 0x00,
				0xAD, 0x56, 0x56, 0x56, 0x56, 0x56, 0x57, 0x5C,
				0x60, 0x60, 0x5B, 0x5A, 0x59, 0x00, 0x00, 0x00,
				0xFF, 0xA2, 0x9D, 0x9D, 0xA3, 0xA4, 0xA5, 0xA7,
				0xA9, 0xA8, 0xA7, 0xA7, 0xA7, 0x4F, 0x00, 0x00,
				0xFF, 0xD9, 0xD7, 0xD8, 0xDA, 0xDA, 0xDB, 0xDC,
				0xDC, 0xDC, 0xDB, 0xDC, 0xDC, 0xB8, 0x00, 0x00,

				0x5C, 0x19, 0x1B, 0x26, 0x2C, 0x2B, 0x24, 0x22,
				0x25, 0x26, 0x23, 0x22, 0x21, 0x00, 0x00, 0x00,
				0xA0, 0x48, 0x48, 0x48, 0x4A, 0x49, 0x49, 0x50,
				0x55, 0x54, 0x4E, 0x49, 0x49, 0x20, 0x00, 0x00,
				0xE6, 0x92, 0x8D, 0x8E, 0x90, 0x8A, 0x8C, 0x95,
				0x99, 0x98, 0x96, 0x95, 0x95, 0x46, 0x00, 0x00,
				0xE6, 0xC9, 0xC7, 0xC8, 0xC4, 0xC1, 0xC1, 0xC5,
				0xC9, 0xCA, 0xCA, 0xC9, 0xC9, 0xA9, 0x00, 0x00,

				0x5C, 0x25, 0x1A, 0x19, 0x1E, 0x1C, 0x1F, 0x26,
				0x29, 0x2C, 0x32, 0x30, 0x2B, 0x00, 0x00, 0x00,
				0x9F, 0x4C, 0x48, 0x48, 0x48, 0x48, 0x4C, 0x55,
				0x5A, 0x5A, 0x59, 0x57, 0x54, 0x10, 0x00, 0x00,
				0xEA, 0x95, 0x94, 0x94, 0x95, 0x95, 0x95, 0x9B,
				0x9F, 0x9F, 0x99, 0x96, 0x95, 0x46, 0x00, 0x00,
				0xEA, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xCA,
				0xCA, 0xCA, 0xC9, 0xCA, 0xC9, 0xA9, 0x00, 0x00,

				0x50, 0x16, 0xF0, 0x1A, 0x20, 0x1E, 0x1B, 0x1A,
				0x1D, 0x1E, 0x23, 0x22, 0x1D, 0x00, 0x00, 0x00,
				0x8F, 0x38, 0x38, 0x38, 0x39, 0x38, 0x3B, 0x44,
				0x49, 0x48, 0x46, 0x43, 0x41, 0x00, 0x00, 0x00,
				0xD7, 0x7F, 0x7E, 0x7E, 0x7F, 0x7F, 0x80, 0x85,
				0x89, 0x89, 0x83, 0x7F, 0x7F, 0x3B, 0x00, 0x00,
				0xD7, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB4, 0xB4,
				0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0x98, 0x00, 0x00,

				0x5C, 0x24, 0x19, 0x20, 0x26, 0x25, 0x1C, 0x1E,
				0x21, 0x23, 0x29, 0x2B, 0x2A, 0x00, 0x00, 0x00,
				0x9F, 0x4B, 0x44, 0x44, 0x48, 0x48, 0x49, 0x4A,
				0x4F, 0x4E, 0x4E, 0x55, 0x53, 0x10, 0x00, 0x00,
				0xE8, 0x8A, 0x85, 0x86, 0x8C, 0x95, 0x95, 0x95,
				0x95, 0x95, 0x95, 0x95, 0x93, 0x46, 0x00, 0x00,
				0xE8, 0xC2, 0xC0, 0xC0, 0xC2, 0xC6, 0xC9, 0xC9,
				0xC9, 0xC9, 0xC9, 0xC9, 0xC6, 0xA9, 0x00, 0x00,

				0x50, 0x16, 0x14, 0x1F, 0x25, 0x23, 0x1C, 0x15,
				0x19, 0x19, 0x1E, 0x1D, 0x1C, 0x00, 0x00, 0x00,
				0x8F, 0x38, 0x37, 0x39, 0x40, 0x3E, 0x38, 0x3E,
				0x42, 0x42, 0x3F, 0x41, 0x40, 0x00, 0x00, 0x00,
				0xD7, 0x7A, 0x75, 0x75, 0x7B, 0x7F, 0x7F, 0x80,
				0x81, 0x81, 0x7F, 0x7F, 0x7F, 0x3B, 0x00, 0x00,
				0xD7, 0xB2, 0xAF, 0xB0, 0xB2, 0xB3, 0xB3, 0xB4,
				0xB4, 0xB3, 0xB3, 0xB4, 0xB3, 0x98, 0x00, 0x00,

				0x4F, 0x1C, 0x12, 0x17, 0x1C, 0x1B, 0x15, 0x17,
				0x1A, 0x22, 0x27, 0x25, 0x23, 0x00, 0x00, 0x00,
				0x8F, 0x40, 0x38, 0x37, 0x38, 0x38, 0x38, 0x41,
				0x45, 0x45, 0x4A, 0x4A, 0x49, 0x00, 0x00, 0x00,
				0xD7, 0x7E, 0x79, 0x7A, 0x7F, 0x7F, 0x7F, 0x80,
				0x84, 0x84, 0x80, 0x80, 0x7F, 0x3B, 0x00, 0x00,
				0xD7, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB4,
				0xB4, 0xB4, 0xB3, 0xB4, 0xB3, 0x98, 0x00, 0x00,

				0x45, 0x15, 0x10, 0x19, 0x1F, 0x1D, 0x18, 0x14,
				0x17, 0x1B, 0x21, 0x1F, 0x1C, 0x00, 0x00, 0x00,
				0x85, 0x3A, 0x36, 0x36, 0x3A, 0x38, 0x37, 0x40,
				0x45, 0x44, 0x44, 0x44, 0x43, 0x00, 0x00, 0x00,
				0xCE, 0x7E, 0x79, 0x7A, 0x7F, 0x7F, 0x7F, 0x80,
				0x85, 0x85, 0x80, 0x80, 0x7F, 0x30, 0x00, 0x00,
				0xCE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
				0xAF, 0xAF, 0xAE, 0xAE, 0xAE, 0x8E, 0x00, 0x00
			};

			#ifdef NST_MSVC_OPTIMIZE
			#pragma optimize("s", on)
			#endif

			Zapper::Zapper(const Cpu& c,Ppu& p)
			:
			Device (c,Api::Input::ZAPPER),
			arcade (false),
			ppu    (p)
			{
				Zapper::Reset();
			}

			void Zapper::Reset()
			{
				shifter = 1;
				stream = 0x10;
				pos = ~0U;
				fire = 0;
			}

			void Zapper::Initialize(bool a)
			{
				arcade = a;
			}

			void Zapper::SaveState(State::Saver& saver,const byte id) const
			{
				const byte data[2] =
				{
					arcade ? shifter ? 0x1 : 0x3 : 0x0,
					arcade ? stream : 0x00
				};

				saver.Begin( AsciiId<'Z','P'>::R(0,0,id) ).Write( data ).End();
			}

			void Zapper::LoadState(State::Loader& loader,const dword id)
			{
				if (id == AsciiId<'Z','P'>::V)
				{
					State::Loader::Data<2> data( loader );

					if (data[0] & 0x1U)
					{
						shifter = ~data[0] >> 1 & 0x1;
						stream = data[1];
					}
				}
			}

			#ifdef NST_MSVC_OPTIMIZE
			#pragma optimize("", on)
			#endif

			uint Zapper::Poll()
			{
				if (input)
				{
					Controllers::Zapper& zapper = input->zapper;
					input = NULL;

					if (Controllers::Zapper::callback( zapper ))
					{
						fire = (zapper.fire ? arcade ? 0x80 : 0x10 : 0x00);

						if (zapper.y < Video::Screen::HEIGHT && zapper.x < Video::Screen::WIDTH)
							pos = zapper.y * Video::Screen::WIDTH + zapper.x;
						else
							pos = ~0U;
					}
				}

				if (pos < Video::Screen::WIDTH * Video::Screen::HEIGHT)
				{
					ppu.Update();

					uint pixel = ppu.GetPixelCycles();

					if (pos < pixel && pos >= pixel - PHOSPHOR_DECAY)
					{
						pixel = ppu.GetPixel( pos );

						if (arcade)
						{
							//NST_COMPILE_ASSERT( LIGHT_SENSOR >= 0x3F );
							NST_VERIFY( pixel <= 0x3F );

							if (pixel > 0x3F)
								return pixel;

							pixel = ppu.GetYuvColor( pixel );
						}

						return GetLightMap( pixel );
					}
				}

				return 0;
			}

			void Zapper::Poke(const uint data)
			{
				if (arcade)
				{
					shifter = ~data & 0x1;
					stream  = (Poll() >= LIGHT_SENSOR ? 0x40 : 0x00);
					stream |= 0x10 | fire;
				}
			}

			uint Zapper::Peek(uint)
			{
				if (!arcade)
				{
					uint data = (Poll() >= LIGHT_SENSOR ? 0x0 : 0x8);
					return data | fire;
				}
				else
				{
					const uint data = stream;
					stream >>= shifter;
					return data & 0x1;
				}
			}
		}
	}
}
