
// These are all the includes we need.  One for the basic Windows stuff
// (which we will use as rarely as possible, I'm not a fan of the Windows
// API) and one for Direct3D 8.
#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <xtl.h>
#include <D3DX8.h>
#include <d3d8types.h>

// This is causes the d3d8.lib to be linked in, the same thing can be accomplished by
// adding it to your compiler's link list (Project->Settings->Link in VC++),
// but I prefer this method.
#pragma comment(lib,"d3d8.lib")
#pragma comment(lib,"d3dx8.lib")

// Forward declarations for all of our functions, see their definitions for more detail
void FatalError(const char *error_msg);
void ask_fullscreen(void);
LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam);
void init_window(void);
void kill_window(void);
void init_d3d(void);
void kill_d3d(void);
void init_scene(void);
void kill_scene(void);
void message_pump(void);
D3DFORMAT find_16bit_mode(void);
void render(void);
void NOP(HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show);

// The name of our application.  Used for window titles, MessageBox titles and
// error reporting
const char g_app_name[]="DirectX 8 Lesson 6";

// Our screen/window sizes.  A better app would allow the user to choose the
// sizes.  I'll do that in a later tutorial, for now this is good enough.
const int g_width=640;
const int g_height=480;

// A global flag to determine if we're windowed (false) or full-screen(true)
bool g_fullscreen=true;

// A global handle to our main window, initializing pointers to NULL can save you
// a lot of hassle in the future.
HWND g_main_window=NULL;

// A global handle to our 'instance'.  This is needed in various places by the Windows API.
HINSTANCE g_instance;

// Our global flag to track whether we should quit or not.  When it becomes true, we clean
// up and exit.
bool g_app_done=false;

// Our main Direct3D interface, it doesn't do much on it's own, but all the more commonly
// used interfaces are created by it.  It's the first D3D object you create, and the last
// one you release.
LPDIRECT3D8 g_D3D=NULL;

// The D3DDevice is your main rendering interface.  It represents the display and all of its
// capabilities.  When you create, modify, or render any type of resource, you will likely
// do it through this interface.
IDirect3DDevice8 *g_d3d_device=NULL;

//A texture is an image that is to be 'applied' to a polygon/triangle.  By 'painting'
//our triangles with images we can achieve some amazing effects.
LPDIRECT3DTEXTURE8 g_texture=NULL;

//We add 2 floats to our vertex structure to store the texture coordinates.
//tu & tv represent the location of the texture that should be mapped to this
//vertex. tu can be thought of as the x-coordinate of the texture, and tv as
//the y-coordinate.  So if a vertex was to be mapped to the top left of a texture
// you would set tu & tv to (0,0).  Note-Texture coordinates range from 0.0 (left, or top)
// to 1.0 (right, or bottom).
//NOTE:Since we want our texture to provide all of our colour information we don't have
//a diffuse component in our vertex definition.
struct my_vertex{
	FLOAT x, y, z;   // The untransformed position for the vertex.
	FLOAT tu, tv;    // The texture coordinates
};

//A handy little 'macro' for our definition of the vertex.  When we use the vertex data
//we have to tell D3D what data we're passing it.  
//D3DFVF_XYZ specifies that the vertex will have coordinate given in model space.
//D3DFVF_TEX1 specifies that 1 set of texture coordinates will be provided as well.
#define D3D8T_CUSTOMVERTEX (D3DFVF_XYZ|D3DFVF_TEX1)

//The last 2 floats in this structure are the texture coordinates.  As mentioned
//before, their values range from 0.0 to 1.0.  Our texture is a solid square and
//we want to map it on to the cube so that each cube face is covered by the texture.
//Looking at the front face of our cube, we see that (-1,1) is the top left of the
//face, so we set its texture coordinates to (0,0), the top left of the texture.
//Similarly, since vertex (1,-1) is the bottom right position, we set it to the
//bottom right (1,1) of the texture.
my_vertex g_vertices[] ={
	{ -1.0f, -1.0f, -1.0f, 0.0f, 1.0f },  //Front face
	{ -1.0f,  1.0f, -1.0f, 0.0f, 0.0f },
	{  1.0f,  1.0f, -1.0f, 1.0f, 0.0f },
	{  1.0f,  1.0f, -1.0f, 1.0f, 0.0f },
	{  1.0f, -1.0f, -1.0f, 1.0f, 1.0f },
	{ -1.0f, -1.0f, -1.0f, 0.0f, 1.0f },

	//{  1.0f, -1.0f,  1.0f, 0.0f, 1.0f },  //Back face
	//{  1.0f,  1.0f,  1.0f, 0.0f, 0.0f },
	//{ -1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{ -1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{ -1.0f, -1.0f,  1.0f, 1.0f, 1.0f },
	//{  1.0f, -1.0f,  1.0f, 0.0f, 1.0f },

	//{ -1.0f,  1.0f, -1.0f, 0.0f, 1.0f },  //Top face
	//{ -1.0f,  1.0f,  1.0f, 0.0f, 0.0f },
	//{  1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{  1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{  1.0f,  1.0f, -1.0f, 1.0f, 1.0f },
	//{ -1.0f,  1.0f, -1.0f, 0.0f, 1.0f },

	//{  1.0f, -1.0f, -1.0f, 0.0f, 1.0f },  //Bottom face
	//{  1.0f, -1.0f,  1.0f, 0.0f, 0.0f },
	//{ -1.0f, -1.0f,  1.0f, 1.0f, 0.0f },
	//{ -1.0f, -1.0f,  1.0f, 1.0f, 0.0f },
	//{ -1.0f, -1.0f, -1.0f, 1.0f, 1.0f },
	//{  1.0f, -1.0f, -1.0f, 0.0f, 1.0f },

	//{ -1.0f, -1.0f,  1.0f, 0.0f, 1.0f },  //Left face
	//{ -1.0f,  1.0f,  1.0f, 0.0f, 0.0f },
	//{ -1.0f,  1.0f, -1.0f, 1.0f, 0.0f },
	//{ -1.0f,  1.0f, -1.0f, 1.0f, 0.0f },
	//{ -1.0f, -1.0f, -1.0f, 1.0f, 1.0f },
	//{ -1.0f, -1.0f,  1.0f, 0.0f, 1.0f },

	//{  1.0f, -1.0f, -1.0f, 0.0f, 1.0f },  //Right face
	//{  1.0f,  1.0f, -1.0f, 0.0f, 0.0f },
	//{  1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{  1.0f,  1.0f,  1.0f, 1.0f, 0.0f },
	//{  1.0f, -1.0f,  1.0f, 1.0f, 1.0f },
	//{  1.0f, -1.0f, -1.0f, 0.0f, 1.0f },

};

//This handy macro makes sure that we always reference the right number of
//vertices.  This way we always allocate a correctly sized vertex buffer.
#define NUM_VERTICES (sizeof(g_vertices)/sizeof(my_vertex))


//Vertex buffers are a method of storing vertices to be rendered in an optimized manner.
IDirect3DVertexBuffer8 *g_vb=NULL;

extern int cur_width;
extern int cur_height;

void ZeroTexture() {
	D3DSURFACE_DESC desc;
	g_texture->GetLevelDesc( 0, &desc );
	DWORD dwTexWidth  = desc.Width;
	DWORD dwTexHeight = desc.Height;
	D3DLOCKED_RECT lock;
	g_texture->LockRect( 0, &lock, 0, 0L );
	VOID* pBits = lock.pBits;
	for( DWORD y = 0; y < dwTexHeight; y++ ) {
		for( DWORD x = 0; x < dwTexWidth; x++ ) {
			((unsigned int*)pBits)[y*dwTexWidth+x] = 0x00000000;
		}
	}
	g_texture->UnlockRect( 0 );
}

void VideoExit() {
    kill_scene();

	//Clean up all of our Direct3D objects
	kill_d3d();

	//Close down our window
	kill_window();
}
// WinMain is the first function called by Windows when our app is run.  It's the entry
// point of our application.
int framework(){

	// Set our global instance handle so we don't have to pass it around
	HINSTANCE g_instance = NULL;//p_instance;

	//This function exists to quiet compiler warnings, see its definition for more detail
	//NOP(p_prev_instance,p_cmd_line,p_show);

	// Prompt the user, Full Screen?  Windowed?  Cancel?
	ask_fullscreen();

	// Build our window.  Cover the screen if full-screen, otherwise make a standard window
	init_window();

	//Build the D3D objects we'll require
	init_d3d();

	//One-time preparation of objects and other stuff required for rendering
	init_scene();

	//Exit happily
	return 0;
}

// Procedure: NOP
// Whazzit:This procedure does nothing.  If set to a high warning level
//         (which I like to do) the compiler will complain because the
//         parameters passed into WinMain are never used.  The purpose
//         of this procedure is to make it think that they are used, so
//         it doesn't complain.  
void NOP(HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show){

	p_prev_instance=p_prev_instance;
	p_cmd_line=p_cmd_line;
	p_show=p_show;
}


// Procedure: message_pump
// Whazzit:Checks the message queue to see if any windows messages
//         (window is closing, window needs repainting, etc)
//         are waiting and if there are, the messages are dispatched
//         to our message handler.
void message_pump(void){
	MSG msg;

	if(PeekMessage(&msg, NULL, 0, 0,PM_REMOVE)){
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

}


// Function:init_d3d
// Whazzit:Sets up Direct3D and creates the device.  The device is create differently
//         if we're full-screen as opposed to running in a desktop window.
void init_d3d(void){
	HRESULT hr;
	D3DPRESENT_PARAMETERS d3dpp;
	D3DDISPLAYMODE display_mode;

	//Create Direct3D8, this is the first thing you have to do in any D3D8 program
	g_D3D = Direct3DCreate8( D3D_SDK_VERSION );
	if(!g_D3D ){
		FatalError("Error getting Direct3D");
	}

	//Get the current(desktop) display mode.  This is really only needed if
	//we're running in a window.
	hr=g_D3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT,&display_mode);
	if(FAILED(hr)){
		FatalError("Error getting display mode\n");
	}

	//Clear out our D3DPRESENT_PARAMETERS structure.  Even though we're going
	//to set virtually all of it members, it's good practice to zero it out first.
	ZeroMemory(&d3dpp,sizeof(d3dpp));

	//Whether we're full-screen or windowed these are the same.
	d3dpp.SwapEffect     = D3DSWAPEFFECT_DISCARD; // Throw away previous frames, we don't need them
	d3dpp.hDeviceWindow  = g_main_window;  //This is our main (and only) window
	d3dpp.BackBufferCount= 1;  //We only need a single back buffer

	//We need a Z-Buffer so everything will be drawn properly
	d3dpp.EnableAutoDepthStencil = TRUE;
	d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

	// BackBufferWidth/Height have to be set for full-screen apps, these values are
	//used (along with BackBufferFormat) to determine the display mode.
	//They aren't needed in windowed mode since the size of the window will be used.
	// BackBufferFormat is the pixel format we want.  In windowed mode we use the same
	//format as the desktop (which we found by using GetAdapterDisplayMode() above).
	//In full-screen we need to find a pixel format we like, see find_16bit_mode()
	//below for more details.
	if(g_fullscreen){
		d3dpp.Windowed          = FALSE;
		d3dpp.BackBufferWidth   = g_width;
		d3dpp.BackBufferHeight  = g_height;
		d3dpp.BackBufferFormat  = find_16bit_mode();
	}else{
		d3dpp.Windowed          = TRUE;
		d3dpp.BackBufferFormat  = display_mode.Format;
	}


	hr=g_D3D->CreateDevice(D3DADAPTER_DEFAULT, //The default adapter, on a multimonitor system
		//there can be more than one.
		//Use hardware acceleration rather than the software renderer
		D3DDEVTYPE_HAL,
		//Our Window
		g_main_window,
		//Process vertices in software. This is slower than in hardware,
		//But will work on all graphics cards.
		D3DCREATE_SOFTWARE_VERTEXPROCESSING,
		//Our D3DPRESENT_PARAMETERS structure, so it knows what we want to build
		&d3dpp,
		//This will be set to point to the new device
		&g_d3d_device);
	if(FAILED(hr)){
		FatalError("Error creating device\n");
	}


}
// Function:kill_d3d
// Whazzit:Releases all of our D3D resources in the opposite order from their creation.
//         Note-Since we initially set the pointers to be NULL, we can safely test them
//         for a non-NULL state and we know if they've been created.  Thus we never Release
//         something we didn't create (which causes bad things to happen).
void kill_d3d(void){

	if(g_d3d_device){
		g_d3d_device->Release();
		g_d3d_device=NULL;
	}

	if(g_D3D){
		g_D3D->Release();
		g_D3D=NULL;
	}

}
// Function:init_scene
// Whazzit:One-time preparation of objects required for rendering.
void init_scene(void){
	HRESULT hr;
	unsigned char *vb_vertices;
	D3DXMATRIX view_matrix;
	D3DXMATRIX matProj;

	//The following SetTextureStageState calls determine how textures will
	//be applied to your objects in all of the following Draw operations.

	//D3DTSS_COLOROP specifies how the colour of each pixel will be determined
	//In this case it's set to D3DTOP_SELECTARG1 which means we look to the
	//setting of D3DTSS_COLORARG1 to determine the colour.
	g_d3d_device->SetTextureStageState(0,D3DTSS_COLOROP,   D3DTOP_SELECTARG1);

	//D3DTSS_COLORARG1 is set to D3DTA_TEXTURE which means the colour is
	//entirely taken from the texture and nothing else.  Alternately we could
	//have set D3DTA_DIFFUSE and then the colour would have come from our
	//vertex colour, completely ignoring the texture.
	g_d3d_device->SetTextureStageState(0,D3DTSS_COLORARG1, D3DTA_TEXTURE);

	//We don't want to use Alpha Blending (Transparency) so we make sure it's
	//turned off for extra speed.
	g_d3d_device->SetTextureStageState(0,D3DTSS_ALPHAOP,   D3DTOP_DISABLE);

	//The MAGFILTER specifies how the texture should be filtered when it's
	//drawn larger than it's real size, MINFILTER is for when it's drawn smaller
	//than it's real size.
	//The 2 most common choices are D3DTEXF_LINEAR & D3DTEXF_POINT.  Linear
	//filtering is smoother but is also more expensive, point is fast but
	//generally doesn't look as good.
	g_d3d_device->SetTextureStageState(0,D3DTSS_MAGFILTER, D3DTEXF_LINEAR);
	g_d3d_device->SetTextureStageState(0,D3DTSS_MINFILTER, D3DTEXF_LINEAR);

	//Turn off D3D lighting.  For this tutorial we don't want Direct3D doing
	//any lighting.
	g_d3d_device->SetRenderState( D3DRS_LIGHTING,FALSE);

	// Turn on the zbuffer.  The ZBuffer makes sure that objects appear in the
	//proper order.  Without it we'd have to make sure we draw our triangles
	//in back-to-front order, with a ZBuffer we don't have to worry about it.
	g_d3d_device->SetRenderState( D3DRS_ZENABLE, TRUE );

	//Given a pointer to our device and a file name for our bitmap,
	//this function loads our texture and set g_texture to point to it.
	hr=g_d3d_device->CreateTexture(cur_width, cur_height, 0, 0L, D3DFMT_A8R8G8B8, D3DPOOL_MANAGED, &g_texture); //D3DXCreateTextureFromFile( g_d3d_device, "DH.bmp",&g_texture);
	if(FAILED(hr)){
		FatalError("Error loading texture");
	}

	ZeroTexture();

	hr=g_d3d_device->CreateVertexBuffer(NUM_VERTICES*sizeof(my_vertex), //Size of memory to be allocated
		// Number of vertices * size of a vertex
		D3DUSAGE_WRITEONLY, // We never need to read from it so
		// we specify write only, it's faster
		D3D8T_CUSTOMVERTEX, //Our custom vertex specifier (coordinates & a colour)
		D3DPOOL_MANAGED,   //Tell DirectX to manage the memory of this resource
		&g_vb);            //Pointer to our Vertex Buffer, after this call
	//It will point to a valid vertex buffer
	if(FAILED(hr)){
		FatalError("Error creating vertex buffer");
	}



	//Now we have our Vertex Buffers, but they're empty. To put our data into them
	//we Lock the Vertex Buffer so Direct3D knows we're modifying it, then we copy
	//our data in and Unlock it so Direct3D knows we're done.
	hr=g_vb->Lock(0, //Offset, we want to start at the beginning
		0, //SizeToLock, 0 means lock the whole thing
		&vb_vertices, //If successful, this will point to the data in the VB
		0);  //Flags, nothing special
	if(FAILED(hr)){
		FatalError("Error Locking triangle buffer");
	}

	//vb_vertices now points to our vertices inside the Vertex buffer, so
	//to fill in our VB, we copy to vb_vertices.
	memcpy(vb_vertices, g_vertices, sizeof(g_vertices) );

	//Unlock so Direct3D knows we're done and can do any behind-the-scenes magic required
	g_vb->Unlock();


	//Here we build our View Matrix, think of it as our camera.
	//First we specify that our viewpoint is 8 units back on the Z-axis
	//We are looking towards the origin
	//And the y-axis is up
	D3DXMatrixLookAtLH(&view_matrix,&D3DXVECTOR3( 0.0f, 0.0f,-3.5f ),
		&D3DXVECTOR3( 0.0f, 0.0f, 0.0f ),
		&D3DXVECTOR3( 0.0f, 1.0f, 0.0f ));

	//Since our 'camera' will never move, we can set this once at the
	//beginning and never worry about it again
	g_d3d_device->SetTransform(D3DTS_VIEW,&view_matrix);


	D3DXMatrixPerspectiveFovLH(&matProj, //Result Matrix
		D3DX_PI/4,//Field of View, in radians. (PI/4) is typical
		((float)g_width / (float)g_height),     //Aspect ratio
		1.0f,     //Near view plane
		100.0f ); // Far view plane

	//Our Projection matrix won't change either, so we set it now and never touch
	//it again.
	g_d3d_device->SetTransform( D3DTS_PROJECTION, &matProj );

}
// Function:kill_scene
// Whazzit:Clean up any objects we required for rendering.
void kill_scene(void){

	if(g_texture){
		g_texture->Release();
		g_texture=NULL;
	}

	if(g_vb){
		g_vb->Release();
		g_vb=NULL;
	}

}
// Function:find_16bit_mode
// Whazzit:Tests a couple of 16-bit modes to see if they are supported.  Virtually every graphics
//         card in existance will support one of these 2 formats.
D3DFORMAT find_16bit_mode(void){
	HRESULT hr;

	//First we test for R5G6B5.  All 16-bits are used in this format giving us a full 64K worth
	//worth of colours
	hr=g_D3D->CheckDeviceType(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,D3DFMT_R5G6B5,D3DFMT_R5G6B5,FALSE);
	if(SUCCEEDED(hr)){
		OutputDebugString("D3DFMT_R5G6B5\n");
		return D3DFMT_R5G6B5;
	}

	//Next try X1R5G5B5. Since 1 bit is wasted it's technically a 15-bit mode and only
	//provides 32K colours, though you'd be hard pressed to tell the difference between
	//15- & 16-bit modes.
	hr=g_D3D->CheckDeviceType(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,D3DFMT_X1R5G5B5,D3DFMT_X1R5G5B5,FALSE);
	if(SUCCEEDED(hr)){
		OutputDebugString("D3DFMT_X1R5G5B5\n");
		return D3DFMT_X1R5G5B5;
	}

	//This is a freaky card.  Complain and bail out.
	FatalError("Couldn't find a decent mode\n");

	//Won't actually hit this line since FatalError() kills us, but it makes the compiler happy.
	return (D3DFORMAT)NULL;
}
// Function:ask_fullscreen
// Whazzit:Ask the user if they would like to run in full-screen or windowed mode or if they
//         would like to Cancel (abort).
void ask_fullscreen(void){
	int full_result;

	full_result=MessageBox(NULL,"Would you like to run in fullscreen mode?",g_app_name,
		MB_YESNOCANCEL|MB_ICONQUESTION);
	switch(full_result){
case IDCANCEL:
	OutputDebugString("User Abort");
	MessageBox(NULL,"User Abort",g_app_name,MB_OK);
	exit(5);
	break;
case IDNO:
	g_fullscreen=false;
	break;
case IDYES:
	g_fullscreen=true;
	break;
case 0:
	OutputDebugString("Couldn't open MessageBox, dying");
	exit(10);
	break;
	}

}

// Function: render
// Whazzit:Draws out pretty(?) spinning cube
void render(void){
	D3DXMATRIX matWorld;
	D3DXMATRIX rot_x_matrix;   //Our rotation matrix for the x-axis
	D3DXMATRIX rot_y_matrix;   //Our rotation matrix for the y_axis
	D3DXMATRIX trans_matrix;   //Our translation matrix
	static float rot_x=0;      //Tracks rotation for the x-axis
	static float rot_y=0;      //Tracks rotation for the y-axis

	rot_x+=0;//0.007f;
	rot_y+=0;//0.002f;

	//Clear the buffer to our new colour and clear the ZBuffer
	g_d3d_device->Clear( 0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER , D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );

	//Notify the device that we're ready to render
	if(SUCCEEDED(g_d3d_device->BeginScene())){

		//Vertex shaders are a complex topic, but you can do some amazing things with them
		//For this example we're not creating one, so we tell Direct3D that we're just
		//using a plain vertex format.
		g_d3d_device->SetVertexShader(D3D8T_CUSTOMVERTEX);

		//D3D's rendering functions read from streams.  Here we tell D3D that the
		//VB we created for our triangle is the stream it should read from.
		g_d3d_device->SetStreamSource(0,g_vb,sizeof(my_vertex));

		//Here we set our texture to be the current texture in the first texture 'stage'
		//There can be up to 8 textures and they can be blended in a variety of ways.
		//For now we only need one.
		g_d3d_device->SetTexture( 0, g_texture );

		//Set up the rotation matrix for the triangle
		D3DXMatrixRotationY(&rot_y_matrix,rot_y);
		D3DXMatrixRotationX(&rot_x_matrix,rot_x);

		//D3DXMatrixTranslation(&trans_matrix,-1.5,0.0f,1.0f);
		//Combine the 2 matrices to get our final World Matrix
		D3DXMatrixMultiply(&matWorld,&rot_x_matrix,&rot_y_matrix);

		//Set our World Matrix
		g_d3d_device->SetTransform(D3DTS_WORLD,&matWorld );

		//Draw our cube
		g_d3d_device->DrawPrimitive(D3DPT_TRIANGLELIST,0,NUM_VERTICES/3);

		//Notify the device that we're finished rendering for this frame
		g_d3d_device->EndScene();
	}

	//After drawing, empty the first texture stage.  When you SetTexture a texture
	//Direct3D adds to it's internal reference count, this prevents a texture from
	//being released when it's still being used.
	//When you're done drawing it's a good idea to set all of your texture stages to NULL
	//so your textures can be released when they should be.
	g_d3d_device->SetTexture(0,NULL);

	//Show the results
	g_d3d_device->Present( NULL, NULL, NULL, NULL );

}
// Function: init_window
// Whazzit:Registers a window class and then creates our window.
void init_window(void){
	ULONG window_width, window_height;
	WNDCLASS window_class;
	DWORD style;

	//Fill in all the fields for the WNDCLASS structure.  Window classes
	//are a sort of template for window creation.  You could create many
	//windows using the same window class.
	window_class.style          = CS_OWNDC;
	window_class.cbClsExtra     = 0;
	window_class.cbWndExtra     = 0;
	window_class.hInstance      = g_instance;
	window_class.hIcon          = LoadIcon(NULL,IDI_APPLICATION);
	window_class.hCursor        = LoadCursor(NULL,IDC_ARROW);
	window_class.hbrBackground  = (HBRUSH)GetStockObject(BLACK_BRUSH);
	window_class.lpszMenuName   = NULL;
	window_class.lpszClassName  = "DH Class";
	//Here we provide our default window handler, all windows messages
	//will be sent to this function.
	window_class.lpfnWndProc    = default_window_proc;


	//Register the class with windows
	if(!RegisterClass(&window_class)){
		FatalError("Error registering window class");
	}

	//If we're running full screen, we cover the desktop with our window.
	//This isn't necessary, but it provides a smoother transition for the
	//user, especially when we're going to change screen modes.
	if(g_fullscreen){
		window_width=GetSystemMetrics(SM_CXSCREEN);
		window_height=GetSystemMetrics(SM_CYSCREEN);
		style=WS_POPUP;
	}else{
		//In windowed mode, we just make the window whatever size we need.
		window_width=g_width;
		window_height=g_height;
		style=WS_OVERLAPPED|WS_SYSMENU;
	}


	g_main_window=CreateWindow("DH Class",g_app_name,style,
		0,0,window_width,window_height,NULL,NULL,
		g_instance,NULL);

	if(!g_main_window){
		FatalError("Error opening window");
	}

	//The next 3 lines just make sure that our window is visible and has the
	//input focus.  It's not strictly necessary, but it doesn't hurt to be
	//thorough.
	ShowWindow(g_main_window,SW_SHOW);
	UpdateWindow(g_main_window);
	SetFocus(g_main_window);

}
// Function: kill_window
// Whazzit:Closes the window, clean up any waiting messages, and then unregister 
//         our window class.  Note - This is not the standard Win32 way of cleaning
//         up your window.  The standard way involves putting the clean-up code in
//         your window handler, I don't like that method.
void kill_window(void){

	//Test if our window is valid
	if(g_main_window){
		if(!DestroyWindow(g_main_window)){
			//We failed to destroy our window, this shouldn't ever happen
			OutputDebugString("   Failed to DestroyWindow\n");
			MessageBox(NULL,"Destroy Window Failed",g_app_name,MB_OK|MB_ICONERROR|MB_TOPMOST);
		}else{
			MSG msg;
			//Clean up any pending messages
			while(PeekMessage(&msg, NULL, 0, 0,PM_REMOVE)){
				DispatchMessage(&msg);
			}
		}
		//Set our window handle to NULL just to be safe
		g_main_window=NULL;
	}

	//Unregister our window, if we had opened multiple windows using this
	//class, we would have to close all of them before we unregistered the class.
	if(!UnregisterClass("DH Class",g_instance)){
		OutputDebugString("   Failed to Unregister Window\n");
		MessageBox(NULL,"Unregister Failed",g_app_name,MB_OK|MB_ICONERROR|MB_TOPMOST);
	}

}
// Function:FatalError
// Whazzit:Close down all resources, alert the user and quit
void FatalError(const char *error_msg){

	kill_scene();

	kill_d3d();

	kill_window();

	OutputDebugString( error_msg );
	OutputDebugString("\n");
	MessageBox(NULL, error_msg,g_app_name, MB_OK );

	exit(5);

}
// Function:default_window_proc
// Whazzit:All Windows messages get passed through this function.  We only handle
//         a tiny subset of the available messages, all unhandled messages get
//         passed through to DefWindowProc() which is part of the Win32 API.
LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam){
	HDC hdc;
	PAINTSTRUCT ps;

	switch(msg){
case WM_KEYDOWN:  // A key has been pressed, end the app
	g_app_done=true;
	return 0;
case WM_PAINT:    //Repaint our window
	hdc=BeginPaint(hwnd,&ps);
	EndPaint(hwnd,&ps);
	return 0;
case WM_CLOSE:    //User hit the Close Window button, end the app
	g_app_done=true;
	return 0;
case WM_DESTROY:  //This window is being destroyed, tell Windows we're quitting
	PostQuitMessage(0);
	return 0;
	}

	return (DefWindowProc(hwnd,msg,wparam,lparam));

}
