#include "toaplan.h"
// Truxton 2

static unsigned char DrvButton[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvJoy1[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvJoy2[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvInput[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static unsigned char *Mem = NULL, *MemEnd = NULL;
static unsigned char *RamStart, *RamEnd;
static unsigned char *Rom01;
static unsigned char *Ram01, *RamPal;

static int nColCount = 0x0800;

static unsigned char DrvReset = 0;
static unsigned char bDrawScreen;
static bool bVBlank;

static struct BurnInputInfo truxton2InputList[] = {
	{"P1 Coin",		0, DrvButton + 3,	"p1 coin"},
	{"P1 Start",	0, DrvButton + 5,	"p1 start"},

	{"P1 Up",		0, DrvJoy1 + 0, 	"p1 up"},
	{"P1 Down",		0, DrvJoy1 + 1, 	"p1 down"},
	{"P1 Left",		0, DrvJoy1 + 2, 	"p1 left"},
	{"P1 Right",	0, DrvJoy1 + 3, 	"p1 right"},
	{"P1 Button 1",	0, DrvJoy1 + 4,		"p1 fire 1"},
	{"P1 Button 2",	0, DrvJoy1 + 5,		"p1 fire 2"},
	{"P1 Button 3",	0, DrvJoy1 + 6,		"p1 fire 3"},

	{"P2 Coin",		0, DrvButton + 4,	"p2 coin"},
	{"P2 Start",	0, DrvButton + 6,	"p2 start"},

	{"P2 Up",		0, DrvJoy2 + 0, 	"p2 up"},
	{"P2 Down",		0, DrvJoy2 + 1, 	"p2 down"},
	{"P2 Left",		0, DrvJoy2 + 2, 	"p2 left"},
	{"P2 Right",	0, DrvJoy2 + 3, 	"p2 right"},
	{"P2 Button 1",	0, DrvJoy2 + 4,		"p2 fire 1"},
	{"P2 Button 2",	0, DrvJoy2 + 5,		"p2 fire 2"},
	{"P2 Button 3",	0, DrvJoy2 + 6,		"p2 fire 3"},

	{"Reset",		0, &DrvReset,		"reset"},
	{"Diagnostics",	0, DrvButton + 0,	"diag"},
	{"Dip 1",		2, DrvInput + 3,	"dip"},
	{"Dip 2",		2, DrvInput + 4,	"dip"},
	{"Dip 3",		2, DrvInput + 5,	"dip"},
};

STDINPUTINFO(truxton2);

static struct BurnDIPInfo truxton2DIPList[] = {
	// Defaults
	{0x14,	0xFF, 0xFE,	0x00, NULL},
	{0x15,	0xFF, 0xFF,	0x00, NULL},
	{0x16,	0xFF, 0x0F,	0x02, NULL},
	
	// DIP 1
	{0,		0xFE, 0,	2,	  NULL},
	{0x14,	0x01, 0x02,	0x00, "Normal screen"},
	{0x14,	0x01, 0x02,	0x02, "Invert screen"},
	{0,		0xFE, 0,	2,	  NULL},
	{0x14,	0x01, 0x04,	0x00, "Normal mode"},
	{0x14,	0x01, 0x04,	0x04, "Test mode"},
	{0,		0xFE, 0,	2,	  "Advertise sound"},
	{0x14,	0x01, 0x08,	0x00, "On"},
	{0x14,	0x01, 0x08,	0x08, "Off"},
	
		// Normal coin settings
	{0,		0xFE, 0,	4,	  "Coin A"},
	{0x14,	0x82, 0x30,	0x00, "1 coin 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0x30,	0x10, "1 coin 2 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0x30,	0x20, "2 coins 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0x30,	0x30, "2 coins 3 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0,		0xFE, 0,	4,	  "Coin B"},
	{0x14,	0x82, 0xC0,	0x00, "1 coin 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0xC0,	0x40, "1 coin 2 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0xC0,	0x80, "2 coins 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x82, 0xC0,	0xC0, "2 coins 3 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	
	// European coin settings
	{0,		0xFE, 0,	4,	  "Coin A"},
	{0x14,	0x02, 0x30,	0x00, "1 coin 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0x30,	0x10, "2 coins 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0x30,	0x20, "3 coins 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0x30,	0x30, "3 coins 1 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0,		0xFE, 0,	4,	  "Coin B"},
	{0x14,	0x02, 0xC0,	0x00, "1 coin 2 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0xC0,	0x40, "1 coin 3 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0xC0,	0x80, "1 coin 4 play"},
	{0x16,	0x00, 0x0F, 0x02, NULL},
	{0x14,	0x02, 0xC0,	0xC0, "1 coin 6 plays"},
	{0x16,	0x00, 0x0F, 0x02, NULL},

	// DIP 2
	{0,		0xFE, 0,	4,	  "Game difficulty"},
	{0x15,	0x01, 0x03,	0x00, "B"},
	{0x15,	0x01, 0x03,	0x01, "A"},
	{0x15,	0x01, 0x03,	0x02, "C"},
	{0x15,	0x01, 0x03,	0x03, "D"},
	{0,		0xFE, 0,	4,	  "Extend"},
	{0x15,	0x01, 0x0C,	0x00, "70000, 200000"},
	{0x15,	0x01, 0x0C,	0x04, "100000, 250000"},
	{0x15,	0x01, 0x0C,	0x08, "100000"},
	{0x15,	0x01, 0x0C,	0x0C, "200000"},
	{0,		0xFE, 0,	4,	  "Hero counts"},
	{0x15,	0x01, 0x30,	0x00, "3"},
	{0x15,	0x01, 0x30,	0x01, "5"},
	{0x15,	0x01, 0x30,	0x02, "4"},
	{0x15,	0x01, 0x30,	0x03, "2"},
	{0,		0xFE, 0,	2,	  NULL},
    {0x15,	0x01, 0x40,	0x00, "Normal game"},
    {0x15,	0x01, 0x40,	0x40, "No-death & stop mode"},
	{0,		0xFE, 0,	2,	  "Continue play"},
    {0x15,	0x01, 0x80,	0x00, "On"},
	{0x15,	0x01, 0x80,	0x80, "Off"},
	
	// DIP 3
	{0,		0xFE, 0,	7,	  "For"},
	{0x16,	0x01, 0x0F,	0x00, "Japan"},
	{0x16,	0x01, 0x0F,	0x01, "U.S.A."},
	{0x16,	0x01, 0x0F,	0x02, "Europe"},
	{0x16,	0x01, 0x0F,	0x03, "Hong Kong"},
	{0x16,	0x01, 0x0F,	0x04, "Korea"},
	{0x16,	0x01, 0x0F,	0x05, "Taiwan"},
	{0x16,	0x01, 0x0F,	0x06, "South East Asia"},
	{0x16,	0x01, 0x0F,	0x07, "U.S.A."},
	{0x16,	0x01, 0x0F,	0x08, "Hong Kong"},
	{0x16,	0x01, 0x0F,	0x09, "Korea"},
	{0x16,	0x01, 0x0F,	0x0A, "U.S.A."},
	{0x16,	0x01, 0x0F,	0x0B, ""},
	{0x16,	0x01, 0x0F,	0x0C, ""},
	{0x16,	0x01, 0x0F,	0x0D, ""},
	{0x16,	0x01, 0x0F,	0x0E, ""},
	{0x16,	0x01, 0x0F,	0x0F, "Japan"},
};

STDDIPINFO(truxton2);

static int __fastcall DrvResetCallback()
{
	// Reset instruction on 68000

	return 0;
}

static unsigned char __fastcall Drv1ReadByte(unsigned int sekAddress)
{
	switch (sekAddress) {

		case 0x700007:								// Player 1 inputs
			return DrvInput[0];
		case 0x700009:								// Player 2 inputs
			return DrvInput[1];
		case 0x70000B:								// Other inputs
			return DrvInput[2];
		case 0x700001:								// Dipswitch A
			return DrvInput[3];
		case 0x700003:								// Dipswitch B
			return DrvInput[4];
		case 0x700005:								// Dipswitch C - Territory
			return DrvInput[5];

		case 0x700011:
			return OKIM6295ReadStatus(0);
		case 0x700017:
			return YM2151ReadStatus(0);

		default: {
//			printf("Attempt to read byte value of location %x\n", sekAddress);

			if ((sekAddress & 0x00FF0000) == 0x00500000) {
				return ExtraTROM[(sekAddress & 0xFFFF) >> 1];
			}
		}
	}	
	return 0;
}

static unsigned short __fastcall Drv1ReadWord(unsigned int sekAddress)
{
	switch (sekAddress) {

		case 0x200004:
			return ToaGP9001ReadRAM_Hi(0);
		case 0x200006:
			return ToaGP9001ReadRAM_Lo(0);

		case 0x20000C:
			return ToaVBlankRegister();

		case 0x600000:
			return ToaScanlineRegister();

		case 0x700006:								// Player 1 inputs
			return DrvInput[0];
		case 0x700008:								// Player 2 inputs
			return DrvInput[1];
		case 0x70000A:								// Other inputs
			return DrvInput[2];
		case 0x700000:								// Dipswitch A
			return DrvInput[3];
		case 0x700002:								// Dipswitch B
			return DrvInput[4];
		case 0x700004:								// Dipswitch C - Territory
			return DrvInput[5];

		case 0x700010:
			return OKIM6295ReadStatus(0);
		case 0x700016:
			return YM2151ReadStatus(0);

		default: {
// 			printf("Attempt to read word value of location %x\n", sekAddress);

			if ((sekAddress & 0x00FF0000) == 0x00500000) {
				return ExtraTROM[(sekAddress & 0xFFFF) >> 1] | (ExtraTROM[0x8000 + ((sekAddress & 0xFFFF) >> 1)] << 8);
			}
		}
	}	
	return 0;
}

static void __fastcall Drv1WriteByte(unsigned int sekAddress, unsigned char byteValue)
{
	switch (sekAddress) {
		case 0x700011:
			OKIM6295Command(0, byteValue);
			break;

		case 0x700015:
			ToaYM2151SelectRegister(byteValue);
			break;
		case 0x700017:
			ToaYM2151WriteRegister(byteValue);
			break;

		default: {
//			printf("Attempt to write byte value %x to location %x\n", byteValue, sekAddress);

			if ((sekAddress & 0x00FF0000) == 0x00500000) {
				ExtraTROM[(sekAddress & 0xFFFF) >> 1] = byteValue;
			}
		}
	}
}

static void __fastcall Drv1WriteWord(unsigned int sekAddress, unsigned short wordValue)
{
	switch (sekAddress) {
		case 0x200000:								// Set GP9001 VRAM address-pointer
			ToaGP9001SetRAMPointer(wordValue);
			break;

		case 0x200004:
		case 0x200006:
			ToaGP9001WriteRAM(wordValue, 0);
			break;

		case 0x200008:
			ToaGP9001SelectRegister(wordValue);
			break;

		case 0x20000C:
			ToaGP9001WriteRegister(wordValue);
			break;

		case 0x700010:
			OKIM6295Command(0, wordValue & 0xFF);
			break;
		case 0x700014:
			ToaYM2151SelectRegister(wordValue);
			break;
		case 0x700016:
			ToaYM2151WriteRegister(wordValue);
			break;

		default: {
//			printf("Attempt to write word value %x to location %x\n", wordValue, sekAddress);

			if ((sekAddress & 0x00FF0000) == 0x00500000) {
				ExtraTROM[(sekAddress & 0xFFFF) >> 1] = wordValue & 0xFF;
				ExtraTROM[0x8000 + ((sekAddress & 0xFFFF) >> 1)] = wordValue << 8;
			}
		}
	}
}

static int DrvExit()
{
	OKIM6295Exit(0);
	ToaYM2151Exit();

	ToaPalExit();

	ToaExtraTextExit();
	ToaExitGP9001();
	SekExit();				// Deallocate 68000s

	// Deallocate all used memory
	free(Mem);
	Mem = NULL;

	return 0;
}

static int DrvDoReset()
{
	SekOpen(0);
	SekReset();
	SekClose();

	return 0;
}

static int DrvDraw()
{
	ToaClearScreen();
	
	if (bDrawScreen) {
		ToaGetBitmap();		
		ToaRenderGP9001();					// Render GP9001 graphics
		ToaExtraTextLayer();				// Render extra text layer
	}
	
	ToaPalUpdate();							// Update the palette

	return 0;
}

inline static int CheckSleep(int)
{
	return 0;
}

static int DrvFrame()
{
	int nCyclesVBlank;
	int nInterleave = 4;

	if (DrvReset) {														// Reset machine
		DrvDoReset();
	}

	// Compile digital inputs
	DrvInput[0] = 0x00;													// Buttons
	DrvInput[1] = 0x00;													// Player 1
	DrvInput[2] = 0x00;													// Player 2
	for (int i = 0; i < 8; i++) {
		DrvInput[0] |= (DrvJoy1[i] & 1) << i;
		DrvInput[1] |= (DrvJoy2[i] & 1) << i;
		DrvInput[2] |= (DrvButton[i] & 1) << i;
	}
	ToaClearOpposites(&DrvInput[0]);
	ToaClearOpposites(&DrvInput[1]);

	nCyclesTotal[0] = TOA_68K_SPEED / 60;
	nCyclesDone[0] = 0;

	nCyclesVBlank = nCyclesTotal[0] - ((nCyclesTotal[0] * TOA_VBLANK_LINES) / 0x0106);
	bVBlank = false;

	int nSoundBufferPos = 0;

	SekOpen(0);

	for (int i = 0; i < nInterleave; i++) {
    	int nCurrentCPU;
		int nNext;

		// Run 68000

		nCurrentCPU = 0;
		nNext = (i + 1) * nCyclesTotal[nCurrentCPU] / nInterleave;

		// Trigger VBlank interrupt
		if (!bVBlank && nNext > nCyclesVBlank) {
			if (nCyclesDone[nCurrentCPU] < nCyclesVBlank) {
				nCyclesSegment = nCyclesVBlank - nCyclesDone[nCurrentCPU];
				nCyclesDone[nCurrentCPU] += SekRun(nCyclesSegment);
			}

			ToaBufferGP9001Sprites();
			
			bVBlank = true;
			SekInterrupt(2);
		}

		nCyclesSegment = nNext - nCyclesDone[nCurrentCPU];
		if (bVBlank || (!CheckSleep(nCurrentCPU))) {					// See if this CPU is busywaiting
			nCyclesDone[nCurrentCPU] += SekRun(nCyclesSegment);
		} else {
			nCyclesDone[nCurrentCPU] += nCyclesSegment;
		}

		{
			// Render sound segment
			if (pBurnSoundOut) {
				int nSegmentLength = nBurnSoundLen / nInterleave;
				short* pSoundBuf = pBurnSoundOut + (nSoundBufferPos << 1);
				YM2151UpdateOne(0, pYM2151Buffer, nSegmentLength);
				for (int n = 0; n < nSegmentLength; n++) {
					pSoundBuf[(n << 1) + 0] = pYM2151Buffer[0][n] >> 1;
					pSoundBuf[(n << 1) + 1] = pYM2151Buffer[1][n] >> 1;
				}
				OKIM6295Render(0, pSoundBuf, nSegmentLength);
				nSoundBufferPos += nSegmentLength;
			}
		}
	}

	SekClose();

	{
		// Make sure the buffer is entirely filled.
		if (pBurnSoundOut) {
			int nSegmentLength = nBurnSoundLen - nSoundBufferPos;
			short* pSoundBuf = pBurnSoundOut + (nSoundBufferPos << 1);
			if (nSegmentLength) {
				YM2151UpdateOne(0, pYM2151Buffer, nSegmentLength);
				for (int n = 0; n < nSegmentLength; n++) {
					pSoundBuf[(n << 1) + 0] = pYM2151Buffer[0][n] >> 1;
					pSoundBuf[(n << 1) + 1] = pYM2151Buffer[1][n] >> 1;
				}
				OKIM6295Render(0, pSoundBuf, nSegmentLength);
			}
		}
	}

	if (pBurnDraw != NULL) {
		DrvDraw();												// Draw screen if needed
	}
	
	return 0;
}

// This routine is called first to determine how much memory is needed (MemEnd-(unsigned char *)0),
// and then afterwards to set up all the pointers
static int MemIndex()
{
	unsigned char *Next; Next = Mem;
	Rom01		= Next; Next += 0x080000;		//
	GP9001ROM[0]= Next; Next += nGP9001ROMSize[0];	// GP9001 tile data
	OKIM6295ROM	= Next; Next += 0x080000;
	RamStart	= Next;
	Ram01		= Next; Next += 0x010000;		// CPU #0 work RAM
	ExtraTROM	= Next; Next += 0x010000;		// Extra tile layer
	ExtraTRAM	= Next; Next += 0x002000;		// Extra tile layer
	ExtraTOffset= Next; Next += 0x001000;		//
	ExtraTScroll= Next; Next += 0x001000;		//
	RamPal		= Next; Next += 0x001000;		// palette
	GP9001RAM[0]= Next; Next += 0x004000;
	GP9001Reg[0]= (unsigned short*)Next; Next += 0x0100 * sizeof(short);
	RamEnd		= Next;
	ToaPalette	= (unsigned int *)Next; Next += nColCount * sizeof(unsigned int);
	MemEnd		= Next;

	return 0;
}

static int LoadRoms()
{
	// Load 68000 ROM
	BurnLoadRom(Rom01, 0, 1);
	BurnByteswap(Rom01, 0x080000);

	// Load GP9001 tile data
	ToaLoadGP9001Tiles(GP9001ROM[0], 1, 2, nGP9001ROMSize[0]);

	// Load OKIM6295 ADPCM data
	BurnLoadRom(OKIM6295ROM, 3, 1);
	return 0;
}

// Scan ram
static int DrvScan(int nAction,int *pnMin)
{
	struct BurnArea ba;

	if (nAction & 4) {					// Scan volatile ram
		if (pnMin != NULL) {			// Return minimum compatible version
			*pnMin = 0x020902;
		}

		memset(&ba, 0, sizeof(ba));
    	ba.Data		= RamStart;
		ba.nLen		= RamEnd - RamStart;
		ba.szName	= "RAM";
		BurnAcb(&ba);

		SekScan(nAction & 3);			// scan 68000 states

		OKIM6295Scan(0);
		ToaYM2151Scan(nAction);

		SCAN_VAR(DrvInput);
	}
	return 0;
}

static int DrvInit()
{
	int nLen;

	bRotatedScreen = true;
	nGP9001ROMSize[0] = 0x200000;

	// Find out how much memory is needed
	Mem = NULL;
	MemIndex();
	nLen = MemEnd - (unsigned char *)0;
	if ((Mem = (unsigned char *)malloc(nLen)) == NULL) {
		return 1;
	}
	memset(Mem, 0, nLen);										// blank all memory
	MemIndex();													// Index the allocated memory

	// Load the roms into memory
	if (LoadRoms()) {
		return 1;
	}

	{
		SekInit(1);												// Allocate 68000
		SekExt[0].ResetCallback = DrvResetCallback;				// Get cpu 1 reset requests

		// Map 68000 memory:
	    SekOpen(0);
		SekMemory(Rom01,		0x000000, 0x07FFFF, SM_ROM);	// CPU 0 ROM
		SekMemory(Ram01,		0x100000, 0x10FFFF, SM_RAM);
		SekMemory(RamPal,		0x300000, 0x300FFF, SM_RAM);	// Palette RAM
		SekMemory(ExtraTRAM,	0x400000, 0x401FFF, SM_RAM);
		SekMemory(ExtraTScroll,	0x402000, 0x402FFF, SM_RAM);	// 0x502000 - Scroll; 0x502200 - RAM
		SekMemory(ExtraTOffset,	0x403000, 0x403FFF, SM_RAM);	// 0x203000 - Offset; 0x503200 - RAM

		SekExt[0].ReadWord	= Drv1ReadWord;
		SekExt[0].ReadByte	= Drv1ReadByte;
		SekExt[0].WriteWord	= Drv1WriteWord;
		SekExt[0].WriteByte	= Drv1WriteByte;

		SekClose();
	}

	nLayer0XOffset = -0x01D6;
	nLayer1XOffset = -0x01D8;
	nLayer2XOffset = -0x01DA;

	nSpriteYOffset = 0x0001;
	ToaInitGP9001();

	ToaExtraTextInit();

	nToaPalLen = nColCount;
	ToaPalSrc = RamPal;
	ToaPalInit();

	ToaYM2151Init(27000000 / 8);
	OKIM6295Init(0, 27000000 / 10 / 132, 50.0);

	bDrawScreen = true;

	DrvDoReset(); // Reset machine
	return 0;
}

// Rom information
static struct BurnRomInfo truxton2RomDesc[] = {
	{"tp024_1.bin", 0x080000, 0xF5CFE6EE, 0x10}, //  0 CPU #0 code

	{"tp024_4.bin", 0x100000, 0x805C449E,    1}, //  1 GP9001 Tile data
	{"tp024_3.bin", 0x100000, 0x47587164,    1}, //  2

	{"tp024_2.bin", 0x080000, 0xF2F6CAE4,    2}, //  3 OKIM6295 ADPCM data
};

STD_ROM_PICK(truxton2);
STD_ROM_FN(truxton2);


struct BurnDriver BurnDrvTruxton2 = {
	{"truxton2", "Truxton II /  Tatsujin Oh", NULL, "Toaplan", "Toaplan", "1992", NULL, NULL},
	1, 2, HARDWARE_TOAPLAN_68K_ONLY,
	NULL, truxton2RomInfo, truxton2RomName, truxton2InputInfo, truxton2DIPInfo,
	DrvInit, DrvExit, DrvFrame, DrvDraw, DrvScan, &ToaRecalcPalette,
	240, 320, 3, 4
};

