#include "toaplan.h"
// Battle Garegga

static unsigned char DrvButton[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvJoy1[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvJoy2[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char DrvInput[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static unsigned char DrvReset = 0;
static unsigned char bDrawScreen;
static bool bVBlank;

static int nSoundCommand;

// Z80 ROM bank
static int nCurrentBank;


// Rom information
static struct BurnRomInfo battlegRomDesc[] = {
	{"u123",	 0x080000, 0x88A4E66A, 0x10}, //  0 CPU #0 code (even)
	{"u65",		 0x080000, 0x5DEA32A3, 0x10}, //  1				(odd)

	{"rom4.bin", 0x200000, 0xB333D81F,    1}, //  2 GP9001 Tile data
	{"rom3.bin", 0x200000, 0x51B9EBFB,    1}, //  3
	{"rom2.bin", 0x200000, 0xB330E5E2,    1}, //  4
	{"rom1.bin", 0x200000, 0x7EAFDD70,    1}, //  5

	{"text.u81", 0x008000, 0xE67FD534,    1}, //  6 Extra text layer tile data

	{"snd.bin",	 0x020000, 0x68632952, 0x10}, //  7 Z80 program

	{"rom5.bin", 0x100000, 0xF6D49863,    2}, //  8 OKIM6295 ADPCM data
};

STD_ROM_PICK(battleg);
STD_ROM_FN(battleg);

static struct BurnRomInfo battlegaRomDesc[] = {
	{"prg0.bin", 0x080000, 0xF80C2FC2, 0x10}, //  0 CPU #0 code (even)
	{"prg1.bin", 0x080000, 0x2CCFDD1E, 0x10}, //  1				(odd)

	{"rom4.bin", 0x200000, 0xB333D81F,    1}, //  2 GP9001 Tile data
	{"rom3.bin", 0x200000, 0x51B9EBFB,    1}, //  3
	{"rom2.bin", 0x200000, 0xB330E5E2,    1}, //  4
	{"rom1.bin", 0x200000, 0x7EAFDD70,    1}, //  5

	{"text.u81", 0x008000, 0xE67FD534,    1}, //  6 Extra text layer tile data

	{"snd.bin",	 0x020000, 0x68632952, 0x10}, //  7 Z80 program

	{"rom5.bin", 0x100000, 0xF6D49863,    2}, //  8 OKIM6295 ADPCM data
};

STD_ROM_PICK(battlega);
STD_ROM_FN(battlega);

static struct BurnInputInfo battlegInputList[] = {
	{"P1 Coin",		0, DrvButton + 3,	"p1 coin"},
	{"P1 Start",	0, DrvButton + 5,	"p1 start"},

	{"P1 Up",		0, DrvJoy1 + 0, 	"p1 up"},
	{"P1 Down",		0, DrvJoy1 + 1, 	"p1 down"},
	{"P1 Left",		0, DrvJoy1 + 2, 	"p1 left"},
	{"P1 Right",	0, DrvJoy1 + 3, 	"p1 right"},
	{"P1 Button 1",	0, DrvJoy1 + 4,		"p1 fire 1"},
	{"P1 Button 2",	0, DrvJoy1 + 5,		"p1 fire 2"},
	{"P1 Button 3",	0, DrvJoy1 + 6,		"p1 fire 3"},

	{"P2 Coin",		0, DrvButton + 4,	"p2 coin"},
	{"P2 Start",	0, DrvButton + 6,	"p2 start"},
	
	{"P2 Up",		0, DrvJoy2 + 0, 	"p2 up"},
	{"P2 Down",		0, DrvJoy2 + 1, 	"p2 down"},
	{"P2 Left",		0, DrvJoy2 + 2, 	"p2 left"},
	{"P2 Right",	0, DrvJoy2 + 3, 	"p2 right"},
	{"P2 Button 1",	0, DrvJoy2 + 4,		"p2 fire 1"},
	{"P2 Button 2",	0, DrvJoy2 + 5,		"p2 fire 2"},
	{"P2 Button 3",	0, DrvJoy2 + 6,		"p2 fire 3"},

	{"Reset",		0, &DrvReset,		"reset"},
	{"Diagnostics",	0, DrvButton + 0,	"diag"},
	{"Dip 1",		2, DrvInput + 3,	"dip"},
	{"Dip 2",		2, DrvInput + 4,	"dip"},
	{"Dip 3",		2, DrvInput + 5,	"dip"},
};

STDINPUTINFO(battleg);

static struct BurnDIPInfo battlegDIPList[] = {
	
	// Defaults
	{0x14,	0xFF, 0xFF,	0x00, NULL},
	{0x15,	0xFF, 0xFF,	0x00, NULL},
	
	// DIP 1
	{0,		0xFE, 0,	2,	  "Test mode"},
	{0x14,	0x01, 0x01,	0x00, "Normal"},
	{0x14,	0x01, 0x01,	0x01, "Test"},
	{0,		0xFE, 0,	2,	  "Starting coin"},
	{0x14,	0x01, 0x02,	0x00, "1 credit"},
	{0x14,	0x01, 0x02,	0x02, "2 credits"},
	{0,		0xFE, 0,	8,	  "Coin 1"},
	{0x14,	0x01, 0x1C, 0x00, "1 coin 1 credit"},
	{0x14,	0x01, 0x1C, 0x04, "1 coin 2 credits"},
	{0x14,	0x01, 0x1C, 0x08, "1 coin 3 credits"},
	{0x14,	0x01, 0x1C, 0x0C, "1 coin 3 credits"},
	{0x14,	0x01, 0x1C, 0x10, "2 coins 1 credit"},
	{0x14,	0x01, 0x1C, 0x14, "3 coins 1 credit"},
	{0x14,	0x01, 0x1C, 0x18, "4 coins 1 credit"},
	{0x14,	0x01, 0x1C, 0x1C, "Free Play"},
	// 0x1C: Free play settings active
	// Normal
	{0,		0xFE, 0,	7,	  "Coin 2"},
	{0x14,	0x82, 0xE0,	0x00, "1 coin 1 credit"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0x20, "1 coin 2 credits"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0x40, "1 coin 3 credits"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0x60, "1 coin 3 credits"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0x80, "2 coins 1 credit"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0xA0, "3 coins 1 credit"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0xC0, "4 coins 1 credit"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x82, 0xE0,	0x04, "1 coin 1 credit"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	// Free play
	{0,		0xFE, 0,	2,	  "Stick mode"},
	{0x14,	0x02, 0x20,	0x00, "Special"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x02, 0x20,	0x20, "Normal"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0,		0xFE, 0,	2,	  "Effect"},
	{0x14,	0x02, 0x40,	0x00, "Off"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x02, 0x40,	0x40, "On"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0,		0xFE, 0,	2,	  "Music"},
	{0x14,	0x02, 0x80,	0x00, "Off"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	{0x14,	0x02, 0x80,	0x80, "On"},
	{0x14,	0x00, 0x1C, 0x1C, NULL},
	
	// DIP 2
	{0,		0xFE, 0,	4,	  "Difficulty"},
	{0x15,	0x01, 0x03, 0x00, "Normal"},
	{0x15,	0x01, 0x03, 0x01, "Easy"},
	{0x15,	0x01, 0x03, 0x02, "Hard"},
	{0x15,	0x01, 0x03, 0x03, "Very hard"},
	{0,		0xFE, 0,	2,	  "Screen flip"},
	{0x15,	0x01, 0x04, 0x00, "Off"},
	{0x15,	0x01, 0x04, 0x04, "On"},
	{0,		0xFE, 0,	2,	  "Demo sound"},
	{0x15,	0x01, 0x08, 0x00, "On"},
	{0x15,	0x01, 0x08, 0x08, "Off"},
	{0,		0xFE, 0,	8,	  "Player counts"},
	{0x15,	0x01, 0x70, 0x00, "3"},
	{0x15,	0x01, 0x70, 0x10, "4"},
	{0x15,	0x01, 0x70, 0x20, "2"},
	{0x15,	0x01, 0x70, 0x30, "1"},
	{0x15,	0x01, 0x70, 0x40, "5"},
	{0x15,	0x01, 0x70, 0x50, "6"},
	{0x15,	0x01, 0x70, 0x60, "Multiple"},
	{0x15,	0x01, 0x70, 0x70, "Invincible"},
    // Nippon and U.S.A. regions
	{0,		0xFE, 0,	2,	  "Extra player"},
    {0x15,	0x02, 0x80, 0x00, "1000000 each"},
	{0x16,	0x00, 0x01, 0x00, NULL},
	{0x15,	0x02, 0x80, 0x80, "1000000 2000000"},
	{0x16,	0x00, 0x01, 0x00, NULL},
    // Europe/Denmark and Asia regions
	{0,		0xFE, 0,	2,	  "Extra player"},
	{0x15,	0x02, 0x80, 0x80, "2000000 each"},
	{0x16,	0x00, 0x01, 0x01, NULL},
	{0x15,	0x02, 0x80, 0x00, "No extra"},
	{0x16,	0x00, 0x01, 0x01, NULL},

	// DIP 3	
	{0,		0xFE, 0,	2,	  "Continue play"},
	{0x16,	0x01, 0x04, 0x00, "On"},
	{0x16,	0x01, 0x04, 0x04, "Off"},
};
	
static struct BurnDIPInfo battlegRegionDIPList[] = {	
	// Region
	{0x16,	0xFF, 0x0F,	0x01, NULL},
	{0,		0xFE, 0,	2,	  "Region"},
	{0x16,	0x01, 0x03, 0x01, "Denmark"},
	{0x16,	0x01, 0x03, 0x03, "China"},
	{0x16,	0x01, 0x03, 0x00, "Nippon [illegal setting]"},
	{0x16,	0x01, 0x03, 0x02, "U.S.A. [illegal setting]"},
};

static struct BurnDIPInfo battlegaRegionDIPList[] = {
	// DIP 3	
	{0,		0xFE, 0,	2,	  "Stage edit"},
	{0x16,	0x01, 0x08, 0x00, "Disable"},
	{0x16,	0x01, 0x08, 0x08, "Enable"},
	
	// Region
	{0x16,	0xFF, 0x0F,	0x00, NULL},
	{0,		0xFE, 0,	4,	  "Region"},
	{0x16,	0x01, 0x03, 0x00, "Nippon"},
	{0x16,	0x01, 0x03, 0x01, "Europe"},
	{0x16,	0x01, 0x03, 0x02, "U.S.A."},
	{0x16,	0x01, 0x03, 0x03, "Asia"},
};

STDDIPINFOEXT(battleg, battleg, battlegRegion);
STDDIPINFOEXT(battlega, battleg, battlegaRegion);

static int __fastcall DrvResetCallback()
{
	// Reset instruction on 68000

	ZetReset();						// Reset Z80 (CPU #1)

	return 0;
}

static unsigned char *Mem = NULL, *MemEnd = NULL;
static unsigned char *RamStart, *RamEnd;
static unsigned char *Rom01;
static unsigned char *Ram01, *Ram02, *RamPal;

static int nColCount = 0x0800;
static int nOKIM6295ROMSize = 0x100000;

// This routine is called first to determine how much memory is needed (MemEnd-(unsigned char *)0),
// and then afterwards to set up all the pointers
static int MemIndex()
{
	unsigned char *Next; Next = Mem;
	Rom01		= Next; Next += 0x100000;			//
	RomZ80		= Next; Next += 0x020000;			// Z80 ROM
	GP9001ROM[0]= Next; Next += nGP9001ROMSize[0];	// GP9001 tile data
	ExtraTROM	= Next; Next += 0x010000;			// Extra Text layer tile data
	OKIM6295ROM	= Next; Next += nOKIM6295ROMSize;	// ADPCM data
	RamStart	= Next;
	Ram01		= Next; Next += 0x010000;			// CPU #0 work RAM
	Ram02		= Next; Next += 0x000800;			//
	ExtraTRAM	= Next; Next += 0x002000;			// Extra tile layer
	ExtraTOffset= Next; Next += 0x001000;			//
	ExtraTScroll= Next; Next += 0x001000;			//
	RamPal		= Next; Next += 0x001000;			// palette
	RamZ80		= Next; Next += 0x004000;			// Z80 RAM
	GP9001RAM[0]= Next; Next += 0x004000;
	GP9001Reg[0]= (unsigned short*)Next; Next += 0x0100 * sizeof(short);
	RamEnd		= Next;
	ToaPalette	= (unsigned int *)Next; Next += nColCount * sizeof(unsigned int);
	MemEnd		= Next;

	return 0;
}

// Scan ram
static int DrvScan(int nAction,int *pnMin)
{
	struct BurnArea ba;

	if (nAction & 4) {					// Scan volatile ram
		if (pnMin != NULL) {			// Return minimum compatible version
			*pnMin = 0x020902;
		}

		memset(&ba, 0, sizeof(ba));
    	ba.Data		= RamStart;
		ba.nLen		= RamEnd-RamStart;
		ba.szName	= "All Ram";
		BurnAcb(&ba);

		SekScan(nAction & 3);			// scan 68000 states
		ZetScan(nAction);				// Scan Z80
		SCAN_VAR(nCurrentBank);

		OKIM6295Scan(0);
		ToaYM2151Scan(nAction);

		SCAN_VAR(DrvInput);
		SCAN_VAR(nSoundCommand);
		
	}
	return 0;
}

static int LoadRoms()
{
	// Load 68000 ROM
	if (ToaLoadCode(Rom01, 0, 2)) {
		return 1;
	}

	// Load GP9001 tile data
	ToaLoadGP9001Tiles(GP9001ROM[0], 2, 4, nGP9001ROMSize[0]);

	// Load Extra text layer tile data
	BurnLoadRom(ExtraTROM, 6, 1);

	// Load the Z80 ROM
	BurnLoadRom(RomZ80, 7, 1);

	// Load ADPCM data
	BurnLoadRom(OKIM6295ROM, 8, 1);

	return 0;
}

static unsigned char DrvZ80Read(unsigned short nAddress)
{
//	printf("z80 read %4X\n", nAddress);
	switch (nAddress) {
		case 0xE001:
			return YM2151ReadStatus(0);
		
		case 0xE004:
			return OKIM6295ReadStatus(0);
		
		case 0xE01C:
			return nSoundCommand;
		
		case 0xE01D:
			// Bit 0 enables/disables interrupt processing (0 = enabled)
			return 0;
	}
	return 0;
}

static void DrvZ80Write(unsigned short nAddress, unsigned char nValue)
{
//	printf("z80 wrote %4X with %2X\n", nAddress, nValue);
	switch (nAddress) {
		case 0xE000:
			ToaYM2151SelectRegister(nValue);
			break;
		case 0xE001:
			ToaYM2151WriteRegister(nValue);
			break;
		
		case 0xE004:
			OKIM6295Command(0, nValue);
			break;
		
		case 0xE006:
			OKIM6295SampleInfo[0][0] = OKIM6295ROM + ((nValue & 0x0F) << 16);
			OKIM6295SampleData[0][0] = OKIM6295ROM + ((nValue & 0x0F) << 16);
			OKIM6295SampleInfo[0][1] = OKIM6295ROM + ((nValue & 0xF0) << 12) + 0x0100;
			OKIM6295SampleData[0][1] = OKIM6295ROM + ((nValue & 0xF0) << 12);
			break;
		case 0xE008:
			OKIM6295SampleInfo[0][2] = OKIM6295ROM + ((nValue & 0x0F) << 16) + 0x0200;
			OKIM6295SampleData[0][2] = OKIM6295ROM + ((nValue & 0x0F) << 16);
			OKIM6295SampleInfo[0][3] = OKIM6295ROM + ((nValue & 0xF0) << 12) + 0x0300;
			OKIM6295SampleData[0][3] = OKIM6295ROM + ((nValue & 0xF0) << 12);
			break;
		
		case 0xE00A: {
			nValue &= 0x07;
  			if (nValue != nCurrentBank) {
  				unsigned char* nStartAddress = RomZ80 + (nValue << 14);
  				nCurrentBank = nValue;
  				ZetMapArea(0x8000, 0xBFFF, 0, nStartAddress);
	  			ZetMapArea(0x8000, 0xBFFF, 2, nStartAddress);
			}
			break;
  		}
		
		case 0xE00C:
			// Once a sound command is processed, it is written to this address
			// printf("z80 wrote %4X -> %2X\n", nAddress, nValue);
			break;
	
	}
}

static int DrvZ80Init()
{
	ZetRead = DrvZ80Read;
	ZetWrite = DrvZ80Write;

	ZetInit();
	// ROM bank 1
	ZetMapArea    (0x0000, 0x7FFF, 0, RomZ80 + 0x0000); // Direct Read from ROM
	ZetMapArea    (0x0000, 0x7FFF, 2, RomZ80 + 0x0000); // Direct Fetch from ROM
	// ROM bank 2
	ZetMapArea    (0x8000, 0xBFFF, 0, RomZ80 + 0x8000); // Direct Read from ROM
	ZetMapArea    (0x8000, 0xBFFF, 2, RomZ80 + 0x8000); // Direct Fetch from ROM
	// RAM
	ZetMapArea    (0xC000, 0xDFFF, 0, RamZ80);			// Direct Read from RAM
	ZetMapArea    (0xC000, 0xDFFF, 1, RamZ80);			// Direct Write to RAM
	ZetMapArea    (0xC000, 0xDFFF, 2, RamZ80);			//
	// Ports
	ZetMemCallback(0xE000, 0xE0FF, 0);					// Read
	ZetMemCallback(0xE000, 0xE0FF, 1);					// Write

	ZetMemEnd();

	nCurrentBank = 2;

	return 0;
}

static unsigned char __fastcall Drv1ReadByte(unsigned int sekAddress)
{
	switch (sekAddress) {
		
		case 0x218021:								// The 68K has access to the Z80 RAM
			return RamZ80[0x10];					// Only these addresses are used, however
		case 0x218023:								//
			return RamZ80[0x11];					//

		case 0x21C021:								// Player 1 inputs
			return DrvInput[0];
		case 0x21C025:								// Player 2 inputs
			return DrvInput[1];
		case 0x21C029:								// Other inputs
			return DrvInput[2];
		case 0x21C02D:								// Dipswitch A
			return DrvInput[3];
		case 0x21C031:			   					// Dipswitch B
			return DrvInput[4];
		case 0x21C035:								// Dipswitch C - Territory
			return DrvInput[5];

//		default:
//			printf("Attempt to read byte value of location %x\n", sekAddress);
	}
	return 0;
}

static unsigned short __fastcall Drv1ReadWord(unsigned int sekAddress)
{
	switch (sekAddress) {

		case 0x21C03C:
			return ToaScanlineRegister();

		case 0x300004:
			return ToaGP9001ReadRAM_Hi(0);
		case 0x300006:
			return ToaGP9001ReadRAM_Lo(0);

//		default:
//			printf("Attempt to read word value of location %x\n", sekAddress);
	}
	return 0;
}

static void __fastcall Drv1WriteByte(unsigned int sekAddress, unsigned char byteValue)
{
	switch (sekAddress) {
		
		case 0x600001:
			nSoundCommand = byteValue;
			
			// Trigger Z80 interrupt, and allow the Z80 to process it
			ZetRaiseIrq(255);
			ZetRun(0x0200);
			nCyclesDone[1] += 0x0200;
			break;

//		default:
//			printf("Attempt to write byte value %x to location %x\n", byteValue, sekAddress);
	}
}

static void __fastcall Drv1WriteWord(unsigned int sekAddress, unsigned short wordValue)
{
	switch (sekAddress) {

		case 0x300000:								// Set GP9001 VRAM address-pointer
			ToaGP9001SetRAMPointer(wordValue);
			break;

		case 0x300004:
		case 0x300006:
			ToaGP9001WriteRAM(wordValue, 0);
			break;

		case 0x300008:
			ToaGP9001SelectRegister(wordValue);
			break;

		case 0x30000C: {
			ToaGP9001WriteRegister(wordValue);
			break;
		}

//		default:
//			printf("Attempt to write word value %x to location %x\n", wordValue, sekAddress);
	}
}

static int DrvDoReset()
{
	SekOpen(0);
	SekReset();
	SekClose();

	ZetReset();
	return 0;
}

static int battlegInit()
{
	int nLen;

	bRotatedScreen = true;
	nGP9001ROMSize[0] = 0x800000;

	// Find out how much memory is needed
	Mem = NULL;
	MemIndex();
	nLen = MemEnd - (unsigned char *)0;
	if ((Mem = (unsigned char *)malloc(nLen)) == NULL) {
		return 1;
	}
	memset(Mem, 0, nLen);										// blank all memory
	MemIndex();													// Index the allocated memory

	// Load the roms into memory
	if (LoadRoms()) {
		return 1;
	}

	{
		SekInit(1);												// Allocate 68000
		SekExt[0].ResetCallback = DrvResetCallback;				// Get cpu 1 reset requests

		// Map 68000 memory:
	    SekOpen(0);
		SekMemory(Rom01,		0x000000, 0x0FFFFF, SM_ROM);	// CPU 0 ROM
		SekMemory(Ram01,		0x100000, 0x10FFFF, SM_RAM);
		SekMemory(RamPal,		0x400000, 0x400FFF, SM_RAM);	// Palette RAM
		SekMemory(Ram02,		0x401000, 0x4017FF, SM_RAM);	// Unused
		SekMemory(ExtraTRAM,	0x500000, 0x501FFF, SM_RAM);
		SekMemory(ExtraTScroll,	0x502000, 0x502FFF, SM_RAM);	// 0x502000 - Scroll; 0x502200 - RAM
		SekMemory(ExtraTOffset,	0x503000, 0x503FFF, SM_RAM);	// 0x203000 - Offset; 0x503200 - RAM

		SekExt[0].ReadWord	= Drv1ReadWord;
		SekExt[0].ReadByte	= Drv1ReadByte;
		SekExt[0].WriteWord	= Drv1WriteWord;
		SekExt[0].WriteByte	= Drv1WriteByte;

		SekClose();
	}

	nSpriteXOffset = 0x0024;
	nSpriteYOffset = 0x0001;

	nLayer0XOffset = -0x01D6;
	nLayer1XOffset = -0x01D8;
	nLayer2XOffset = -0x01DA;

	ToaInitGP9001();

	nExtraTXOffset = 0x2C;
	ToaExtraTextInit();

	DrvZ80Init();												// Initialize Z80

	ToaYM2151Init(32000000 / 8);
	OKIM6295Init(0, 32000000 / 16 / 132, 50.0);

	nToaPalLen = nColCount;
	ToaPalSrc = RamPal;
	ToaPalInit();

	bDrawScreen = true;

	DrvDoReset();												// Reset machine
	return 0;
}

static int DrvExit()
{
	OKIM6295Exit(0);
	ToaYM2151Exit();

	ToaPalExit();
	ToaExitGP9001();
	ToaExtraTextExit();
	ToaZExit();				// Z80 exit
	SekExit();				// Deallocate 68000s

	// Deallocate all used memory
	free(Mem);
	Mem = NULL;

	return 0;
}

static int DrvDraw()
{
	ToaClearScreen();
	
	if (bDrawScreen) {
		ToaGetBitmap();		
		ToaRenderGP9001();					// Render GP9001 graphics
		ToaExtraTextLayer();				// Render extra text layer
	}
	
	ToaPalUpdate();							// Update the palette

	return 0;
}

inline static int CheckSleep(int)
{
	return 0;
}

static int DrvFrame()
{
	int nCyclesVBlank;
	int nInterleave = 16;

	if (DrvReset) {														// Reset machine
		DrvDoReset();
	}

	// Compile digital inputs
	DrvInput[0] = 0x00;													// Buttons
	DrvInput[1] = 0x00;													// Player 1
	DrvInput[2] = 0x00;													// Player 2
	for (int i = 0; i < 8; i++) {
		DrvInput[0] |= (DrvJoy1[i] & 1) << i;
		DrvInput[1] |= (DrvJoy2[i] & 1) << i;
		DrvInput[2] |= (DrvButton[i] & 1) << i;
	}
	ToaClearOpposites(&DrvInput[0]);
	ToaClearOpposites(&DrvInput[1]);

	nCyclesTotal[0] = TOA_68K_SPEED / 60;
	nCyclesTotal[1] = TOA_Z80_SPEED / 60;
	nCyclesDone[0] = nCyclesDone[1] = 0;

	nCyclesVBlank = nCyclesTotal[0] - ((nCyclesTotal[0] * TOA_VBLANK_LINES) / 0x0106);
	bVBlank = false;

	int nSoundBufferPos = 0;

	SekOpen(0);
	for (int i = 0; i < nInterleave; i++) {
    	int nCurrentCPU;
		int nNext;

		// Run 68000

		nCurrentCPU = 0;
		nNext = (i + 1) * nCyclesTotal[nCurrentCPU] / nInterleave;

		// Trigger VBlank interrupt
		if (!bVBlank && nNext > nCyclesVBlank) {
			if (nCyclesDone[nCurrentCPU] < nCyclesVBlank) {
				nCyclesSegment = nCyclesVBlank - nCyclesDone[nCurrentCPU];
				nCyclesDone[nCurrentCPU] += SekRun(nCyclesSegment);
			}

			ToaBufferGP9001Sprites();
			
			bVBlank = true;
			SekInterrupt(4);
		}

		nCyclesSegment = nNext - nCyclesDone[nCurrentCPU];
		if (bVBlank || (!CheckSleep(nCurrentCPU))) {					// See if this CPU is busywaiting
			nCyclesDone[nCurrentCPU] += SekRun(nCyclesSegment);
		} else {
			nCyclesDone[nCurrentCPU] += nCyclesSegment;
		}

		// Run Z80
		nCurrentCPU = 1;
		nNext = (i + 1) * nCyclesTotal[nCurrentCPU] / nInterleave;
		nCyclesSegment = nNext - nCyclesDone[nCurrentCPU];
		nCyclesSegment = ZetRun(nCyclesSegment);
		nCyclesDone[nCurrentCPU] += nCyclesSegment;

		{
			// Render sound segment
			if (pBurnSoundOut) {
				int nSegmentLength = nBurnSoundLen / nInterleave;
				short* pSoundBuf = pBurnSoundOut + (nSoundBufferPos << 1);
				YM2151UpdateOne(0, pYM2151Buffer, nSegmentLength);
				for (int n = 0; n < nSegmentLength; n++) {
					pSoundBuf[(n << 1) + 0] = pYM2151Buffer[0][n] >> 1;
					pSoundBuf[(n << 1) + 1] = pYM2151Buffer[1][n] >> 1;
				}
				OKIM6295Render(0, pSoundBuf, nSegmentLength);
				nSoundBufferPos += nSegmentLength;
			}
		}
	}
	
	SekClose();

	{
		// Make sure the buffer is entirely filled.
		if (pBurnSoundOut) {
			int nSegmentLength = nBurnSoundLen - nSoundBufferPos;
			short* pSoundBuf = pBurnSoundOut + (nSoundBufferPos << 1);
			if (nSegmentLength) {
				YM2151UpdateOne(0, pYM2151Buffer, nSegmentLength);
				for (int n = 0; n < nSegmentLength; n++) {
					pSoundBuf[(n << 1) + 0] = pYM2151Buffer[0][n] >> 1;
					pSoundBuf[(n << 1) + 1] = pYM2151Buffer[1][n] >> 1;
				}
				OKIM6295Render(0, pSoundBuf, nSegmentLength);
			}
		}
	}

	if (pBurnDraw) {
		DrvDraw();																// Draw screen if needed
	}

	return 0;
}

struct BurnDriver BurnDrvBattleG = {
	{"battleg", "Battle Garegga (ver. Tue Apr 2 1996)", "Type 2 (China/Denmark region only, round bullets only)", "Raizing", "Toaplan", "1996", NULL, NULL},
	1, 2, HARDWARE_TOAPLAN_RAIZING,
	NULL, battlegRomInfo, battlegRomName, battlegInputInfo, battlegDIPInfo,
	battlegInit, DrvExit, DrvFrame, DrvDraw, DrvScan, &ToaRecalcPalette,
	240, 320, 3, 4
};

struct BurnDriver BurnDrvBattleGA = {
	{"battlega", "Battle Garegga (ver. Sat Feb 3 1996)", NULL, "Raizing", "Toaplan", "1996", "battleg", NULL},
	1, 2, HARDWARE_TOAPLAN_RAIZING,
	NULL, battlegaRomInfo, battlegaRomName, battlegInputInfo, battlegaDIPInfo,
	battlegInit, DrvExit, DrvFrame, DrvDraw, DrvScan, &ToaRecalcPalette,
	240, 320, 3, 4
};

