#pragma code_seg("C10")
#pragma data_seg("D10")
#pragma bss_seg("B10")
#pragma const_seg("K10")
#pragma comment(linker, "/merge:D10=10")
#pragma comment(linker, "/merge:C10=10")
#pragma comment(linker, "/merge:B10=10")
#pragma comment(linker, "/merge:K10=10")
/***************************************************************************

Pac-Land  (c) 1984 Namco

Ernesto Corvi
ernesto@imagina.com

Custom ICs:
----------
04XX     sprite address generator
11XX     gfx data shifter and mixer (16-bit in, 4-bit out)
12XX     sprite generator
13XX     dual scrolling tilemap address generator
98XX     lamp/coin output
99XX     sound volume
CUS27    clock divider
CUS29    sprite line buffer and sprite/tilemap mixer
CUS30    sound control
CUS34A   address decoder
CUS36    dual tilemap generator
CUS60    MCU (63701) [1]

[1] 60A1 found in at least one version


Memory map
----------
Part of the address decoding is done by a custom IC (CUS34A), so the memory
map is inferred by program behaviour. The CUS34A also handles internally the
main and sub irq, some latches, and a watchdog.
Also, most of the address decoding for the MCU is done by CUS30.

MAIN CPU:

Address          Dir Data     Name      Description
---------------- --- -------- --------- -----------------------
0000xxxxxxxxxxx0 R/W xxxxxxxx RAM2      tilemap RAM
0000xxxxxxxxxxx1 R/W xxxxxxxx RAM3      tilemap RAM
0001xxxxxxxxxxx0 R/W xxxxxxxx RAM0      tilemap RAM
0001xxxxxxxxxxx1 R/W xxxxxxxx RAM1      tilemap RAM
00100xxxxxxxxxxx R/W xxxxxxxx ORAM0     work RAM
001001111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (sprite number & color)
00101xxxxxxxxxxx R/W xxxxxxxx ORAM1     work RAM
001011111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x, y)
00110xxxxxxxxxxx R/W xxxxxxxx ORAM2     work RAM
001101111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x msb, flip, size)
0011100--------x   W xxxxxxxx POSI/S    fg X scroll (9-bit data: A0 is the msb)
0011101--------x   W xxxxxxxx POSI/W    bg X scroll (9-bit data: A0 is the msb)
0011110---------   W -----xxx BANK      ROM bank select
0011110---------   W ----x--- SEASON0   \ palette bank select
0011110---------   W ---x---- SEASON1   /
0011110---------   W --x----- PBC       ?? tilemap enable ?? (doesn't seem to be used)
0011111---------              n.c.
010xxxxxxxxxxxxx R   xxxxxxxx ROM 8EFHJ banked program ROMs
01101xxxxxxxxxxx R/W xxxxxxxx RAM 3J/3K sound RAM (through CUS30, shared with MCU) [1]
01101000xxxxxxxx R/W xxxxxxxx           portion holding the sound wave data
0110100100xxxxxx R/W xxxxxxxx           portion holding the sound registers
0111x-----------   W --------           main CPU irq enable (data is in A11) (IRQ1 generated by CUS34A)
01111----------- R   --------           watchdog reset (RESET generated by CUS34A)
1000x-----------   W -------- SUBRESET  reset MCU (data is in A11) (latch in CUS34A)
1001x-----------   W -------- FLIP      flip screen (data is in A11) (latch in CUS34A)
10xxxxxxxxxxxxxx R   xxxxxxxx ROM 8B    program ROM
11xxxxxxxxxxxxxx R   xxxxxxxx ROM 8D    program ROM

[1] two 1024x4bit 8148 RAMs; can be replaced by one 2048x8bit 2018 RAM @ 3L


MCU:

Address          Dir Data     Name      Description
---------------- --- -------- --------- -----------------------
00000000xxxxxxxx                        MCU internal registers, timers, ports and RAM
0001-xxxxxxxxxxx R/W xxxxxxxx RAM 3J/3K sound RAM (through CUS30, shared with main CPU) [1]
0001-000xxxxxxxx R/W xxxxxxxx           portion holding the sound wave data
0001-00100xxxxxx R/W xxxxxxxx           portion holding the sound registers
001------------- R/W --------           watchdog reset? (CUS34A)
01x-------------   W --------           sound CPU irq enable (data is in A13) (IRQ2 generated by CUS34A)
10xxxxxxxxxxxxxx R   xxxxxxxx ROM 3E    program ROM
1100-xxxxxxxxxxx R/W xxxxxxxx RAM 4E    work RAM
1101----------00 R   xxxx----           dip SWA 1-4
1101----------00 R   ----xxxx           dip SWB 1-4
1101----------01 R   xxxx----           dip SWA 5-8
1101----------01 R   ----xxxx           dip SWB 5-8
1101----------10 R   xxxxxxxx           switch inputs
1101----------11 R   xxxxxxxx           switch inputs
1111xxxxxxxxxxxx R   xxxxxxxx           MCU internal ROM

[1] two 1024x4bit 8148 RAMs; can be replaced by one 2048x8bit 2018 RAM @ 3L


ROM locations:
PL5_01B -> 8B
PL5_02  -> 8D
PL1_03  -> 8E
PL1_04  -> 8F
PL1_05  -> 8H
PL1_06  -> 8J
PL4_10  -> 7E
PL4_08  -> 6E
PL4_11  -> 7F
PL4_09  -> 6F
PL1_07  -> 3E
PL2_12  -> 6N
PL4_13  -> 6T
PL1-3   -> 6L
PL1-5   -> 5T
PL1-4   -> 4N
PL1-2   -> 1T
PL1-1   -> 1R



Pacland
Namco, 1984

PCB Layout
----------

2234961101 (2234963101)
|------------------------------------------------------|
| PL1-2.1T  2148 2148  PL1-5.5T  PL1-13.6T  6116 6116  |
|                2148                       6116 6116  |
| PL1-1.1R                                             |
|            29         36                  6116       |
|                                           6116       |
|      PL1-4.4N   PL6-12.6N                            |
|                                           6116  0482 |
|2  DSWA DSWB                                          |
|2                PL1-3.6L                             |
|W                                                     |
|A                                          1371       |
|Y         2148                                        |
|          2148                         PL1-6.8J       |
|                 1179      1275        PL1-5.8H       |
|          30                                          |
|                 PL1-9.6F  PL1-11.7F   PL1-4.8F       |
|            6116                       PL1-3.8E       |
|   PL1-7.3E      PL1-8.6E  PL1-10.7E                  |
|                                       PL6-2.8D       |
|                 27                    PL6-1.8B       |
|    60A1    49.152MHz                                 |
|                           34          6809           |
|------------------------------------------------------|
Notes:
      6809 clock :
      63701 clock:
      VSync      : 60.606060
      6116       : 2K x8 SRAM
      2148       : 1K x4 SRAM

      Namco Customs
      27   (DIP40)
      29   (SDIP64)
      30   (SDIP64)
      34   (DIP24)
      36   (SDIP64)
      0482 (DIP28)
      1179 (DIP28)
      1275 (DIP28)
      1371 (DIP28)
      60A1 (DIP40, known 63701 MCU)


Notes:
-----
- Is there a service mode Easter egg? Maybe with this game they stopped putting
  them in, because I haven't found them in the later games either (skykid,
  drgnbstr, etc.)

- Sprites cover the top and bottom non-scrolling portions of the fg. This
  includes the cookie cut light in round 19, which makes text disappear from
  those areas. This looks odd, but it's the correct behaviour verified on the
  real hardware.

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/m6800/m6800.h"


static data8_t *sharedram;

extern data8_t *pacland_videoram,*pacland_videoram2,*pacland_spriteram;

WRITE8_HANDLER( pacland_videoram_w );
WRITE8_HANDLER( pacland_videoram2_w );
WRITE8_HANDLER( pacland_scroll0_w );
WRITE8_HANDLER( pacland_scroll1_w );
WRITE8_HANDLER( pacland_bankswitch_w );

PALETTE_INIT( pacland );
VIDEO_START( pacland );
VIDEO_UPDATE( pacland );


static READ8_HANDLER( sharedram_r )
{
	return sharedram[offset];
}

static WRITE8_HANDLER( sharedram_w )
{
	sharedram[offset] = data;
}

static WRITE8_HANDLER( pacland_subreset_w )
{
	int bit = !BIT(offset,11);
	cpunum_set_input_line(1, INPUT_LINE_RESET, bit ? CLEAR_LINE : ASSERT_LINE);
}

static WRITE8_HANDLER( pacland_flipscreen_w )
{
	int bit = !BIT(offset,11);
	/* can't use flip_screen_set() because the visible area is asymmetrical */
	flip_screen = bit;
	tilemap_set_flip(ALL_TILEMAPS,flip_screen ? (TILEMAP_FLIPX | TILEMAP_FLIPY) : 0);
}


static READ8_HANDLER( pacland_input_r )
{
	int shift = 4 * (offset & 1);
	int port = offset & 2;
	int r = ( readinputport( port+0 ) << shift ) & 0xf0;
	r |= ( readinputport( port+1 ) >> (4-shift) ) & 0x0f;
	return r;
}

static WRITE8_HANDLER( pacland_coin_w )
{
	coin_lockout_global_w(data & 1);
	coin_counter_w(0,~data & 2);
	coin_counter_w(1,~data & 4);
}

static WRITE8_HANDLER( pacland_led_w )
{
	set_led_status(0,data & 0x08);
	set_led_status(1,data & 0x10);
}

static WRITE8_HANDLER( pacland_irq_1_ctrl_w )
{
	int bit = !BIT(offset,11);
	cpu_interrupt_enable(0,bit);
	if (!bit)
		cpunum_set_input_line(0, 0, CLEAR_LINE);
}

static WRITE8_HANDLER( pacland_irq_2_ctrl_w )
{
	int bit = !BIT(offset,13);
	cpu_interrupt_enable(1,bit);
	if (!bit)
		cpunum_set_input_line(1, 0, CLEAR_LINE);
}



static ADDRESS_MAP_START( main_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x0fff) AM_READWRITE(MRA8_RAM, pacland_videoram_w) AM_BASE(&pacland_videoram)
	AM_RANGE(0x1000, 0x1fff) AM_READWRITE(MRA8_RAM, pacland_videoram2_w) AM_BASE(&pacland_videoram2)
	AM_RANGE(0x2000, 0x37ff) AM_RAM AM_BASE(&pacland_spriteram)
	AM_RANGE(0x3800, 0x3801) AM_WRITE(pacland_scroll0_w)
	AM_RANGE(0x3a00, 0x3a01) AM_WRITE(pacland_scroll1_w)
	AM_RANGE(0x3c00, 0x3c00) AM_WRITE(pacland_bankswitch_w)
	AM_RANGE(0x4000, 0x5fff) AM_READ(MRA8_BANK1)

	AM_RANGE(0x6800, 0x68ff) AM_READWRITE(namcos1_wavedata_r, namcos1_wavedata_w)		/* PSG device, shared RAM */
	AM_RANGE(0x6800, 0x6bff) AM_READWRITE(sharedram_r, sharedram_w) AM_BASE(&sharedram)

	AM_RANGE(0x7000, 0x7fff) AM_WRITE(pacland_irq_1_ctrl_w)
	AM_RANGE(0x7800, 0x7fff) AM_READ(watchdog_reset_r)
	AM_RANGE(0x8000, 0xffff) AM_READ(MRA8_ROM)
	AM_RANGE(0x8000, 0x8fff) AM_WRITE(pacland_subreset_w)
	AM_RANGE(0x9000, 0x9fff) AM_WRITE(pacland_flipscreen_w)
ADDRESS_MAP_END

static ADDRESS_MAP_START( mcu_map, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x001f) AM_READWRITE(hd63701_internal_registers_r, hd63701_internal_registers_w)
	AM_RANGE(0x0080, 0x00ff) AM_RAM

	AM_RANGE(0x1000, 0x10ff) AM_READWRITE(namcos1_wavedata_r, namcos1_wavedata_w) AM_BASE(&namco_wavedata)		/* PSG device, shared RAM */
	AM_RANGE(0x1100, 0x113f) AM_READWRITE(MRA8_RAM, namcos1_sound_w) AM_BASE(&namco_soundregs) /* PSG device */
	AM_RANGE(0x1000, 0x13ff) AM_READWRITE(sharedram_r, sharedram_w)

	AM_RANGE(0x2000, 0x3fff) AM_WRITE(watchdog_reset_w)		/* watchdog? */
	AM_RANGE(0x4000, 0x7fff) AM_WRITE(pacland_irq_2_ctrl_w)
	AM_RANGE(0x8000, 0xbfff) AM_READ(MRA8_ROM)
	AM_RANGE(0xc000, 0xc7ff) AM_RAM
	AM_RANGE(0xd000, 0xd003) AM_READ(pacland_input_r)
	AM_RANGE(0xf000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static READ8_HANDLER( readFF )
{
	return 0xff;
}

static ADDRESS_MAP_START( mcu_port_map, ADDRESS_SPACE_IO, 8 )
	AM_RANGE(HD63701_PORT1, HD63701_PORT1) AM_READ(input_port_4_r)
	AM_RANGE(HD63701_PORT1, HD63701_PORT1) AM_WRITE(pacland_coin_w)
	AM_RANGE(HD63701_PORT2, HD63701_PORT2) AM_READ(readFF)	/* leds won't work otherwise */
	AM_RANGE(HD63701_PORT2, HD63701_PORT2) AM_WRITE(pacland_led_w)
ADDRESS_MAP_END



INPUT_PORTS_START( pacland )
	PORT_START      /* DSWA */
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x40, "2" )
	PORT_DIPSETTING(    0x60, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x18, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )

	PORT_START      /* DSWB */
	PORT_DIPNAME( 0xe0, 0xe0, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0xe0, "30K 80K 130K 300K 500K 1M" )
	PORT_DIPSETTING(    0x80, "30K 80K every 100K" )
	PORT_DIPSETTING(    0x40, "30K 80K 150K" )
	PORT_DIPSETTING(    0xc0, "30K 100K 200K 400K 600K 1M" )
	PORT_DIPSETTING(    0xa0, "40K 100K 180K 300K 500K 1M" )
	PORT_DIPSETTING(    0x20, "40K 100K 200K" )
	PORT_DIPSETTING(    0x00, "40K" )
	PORT_DIPSETTING(    0x60, "50K 150K every 200K" )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x10, "B (Easy)" )
	PORT_DIPSETTING(    0x18, "A (Average)" )
	PORT_DIPSETTING(    0x08, "C (Hard)" )
	PORT_DIPSETTING(    0x00, "D (Very Hard)" )
	PORT_DIPNAME( 0x04, 0x04, "Round Select" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Freeze" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x01, 0x01, "Trip Select" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )

	PORT_START	/* Memory Mapped Port */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )	// service mode again
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* Memory Mapped Port */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )	// IPT_JOYSTICK_DOWN according to schematics
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )	// IPT_JOYSTICK_UP according to schematics
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )	// IPT_JOYSTICK_DOWN | IPF_COCKTAIL according to schematics
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )	// IPT_JOYSTICK_UP | IPF_COCKTAIL according to schematics

	PORT_START	/* MCU Input Port */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SPECIAL )	/* OUT:coin lockout */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SPECIAL )	/* OUT:coin counter 1 */
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SPECIAL )	/* OUT:coin counter 2 */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_COCKTAIL )
INPUT_PORTS_END



static struct GfxLayout spritelayout =
{
	16,16,
	RGN_FRAC(1,2),
	4,
	{ 0, 4, RGN_FRAC(1,2)+0, RGN_FRAC(1,2)+4 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3,
			16*8+0, 16*8+1, 16*8+2, 16*8+3, 24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8,
			32*8, 33*8, 34*8, 35*8, 36*8, 37*8, 38*8, 39*8 },
	64*8
};

static struct GfxLayout charlayout =
{
	8,8,
	RGN_FRAC(1,1),
	2,
	{ 0, 4 },
	{ 8*8, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
	16*8
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &charlayout,              0, 256 },
	{ REGION_GFX2, 0, &charlayout,          256*4, 256 },
	{ REGION_GFX3, 0, &spritelayout,  256*4+256*4, 2*64 },
	{ -1 } /* end of array */
};



static struct namco_interface namco_interface =
{
	49152000/2/1024,	/* 24kHz sample rate */
	8,		/* number of voices */
	100,	/* playback volume */
	-1,		/* memory region */
	0		/* stereo */
};



static MACHINE_DRIVER_START( pacland )

	/* basic machine hardware */
	MDRV_CPU_ADD(M6809, 49152000/32)	/* 1.536 MHz */
	MDRV_CPU_PROGRAM_MAP(main_map,0)
	MDRV_CPU_VBLANK_INT(irq0_line_assert,1)

	MDRV_CPU_ADD(HD63701, 49152000/32)	/* 1.536 MHz? */
	MDRV_CPU_PROGRAM_MAP(mcu_map,0)
	MDRV_CPU_IO_MAP(mcu_port_map,0)
	MDRV_CPU_VBLANK_INT(irq0_line_assert,1)

	MDRV_FRAMES_PER_SECOND(60.606060)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(100)	/* we need heavy synching between the MCU and the CPU */

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(64*8, 32*8)
	MDRV_VISIBLE_AREA(3*8, 39*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(256)
	MDRV_COLORTABLE_LENGTH(256*4+256*4+2*64*16)

	MDRV_PALETTE_INIT(pacland)
	MDRV_VIDEO_START(pacland)
	MDRV_VIDEO_UPDATE(pacland)

	/* sound hardware */
	MDRV_SOUND_ADD(NAMCO_CUS30, namco_interface)
MACHINE_DRIVER_END


/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( pacland )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 128k for code */
	ROM_LOAD( "pl5_01b.bin",  0x08000, 0x4000, CRC(b0ea7631) SHA1(424afa6f397310c7af39c9e8b580aa9ccd42c39c) )
	ROM_LOAD( "pl5_02.bin",   0x0C000, 0x4000, CRC(d903e84e) SHA1(25338726227bfbec65847879aac5228a6a307db4) )
	/* all the following are banked at 0x4000-0x5fff */
	ROM_LOAD( "pl1-3",        0x10000, 0x4000, CRC(aa9fa739) SHA1(7b1f7857eb5f68e166b1f8988c82051aaf05df48) )
	ROM_LOAD( "pl1-4",        0x14000, 0x4000, CRC(2b895a90) SHA1(820f8873c6a5a736089406d0f03d491dfb82d00d) )
	ROM_LOAD( "pl1-5",        0x18000, 0x4000, CRC(7af66200) SHA1(f44161ded1633e9801b7a9cd84d481e53823f5d9) )
	ROM_LOAD( "pl3_06.bin",   0x1c000, 0x4000, CRC(2ffe3319) SHA1(c2540321cd5a1fe29ecb077abdf8f997893192e9) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for code */
	ROM_LOAD( "pl1-7",        0x8000, 0x2000, CRC(8c5becae) SHA1(14d67136395c4c64472980a69648ce2d479ae67f) ) /* sub program for the mcu */
	ROM_LOAD( "cus60",        0xf000, 0x1000, CRC(6ef08fb3) SHA1(4842590d60035a0059b0899eb2d5f58ae72c2529) ) /* microcontroller */

	ROM_REGION( 0x02000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pl2_12.bin",   0x00000, 0x2000, CRC(a63c8726) SHA1(b15903fa2267375280af03af0a7157e1b0bcb86d) )	/* chars */

	ROM_REGION( 0x02000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pl4_13.bin",   0x00000, 0x2000, CRC(3ae582fd) SHA1(696b2cfadb6b071de8e43d20cd65b37713ca3b30) )

	ROM_REGION( 0x10000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-9",        0x00000, 0x4000, CRC(f5d5962b) SHA1(8d008a9bc06dc562c241955d9c551647b5c1f4e9) )	/* sprites */
	ROM_LOAD( "pl1-8",        0x04000, 0x4000, CRC(a2ebfa4a) SHA1(4a2a2b43a23a7a46266751415d1bde118143429c) )
	ROM_LOAD( "pl1-10",       0x08000, 0x4000, CRC(c7cf1904) SHA1(7ca8ed20ee32eb8609ac96b4e4fcb3b6027b598a) )
	ROM_LOAD( "pl1-11",       0x0c000, 0x4000, CRC(6621361a) SHA1(4efa40adba803006e86d5e12514983d4132b5efb) )

	ROM_REGION( 0x1400, REGION_PROMS, 0 )
	ROM_LOAD( "pl1-2.bin",    0x0000, 0x0400, CRC(472885de) SHA1(8d552c90b8d5bc6ad6c60934c00f4303cd180ce7) )	/* red and green component */
	ROM_LOAD( "pl1-1.bin",    0x0400, 0x0400, CRC(a78ebdaf) SHA1(8ea215701eb5e1a2a329ef92c19fc69b18fc28c7) )	/* blue component */
	ROM_LOAD( "pl1-5.bin",    0x0800, 0x0400, CRC(4b7ee712) SHA1(dd0ec4c632d8b160f7b54d8f18fcf4ef1508d832) )	/* foreground lookup table */
	ROM_LOAD( "pl1-4.bin",    0x0c00, 0x0400, CRC(3a7be418) SHA1(475cdc68205e3acce83fe79b00b74c6a7e28dde4) )	/* background lookup table */
	ROM_LOAD( "pl1-3.bin",    0x1000, 0x0400, CRC(80558da8) SHA1(7e1483467817295f36d1e2bdb32934c4f2617d52) )	/* sprites lookup table */
ROM_END

ROM_START( pacland2 )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 128k for code */
	ROM_LOAD( "pl6_01.bin",   0x08000, 0x4000, CRC(4c96e11c) SHA1(c136dc3877155b7a600173c876f6a53394d9260d) )
	ROM_LOAD( "pl6_02.bin",   0x0C000, 0x4000, CRC(8cf5bd8d) SHA1(0771ca1ab5db58f5632583a5e6e84660e8ab727d) )
	/* all the following are banked at 0x4000-0x5fff */
	ROM_LOAD( "pl1-3",        0x10000, 0x4000, CRC(aa9fa739) SHA1(7b1f7857eb5f68e166b1f8988c82051aaf05df48) )
	ROM_LOAD( "pl1-4",        0x14000, 0x4000, CRC(2b895a90) SHA1(820f8873c6a5a736089406d0f03d491dfb82d00d) )
	ROM_LOAD( "pl1-5",        0x18000, 0x4000, CRC(7af66200) SHA1(f44161ded1633e9801b7a9cd84d481e53823f5d9) )
	ROM_LOAD( "pl1-6",        0x1c000, 0x4000, CRC(b01e59a9) SHA1(e5b093852d33a4d09969d111fa6e42e964aa4dac) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for code */
	ROM_LOAD( "pl1-7",        0x8000, 0x2000, CRC(8c5becae) SHA1(14d67136395c4c64472980a69648ce2d479ae67f) ) /* sub program for the mcu */
	ROM_LOAD( "cus60",        0xf000, 0x1000, CRC(6ef08fb3) SHA1(4842590d60035a0059b0899eb2d5f58ae72c2529) ) /* microcontroller */

	ROM_REGION( 0x02000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pl0_12.bin",   0x00000, 0x2000, CRC(c8cb61ab) SHA1(ec33d64949a8c011430e889f55f54816b33c4218) )	/* chars */

	ROM_REGION( 0x02000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-13",       0x00000, 0x2000, CRC(6c5ed9ae) SHA1(db919c9254289179e98ba5d2ed8c66d67ae95f35) )

	ROM_REGION( 0x10000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1_09b.bin",  0x00000, 0x4000, CRC(80768a87) SHA1(1572f309e810d9eb007a1c8b2aa8463027c146ca) )	/* sprites */
	ROM_LOAD( "pl1_08.bin",   0x04000, 0x4000, CRC(2b20e46d) SHA1(9f78952ae94fef6a83a15de35d5fefdf71e78488) )
	ROM_LOAD( "pl1_10b.bin",  0x08000, 0x4000, CRC(ffd9d66e) SHA1(9a6e9ad500fcb7a67cb3c45d029c2aa7636a64f9) )
	ROM_LOAD( "pl1_11.bin",   0x0c000, 0x4000, CRC(c59775d8) SHA1(034281c8101719d79043df31ef845fd28c0c69c0) )

	ROM_REGION( 0x1400, REGION_PROMS, 0 )
	ROM_LOAD( "pl1-2.bin",    0x0000, 0x0400, CRC(472885de) SHA1(8d552c90b8d5bc6ad6c60934c00f4303cd180ce7) )	/* red and green component */
	ROM_LOAD( "pl1-1.bin",    0x0400, 0x0400, CRC(a78ebdaf) SHA1(8ea215701eb5e1a2a329ef92c19fc69b18fc28c7) )	/* blue component */
	ROM_LOAD( "pl1-5.bin",    0x0800, 0x0400, CRC(4b7ee712) SHA1(dd0ec4c632d8b160f7b54d8f18fcf4ef1508d832) )	/* foreground lookup table */
	ROM_LOAD( "pl1-4.bin",    0x0c00, 0x0400, CRC(3a7be418) SHA1(475cdc68205e3acce83fe79b00b74c6a7e28dde4) )	/* background lookup table */
	ROM_LOAD( "pl1-3.bin",    0x1000, 0x0400, CRC(80558da8) SHA1(7e1483467817295f36d1e2bdb32934c4f2617d52) )	/* sprites lookup table */
ROM_END

ROM_START( pacland3 )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 128k for code */
	ROM_LOAD( "pln1-1",       0x08000, 0x4000, CRC(f729fb94) SHA1(332ff2e4aae67eb8ed0f52048097f74323a176f8) )
	ROM_LOAD( "pln1-2",       0x0C000, 0x4000, CRC(5c66eb6f) SHA1(376233f51e655df8922886c1e808a2f37ccae5d4) )
	/* all the following are banked at 0x4000-0x5fff */
	ROM_LOAD( "pl1-3",        0x10000, 0x4000, CRC(aa9fa739) SHA1(7b1f7857eb5f68e166b1f8988c82051aaf05df48) )
	ROM_LOAD( "pl1-4",        0x14000, 0x4000, CRC(2b895a90) SHA1(820f8873c6a5a736089406d0f03d491dfb82d00d) )
	ROM_LOAD( "pl1-5",        0x18000, 0x4000, CRC(7af66200) SHA1(f44161ded1633e9801b7a9cd84d481e53823f5d9) )
	ROM_LOAD( "pl1-6",        0x1c000, 0x4000, CRC(b01e59a9) SHA1(e5b093852d33a4d09969d111fa6e42e964aa4dac) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for code */
	ROM_LOAD( "pl1-7",        0x8000, 0x2000, CRC(8c5becae) SHA1(14d67136395c4c64472980a69648ce2d479ae67f) ) /* sub program for the mcu */
	ROM_LOAD( "cus60",        0xf000, 0x1000, CRC(6ef08fb3) SHA1(4842590d60035a0059b0899eb2d5f58ae72c2529) ) /* microcontroller */

	ROM_REGION( 0x02000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-12",       0x00000, 0x2000, CRC(c159fbce) SHA1(b0326c85b7df407f3e94c38a5971f911968d7b27) )	/* chars */

	ROM_REGION( 0x02000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-13",       0x00000, 0x2000, CRC(6c5ed9ae) SHA1(db919c9254289179e98ba5d2ed8c66d67ae95f35) )

	ROM_REGION( 0x10000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1_09b.bin",  0x00000, 0x4000, CRC(80768a87) SHA1(1572f309e810d9eb007a1c8b2aa8463027c146ca) )	/* sprites */
	ROM_LOAD( "pl1_08.bin",   0x04000, 0x4000, CRC(2b20e46d) SHA1(9f78952ae94fef6a83a15de35d5fefdf71e78488) )
	ROM_LOAD( "pl1_10b.bin",  0x08000, 0x4000, CRC(ffd9d66e) SHA1(9a6e9ad500fcb7a67cb3c45d029c2aa7636a64f9) )
	ROM_LOAD( "pl1_11.bin",   0x0c000, 0x4000, CRC(c59775d8) SHA1(034281c8101719d79043df31ef845fd28c0c69c0) )

	ROM_REGION( 0x1400, REGION_PROMS, 0 )
	ROM_LOAD( "pl1-2.bin",    0x0000, 0x0400, CRC(472885de) SHA1(8d552c90b8d5bc6ad6c60934c00f4303cd180ce7) )	/* red and green component */
	ROM_LOAD( "pl1-1.bin",    0x0400, 0x0400, CRC(a78ebdaf) SHA1(8ea215701eb5e1a2a329ef92c19fc69b18fc28c7) )	/* blue component */
	ROM_LOAD( "pl1-5.bin",    0x0800, 0x0400, CRC(4b7ee712) SHA1(dd0ec4c632d8b160f7b54d8f18fcf4ef1508d832) )	/* foreground lookup table */
	ROM_LOAD( "pl1-4.bin",    0x0c00, 0x0400, CRC(3a7be418) SHA1(475cdc68205e3acce83fe79b00b74c6a7e28dde4) )	/* background lookup table */
	ROM_LOAD( "pl1-3.bin",    0x1000, 0x0400, CRC(80558da8) SHA1(7e1483467817295f36d1e2bdb32934c4f2617d52) )	/* sprites lookup table */
ROM_END

ROM_START( paclandm )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 128k for code */
	ROM_LOAD( "pl1-1",        0x08000, 0x4000, CRC(a938ae99) SHA1(bf12097d8c69685cb7af763f9b9617c767aaed2f) )
	ROM_LOAD( "pl1-2",        0x0C000, 0x4000, CRC(3fe43bb5) SHA1(14e6144d06ff2fd786f383f36f1b8238ac364849) )
	/* all the following are banked at 0x4000-0x5fff */
	ROM_LOAD( "pl1-3",        0x10000, 0x4000, CRC(aa9fa739) SHA1(7b1f7857eb5f68e166b1f8988c82051aaf05df48) )
	ROM_LOAD( "pl1-4",        0x14000, 0x4000, CRC(2b895a90) SHA1(820f8873c6a5a736089406d0f03d491dfb82d00d) )
	ROM_LOAD( "pl1-5",        0x18000, 0x4000, CRC(7af66200) SHA1(f44161ded1633e9801b7a9cd84d481e53823f5d9) )
	ROM_LOAD( "pl1-6",        0x1c000, 0x4000, CRC(b01e59a9) SHA1(e5b093852d33a4d09969d111fa6e42e964aa4dac) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for code */
	ROM_LOAD( "pl1-7",        0x8000, 0x2000, CRC(8c5becae) SHA1(14d67136395c4c64472980a69648ce2d479ae67f) ) /* sub program for the mcu */
	ROM_LOAD( "cus60",        0xf000, 0x1000, CRC(6ef08fb3) SHA1(4842590d60035a0059b0899eb2d5f58ae72c2529) ) /* microcontroller */

	ROM_REGION( 0x02000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-12",       0x00000, 0x2000, CRC(c159fbce) SHA1(b0326c85b7df407f3e94c38a5971f911968d7b27) )	/* chars */

	ROM_REGION( 0x02000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-13",       0x00000, 0x2000, CRC(6c5ed9ae) SHA1(db919c9254289179e98ba5d2ed8c66d67ae95f35) )

	ROM_REGION( 0x10000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "pl1-9",        0x00000, 0x4000, CRC(f5d5962b) SHA1(8d008a9bc06dc562c241955d9c551647b5c1f4e9) )	/* sprites */
	ROM_LOAD( "pl1-8",        0x04000, 0x4000, CRC(a2ebfa4a) SHA1(4a2a2b43a23a7a46266751415d1bde118143429c) )
	ROM_LOAD( "pl1-10",       0x08000, 0x4000, CRC(c7cf1904) SHA1(7ca8ed20ee32eb8609ac96b4e4fcb3b6027b598a) )
	ROM_LOAD( "pl1-11",       0x0c000, 0x4000, CRC(6621361a) SHA1(4efa40adba803006e86d5e12514983d4132b5efb) )

	ROM_REGION( 0x1400, REGION_PROMS, 0 )
	ROM_LOAD( "pl1-2.bin",    0x0000, 0x0400, CRC(472885de) SHA1(8d552c90b8d5bc6ad6c60934c00f4303cd180ce7) )	/* red and green component */
	ROM_LOAD( "pl1-1.bin",    0x0400, 0x0400, CRC(a78ebdaf) SHA1(8ea215701eb5e1a2a329ef92c19fc69b18fc28c7) )	/* blue component */
	ROM_LOAD( "pl1-5.bin",    0x0800, 0x0400, CRC(4b7ee712) SHA1(dd0ec4c632d8b160f7b54d8f18fcf4ef1508d832) )	/* foreground lookup table */
	ROM_LOAD( "pl1-4.bin",    0x0c00, 0x0400, CRC(3a7be418) SHA1(475cdc68205e3acce83fe79b00b74c6a7e28dde4) )	/* background lookup table */
	ROM_LOAD( "pl1-3.bin",    0x1000, 0x0400, CRC(80558da8) SHA1(7e1483467817295f36d1e2bdb32934c4f2617d52) )	/* sprites lookup table */
ROM_END



GAME( 1984, pacland,  0,       pacland, pacland, 0, ROT0, "Namco", "Pac-Land (set 1)" )
GAME( 1984, pacland2, pacland, pacland, pacland, 0, ROT0, "Namco", "Pac-Land (set 2)" )
GAME( 1984, pacland3, pacland, pacland, pacland, 0, ROT0, "Namco", "Pac-Land (set 3)" )
GAME( 1984, paclandm, pacland, pacland, pacland, 0, ROT0, "[Namco] (Bally Midway license)", "Pac-Land (Midway)" )
#pragma code_seg()
#pragma data_seg()
#pragma bss_seg()
#pragma const_seg()
