#pragma code_seg("C14")
#pragma data_seg("D14")
#pragma bss_seg("B14")
#pragma const_seg("K14")
#pragma comment(linker, "/merge:D14=14")
#pragma comment(linker, "/merge:C14=14")
#pragma comment(linker, "/merge:B14=14")
#pragma comment(linker, "/merge:K14=14")
/***************************************************************************

Gottlieb driver : dedicated to Warren Davis, Jeff Lee, Tim Skelly & David Thiel

driver by Fabrice Frances & Nicola Salmoria

Notes:
There was a bug in the hardware of the GG1 and GG2 boards, which is not
emulated. The bug seems to have disappeared with the later revision of the
board, e.g the board used by 3Stooges and Mach3 don't seem to have it).
The bug was affecting the first character column (on horizontal games):
screen memory could be used, but whatever was stored in this column, always
the same character was displayed.
This led to two consequences:
- the image on the monitor had to be stretched so that the column was not
  visible
- game designers were not using the first column. In fact, when the first
  column was ejected from the screen, the last one was usually out too,
  so it wasn't used either...

****************************************************************************

Reactor: earlier version of the board, with a different memory map

Main processor (8088 minimum mode)
0000-1fff RAM (NOT battery backed unlike all the others)
2000-2fff sprites
3000-37ff video ram
4000-4fff char generator ram
6000-67ff palette ram (palette of 16 colors)
7000-77ff i/o ports
8000-ffff ROM

memory mapped ports:

read:
7000    Dip switch
7001    Inputs 10-17
7002    trackball H
7003    trackball V
7004    Inputs 40-47

write:
7000    watchdog reset
7001    trackball clear
7002    Outputs 20-27
7003    Flipflop outputs:
		b0: F/B priority
		b1: horiz. flip
		b2: vert. flip
		b3: Output 33
		b4: coin counter
		b5: left lamp (1P/3Lives start)
		b6: middle lamps (2P/3Lives start, 1P/7Lives start)
		b7: right lamp (2P/7Lives start)
7004    Outputs 40-47

interrupts:
INTR not connected
NMI connected to vertical blank



Gottlieb games memory map

Main processor (8088 minimum mode)
0000-0fff RAM (battery backed)
1000-1fff RAM or ROM (selected with jumpers on the board)
2000-2fff RAM or ROM (selected with jumpers on the board)
3000-37ff sprites. The manual says there are 63 sprites (NOT 64),
          but the Q*Bert object priority test leaves sprite #63 dangling, so
		  they are probably only 62.
3800-3fff video RAM
4000-4fff char generator RAM (can be replaced by a ROM twice as large,
          selection made with jumpers on the board. If it's ROM, the CPU
		  cannot fully access it, I think it could read half the data if it
		  wanted to but none of the games do that)
5000-57ff palette ram (palette of 16 colors)
5800-5fff i/o ports
6000-ffff ROM (not necessarily fully populated)

memory mapped ports:

read:
5800    Dip switch
5801    Inputs 10-17
5802    trackball H (optional)
5803    trackball V (optional)
5804    Inputs 40-47

write:
5800    watchdog reset
5801    trackball clear (optional)
5802    Outputs 20-27
5803    Flipflop outputs:
		b0: F/B priority
		b1: horiz. flip (sprite bank in Us vs. Them)
		b2: vert. flip (maybe genlock control in the laser disc games)
		b3: Output 33
		b4: coin counter (sprite bank in Q*Bert Qubes)
		b5: Q*Bert: kicker; Q*Bert Qubes: coin counter
		b5/b6: 3 Stooges: joystick input multiplexer
		b7: ?
5804    Outputs 40-47

interrupts:
INTR not connected
NMI connected to vertical blank



Sound processor (6502) memory map (earlier revision, used by games up to Krull):
0000-0fff RIOT (6532)
1000-1fff amplitude DAC
2000-2fff SC01 voice chip
3000-3fff voice clock DAC
4000-4fff expansion socket
5000-5fff expansion socket
6000-6fff expansion socket or ROM (selected with jumpers on the board)
7000-7fff ROM
(repeated in 8000-ffff, A15 only used in expansion socket)

Use of I/Os on the RIOT:
both ports A and B are programmed as inputs, A is connected to the main
motherboard, and B has SW1 (test) connected on bit 6.

interrupts:
INTR is connected to the RIOT, so an INTR can be generated by a variety
of sources, e.g active edge detection on PA7, or timer countdown.
It seems that all gottlieb games program the interrupt conditions so that
a positive active edge on PA7 triggers an interrupt, so the
main board ensures a command is correctly received by sending nul (0)
commands between two commands. Also, the timer interrupt is enabled but
doesn't seem to serve any purpose...(?)


In the later revision of the sound board, used from M.A.C.H. 3 onwards, there
are two 6502, two 8910, a DAC and a GI SP-0250 speech chip.


Video timings:
XTAL = 20 MHz
Horizontal video frequency: HSYNC = XTAL/4/318 = 15.72327 kHz
Video frequency: VSYNC = HSYNC/256 = 61.41903 Hz
VBlank duration: 1/VSYNC * (16/256) = 1017.6 us

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"

extern UINT8 *gottlieb_charram;

extern WRITE_HANDLER( gottlieb_videoram_w );
extern WRITE_HANDLER( gottlieb_charram_w );
extern WRITE_HANDLER( gottlieb_video_outputs_w );
extern WRITE_HANDLER( usvsthem_video_outputs_w );
extern WRITE_HANDLER( gottlieb_paletteram_w );

extern VIDEO_START( gottlieb );
extern VIDEO_UPDATE( gottlieb );

extern WRITE_HANDLER( gottlieb_sh_w );

extern UINT8 *riot_ram;
extern READ_HANDLER( riot_ram_r );
extern READ_HANDLER( gottlieb_riot_r );
extern WRITE_HANDLER( riot_ram_w );
extern WRITE_HANDLER( gottlieb_riot_w );
extern WRITE_HANDLER( gottlieb_speech_w );
extern WRITE_HANDLER( gottlieb_speech_clock_DAC_w );
extern void gottlieb_sound_init(void);
extern READ_HANDLER( stooges_sound_input_r );
extern WRITE_HANDLER( stooges_8910_latch_w );
extern WRITE_HANDLER( stooges_sound_control_w );
extern WRITE_HANDLER( gottlieb_nmi_rate_w );
extern WRITE_HANDLER( gottlieb_cause_dac_nmi_w );


static UINT8 *audiobuffer_region;

MACHINE_INIT( gottlieb )
{
	UINT8 *ram = memory_region(REGION_CPU1);
	cpu_setbank(1, &ram[0x8000]);
	cpu_setbank(2, &ram[0x0000]);
	audiobuffer_region = memory_region(REGION_USER1);
}


static int track[2];

READ_HANDLER( gottlieb_track_0_r )
{
	return input_port_2_r(offset) - track[0];
}

READ_HANDLER( gottlieb_track_1_r )
{
	return input_port_3_r(offset) - track[1];
}

WRITE_HANDLER( gottlieb_track_reset_w )
{
	/* reset the trackball counters */
	track[0] = input_port_2_r(offset);
	track[1] = input_port_3_r(offset);
}

static int joympx;

READ_HANDLER( stooges_IN4_r )
{
	int joy;

	switch (joympx)
	{
		case 0:
		default:
			joy = ((readinputport(4) >> 0) & 0x0f);	/* joystick 1 */
			break;
		case 1:
			joy = ((readinputport(5) >> 0) & 0x0f);	/* joystick 2 */
			break;
		case 2:
			joy = ((readinputport(5) >> 4) & 0x0f);	/* joystick 3 */
			break;
	}

	return joy | (readinputport(4) & 0xf0);
}

WRITE_HANDLER( reactor_output_w )
{
	set_led_status(0,data & 0x20);
	set_led_status(1,data & 0x40);
	set_led_status(2,data & 0x80);
	gottlieb_video_outputs_w(offset,data);
}

WRITE_HANDLER( stooges_output_w )
{
	joympx = (data >> 5) & 0x03;
	gottlieb_video_outputs_w(offset,data);
}


static int current_frame = 1;
static int laserdisc_playing;
static int lasermpx;
static int audioptr;
static int audioready=1;
static int skipfirstbyte;
static int discready;
static int lastcmd;
static int odd_field;
static int access_time;


/* The only sure thing I know about the Philips 24-bit frame number code is that the
 * 4 most significant bits are 1's and these four 1's are used to detect a valid frame
 * number when shifting bits from right to left...
 * From the audio code it seems it's a simple BCD numbering...
 * (what happens when four consecutive 1's are inside the number, eg 00078 ?)
 * Only 19 bits are returned to the game machine, this gives a max frame number of 79999.
 * The size of the audio buffer is 1024 bytes. When it is full, the "audio buffer ready"
 * flag is raised, and the frame decoder tries again to synchronize with a 0x67 pattern
 * Each 1024-byte buffer contains target data for 53 frames;
 * target data for each frame is 19-bytes long including one byte of checksum
 * and the frame number code of the first of these 53 frames is stored at the beginning
 * This gives a total of 1+3+3+19*53=1014 bytes, the 10 last bytes are ignored
 */

READ_HANDLER( gottlieb_laserdisc_status_r )
{
	int tmp;
	switch (offset)
	{
		case 0:
			tmp = current_frame % 100;
			logerror("LSB frame read: %d\n",tmp);
			return ((tmp / 10) << 4) | (tmp % 10);
			break;
		case 1:
			tmp = (current_frame / 100) % 100;
			logerror("MSB frame read: %d\n",tmp);
			return ((tmp / 10) << 4) | (tmp % 10);
			break;
		case 2:
			if (lasermpx == 1) {
				/* bits 0-2 frame number MSN */
				/* bit 3 audio buffer ready */
				/* bit 4 ready to send new laserdisc command? */
				/* bit 5 disc ready */
				/* bit 6 break in audio trasmission */
				/* bit 7 missing audio clock */
				return ((current_frame / 10000) & 0x7) | (audioready << 3) | 0x10 | (discready << 5);
			} else {	/* read audio buffer */
				if (skipfirstbyte) audioptr++;
				skipfirstbyte = 0;
				if (audiobuffer_region) {
					logerror("audio bufread: %02x\n",audiobuffer_region[audioptr]);
					return audiobuffer_region[audioptr++];
				} else {
					logerror("audiobuffer is null !!");
					return 0xFF; /* don't know what to do in this case ;-) */
				}
			}
			break;
	}

	return 0;
}

WRITE_HANDLER( gottlieb_laserdisc_mpx_w )
{
	lasermpx = data & 1;
	if (lasermpx==0) skipfirstbyte=1;	/* first byte of the 1K buffer (0x67) is not returned... */
}

WRITE_HANDLER( gottlieb_laserdisc_command_w )
{
	static int loop;
	int cmd;

	/* commands are written in three steps, the first two the command is */
	/* written (maybe one to load the latch, the other to start the send), */
	/* the third 0 (maybe to clear the latch) */
	if (data == 0) return;
	if (loop++ & 1) return;

	if ((data & 0xe0) != 0x20)
	{
logerror("error: laserdisc command %02x\n",data);
		return;
	}

	cmd =	((data & 0x10) >> 4) |
			((data & 0x08) >> 2) |
			((data & 0x04) >> 0) |
			((data & 0x02) << 2) |
			((data & 0x01) << 4);

logerror("laserdisc command %02x -> %02x\n",data,cmd);
	if (lastcmd == 0x0b && (cmd & 0x10))	/* seek frame # */
	{
		current_frame = current_frame * 10 + (cmd & 0x0f);
		while (current_frame >= 100000)
			current_frame -= 100000;
		audioptr = -1;
	}
	else
	{
		if (cmd == 0x04)					/* step forward */
		{
			laserdisc_playing = 0;
			current_frame++;
		}
		if (cmd == 0x05) 					/* play */
		{
			laserdisc_playing = 1;
			discready = 1;
		}
		if (cmd == 0x0b)					/* seek frame */
		{
			laserdisc_playing = 0;
			discready = 0;
			access_time = 60;		/* 1s access time */
		}
		if (cmd == 0x0f)	 				/* stop */
		{
			laserdisc_playing = 0;
			discready = 0;
		}
		lastcmd = cmd;
	}
}

INTERRUPT_GEN( gottlieb_interrupt )
{
	if (access_time > 0) {
		access_time--;
		if (access_time == 0)
			discready = 1;
	} else if (laserdisc_playing) {
		odd_field ^= 1;
		if (odd_field)		/* the manual says the video frame number is only present in the odd field) */
		{
			current_frame++;
logerror("current frame : %d\n",current_frame);

			if (current_frame%53==0)
			{
				int seq = current_frame / 53;
				if (seq >= 44) {	/* 44*53 frames without target data at the beginning of the disc */
					audioptr = (seq - 44)*1024;
					audioready = 1;
				}
			}
			else audioready = 0;
		}
	}
	cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
}


static MEMORY_READ_START( reactor_readmem )
	{ 0x00000, 0x01fff, MRA_RAM },
	{ 0x03000, 0x033ff, MRA_RAM },
	{ 0x04000, 0x04fff, MRA_RAM },
	{ 0x07000, 0x07000, input_port_0_r },	/* DSW */
	{ 0x07001, 0x07001, input_port_1_r },	/* buttons */
	{ 0x07002, 0x07002, gottlieb_track_0_r },	/* trackball H */
	{ 0x07003, 0x07003, gottlieb_track_1_r },	/* trackball V */
	{ 0x07004, 0x07004, input_port_4_r },	/* joystick */
	{ 0x08000, 0x0ffff, MRA_ROM },
	/* map the ROM into Fxxxx so we can boot */
	{ 0xf8000, 0xfffff, MRA_BANK1 },
MEMORY_END

static MEMORY_WRITE_START( reactor_writemem )
	{ 0x00000, 0x01fff, MWA_RAM },
	{ 0x02000, 0x020ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x03000, 0x033ff, gottlieb_videoram_w, &videoram },
	{ 0x03400, 0x037ff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0x04000, 0x04fff, gottlieb_charram_w, &gottlieb_charram },
	{ 0x06000, 0x0601f, gottlieb_paletteram_w, &paletteram },
	{ 0x07000, 0x07000, watchdog_reset_w },
	{ 0x07001, 0x07001, gottlieb_track_reset_w },
	{ 0x07002, 0x07002, gottlieb_sh_w }, /* sound/speech command */
	{ 0x07003, 0x07003, reactor_output_w },       /* OUT1 */
	{ 0x08000, 0x0ffff, MWA_ROM },
MEMORY_END

static MEMORY_READ_START( gottlieb_readmem )
	{ 0x00000, 0x00fff, MRA_RAM },
	{ 0x01000, 0x01fff, MRA_RAM },	/* or ROM */
	{ 0x02000, 0x02fff, MRA_RAM },	/* or ROM */
	{ 0x03800, 0x03bff, MRA_RAM },
	{ 0x04000, 0x04fff, MRA_RAM },
	{ 0x05800, 0x05800, input_port_0_r },	/* DSW */
	{ 0x05801, 0x05801, input_port_1_r },	/* buttons */
	{ 0x05802, 0x05802, gottlieb_track_0_r },	/* trackball H */
	{ 0x05803, 0x05803, gottlieb_track_1_r },	/* trackball V */
	{ 0x05804, 0x05804, input_port_4_r },	/* joystick */
	{ 0x05805, 0x05807, gottlieb_laserdisc_status_r },
	{ 0x06000, 0x0ffff, MRA_ROM },
	/* Us vs Them and Q*bert Qubes use the Exxxx address space */
	{ 0xe5800, 0xe5800, input_port_0_r },	/* DSW */
	{ 0xe5801, 0xe5801, input_port_1_r },	/* buttons */
	{ 0xe5802, 0xe5802, gottlieb_track_0_r },	/* trackball H */
	{ 0xe5803, 0xe5803, gottlieb_track_1_r },	/* trackball V */
	{ 0xe5804, 0xe5804, input_port_4_r },	/* joystick */
	{ 0xe5805, 0xe5807, gottlieb_laserdisc_status_r },
	{ 0xe0000, 0xeffff, MRA_BANK2 },
	/* map the ROM into Fxxxx so we can boot */
	{ 0xf8000, 0xfffff, MRA_BANK1 },
MEMORY_END

static MEMORY_WRITE_START( gottlieb_writemem )
	{ 0x00000, 0x00fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x01000, 0x01fff, MWA_RAM },	/* ROM in Krull */
	{ 0x02000, 0x02fff, MWA_RAM },	/* ROM in Krull and 3 Stooges */
	{ 0x03000, 0x030ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x03800, 0x03bff, gottlieb_videoram_w, &videoram },
	{ 0x03c00, 0x03fff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0x04000, 0x04fff, gottlieb_charram_w, &gottlieb_charram },
	{ 0x05000, 0x0501f, gottlieb_paletteram_w, &paletteram },
	{ 0x05800, 0x05800, watchdog_reset_w },
	{ 0x05801, 0x05801, gottlieb_track_reset_w },
	{ 0x05802, 0x05802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x05803, 0x05803, gottlieb_video_outputs_w },       /* OUT1 */
	{ 0x06000, 0x0ffff, MWA_ROM },
	/* Q*bert Qubes uses the Exxxx address space */
	{ 0xe0000, 0xe37ff, MWA_BANK2 },
	{ 0xe3800, 0xe3bff, gottlieb_videoram_w },
	{ 0xe3c00, 0xe3fff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0xe4000, 0xe4fff, gottlieb_charram_w },
	{ 0xe5000, 0xe501f, gottlieb_paletteram_w },
	{ 0xe5800, 0xe5800, watchdog_reset_w },
	{ 0xe5801, 0xe5801, gottlieb_track_reset_w },
	{ 0xe5802, 0xe5802, gottlieb_sh_w }, /* sound/speech command */
	{ 0xe5803, 0xe5803, gottlieb_video_outputs_w },       /* OUT1 */
MEMORY_END


/* same as above, different video_outputs plus laser disc control outputs */
static MEMORY_WRITE_START( usvsthem_writemem )
	{ 0x00000, 0x00fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x01000, 0x01fff, MWA_RAM },	/* ROM in Krull */
	{ 0x02000, 0x02fff, MWA_RAM },	/* ROM in Krull and 3 Stooges */
	{ 0x03000, 0x030ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x03800, 0x03bff, gottlieb_videoram_w, &videoram },
	{ 0x03c00, 0x03fff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0x04000, 0x04fff, gottlieb_charram_w, &gottlieb_charram },
	{ 0x05000, 0x0501f, gottlieb_paletteram_w, &paletteram },
	{ 0x05800, 0x05800, watchdog_reset_w },
	{ 0x05801, 0x05801, gottlieb_track_reset_w },
	{ 0x05802, 0x05802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x05803, 0x05803, usvsthem_video_outputs_w },       /* OUT1 */
	{ 0x05805, 0x05805, gottlieb_laserdisc_command_w },	/* command for the player */
	{ 0x05806, 0x05806, gottlieb_laserdisc_mpx_w },
	{ 0x06000, 0x0ffff, MWA_ROM },
	/* Us vs Them uses the Exxxx address space */
	{ 0xe0000, 0xe37ff, MWA_BANK2 },
	{ 0xe3800, 0xe3bff, gottlieb_videoram_w },
	{ 0xe3c00, 0xe3fff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0xe4000, 0xe4fff, gottlieb_charram_w },
	{ 0xe5000, 0xe501f, gottlieb_paletteram_w },
	{ 0xe5800, 0xe5800, watchdog_reset_w },
	{ 0xe5801, 0xe5801, gottlieb_track_reset_w },
	{ 0xe5802, 0xe5802, gottlieb_sh_w }, /* sound/speech command */
	{ 0xe5803, 0xe5803, usvsthem_video_outputs_w },       /* OUT1 */
	{ 0xe5805, 0xe5805, gottlieb_laserdisc_command_w },	/* command for the player */
	{ 0xe5806, 0xe5806, gottlieb_laserdisc_mpx_w },
MEMORY_END

/* same as above, different IN4 */
static MEMORY_READ_START( stooges_readmem )
	{ 0x00000, 0x00fff, MRA_RAM },
	{ 0x01000, 0x01fff, MRA_RAM },
	{ 0x02000, 0x02fff, MRA_ROM },
	{ 0x03800, 0x03bff, MRA_RAM },
	{ 0x04000, 0x04fff, MRA_RAM },
	{ 0x05800, 0x05800, input_port_0_r },	/* DSW */
	{ 0x05801, 0x05801, input_port_1_r },	/* buttons */
	{ 0x05802, 0x05802, gottlieb_track_0_r },	/* trackball H */
	{ 0x05803, 0x05803, gottlieb_track_1_r },	/* trackball V */
	{ 0x05804, 0x05804, stooges_IN4_r },	/* joystick */
	{ 0x06000, 0x0ffff, MRA_ROM },
	/* map the ROM into Fxxxx so we can boot */
	{ 0xf8000, 0xfffff, MRA_BANK1 },
MEMORY_END

/* same as above, different video_outputs */
static MEMORY_WRITE_START( stooges_writemem )
	{ 0x00000, 0x00fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x01000, 0x01fff, MWA_RAM },
	{ 0x02000, 0x02fff, MWA_ROM },
	{ 0x03000, 0x030ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x03800, 0x03bff, gottlieb_videoram_w, &videoram },
	{ 0x03c00, 0x03fff, gottlieb_videoram_w },	/* mirror address, some games write to it */
	{ 0x04000, 0x04fff, gottlieb_charram_w, &gottlieb_charram },
	{ 0x05000, 0x0501f, gottlieb_paletteram_w, &paletteram },
	{ 0x05800, 0x05800, watchdog_reset_w },
	{ 0x05801, 0x05801, gottlieb_track_reset_w },
	{ 0x05802, 0x05802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x05803, 0x05803, stooges_output_w },       /* OUT1 */
	{ 0x06000, 0x0ffff, MWA_ROM },
MEMORY_END



MEMORY_READ_START( gottlieb_sound_readmem )
	{ 0x0000, 0x01ff, riot_ram_r },
	{ 0x0200, 0x03ff, gottlieb_riot_r },
	{ 0x6000, 0x7fff, MRA_ROM },
			 /* A15 not decoded except in expansion socket */
	{ 0x8000, 0x81ff, riot_ram_r },
	{ 0x8200, 0x83ff, gottlieb_riot_r },
	{ 0xe000, 0xffff, MRA_ROM },
MEMORY_END

MEMORY_WRITE_START( gottlieb_sound_writemem )
	{ 0x0000, 0x01ff, riot_ram_w, &riot_ram },
	{ 0x0200, 0x03ff, gottlieb_riot_w },
	{ 0x1000, 0x1000, DAC_0_data_w },
	{ 0x2000, 0x2000, gottlieb_speech_w },
	{ 0x3000, 0x3000, gottlieb_speech_clock_DAC_w },
	{ 0x6000, 0x7fff, MWA_ROM },
			 /* A15 not decoded except in expansion socket */
	{ 0x8000, 0x81ff, riot_ram_w },
	{ 0x8200, 0x83ff, gottlieb_riot_w },
	{ 0x9000, 0x9000, DAC_0_data_w },
	{ 0xa000, 0xa000, gottlieb_speech_w },
	{ 0xb000, 0xb000, gottlieb_speech_clock_DAC_w },
	{ 0xe000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( stooges_sound_readmem )
	{ 0x0000, 0x03ff, MRA_RAM },
	{ 0x8000, 0x8000, soundlatch_r },
	{ 0xe000, 0xffff, MRA_ROM },
MEMORY_END

MEMORY_WRITE_START( stooges_sound_writemem )
	{ 0x0000, 0x03ff, MWA_RAM },
	{ 0x4000, 0x4001, DAC_0_data_w },
	{ 0xe000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( stooges_sound2_readmem )
	{ 0x0000, 0x03ff, MRA_RAM },
	{ 0x6000, 0x6000, stooges_sound_input_r },	/* various signals */
	{ 0xa800, 0xa800, soundlatch_r },
	{ 0xc000, 0xffff, MRA_ROM },
MEMORY_END

MEMORY_WRITE_START( stooges_sound2_writemem )
	{ 0x0000, 0x03ff, MWA_RAM },
	{ 0x2000, 0x2000, MWA_NOP },	/* speech chip. The game sends strings */
									/* of 15 bytes (clocked by 4000). The chip also */
									/* checks a DATA REQUEST bit in 6000. */
	{ 0x4000, 0x4000, stooges_sound_control_w },
	{ 0x8000, 0x8000, stooges_8910_latch_w },
	{ 0xa000, 0xa000, gottlieb_nmi_rate_w },	/* the timer generates NMIs */
	{ 0xb000, 0xb000, gottlieb_cause_dac_nmi_w },
	{ 0xc000, 0xffff, MWA_ROM },
MEMORY_END



INPUT_PORTS_START( reactor )
	PORT_START	/* DSW */
	PORT_DIPNAME( 0x01, 0x01, "Sound with Logos" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Bounce Chambers Points" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPSETTING(    0x02, "15" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Sound with Instructions" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x40, "12000" )
	PORT_DIPSETTING(    0xc0, "15000" )
	PORT_DIPSETTING(    0x80, "20000" )

	PORT_START	/* IN1 */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT ( 0xfc, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H */
	PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X, 15, 20, 0, 0 )

	PORT_START	/* trackball V */
	PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y, 15, 20, 0, 0 )

	PORT_START	/* IN4 */
	PORT_BIT ( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT ( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT ( 0x04, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT ( 0x08, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT ( 0x10, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT ( 0x20, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT ( 0xc0, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( mplanets )
	PORT_START	/* DSW */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x02, "12000" )
	PORT_DIPNAME( 0x08, 0x00, "Allow Round Select" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x20, "5" )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Easy" )
	PORT_DIPSETTING(    0x00, "Medium" )
	PORT_DIPSETTING(    0x80, "Hard" )
	PORT_DIPSETTING(    0xc0, "Hardest" )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x3c, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x40, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V (dial) */
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL, 5, 10, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_BUTTON2 )
INPUT_PORTS_END

INPUT_PORTS_START( qbert )
	PORT_START      /* DSW */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Kicker" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_BITX( 0x08, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Auto Round Advance", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "SW5" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, "SW7" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "SW8" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNUSED )
/* 0x40 must be connected to the IP16 line */

	PORT_START      /* buttons */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_SERVICE( 0x40, IP_ACTIVE_LOW )
	PORT_BITX(0x80, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* joystick */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
INPUT_PORTS_END

INPUT_PORTS_START( qbertqub )
	PORT_START      /* DSW */
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x35, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x24, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x14, "A 1/1 B 4/1" )
	PORT_DIPSETTING(    0x30, "A 1/1 B 3/1" )
	PORT_DIPSETTING(    0x10, "A 1/1 B 2/1" )
	PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0x11, "A 2/3 B 2/1" )
	PORT_DIPSETTING(    0x15, "A 1/2 B 3/1" )
	PORT_DIPSETTING(    0x20, "A 1/2 B 2/1" )
	PORT_DIPSETTING(    0x21, "A 1/2 B 1/1" )
	PORT_DIPSETTING(    0x31, "A 1/2 B 1/5" )
	PORT_DIPSETTING(    0x04, "A 1/3 B 2/1" )
	PORT_DIPSETTING(    0x05, "A 1/3 B 1/1" )
	PORT_DIPSETTING(    0x35, DEF_STR( Free_Play ) )
/* 0x25 DEF_STR( 2C_1C )
   0x01 DEF_STR( 1C_1C )
   0x34 DEF_STR( Free_Play ) */
	PORT_DIPNAME( 0x02, 0x00, "1st Bonus Life" )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x02, "15000" )
	PORT_DIPNAME( 0x40, 0x00, "Additional Bonus Life" )
	PORT_DIPSETTING(    0x00, "20000" )
	PORT_DIPSETTING(    0x40, "25000" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x80, "Hard" )

	PORT_START      /* buttons */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_SERVICE( 0x40, IP_ACTIVE_LOW )
	PORT_BITX(0x80, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* joystick */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( krull )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x02, "Hard" )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPNAME( 0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, "Hexagon" )
	PORT_DIPSETTING(    0x00, "Roving" )
	PORT_DIPSETTING(    0x20, "Stationary" )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x40, "30000 30000" )
	PORT_DIPSETTING(    0x00, "30000 50000" )
	PORT_DIPSETTING(    0x80, "40000 50000" )
	PORT_DIPSETTING(    0xc0, "50000 75000" )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP     | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT  | IPF_8WAY )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN   | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT   | IPF_8WAY )
INPUT_PORTS_END

INPUT_PORTS_START( mach3 )
	PORT_START      /* DSW0 */
	/* TODO: values are different for 5 lives */
	PORT_DIPNAME( 0x09, 0x08, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x02, "Hard" )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPNAME( 0x24, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000 10000" )
	PORT_DIPSETTING(    0x04, "10000 20000" )
	PORT_DIPSETTING(    0x20, "10000 40000" )
	PORT_DIPSETTING(    0x24, "20000 60000" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_BITX(    0x80, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Infinite Lives", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( usvsthem )
	PORT_START      /* DSW0 */
	/* TODO: values are different for 5 lives */
	PORT_DIPNAME( 0x09, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Free_Play ) )
/*	PORT_DIPSETTING(    0x09, DEF_STR( Free_Play ) ) */
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x04, "Hard" )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( 3stooges )
	PORT_START	/* DSW */
	PORT_DIPNAME (0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING (   0x01, DEF_STR( Off ) )
	PORT_DIPSETTING (   0x00, DEF_STR( On ) )
	PORT_DIPNAME (0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING (   0x00, "Normal" )
	PORT_DIPSETTING (   0x02, "Hard" )
	PORT_DIPNAME (0x08, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING (   0x00, "3" )
	PORT_DIPSETTING (   0x08, "5" )
	PORT_DIPNAME (0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING (   0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING (   0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING (   0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING (   0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME (0x40, 0x00, "1st Bonus Life at" )
	PORT_DIPSETTING (   0x40, "10000" )
	PORT_DIPSETTING (   0x00, "20000" )
	PORT_DIPNAME (0x80, 0x00, "And Bonus Life Every" )
	PORT_DIPSETTING (   0x80, "10000" )
	PORT_DIPSETTING (   0x00, "20000" )

	PORT_START	/* IN1 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* joystick 2 (Moe) */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	/* the bottom four bits of the previous port are multiplexed among */
	/* three joysticks - the following port contains settings for the other two */
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
INPUT_PORTS_END

INPUT_PORTS_START( curvebal )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x08, 0x00, "2 Players Game" )
	PORT_DIPSETTING(    0x08, "1 Credit" )
	PORT_DIPSETTING(    0x00, "2 Credits" )
	PORT_DIPNAME( 0x11, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x10, "Medium" )
	PORT_DIPSETTING(    0x01, "Hard" )
	PORT_DIPSETTING(    0x11, "Hardest" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "Coins" )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x20, "French" )
	/* TODO: coinage is different when French is selected */
/* PORT_DIPNAME( 0xc2, 0x00, "French Coinage" )
PORT_DIPSETTING(    0x42, "A 3/1 B 1/2" )
PORT_DIPSETTING(    0x82, "A 1/5 B 1/2" )
PORT_DIPSETTING(    0x02, "A 2/1 B 2/3" )
PORT_DIPSETTING(    0xc0, "A 2/1 B 2/1" )
PORT_DIPSETTING(    0x80, "A 1/1 B 1/2" )
PORT_DIPSETTING(    0x40, "A 1/1 B 1/3" )
PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
PORT_DIPSETTING(    0xc2, DEF_STR( Free_Play ) ) */
	PORT_DIPNAME( 0xc2, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x42, "A 4/1 B 1/1" )
	PORT_DIPSETTING(    0x82, "A 3/1 B 1/1" )
	PORT_DIPSETTING(    0x02, "A 2/1 B 1/1" )
	PORT_DIPSETTING(    0xc0, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x80, "A 2/1 B 1/2" )
	PORT_DIPSETTING(    0x40, "A 2/1 B 1/3" )
	PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0xc2, DEF_STR( Free_Play ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER1, "Swing", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX(0x04, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2, "Pitch Left", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_PLAYER2, "Pitch Right", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x40, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_PLAYER1, "Bunt", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( screwloo )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Demo mode" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "1st Bonus Atom at" )
	PORT_DIPSETTING(    0x00, "5000" )
	PORT_DIPSETTING(    0x04, "20000" )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x50, 0x40, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x20, 0x00, "1st Bonus Hand at" )
	PORT_DIPSETTING(    0x00, "25000" )
	PORT_DIPSETTING(    0x20, "50000" )
	PORT_DIPNAME( 0x80, 0x00, "Hands" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x80, "5" )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT  | IPF_8WAY )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN  | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP    | IPF_8WAY )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_PLAYER1, "Start 2P", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER1, "Start 1P", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( insector )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "25000" )
	PORT_DIPSETTING(    0x01, "30000" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "Demo mode" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x50, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Cocktail ) )

	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_COCKTAIL )
	PORT_SERVICE( 0x40, IP_ACTIVE_LOW )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_COCKTAIL )
INPUT_PORTS_END



/* the games can store char gfx data in either a 4k RAM area (128 chars), or */
/* a 8k ROM area (256 chars). */
static struct GfxLayout charRAMlayout =
{
	8,8,    /* 8*8 characters */
	128,    /* 128 characters */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 },
	{ 0, 4, 8, 12, 16, 20, 24, 28},
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32 },
	32*8    /* every char takes 32 consecutive bytes */
};

static struct GfxLayout charROMlayout =
{
	8,8,    /* 8*8 characters */
	256,    /* 256 characters */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 },
	{ 0, 4, 8, 12, 16, 20, 24, 28},
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32 },
	32*8    /* every char takes 32 consecutive bytes */
};

static struct GfxLayout spritelayout =
{
	16,16,
	RGN_FRAC(1,4),
	4,      /* 4 bits per pixel */
	{ RGN_FRAC(0,4), RGN_FRAC(1,4), RGN_FRAC(2,4), RGN_FRAC(3,4) },
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,
			8*16, 9*16, 10*16, 11*16, 12*16, 13*16, 14*16, 15*16 },
	32*8
};

static struct GfxDecodeInfo charRAM_gfxdecodeinfo[] =
{
	{ 0,           0x4000, &charRAMlayout, 0, 1 },	/* the game dynamically modifies this */
	{ REGION_GFX2, 0x0000, &spritelayout,  0, 1 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo charROM_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x0000, &charROMlayout, 0, 1 },
	{ REGION_GFX2, 0x0000, &spritelayout,  0, 1 },
	{ -1 } /* end of array */
};



static struct DACinterface dac1_interface =
{
	1,
	{ 50 }
};

static struct DACinterface dac2_interface =
{
	2,
	{ 25, 25 }
};

static const char *reactor_sample_names[] =
{
	"*reactor",
	"fx_53.wav", /* "8 left" */
	"fx_54.wav", /* "16 left" */
	"fx_55.wav", /* "24 left" */
	"fx_56.wav", /* "32 left" */
	"fx_57.wav", /* "40 left" */
	"fx_58.wav", /* "warning, core unstable" */
	"fx_59.wav", /* "bonus" */
	"fx_31.wav", /* "chamber activated" */
	"fx_39a.wav", /* "2000" */
	"fx_39b.wav", /* "5000" */
	"fx_39c.wav", /* "10000" */
	"fx_39d.wav", /* "15000" */
	"fx_39e.wav", /* "20000" */
	"fx_39f.wav", /* "25000" */
	"fx_39g.wav", /* "30000" */
	"fx_39h.wav", /* "35000" */
	"fx_39i.wav", /* "40000" */
	"fx_39j.wav", /* "45000" */
	"fx_39k.wav", /* "50000" */
	"fx_39l.wav", /* "55000" */
     0	/* end of array */
};

static const char *qbert_sample_names[] =
{
	"*qbert",
	"fx_17a.wav", /* random speech, voice clock 255 */
	"fx_17b.wav", /* random speech, voice clock 255 */
	"fx_17c.wav", /* random speech, voice clock 255 */
	"fx_17d.wav", /* random speech, voice clock 255 */
	"fx_17e.wav", /* random speech, voice clock 255 */
	"fx_17f.wav", /* random speech, voice clock 255 */
	"fx_17g.wav", /* random speech, voice clock 255 */
	"fx_17h.wav", /* random speech, voice clock 255 */
	"fx_18a.wav", /* random speech, voice clock 176 */
	"fx_18b.wav", /* random speech, voice clock 176 */
	"fx_18c.wav", /* random speech, voice clock 176 */
	"fx_18d.wav", /* random speech, voice clock 176 */
	"fx_18e.wav", /* random speech, voice clock 176 */
	"fx_18f.wav", /* random speech, voice clock 176 */
	"fx_18g.wav", /* random speech, voice clock 176 */
	"fx_18h.wav", /* random speech, voice clock 176 */
	"fx_19a.wav", /* random speech, voice clock 128 */
	"fx_19b.wav", /* random speech, voice clock 128 */
	"fx_19c.wav", /* random speech, voice clock 128 */
	"fx_19d.wav", /* random speech, voice clock 128 */
	"fx_19e.wav", /* random speech, voice clock 128 */
	"fx_19f.wav", /* random speech, voice clock 128 */
	"fx_19g.wav", /* random speech, voice clock 128 */
	"fx_19h.wav", /* random speech, voice clock 128 */
	"fx_20a.wav", /* random speech, voice clock 96 */
	"fx_20b.wav", /* random speech, voice clock 96 */
	"fx_20c.wav", /* random speech, voice clock 96 */
	"fx_20d.wav", /* random speech, voice clock 96 */
	"fx_20e.wav", /* random speech, voice clock 96 */
	"fx_20f.wav", /* random speech, voice clock 96 */
	"fx_20g.wav", /* random speech, voice clock 96 */
	"fx_20h.wav", /* random speech, voice clock 96 */
	"fx_21a.wav", /* random speech, voice clock 62 */
	"fx_21b.wav", /* random speech, voice clock 62 */
	"fx_21c.wav", /* random speech, voice clock 62 */
	"fx_21d.wav", /* random speech, voice clock 62 */
	"fx_21e.wav", /* random speech, voice clock 62 */
	"fx_21f.wav", /* random speech, voice clock 62 */
	"fx_21g.wav", /* random speech, voice clock 62 */
	"fx_21h.wav", /* random speech, voice clock 62 */
	"fx_22.wav", /* EH2 with decreasing voice clock */
	"fx_23.wav", /* O1 with varying voice clock */
	"fx_28.wav", /* "hello, I'm ready" */
	"fx_36.wav", /* "byebye" */
	"knocker.wav",
	0	/* end of array */
};

static struct Samplesinterface qbert_samples_interface =
{
 	1,	/* one channel */
	100,	/* volume */
	qbert_sample_names
};

static struct Samplesinterface reactor_samples_interface =
{
	1,	/* one channel */
	100,	/* volume */
	reactor_sample_names
};

#define gottlieb_samples_interface qbert_samples_interface	/* not used */
#define krull_samples_interface qbert_samples_interface		/* not used */


static struct AY8910interface ay8910_interface =
{
	2,	/* 2 chips */
	2000000,	/* 2 MHz */
	{ 25, 25 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};



/********************************************************************
*
*  Machine Driver macro
*  ====================
*
*  There are two versions of the machine driver,
*  one for revision 1 sound board (1x6502 + DAC + Votrax),
*  one for revision 2 (2x6502 + DAC + GI SP-0250 speech chip + 2x8910).
*  They are identical apart form the sound CPU subsections.
*
********************************************************************/

/* games using the revision 1 sound board */
static MACHINE_DRIVER_START( gottlieb )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", I86, 5000000)	/* 5 MHz */
	MDRV_CPU_MEMORY(gottlieb_readmem,gottlieb_writemem)
	MDRV_CPU_VBLANK_INT(gottlieb_interrupt,1)

	MDRV_CPU_ADD_TAG("sound", M6502, 3579545/4)	/* the board can be set to /2 as well */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(gottlieb_sound_readmem,gottlieb_sound_writemem)
								/* NMIs are triggered by the Votrax SC-01 */
	MDRV_FRAMES_PER_SECOND(61)
	MDRV_VBLANK_DURATION(1018)	/* frames per second, vblank duration */

	MDRV_MACHINE_INIT(gottlieb)
	MDRV_NVRAM_HANDLER(generic_1fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(charROM_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)

	MDRV_VIDEO_START(gottlieb)
	MDRV_VIDEO_UPDATE(gottlieb)

	/* sound hardware */
	MDRV_SOUND_ADD(DAC, dac1_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( reactor )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(gottlieb)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(reactor_readmem, reactor_writemem)

	MDRV_NVRAM_HANDLER(NULL)

	/* video hardware */
	MDRV_GFXDECODE(charRAM_gfxdecodeinfo)
	MDRV_SOUND_ADD(SAMPLES, reactor_samples_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( qbert )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(gottlieb)

	/* video hardware */
	MDRV_SOUND_ADD(SAMPLES, qbert_samples_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( krull )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(gottlieb)

	/* video hardware */
	MDRV_GFXDECODE(charRAM_gfxdecodeinfo)
MACHINE_DRIVER_END


/* games using the revision 2 sound board */
static MACHINE_DRIVER_START( gottlieb2 )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", I86, 5000000)	/* 5 MHz */
	MDRV_CPU_MEMORY(gottlieb_readmem,gottlieb_writemem)
	MDRV_CPU_VBLANK_INT(gottlieb_interrupt,1)

	MDRV_CPU_ADD_TAG("sound", M6502, 1000000)	/* 1 MHz */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(stooges_sound_readmem,stooges_sound_writemem)

	MDRV_CPU_ADD_TAG("sound2", M6502, 1000000)	/* 1 MHz */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(stooges_sound2_readmem,stooges_sound2_writemem)
								/* NMIs are triggered by the Votrax SC-01 */
	MDRV_FRAMES_PER_SECOND(61)
	MDRV_VBLANK_DURATION(1018)	/* frames per second, vblank duration */

	MDRV_MACHINE_INIT(gottlieb)
	MDRV_NVRAM_HANDLER(generic_1fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(charROM_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)

	MDRV_VIDEO_START(gottlieb)
	MDRV_VIDEO_UPDATE(gottlieb)

	/* sound hardware */
	MDRV_SOUND_ADD(DAC, dac2_interface)
	MDRV_SOUND_ADD(AY8910, ay8910_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mach3 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(gottlieb2)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(gottlieb_readmem,usvsthem_writemem)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( usvsthem )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mach3)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( stooges )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(gottlieb2)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(stooges_readmem,stooges_writemem)

	/* video hardware */
	MDRV_GFXDECODE(charRAM_gfxdecodeinfo)
MACHINE_DRIVER_END


/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( reactor )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "rom7",         0x8000, 0x1000, CRC(a62d86fd) SHA1(9ffce668c6f0155568ccb62a6f12a9513f3c513f) )
	ROM_LOAD( "rom6",         0x9000, 0x1000, CRC(6ed841f3) SHA1(b94272457e6a2d0ec46b510d71d69b4ab92a44aa) )
	ROM_LOAD( "rom5",         0xa000, 0x1000, CRC(d90576a3) SHA1(52af7164dc59b9f7d8a1b3ef9d4c1fc81496899f) )
	ROM_LOAD( "rom4",         0xb000, 0x1000, CRC(0155daae) SHA1(3a94952c2830c7bea126cf39b84108936d96dc0e) )
	ROM_LOAD( "rom3",         0xc000, 0x1000, CRC(f8881385) SHA1(baa7ef7309aab9b382442b25ede5d7ecd6e2c370) )
	ROM_LOAD( "rom2",         0xd000, 0x1000, CRC(3caba35b) SHA1(4d3b894821a2cff0ef55cec2b1f62f2e33009b4d) )
	ROM_LOAD( "rom1",         0xe000, 0x1000, CRC(944e1ddf) SHA1(6b487f1cb405e2ba9345190e8ab6022c790882c1) )
	ROM_LOAD( "rom0",         0xf000, 0x1000, CRC(55930aed) SHA1(37ed60386935741e8cc0b8750bfcdf6f54c1bf9e) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1",         0xf000, 0x800, CRC(d958a0fd) SHA1(3c383076c68a929f96d844e89b09f3075f331906) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "snd2",         0xf800, 0x800, CRC(5dc86942) SHA1(a449fcfb25521a0e7523184518b5204dac56e5f8) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	/* 0000-0fff empty */
	ROM_LOAD( "fg3",          0x1000, 0x1000, CRC(8416ad53) SHA1(f868259b97675e58b6a7f1dc3c2a4ecf6aa0570e) )	/* sprites */
	/* 2000-2fff empty */
	ROM_LOAD( "fg2",          0x3000, 0x1000, CRC(5489605a) SHA1(f4bbaaa8cd881dc164b118d1e516edeeea54c1d8) )
	/* 4000-4fff empty */
	ROM_LOAD( "fg1",          0x5000, 0x1000, CRC(18396c57) SHA1(39d90a842a03091414ed58d4128b524ecc20c9f1) )
	/* 6000-6fff empty */
	ROM_LOAD( "fg0",          0x7000, 0x1000, CRC(d1f20e15) SHA1(dba9aa0fec8b720a33d78b3dd1d7f74040048f7e) )
ROM_END

ROM_START( mplanets )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "rom4",         0x6000, 0x2000, CRC(5402077f) SHA1(f4e8699ab3c6dfc0f86b6df86d2a5b35caf2ca73) )
	ROM_LOAD( "rom3",         0x8000, 0x2000, CRC(5d18d740) SHA1(30307d98704c49dec5aecd0a1ec2f06f1869a5d2) )
	ROM_LOAD( "rom2",         0xa000, 0x2000, CRC(960c3bb1) SHA1(305a7904fa8c0b9823ad186d1c5c7460c0900bad) )
	ROM_LOAD( "rom1",         0xc000, 0x2000, CRC(eb515f10) SHA1(31c3519328eba7adc4a3b0adcc0384f606d81a57) )
	ROM_LOAD( "rom0",         0xe000, 0x2000, CRC(74de78aa) SHA1(7ebd02e660c1413eff284a7ca77feeff41c1e2b7) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1",         0xf000, 0x800, CRC(453193a1) SHA1(317ec81f71661eaa92624c0304a52b635dcd5613) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "snd2",         0xf800, 0x800, CRC(f5ffc98f) SHA1(516e895df94942fc51f1b51eb9316d4296df82e7) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "bg0",          0x0000, 0x1000, CRC(709aa24c) SHA1(95be072bf63320f4b44feaf88003ba011754e20f) )	/* chars */
	ROM_LOAD( "bg1",          0x1000, 0x1000, CRC(4921e345) SHA1(7b6e03458222be501ed150ffbd489433027fc6cb) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg3",          0x0000, 0x2000, CRC(c990b39f) SHA1(d1b6060744b78df430e914504b20e8693829bbd5) )	/* sprites */
	ROM_LOAD( "fg2",          0x2000, 0x2000, CRC(735e2522) SHA1(9fac59e9b1d81695d3da32107d640726cf96e31a) )
	ROM_LOAD( "fg1",          0x4000, 0x2000, CRC(6456cc1c) SHA1(12c20f0ce49a7d3579049e8ba95e542c4aaee241) )
	ROM_LOAD( "fg0",          0x6000, 0x2000, CRC(a920e325) SHA1(60f15d5014a55d9c18b06c17c7587d45716619e4) )
ROM_END

ROM_START( qbert )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "qb-rom2.bin",  0xa000, 0x2000, CRC(fe434526) SHA1(4cfc5d52dd6c82163e035af82d6112c0c93a3797) )
	ROM_LOAD( "qb-rom1.bin",  0xc000, 0x2000, CRC(55635447) SHA1(ca6acdef1c9e06b33efe1f0a2df2dfb03723cfbe) )
	ROM_LOAD( "qb-rom0.bin",  0xe000, 0x2000, CRC(8e318641) SHA1(7f8f66d1e6a7905e93cce07fc92e8801370b7194) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, CRC(15787c07) SHA1(8b7d03fbf2ebaa71b3a7e2f636a0d1bb9b796e43) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, CRC(58437508) SHA1(09d8053e7e99679b602dcda230d64db7fe6cb7f5) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, CRC(7a9ba824) SHA1(12aa6df499eb6996ee35f56acac403ff6290f844) )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, CRC(22e5b891) SHA1(5bb67e333255c0ea679ab4312256a8a71a950db8) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, CRC(dd436d3a) SHA1(ae16087a6ceec84551b5d7aae4036e0ed432cbb7) )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, CRC(f69b9483) SHA1(06894a1474c79c1274efbd32d7371179e7e0a661) )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, CRC(224e8356) SHA1(f7f26b879aa8b964ff6311136ed8157e44de736c) )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, CRC(2f695b85) SHA1(807d16459838f129e10b913890bbc95065d5dd40) )
ROM_END

ROM_START( qbertjp )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "qbj-rom2.bin", 0xa000, 0x2000, CRC(67bb1cb2) SHA1(23a7f8c86d6db9220a98b3f630c5d000e80f2d53) )
	ROM_LOAD( "qbj-rom1.bin", 0xc000, 0x2000, CRC(c61216e7) SHA1(e727b85dddc2963e33af6c02b675243f6fbe2710) )
	ROM_LOAD( "qbj-rom0.bin", 0xe000, 0x2000, CRC(69679d5c) SHA1(996d45517d0c01a1952fead05dbe201dbb7dca35) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, CRC(15787c07) SHA1(8b7d03fbf2ebaa71b3a7e2f636a0d1bb9b796e43) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, CRC(58437508) SHA1(09d8053e7e99679b602dcda230d64db7fe6cb7f5) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, CRC(7a9ba824) SHA1(12aa6df499eb6996ee35f56acac403ff6290f844) )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, CRC(22e5b891) SHA1(5bb67e333255c0ea679ab4312256a8a71a950db8) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, CRC(dd436d3a) SHA1(ae16087a6ceec84551b5d7aae4036e0ed432cbb7) )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, CRC(f69b9483) SHA1(06894a1474c79c1274efbd32d7371179e7e0a661) )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, CRC(224e8356) SHA1(f7f26b879aa8b964ff6311136ed8157e44de736c) )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, CRC(2f695b85) SHA1(807d16459838f129e10b913890bbc95065d5dd40) )
ROM_END

ROM_START( myqbert )
 ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
 ROM_LOAD( "mqb-rom2.bin",  0xa000, 0x2000, CRC(6860f957) SHA1(ebd68aeb6d54868295bd20cf64ee0187a52df0e3) )
 ROM_LOAD( "mqb-rom1.bin",  0xc000, 0x2000, CRC(11f0a4e4) SHA1(a805e51c40042fae209ace277abd9b35a990905b) )
 ROM_LOAD( "mqb-rom0.bin",  0xe000, 0x2000, CRC(12a90cb2) SHA1(a33203aea79fe43d1233a16e3fdddaceac6e4a20) )

 ROM_REGION( 0x10000, REGION_CPU2, 0 ) /* 64k for sound cpu */
 ROM_LOAD( "mqb-snd1.bin",  0xf000, 0x800, CRC(e704b450) SHA1(d509f54658e9f0264b9ab865a6f36e5423a28904) )
 ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
 ROM_LOAD( "mqb-snd2.bin",  0xf800, 0x800, CRC(c6a98bf8) SHA1(cc5b5bb5966f5d79226f1f665a3f9fc934f4ef7f) )
 ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

 ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
 ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, CRC(7a9ba824) SHA1(12aa6df499eb6996ee35f56acac403ff6290f844) ) /* chars */
 ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, CRC(22e5b891) SHA1(5bb67e333255c0ea679ab4312256a8a71a950db8) )

 ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
 ROM_LOAD( "mqb-fg3.bin",   0x0000, 0x2000, CRC(8b5d0852) SHA1(e2cf1679a7ec9e88b254b0a8d690a74d88db0cdc) ) /* sprites */
 ROM_LOAD( "mqb-fg2.bin",   0x2000, 0x2000, CRC(823f1e57) SHA1(a7305815d71d6e3b1c92c387a675e969edc77b7d) )
 ROM_LOAD( "mqb-fg1.bin",   0x4000, 0x2000, CRC(05343ae6) SHA1(6ae3e6949c9eb0df85216575ffd21adc939df0a2) )
 ROM_LOAD( "mqb-fg0.bin",   0x6000, 0x2000, CRC(abc71bdd) SHA1(b72c6867d8e342a6794a4bbab991761c01cfae44) )
ROM_END

ROM_START( insector )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "rom3",         0x8000, 0x2000, CRC(640881fd) SHA1(2832183e41ae7e631b61e4845fa68ce1c49edf29) )
	ROM_LOAD( "rom2",         0xa000, 0x2000, CRC(456bc3f4) SHA1(b61a56a65639f97399a8a3a4293ac2292edfd159) )
	ROM_LOAD( "rom1",         0xc000, 0x2000, CRC(706962af) SHA1(e40b567fdf6a3f7c6485808b4db45cea322c7724) )
	ROM_LOAD( "rom0",         0xe000, 0x2000, CRC(31cee24b) SHA1(3d21f5d530cc022f9633ad487e13a664848dd3e6) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "sound",        0xf000, 0x1000, NO_DUMP )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "bg0",          0x0000, 0x1000, CRC(0dc2037e) SHA1(aa3fdec7884aad782e430182326f5b113f59bf00) )	/* chars */
	ROM_LOAD( "bg1",          0x1000, 0x1000, CRC(3dd73b94) SHA1(98b6592a907d6cb6a692c576f757bb612f8d3b72) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg3",          0x0000, 0x2000, CRC(9bbf5b6b) SHA1(3866d4f5231140e0b7ffe5daa4752d09d3bd7241) )	/* sprites */
	ROM_LOAD( "fg2",          0x2000, 0x2000, CRC(5adf9986) SHA1(2ef0d002d7ab4c9199e40dfcb25747564ecc0495) )
	ROM_LOAD( "fg1",          0x4000, 0x2000, CRC(4bb16111) SHA1(88ac8c957c6968d355c494bf95ccf8a99152e5e9) )
	ROM_LOAD( "fg0",          0x6000, 0x2000, CRC(965f6b76) SHA1(faec9d43f39e730eefd08d4c3337ed4b230bf7e1) )
ROM_END

ROM_START( screwloo )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "rom4",         0x6000, 0x2000, CRC(744a2513) SHA1(d6e8e016d9af984fae9b6667bedeea069637bd3a) )
	ROM_LOAD( "rom3",         0x8000, 0x2000, CRC(ffde5b5d) SHA1(50ac0e600935b8b1f8b68c40ad21a179b176e535) )
	ROM_LOAD( "rom2",         0xa000, 0x2000, CRC(97932b05) SHA1(6807c6e08edeb93aa7b4969eb78de3cf88a00b72) )
	ROM_LOAD( "rom1",         0xc000, 0x2000, CRC(571b65ca) SHA1(75077f4fab296b3802271fa77af588003570cde6) )
	ROM_LOAD( "rom0",         0xe000, 0x2000, CRC(6447fe54) SHA1(6391c841cafd35dd315d9fac99ed5d8304018747) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "drom1",        0xe000, 0x2000, CRC(ae965ade) SHA1(84a690cba8990fe6406b7cfbd6ea643a48446567) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )	/* 64k for second sound cpu */
	ROM_LOAD( "yrom1",        0xe000, 0x2000, CRC(3719b0b5) SHA1(4f215ca2f15956374c4cd9484b6798f1c4d60fc7) )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "bg0",          0x0000, 0x1000, CRC(1fd5b649) SHA1(4e2127a4458d54f094934c6f5f154a6db5d0c8b0) )	/* chars */
	ROM_LOAD( "bg1",          0x1000, 0x1000, CRC(c8ddb8ba) SHA1(58137a5043157d08af4d628cf5384d7f530393f7) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg3",          0x0000, 0x2000, CRC(97d4e63b) SHA1(e06ecb73148f16b33132767f8f0dffc37eeba5f1) )	/* sprites */
	ROM_LOAD( "fg2",          0x2000, 0x2000, CRC(f76e56ca) SHA1(32252d424f31aa899d6c614f1d1006c2ef49df84) )
	ROM_LOAD( "fg1",          0x4000, 0x2000, CRC(698c395f) SHA1(46c6700b2943c35c9ac45791652eafbb4a4e059a) )
	ROM_LOAD( "fg0",          0x6000, 0x2000, CRC(f23269fb) SHA1(9a418a372da18cf33bcfba07202cf2ac7a1dcd1d) )
ROM_END

ROM_START( sqbert )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "qb-rom2.bin",  0xa000, 0x2000, CRC(1e3d4038) SHA1(d4402c5d16c0aa55efbceb83f0b30082b8434df7) )
	ROM_LOAD( "qb-rom1.bin",  0xc000, 0x2000, CRC(eaf3076c) SHA1(749a87b3c40ba0a2ecd2ca962786e066daf63e30) )
	ROM_LOAD( "qb-rom0.bin",  0xe000, 0x2000, CRC(61260a7e) SHA1(e2028a453aa34aaffa2c465f64a963504315df3c) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, CRC(15787c07) SHA1(8b7d03fbf2ebaa71b3a7e2f636a0d1bb9b796e43) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, CRC(58437508) SHA1(09d8053e7e99679b602dcda230d64db7fe6cb7f5) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, CRC(c3118eef) SHA1(2c320eb8aae8841046ac3fca3bdaeeba778360e4) )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, CRC(4f6d8075) SHA1(dc1897f939c8d837627ff6e06609afe305566a3b) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, CRC(ee595eda) SHA1(11777d95ba79bd0ec7b964b76c1dc129db857816) )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, CRC(59884c78) SHA1(5e77ef46ccd55f79a5fa90521baa7c22e3783fe5) )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, CRC(2a60e3ad) SHA1(9ed83017f6c8e44337ad76c68b095f2c3300aadb) )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, CRC(b11ad9d8) SHA1(5264598f33aa76455ae4107d0f265c2a372ed67a) )
ROM_END

ROM_START( qbertqub )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "qq-rom3.bin",  0x8000, 0x2000, CRC(c4dbdcd7) SHA1(34aaa4762073680e2b4d024ce7106315ffc6bcf3) )
	ROM_LOAD( "qq-rom2.bin",  0xa000, 0x2000, CRC(21a6c6cc) SHA1(6d4d81d9ad85be3792584e39dbeaf0dfeeda1503) )
	ROM_LOAD( "qq-rom1.bin",  0xc000, 0x2000, CRC(63e6c43d) SHA1(9435eb06dc069e5bf1c439f0c772fef3183745b0) )
	ROM_LOAD( "qq-rom0.bin",  0xe000, 0x2000, CRC(8ddbe438) SHA1(31112d711af5d4039491e99a0be0c088b3272482) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, CRC(15787c07) SHA1(8b7d03fbf2ebaa71b3a7e2f636a0d1bb9b796e43) )
	ROM_RELOAD(               0x7000, 0x800 ) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, CRC(58437508) SHA1(09d8053e7e99679b602dcda230d64db7fe6cb7f5) )
	ROM_RELOAD(               0x7800, 0x800 ) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "qq-bg0.bin",   0x0000, 0x1000, CRC(050badde) SHA1(d049367e262cc6080e01d32227e86310166e00bb) )	/* chars */
	ROM_LOAD( "qq-bg1.bin",   0x1000, 0x1000, CRC(8875902f) SHA1(715e05b555d52c0445c4bec7fb6d1c02053505e6) )

	ROM_REGION( 0x10000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "qq-fg3.bin",   0x0000, 0x4000, CRC(91a949cc) SHA1(bd27a6bb744f73cb918318f1bc3aa5bde39417c2) )	/* sprites */
	ROM_LOAD( "qq-fg2.bin",   0x4000, 0x4000, CRC(782d9431) SHA1(04d46014e9a5b80b8215b5338fa9de27e530e4a9) )
	ROM_LOAD( "qq-fg1.bin",   0x8000, 0x4000, CRC(71c3ac4c) SHA1(ed3444b3c8bd332b83f2700f1d6ea6bc9e185d8c) )
	ROM_LOAD( "qq-fg0.bin",   0xc000, 0x4000, CRC(6192853f) SHA1(907e62a3835bacda4104dcc1d7e946e4e8e821ef) )
ROM_END

ROM_START( krull )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "ram2.bin",     0x1000, 0x1000, CRC(302feadf) SHA1(9d70de35e4f0490dc4e601070993ad146f250dea) )
	ROM_LOAD( "ram4.bin",     0x2000, 0x1000, CRC(79355a60) SHA1(57ad5c904b9ac4bf7c7d828bf755bbcbba6a4fd7) )
	ROM_LOAD( "rom4.bin",     0x6000, 0x2000, CRC(2b696394) SHA1(b18270f4ad97743f6ff8c4cbc93e523c77a8e794) )
	ROM_LOAD( "rom3.bin",     0x8000, 0x2000, CRC(14b0ee42) SHA1(276c4008a013806b3989c529f41cbc358ec49fd6) )
	ROM_LOAD( "rom2.bin",     0xa000, 0x2000, CRC(b5fad94a) SHA1(1bae895fbdd658cfb56c53cc2139282cc1e778de) )
	ROM_LOAD( "rom1.bin",     0xc000, 0x2000, CRC(1ad956a3) SHA1(f5b74b196fe1bd9ab48336e0051cbf29c650cfc1) )
	ROM_LOAD( "rom0.bin",     0xe000, 0x2000, CRC(a466afae) SHA1(d691cbb46e8c3b71f9b1688d7fcef36df82aa854) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1.bin",     0xe000, 0x1000, CRC(dd2b30b4) SHA1(f01cb64932493bf69d4fc75a7fa809ff6f6e4263) )
	ROM_RELOAD(               0x6000, 0x1000 ) /* A15 is not decoded */
	ROM_LOAD( "snd2.bin",     0xf000, 0x1000, CRC(8cab901b) SHA1(b886532828efc8cf442e2ee4ebbfe37acd489f62) )
	ROM_RELOAD(               0x7000, 0x1000 ) /* A15 is not decoded */

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg3.bin",      0x0000, 0x2000, CRC(82d77a45) SHA1(753476609c4bf4f0f0cd28d61fd8aef6967bda57) )	/* sprites */
	ROM_LOAD( "fg2.bin",      0x2000, 0x2000, CRC(25a24317) SHA1(33d2c23a388b09c4a09b9893648c30fbd5482cc3) )
	ROM_LOAD( "fg1.bin",      0x4000, 0x2000, CRC(7e3ad7b0) SHA1(0de86e632e5a9e6c1ec82550b15dc25a17ab7066) )
	ROM_LOAD( "fg0.bin",      0x6000, 0x2000, CRC(7402dc19) SHA1(d6d1b8aca8e9ee3bdc57f4474d22b405963909ec) )
ROM_END

ROM_START( mach3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "m3rom4.bin",   0x6000, 0x2000, CRC(8bfd5a44) SHA1(61f5c6c39047c1d0296e2cacce2be9525cb47176) )
	ROM_LOAD( "m3rom3.bin",   0x8000, 0x2000, CRC(b1b045af) SHA1(4e71ca4661bf5daaf9e2ffbb930ac3b13e2e57bd) )
	ROM_LOAD( "m3rom2.bin",   0xa000, 0x2000, CRC(fbdfb03d) SHA1(61d587558fc036fc3b55d3e36ebb6940e9eda258) )
	ROM_LOAD( "m3rom1.bin",   0xc000, 0x2000, CRC(3b0ba80b) SHA1(bc7e961311b40f05f2998f10f0a68f2e515c8e66) )
	ROM_LOAD( "m3rom0.bin",   0xe000, 0x2000, CRC(70c12bf4) SHA1(c26127b6e2a16791b3be8abac93be6af4f30fb3b) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "m3drom1.bin",  0xf000, 0x1000, CRC(a6e29212) SHA1(a73aafc2efa99e9ae0aecbb6075a10f7178ac938) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )	/* 64k for second sound cpu */
	ROM_LOAD( "m3yrom1.bin",  0xf000, 0x1000, CRC(eddf8872) SHA1(29ed0d1828639849bab826b3e2ab4eefac45fd85) )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "mach3bg0.bin", 0x0000, 0x1000, CRC(ea2f5257) SHA1(664502dd2b7ee4ce96820da532615f3902b45557) )
	ROM_LOAD( "mach3bg1.bin", 0x1000, 0x1000, CRC(f543e4ce) SHA1(2a1b7dbbcd9756f836ca2e42973043b98a303082) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "mach3fg3.bin", 0x0000, 0x2000, CRC(472128b4) SHA1(8c6f36cab5ec8abb6db2e6d52530560664b950fe) )
	ROM_LOAD( "mach3fg2.bin", 0x2000, 0x2000, CRC(2a59e99e) SHA1(5c1faa244fc0f53cc2a52c8d4d40fb178706c2ed) )
	ROM_LOAD( "mach3fg1.bin", 0x4000, 0x2000, CRC(9b88767b) SHA1(8071e11906b3f0026f9a210cc5a236d95ca1f659) )
	ROM_LOAD( "mach3fg0.bin", 0x6000, 0x2000, CRC(0bae12a5) SHA1(7bc0b82ccab0e4498a7a2a9dc85f03125f25826e) )

	ROM_REGION( 1024*1024, REGION_USER1, 0)	/* about 30 min of target data from laserdisc */
	ROM_LOAD( "m3target.bin", 0, 1024*1024, CRC(6e779a6f) SHA1(e556ad438e637a71f17ea04088de10b39b45f8df) )
ROM_END

ROM_START( usvsthem )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "usvs.rm4",     0x6000, 0x2000, CRC(0d7a4072) SHA1(84a7eec31037243185ab40ab269be0f83946ebd5) )
	ROM_LOAD( "usvs.rm3",     0x8000, 0x2000, CRC(6f32a671) SHA1(d54c1e09988f78ea706b4b0da675b27d9d407696) )
	ROM_LOAD( "usvs.rm2",     0xa000, 0x2000, CRC(36770716) SHA1(797b48ef48f563ed21ea263297fe0ed61c69f41e) )
	ROM_LOAD( "usvs.rm1",     0xc000, 0x2000, CRC(697bc989) SHA1(ebfc0868f949e5aba1efb8fbce06f795888d8e00) )
	ROM_LOAD( "usvs.rm0",     0xe000, 0x2000, CRC(30cf6bd9) SHA1(527ad3b96ea4a77f6d6f8a89a9215da490292297) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "usvsdrom.1",   0xe000, 0x2000, CRC(c0b5cab0) SHA1(b18e8fd9837bb52d6b3d86f2b652f6573c7cd3b3) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )	/* 64k for second sound cpu */
	ROM_LOAD( "usvsyrom.1",   0xe000, 0x2000, CRC(c3d245ca) SHA1(d281b139ae6c58e855b2914a24ca4bc5f8800681) )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "usvs.bg0",     0x0000, 0x1000, CRC(8a0de09f) SHA1(6b56ffbe8569625ff307510645683b4e3f04e753) )
	ROM_LOAD( "usvs.bg1",     0x1000, 0x1000, CRC(6fb64d3c) SHA1(0cf0fd60995e56eddc56c7fac6280abe3b87dbc9) )

	ROM_REGION( 0x10000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "usvs.fg3",     0x0000, 0x4000, CRC(98703015) SHA1(120e87ef640db3f23c1d2c0abf4b6acff7714512) )
	ROM_LOAD( "usvs.fg2",     0x4000, 0x4000, CRC(d3990707) SHA1(81d58f6bc6ec04b95036f81c4cd3516d0adf348e) )
	ROM_LOAD( "usvs.fg1",     0x8000, 0x4000, CRC(a2057430) SHA1(e24aa35cb27fa41b75f5c01f4c083dc6eeb04c0d) )
	ROM_LOAD( "usvs.fg0",     0xc000, 0x4000, CRC(7734e53f) SHA1(c1307596ba098c98e741f3c00686b514587e1d0a) )
ROM_END

ROM_START( 3stooges )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "gv113ram.4",   0x2000, 0x1000, CRC(533bff2a) SHA1(58d0be8add4b02dc3e27cf6b17a05baf4304f3ce) )
	ROM_LOAD( "gv113rom.4",   0x6000, 0x2000, CRC(8b6e52b8) SHA1(6e17e11afce92a7fa1735a724f0c0faf9375ac89) )
	ROM_LOAD( "gv113rom.3",   0x8000, 0x2000, CRC(b816d8c4) SHA1(86e16888492390034ac04e3f6a9f56422575c778) )
	ROM_LOAD( "gv113rom.2",   0xa000, 0x2000, CRC(b45b2a79) SHA1(7d0b19bec462ab67f518361afdf4b6982829ed07) )
	ROM_LOAD( "gv113rom.1",   0xc000, 0x2000, CRC(34ab051e) SHA1(df416aaf34d6bbbdd79ae633842355a292ed935d) )
	ROM_LOAD( "gv113rom.0",   0xe000, 0x2000, CRC(ab124329) SHA1(de1bc721eea74426035eec10a389f77b435aa9b9) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "drom1",        0xe000, 0x2000, CRC(87a9fa10) SHA1(9c07837dce1384d6b51b716aa8ceeb5bc247a911) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )	/* 64k for second sound cpu */
	ROM_LOAD( "yrom2",        0xc000, 0x2000, CRC(90f9c940) SHA1(646dacc902cf235948ac9c064c92390e2764370b) )
	ROM_LOAD( "yrom1",        0xe000, 0x2000, CRC(55f8ab30) SHA1(a6b6318f12fd4a1fab61b82cde33759da615d5b2) )

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "gv113fg3",     0x0000, 0x2000, CRC(28071212) SHA1(33ce5cfae3491658f8b4cb977dc2da0a75dffee4) )	/* sprites */
	ROM_LOAD( "gv113fg2",     0x2000, 0x2000, CRC(9fa3dfde) SHA1(693327c60691748bf16e486b2962ebe019009a69) )
	ROM_LOAD( "gv113fg1",     0x4000, 0x2000, CRC(fb223854) SHA1(b1eaad971edd2f3a2aed66c5f9d0f27d6ffd5466) )
	ROM_LOAD( "gv113fg0",     0x6000, 0x2000, CRC(95762c53) SHA1(e486a232e6b62ab21b9c3be521606dea2a451889) )
ROM_END

ROM_START( curvebal )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "cb-rom-3.chp", 0x8000, 0x2000, CRC(72ad4d45) SHA1(9537eb360ed1d33d399cc2d8761c36b7d25fdae0) )
	ROM_LOAD( "cb-rom-2.chp", 0xa000, 0x2000, CRC(d46c3db5) SHA1(d4f464a6ebc090d100e890303557f0d05214033b) )
	ROM_LOAD( "cb-rom-1.chp", 0xc000, 0x2000, CRC(eb1e08bd) SHA1(f558664df12e4e15ef2779a0bdf99538f8c43ca3) )
	ROM_LOAD( "cb-rom-0.chp", 0xe000, 0x2000, CRC(401fc7e3) SHA1(86ca53cb6f1d88d5a95baa9524c6b42a1f7fc9c2) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for sound cpu */
	ROM_LOAD( "yrom.sbd",     0xe000, 0x1000, CRC(4c313d9b) SHA1(c61a8c827f4b199fdfb6ffc0bc6cca396c73625e) )
	ROM_LOAD( "drom.sbd",     0xf000, 0x1000, CRC(cecece88) SHA1(4c6639f6f89f80b04b6ffbb5004ea2121e71f504) )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "cb-bg-0.chp",  0x0000, 0x1000, CRC(d666a179) SHA1(3b9aca5272ae3f3d99ba55f5dc2db4eac82896bc) )
	ROM_LOAD( "cb-bg-1.chp",  0x1000, 0x1000, CRC(5e34ff4e) SHA1(f88234c0f46533540815e05479938810ea4fb4f8) )

	ROM_REGION( 0x8000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "cb-fg-3.chp",  0x0000, 0x2000, CRC(9c9452fe) SHA1(4cf5ee2efa5734849781aa57c2b92ed542d5cb4e) )
	ROM_LOAD( "cb-fg-2.chp",  0x2000, 0x2000, CRC(065131af) SHA1(fe78ee907f1d0e9f6fc3c96e4fa4aff390115147) )
	ROM_LOAD( "cb-fg-1.chp",  0x4000, 0x2000, CRC(1b7b7f94) SHA1(ffb0f163531c2f9d184d8733466d23ab9d48ea2e) )
	ROM_LOAD( "cb-fg-0.chp",  0x6000, 0x2000, CRC(e3a8230e) SHA1(c256b5ca25dc15c11d574d0ad823b34093933802) )
ROM_END


static DRIVER_INIT( gottlieb )
{
	gottlieb_sound_init();
}


GAME( 1982, reactor,  0,     reactor,  reactor,  0,        ROT0,   "Gottlieb", "Reactor" )
GAME( 1982, qbert,    0,     qbert,    qbert,    0,        ROT270, "Gottlieb", "Q*bert (US)" )
GAME( 1982, qbertjp,  qbert, qbert,    qbert,    0,        ROT270, "Gottlieb (Konami license)", "Q*bert (Japan)" )
GAME( 1982, myqbert,  qbert, qbert,    qbert,    0,        ROT270, "Gottlieb", "Mello Yello Q*bert" )
GAMEX(1982, insector, 0,     gottlieb, insector, 0,        ROT0,   "Gottlieb", "Insector (prototype)", GAME_NO_SOUND )
GAME( 1983, mplanets, 0,     gottlieb, mplanets, 0,        ROT270, "Gottlieb", "Mad Planets" )
GAME( 1983, krull,    0,     krull,    krull,    0,        ROT270, "Gottlieb", "Krull" )
GAME( 1983, sqbert,   0,     qbert,    qbert,    0,        ROT270, "Mylstar", "Faster, Harder, More Challenging Q*bert (prototype)" )
GAMEX(1983, mach3,    0,     mach3,    mach3,    gottlieb, ROT0,   "Mylstar", "M.A.C.H. 3", GAME_NOT_WORKING )
GAME( 1983, qbertqub, 0,     qbert,    qbertqub, 0,        ROT270, "Mylstar", "Q*bert's Qubes" )
GAME( 1983, screwloo, 0,     gottlieb2,screwloo, gottlieb, ROT0,   "Mylstar", "Screw Loose (prototype)" )
GAME( 1984, curvebal, 0,     gottlieb, curvebal, 0,        ROT270, "Mylstar", "Curve Ball" )
GAMEX(1984, usvsthem, 0,     usvsthem, usvsthem, gottlieb, ROT0,   "Mylstar", "Us vs. Them", GAME_NOT_WORKING )
GAMEX(1984, 3stooges, 0,     stooges,  3stooges, gottlieb, ROT0,   "Mylstar", "The Three Stooges In Brides Is Brides", GAME_IMPERFECT_SOUND )
#pragma code_seg()
#pragma data_seg()
#pragma bss_seg()
#pragma const_seg()
