#pragma code_seg("C174")
#pragma data_seg("D174")
#pragma bss_seg("B174")
#pragma const_seg("K174")
#pragma comment(linker, "/merge:D174=174")
#pragma comment(linker, "/merge:C174=174")
#pragma comment(linker, "/merge:B174=174")
#pragma comment(linker, "/merge:K174=174")
/***************************************************************************

						-= Dynax / Nakanihon Games =-

					driver by	Luca Elia (l.elia@tin.it)


	hanamai:
	There are four scrolling layers. Each layer consists
	of 2 frame buffers. The 2 images are interleaved to form the final picture
	sent to the screen.

	drgpunch:
	There are three scrolling layers. Each layer consists of 2 frame buffers.
	The 2 images are interleaved to form the final picture sent to the screen.

	mjdialq2:
	Two scrolling layers.

	The gfx roms do not contain tiles: the CPU controls a video blitter
	that can read data from them (instructions to draw pixel by pixel,
	in a compressed form) and write to the 6 frame buffers.

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "includes/dynax.h"

// Log Blitter
#define VERBOSE 0


/* 0 B01234 G01234 R01234 */
PALETTE_INIT( sprtmtch )
{
	int i;

	for (i = 0;i < Machine->drv->total_colors;i++)
	{
		int x =	(color_prom[i]<<8) + color_prom[0x200+i];
		/* The bits are in reverse order! */
		int r = BITSWAP8((x >>  0) & 0x1f, 7,6,5, 0,1,2,3,4 );
		int g = BITSWAP8((x >>  5) & 0x1f, 7,6,5, 0,1,2,3,4 );
		int b = BITSWAP8((x >> 10) & 0x1f, 7,6,5, 0,1,2,3,4 );
		r =  (r << 3) | (r >> 2);
		g =  (g << 3) | (g >> 2);
		b =  (b << 3) | (b >> 2);
		palette_set_color(i,r,g,b);
	}
}

/***************************************************************************


								Video Blitter(s)


***************************************************************************/


static int dynax_blit_scroll_x;
static int dynax_blit_scroll_y;

static int blit_dest;
static int blit_src;
static int dynax_blit_dest;
static int hanamai_layer_dest;

static int dynax_blit_pen;
static int dynax_blit_backpen;
static int dynax_blit_palettes;
static int dynax_blit_palbank;

static int dynax_layer_enable;

static int extra_scroll_x,extra_scroll_y;
static int flipscreen;

#define LAYOUT_HANAMAI	0	// 4 layers, interleaved
#define LAYOUT_HNORIDUR	1	// same as hanamai but some bits are inverted and layer order is reversed
#define LAYOUT_DRGPUNCH	2	// 3 couples of layers, interleaved
#define LAYOUT_MJDIALQ2	3	// 2 layers

static int layer_layout;

static int trigger_irq;	// some games trigger IRQ at blitter end, some don't

// 4 layers, 2 images per layer (interleaved on screen)
static UINT8 *dynax_pixmap[4][2];



WRITE_HANDLER( dynax_extra_scrollx_w )
{
	extra_scroll_x = data;
}

WRITE_HANDLER( dynax_extra_scrolly_w )
{
	extra_scroll_y = data;
}


/* Destination Pen */
WRITE_HANDLER( dynax_blit_pen_w )
{
	dynax_blit_pen = data;
#if VERBOSE
	logerror("P=%02X ",data);
#endif
}

/* Background Color */
WRITE_HANDLER( dynax_blit_backpen_w )
{
	dynax_blit_backpen = data;
#if VERBOSE
	logerror("B=%02X ",data);
#endif
}

/* Layers 0&1 Palettes (Low Bits) */
WRITE_HANDLER( dynax_blit_palette01_w )
{
	dynax_blit_palettes = (dynax_blit_palettes & ~0xff) | data;
#if VERBOSE
	logerror("P1=%02X ",data);
#endif
}

/* Layer 2 Palette (Low Bits) */
WRITE_HANDLER( dynax_blit_palette2_w )
{
	dynax_blit_palettes = (dynax_blit_palettes & ~0xff00) | (data<<8);
#if VERBOSE
	logerror("P2=%02X ",data);
#endif
}

/* Layers Palettes (High Bits) */
WRITE_HANDLER( dynax_blit_palbank_w )
{
	dynax_blit_palbank = data;
#if VERBOSE
	logerror("PB=%02X ",data);
#endif
}

/* Destination Layers */
WRITE_HANDLER( dynax_blit_dest_w )
{
	dynax_blit_dest = data;
	if (layer_layout == LAYOUT_HNORIDUR)
		dynax_blit_dest = BITSWAP8(dynax_blit_dest ^ 0x0f, 7,6,5,4, 0,1,2,3);

#if VERBOSE
	logerror("D=%02X ",data);
#endif
}

WRITE_HANDLER( hanamai_layer_dest_w )
{
	hanamai_layer_dest = data & 1;
}

WRITE_HANDLER( mjdialq2_blit_dest_w )
{
	int mask = (2 >> offset);	/* 1 or 2 */

	dynax_blit_dest &= ~mask;
	if (~data & 1) dynax_blit_dest |= mask;
}

/* Layers Enable */
WRITE_HANDLER( dynax_layer_enable_w )
{
	dynax_layer_enable = data;

#if VERBOSE
	logerror("E=%02X ",data);
#endif
}

WRITE_HANDLER( mjdialq2_layer_enable_w )
{
	int mask = (2 >> offset);	/* 1 or 2 */

	dynax_layer_enable &= ~mask;
	if (~data & 1) dynax_layer_enable |= mask;
}


WRITE_HANDLER( dynax_flipscreen_w )
{
	flipscreen = data & 1;
	if (data & ~1)
		logerror("CPU#0 PC %06X: Warning, flip screen <- %02X\n", activecpu_get_pc(), data);
}


/***************************************************************************

							Blitter Data Format

	The blitter reads its commands from the gfx ROMs. They are
	instructions to draw an image pixel by pixel (in a compressed
	form) in a frame buffer.

	Fetch 1 Byte from the ROM:

	7654 ----	Pen to draw with
	---- 3210	Command

	Other bytes may follow, depending on the command

	Commands:

	0		Stop.
	1-b		Draw 1-b pixels along X.
	c		Followed by 1 byte (N): draw N pixels along X.
	d		Followed by 2 bytes (X,N): skip X pixels, draw N pixels along X.
	e		? unused
	f		Increment Y

***************************************************************************/


/* Plot a pixel (in the pixmaps specified by dynax_blit_dest) */
INLINE void sprtmtch_plot_pixel(int x, int y, int pen, int flags)
{
	x &= 0xff;	// confirmed by some mjdialq2 gfx and especially by mjfriday, which
				// uses the front layer to mask out the right side of the screen as
				// it draws stuff on the left, when it shows the girls scrolling
				// horizontally after you win.
	y &= 0xff;	// seems confirmed by mjdialq2 last picture of gal 6, but it breaks
				// mjdialq2 title screen so there's something we are missing.

	/* "Flip Screen" just means complement the coordinates to 256 */
	if (flipscreen)	{	x ^= 0xff;	y ^= 0xff;	}

	/* Rotate: rotation = SWAPXY + FLIPY */
	if (flags & 0x08)	{ int t = x; x = y; y = t;	}

	/* Ignore the pens specified in ROM, draw everything with the
	   supplied one instead */
	if (flags & 0x02)	{ pen = (dynax_blit_pen >> 4) & 0xf;	}

	if (	(x >= 0) && (x <= 0xff) &&
			(y >= 0) && (y <= 0xff)	)
	{
		switch (layer_layout)
		{
			case LAYOUT_HANAMAI:
			case LAYOUT_HNORIDUR:
				if (dynax_blit_dest & 0x01)	dynax_pixmap[0][hanamai_layer_dest][256*y+x] = pen;
				if (dynax_blit_dest & 0x02)	dynax_pixmap[1][hanamai_layer_dest][256*y+x] = pen;
				if (dynax_blit_dest & 0x04)	dynax_pixmap[2][hanamai_layer_dest][256*y+x] = pen;
				if (dynax_blit_dest & 0x08)	dynax_pixmap[3][hanamai_layer_dest][256*y+x] = pen;
				break;

			case LAYOUT_DRGPUNCH:
				if (dynax_blit_dest & 0x01)	dynax_pixmap[0][0][256*y+x] = pen;
				if (dynax_blit_dest & 0x02)	dynax_pixmap[0][1][256*y+x] = pen;
				if (dynax_blit_dest & 0x04)	dynax_pixmap[1][0][256*y+x] = pen;
				if (dynax_blit_dest & 0x08)	dynax_pixmap[1][1][256*y+x] = pen;
				if (dynax_blit_dest & 0x10)	dynax_pixmap[2][0][256*y+x] = pen;
				if (dynax_blit_dest & 0x20)	dynax_pixmap[2][1][256*y+x] = pen;
				break;

			case LAYOUT_MJDIALQ2:
				if (dynax_blit_dest & 0x01)	dynax_pixmap[0][0][256*y+x] = pen;
				if (dynax_blit_dest & 0x02)	dynax_pixmap[1][0][256*y+x] = pen;
				break;
		}
	}
}


int sprtmtch_drawgfx( int i, int dest, int flags )
{
	data8_t cmd, pen;
	int x = dest & 0xff;
	int y = dest >> 8;

	data8_t *SRC		=	memory_region( REGION_GFX1 );
	size_t   size_src	=	memory_region_length( REGION_GFX1 );

	int sx;

if (flags & 0xf4) usrintf_showmessage("flags %02x",flags);
	if ( flags & 1 )
	{
		int start,len;
		pen = (dynax_blit_pen >> 4) & 0xf;

		/* Clear the buffer(s) starting from the given scanline and exit */

		if (flipscreen)
			start = 0;
		else
			start = dest;

		len = 0x10000 - dest;

		switch (layer_layout)
		{
			case LAYOUT_HANAMAI:
			case LAYOUT_HNORIDUR:
				if (dynax_blit_dest & 0x01)	memset(&dynax_pixmap[0][0][start],pen,len);
				if (dynax_blit_dest & 0x01)	memset(&dynax_pixmap[0][1][start],pen,len);
				if (dynax_blit_dest & 0x02)	memset(&dynax_pixmap[1][0][start],pen,len);
				if (dynax_blit_dest & 0x02)	memset(&dynax_pixmap[1][1][start],pen,len);
				if (dynax_blit_dest & 0x04)	memset(&dynax_pixmap[2][0][start],pen,len);
				if (dynax_blit_dest & 0x04)	memset(&dynax_pixmap[2][1][start],pen,len);
				if (dynax_blit_dest & 0x08)	memset(&dynax_pixmap[3][0][start],pen,len);
				if (dynax_blit_dest & 0x08)	memset(&dynax_pixmap[3][1][start],pen,len);
				break;

			case LAYOUT_DRGPUNCH:
				if (dynax_blit_dest & 0x01)	memset(&dynax_pixmap[0][0][start],pen,len);
				if (dynax_blit_dest & 0x02)	memset(&dynax_pixmap[0][1][start],pen,len);
				if (dynax_blit_dest & 0x04)	memset(&dynax_pixmap[1][0][start],pen,len);
				if (dynax_blit_dest & 0x08)	memset(&dynax_pixmap[1][1][start],pen,len);
				if (dynax_blit_dest & 0x10)	memset(&dynax_pixmap[2][0][start],pen,len);
				if (dynax_blit_dest & 0x20)	memset(&dynax_pixmap[2][1][start],pen,len);
				break;

			case LAYOUT_MJDIALQ2:
				if (dynax_blit_dest & 0x01)	memset(&dynax_pixmap[0][0][start],pen,len);
				if (dynax_blit_dest & 0x02)	memset(&dynax_pixmap[1][0][start],pen,len);
				break;
		}

		return i;
	}

	sx = x;

	for ( ;; )
	{
		if (i >= size_src)
		{
usrintf_showmessage("GFXROM OVER %08x",i);
			return i;
		}
		cmd = SRC[i++];
		pen = (cmd & 0xf0)>>4;
		cmd = (cmd & 0x0f)>>0;

		switch (cmd)
		{
		case 0xf:	// Increment Y
			/* Rotate: rotation = SWAPXY + FLIPY */
			if (flags & 0x08)
				y--;
			else
				y++;
			x = sx;
			break;

		case 0xe:	// unused ? was "change dest mask" in the "rev1" blitter
			usrintf_showmessage("Blitter unknown command %06X: %02X\n", i-1, cmd);

		case 0xd:	// Skip X pixels
			if (i >= size_src)
			{
usrintf_showmessage("GFXROM OVER %08x",i);
				return i;
			}
			x = sx + SRC[i++];
			/* fall through into next case */

		case 0xc:	// Draw N pixels
			if (i >= size_src)
			{
usrintf_showmessage("GFXROM OVER %08x",i);
				return i;
			}
			cmd = SRC[i++];
			/* fall through into next case */

		case 0xb:
		case 0xa:
		case 0x9:
		case 0x8:
		case 0x7:
		case 0x6:
		case 0x5:
		case 0x4:
		case 0x3:
		case 0x2:
		case 0x1:	// Draw N pixels
			while (cmd--)
				sprtmtch_plot_pixel(x++, y, pen, flags);
			break;

		case 0x0:	// Stop
			return i;
		}
	}
}



static void dynax_blitter_start(int flags)
{
	int i =
	sprtmtch_drawgfx(
		blit_src & 0x3fffff,
		blit_dest,
		flags
	);

#if VERBOSE
	logerror("BLIT=%02X\n",data);
#endif

	blit_src	=	(blit_src & ~0x3fffff) |
							(i                  &  0x3fffff) ;

	/* Generate an IRQ */
	if (trigger_irq)
	{
		dynax_blitter_irq = 1;
		sprtmtch_update_irq();
	}
}




WRITE_HANDLER( dynax_blit_scroll_w )
{
//logerror("%04x blit_scroll_w data = %02x addr = %06x\n",activecpu_get_pc(),data,blit_src);
	// 0x800000 also used!
	if (blit_src & 0x800000)
	{
//usrintf_showmessage("dynax_blit_scroll %02x",data);
	}
	else
	{
		if (blit_src & 0x400000)
		{
			dynax_blit_scroll_y = data;
#if VERBOSE
			logerror("SY=%02X ",data);
#endif
		}
		else
		{
			dynax_blit_scroll_x = data;
#if VERBOSE
			logerror("SX=%02X ",data);
#endif
		}
	}
}


WRITE_HANDLER( dynax_blitter_rev2_w )
{
	switch (offset)
	{
		case 0: dynax_blitter_start(data); break;
		case 1:	blit_dest = (blit_dest & 0xff00) | (data << 0); break;
		case 2: blit_dest = (blit_dest & 0x00ff) | (data << 8); break;
		case 3:	blit_src = (blit_src & 0xffff00) | (data << 0); break;
		case 4: blit_src = (blit_src & 0xff00ff) | (data << 8); break;
		case 5: blit_src = (blit_src & 0x00ffff) | (data <<16); break;
		case 6: dynax_blit_scroll_w(0,data); break;
	}
}


/***************************************************************************


								Video Init


***************************************************************************/

VIDEO_START( hanamai )
{
	if (!(dynax_pixmap[0][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[0][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[3][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[3][1] = auto_malloc(256*256)))	return 1;

	extra_scroll_x = 0;
	extra_scroll_y = 0;

	layer_layout = LAYOUT_HANAMAI;
	trigger_irq = 1;

	return 0;
}

VIDEO_START( hnoridur )
{
	if (!(dynax_pixmap[0][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[0][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[3][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[3][1] = auto_malloc(256*256)))	return 1;

	extra_scroll_x = 0;
	extra_scroll_y = 0;

	layer_layout = LAYOUT_HNORIDUR;
	trigger_irq = 1;

	return 0;
}

VIDEO_START( sprtmtch )
{
	if (!(dynax_pixmap[0][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[0][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][1] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[2][1] = auto_malloc(256*256)))	return 1;

	layer_layout = LAYOUT_DRGPUNCH;
	trigger_irq = 1;

	return 0;
}

VIDEO_START( mjdialq2 )
{
	if (!(dynax_pixmap[0][0] = auto_malloc(256*256)))	return 1;
	if (!(dynax_pixmap[1][0] = auto_malloc(256*256)))	return 1;

	layer_layout = LAYOUT_MJDIALQ2;
	trigger_irq = 0;

	return 0;
}


/***************************************************************************


								Screen Drawing


***************************************************************************/

void hanamai_copylayer(struct mame_bitmap *bitmap,const struct rectangle *cliprect,int i)
{
	int color;
	int scrollx,scrolly;

	switch ( i )
	{
		case 0:	color = (dynax_blit_palettes >>  0) & 0xf;	break;
		case 1:	color = (dynax_blit_palettes >>  4) & 0xf;	break;
		case 2:	color = (dynax_blit_palettes >>  8) & 0xf;	break;
		case 3:	color = (dynax_blit_palettes >> 12) & 0xf;	break;
		default:	return;
	}

	color += (dynax_blit_palbank & 1) * 16;

	scrollx = dynax_blit_scroll_x;
	scrolly = dynax_blit_scroll_y;

	if ((layer_layout == LAYOUT_HANAMAI && i == 1)
			|| (layer_layout == LAYOUT_HNORIDUR && i == 1))
	{
		scrollx = extra_scroll_x;
		scrolly = extra_scroll_y - 1;
	}

	{
		int dy,length,pen,offs;
		UINT8 *src1 = dynax_pixmap[i][1];
		UINT8 *src2 = dynax_pixmap[i][0];

		int palbase = 16*color;
		offs = 0;

		for (dy = 0; dy < 256; dy++)
		{
			UINT16 *dst;
			UINT16 *dstbase = (UINT16 *)bitmap->base + ((dy - scrolly) & 0xff) * bitmap->rowpixels;

			length = scrollx;
			dst = dstbase + 2*(256 - length);
			while (length--)
			{
				pen = *(src1++);
				if (pen) *dst     = palbase + pen;
				pen = *(src2++);
				if (pen) *(dst+1) = palbase + pen;
				dst += 2;
			}

			length = 256 - scrollx;
			dst = dstbase;
			while (length--)
			{
				pen = *(src1++);
				if (pen) *dst     = palbase + pen;
				pen = *(src2++);
				if (pen) *(dst+1) = palbase + pen;
				dst += 2;
			}
		}
	}
}



void mjdialq2_copylayer(struct mame_bitmap *bitmap,const struct rectangle *cliprect,int i)
{
	int color;
	int scrollx,scrolly;

	switch ( i )
	{
		case 0:	color = (dynax_blit_palettes >>  4) & 0xf;	break;
		case 1:	color = (dynax_blit_palettes >>  0) & 0xf;	break;
		default:	return;
	}

	color += (dynax_blit_palbank & 1) * 16;

	scrollx = dynax_blit_scroll_x;
	scrolly = dynax_blit_scroll_y;

	{
		int dy,length,pen,offs;
		UINT8 *src = dynax_pixmap[i][0];

		int palbase = 16*color;
		offs = 0;

		for (dy = 0; dy < 256; dy++)
		{
			UINT16 *dst;
			UINT16 *dstbase = (UINT16 *)bitmap->base + ((dy - scrolly) & 0xff) * bitmap->rowpixels;

			length = scrollx;
			dst = dstbase + 256 - length;
			while (length--)
			{
				pen = *(src++);
				if (pen) *dst = palbase + pen;
				dst++;
			}

			length = 256 - scrollx;
			dst = dstbase;
			while (length--)
			{
				pen = *(src++);
				if (pen) *dst = palbase + pen;
				dst++;
			}
		}
	}
}



#if 0
/*	A primitive gfx viewer:

	T          -  Toggle viewer
	I,O        -  Change palette (-,+)
	J,K & N,M  -  Change "tile"  (-,+, slow & fast)
	R          -  "tile" = 0		*/

static int toggle;
if (keyboard_pressed_memory(KEYCODE_T))	toggle = 1-toggle;
if (toggle)	{
	data8_t *RAM = memory_region( REGION_GFX1 );
	static int i = 0, c = 0;

	if (keyboard_pressed_memory(KEYCODE_I))	c = (c-1) & 0x1f;
	if (keyboard_pressed_memory(KEYCODE_O))	c = (c+1) & 0x1f;
	if (keyboard_pressed_memory(KEYCODE_R))	i = 0;
	if (keyboard_pressed(KEYCODE_M) | keyboard_pressed_memory(KEYCODE_K))	{
		while( RAM[i] )	i++;		i++;	}
	if (keyboard_pressed(KEYCODE_N) | keyboard_pressed_memory(KEYCODE_J))	{
		i-=2;	while( RAM[i] )	i--;		i++;	}

	dynax_blit_palettes = (c & 0xf) * 0x111;
	dynax_blit_palbank  = (c >>  4) & 1;
	dynax_blit_dest = 1;

	fillbitmap(bitmap,Machine->pens[0],&Machine->visible_area);
	memset(dynax_pixmap[0][0],0,sizeof(UINT8)*0x100*0x100);
	memset(dynax_pixmap[0][1],0,sizeof(UINT8)*0x100*0x100);
	hanamai_layer_dest = 0;
	sprtmtch_drawgfx(i, Machine->visible_area.min_x, Machine->visible_area.min_y, 0);
	hanamai_layer_dest = 1;
	sprtmtch_drawgfx(i, Machine->visible_area.min_x, Machine->visible_area.min_y, 0);
	hanamai_copylayer(bitmap, cliprect, 0);
	usrintf_showmessage("%06X C%02X",i,c);
}
else
#endif


static int hanamai_priority;

WRITE_HANDLER( hanamai_priority_w )
{
	hanamai_priority = data;
}


VIDEO_UPDATE( hanamai )
{
	int layers_ctrl = ~dynax_layer_enable;
	int lay[4];

	fillbitmap(
		bitmap,
		Machine->pens[(dynax_blit_backpen & 0xff) + (dynax_blit_palbank & 1) * 256],
		cliprect);

	/* bit 4 = display enable? */
	if (!(hanamai_priority & 0x10)) return;

	switch (hanamai_priority)
	{
		default:
			usrintf_showmessage("unknown priority %02x",hanamai_priority);
		case 0x10:
			lay[0] = 0; lay[1] = 1; lay[2] = 2; lay[3] = 3; break;
		case 0x11:
			lay[0] = 0; lay[1] = 3; lay[2] = 2; lay[3] = 1; break;
		case 0x12:
			lay[0] = 0; lay[1] = 1; lay[2] = 3; lay[3] = 2; break;
		case 0x13:
			lay[0] = 0; lay[1] = 3; lay[2] = 1; lay[3] = 2; break;
		case 0x14:
			lay[0] = 0; lay[1] = 2; lay[2] = 1; lay[3] = 3; break;
		case 0x15:
			lay[0] = 0; lay[1] = 2; lay[2] = 3; lay[3] = 1; break;
	}

	if (layers_ctrl & (1 << lay[0]))	hanamai_copylayer( bitmap, cliprect, lay[0] );
	if (layers_ctrl & (1 << lay[1]))	hanamai_copylayer( bitmap, cliprect, lay[1] );
	if (layers_ctrl & (1 << lay[2]))	hanamai_copylayer( bitmap, cliprect, lay[2] );
	if (layers_ctrl & (1 << lay[3]))	hanamai_copylayer( bitmap, cliprect, lay[3] );
}


VIDEO_UPDATE( hnoridur )
{
	int layers_ctrl = ~BITSWAP8(hanamai_priority, 7,6,5,4, 0,1,2,3);
	int lay[4];

	fillbitmap(
		bitmap,
		Machine->pens[(dynax_blit_backpen & 0xff) + (dynax_blit_palbank & 1) * 256],
		cliprect);

	switch (hanamai_priority & 0xf0)
	{
		default:
			usrintf_showmessage("unknown priority %02x",hanamai_priority);
		case 0x00:
			lay[0] = 0; lay[1] = 2; lay[2] = 3; lay[3] = 1; break;
		case 0x10:
			lay[0] = 2; lay[1] = 1; lay[2] = 0; lay[3] = 3; break;
		case 0x20:
			lay[0] = 3; lay[1] = 1; lay[2] = 0; lay[3] = 2; break;
		case 0x30:
			lay[0] = 2; lay[1] = 0; lay[2] = 3; lay[3] = 1; break;
		case 0x40:
			lay[0] = 3; lay[1] = 0; lay[2] = 2; lay[3] = 1; break;
		case 0x50:
			lay[0] = 1; lay[1] = 3; lay[2] = 0; lay[3] = 2; break;
		case 0x60:
			lay[0] = 2; lay[1] = 3; lay[2] = 1; lay[3] = 0; break;
		case 0x70:
			lay[0] = 1; lay[1] = 0; lay[2] = 2; lay[3] = 3; break;
	}

	if (layers_ctrl & (1 << lay[0]))	hanamai_copylayer( bitmap, cliprect, lay[0] );
	if (layers_ctrl & (1 << lay[1]))	hanamai_copylayer( bitmap, cliprect, lay[1] );
	if (layers_ctrl & (1 << lay[2]))	hanamai_copylayer( bitmap, cliprect, lay[2] );
	if (layers_ctrl & (1 << lay[3]))	hanamai_copylayer( bitmap, cliprect, lay[3] );
}


VIDEO_UPDATE( sprtmtch )
{
	int layers_ctrl = ~dynax_layer_enable;

	fillbitmap(
		bitmap,
		Machine->pens[(dynax_blit_backpen & 0xff) + (dynax_blit_palbank & 1) * 256],
		cliprect);

	if (layers_ctrl & 1)	hanamai_copylayer( bitmap, cliprect, 0 );
	if (layers_ctrl & 2)	hanamai_copylayer( bitmap, cliprect, 1 );
	if (layers_ctrl & 4)	hanamai_copylayer( bitmap, cliprect, 2 );
}


VIDEO_UPDATE( mjdialq2 )
{
	int layers_ctrl = ~dynax_layer_enable;

	fillbitmap(
		bitmap,
		Machine->pens[(dynax_blit_backpen & 0xff) + (dynax_blit_palbank & 1) * 256],
		cliprect);

	if (layers_ctrl & 1)	mjdialq2_copylayer( bitmap, cliprect, 0 );
	if (layers_ctrl & 2)	mjdialq2_copylayer( bitmap, cliprect, 1 );
}
#pragma code_seg()
#pragma data_seg()
#pragma bss_seg()
#pragma const_seg()
