#pragma code_seg("CC48")
#pragma data_seg("CD48")
#pragma bss_seg("CB48")
#pragma const_seg("CK48")
#pragma comment(linker, "/merge:CD48=CPU48")
#pragma comment(linker, "/merge:CC48=CPU48")
#pragma comment(linker, "/merge:CB48=CPU48")
#pragma comment(linker, "/merge:CK48=CPU48")
/* ======================================================================== */
/* =============================== COPYRIGHT ============================== */
/* ======================================================================== */
/*

G65C816 CPU Emulator V0.93

Copyright (c) 2000 Karl Stenerud
All rights reserved.

Permission is granted to use this source code for non-commercial purposes.
To use this code for commercial purposes, you must get permission from the
author (Karl Stenerud) at karl@higashiyama-unet.ocn.ne.jp.


*/
/* ======================================================================== */
/* ================================= NOTES ================================ */
/* ======================================================================== */
/*

Changes:
	0.93 (2003-07-05):
			Angelo Salese <lordkale@libero.it>
			- Fixed the BCD conversion when using the Decimal Flag in ADC and SBC.
			- Removed the two conversion tables for ADC and SBC as they aren't
			  needed anymore.

	0.92 (2000-05-28):
			Lee Hammerton <lee-hammerton@hammerhead.ltd.uk>
			- Fixed debugger bug that caused D to be misrepresented.
			- Fixed MVN and MVP (they were reversed)

	0.91 (2000-05-22):
			Lee Hammerton <lee-hammerton@hammerhead.ltd.uk>
			- Fixed reset vector fetch to be little endian
			- Fixed disassembler call bug
			- Fixed C flag in SBC (should be inverted before operation)
			- Fixed JSR to stack PC-1 and RTS to pull PC and add 1

			Karl Stenerud <karl@higashiyama-unet.ocn.ne.jp>
			- Added correct timing for absolute indexed operations
			- SBC: fixed corruption of interim values

	0.90 (2000-05-17):
			Karl Stenerud <karl@higashiyama-unet.ocn.ne.jp>
			- first public release


Note on timings:
	- For instructions that write to memory (ASL, ASR, LSL, ROL, ROR, DEC,
	  INC, STA, STZ), the absolute indexed addressing mode takes 1 extra
	  cycle to complete.
	- The spec says fc (JMP axi) is 6 cyles, but elsewhere says 8 cycles
	  (which is what it should be)


TODO general:
	- WAI will not stop if RDY is held high.

	- RDY internally held low when WAI executed and returned to hi when RES,
	  ABORT, NMI, or IRQ asserted.

	- ABORT will terminate WAI instruction but wil not restart the processor

	- If interrupt occurs after ABORT of WAI, processor returns to WAI
	  instruction.

	- Add one cycle when indexing across page boundary and E=1 except for STA
	  and STZ instructions.

	- Add 1 cycle if branch is taken. In Emulation (E= 1 ) mode only --add 1
	  cycle if the branch is taken and crosses a page boundary.

	- Add 1 cycle in Emulation mode (E=1) for (dir),y; abs,x; and abs,y
	  addressing modes.

*/
/* ======================================================================== */
/* ================================= DATA ================================= */
/* ======================================================================== */

#include "g65816cm.h"

/* Our CPU structure */
g65816i_cpu_struct g65816i_cpu = {0};

int g65816_ICount = 0;

/* Temporary Variables */
uint g65816i_source;
uint g65816i_destination;

/* Layout of the registers in the MAME debugger */
static unsigned char g65816i_register_layout[] =
{
	G65816_PB, G65816_PC, G65816_S, G65816_DB, G65816_D, G65816_P, G65816_E, -1,
	G65816_A, G65816_X, G65816_Y, G65816_NMI_STATE, G65816_IRQ_STATE, 0
};

/* Layout of the MAME debugger windows x,y,w,h */
static unsigned char g65816i_window_layout[] = {
	25, 0,55, 2, /* register window (top, right rows) */
	 0, 0,24,22, /* disassembler window (left colums) */
	25, 3,55, 9, /* memory #1 window (right, upper middle) */
	25,13,55, 9, /* memory #2 window (right, lower middle) */
	 0,23,80, 1, /* command line window (bottom rows) */
};

extern void (*g65816i_opcodes_M0X0[])(void);
extern uint g65816i_get_reg_M0X0(int regnum);
extern void g65816i_set_reg_M0X0(int regnum, uint val);
extern void g65816i_set_line_M0X0(int line, int state);
extern int  g65816i_execute_M0X0(int cycles);

extern void (*g65816i_opcodes_M0X1[])(void);
extern uint g65816i_get_reg_M0X1(int regnum);
extern void g65816i_set_reg_M0X1(int regnum, uint val);
extern void g65816i_set_line_M0X1(int line, int state);
extern int  g65816i_execute_M0X1(int cycles);

extern void (*g65816i_opcodes_M1X0[])(void);
extern uint g65816i_get_reg_M1X0(int regnum);
extern void g65816i_set_reg_M1X0(int regnum, uint val);
extern void g65816i_set_line_M1X0(int line, int state);
extern int  g65816i_execute_M1X0(int cycles);

extern void (*g65816i_opcodes_M1X1[])(void);
extern uint g65816i_get_reg_M1X1(int regnum);
extern void g65816i_set_reg_M1X1(int regnum, uint val);
extern void g65816i_set_line_M1X1(int line, int state);
extern int  g65816i_execute_M1X1(int cycles);

extern void (*g65816i_opcodes_E[])(void);
extern uint g65816i_get_reg_E(int regnum);
extern void g65816i_set_reg_E(int regnum, uint val);
extern void g65816i_set_line_E(int line, int state);
extern int  g65816i_execute_E(int cycles);

void (**g65816i_opcodes[5])(void) =
{
	g65816i_opcodes_M0X0,
	g65816i_opcodes_M0X1,
	g65816i_opcodes_M1X0,
	g65816i_opcodes_M1X1,
	g65816i_opcodes_E
};

uint (*g65816i_get_reg[5])(int regnum) =
{
	g65816i_get_reg_M0X0,
	g65816i_get_reg_M0X1,
	g65816i_get_reg_M1X0,
	g65816i_get_reg_M1X1,
	g65816i_get_reg_E
};

void (*g65816i_set_reg[5])(int regnum, uint val) =
{
	g65816i_set_reg_M0X0,
	g65816i_set_reg_M0X1,
	g65816i_set_reg_M1X0,
	g65816i_set_reg_M1X1,
	g65816i_set_reg_E
};

void (*g65816i_set_line[5])(int line, int state) =
{
	g65816i_set_line_M0X0,
	g65816i_set_line_M0X1,
	g65816i_set_line_M1X0,
	g65816i_set_line_M1X1,
	g65816i_set_line_E
};

int (*g65816i_execute[5])(int cycles) =
{
	g65816i_execute_M0X0,
	g65816i_execute_M0X1,
	g65816i_execute_M1X0,
	g65816i_execute_M1X1,
	g65816i_execute_E
};

/* ======================================================================== */
/* ================================= API ================================== */
/* ======================================================================== */


void g65816_reset(void* param)
{
		/* Start the CPU */
		CPU_STOPPED = 0;

		/* Put into emulation mode */
		REGISTER_D = 0;
		REGISTER_PB = 0;
		REGISTER_DB = 0;
		REGISTER_S = (REGISTER_S & 0xff) | 0x100;
		REGISTER_X &= 0xff;
		REGISTER_Y &= 0xff;
		if(!FLAG_M)
		{
			REGISTER_B = REGISTER_A & 0xff00;
			REGISTER_A &= 0xff;
		}
		FLAG_E = EFLAG_SET;
		FLAG_M = MFLAG_SET;
		FLAG_X = XFLAG_SET;

		/* Clear D and set I */
		FLAG_D = DFLAG_CLEAR;
		FLAG_I = IFLAG_SET;

		/* Clear all pending interrupts (should we really do this?) */
		LINE_IRQ = 0;
		LINE_NMI = 0;
		IRQ_DELAY = 0;

		/* Set the function tables to emulation mode */
		g65816i_set_execution_mode(EXECUTION_MODE_E);

		/* 6502 expects these, but its not in the 65816 spec */
		FLAG_Z = ZFLAG_CLEAR;
		REGISTER_S = 0x1ff;

		/* Fetch the reset vector */
		REGISTER_PC = g65816_read_8(VECTOR_RESET) | (g65816_read_8(VECTOR_RESET+1)<<8);
		g65816i_jumping(REGISTER_PB | REGISTER_PC);
}

/* Exit and clean up */
void g65816_exit(void)
{
	/* nothing to do yet */
}

/* Execute some instructions */
int g65816_execute(int cycles)
{
	return FTABLE_EXECUTE(cycles);
}


/* Get the current CPU context */
unsigned g65816_get_context(void *dst_context)
{
	if(dst_context)
		*(g65816i_cpu_struct*)dst_context = g65816i_cpu;
	return sizeof(g65816i_cpu);
}

static void mame_g65816_get_context(void *dst_context)
{
	g65816_get_context(dst_context);
}

/* Set the current CPU context */
void g65816_set_context(void *src_context)
{
	if(src_context)
	{
		g65816i_cpu = *(g65816i_cpu_struct*)src_context;
		g65816i_jumping(REGISTER_PB | REGISTER_PC);
	}
}

/* Get the current Program Counter */
unsigned g65816_get_pc(void)
{
	return REGISTER_PC;
}

/* Set the Program Counter */
void g65816_set_pc(unsigned val)
{
	REGISTER_PC = MAKE_UINT_16(val);
	g65816_jumping(REGISTER_PB | REGISTER_PC);
}

/* Get the current Stack Pointer */
unsigned g65816_get_sp(void)
{
	return REGISTER_S;
}

/* Set the Stack Pointer */
void g65816_set_sp(unsigned val)
{
	REGISTER_S = FLAG_E ? MAKE_UINT_8(val) | 0x100 : MAKE_UINT_16(val);
}

/* Get a register */
unsigned g65816_get_reg(int regnum)
{
	/* Set the function tables to emulation mode if the FTABLE is NULL */
	if( FTABLE_GET_REG == NULL )
		g65816i_set_execution_mode(EXECUTION_MODE_E);

	return FTABLE_GET_REG(regnum);
}

/* Set a register */
void g65816_set_reg(int regnum, unsigned value)
{
	FTABLE_SET_REG(regnum, value);
}

/* Load a CPU state */
void g65816_state_load(void *file)
{
}

/* Save the current CPU state */
void g65816_state_save(void *file)
{
}

/* Set the non-maskable interrupt line */
void g65816_set_nmi_line(int state)
{
	FTABLE_SET_LINE(G65816_LINE_NMI, state);
}

/* Set an interrupt line */
void g65816_set_irq_line(int line, int state)
{
	FTABLE_SET_LINE(line, state);
}

/* Set the callback that is called when servicing an interrupt */
void g65816_set_irq_callback(int (*callback)(int))
{
	INT_ACK = callback;
}


/* Disassemble an instruction */
#ifdef MAME_DEBUG
#include "g65816ds.h"
#endif
unsigned g65816_dasm(char *buffer, unsigned pc)
{
#ifdef MAME_DEBUG
	return g65816_disassemble(buffer, (pc&0xffff), REGISTER_PB>>16, FLAG_M, FLAG_X);
#else
	sprintf(buffer, "$%02X", g65816_read_8_immediate(REGISTER_PB | (pc&0xffff)));
	return 1;
#endif
}


void g65816_init(void){ return; }


/**************************************************************************
 * Generic set_info
 **************************************************************************/

static void g65816_set_info(UINT32 state, union cpuinfo *info)
{
	switch (state)
	{
		/* --- the following bits of info are set as 64-bit signed integers --- */
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_IRQ:		g65816_set_irq_line(G65816_LINE_IRQ, info->i); break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_NMI:		g65816_set_irq_line(G65816_LINE_NMI, info->i); break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_ABORT:	g65816_set_irq_line(G65816_LINE_ABORT, info->i); break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_SO:		g65816_set_irq_line(G65816_LINE_SO, info->i); break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_RDY:		g65816_set_irq_line(G65816_LINE_RDY, info->i); break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_RESET:	g65816_set_irq_line(G65816_LINE_RESET, info->i); break;

		case CPUINFO_INT_PC:							g65816_set_pc(info->i);					break;
		case CPUINFO_INT_SP:							g65816_set_sp(info->i);					break;

		case CPUINFO_INT_REGISTER + G65816_PC:			g65816_set_reg(G65816_PC, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_S:			g65816_set_reg(G65816_S, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_P:			g65816_set_reg(G65816_P, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_A:			g65816_set_reg(G65816_A, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_X:			g65816_set_reg(G65816_X, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_Y:			g65816_set_reg(G65816_Y, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_PB:			g65816_set_reg(G65816_PB, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_DB:			g65816_set_reg(G65816_DB, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_D:			g65816_set_reg(G65816_D, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_E:			g65816_set_reg(G65816_E, info->i);		break;
		case CPUINFO_INT_REGISTER + G65816_NMI_STATE:	g65816_set_reg(G65816_NMI_STATE, info->i); break;
		case CPUINFO_INT_REGISTER + G65816_IRQ_STATE:	g65816_set_reg(G65816_IRQ_STATE, info->i); break;

		/* --- the following bits of info are set as pointers to data or functions --- */
		case CPUINFO_PTR_IRQ_CALLBACK:					g65816_set_irq_callback(info->irqcallback); break;
	}
}



/**************************************************************************
 * Generic get_info
 **************************************************************************/

void g65816_get_info(UINT32 state, union cpuinfo *info)
{
	switch (state)
	{
		/* --- the following bits of info are returned as 64-bit signed integers --- */
		case CPUINFO_INT_CONTEXT_SIZE:					info->i = sizeof(g65816i_cpu);			break;
		case CPUINFO_INT_INPUT_LINES:					info->i = 1;							break;
		case CPUINFO_INT_DEFAULT_IRQ_VECTOR:			info->i = 0;							break;
		case CPUINFO_INT_ENDIANNESS:					info->i = CPU_IS_BE;					break;
		case CPUINFO_INT_CLOCK_DIVIDER:					info->i = 1;							break;
		case CPUINFO_INT_MIN_INSTRUCTION_BYTES:			info->i = 1;							break;
		case CPUINFO_INT_MAX_INSTRUCTION_BYTES:			info->i = 3;							break;
		case CPUINFO_INT_MIN_CYCLES:					info->i = 1;							break;
		case CPUINFO_INT_MAX_CYCLES:					info->i = 20; /* rough guess */			break;
		
		case CPUINFO_INT_DATABUS_WIDTH + ADDRESS_SPACE_PROGRAM:	info->i = 8;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH + ADDRESS_SPACE_PROGRAM: info->i = 24;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT + ADDRESS_SPACE_PROGRAM: info->i = 0;					break;
		case CPUINFO_INT_DATABUS_WIDTH + ADDRESS_SPACE_DATA:	info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH + ADDRESS_SPACE_DATA: 	info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT + ADDRESS_SPACE_DATA: 	info->i = 0;					break;
		case CPUINFO_INT_DATABUS_WIDTH + ADDRESS_SPACE_IO:		info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH + ADDRESS_SPACE_IO: 		info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT + ADDRESS_SPACE_IO: 		info->i = 0;					break;

		case CPUINFO_INT_INPUT_STATE + G65816_LINE_IRQ:		info->i = LINE_IRQ;						break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_NMI:		info->i = LINE_NMI;						break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_ABORT:	info->i = 0;							break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_SO:		info->i = 0;							break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_RDY:		info->i = 0;							break;
		case CPUINFO_INT_INPUT_STATE + G65816_LINE_RESET:	info->i = 0;							break;

		case CPUINFO_INT_PREVIOUSPC:					/* not supported */						break;

		case CPUINFO_INT_PC:							info->i = g65816_get_pc();				break;
		case CPUINFO_INT_SP:							info->i = g65816_get_sp();				break;

		case CPUINFO_INT_REGISTER + G65816_PC:			info->i = g65816_get_reg(G65816_PC);	break;
		case CPUINFO_INT_REGISTER + G65816_S:			info->i = g65816_get_reg(G65816_S);		break;
		case CPUINFO_INT_REGISTER + G65816_P:			info->i = g65816_get_reg(G65816_P);		break;
		case CPUINFO_INT_REGISTER + G65816_A:			info->i = g65816_get_reg(G65816_A);		break;
		case CPUINFO_INT_REGISTER + G65816_X:			info->i = g65816_get_reg(G65816_X);		break;
		case CPUINFO_INT_REGISTER + G65816_Y:			info->i = g65816_get_reg(G65816_Y);		break;
		case CPUINFO_INT_REGISTER + G65816_PB:			info->i = g65816_get_reg(G65816_PB);	break;
		case CPUINFO_INT_REGISTER + G65816_DB:			info->i = g65816_get_reg(G65816_DB);	break;
		case CPUINFO_INT_REGISTER + G65816_D:			info->i = g65816_get_reg(G65816_D);		break;
		case CPUINFO_INT_REGISTER + G65816_E:			info->i = g65816_get_reg(G65816_E);		break;
		case CPUINFO_INT_REGISTER + G65816_NMI_STATE:	info->i = g65816_get_reg(G65816_NMI_STATE); break;
		case CPUINFO_INT_REGISTER + G65816_IRQ_STATE:	info->i = g65816_get_reg(G65816_IRQ_STATE); break;

		/* --- the following bits of info are returned as pointers to data or functions --- */
		case CPUINFO_PTR_SET_INFO:						info->setinfo = g65816_set_info;		break;
		case CPUINFO_PTR_GET_CONTEXT:					info->getcontext = mame_g65816_get_context;	break;
		case CPUINFO_PTR_SET_CONTEXT:					info->setcontext = g65816_set_context;	break;
		case CPUINFO_PTR_INIT:							info->init = g65816_init;				break;
		case CPUINFO_PTR_RESET:							info->reset = g65816_reset;				break;
		case CPUINFO_PTR_EXIT:							info->exit = g65816_exit;				break;
		case CPUINFO_PTR_EXECUTE:						info->execute = g65816_execute;			break;
		case CPUINFO_PTR_BURN:							info->burn = NULL;						break;
		case CPUINFO_PTR_DISASSEMBLE:					info->disassemble = g65816_dasm;		break;
		case CPUINFO_PTR_IRQ_CALLBACK:					info->irqcallback = INT_ACK;			break;
		case CPUINFO_PTR_INSTRUCTION_COUNTER:			info->icount = &g65816_ICount;			break;
		case CPUINFO_PTR_REGISTER_LAYOUT:				info->p = g65816i_register_layout;		break;
		case CPUINFO_PTR_WINDOW_LAYOUT:					info->p = g65816i_window_layout;		break;

		/* --- the following bits of info are returned as NULL-terminated strings --- */
		case CPUINFO_STR_NAME:							strcpy(info->s = cpuintrf_temp_str(), "G65C816"); break;
		case CPUINFO_STR_CORE_FAMILY:					strcpy(info->s = cpuintrf_temp_str(), "6500"); break;
		case CPUINFO_STR_CORE_VERSION:					strcpy(info->s = cpuintrf_temp_str(), "0.90"); break;
		case CPUINFO_STR_CORE_FILE:						strcpy(info->s = cpuintrf_temp_str(), __FILE__); break;
		case CPUINFO_STR_CORE_CREDITS:					strcpy(info->s = cpuintrf_temp_str(), "Copyright (c) 2000 Karl Stenerud, all rights reserved."); break;

		case CPUINFO_STR_FLAGS:
			sprintf(info->s = cpuintrf_temp_str(), "%c%c%c%c%c%c%c%c",
				g65816i_cpu.flag_n & NFLAG_SET ? 'N':'.',
				g65816i_cpu.flag_v & VFLAG_SET ? 'V':'.',
				g65816i_cpu.flag_m & MFLAG_SET ? 'M':'.',
				g65816i_cpu.flag_x & XFLAG_SET ? 'X':'.',
				g65816i_cpu.flag_d & DFLAG_SET ? 'D':'.',
				g65816i_cpu.flag_i & IFLAG_SET ? 'I':'.',
				g65816i_cpu.flag_z == 0        ? 'Z':'.',
				g65816i_cpu.flag_c & CFLAG_SET ? 'C':'.');
			break;

		case CPUINFO_STR_REGISTER + G65816_PC:			sprintf(info->s = cpuintrf_temp_str(), "PC:%04X", g65816i_cpu.pc); break;
		case CPUINFO_STR_REGISTER + G65816_PB:			sprintf(info->s = cpuintrf_temp_str(), "PB:%02X", g65816i_cpu.pb>>16); break;
		case CPUINFO_STR_REGISTER + G65816_DB:			sprintf(info->s = cpuintrf_temp_str(), "DB:%02X", g65816i_cpu.db>>16); break;
		case CPUINFO_STR_REGISTER + G65816_D:			sprintf(info->s = cpuintrf_temp_str(), "D:%04X", g65816i_cpu.d); break;
		case CPUINFO_STR_REGISTER + G65816_S:			sprintf(info->s = cpuintrf_temp_str(), "S:%04X", g65816i_cpu.s); break;
		case CPUINFO_STR_REGISTER + G65816_P:			sprintf(info->s = cpuintrf_temp_str(), "P:%02X",
																 (g65816i_cpu.flag_n&0x80)		|
																((g65816i_cpu.flag_v>>1)&0x40)	|
																g65816i_cpu.flag_m				|
																g65816i_cpu.flag_x				|
																g65816i_cpu.flag_d				|
																g65816i_cpu.flag_i				|
																((!g65816i_cpu.flag_z)<<1)		|
																((g65816i_cpu.flag_c>>8)&1)); break;
		case CPUINFO_STR_REGISTER + G65816_E:			sprintf(info->s = cpuintrf_temp_str(), "E:%d", g65816i_cpu.flag_e); break;
		case CPUINFO_STR_REGISTER + G65816_A:			sprintf(info->s = cpuintrf_temp_str(), "A:%04X", g65816i_cpu.a | g65816i_cpu.b); break;
		case CPUINFO_STR_REGISTER + G65816_X:			sprintf(info->s = cpuintrf_temp_str(), "X:%04X", g65816i_cpu.x); break;
		case CPUINFO_STR_REGISTER + G65816_Y:			sprintf(info->s = cpuintrf_temp_str(), "Y:%04X", g65816i_cpu.y); break;
		case CPUINFO_STR_REGISTER + G65816_NMI_STATE:	sprintf(info->s = cpuintrf_temp_str(), "NMI:%X", g65816i_cpu.line_nmi); break;
		case CPUINFO_STR_REGISTER + G65816_IRQ_STATE:	sprintf(info->s = cpuintrf_temp_str(), "IRQ:%X", g65816i_cpu.line_irq); break;
	}
}

/* ======================================================================== */
/* ============================== END OF FILE ============================= */
/* ======================================================================== */
#pragma code_seg()
#pragma data_seg()
#pragma bss_seg()
#pragma const_seg()
