/* IGS PGM System Encryptions */

#include "pgm.h"
#include "bitswap.h"

void pgm_decrypt_dw2()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i<nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x020890) == 0x000000)				      x ^= 0x0002;
		if ((i & 0x020000) == 0x020000 && (i & 0x001500) != 0x001400) x ^= 0x0002;
		if ((i & 0x020400) == 0x000000 && (i & 0x002010) != 0x002010) x ^= 0x0400;
		if ((i & 0x020000) == 0x020000 && (i & 0x000148) != 0x000140) x ^= 0x0400;
	
		src[i] = x;
	}
}

void pgm_decrypt_dw3()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x005460) == 0x001400) x ^= 0x0100;
		if ((i & 0x005450) == 0x001040) x ^= 0x0100;
		if ((i & 0x005e00) == 0x001c00) x ^= 0x0040;
		if ((i & 0x005580) == 0x001100) x ^= 0x0040;

		src[i] = x;
	}
}

void pgm_decrypt_killbld()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x006d00) == 0x000400) x ^= 0x0008;
		if ((i & 0x006c80) == 0x000880) x ^= 0x0008;
		if ((i & 0x007500) == 0x002400) x ^= 0x1000;
		if ((i & 0x007600) == 0x003200) x ^= 0x1000;

		src[i] = x;
	}
}


static const unsigned char kov_tab[256] = {
	0x17, 0x1c, 0xe3, 0x02, 0x62, 0x59, 0x97, 0x4a, 0x67, 0x4d, 0x1f, 0x11, 0x76, 0x64, 0xc1, 0xe1,
	0xd2, 0x41, 0x9f, 0xfd, 0xfa, 0x04, 0xfe, 0xab, 0x89, 0xeb, 0xc0, 0xf5, 0xac, 0x2b, 0x64, 0x22,
	0x90, 0x7d, 0x88, 0xc5, 0x8c, 0xe0, 0xd9, 0x70, 0x3c, 0xf4, 0x7d, 0x31, 0x1c, 0xca, 0xe2, 0xf1,
	0x31, 0x82, 0x86, 0xb1, 0x55, 0x95, 0x77, 0x01, 0x77, 0x3b, 0xab, 0xe6, 0x88, 0xef, 0x77, 0x11,
	0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d, 0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a,
	0x43, 0xcc, 0x35, 0xfe, 0xa1, 0x0d, 0xb3, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce, 0x46,
	0x9b, 0x5d, 0x68, 0x75, 0xfe, 0x1e, 0x25, 0x41, 0x24, 0xa0, 0x79, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0x3a, 0x78, 0x24, 0x64, 0xe1, 0xa3, 0x62, 0x75, 0x38, 0x65, 0x8a, 0xbf, 0xf9, 0x7c, 0x00, 0xa0,
	0x6d, 0xdb, 0x1f, 0x80, 0x37, 0x37, 0x8e, 0x97, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0xf2, 0x44, 0x91, 0x3e, 0x62, 0x44, 0xc5, 0x55, 0xe6, 0x8e, 0x5a, 0x25, 0x8a, 0x90, 0x25, 0x74,
	0xa0, 0x95, 0x33, 0xf7, 0x51, 0xce, 0xe4, 0xa0, 0x13, 0xcf, 0x33, 0x1e, 0x59, 0x5b, 0xec, 0x42,
	0xc5, 0xb8, 0xe4, 0xc5, 0x71, 0x38, 0xc5, 0x6b, 0x8d, 0x1d, 0x84, 0xf8, 0x4e, 0x21, 0x6d, 0xdc,
	0x2c, 0xf1, 0xae, 0xad, 0x19, 0xc5, 0xed, 0x8e, 0x36, 0xb5, 0x81, 0x94, 0xfe, 0x62, 0x3a, 0xe8,
	0xc9, 0x95, 0x84, 0xbd, 0x65, 0x15, 0x16, 0x15, 0xd2, 0xe7, 0x16, 0xd7, 0x9c, 0xd3, 0xd2, 0x66,
	0xf6, 0x46, 0xe3, 0x32, 0x62, 0x51, 0x86, 0x4a, 0x67, 0xcc, 0x4d, 0xea, 0x37, 0x45, 0xd5, 0xa6,
	0x80, 0xe6, 0xba, 0xb3, 0x08, 0xd8, 0x30, 0x5b, 0x5f, 0xf2, 0x5a, 0xfb, 0x63, 0xb0, 0xa4, 0x41
};

void pgm_decrypt_kov()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010 && (i & 0x180000) != 0x080000) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x004820) == 0x004820) x ^= 0x0080;

		x ^= kov_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char kovsh_tab[256] = {
	0xe7, 0x06, 0xa3, 0x70, 0xf2, 0x58, 0xe6, 0x59, 0xe4, 0xcf, 0xc2, 0x79, 0x1d, 0xe3, 0x71, 0x0e,
	0xb6, 0x90, 0x9a, 0x2a, 0x8c, 0x41, 0xf7, 0x82, 0x9b, 0xef, 0x99, 0x0c, 0xfa, 0x2f, 0xf1, 0xfe,
	0x8f, 0x70, 0xf4, 0xc1, 0xb5, 0x3d, 0x7c, 0x60, 0x4c, 0x09, 0xf4, 0x2e, 0x7c, 0x87, 0x63, 0x5f,
	0xce, 0x99, 0x84, 0x95, 0x06, 0x9a, 0x20, 0x23, 0x5a, 0xb9, 0x52, 0x95, 0x48, 0x2c, 0x84, 0x60,
	0x69, 0xe3, 0x93, 0x49, 0xb9, 0xd6, 0xbb, 0xd6, 0x9e, 0xdc, 0x96, 0x12, 0xfa, 0x60, 0xda, 0x5f,
	0x55, 0x5d, 0x5b, 0x20, 0x07, 0x1e, 0x97, 0x42, 0x77, 0xea, 0x1d, 0xe0, 0x70, 0xfb, 0x6a, 0x00,
	0x77, 0x9a, 0xef, 0x1b, 0xe0, 0xf9, 0x0d, 0xc1, 0x2e, 0x2f, 0xef, 0x25, 0x29, 0xe5, 0xd8, 0x2c,
	0xaf, 0x01, 0xd9, 0x6c, 0x31, 0xce, 0x5c, 0xea, 0xab, 0x1c, 0x92, 0x16, 0x61, 0xbc, 0xe4, 0x7c,
	0x5a, 0x76, 0xe9, 0x92, 0x39, 0x5b, 0x97, 0x60, 0xea, 0x57, 0x83, 0x9c, 0x92, 0x29, 0xa7, 0x12,
	0xa9, 0x71, 0x7a, 0xf9, 0x07, 0x68, 0xa7, 0x45, 0x88, 0x10, 0x81, 0x12, 0x2c, 0x67, 0x4d, 0x55,
	0x33, 0xf0, 0xfa, 0xd7, 0x1d, 0x4d, 0x0e, 0x63, 0x03, 0x34, 0x65, 0xe2, 0x76, 0x0f, 0x98, 0xa9,
	0x5f, 0x9a, 0xd3, 0xca, 0xdd, 0xc1, 0x5b, 0x3d, 0x4d, 0xf8, 0x40, 0x08, 0xdc, 0x05, 0x38, 0x00,
	0xcb, 0x24, 0x02, 0xff, 0x39, 0xe2, 0x9e, 0x04, 0x9a, 0x08, 0x63, 0xc8, 0x2b, 0x5a, 0x34, 0x06,
	0x62, 0xc1, 0xbb, 0x8a, 0xd0, 0x54, 0x4c, 0x43, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce,
	0x2a, 0x7b, 0x09, 0x62, 0x1a, 0x91, 0x9b, 0xc3, 0x41, 0x24, 0xa0, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0xa7, 0xb8, 0x85, 0x8a, 0xa1, 0x1e, 0x4f, 0xb6, 0x75, 0x38, 0x65, 0x8a, 0xf9, 0x7c, 0x00, 0xa0
};

void pgm_decrypt_kovsh()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008 && (i & 0x180000) != 0x000000) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= kovsh_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char kovshp_tab[256] = { // IGS0009rd1040219
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x39, 0x72, 0x64, 0x31, 0x30, 0x34, 0x30, 0x32, 0x31, 0x39,
	0xf9, 0x8c, 0xbd, 0x87, 0x16, 0x07, 0x39, 0xeb, 0x29, 0x9e, 0x17, 0xef, 0x4f, 0x64, 0x7c, 0xe0,
	0x5f, 0x73, 0x5b, 0xa1, 0x5e, 0x95, 0x0d, 0xf1, 0x40, 0x36, 0x2f, 0x00, 0xe2, 0x8a, 0xbc, 0x32,
	0x44, 0xfa, 0x6c, 0x33, 0x0b, 0xd5, 0x4c, 0x3b, 0x36, 0x34, 0x9e, 0xa3, 0x20, 0x2e, 0xf3, 0xa9,
	0xb7, 0x3e, 0x87, 0x80, 0xfb, 0xf1, 0xdd, 0x9c, 0xba, 0xd3, 0x9b, 0x3b, 0x8a, 0x9c, 0xa8, 0x37,
	0x07, 0x97, 0x84, 0x0c, 0x4e, 0x54, 0xe7, 0x25, 0xba, 0x8e, 0x9d, 0x6b, 0xde, 0x5f, 0xa1, 0x10,
	0xc3, 0xa2, 0x79, 0x99, 0x63, 0xa9, 0xd1, 0x2a, 0x65, 0x20, 0x5b, 0x16, 0x1b, 0x41, 0xe6, 0xa7,
	0xba, 0x3a, 0xbd, 0x2a, 0xd8, 0xdb, 0x43, 0x3f, 0x2b, 0x85, 0xcc, 0x5f, 0x80, 0x4f, 0xbe, 0xae,
	0xfa, 0x79, 0xe8, 0x03, 0x8d, 0x16, 0x22, 0x35, 0xbb, 0xf6, 0x26, 0xa9, 0x8d, 0xd2, 0xaf, 0x19,
	0xd4, 0xbb, 0xd0, 0xa6, 0xa1, 0xc4, 0x96, 0x21, 0x02, 0xef, 0xe1, 0x96, 0x00, 0x56, 0x80, 0x1b,
	0xd6, 0x9a, 0x8c, 0xd7, 0x73, 0x91, 0x07, 0x55, 0x32, 0x2b, 0xb5, 0x0b, 0xd8, 0xa5, 0x39, 0x26,
	0xce, 0xf2, 0x74, 0x98, 0xa1, 0x66, 0x1a, 0x64, 0xb8, 0xa5, 0x96, 0x29, 0x54, 0xcb, 0x21, 0xed,
	0xcd, 0xdd, 0x1e, 0x2c, 0x0b, 0x70, 0xb8, 0x22, 0x43, 0x98, 0xbe, 0x54, 0xf3, 0x14, 0xbe, 0x65,
	0x21, 0xb7, 0x61, 0x17, 0xcf, 0x19, 0x07, 0xa0, 0xc2, 0x7f, 0xa3, 0x30, 0x75, 0x08, 0xd8, 0xbf,
	0x58, 0x1a, 0x55, 0x1b, 0x4e, 0x0d, 0x6d, 0x32, 0x65, 0x15, 0xfb, 0x9e, 0xd8, 0x75, 0x76, 0x6f,
	0x42, 0xe2, 0x4f, 0x3c, 0x25, 0x35, 0x93, 0x6c, 0x9b, 0x56, 0xbe, 0xc1, 0x5b, 0x65, 0xde, 0x27
};

void pgm_decrypt_kovshp()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008 && (i & 0x180000) != 0x000000) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= kovshp_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char photoy2k_tab[256] = {
	0xd9, 0x92, 0xb2, 0xbc, 0xa5, 0x88, 0xe3, 0x48, 0x7d, 0xeb, 0xc5, 0x4d, 0x31, 0xe4, 0x82, 0xbc,
	0x82, 0xcf, 0xe7, 0xf3, 0x15, 0xde, 0x8f, 0x91, 0xef, 0xc6, 0xb8, 0x81, 0x97, 0xe3, 0xdf, 0x4d,
	0x88, 0xbf, 0xe4, 0x05, 0x25, 0x73, 0x1e, 0xd0, 0xcf, 0x1e, 0xeb, 0x4d, 0x18, 0x4e, 0x6f, 0x9f,
	0x00, 0x72, 0xc3, 0x74, 0xbe, 0x02, 0x09, 0x0a, 0xb0, 0xb1, 0x8e, 0x9b, 0x08, 0xed, 0x68, 0x6d,
	0x25, 0xe8, 0x28, 0x94, 0xa6, 0x44, 0xa6, 0xfa, 0x95, 0x69, 0x72, 0xd3, 0x6d, 0xb6, 0xff, 0xf3,
	0x45, 0x4e, 0xa3, 0x60, 0xf2, 0x58, 0xe7, 0x59, 0xe4, 0x4f, 0x70, 0xd2, 0xdd, 0xc0, 0x6e, 0xf3,
	0xd7, 0xb2, 0xdc, 0x1e, 0xa8, 0x41, 0x07, 0x5d, 0x60, 0x15, 0xea, 0xcf, 0xdb, 0xc1, 0x1d, 0x4d,
	0xb7, 0x42, 0xec, 0xc4, 0xca, 0xa9, 0x40, 0x30, 0x0f, 0x3c, 0xe2, 0x81, 0xe0, 0x5c, 0x51, 0x07,
	0xb0, 0x1e, 0x4a, 0xb3, 0x64, 0x3e, 0x1c, 0x62, 0x17, 0xcd, 0xf2, 0xe4, 0x14, 0x9d, 0xa6, 0xd4,
	0x64, 0x36, 0xa5, 0xe8, 0x7e, 0x84, 0x0e, 0xb3, 0x5d, 0x79, 0x57, 0xea, 0xd7, 0xad, 0xbc, 0x9e,
	0x2d, 0x90, 0x03, 0x9e, 0x0e, 0xc6, 0x98, 0xdb, 0xe3, 0xb6, 0x9f, 0x9b, 0xf6, 0x21, 0xe6, 0x98,
	0x94, 0x77, 0xb7, 0x2b, 0xaa, 0xc9, 0xff, 0xef, 0x7a, 0xf2, 0x71, 0x4e, 0x52, 0x06, 0x85, 0x37,
	0x81, 0x8e, 0x86, 0x64, 0x39, 0x92, 0x2a, 0xca, 0xf3, 0x3e, 0x87, 0xb5, 0x0c, 0x7b, 0x42, 0x5e,
	0x04, 0xa7, 0xfb, 0xd7, 0x13, 0x7f, 0x83, 0x6a, 0x77, 0x0f, 0xa7, 0x34, 0x51, 0x88, 0x9c, 0xac,
	0x23, 0x90, 0x4d, 0x4d, 0x72, 0x4e, 0xa3, 0x26, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a, 0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d
};

void pgm_decrypt_photoy2k()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x40080) != 0x00080) x ^= 0x0001;
		if ((i & 0x84008) == 0x84008) x ^= 0x0002;
		if ((i & 0x00030) == 0x00010) x ^= 0x0004;
		if ((i & 0x00242) != 0x00042) x ^= 0x0008;
		if ((i & 0x48100) == 0x48000) x ^= 0x0010;
		if ((i & 0x02004) != 0x00004) x ^= 0x0020;
		if ((i & 0x01800) != 0x00000) x ^= 0x0040;
		if ((i & 0x04820) == 0x04820) x ^= 0x0080;

		x ^= photoy2k_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char puzlstar_tab[256] = {
	0x62, 0x59, 0x17, 0xe3, 0xe1, 0x11, 0x02, 0x97, 0x67, 0x4d, 0x4a, 0x1c, 0x1f, 0x76, 0x64, 0xc1,
	0xfa, 0x04, 0xd2, 0x9f, 0x22, 0xf5, 0xfd, 0xfe, 0x89, 0xeb, 0xab, 0x41, 0xc0, 0xac, 0x2b, 0x64,
	0xfe, 0x1e, 0x9b, 0x68, 0x07, 0xfd, 0x75, 0x25, 0x24, 0xa0, 0x41, 0x5d, 0x79, 0xb5, 0x67, 0x93,
	0xe1, 0xa3, 0x3a, 0x24, 0xa0, 0xbf, 0x64, 0x62, 0x38, 0x65, 0x75, 0x78, 0x8a, 0xf9, 0x7c, 0x00,
	0x71, 0x38, 0xc5, 0xe4, 0xdc, 0xf8, 0xc5, 0xc5, 0x8d, 0x1d, 0x6b, 0xb8, 0x84, 0x4e, 0x21, 0x6d,
	0x55, 0x95, 0x31, 0x86, 0x11, 0xe6, 0xb1, 0x77, 0x77, 0x3b, 0x01, 0x82, 0xab, 0x88, 0xef, 0x77,
	0x08, 0xd8, 0x80, 0xba, 0x41, 0xfb, 0xb3, 0x30, 0x5f, 0xf2, 0x5b, 0xe6, 0x5a, 0x63, 0xb0, 0xa4,
	0x37, 0x37, 0x6d, 0x1f, 0x27, 0x0e, 0x80, 0x8e, 0x1a, 0x45, 0x97, 0xdb, 0x61, 0x10, 0x24, 0x8a,
	0x62, 0x44, 0xf2, 0x91, 0x74, 0x25, 0x3e, 0xc5, 0xe6, 0x8e, 0x55, 0x44, 0x5a, 0x8a, 0x90, 0x25,
	0xa1, 0x0d, 0x43, 0x35, 0x46, 0x80, 0xfe, 0xb3, 0x4e, 0x4c, 0x21, 0xcc, 0x99, 0xc2, 0x3d, 0xce,
	0x19, 0xc5, 0x2c, 0xae, 0xe8, 0x94, 0xad, 0xed, 0x36, 0xb5, 0x8e, 0xf1, 0x81, 0xfe, 0x62, 0x3a,
	0x8c, 0xe0, 0x90, 0x88, 0xf1, 0x31, 0xc5, 0xd9, 0x3c, 0xf4, 0x70, 0x7d, 0x7d, 0x1c, 0xca, 0xe2,
	0x51, 0xce, 0xa0, 0x33, 0x42, 0x1e, 0xf7, 0xe4, 0x13, 0xcf, 0xa0, 0x95, 0x33, 0x59, 0x5b, 0xec,
	0xf7, 0x6d, 0x56, 0xac, 0x6a, 0xae, 0x55, 0x9b, 0x92, 0x14, 0x6d, 0x01, 0x23, 0x4b, 0x80, 0xae,
	0x65, 0x15, 0xc9, 0x84, 0x66, 0xd7, 0xbd, 0x16, 0xd2, 0xe7, 0x15, 0x95, 0x16, 0x9c, 0xd3, 0xd2,
	0x62, 0x51, 0xf6, 0xe3, 0xa6, 0xea, 0x32, 0x86, 0x67, 0xcc, 0x4a, 0x46, 0x4d, 0x37, 0x45, 0xd5
};

void pgm_decrypt_puzlstar()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x40480) != 0x00080) x ^= 0x0001;
		if ((i & 0x00030) == 0x00010) x ^= 0x0004;
		if ((i & 0x00242) != 0x00042) x ^= 0x0008;
		if ((i & 0x08100) == 0x08000) x ^= 0x0010;
		if ((i & 0x22004) != 0x00004) x ^= 0x0020;
		if ((i & 0x11800) != 0x10000) x ^= 0x0040;
		if ((i & 0x04820) == 0x04820) x ^= 0x0080;

		x ^= puzlstar_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char puzzli2_tab[256] = {
	0xb7, 0x66, 0xa3, 0xc0, 0x51, 0x55, 0x6d, 0x63, 0x86, 0x60, 0x64, 0x6c, 0x67, 0x18, 0x0b, 0x05,
	0x62, 0xff, 0xe0, 0x1e, 0x30, 0x21, 0x2e, 0x40, 0x41, 0xb9, 0x60, 0x38, 0xd1, 0x24, 0x7e, 0x36,
	0x7a, 0x0b, 0x1c, 0x69, 0x4f, 0x09, 0xe1, 0x9e, 0xcf, 0xcd, 0x7c, 0x00, 0x73, 0x08, 0x77, 0x37,
	0x5f, 0x50, 0x32, 0x3e, 0xd3, 0x54, 0x77, 0x6b, 0x60, 0x60, 0x74, 0x7c, 0x55, 0x4f, 0x44, 0x5e,
	0x66, 0x5c, 0x58, 0x26, 0x35, 0x29, 0x3f, 0x35, 0x3f, 0x1c, 0x0b, 0x0d, 0x08, 0x5b, 0x59, 0x5c,
	0xa0, 0xa5, 0x87, 0x85, 0x24, 0x75, 0x5f, 0x42, 0x1b, 0xf3, 0x1a, 0x58, 0x17, 0x58, 0x71, 0x6b,
	0x69, 0x89, 0x7d, 0x3a, 0xf3, 0xc4, 0x5d, 0xa0, 0x4f, 0x27, 0x58, 0xc4, 0xa8, 0xdd, 0xa8, 0xfb,
	0xbe, 0xa4, 0xe2, 0xee, 0x07, 0x10, 0x90, 0x72, 0x99, 0x08, 0x68, 0x6d, 0x5c, 0x5c, 0x6d, 0x58,
	0x2f, 0xdc, 0x15, 0xd5, 0xd6, 0xd6, 0x3b, 0x3b, 0xf9, 0x32, 0xcc, 0xdd, 0xd4, 0xf1, 0xea, 0xed,
	0xe4, 0xf6, 0xf2, 0x91, 0xca, 0xc1, 0xed, 0xf2, 0xf6, 0xfb, 0xc0, 0xe8, 0xe3, 0xe7, 0xfa, 0xf1,
	0xf5, 0x08, 0x26, 0x2b, 0x2f, 0x34, 0x39, 0x13, 0x28, 0x07, 0x88, 0x5b, 0x8f, 0x94, 0x9b, 0x2e,
	0xf5, 0xab, 0x72, 0x76, 0x7a, 0x40, 0xb9, 0x09, 0xd8, 0x3b, 0xcd, 0x31, 0x3d, 0x42, 0xab, 0xb1,
	0xb5, 0xb9, 0x3b, 0xe3, 0x0b, 0x65, 0x18, 0xfb, 0x1f, 0x12, 0xe4, 0xe8, 0xec, 0xf2, 0xf7, 0xfc,
	0xc0, 0xe8, 0xe0, 0xe6, 0xfa, 0xf1, 0xf4, 0x0b, 0x26, 0x2b, 0x30, 0x35, 0x39, 0x13, 0x29, 0x21,
	0x0c, 0x11, 0x16, 0x1b, 0x1f, 0x64, 0x0e, 0x60, 0x05, 0x79, 0x7c, 0x37, 0x00, 0x0f, 0x4f, 0x38,
	0x1d, 0x18, 0xa2, 0xb6, 0xb2, 0xa9, 0xac, 0xab, 0xae, 0x91, 0x98, 0x8d, 0x91, 0xbb, 0xb1, 0xc0
};

void pgm_decrypt_puzzli2()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

	    	if ((i & 0x040080) != 0x000080) x ^= 0x0001;
	    	if ((i & 0x004008) == 0x004008) x ^= 0x0002;
	    	if ((i & 0x000030) == 0x000010) x ^= 0x0004;
	    	if ((i & 0x000242) != 0x000042) x ^= 0x0008;
	    	if ((i & 0x008100) == 0x008000) x ^= 0x0010;
	    	if ((i & 0x022004) != 0x000004) x ^= 0x0020;
	    	if ((i & 0x011800) != 0x010000) x ^= 0x0040;
	    	if ((i & 0x004820) == 0x004820) x ^= 0x0080;

	    	x ^= puzzli2_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char oldsplus_tab[256] = { // IGS0013RD1040727
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x33, 0x52, 0x44, 0x31, 0x30, 0x34, 0x30, 0x37, 0x32, 0x37,
	0xf5, 0x79, 0x6d, 0xab, 0x04, 0x22, 0x51, 0x96, 0xf2, 0x72, 0xe8, 0x3a, 0x96, 0xd2, 0x9a, 0xcc,
	0x3f, 0x47, 0x3c, 0x09, 0xf2, 0xd9, 0x72, 0x41, 0xe6, 0x44, 0x43, 0xa7, 0x3e, 0xe2, 0xfd, 0xd8,
	0x06, 0xd8, 0x4c, 0xa9, 0x70, 0x80, 0x95, 0x35, 0x50, 0x17, 0x99, 0x27, 0xd5, 0xa8, 0x47, 0x45,
	0x89, 0x38, 0xe1, 0x3d, 0x8c, 0x33, 0x53, 0xb4, 0x0d, 0x17, 0xd1, 0x8d, 0x09, 0x5f, 0xaf, 0x76,
	0x48, 0xb2, 0x85, 0xb9, 0x95, 0x4c, 0x83, 0x42, 0x3d, 0xad, 0x11, 0xec, 0xca, 0x82, 0xac, 0x10,
	0x01, 0xd0, 0xfd, 0x50, 0x19, 0x67, 0x3b, 0xa0, 0x3e, 0x86, 0xc2, 0x97, 0x46, 0xcb, 0xf4, 0xf5,
	0xb3, 0x5f, 0x50, 0x74, 0xe9, 0x5f, 0xd2, 0xd4, 0xb0, 0x8d, 0x8a, 0x21, 0xed, 0x37, 0x80, 0x47,
	0x9d, 0x68, 0xc7, 0xd9, 0x12, 0x4e, 0xdf, 0x1e, 0x72, 0xeb, 0x50, 0x5e, 0x6d, 0x00, 0x85, 0x6b,
	0x3e, 0x37, 0xe6, 0x72, 0xe5, 0x8f, 0x3a, 0x03, 0xa3, 0x0d, 0x3b, 0x5f, 0xb6, 0xa1, 0x7b, 0x02,
	0x56, 0x56, 0x77, 0x71, 0xef, 0xbe, 0xf9, 0x46, 0xa1, 0x9d, 0xb3, 0x79, 0xf6, 0xd5, 0x19, 0xf0,
	0xe2, 0x91, 0x7e, 0x4a, 0x01, 0xb6, 0x73, 0xe8, 0x0c, 0x86, 0x5d, 0x3e, 0x9c, 0x97, 0x55, 0x58,
	0x23, 0xf4, 0x45, 0xb0, 0x28, 0x91, 0x40, 0x2f, 0xc2, 0xf4, 0x21, 0x81, 0x58, 0x22, 0x68, 0x9d,
	0x97, 0xc7, 0x51, 0x95, 0xb4, 0xaa, 0x36, 0x9b, 0xe4, 0x51, 0x27, 0x55, 0x18, 0xf0, 0xc7, 0x62,
	0xfe, 0x98, 0x6a, 0x2d, 0x35, 0x9d, 0x6c, 0xf1, 0xcf, 0x48, 0xd4, 0x0d, 0x0c, 0xbe, 0x2a, 0x8a,
	0x55, 0x31, 0x96, 0xea, 0x78, 0x45, 0x3a, 0x33, 0x23, 0xc5, 0xd1, 0x3c, 0xa3, 0x86, 0x88, 0x38
};

void pgm_decrypt_oldsplus()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x048100) == 0x048000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= oldsplus_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char py2k2_tab[256] = {
	0x74, 0xe8, 0xa8, 0x64, 0x26, 0x44, 0xa6, 0x9a, 0xa5, 0x69, 0xa2, 0xd3, 0x6d, 0xba, 0xff, 0xf3,
	0xeb, 0x6e, 0xe3, 0x70, 0x72, 0x58, 0x27, 0xd9, 0xe4, 0x9f, 0x50, 0xa2, 0xdd, 0xce, 0x6e, 0xf6,
	0x44, 0x72, 0x0c, 0x7e, 0x4d, 0x41, 0x77, 0x2d, 0x00, 0xad, 0x1a, 0x5f, 0x6b, 0xc0, 0x1d, 0x4e,
	0x4c, 0x72, 0x62, 0x3c, 0x32, 0x28, 0x43, 0xf8, 0x9d, 0x52, 0x05, 0x7e, 0xd1, 0xee, 0x82, 0x61,
	0x3b, 0x3f, 0x77, 0xf3, 0x8f, 0x7e, 0x3f, 0xf1, 0xdf, 0x8f, 0x68, 0x43, 0xd7, 0x68, 0xdf, 0x19,
	0x87, 0xff, 0x74, 0xe5, 0x3f, 0x43, 0x8e, 0x80, 0x0f, 0x7e, 0xdb, 0x32, 0xe8, 0xd1, 0x66, 0x8f,
	0xbe, 0xe2, 0x33, 0x94, 0xc8, 0x32, 0x39, 0xfa, 0xf0, 0x43, 0xde, 0x84, 0x18, 0xd0, 0x6d, 0xd5,
	0x74, 0x98, 0xf8, 0x64, 0xcf, 0x84, 0xc6, 0xea, 0x55, 0x32, 0xe2, 0x38, 0xdd, 0xea, 0xfd, 0x6c,
	0xeb, 0x6e, 0xe3, 0x70, 0xae, 0x38, 0xc7, 0xd9, 0x54, 0x84, 0x10, 0xc1, 0xfd, 0x1e, 0x6e, 0x6d,
	0x37, 0xe0, 0x03, 0x9e, 0x06, 0x36, 0x68, 0x5b, 0xe3, 0xf6, 0x7f, 0x0b, 0x56, 0x79, 0xe0, 0xa8,
	0x98, 0x77, 0xc7, 0x2b, 0xa5, 0x79, 0xff, 0x2f, 0xca, 0x15, 0x71, 0x7e, 0x02, 0xbf, 0x87, 0xb7,
	0x7a, 0x8e, 0xe6, 0x64, 0x32, 0x62, 0x2a, 0xca, 0x23, 0x72, 0x87, 0xb5, 0x0c, 0x02, 0x4b, 0xee,
	0x44, 0x72, 0x9c, 0x7e, 0x5d, 0xc1, 0xa7, 0x1d, 0x30, 0x38, 0xda, 0xc9, 0x5b, 0xd0, 0x11, 0xf9,
	0xb1, 0x72, 0x6c, 0x04, 0x31, 0xc9, 0x50, 0x60, 0x6f, 0xc1, 0xf2, 0xae, 0x00, 0xf4, 0x5d, 0x66,
	0x43, 0x0e, 0x7a, 0xc3, 0x76, 0xae, 0x3c, 0xc2, 0xb7, 0xc9, 0x52, 0xf4, 0x74, 0x51, 0xaf, 0x12,
	0x19, 0xc6, 0x75, 0xe8, 0x6c, 0x54, 0x7e, 0x63, 0xdd, 0xae, 0x07, 0x5a, 0xb7, 0x00, 0xb5, 0x5e
};

void pgm_decrypt_py2k2() // and ddp3
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x084008) == 0x084008) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010 && (i & 0x180000) != 0x080000) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x004820) == 0x004820) x ^= 0x0080;

		x ^= py2k2_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char ketsui_tab[256] = { // IGS0004RD1021015
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x34, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x35,
	0x7c, 0x49, 0x27, 0xa5, 0xff, 0xf6, 0x98, 0x2d, 0x0f, 0x3d, 0x12, 0x23, 0xe2, 0x30, 0x50, 0xcf,
	0xf1, 0x82, 0xf0, 0xce, 0x48, 0x44, 0x5b, 0xf3, 0x0d, 0xdf, 0xf8, 0x5d, 0x50, 0x53, 0x91, 0xd9,
	0x12, 0xaf, 0x05, 0x7a, 0x98, 0xd0, 0x2f, 0x76, 0xf1, 0x5d, 0x17, 0x44, 0xc5, 0x03, 0x58, 0xf4,
	0x61, 0xee, 0xd1, 0xce, 0x00, 0x88, 0x90, 0x2e, 0x5c, 0x76, 0xfb, 0x9f, 0x75, 0xcf, 0x40, 0x37,
	0xa1, 0x9f, 0x00, 0x32, 0xd5, 0x9c, 0x37, 0xd2, 0x32, 0x27, 0x6f, 0x76, 0xd3, 0x86, 0x25, 0xf9,
	0xd6, 0x60, 0x7b, 0x4e, 0xa9, 0x7a, 0x20, 0x59, 0x96, 0xb1, 0x7d, 0x10, 0x92, 0x37, 0x22, 0xd2,
	0x42, 0x12, 0x6f, 0x07, 0x4f, 0xd2, 0x87, 0xfa, 0xeb, 0x92, 0x71, 0xf3, 0xa4, 0x31, 0x91, 0x98,
	0x68, 0xd2, 0x47, 0x86, 0xda, 0x92, 0xe5, 0x2b, 0xd4, 0x89, 0xd7, 0xe7, 0x3d, 0x03, 0x0d, 0x63,
	0x0c, 0x00, 0xac, 0x31, 0x9d, 0xe9, 0xf6, 0xa5, 0x34, 0x95, 0x77, 0xf2, 0xcf, 0x7c, 0x72, 0x89,
	0x31, 0x3a, 0x8b, 0xae, 0x2b, 0x47, 0xb6, 0x5d, 0x2d, 0xf5, 0x5f, 0x5c, 0x0e, 0xab, 0xdb, 0xa1,
	0x18, 0x60, 0x0e, 0xe6, 0x58, 0x5b, 0x5e, 0x8b, 0x24, 0x29, 0xd8, 0xac, 0xed, 0xdf, 0xa2, 0x83,
	0x46, 0x91, 0xa1, 0xff, 0x35, 0x13, 0x6a, 0xa5, 0xba, 0xef, 0x6e, 0xa8, 0x9e, 0xa6, 0x62, 0x44,
	0x7e, 0x2c, 0xed, 0x60, 0x17, 0x9e, 0x96, 0x64, 0xd3, 0x46, 0xec, 0x58, 0x95, 0xd1, 0xf7, 0x3e,
	0xc2, 0xcf, 0xdf, 0xb0, 0x90, 0x6c, 0xdb, 0xbe, 0x93, 0x6d, 0x5d, 0x02, 0x85, 0x6e, 0x7c, 0x05,
	0x55, 0x5a, 0xa1, 0xd7, 0x73, 0x2b, 0x76, 0xe9, 0x5b, 0xe4, 0x0c, 0x2e, 0x60, 0xcb, 0x4b, 0x72
};

void pgm_decrypt_ketsui()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x080030) == 0x000010) x ^= 0x0004; // due to address starting at 0 and not 100000/2!
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= ketsui_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char espgal_tab[256] = { // IGS0007RD1030909
        0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x37, 0x52, 0x44, 0x31, 0x30, 0x33, 0x30, 0x39, 0x30, 0x39,
        0xa7, 0xf1, 0x0a, 0xca, 0x69, 0xb2, 0xce, 0x86, 0xec, 0x3d, 0xa2, 0x5a, 0x03, 0xe9, 0xbf, 0xba,
        0xf7, 0xd5, 0xec, 0x68, 0x03, 0x90, 0x15, 0xcc, 0x0d, 0x08, 0x2d, 0x76, 0xa5, 0xb5, 0x41, 0xf1,
        0x43, 0x06, 0xdd, 0xcb, 0xbd, 0x0c, 0xa4, 0xe2, 0x08, 0x65, 0x2a, 0xf0, 0x30, 0x6b, 0x15, 0x59,
        0x99, 0x9e, 0x75, 0x35, 0x77, 0x4f, 0x60, 0x99, 0x8c, 0x8f, 0xd2, 0x2b, 0x21, 0x57, 0xc3, 0xe5,
        0x48, 0xf9, 0x8a, 0x29, 0x50, 0xc6, 0x71, 0x06, 0x89, 0x01, 0x9a, 0xc9, 0x39, 0x04, 0x12, 0xc8,
        0xdf, 0xb1, 0x33, 0x6b, 0xa7, 0x1c, 0x3f, 0x7b, 0x2d, 0x76, 0x3a, 0xaf, 0x76, 0x3d, 0x08, 0x74,
        0x2c, 0xa2, 0xc8, 0xfd, 0x1a, 0x3a, 0x6f, 0x8b, 0xe8, 0xe9, 0xa9, 0xfe, 0x17, 0x0c, 0xed, 0x9d,
        0x40, 0xe6, 0xdf, 0x22, 0x89, 0x4d, 0xea, 0x09, 0x68, 0x96, 0x1e, 0x1a, 0x9c, 0xbd, 0x47, 0x35,
        0x68, 0xd9, 0x4f, 0x5e, 0x12, 0xbf, 0xd6, 0x09, 0x9d, 0xf6, 0x0f, 0xa7, 0xc2, 0xdb, 0xde, 0x70,
        0x35, 0x15, 0x2f, 0x73, 0x16, 0x3c, 0x9a, 0xdc, 0xb5, 0xc5, 0x35, 0x86, 0x8a, 0x31, 0xb8, 0xc1,
        0x74, 0x76, 0xd7, 0x65, 0x32, 0xad, 0xdc, 0x17, 0x1f, 0xfe, 0x85, 0xda, 0x32, 0xc9, 0x1d, 0xda,
        0x36, 0x16, 0xde, 0x76, 0x45, 0x3f, 0x85, 0x8c, 0x8b, 0xdc, 0x37, 0x08, 0x39, 0xef, 0x94, 0xaf,
        0xc8, 0x51, 0x19, 0x29, 0x70, 0x5d, 0xbb, 0x4e, 0xe8, 0xdb, 0xc2, 0xb2, 0x5f, 0x2e, 0xe3, 0x73,
        0xba, 0xc2, 0xa1, 0x42, 0x10, 0xb0, 0xe5, 0xb0, 0x64, 0xb4, 0xdc, 0xbb, 0xa1, 0x51, 0x12, 0x98,
        0xdc, 0x43, 0xcc, 0xc3, 0xc5, 0x25, 0xab, 0x45, 0x6e, 0x63, 0x7e, 0x45, 0x40, 0x63, 0x67, 0xd2
};

void pgm_decrypt_espgaluda()
{
	unsigned short *src = (unsigned short *)PGM68KROM;

	for (int i = 0; i < nPGM68KROMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x084008) == 0x084008) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x048100) == 0x048000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= espgal_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

void pgm_decrypt_svg()
{
	unsigned short *src = (unsigned short *)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x048100) == 0x048000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		src[i] = x;
	}
}

static unsigned char dfront_tab[256] = {
	0x51, 0xc4, 0xe3, 0x10, 0x1c, 0xad, 0x8a, 0x39, 0x8c, 0xe0, 0xa5, 0x04, 0x0f, 0xe4, 0x35, 0xc3,
	0x2d, 0x6b, 0x32, 0xe2, 0x60, 0x54, 0x63, 0x06, 0xa3, 0xf1, 0x0b, 0x5f, 0x6c, 0x5c, 0xb3, 0xec,
	0x77, 0x61, 0x69, 0xe7, 0x3c, 0xb7, 0x42, 0x72, 0x1a, 0x70, 0xb0, 0x96, 0xa4, 0x28, 0xc0, 0xfb,
	0x0a, 0x00, 0xcb, 0x15, 0x49, 0x48, 0xd3, 0x94, 0x58, 0xcf, 0x41, 0x86, 0x17, 0x71, 0xb1, 0xbd,
	0x21, 0x01, 0x37, 0x1e, 0xba, 0xeb, 0xf3, 0x59, 0xf6, 0xa7, 0x29, 0x4f, 0xb5, 0xca, 0x4c, 0x34,
	0x20, 0xa2, 0x62, 0x4b, 0x93, 0x9e, 0x47, 0x9f, 0x8d, 0x0e, 0x1b, 0xb6, 0x4d, 0x82, 0xd5, 0xf4,
	0x85, 0x79, 0x53, 0x92, 0x9b, 0xf7, 0xea, 0x44, 0x76, 0x1f, 0x22, 0x45, 0xed, 0xbe, 0x11, 0x55,
	0xaf, 0xf5, 0xf8, 0x50, 0x07, 0xe6, 0xc7, 0x5e, 0xd7, 0xde, 0xe5, 0x26, 0x2b, 0xf2, 0x6a, 0x8b,
	0xb8, 0x98, 0x89, 0xdb, 0x14, 0x5b, 0xc5, 0x78, 0xdc, 0xd0, 0x87, 0x5d, 0xc1, 0x0d, 0x95, 0x97,
	0x7e, 0xa8, 0x24, 0x3d, 0xe1, 0xd1, 0x19, 0xa6, 0x99, 0xd8, 0x83, 0x1d, 0xff, 0x30, 0x9d, 0x05,
	0xd4, 0x02, 0x27, 0x7b, 0x13, 0xb2, 0x7f, 0x40, 0x12, 0xa0, 0x68, 0x67, 0x4e, 0x3a, 0x46, 0xb9,
	0xee, 0xdf, 0x66, 0xd6, 0x8f, 0xa9, 0x0c, 0x91, 0x65, 0x18, 0x52, 0x56, 0xd9, 0x74, 0x09, 0x6e,
	0xc6, 0x73, 0xc9, 0xfc, 0x03, 0x43, 0xef, 0xaa, 0x7c, 0xbb, 0x2c, 0x90, 0xcc, 0xce, 0xe8, 0xae,
	0x2a, 0xf9, 0x57, 0x88, 0xc8, 0xe9, 0x5a, 0xdd, 0x2e, 0x7d, 0x64, 0xc2, 0x6d, 0x3e, 0xfa, 0x80,
	0x16, 0xcd, 0x6f, 0x84, 0x8e, 0x9c, 0xf0, 0xac, 0xb4, 0x9a, 0x2f, 0xbc, 0x31, 0x23, 0xfe, 0x38,
	0x08, 0x75, 0xa1, 0x33, 0xab, 0xd2, 0xda, 0x81, 0xbf, 0x7a, 0x3b, 0x3f, 0x4a, 0xfd, 0x25, 0x36
};

void pgm_decrypt_dfront()
{
	unsigned short *src = (unsigned short *)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
	    	unsigned short x = src[i];

		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x104008) == 0x104008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x004820) == 0x004820) x ^= 0x0080;

		x ^= dfront_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static unsigned char ddp2_tab[256] = {
	0x2a, 0x4a, 0x39, 0x98, 0xac, 0x39, 0xb2, 0x55, 0x72, 0xf3, 0x7b, 0x3c, 0xee, 0x94, 0x6e, 0xd5,
	0xcd, 0xbc, 0x9a, 0xd0, 0x45, 0x7d, 0x49, 0x68, 0xb1, 0x61, 0x54, 0xef, 0xa2, 0x84, 0x29, 0x20,
	0x32, 0x52, 0x82, 0x04, 0x38, 0x69, 0x9f, 0x24, 0x46, 0xf4, 0x3f, 0xc2, 0xf1, 0x25, 0xac, 0x2d,
	0xdf, 0x2d, 0xb4, 0x51, 0xc7, 0xb5, 0xe5, 0x88, 0xbd, 0x3b, 0x5a, 0x25, 0x5b, 0xc7, 0xae, 0x5f,
	0x43, 0xcf, 0x89, 0xd9, 0xe2, 0x63, 0xc6, 0x76, 0x21, 0x2b, 0x77, 0xc0, 0x27, 0x98, 0xfd, 0x09,
	0xe1, 0x8c, 0x26, 0x2e, 0x92, 0x99, 0xbc, 0xbe, 0x0e, 0xba, 0xbf, 0x70, 0xe7, 0xb7, 0xe9, 0x37,
	0x5c, 0xd1, 0x5e, 0xad, 0x22, 0x17, 0xc5, 0x67, 0x9d, 0xc6, 0xfb, 0x53, 0xc7, 0x4d, 0x32, 0xb4,
	0xf2, 0x43, 0x53, 0x7c, 0x01, 0xfe, 0xd2, 0x91, 0x40, 0x85, 0xa3, 0xe8, 0xdf, 0xdb, 0xff, 0x6c,
	0x64, 0x15, 0xcd, 0x8e, 0x07, 0x82, 0x78, 0x8d, 0x4e, 0x2d, 0x66, 0x8a, 0x62, 0x6f, 0xd3, 0x6a,
	0xae, 0x16, 0x44, 0x1e, 0xed, 0xc4, 0x12, 0x7a, 0xbe, 0x05, 0x06, 0xce, 0x9b, 0x8a, 0xf7, 0xf8,
	0x74, 0x23, 0x73, 0x74, 0xb8, 0x13, 0xc2, 0x42, 0xea, 0xf9, 0x7f, 0xa9, 0xaf, 0x56, 0xd6, 0xb3,
	0xb7, 0xc4, 0x47, 0x31, 0x67, 0xaa, 0x58, 0x8b, 0x47, 0x1b, 0xf5, 0x75, 0x95, 0x8f, 0xf0, 0x3a,
	0x85, 0x76, 0x59, 0x24, 0x0c, 0xd7, 0x00, 0xb3, 0xdc, 0xfc, 0x65, 0x34, 0xde, 0xfa, 0xd8, 0xc3,
	0xc3, 0x5e, 0xe3, 0x9e, 0x02, 0x28, 0x50, 0x81, 0x95, 0x2f, 0xe4, 0xb5, 0xa0, 0x4d, 0xa1, 0x36,
	0x9d, 0x18, 0x6d, 0x79, 0x19, 0x3b, 0x1d, 0xb8, 0xe1, 0xcc, 0x61, 0x1a, 0xe2, 0x31, 0x4c, 0x3f,
	0xdc, 0xca, 0xd4, 0xda, 0xcd, 0xd2, 0x83, 0xca, 0xeb, 0x4f, 0xf2, 0x2f, 0x2d, 0x2a, 0xec, 0x1f
};

void pgm_decrypt_ddp2()
{
	unsigned short *src = (unsigned short *)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
    		unsigned short x = src[i];

		if ((i & 0x0480) != 0x0080) x ^= 0x0001;
		if ((i & 0x0042) != 0x0042) x ^= 0x0008;
		if ((i & 0x8100) == 0x8000) x ^= 0x0010;
		if ((i & 0x2004) != 0x0004) x ^= 0x0020;
		if ((i & 0x1800) != 0x0000) x ^= 0x0040;
		if ((i & 0x0820) == 0x0820) x ^= 0x0080;

		x ^= ddp2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static unsigned char mm_tab[256] = {
	0xd0, 0x45, 0xbc, 0x84, 0x93, 0x60, 0x7d, 0x49, 0x68, 0xb1, 0x54, 0xa2, 0x05, 0x29, 0x41, 0x20,
	0x04, 0x08, 0x52, 0x25, 0x89, 0xf4, 0x69, 0x9f, 0x24, 0x46, 0x3d, 0xf1, 0xf9, 0xab, 0xa6, 0x2d,
	0x18, 0x19, 0x6d, 0x33, 0x79, 0x23, 0x3b, 0x1d, 0xe0, 0xb8, 0x61, 0x1a, 0xe1, 0x4c, 0x5d, 0x3f,
	0x5e, 0x02, 0xe3, 0x4d, 0x9e, 0x80, 0x28, 0x50, 0xa0, 0x81, 0xe4, 0xa5, 0x97, 0xa1, 0x86, 0x36,
	0x1e, 0xed, 0x16, 0x8a, 0x44, 0x06, 0x64, 0x12, 0x9a, 0x7e, 0xce, 0x9b, 0xef, 0xf7, 0x3e, 0xf8,
	0x15, 0x07, 0xcb, 0x6f, 0x8e, 0x3c, 0x82, 0x70, 0x62, 0x8d, 0x66, 0x7a, 0x4e, 0xd3, 0xb6, 0x6a,
	0x51, 0xa7, 0x2c, 0xc7, 0xa4, 0x0b, 0xb5, 0xe5, 0x88, 0xbd, 0x5a, 0x5b, 0x1b, 0xae, 0xe6, 0x5f,
	0x2e, 0x92, 0x8c, 0xb7, 0x96, 0xba, 0x99, 0xbb, 0xbe, 0x0e, 0xbf, 0xe7, 0x2f, 0xe9, 0x30, 0x37,
	0x98, 0xac, 0x4a, 0x94, 0x38, 0xf3, 0x39, 0xb2, 0x55, 0x72, 0x7b, 0xee, 0xdd, 0x6e, 0x11, 0xd5,
	0x26, 0xa8, 0x71, 0xd6, 0x74, 0x7f, 0x13, 0xc2, 0x56, 0xea, 0xa9, 0xaf, 0xc3, 0x42, 0x03, 0xb3,
	0xc4, 0x6b, 0x47, 0xf0, 0x31, 0xf5, 0xaa, 0x58, 0x8f, 0x48, 0x75, 0x95, 0x35, 0x8b, 0x57, 0x3a,
	0x73, 0x0c, 0x59, 0xd8, 0x14, 0x65, 0xd7, 0x00, 0xfa, 0xdc, 0x34, 0xde, 0xc0, 0xb0, 0x87, 0xc1,
	0xc8, 0xcd, 0xd4, 0x2a, 0xda, 0xe8, 0xd2, 0x83, 0x0d, 0xca, 0xf2, 0x0f, 0xeb, 0xec, 0x9c, 0x1f,
	0xad, 0x22, 0xd1, 0x4b, 0x5c, 0xf6, 0x17, 0xc5, 0x67, 0x9d, 0xfb, 0xc9, 0xcc, 0x32, 0x1c, 0xb4,
	0xd9, 0xe2, 0xcf, 0x90, 0xb9, 0x2b, 0x63, 0xc6, 0x76, 0x21, 0x77, 0x27, 0xfc, 0xfd, 0x0a, 0x09,
	0x7c, 0x01, 0x43, 0xdb, 0x53, 0x85, 0xfe, 0x78, 0x91, 0x40, 0xa3, 0xdf, 0x4f, 0xff, 0x10, 0x6c
};

void pgm_decrypt_martmast() // and dw2001
{
	unsigned short *src = (unsigned short *)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
    		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x000030) == 0x000010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= mm_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char kov2_tab[256] = {
	0x11, 0x4a, 0x38, 0x98, 0xac, 0x39, 0xb2, 0x55, 0x72, 0xf3, 0x7b, 0x3c, 0xee, 0x94, 0x6e, 0xd5,
	0x41, 0xbc, 0x93, 0xd0, 0x45, 0x7d, 0x49, 0x68, 0xb1, 0x60, 0x54, 0xef, 0xa2, 0x84, 0x29, 0x20,
	0xa6, 0x52, 0x89, 0x04, 0x08, 0x69, 0x9f, 0x24, 0x46, 0xf4, 0x3d, 0xc3, 0xf1, 0x25, 0xab, 0x2d,
	0xe6, 0x2c, 0xa4, 0x51, 0xa7, 0xb5, 0xe5, 0x88, 0xbd, 0x0b, 0x5a, 0x35, 0x5b, 0xc7, 0xae, 0x5f,
	0x0a, 0xcf, 0xb9, 0xd9, 0xe2, 0x63, 0xc6, 0x76, 0x21, 0x2b, 0x77, 0xc0, 0x27, 0x90, 0xfd, 0x09,
	0x30, 0x8c, 0x96, 0x2e, 0x92, 0x99, 0xbb, 0xbe, 0x0e, 0xba, 0xbf, 0x80, 0xe7, 0xb7, 0xe9, 0x37,
	0x1c, 0xd1, 0x5c, 0xad, 0x22, 0x17, 0xc5, 0x67, 0x9d, 0xf6, 0xfb, 0x23, 0xc9, 0x4b, 0x32, 0xb4,
	0x10, 0x43, 0x53, 0x7c, 0x01, 0xfe, 0x78, 0x91, 0x40, 0x85, 0xa3, 0xe8, 0xdf, 0xdb, 0xff, 0x6c,
	0xb6, 0x15, 0xcb, 0x8e, 0x07, 0x82, 0x70, 0x8d, 0x4e, 0xdd, 0x66, 0x7a, 0x62, 0x6f, 0xd3, 0x6a,
	0x3e, 0x16, 0x44, 0x1e, 0xed, 0x64, 0x12, 0x9a, 0x7e, 0x05, 0x06, 0xce, 0x9b, 0x8a, 0xf7, 0xf8,
	0x03, 0x26, 0x71, 0x74, 0xa8, 0x13, 0xc2, 0x42, 0xea, 0xf9, 0x7f, 0xa9, 0xaf, 0x56, 0xd6, 0xb3,
	0x57, 0xc4, 0x47, 0x31, 0x6b, 0xaa, 0x58, 0x8b, 0x48, 0x1b, 0xf5, 0x75, 0x95, 0x8f, 0xf0, 0x3a,
	0x87, 0x73, 0x59, 0x14, 0x0c, 0xd7, 0x00, 0xb0, 0xdc, 0xfc, 0x65, 0x34, 0xde, 0xfa, 0xd8, 0xc1,
	0x86, 0x5e, 0xe3, 0x9e, 0x02, 0x28, 0x50, 0x81, 0x97, 0x2f, 0xe4, 0xa5, 0xa0, 0x4d, 0xa1, 0x36,
	0x5d, 0x18, 0x6d, 0x79, 0x19, 0x3b, 0x1d, 0xb8, 0xe1, 0xcc, 0x61, 0x1a, 0xe0, 0x33, 0x4c, 0x3f,
	0x9c, 0xc8, 0xd4, 0xda, 0xcd, 0xd2, 0x83, 0xca, 0xeb, 0x4f, 0xf2, 0x0f, 0x0d, 0x2a, 0xec, 0x1f
};

void pgm_decrypt_kov2()
{
	unsigned short *src = (unsigned short *)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
	    	unsigned short x = src[i];

	    	if ((i & 0x40080) != 0x00080) x ^= 0x0001;
	    	if ((i & 0x80030) == 0x80010) x ^= 0x0004;
	    	if ((i & 0x00042) != 0x00042) x ^= 0x0008;
	    	if ((i & 0x48100) == 0x48000) x ^= 0x0010;
	    	if ((i & 0x22004) != 0x00004) x ^= 0x0020;
	    	if ((i & 0x01800) != 0x00000) x ^= 0x0040;
	    	if ((i & 0x00820) == 0x00820) x ^= 0x0080;

	    	x ^= kov2_tab[(i >> 1) & 0xff] << 8;

	    	src[i] = x;
	}
}

static const unsigned char kov2p_tab[256] = {
	0x44, 0x47, 0xb8, 0x28, 0x03, 0xa2, 0x21, 0xbc, 0x17, 0x32, 0x4e, 0xe2, 0xdf, 0x69, 0x35, 0xc7,
	0xa2, 0x06, 0xec, 0x36, 0xd2, 0x44, 0x12, 0x6a, 0x8d, 0x51, 0x6b, 0x20, 0x69, 0x01, 0xca, 0xf0,
	0x71, 0xc4, 0x34, 0xdc, 0x6b, 0xd6, 0x42, 0x2a, 0x5d, 0xb5, 0xc7, 0x6f, 0x4f, 0xd8, 0xb3, 0xed,
	0x51, 0x9e, 0x37, 0x1e, 0xc0, 0x85, 0x2a, 0x91, 0xc6, 0x9c, 0xac, 0xf5, 0x20, 0x3b, 0x09, 0x74,
	0x24, 0xf1, 0xe0, 0x42, 0x02, 0xbe, 0x84, 0x75, 0x4a, 0x82, 0xa2, 0x17, 0xae, 0xb6, 0x24, 0x79,
	0x0a, 0x5a, 0x56, 0xcb, 0xa1, 0x2e, 0x47, 0xea, 0xa9, 0x25, 0x73, 0x79, 0x0b, 0x17, 0x9e, 0x33,
	0x64, 0xb6, 0x03, 0x7f, 0x4f, 0xc3, 0xae, 0x45, 0xe6, 0x82, 0x27, 0x01, 0x86, 0x6b, 0x50, 0x16,
	0xd3, 0x22, 0x90, 0x64, 0xfc, 0xa9, 0x31, 0x1c, 0x41, 0xd5, 0x07, 0xd3, 0xb2, 0xfe, 0x53, 0xd6,
	0x39, 0xfb, 0xe6, 0xbe, 0xda, 0x4d, 0x8a, 0x44, 0x3a, 0x9b, 0x9d, 0x56, 0x5e, 0x5f, 0xff, 0x6a,
	0xb6, 0xde, 0x2f, 0x12, 0x5a, 0x5d, 0xb0, 0xd0, 0x93, 0x92, 0xb2, 0x2c, 0x9d, 0x59, 0xee, 0x05,
	0xab, 0xa8, 0xd2, 0x25, 0x2c, 0xc5, 0xde, 0x18, 0x4d, 0xb6, 0x4e, 0x3d, 0xbf, 0xfa, 0xf9, 0x1d,
	0xba, 0x76, 0x79, 0xfc, 0x42, 0xb2, 0x8c, 0xae, 0xa9, 0x45, 0xba, 0xac, 0x55, 0x8e, 0x38, 0x67,
	0xc3, 0xa5, 0x0d, 0xdc, 0xcc, 0x91, 0x73, 0x69, 0x27, 0xbc, 0x80, 0xdf, 0x30, 0xa4, 0x05, 0xd8,
	0xe7, 0xd2, 0xb7, 0x4b, 0x3c, 0x10, 0x8c, 0x5d, 0x8a, 0xd7, 0x68, 0x7a, 0x61, 0x07, 0xf9, 0xa5,
	0x88, 0xda, 0xdf, 0x0c, 0x42, 0x1b, 0x11, 0xe0, 0xd1, 0x93, 0x7c, 0x63, 0x39, 0xc5, 0xed, 0x43,
	0x46, 0xdb, 0x30, 0x26, 0xd0, 0xdf, 0x7a, 0x86, 0x3e, 0x2e, 0x04, 0xbf, 0x49, 0x2a, 0xf9, 0x66
};

void pgm_decrypt_kov2p()
{
	unsigned short *src = (unsigned short*)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
		unsigned short x = src[i];

    		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x004008) == 0x004008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
    		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= kov2p_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char theglad_tab[256] = { // IGS0005RD1021203
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x35, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x32, 0x30, 0x33, 
	0xc4, 0xa3, 0x46, 0x78, 0x30, 0xb3, 0x8b, 0xd5, 0x2f, 0xc4, 0x44, 0xbf, 0xdb, 0x76, 0xdb, 0xea, 
	0xb4, 0xeb, 0x95, 0x4d, 0x15, 0x21, 0x99, 0xa1, 0xd7, 0x8c, 0x40, 0x1d, 0x43, 0xf3, 0x9f, 0x71, 
	0x3d, 0x8c, 0x52, 0x01, 0xaf, 0x5b, 0x8b, 0x63, 0x34, 0xc8, 0x5c, 0x1b, 0x06, 0x7f, 0x41, 0x96, 
	0x2a, 0x8d, 0xf1, 0x64, 0xda, 0xb8, 0x67, 0xba, 0x33, 0x1f, 0x2b, 0x28, 0x20, 0x13, 0xe6, 0x96, 
	0x86, 0x34, 0x25, 0x85, 0xb0, 0xd0, 0x6d, 0x85, 0xfe, 0x78, 0x81, 0xf1, 0xca, 0xe4, 0xef, 0xf2, 
	0x9b, 0x09, 0xe1, 0xb4, 0x8d, 0x79, 0x22, 0xe2, 0x00, 0xfb, 0x6f, 0x68, 0x80, 0x6a, 0x00, 0x69, 
	0xf5, 0xd3, 0x57, 0x7e, 0x0c, 0xca, 0x48, 0x31, 0xe5, 0x0d, 0x4a, 0xb9, 0xfd, 0x5c, 0xfd, 0xf8, 
	0x5f, 0x98, 0xfb, 0xb3, 0x07, 0x1a, 0xe3, 0x10, 0x96, 0x56, 0xa3, 0x56, 0x3d, 0xb1, 0x07, 0xe0, 
	0xe3, 0x9f, 0x7f, 0x62, 0x99, 0x01, 0x35, 0x60, 0x40, 0xbe, 0x4f, 0xeb, 0x79, 0xa0, 0x82, 0x9f, 
	0xcd, 0x71, 0xd8, 0xda, 0x1e, 0x56, 0xc2, 0x3e, 0x4e, 0x6b, 0x60, 0x69, 0x2d, 0x9f, 0x10, 0xf4, 
	0xa9, 0xd3, 0x36, 0xaa, 0x31, 0x2e, 0x4c, 0x0a, 0x69, 0xc3, 0x2a, 0xff, 0x15, 0x67, 0x96, 0xde, 
	0x3f, 0xcc, 0x0f, 0xa1, 0xac, 0xe2, 0xd6, 0x62, 0x7e, 0x6f, 0x3e, 0x1b, 0x2a, 0xed, 0x36, 0x9c, 
	0x9d, 0xa4, 0x14, 0xcd, 0xaa, 0x08, 0xa4, 0x26, 0xb7, 0x55, 0x70, 0x6c, 0xa9, 0x69, 0x52, 0xae, 
	0x0c, 0xe1, 0x38, 0x7f, 0x87, 0x78, 0x38, 0x75, 0x80, 0x9c, 0xd4, 0xe2, 0x0b, 0x52, 0x8f, 0xd2, 
	0x19, 0x4c, 0xb0, 0x45, 0xde, 0x48, 0x55, 0xae, 0x82, 0xab, 0xbc, 0xab, 0x0c, 0x5e, 0xce, 0x07
};

void pgm_decrypt_theglad()
{
	unsigned short *src = (unsigned short*)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040080) != 0x000080) x ^= 0x0001;
		if ((i & 0x104008) == 0x104008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000042) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x022004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= theglad_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char killbldp_tab[] = { // IGS0024RD1050908
	0x49, 0x47, 0x53, 0x30, 0x30, 0x32, 0x34, 0x52, 0x44, 0x31, 0x30, 0x35, 0x30, 0x39, 0x30, 0x38,
	0x12, 0xa0, 0xd1, 0x9e, 0xb1, 0x8a, 0xfb, 0x1f, 0x50, 0x51, 0x4b, 0x81, 0x28, 0xda, 0x5f, 0x41,
	0x78, 0x6c, 0x7a, 0xf0, 0xcd, 0x6b, 0x69, 0x14, 0x94, 0x55, 0xb6, 0x42, 0xdf, 0xfe, 0x10, 0x79,
	0x74, 0x08, 0xfa, 0xc0, 0x1c, 0xa5, 0xb4, 0x03, 0x2a, 0x91, 0x67, 0x2b, 0x49, 0x4a, 0x94, 0x7d,
	0x8b, 0x92, 0xbe, 0x35, 0xaf, 0x28, 0x56, 0x63, 0xb3, 0xc2, 0xe8, 0x06, 0x9b, 0x4e, 0x85, 0x66,
	0x7f, 0x6b, 0x70, 0xb7, 0xdb, 0x22, 0x0c, 0xeb, 0x13, 0xe9, 0x06, 0xd7, 0x45, 0xda, 0xbe, 0x8b,
	0x54, 0x30, 0xfc, 0xeb, 0x32, 0x02, 0xd0, 0x92, 0x6d, 0x44, 0xca, 0xe8, 0xfd, 0xfb, 0x5b, 0x81,
	0x4c, 0xc0, 0x8b, 0xb9, 0x87, 0x78, 0xdd, 0x8e, 0x24, 0x52, 0x80, 0xbe, 0xb4, 0x01, 0xb7, 0x21,
	0xeb, 0x3c, 0x8a, 0x49, 0xed, 0x73, 0xae, 0x58, 0xdb, 0xd2, 0xb2, 0x21, 0x9e, 0x7c, 0x6c, 0x82,
	0xf3, 0x01, 0xa3, 0x00, 0xb7, 0x21, 0xfe, 0xa5, 0x75, 0xc4, 0x2d, 0x17, 0x2d, 0x39, 0x56, 0xf9,
	0x67, 0xae, 0xc2, 0x87, 0x79, 0xf1, 0xc8, 0x6d, 0x15, 0x66, 0xfa, 0xe8, 0x16, 0x48, 0x8f, 0x1f,
	0x8b, 0x24, 0x10, 0xc4, 0x04, 0x93, 0x47, 0xe6, 0x1d, 0x37, 0x65, 0x1a, 0x49, 0xf8, 0x72, 0xcb,
	0xe1, 0x80, 0xfa, 0xdd, 0x6d, 0xf5, 0xf6, 0x89, 0x32, 0xf6, 0xf8, 0x75, 0xfc, 0xd8, 0x9b, 0x12,
	0x2d, 0x22, 0x2a, 0x3b, 0x06, 0x46, 0x90, 0x0c, 0x35, 0xa2, 0x80, 0xff, 0xa0, 0xb7, 0xe5, 0x4d,
	0x71, 0xa9, 0x8c, 0x84, 0x62, 0xf7, 0x10, 0x65, 0x4a, 0x7b, 0x06, 0x00, 0xe8, 0xa4, 0x6a, 0x13,
	0xf0, 0xf3, 0x4a, 0x9f, 0x54, 0xb4, 0xb1, 0xcc, 0xd4, 0xff, 0xd6, 0xff, 0xc9, 0xee, 0x86, 0x39
};

void pgm_decrypt_killbldp()
{
	unsigned short *src = (unsigned short*)PGMUSER0;

	for (int i = 0; i< nPGMExternalARMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x104008) == 0x104008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x008100) == 0x008000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= killbldp_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char happy6in1_tab[256] = { // IGS0008RD1031215
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x38, 0x52, 0x44, 0x31, 0x30, 0x33, 0x31, 0x32, 0x31, 0x35,
	0x14, 0xd6, 0x37, 0x5c, 0x5e, 0xc3, 0xd3, 0x62, 0x96, 0x3d, 0xfb, 0x47, 0xf0, 0xcb, 0xbf, 0xb0,
	0x60, 0xa1, 0xc2, 0x3d, 0x90, 0xd0, 0x58, 0x56, 0x22, 0xac, 0xdd, 0x39, 0x27, 0x7e, 0x58, 0x44,
	0xe0, 0x6b, 0x51, 0x80, 0xb4, 0xa4, 0xf0, 0x6f, 0x71, 0xd0, 0x57, 0x18, 0xc7, 0xb6, 0x41, 0x50,
	0x02, 0x2f, 0xdb, 0x4a, 0x08, 0x4b, 0xe3, 0x62, 0x92, 0xc3, 0xff, 0x26, 0xaf, 0x9f, 0x60, 0xa5,
	0x76, 0x28, 0x97, 0xfd, 0x0b, 0x10, 0xb7, 0x1f, 0xd5, 0xe0, 0xac, 0xe6, 0xfd, 0xa3, 0xdb, 0x58,
	0x2a, 0xd1, 0xfc, 0x3b, 0x7c, 0x7e, 0x34, 0xdc, 0xc7, 0xc4, 0x76, 0x1b, 0x11, 0x6d, 0x1b, 0xbb,
	0x4e, 0xe5, 0xc0, 0xe8, 0x5a, 0x60, 0x60, 0x0a, 0x38, 0x47, 0xb3, 0xc9, 0x89, 0xe9, 0xc6, 0x61,
	0x50, 0x5f, 0xdb, 0x28, 0xe5, 0xc0, 0x83, 0x5c, 0x37, 0x86, 0xfa, 0x32, 0x46, 0x40, 0xc3, 0x1d,
	0xdf, 0x7a, 0x85, 0x5c, 0x9a, 0xea, 0x24, 0xc7, 0x12, 0xdc, 0x23, 0xda, 0x65, 0xdf, 0x39, 0x02,
	0xeb, 0xb1, 0x32, 0x28, 0x3a, 0x69, 0x09, 0x7c, 0x5a, 0xe3, 0x44, 0x83, 0x45, 0x71, 0x8f, 0x64,
	0xa3, 0xbf, 0x9c, 0x6f, 0xc4, 0x07, 0x3a, 0xee, 0xdd, 0x77, 0xb4, 0x31, 0x87, 0xdf, 0x6d, 0xd4,
	0x75, 0x9f, 0xb9, 0x53, 0x75, 0xd0, 0xfe, 0xd1, 0xaa, 0xb2, 0x0b, 0x25, 0x08, 0x56, 0xb8, 0x27,
	0x10, 0x8c, 0xbf, 0x39, 0xce, 0x0f, 0xdb, 0x18, 0x10, 0xf0, 0x1f, 0xe5, 0xe8, 0x40, 0x98, 0x6f,
	0x64, 0x02, 0x27, 0xc3, 0x8c, 0x4f, 0x98, 0xf6, 0x9d, 0xcb, 0x07, 0x31, 0x85, 0x48, 0x75, 0xff,
	0x9f, 0xba, 0xa6, 0xd3, 0xb0, 0x5b, 0x3d, 0xdd, 0x22, 0x1f, 0x1b, 0x0e, 0x7f, 0x5a, 0xf4, 0x6a
};

void pgm_decrypt_happy6in1()
{
	unsigned short *src = (unsigned short*)PGMUSER0;

	for (int i = 0; i < nPGMExternalARMLen/2; i++) {
		unsigned short x = src[i];

		if ((i & 0x040480) != 0x000080) x ^= 0x0001;
		if ((i & 0x104008) == 0x104008) x ^= 0x0002;
		if ((i & 0x080030) == 0x080010) x ^= 0x0004;
		if ((i & 0x000242) != 0x000042) x ^= 0x0008;
		if ((i & 0x048100) == 0x048000) x ^= 0x0010;
		if ((i & 0x002004) != 0x000004) x ^= 0x0020;
		if ((i & 0x011800) != 0x010000) x ^= 0x0040;
		if ((i & 0x000820) == 0x000820) x ^= 0x0080;

		x ^= happy6in1_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// ------------------------------------------------------------------------------------------------------------
// Bootleg decryption routines

void pgm_decode_kovqhsgs_gfx_block(unsigned char *src)
{
	int i, j;
	unsigned char *dec = (unsigned char*)malloc(0x800000);

	for (i = 0; i < 0x800000; i++)
	{
		j = BITSWAP24(i, 23, 10, 9, 22, 19, 18, 20, 21, 17, 16, 15, 14, 13, 12, 11, 8, 7, 6, 5, 4, 3, 2, 1, 0);

		dec[j] = src[i];
	}

	memcpy (src, dec, 0x800000);

	free (dec);
}

void pgm_decode_kovqhsgs_tile_data(unsigned char *source)
{
	int i, j;
	unsigned short *src = (unsigned short*)source;
	unsigned short *dst = (unsigned short*)malloc(0x800000);

	for (i = 0; i < 0x800000 / 2; i++)
	{
		j = BITSWAP24(i, 23, 22, 9, 8, 21, 18, 0, 1, 2, 3, 16, 15, 14, 13, 12, 11, 10, 19, 20, 17, 7, 6, 5, 4);

		dst[j] = BITSWAP16(src[i], 1, 14, 8, 7, 0, 15, 6, 9, 13, 2, 5, 10, 12, 3, 4, 11);
	}

	memcpy (src, dst, 0x800000);

	free (dst);
}

static void pgm_decode_kovqhsgs_samples()
{
	int i;
	for (i = 0; i < 0x400000; i+=2) {
		ICSSNDROM[i + 0x400001] = ICSSNDROM[i + 0xc00001];
	}
}

static void pgm_decode_kovqhsgs_program()
{
	int i, j;
	unsigned short *src = (unsigned short*)PGM68KROM;
	unsigned short *dst = (unsigned short*)malloc(0x400000);

	for (i = 0; i < 0x400000 / 2; i++)
	{
		j = BITSWAP24(i, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 6, 7, 5, 4, 3, 2, 1, 0);

		dst[j] = BITSWAP16(src[i], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 4, 5, 3, 2, 1, 0);
	}

	memcpy (src, dst, 0x400000);

	free (dst);
}

void pgm_decrypt_kovqhsgs()
{
	pgm_decode_kovqhsgs_program();

	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x000000);
	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x800000);

	// sprite colors are decoded in pgm_run

	pgm_decode_kovqhsgs_samples();
}

static void pgm_decode_kovlsqh2_program()
{
	int i, j;
	unsigned short *src = (unsigned short*)PGM68KROM;
	unsigned short *dst = (unsigned short*)malloc(0x400000);

	for (i = 0; i < 0x400000 / 2; i++)
	{
		j = BITSWAP24(i, 23, 22, 21, 20, 19, 16, 15, 14, 13, 12, 11, 10, 9, 8, 0, 1, 2, 3, 4, 5, 6, 18, 17, 7);

		dst[j] = src[i];
	}

	memcpy (src, dst, 0x400000);

	free (dst);
}

void pgm_decrypt_kovlsqh2()
{
	pgm_decode_kovlsqh2_program();

	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x000000);
	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x800000);

	// sprite colors are decoded in pgm_run.cpp

	pgm_decode_kovqhsgs_samples();
}

static void pgm_decode_kovassg_program()
{
	int i, j;
	unsigned short *src = (unsigned short *)PGM68KROM;
	unsigned short *dst = (unsigned short *)malloc(0x400000);

	for (i = 0; i < 0x400000/2; i++)
	{
		j = (i & ~0xffff) | (BITSWAP16(i, 15, 14, 13, 12,  11, 10, 7, 3,  1, 9, 4, 8,  6, 0, 2, 5) ^ 0x019c);

		dst[i] = BITSWAP16(src[j], 13, 9, 10, 11, 2, 0, 12 ,5, 4, 1, 14, 8, 15, 6, 3, 7) ^ 0x9d05;
	}

	memcpy (src, dst, 0x400000);

	free (src);
}

void pgm_decrypt_kovassg()
{
	pgm_decode_kovassg_program();

	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x000000);
	pgm_decode_kovqhsgs_gfx_block(PGMSPRMaskROM + 0x800000);

	// sprite colors are decoded in pgm_run.cpp

	pgm_decode_kovqhsgs_samples();
}
