//-----------------------------------------------------------------------------
// File: XbOnline.h
//
// Desc: Shortcut macros and helper functions for the Xbox online samples.
//       Requires linking with XONLINE[D][S].LIB
//
// Hist: 10.11.01 - New for November XDK release
//
// Copyright (c) Microsoft Corporation. All rights reserved.
//-----------------------------------------------------------------------------
#ifndef XBONLINE_H
#define XBONLINE_H

#pragma warning( disable: 4786 ) // ignore STL identifier truncation
#include <xtl.h>
#include <vector>
#include "xonline.h"
#include "XBResource.h"
#include "XBFont.h"




//-----------------------------------------------------------------------------
// Constants
//-----------------------------------------------------------------------------
const DWORD MAX_ERROR_STR = 1024;




//-----------------------------------------------------------------------------
// Name: XBOnline_GetUserList() / XBOnline_UpdateUserList
// Desc: Extract accounts from hard disk and MUs
//-----------------------------------------------------------------------------
typedef std::vector< XONLINE_USER > XBUserList;
HRESULT XBOnline_GetUserList( XBUserList& UserList );




//-----------------------------------------------------------------------------
// Name: XBOnline_GetUserList()
// Desc: Extract all accounts from hard disk and MUs
//-----------------------------------------------------------------------------
HRESULT XBOnline_GetUserList( XONLINE_USER* pUserList, DWORD* pdwNumUsers );




//-----------------------------------------------------------------------------
// Name: XBNet_OnlineInit() / XBNet_OnlineCleanup()
// Desc: Performs all required initialization/shutdown for online and 
//       network play
//-----------------------------------------------------------------------------
HRESULT XBNet_OnlineInit( BYTE cfgFlags = 0, 
                          XONLINE_STARTUP_PARAMS* pxosp = NULL );
HRESULT XBNet_OnlineCleanup();




//-----------------------------------------------------------------------------
// Name: XBOnline_RenderOnlineNotificationIcon()
// Desc: Render an online notification icon
//-----------------------------------------------------------------------------
enum ONLINEICON
{
    ONLINEICON_NONE = 0,                   // No icon
    ONLINEICON_FRIEND_ONLINE,              // Friend is online
    ONLINEICON_FRIEND_RECEIVEDREQUEST,     // Friend request received
    ONLINEICON_FRIEND_SENTREQUEST,         // Friend request sent
    ONLINEICON_FRIEND_RECEIVEDINVITE,      // Game invite from friend
    ONLINEICON_FRIEND_SENTINVITE,          // Game invite to friend
    ONLINEICON_PLAYER_VOICE,               // Player has voice capability        
    ONLINEICON_PLAYER_MUTED,               // Player has been muted
    ONLINEICON_PLAYER_TVCHAT,              // TV Chat -  no voice comm.               
};

VOID XBOnline_RenderOnlineNotificationIcon( CXBFont* pOnlineIconsFont, 
                                            FLOAT fLeft, FLOAT fTop, 
                                            ONLINEICON Icon );



//-----------------------------------------------------------------------------
// Name: XBOnline_GetOnlineHResultString()
// Desc: Display and online error HRESULT
//-----------------------------------------------------------------------------
const WCHAR* XBOnline_GetOnlineHResultString( HRESULT hr );




//-----------------------------------------------------------------------------
// Name: class CXBOnlineUI
// Desc: UI class for standard online UI
//-----------------------------------------------------------------------------
class CXBOnlineUI
{    
public:
    // number of icons and offset multiplier in index
    static const DWORD    ONLINEICON_OFFSET_MULTIPLIER = 20UL;
    static const DWORD    ONLINEICON_NUM_ICONS = 8;

    static const D3DCOLOR COLOR_HIGHLIGHT = 0xffffff00; // Yellow
    static const D3DCOLOR COLOR_GREEN     = 0xff00ff00; // Green
    static const D3DCOLOR COLOR_NORMAL    = 0xffffffff; // White

private:
    // Disabled
    CXBOnlineUI( const CXBOnlineUI& );

    mutable CXBFont     m_Font;                      // Game font
    mutable CXBFont     m_OnlineIconsFont;
    
    WCHAR               m_strError[ MAX_ERROR_STR ]; // Generic err
    WCHAR*              m_strHeader;                 // Header string

public:

    CXBOnlineUI();

    HRESULT Initialize();

    // Accessors
    VOID SetErrorStr( const WCHAR*, va_list );

    VOID SetHeader( WCHAR* strHeader ) { m_strHeader = strHeader; }

    // UI functions
    VOID RenderCreateAccount( BOOL bHasMachineAccount ) const;
    VOID RenderSelectAccount( DWORD, const XBUserList& ) const;
    VOID RenderLoggingOn( const XONLINE_USER *pUsers = NULL ) const;
    VOID RenderError( BOOL bBootToDash = FALSE ) const;
    VOID RenderHeader() const;
    VOID RenderOnlineNotificationIcon( FLOAT fLeft, FLOAT fTop, ONLINEICON Icon ) const;
};

#endif // XBONLINE_H
