{*******************************************************}
{                                                       }
{           MiTeC File Explorer Component               }
{                                                       }
{           version 1.3 for Delphi 5,6                  }
{                                                       }
{       Copyright  1999,2002 Michal Mutl               }
{                                                       }
{*******************************************************}

unit MFileExplorer;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, ComCtrls, CommCtrl, FileCtrl, ShellAPI,
  Menus, ImgList;

const
  cAbout = 'MiTeC File Explorer 1.3 - Copyright  1999,2002 MichaL MutL';

type
  TDiskSign = string[2];

  TMediaType = (dtUnknown, dtNotExists, dtRemovable, dtFixed, dtRemote, dtCDROM, dtRAMDisk);

  TFileFlag = (fsCaseIsPreserved, fsCaseSensitive, fsUnicodeStoredOnDisk,
               fsPersistentAcls, fsFileCompression, fsVolumeIsCompressed,
               fsLongFileNames,
               // following flags are valid only for Windows2000
               fsEncryptedFileSystemSupport, fsObjectIDsSupport, fsReparsePointsSupport,
               fsSparseFilesSupport, fsDiskQuotasSupport);
  TFileFlags = set of TFileFlag;

  TDiskInfo = record
    Sign: TDiskSign;
    MediaType: TMediaType;
    FileFlags: TFileFlags;
    SectorsPerCluster,
    BytesPerSector,
    FreeClusters,
    TotalClusters,
    Serial: DWORD;
    Capacity,
    FreeSpace: Int64;
    VolumeLabel,
    SerialNumber,
    FileSystem: string;
  end;

  TObjectType = (otFile, otDirectory, otDisk);
  TObjectTypes = set of TObjectType;

  TFileAttr = (ftReadOnly, ftHidden, ftSystem, ftArchive, ftNormal);
  TFileType = Set of TFileAttr;

  TMFileExplorer = class(TCustomListView)
  private
    FDirectory: string;
    FDirectorySize: integer;
    FFileName: string;
    FFileType: TFileType;
    FMask: string;
    FContextMenu: Boolean;
    FSortColumn: integer;
    FSortForward: boolean;
    LImageList, SImageList: TImageList;
    FContextAction: Boolean;
    FSelectedFiles :tstringlist;
    fabout: string;
    function GetSelectedNum: Integer;
    function GetSelectedSize: Integer;
    procedure Createimages;
    procedure CompareFiles(Sender: TObject; Item1,Item2: TListItem; Data: Integer; var Compare: Integer);
    procedure ColumnClick(Sender: TObject; Column: TListColumn);
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure DoMouseDown(var Message: TWMMouse; Button: TMouseButton;
      Shift: TShiftState);
    function GetSelectedFilenames: string;
    procedure SetAbout(const Value: string);
    function GetObjectTypes: TObjectTypes;
    function GetDiskCap: Int64;
    function GetDiskFree: Int64;
  protected
    function AddFile(FileMask: string; Attr: DWORD): Boolean;
    function GetFileName: string;
    function GetDirectory: string;
    procedure AddDrives;
    procedure Click; override;
    procedure DblClick; override;
    procedure Keydown(var Key: Word; Shift: TShiftState); override;
    procedure SetFileName(NewFile: string);
    procedure SetDirectory(NewDir: string);
    procedure SetFileType(NewFileType: TFileType);
    procedure SetMask(const NewMasks: string);
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    function GetDiskInfo(Value: TDiskSign): TDiskInfo;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CreateWnd; override;
    procedure OneLevelUp;
    procedure UpdateFileList;
    property FileName: string read GetFileName write SetFileName;
    property DiskCapacity: Int64 read GetDiskCap;
    property DiskFree: Int64 read GetDiskFree;
    function GetMediaTypeStr(MT: TMediaType): string;
  published
    property About :string read fabout write SetAbout;
    property Directory: string read GetDirectory write SetDirectory;
    property FileType: TFileType read FFileType write SetFileType;
    property Mask: string read FMask write SetMask;
    property ContextMenu: Boolean read FContextMenu write FContextMenu;
    property ContextAction: Boolean read FContextAction write FContextAction;
    property SelectedCount: Integer read GetSelectedNum;
    property SelectedSize: Integer read GetSelectedSize;
    property SelectedFilenames :string read GetSelectedFilenames;
    property SelectedTypes :TObjectTypes read GetObjectTypes;
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Dragmode;
    property DragCursor;
    property FlatScrollBars;
    property Font;
    property HideSelection;
//    property HotTrack;
//    property HotTrackStyles;
    property IconOptions;
    property MultiSelect;
    property ParentShowHint;
    property ReadOnly;
    property RowSelect;
    property ShowColumnHeaders;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property ViewStyle;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnColumnClick;
    property OnCompare;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnEdited;
    property OnEditing;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnInsert;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;


procedure Register;

implementation

uses
  ItemProp;

procedure Register;
begin
  RegisterComponents('MiTeC', [TMFileExplorer]);
end;

const
  DefaultMask = '*.*';

  FILE_SUPPORTS_ENCRYPTION = 32;
  FILE_SUPPORTS_OBJECT_IDS = 64;
  FILE_SUPPORTS_REPARSE_POINTS = 128;
  FILE_SUPPORTS_SPARSE_FILES = 256;
  FILE_VOLUME_QUOTAS = 512;

var
  drives: set of 0..25;
  CurPath: string;


function GetMediaPresent(Value: TDiskSign) :Boolean;
var
  ErrorMode: Word;
  bufRoot :pchar;
  a,b,c,d :dword;
begin
  bufRoot:=stralloc(255);
  strpcopy(bufRoot,Value+'\');
  ErrorMode:=SetErrorMode(SEM_FailCriticalErrors);
  try
    try
      result:=GetDiskFreeSpace(bufRoot,a,b,c,d);
    except
      result:=False;
    end;
  finally
    strdispose(bufroot);
    SetErrorMode(ErrorMode);
  end;
end;

constructor TMFileExplorer.Create(AOwner: TComponent);
begin
 inherited Create(AOwner);
 fabout:=cabout;
 fselectedfiles:=tstringlist.create;
 Createimages;
 ShortDateFormat:='mm/dd/yyyy';
 LongTimeFormat:='hh:nn';
 FMask:=DefaultMask;
 FSortForward:=True;
 FSortColumn:=0;
 OnCompare:=CompareFiles;
 OnColumnClick:=ColumnClick;
 if csdesigning in componentstate then
   fdirectory:='c:\';
end;

destructor TMFileExplorer.Destroy;
begin
 LImageList.Free;
 SImageList.Free;
 fSelectedFiles.Free;
 inherited Destroy;
end;

function TMFileExplorer.GetSelectedNum: Integer;
begin
  Result:=SelCount;
  if Result=0 then
    Result:=Items.Count;
end;

function TMFileExplorer.GetSelectedSize: Integer;
var
  i, FSize: UInt;
  FName: string;
  FInfo: TWin32FindData;
  hFindFile: THandle;
begin
  Result:=0;
  FSize:=0;
  hFindFile:=0;
  if SelCount=0 then
    exit;
  for i:=0 to Items.Count-1 do begin
    if Items[i].selected then begin
      FName:=ExtractFileName(Items[i].SubItems[4]+#0);
      hFindFile:=FindFirstFile(pChar(FName),FInfo);
      if hFindFile<>INVALID_HANDLE_VALUE then
        FSize:=FSize+((FInfo.nFileSizeHigh*MAXDWORD)+FInfo.nFileSizeLow);
    end;
  end;
  Windows.FindClose(hFindFile);
  Result:=FSize;
end;

function TMFileExplorer.GetDirectory: string;
begin
  Result:=FDirectory;
end;

procedure TMFileExplorer.SetDirectory(NewDir: string);
begin
  if AnsiCompareText(NewDir,FDirectory)=0 then
    exit;
  if (UpperCase(NewDir)='DRIVES') then begin
    FDirectory:=NewDir;
    UpdateFileList;
  end else begin
    if not DirectoryExists(NewDir) then
      exit;
    NewDir:=IncludeTrailingBackslash(NewDir);
    SetCurrentDir(NewDir);
    FDirectory:=NewDir;
    UpdateFileList;
  end;
end;

procedure TMFileExplorer.SetMask(const NewMasks: string);
begin
  if FMask<>NewMasks then begin
    FMask:=NewMasks;
    UpdateFileList;
  end;
end;

function TMFileExplorer.GetFileName: string;
begin
  Result:=FFileName;
end;

procedure TMFileExplorer.SetFileName(NewFile: string);
begin
  if FFileName=NewFile then
    exit;
  FFileName:=NewFile;
end;

procedure TMFileExplorer.SetFileType(NewFileType: TFileType);
begin
  if NewFileType<>FFileType then begin
    FFileType:=NewFileType;
    UpdateFileList;
  end;
end;

procedure TMFileExplorer.Createimages;
var
  SysImageList: uint;
  SFI: TSHFileInfo;
begin
  Largeimages:=TImageList.Create(self);
  SysImageList:=SHGetFileInfo('',0,SFI,SizeOf(TSHFileInfo),SHGFI_SYSICONINDEX or SHGFI_LARGEICON);
  if SysImageList<>0 then begin
    Largeimages.Handle:=SysImageList;
    Largeimages.ShareImages:=TRUE;
  end;
  Smallimages:=TImageList.Create(Self);
  SysImageList:=SHGetFileInfo('',0,SFI,SizeOf(TSHFileInfo),SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
  if SysImageList<>0 then begin
    Smallimages.Handle:=SysImageList;
    Smallimages.ShareImages:=TRUE;
  end;
end;

procedure TMFileExplorer.CreateWnd;
begin
  inherited;
  Font.Size:=8;
  Font.Name:='MS Sans Serif';
  if Columns.Count=0 then begin
    with Columns.Add do begin
      Caption:='Name';
      Width:=140;
    end;
    with Columns.Add do	begin
      Caption:='Size';
      Width:=70;
      Alignment:=taRightJustify;
    end;
    with Columns.Add do	begin
      Caption:='Type';
      Width:=90;
    end;
    with Columns.Add do	begin
      Caption:='Modified';
      Width:=100;
    end;
    with Columns.Add do	begin
      Caption:='Attributes';
      width:=60;
    end;
    UpdateFileList;
  end;
end;

procedure TMFileExplorer.ColumnClick(Sender: TObject;
  Column: TListColumn);
var
  required_column: integer;
begin
  required_column:=Column.Index;
  if required_column=FSortColumn then
    FSortForward:=not FSortForward
  else begin
    FSortColumn:=required_column;
    FSortForward:=True;
  end;
  SortType:=stData;
  SortType:=stNone;
end;

procedure TMFileExplorer.CompareFiles(Sender: TObject; Item1,
	Item2: TListItem; Data: Integer; var Compare: Integer);
var
  s1,s2,Caption1, Caption2: string;
  size1, size2: Double;
  result: integer;
begin
  Result := 0;
  if (Item1.SubItems[0]=' ') and (Item2.SubItems[0]<>' ') then
    Result:=-1
  else
    if (Item1.SubItems[0]<>' ') and (Item2.SubItems[0]=' ') then
      Result:=1
    else begin
      if FSortColumn=0 then begin
        Caption1:=AnsiUpperCase(Item1.Caption);
        Caption2:=AnsiUpperCase(Item2.Caption);
	if Caption1>Caption2 then
           Result:=1
        else
          if Caption1<Caption2 then
            Result:=-1;
        end else
          if FSortColumn in [1,3] then	begin
            s1:=StringReplace(Item1.SubItems.Strings[FSortColumn-1],ThousandSeparator,'',[rfReplaceAll]);
            s2:=StringReplace(Item2.SubItems.Strings[FSortColumn-1],ThousandSeparator,'',[rfReplaceAll]);
            try
              size1:=StrToFloat(s1);
            except
              try
                size1:=StrToDateTime(s1);
              except
                size1:=0;
              end;
            end;
            try
              size2:=StrToFloat(s2);
            except
              try
                size2:=StrToDateTime(s2);
              except
                size2:=0;
              end;
            end;
	    if Size1>Size2 then
              Result:=1
          else
            if Size1<Size2 then
               Result:=-1;
        end else
           result:=CompareText(Item1.SubItems.Strings[FSortColumn-1],
	     		Item2.SubItems.Strings[FSortColumn-1]);
      end;
  if FSortForward then
    Compare:=-result
  else
    Compare:=result;
end;


procedure TMFileExplorer.Keydown(var Key: Word; Shift: TShiftState);
begin
 if ((Shift=[ssCtrl]) and (key=vk_up)) or (key=vk_back) then
   OneLevelUp
 else
   if (key=vk_return) and assigned(selected) then
     DblClick;
 inherited;
end;

procedure TMFileExplorer.UpdateFileList;
var
  oldCur: TCursor;
  MaskPtr: PChar;
  AttrIndex: TFileAttr;
  Ptr: PChar;
  DirAttr, FileAttr: DWORD;
  FName: string;
const
  dwFileAttr: array[TFileAttr] of DWord = (FILE_ATTRIBUTE_READONLY,
		FILE_ATTRIBUTE_HIDDEN, FILE_ATTRIBUTE_SYSTEM,
		FILE_ATTRIBUTE_ARCHIVE, FILE_ATTRIBUTE_NORMAL);
begin
  Items.beginUpdate;
  Items.Clear;
  OldCur:=Screen.Cursor;
  Screen.Cursor:=crHourGlass;
  FDirectorySize:=0;
  try
    if UpperCase(FDirectory)='DRIVES' then begin
      Column[1].Caption:='Type';
      Column[1].Width:=100;
      Column[1].Alignment:=taLeftJustify;
      Column[2].Caption:='Disk Size';
      Column[2].Width:=100;
      Column[2].Alignment:=taRightJustify;
      Column[3].Caption:='Free Space';
      Column[3].Width:=100;
      Column[3].Alignment:=taRightJustify;
      AddDrives;
    end else begin
      Column[1].Caption:='Size';
      Column[1].Width:=70;
      Column[1].Alignment:=taRightJustify;
      Column[2].Caption:='Type';
      Column[2].Width:=150;
      Column[2].Alignment:=taLeftJustify;
      Column[3].Caption:='Modified';
      Column[3].Width:=110;
      Column[3].Alignment:=taLeftJustify;
      FileAttr:=0;
      for AttrIndex:=ftReadOnly to ftNormal do
    	if AttrIndex in FileType then
	  FileAttr:=FileAttr or dwFileAttr[AttrIndex];
      DirAttr:=FileAttr or FILE_ATTRIBUTE_DIRECTORY;
      CurPath:=IncludeTrailingBackslash(FDirectory);
      FName:=CurPath+ '*.*';
      AddFile(FName, DirAttr);
      MaskPtr:=PChar(FMask);
      while MaskPtr<>nil do begin
    	Ptr:=StrScan(MaskPtr,';');
	if Ptr<>nil then
	  Ptr^:=#0;
	  AddFile((CurPath+StrPas(MaskPtr)),FileAttr);
	  if Ptr<>nil then begin
    	    Ptr^:=';';
	    inc(Ptr);
	  end;
	  MaskPtr:=Ptr;
       end;
     end;
  finally
    FSortForward:=True;
    if not (UpperCase(FDirectory)='DRIVES') then 
      ColumnClick(Self,Columns[0]);
  end;
  Items.EndUpdate;
  Screen.Cursor:=oldCur;
  Application.ProcessMessages;
end;

procedure TMFileExplorer.AddDrives;
var
  shInfo: TSHFileInfo;
  NewItem: TListItem;
  i: Integer;
  Drv: string;
  DI: TDiskInfo;
begin
  Integer(Drives):=GetLogicalDrives;
  for i:=0 to 25 do
    if (i in Drives) then begin
      Drv:=Char(i+Ord('A'))+':';
      NewItem:=Items.Add;
      try
        SHGetFileInfo(PChar(Drv+'\'),0,shInfo,SizeOf(shInfo),SHGFI_SYSICONINDEX or SHGFI_DISPLAYNAME or SHGFI_TYPENAME);
    	if SmallImages<>nil then
     	  NewItem.ImageIndex:=shInfo.Iicon;
        NewItem.Caption:=StrPas(shInfo.szDisplayName);
        DI:=GetDiskInfo(TDiskSign(Drv));
        NewItem.SubItems.Add(GetMediaTypeStr(DI.MediaType));
        NewItem.SubItems.Add(FormatFloat('###,###,##0',DI.Capacity));
        NewItem.SubItems.Add(FormatFloat('###,###,##0',DI.FreeSpace));
        NewItem.SubItems.Add('');
        NewItem.SubItems.Add(Drv+'\');
        NewItem.SubItems.Add('drv');
      except
        Items.Delete(NewItem.Index);
      end;
   end;
end;

function TMFileExplorer.AddFile(FileMask: string; Attr: DWord): Boolean;
var
  ShInfo: TSHFileInfo;
  attributes: string;
  FDate, FName, FileName: string;
  FSize: Integer;
  FI: TSearchRec;

  function AttrStr(Attr: integer): string;
  begin
    Result:='';
    if (FILE_ATTRIBUTE_DIRECTORY and Attr)>0 then
      Result:=Result+'';
    if (FILE_ATTRIBUTE_ARCHIVE and Attr)>0 then
      Result:=Result+'A';
    if (FILE_ATTRIBUTE_READONLY and Attr)>0 then
      Result:=Result+'R';
    if (FILE_ATTRIBUTE_HIDDEN and Attr)>0 then
      Result:=Result+'H';
    if (FILE_ATTRIBUTE_SYSTEM and Attr)>0 then
      Result:=Result+'S';
  end;

begin
  Result := False;
  if not SetCurrentDir(FDirectory) then
    exit;
  if FindFirst(FileMask,faAnyFile,FI)=0 then
    try
      repeat
          if ((Attr and FILE_ATTRIBUTE_DIRECTORY)=(FI.Attr and FILE_ATTRIBUTE_DIRECTORY)) and
             ((Attr and FILE_ATTRIBUTE_READONLY)>=(FI.Attr and FILE_ATTRIBUTE_READONLY)) and
             ((Attr and FILE_ATTRIBUTE_HIDDEN)>=(FI.Attr and FILE_ATTRIBUTE_HIDDEN)) and
             ((Attr and FILE_ATTRIBUTE_SYSTEM)>=(FI.Attr and FILE_ATTRIBUTE_SYSTEM)) then begin
       	    CurPath:=IncludeTrailingBackslash(FDirectory);
	    FName:=FI.Name;
	    FileName:=IncludeTrailingBackslash(FDirectory)+FName;
	    if (FName='.') or (FName='..') then
              continue;
            SHGetFileInfo(PChar(FileName),0,ShInfo,SizeOf(ShInfo),SHGFI_TYPENAME or SHGFI_SYSICONINDEX or SHGFI_DISPLAYNAME);
            FSize:=FI.Size;
            FDate:=DateTimeToStr(FileDateToDateTime(FI.Time));
	    Attributes:=AttrStr(FI.Attr);
	    with Items.Add do begin
              Caption:=FName;
	      if SmallImages<>nil then
		ImageIndex:=ShInfo.iIcon;
              if (FI.Attr and FILE_ATTRIBUTE_DIRECTORY)=FILE_ATTRIBUTE_DIRECTORY then
        	SubItems.Add(' ')
              else
                SubItems.Add(Trim(IntToStr(FSize)));
              SubItems.Add((ShInfo.szTypeName));
	      SubItems.Add(FDate);
	      SubItems.Add(attributes);
	      SubItems.Add(FileName);
	      if (FI.Attr and FILE_ATTRIBUTE_DIRECTORY)=FILE_ATTRIBUTE_DIRECTORY then
                SubItems.Add('dir')
	      else
                SubItems.Add('file');
	    end;
            FDirectorySize:=FDirectorySize+FSize;
            Result:=True;
          end;
     until FindNext(FI)<>0;
  finally
    FindClose(FI);
  end;
end;


procedure TMFileExplorer.OneLevelUp;
var
  NewDir: string;
begin
  if UpperCase(Directory)='DRIVES' then
    exit;
  FDirectory:=IncludeTrailingBackslash(FDirectory);
  if (FDirectory[Length(FDirectory)-1]=':') then
    SetDirectory('Drives')
  else begin
    FDirectory:=Copy(FDirectory,1,Length(FDirectory)-1);
    NewDir:=ExtractFilePath(FDirectory);
    SetDirectory(NewDir);
  end;
end;

procedure TMFileExplorer.Click;
begin
  if Selected<>nil then
    SetFileName(Selected.SubItems[4]);
  inherited;
end;

procedure TMFileExplorer.DblClick;
var
  sDir: string;
begin
  inherited;
  if Selected=nil then
    exit;
  if (Selected.SubItems[5]='dir') or (Selected.SubItems[5]='drv') then begin
    sDir:=Selected.SubItems[4];
    sDir:=IncludeTrailingBackslash(sDir);
    SetDirectory(sDir);
  end else
    if Selected.SubItems[5]='file' then
      if fcontextaction then
        PerformDefaultAction(filename,pointer(handle));
end;

procedure TMFileExplorer.WMRButtonDown(var Message: TWMRButtonDown);
begin
  DoMouseDown(Message, mbRight, []);
end;

procedure TMFileExplorer.DoMouseDown(var Message: TWMMouse;
  Button: TMouseButton; Shift: TShiftState);
begin
  if not (csNoStdEvents in ControlStyle) then
    with Message do
      MouseDown(Button, KeysToShiftState(Keys) + Shift, XPos, YPos);
end;

procedure TMFileExplorer.MouseDown(Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  p :tpoint;
  n :tlistitem;
  b: Boolean;
begin
  inherited;
  if (button=mbright) and fcontextmenu then begin
    n:=getitemat(x,y);
    if assigned(n) then begin
      if pos(n.subitems[4],selectedfilenames)=0 then
        selected:=nil;
      selected:=n;
      click;
      getcursorpos(p);
      b:=False;
      if selectedcount>1 then begin
        getselectedfilenames;
        DisplayContextMenu(directory,fselectedfiles,p,False,b);
      end else
        DisplayContextMenu(filename,Pointer(handle),p,False,b);
    end;
  end;
end;

function TMFileExplorer.GetSelectedFilenames: string;
var
 i: integer;
begin
  Result := '';
  fselectedfiles.clear;
  if SelCount=0 then
    exit;
  for i:=0 to Items.Count-1 do
    if Items[i].selected then begin
      fselectedfiles.add(extractfilename(Items[i].SubItems[4]));
      result:=result+Items[i].SubItems[4]+#13#10;
    end;
  Result:=copy(result,1,length(result)-1);
end;

procedure TMFileExplorer.SetAbout(const Value: string);
begin
end;

function TMFileExplorer.GetObjectTypes: TObjectTypes;
var
 i: integer;
begin
  Result:=[];
  fselectedfiles.clear;
  if SelCount=0 then
    exit;
  for i:=0 to Items.Count-1 do
    if Items[i].selected then begin
      if Items[i].SubItems[5]='file' then
        result:=result+[otfile]
      else
        if Items[i].SubItems[5]='dir' then
          result:=result+[otdirectory]
        else
          if Items[i].SubItems[5]='drv' then
            result:=result+[otdisk];
    end;
end;

function TMFileExplorer.GetDiskCap: Int64;
begin
  Result:=GetDiskInfo(Copy(FDirectory,1,1)+':').Capacity;
end;

function TMFileExplorer.GetDiskFree: Int64;
begin
  Result:=GetDiskInfo(Copy(FDirectory,1,1)+':').FreeSpace;
end;

function TMFileExplorer.GetDiskInfo(Value: TDiskSign): TDiskInfo;
var
  BPS,TC,FC,SPC :integer;
  T,F :TLargeInteger;
  TF :PLargeInteger;
  bufRoot, bufVolumeLabel, bufFileSystem :pchar;
  MCL,Size,Flags :DWORD;
  s :string;
begin
  with Result do begin
    Sign:=Value;
    Size:=255;
    bufRoot:=AllocMem(Size);
    strpcopy(bufRoot,Value+'\');
    case GetDriveType(bufRoot) of
      DRIVE_UNKNOWN     :MediaType:=dtUnknown;
      DRIVE_NO_ROOT_DIR :MediaType:=dtNotExists;
      DRIVE_REMOVABLE   :MediaType:=dtRemovable;
      DRIVE_FIXED       :MediaType:=dtFixed;
      DRIVE_REMOTE      :MediaType:=dtRemote;
      DRIVE_CDROM       :MediaType:=dtCDROM;
      DRIVE_RAMDISK     :MediaType:=dtRAMDisk;
    end;
    FileFlags:=[];
    if GetMediaPresent(Value) then begin
      GetDiskFreeSpace(bufRoot,SectorsPerCluster,BytesPerSector,FreeClusters,TotalClusters);
      try
        new(TF);
        SysUtils.GetDiskFreeSpaceEx(bufRoot,F,T,TF);
        Capacity:=T;
        FreeSpace:=F;
        dispose(TF);
      except
        BPS:=BytesPerSector;
        TC:=TotalClusters;
        FC:=FreeClusters;
        SPC:=SectorsPerCluster;
        Capacity:=TC*SPC*BPS;
        FreeSpace:=FC*SPC*BPS;
      end;
      bufVolumeLabel:=AllocMem(Size);
      bufFileSystem:=AllocMem(Size);
      if GetVolumeInformation(bufRoot,bufVolumeLabel,Size,@Serial,MCL,Flags,bufFileSystem,Size) then begin;
        VolumeLabel:=bufVolumeLabel;
        FileSystem:=bufFileSystem;
        s:=IntToHex(Serial,8);
        SerialNumber:=copy(s,1,4)+'-'+copy(s,5,4);
        FreeMem(bufVolumeLabel);
        FreeMem(bufFileSystem);
        FreeMem(bufRoot);
        if Flags and FS_CASE_SENSITIVE=FS_CASE_SENSITIVE then
          FileFlags:=FileFlags+[fsCaseSensitive];
        if Flags and FS_CASE_IS_PRESERVED=FS_CASE_IS_PRESERVED then
          FileFlags:=FileFlags+[fsCaseIsPreserved];
        if Flags and FS_UNICODE_STORED_ON_DISK=FS_UNICODE_STORED_ON_DISK then
          FileFlags:=FileFlags+[fsUnicodeStoredOnDisk];
        if Flags and FS_PERSISTENT_ACLS=FS_PERSISTENT_ACLS then
          FileFlags:=FileFlags+[fsPersistentAcls];
        if Flags and FS_VOL_IS_COMPRESSED=FS_VOL_IS_COMPRESSED then
          FileFlags:=FileFlags+[fsVolumeIsCompressed];
        if Flags and FS_FILE_COMPRESSION=FS_FILE_COMPRESSION then
          FileFlags:=FileFlags+[fsFileCompression];
        if MCL=255 then
          FileFlags:=FileFlags+[fsLongFileNames];
        if Flags and FILE_SUPPORTS_ENCRYPTION=FILE_SUPPORTS_ENCRYPTION then
          FileFlags:=FileFlags+[fsEncryptedFileSystemSupport];
        if Flags and FILE_SUPPORTS_OBJECT_IDS=FILE_SUPPORTS_OBJECT_IDS then
          FileFlags:=FileFlags+[fsObjectIDsSupport];
        if Flags and FILE_SUPPORTS_REPARSE_POINTS=FILE_SUPPORTS_REPARSE_POINTS then
          FileFlags:=FileFlags+[fsReparsePointsSupport];
        if Flags and FILE_SUPPORTS_SPARSE_FILES=FILE_SUPPORTS_SPARSE_FILES then
          FileFlags:=FileFlags+[fsSparseFilesSupport];
        if Flags and FILE_VOLUME_QUOTAS=FILE_VOLUME_QUOTAS then
          FileFlags:=FileFlags+[fsDiskQuotasSupport];
      end;
    end else begin
      SectorsPerCluster:=0;
      BytesPerSector:=0;
      FreeClusters:=0;
      TotalClusters:=0;
      Capacity:=0;
      FreeSpace:=0;
      VolumeLabel:='';
      SerialNumber:='';
      FileSystem:='';
      Serial:=0;
    end;
  end;
end;

function TMFileExplorer.GetMediaTypeStr(MT: TMediaType): string;
begin
  case MT of
    dtUnknown     :result:='<unknown>';
    dtNotExists   :result:='<not exists>';
    dtRemovable   :result:='Removable';
    dtFixed       :result:='Fixed';
    dtRemote      :result:='Remote';
    dtCDROM       :result:='CDROM';
    dtRAMDisk     :result:='RAM';
  end;
end;

end.

