#include "XTrace.h"
#include "XTraceClient.h"

#if ( USE_XTRACE == 1 || USE_XERROR == 1 )
#include <xtl.h>
#include <stdarg.h>
#include <stdio.h>
#include <xbsockaddr.h>
#include <XBNet.h>

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Static initialization
//
XTraceClient* XTraceClient::m_pInstance = NULL;

/////////////////////////////////////////////////////////////////////////////////////////////
//
extern "C" void _XDebugInit
( 
	const char* pszIpAddress, 
	const int iIpPort 
)
{
	// Call the singleton to process the trace request
	XTraceClient::Instance()->Connect( pszIpAddress, iIpPort );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
extern "C" void _XDebugCleanup( void )
{
	// Call the singleton to process the trace request
	XTraceClient::Cleanup();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Static Implementation of a debug message string that allows for printf style message 
//	formatting.
//
#if ( USE_XTRACE == 1 )
extern "C" void _XTrace
( 
	const char* pszFormat,		// printf style format string
	...							// Variable number of args per pszFormat
)
{
	va_list argList;

	// Set up the vargs
	va_start( argList, pszFormat );

	// Call the singleton to process the trace request
	XTraceClient::Instance()->TraceVA( pszFormat, argList );

	// Clean up the vargs
	va_end( argList );
}
#endif

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Static Implementation of a debug message string that allows for printf style message 
//	formatting.
//
#if ( USE_XERROR == 1 )
extern "C" void _XError
( 
	const char* pszFormat,		// printf style format string
	...							// Variable number of args per pszFormat
)
{
	va_list argList;

	// Set up the vargs
	va_start( argList, pszFormat );

	// Call the singleton to process the trace request
	XTraceClient::Instance()->TraceVA( pszFormat, argList );

	// Clean up the vargs
	va_end( argList );
}
#endif

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Singleton access method.
//
XTraceClient* XTraceClient::Instance( void )
{
	if ( !m_pInstance )
		m_pInstance = new XTraceClient();

	return m_pInstance;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Static cleanup method.
//
void XTraceClient::Cleanup( void )
{ 
	if ( m_pInstance )
		delete m_pInstance;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Constructor.  Protected due to singleton design pattern.  Use Instance()->Fn() to access
//	the singleton.
//
XTraceClient::XTraceClient()
	:
	m_Socket( INVALID_SOCKET ),
	m_bConnected( false ),
	m_iBufferLength( 1024 ),
	m_pszBuffer( NULL )
{
	m_pszBuffer = new char[ m_iBufferLength + 1 ];
	memset( m_pszBuffer, '\0', m_iBufferLength + 1 );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Destructor.  Doesn't need to be virtual as there are no virtual methods.
//
XTraceClient::~XTraceClient()
{
	Close();

	delete[] m_pszBuffer;
	m_pszBuffer = 0;

	m_pInstance = NULL;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Connect to the defined PC
//
void XTraceClient::Connect
( 
	const char* pszIpAddress, 
	const int iIpPort 
)
{
	XBNet_Init( XNET_STARTUP_BYPASS_SECURITY );
	CXBSockAddr	ipAddr( inet_addr( pszIpAddress ), iIpPort );

	if
	( 
		m_bConnected == false &&
		m_Socket.Open( CXBSocket::Type_TCP ) == TRUE &&
		m_Socket.Connect( ipAddr.GetPtr() ) != SOCKET_ERROR
	)
	{
		m_bConnected = true;
		Send( "XTraceClient: Connected to server\n" );
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Connect to the defined PC
//
void XTraceClient::Close( void )
{
	XBNet_Cleanup();

	if ( m_bConnected )
	{
		m_Socket.Close();
		m_bConnected = false;

		Send( "XTraceClient: Disconnecting from server\n" );
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Implementation of a debug message string that allows for printf style message formatting.
//
void XTraceClient::TraceVA
( 
	const char* pszFormat,		// printf style format string
	va_list argList				// Variable number of args per pszFormat
)
{
	XLock lock( m_CriticalSection );

	// Format the message
	vsprintf( m_pszBuffer, pszFormat, argList );

	// Send to the local debugger (if present)
	::OutputDebugString( m_pszBuffer );

	// Send to the remote message server
	if ( m_bConnected )
		Send( m_pszBuffer );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Send strings to a remote PC.
//
void XTraceClient::Send
( 
	const char* pszMessage		// Null terminated message string
)
{
	m_Socket.Send( pszMessage, strlen( pszMessage ) );
}
#endif