#include "stdafx.h"
#include "CSocket.h"
#include "CTrace.h"

#include <stdio.h>

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CSocket::InitWinsock( void )
{
	WSADATA wsa;
	WORD v;

	v = MAKEWORD(2,0); //version 2.0 des sockets windows

	return ( ::WSAStartup( v, &wsa ) == 0 );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

void CSocket::UnloadWinsock( void )
{
	::WSACleanup();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Constructs a synchronous socket
//

CSocket::CSocket
(
)
	:
	m_hSocket( INVALID_SOCKET )
{
	Create();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Protected Constructor that attaches to a valid socket handle.
//

CSocket::CSocket
(
	SOCKET hSocket
)
	:
	m_hSocket( hSocket )
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Destructs a synchronous socket
//

CSocket::~CSocket()
{
	// Always try to close, if we have already closed nothing will happen.
	Close();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	The socket function creates a socket that is bound to a specific service provider.
//

bool CSocket::Create( void )
{
	m_hSocket = ::socket( PF_INET, SOCK_STREAM, 0 );

	return ( m_hSocket != INVALID_SOCKET );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Associates a windows event handle with this socket.  See the documentation for
//		WSAEventSelect for a definition of supported events.
//

bool CSocket::EventSelect
( 
	DWORD dwNetworkEvents		// The specific events
)
{

	int iResult = ::WSAEventSelect( m_hSocket, m_Event, dwNetworkEvents );               

	if ( iResult == SOCKET_ERROR )
		LogLastError();

	return ( iResult != SOCKET_ERROR );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
bool CSocket::GetSignaledEvent
(
	DWORD& dwNetworkEvents
)
{
	WSANETWORKEVENTS networkEvents;    // tells us what events happened     

	int iResult = ::WSAEnumNetworkEvents( m_hSocket, m_Event, &networkEvents );

	if ( iResult == SOCKET_ERROR )
	{
		LogLastError();
		dwNetworkEvents = 0;
	}
	else
	{
		dwNetworkEvents = networkEvents.lNetworkEvents;
		m_Event.Reset();
	}

	return ( iResult != SOCKET_ERROR );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	The listen function places a socket a state where it is listening for an incoming
//	connection.
//

bool CSocket::Listen
(
	LPCTSTR pszHostAddress,		// Ip address or hostname 
	int iHostPort,				// The port on which to listen for clients
	int iBacklog				// The maximum length of the queue of pending connections. 
)
{
	if ( m_hSocket != INVALID_SOCKET )
	{
		// Make up address
		SOCKADDR_IN	sockAddr;

		if ( !InitSocketAddress( &sockAddr, pszHostAddress, iHostPort ) )
			return false;

		// Bind to the address and port
		int iResult = ::bind( m_hSocket, (SOCKADDR*)&sockAddr, sizeof( sockAddr ) );

		if ( iResult == 0 )
		{
//			SetIntOption( SOL_SOCKET, SO_DONTLINGER, 1 );
//			SetIntOption( SOL_SOCKET, SO_KEEPALIVE, 1 );

			// Tell the socket that it will be a listener
			iResult = ::listen( m_hSocket, iBacklog );

			if ( iResult == 0 )
				return true;
		}
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		The accept function blocks until an incoming connection attempt, and then returns
//		a client socket for the new connection.
//
bool CSocket::Accept
( 
	CSocket*& pClientSocket 
)
{
	SOCKET hClientSocket;
	pClientSocket = NULL;

	if ( m_hSocket != INVALID_SOCKET )
	{
		// Accept the client socket
		hClientSocket = ::accept( m_hSocket, NULL, NULL );

		// Now attach hClientSocket to an object and return it
		if ( hClientSocket == INVALID_SOCKET )
			LogLastError();
		else
			pClientSocket = new CSocket( hClientSocket );
	}

	return ( pClientSocket != NULL );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
// The connect function establishes a connection to a specifed socket.
//

bool CSocket::Connect
(
	LPCTSTR pszHostAddress,		// Ip address or hostname 
	int iHostPort				// Port on the host to connect to
)
{
	if ( m_hSocket == INVALID_SOCKET )
		Create();

	if ( m_hSocket != INVALID_SOCKET )
	{
		// Fill address machinery of sockets.
		SOCKADDR_IN sockAddr;

		if ( InitSocketAddress( &sockAddr, pszHostAddress, iHostPort ) )
		{
			// connects to peer
			int iResult = ::connect( m_hSocket, (SOCKADDR*)&sockAddr, sizeof( sockAddr ) );

			if ( iResult == SOCKET_ERROR )
				LogLastError();

			return ( iResult != SOCKET_ERROR );
		}
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	The send function sends data on a connected socket.
//

bool CSocket::Send
( 
	LPCSTR pBuffer,  
	int iBufferLength
)
{
	int iResult;
	int iRemaining = iBufferLength;

	if ( m_hSocket != INVALID_SOCKET )
	{
		while ( iRemaining > 0 )
		{
			iResult = ::send( m_hSocket, (LPSTR)pBuffer, iRemaining, 0 );

			if ( iResult == SOCKET_ERROR )
			{
				LogLastError();
				return false;
			}
			else if
			( 
				iResult == 0			||		// Graceful termination
				iResult < 0						// Send less than 0 bytes?
			)
				return false;

			pBuffer += iResult;
			iRemaining -= iResult;
		}
	}

	return ( iRemaining == 0 );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CSocket::Receive
(
	LPSTR pBuffer, 
	int iBufferLength,
	int& iBytesReceived
) 
{
	iBytesReceived = 0;

	int iResult = ::recv( m_hSocket, pBuffer, iBufferLength, 0 );

	if ( iResult > 0 )
		iBytesReceived += iResult;
	else if ( iResult == 0 )
	{
		// No bytes received, the socket close on us?
		return false; 
	}
	else if ( iResult == SOCKET_ERROR )
	{
		int iError = ::WSAGetLastError();

		switch ( iError )
		{
			case WSAEMSGSIZE:			// The message was too large to fit into the specified buffer and was truncated. 
				iBytesReceived += iBufferLength;
				break;

			default:
				LogLastError();
				return false;
		}
	}

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Close the underlying socket.
//

bool CSocket::Close( void )
{
	if ( m_hSocket != INVALID_SOCKET )
	{
		::closesocket( m_hSocket );
		m_hSocket = INVALID_SOCKET;
	}

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Shutdown a connection but keep resources .
//

bool CSocket::Shutdown( void )
{
	if ( m_hSocket != INVALID_SOCKET )
	{
		::shutdown( m_hSocket, 2 );
		return true;
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CSocket::InitSocketAddress
(
	SOCKADDR_IN* pSockAddr,		
	LPCTSTR pszHostAddress,		// Ip address or hostname 
	int iHostPort				// Port on the host to connect to
)
{
	LPSTR pszAscii = (LPSTR)pszHostAddress;

	// Fill address machinery of sockets.
	memset( pSockAddr, 0, sizeof( SOCKADDR_IN ) );

	pSockAddr->sin_family = AF_INET;
	pSockAddr->sin_port = ::htons( (u_short)iHostPort );

	if ( !pszAscii )
		pSockAddr->sin_addr.s_addr = INADDR_ANY;
	else
	{
		// Try ipaddress
		pSockAddr->sin_addr.s_addr = ::inet_addr( pszAscii );

		if ( pSockAddr->sin_addr.s_addr == INADDR_NONE )
		{
			LPHOSTENT lphost;
			lphost = ::gethostbyname( pszAscii );

			if ( lphost != NULL )
				pSockAddr->sin_addr.s_addr = ((LPIN_ADDR)lphost->h_addr)->s_addr;
			else
				return false;
		}
	}

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
bool CSocket::ConvertIpAddress
( 
	unsigned long nIpAddress,
	std::string& sIpAddress
)
{
	sockaddr_in	clientAddress;

	::memset( &clientAddress, 0, sizeof( sockaddr_in ) );

	clientAddress.sin_family = AF_INET;
	clientAddress.sin_addr.s_addr = nIpAddress;

	sIpAddress = ::inet_ntoa( clientAddress.sin_addr );
	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Returns the ip port of the currently connected client
//

unsigned short CSocket::GetPeerIpPort( void )
{
	sockaddr_in	clientAddress;
	int iResult, iLength = sizeof( sockaddr );

	::memset( &clientAddress, 0, iLength );

	// Get the socket name
	iResult = ::getpeername( m_hSocket, (SOCKADDR*)&clientAddress, &iLength ); 

	return clientAddress.sin_port;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Returns the ip address of the currently connected client
//

unsigned long CSocket::GetPeerIpAddress( void )
{
	sockaddr_in	clientAddress;
	int iResult, iLength = sizeof( sockaddr );

	::memset( &clientAddress, 0, iLength );

	// Get the socket name
	iResult = ::getpeername( m_hSocket, (SOCKADDR*)&clientAddress, &iLength ); 
	char* pszAddress = ::inet_ntoa( clientAddress.sin_addr );

	return ::inet_addr( pszAddress );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Returns the ip port of the currently connected client
//

unsigned short CSocket::GetLocalIpPort( void )
{
	sockaddr_in	clientAddress;
	int iResult, iLength = sizeof( sockaddr );

	::memset( &clientAddress, 0, iLength );

	// Get the socket name
	iResult = ::getsockname( m_hSocket, (SOCKADDR*)&clientAddress, &iLength ); 

	return clientAddress.sin_port;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Returns the ip address of the currently connected client
//

unsigned long CSocket::GetLocalIpAddress( void )
{
	sockaddr_in	clientAddress;
	int iResult, iLength = sizeof( sockaddr );

	::memset( &clientAddress, 0, iLength );

	// Get the socket name
	iResult = ::getsockname( m_hSocket, (SOCKADDR*)&clientAddress, &iLength ); 
	char* pszAddress = ::inet_ntoa( clientAddress.sin_addr );

	return ::inet_addr( pszAddress );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CSocket::SetOption
( 
	int iLevel, 
	int iOptName, 
	const char* pszOptVal, 
	int iOptLen 
)
{
	return ( SOCKET_ERROR == setsockopt( m_hSocket, iLevel, iOptName, pszOptVal, iOptLen ) );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

void CSocket::GetLastError( void )
{
	int iError = ::WSAGetLastError();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
void CSocket::LogLastError( void )
{
	int iError = ::WSAGetLastError();

	switch ( iError )
	{
		case WSAEMSGSIZE:			// The message was too large to fit into the specified buffer and was truncated. 
		case WSANOTINITIALISED:		// A successful WSAStartup must occur before using this function. 
		case WSAENETDOWN:			// The network subsystem has failed. 
		case WSAEFAULT:				// The buf parameter is not completely contained in a valid part of the user address space. 
		case WSAENOTCONN:			// The socket is not connected. 
		case WSAEINTR:				// The (blocking) call was canceled through WSACancelBlockingCall. 
		case WSAEINPROGRESS:		// A blocking Windows Sockets 1.1 call is in progress, or the service provider is still processing a callback function. 
		case WSAENETRESET:			// The connection has been broken due to the keep-alive activity detecting a failure while the operation was in progress. 
		case WSAENOTSOCK:			// The descriptor is not a socket. 
		case WSAEOPNOTSUPP:			// MSG_OOB was specified, but the socket is not stream-style such as type SOCK_STREAM, out-of-band data is not supported in the communication domain associated with this socket, or the socket is unidirectional and supports only send operations. 
		case WSAESHUTDOWN:			// The socket has been shut down; it is not possible to recv on a socket after shutdown has been invoked with how set to SD_RECEIVE or SD_BOTH. 
		case WSAEWOULDBLOCK:		// The socket is marked as nonblocking and the receive operation would block. 
		case WSAEINVAL:				// The socket has not been bound with bind, or an unknown flag was specified, or MSG_OOB was specified for a socket with SO_OOBINLINE enabled or (for byte stream sockets only) len was zero or negative. 
		case WSAECONNABORTED:		// The virtual circuit was terminated due to a time-out or other failure. The application should close the socket as it is no longer usable. 
		case WSAETIMEDOUT:			// The connection has been dropped because of a network failure or because the peer system failed to respond. 
		case WSAECONNRESET:			// The virtual circuit was reset by the remote side executing a "hard" or "abortive" close. The application should close the socket as it is no longer usable. On a UDP datagram socket this error would indicate that a previous send operation resulted in an ICMP "Port Unreachable" message. 
		default:
			TRACE( "Socket %u: WSAGetLastError %u\n", m_hSocket, iError );
	}
}
