#include "Stdafx.h"
#include "CThread.h"
#include "CResourcePool.h"
#include "CRunnable.h"
#include "CTrace.h"

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Constructs an instance of an CThread that encapsulates the functionality of a Win32
//		thread.  Internally either ::CreateThread or ONI_beginThreadEx is used to create the
//		thread based on the USE_ONI_BEGINTHREAD flag. 
//

CThread::CThread()
	:
	m_pRunnable( NULL )
{
	m_hThread = ::CreateThread
	(
		0,
		0,
		CThread::WinThreadEntry,
		this,
		CREATE_SUSPENDED,
		&m_dwThreadId
	);
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Destructs the thread object and closes the underlying thread handle.
//

CThread::~CThread()
{
	::CloseHandle( m_hThread ); 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Run an object in this instances thread.  Returns true if successfull, false if the
//		thread is currently busy.
//

bool CThread::Run
(
	CRunnable* pRunnable		// An object that implements the runnable interface.
)
{
	if ( !IsInUse() )
	{
		// Make sure we shut down completely before allowing reuse of this
		// thread (DEFECT 10247)
		CLock myLock( m_ThreadStateLock );

		// Remember who we belong to
		m_pRunnable = pRunnable;

		// We are created in the suspended state, do resume processing
		Resume(); 

		// Now signal that we should be running
		m_RunEvent.Signal();
		return true;
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Thread entry point when using ONI_BeginThreadEx.  This is a static function that 
//		calls Run on the associated	thread object.
//

void CThread::CThreadEntry
(
	void* pVoid			// Pointer to a valid instance of CThread
)
{
	// pVoid should really be a valid instance of CThread
	CThread* pThis = static_cast< CThread* >( pVoid );

	if ( pThis )
		pThis->Run();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Thread entry point used when using ::CreateThread().  This is a static function that 
//		calls Run() on the associated thread object.
//

DWORD CThread::WinThreadEntry
(
	void* pVoid			// Pointer to a valid instance of CThread
)
{
	// pVoid should really be a valid instance of CThread
	CThread* pThis = static_cast< CThread* >( pVoid );

	if ( pThis )
		return pThis->Run();

	return -1;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Run the runnable object associated with this thread.  When the runnable's run 
//		function exits we do one of the following:
//
//			1.  If the thread is associated with a resource pool the thread is returned 
//				to the pool and sleeps until reallocated.
//			2.  If the thread is not part of a resource pool the thread object is deleted
//				and the thread function is exited, thus ending the thread.
//
//		Note that in either case the current CRunnable object is not destroyed.
//

int CThread::Run( void )
{
	int iReturnValue = 0;

	while ( true )
	{
		TRACE( "CThread(%u): Running\n", m_dwThreadId );

		if ( m_pRunnable )
		{
			m_pRunnable->Run();
			m_pRunnable = NULL;
		}

		// We are done with this runnable object, now what?

		// If this thread came from a thread pool, give it back!
		if ( m_pResourcePool )
		{
			{
				// Make sure we shut down completely before allowing reuse of this
				// thread (DEFECT 10247)
				CLock myLock( m_ThreadStateLock );

				m_RunEvent.Clear();
				m_pResourcePool->Relinquish( this );
				
				TRACE( "CThread(%u): Suspending\n", m_dwThreadId );
			}

			// Now sleep until someone reactivates us
			m_RunEvent.Wait();

			TRACE( "CThread(%u): Resuming\n", m_dwThreadId );
		}
		else
		{
			TRACE( "CThread(%u): Stopping\n", m_dwThreadId );

			// Make sure that we delete ourselves
			delete this;

			// Don't run anymore
			break;
		}
	}

	return iReturnValue;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Increment the suspend count associated with this thread handle.  While the thread 
//		count is greater than zero the thread will not be scheduled to run.
//

DWORD CThread::Suspend( void )
{
	return ::SuspendThread( m_hThread ); 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Decrement the suspend count associated with this thread handle.  While the thread 
//		count is greater than zero the thread will not be scheduled to run.
//

DWORD CThread::Resume( void )
{
	return ::ResumeThread( m_hThread ); 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Implement the DetachFromPool functionality of CPoolable.  We need to stop the 
//		thread and exit.
//

void CThread::DetachFromPool( void )
{
	// Null our m_pResourcePool member
	CPoolable::DetachFromPool(); 

	// Now force the thread to run, which will cause Run() to exit and this CThread instance
	// to be deleted.
	m_RunEvent.Signal(); 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Implements the CancelWait functionality of OniWaitable.  Since we defer our 
//		processing to the CRunnable object we must tell it to stop running.  The correct
//		implementation of CRunnable::Stop is to cancel any wait states and return control
//		to the CThread object.
//

void CThread::CancelWait( void )
{ 
	if ( IsInUse() ) 
	{
		TRACE( "CThread(%u): Canceling wait\n", m_dwThreadId );
		m_pRunnable->Stop(); 
	}
}
