/*
  teetertorture.c
  
  Teeter Torture
  A clone of the 1982 Exidy arcade game of the same name.
  
  Slide a cannon left and right on a teeter toter, shooting at 'weight'
  monsters.  Don't let the teeter toter fall too far to either side, or
  TNT will explode.
  
  By Bill Kendrick
  bill@newbreedsoftware.com
  http://www.newbreedsoftware.com/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  (See COPYING.txt)
  
  August 9, 2005 - October 18, 2005

  *********************************
  *********************************
  **   Xbox Port by: Neobomb     **
  **  http://www.EmuXtras.net    **
  *********************************
  *********************************
*/
#include <xtl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "SDL.h"
#include "SDL_image.h"
#include "SDL_mixer.h"

#define boolean unsigned char
#define FALSE 0
#define TRUE 1

#define TEXT_CENTERED -1

#define FPS 60
#define MAX_CANNON_SPEED 12
#define CANNON_WEIGHT 50
#define MONSTER_WEIGHT 10

#define MAX_LEVEL 20

#define NUM_EXPLOSION_BITS 64

#define MAX_RECTS 512

#define MAX_VOLUME 128

#define CANNON_LIMIT (320 - images[IMG_WHEEL1]->w - images[IMG_CANNON]->w / 2)

enum {
  IMG_TITLE,
  IMG_CANNON,
  IMG_BULLET,
  IMG_WHEEL1,
  IMG_WHEEL2,
  IMG_WHEEL3,
  IMG_BADDIE1,
  IMG_PANEL,
  IMG_AXIS,
  IMG_TNT,
  IMG_FUSE,
  IMG_DETONATOR,
  IMG_PLUNGER,
  IMG_LETTERS,
  IMG_LETTERS_YELLOW,
  IMG_NUMBERS,
  IMG_NUMBERS_YELLOW,
  IMG_RESIZE,
  NUM_IMAGES
};

char * img_filenames[NUM_IMAGES] = {
  "title.png",
  "cannon.png",
  "bullet.png",
  "wheel1.png",
  "wheel2.png",
  "wheel3.png",
  "baddie1.png",
  "panel.png",
  "axis.png",
  "tnt.png",
  "fuse.png",
  "detonator.png",
  "plunger.png",
  "letters.png",
  "letters-yellow.png",
  "numbers.png",
  "numbers-yellow.png",
  "resizer.png"
};

enum{
	SHOOT,
	BOOM,
	POP,
	TICKTOCK,
	DEATH,
	ATTACH,
	NUM_SOUNDS
};

char * sound_filenames[NUM_SOUNDS] = {

	"D:\\sounds\\SHOOT.wav",
	"D:\\sounds\\BOOM.wav",
	"D:\\sounds\\POP.wav",
	"D:\\sounds\\TICKTOC.wav",
	"D:\\sounds\\DEATH.wav",
	"D:\\sounds\\ATTACH.wav"
};

enum {
  CMD_GAME,
  CMD_MENU,
  CMD_QUIT
};


Uint8 level_colors[MAX_LEVEL / 2][3] = {
  {0, 0, 0},
  {0, 64, 64},
  {0, 64, 128},
  {89, 67, 67},
  {90, 100, 90},
  {142, 11, 11},
  {85, 84, 6},
  {34, 33, 134},
  {84, 84, 21},
  {126, 22, 27}
};

typedef struct monster_s {
  boolean alive;
  boolean falling;
  int x, y;
  boolean attached;
} monster_t;

typedef struct explosion_s {
  int active;
  int expl_type;
  int x, y;
  int xm, ym;
} explosion_t;

enum {
  EXPL_MONSTER,
  EXPL_PLAYER
};

enum {
  SCREEN_TITLE,
  SCREEN_INSTRUCTIONS,
  SCREEN_HIGH_SCORE,
  NUM_SCREENS
};

////////////////////
//Our Xbox buttons//
////////////////////
enum
{
    JOY_BTTN_A = 0,
    JOY_BTTN_B,
    JOY_BTTN_X,
    JOY_BTTN_Y,
	JOY_BTTN_BLACK,
    JOY_BTTN_WHITE,
	JOY_BTTN_LTRIG,
    JOY_BTTN_RTRIG,
    JOY_BTTN_START,
    JOY_BTTN_BACK,
	JOY_BTTN_LSTICK,
    JOY_BTTN_RSTICK,
	JOY_BTTN_TOTAL
};

//////////////////////
//Our Xbox joystick //
//////////////////////
enum
{
    JOY_AXIS_LX = 0,
    JOY_AXIS_LY,
    JOY_AXIS_RX,
    JOY_AXIS_RY,
    JOY_AXIS_TOTAL
};

#define JOY_DEADZONE 3200

SDL_Joystick *xJoy;

/* Local function prototypes: */

boolean setup(int argc, char * argv[]);
void do_shutdown(void);
int title_loop(void);
boolean game_loop(void);
int teeter_height(int x);
void adjust_teeter(int position, int weight);
void init_monsters(boolean complete_reset);
void draw_text(int x, int y, char * str, boolean yellow);
void explode_player(int x);
void add_explosion(int x1, int y1, int x2, int y2, int expl_type);
int huge_explosion(void);
void clear_screen(Uint8 r, Uint8 g, Uint8 b);
void add_rect(SDL_Rect rect);
void draw_left_detonator(boolean draw);
void draw_right_detonator(boolean draw);
void erase_rect(SDL_Rect rect);
void draw_score(boolean draw);
void draw_cannon(boolean draw);
void draw_explosions(boolean draw);
void draw_monsters(boolean draw);
void draw_teetertotter(boolean draw);
void show_title(void);
void show_instructions(void);
void show_high_score(void);

//XBOX prototypes
void playsound(int snd);
void load_options(void);
void save_options(void);
void resize_screen();

/* Globals: */

SDL_Surface * screen;
SDL_Surface * images[NUM_IMAGES];
boolean use_sound, fullscreen;
int teeter;
monster_t monsters[18];
int player_exploding;
explosion_t explosion[NUM_EXPLOSION_BITS];
SDL_Rect rects[MAX_RECTS];
int num_rects;
Uint8 bkgd_r, bkgd_g, bkgd_b;
int level, score, lives;
int cannon_x;
int wheel_frame;
char high_score_names[6][4];
int high_score_scores[5];

//Xbox Globals

int axisL, axisLY, axisR, axisRY;
int xpos=0, ypos=0, xstretch=0, ystretch=0; 
Mix_Music *music = NULL, *introMusic = NULL;
Mix_Chunk *sounds[NUM_SOUNDS];
int VOLUME = 64;



/* --- Main function --- */

int main(int argc, char * argv[])
{
  boolean done;
  int cmd;
  
  if (setup(argc, argv))
    {
      do
	{
	  done = FALSE;
	  cmd = title_loop();
	  
	  if (cmd == CMD_GAME)
	    {
	      done = game_loop();
	    }
	  else if (cmd == CMD_QUIT)
	    {
	      done = TRUE;
	    }
	}
      while (!done);
    }
  
  do_shutdown();
  
  return(0);
}


boolean game_loop(void)
{
  boolean done, quit;
  Uint32 prev_time, cur_time;
  SDL_Event event;
  SDL_Rect dest;
  SDLKey key;
  boolean key_left, key_right;
  boolean want_shot;
  int i, j;
  int sx, sy, w;
  boolean bullet_on;
  int bullet_x, bullet_y;
  int cannon_xm;
  int want_cannon_x;
  boolean follow_want;
  int frame;
  int non_attached_monsters, attached_monsters;
  int cmd;
  
  int tempSound = 0;
  int oldTempSound = 0;

  int xHat;

  //In game music!! -Neobomb
  Mix_PlayMusic(music, -1);

  /* Unset key-held flags: */
  
  key_left = FALSE;
  key_right = FALSE;
  
  
  /* Reset game: */
  
  level = 1;
  score = 0;
  lives = 3;
  bullet_on = FALSE;
  bullet_x = 0;
  bullet_y = 0;
  cannon_x = 0;
  cannon_xm = 0; 
  follow_want = FALSE;
  want_cannon_x = 0;
  player_exploding = 0;
  teeter = 0;
  wheel_frame = 0;
  
  init_monsters(TRUE);
  
  for (i = 0; i < NUM_EXPLOSION_BITS; i++)
    explosion[i].active = 0;
  
  
  /* MAIN GAME LOOP! */
  
  done = FALSE;
  quit = FALSE;
  frame = 0;
 

  /* Clear screen */
 
  clear_screen(level_colors[(level - 1) / 2][0],
	       level_colors[(level - 1) / 2][1],
    	       level_colors[(level - 1) / 2][2]);


  do
    {
      frame++;
      prev_time = SDL_GetTicks();
      
      num_rects = 0;



      /* Erase everything: */

      draw_left_detonator(FALSE);
      draw_right_detonator(FALSE);
      draw_score(FALSE);
      
      if (player_exploding == 0)
      {
        if (bullet_on)
        {
          dest.x = 320 + bullet_x - (images[IMG_BULLET]->w / 2);
          dest.y = bullet_y;
          dest.w = images[IMG_BULLET]->w;
          dest.h = images[IMG_BULLET]->h;

	  erase_rect(dest);

          add_rect(dest);
	}

	draw_cannon(FALSE);
      }
      
      draw_explosions(FALSE);
      draw_monsters(FALSE);
      draw_teetertotter(FALSE);

      
      /* Handle pending events: */
      
      want_shot = FALSE;

      
      while (SDL_PollEvent(&event))
	{
	  if (event.type == SDL_QUIT)
	    {
	      /* App Quit! (e.g., window close button) - Quit! */
	      
	      done = TRUE;
	      quit = TRUE;
	    }
	  else if (event.type == SDL_JOYBUTTONDOWN)
	  {
		  if(event.jbutton.button == JOY_BTTN_B)
			  want_shot = TRUE;

		  if(event.jbutton.button == JOY_BTTN_BACK)	
		  {
			  if(high_score_scores[5] < score)
					high_score_scores[5] = score;

			  save_options();

			  done = TRUE;
		  }


		  if(event.jbutton.button == JOY_BTTN_RSTICK)
			  resize_screen();

		  if(event.jbutton.button == JOY_BTTN_RTRIG)
		  {
			  if(VOLUME < MAX_VOLUME)
				  VOLUME = VOLUME + 8;

			  Mix_VolumeMusic(VOLUME); //max 128

		  }
		  if(event.jbutton.button == JOY_BTTN_LTRIG)
		  {
			  if(VOLUME > 0)
				  VOLUME = VOLUME - 8;

			  Mix_VolumeMusic(VOLUME); //max 128

		  }

	  }
	  else if (event.type == SDL_JOYHATMOTION)
	  {
		  xHat = SDL_JoystickGetHat(xJoy, 0);

	      if (xHat == SDL_HAT_LEFT || xHat == SDL_HAT_LEFTUP ||
		      xHat == SDL_HAT_LEFTDOWN)
		  {
		  /* Left is being held */
		  
			key_left = TRUE;
			key_right = FALSE;
			follow_want = FALSE;
		  }
		  else
			key_left  = FALSE;
	  
	      if (xHat == SDL_HAT_RIGHT || xHat == SDL_HAT_RIGHTUP ||
		      xHat == SDL_HAT_RIGHTDOWN)
		  {
			 /* Right is being held */
		  
			key_right = TRUE;
			key_left = FALSE;
			follow_want = FALSE;
		  }
		  else
			key_right = FALSE;

	  }

	  else if (event.type == SDL_JOYAXISMOTION)
	  {
		  axisL = SDL_JoystickGetAxis(xJoy, JOY_AXIS_LX);

		  if(axisL > JOY_DEADZONE)
		  {
			key_right = TRUE;
			key_left = FALSE;
			follow_want = FALSE;
		  }
		  else
			key_right = FALSE;

		  if(axisL < -JOY_DEADZONE)
		  {
			key_left = TRUE;
			key_right = FALSE;
			follow_want = FALSE;
		  }
		  else
			key_left = FALSE;

	  }
	  else if (event.type == SDL_KEYDOWN)
	    {
	      /* Keypress! */
	      
	      key = event.key.keysym.sym;

	      if (key == SDLK_ESCAPE)
		{
		  /* ESCAPE - Go back to main menu! */
		  
		  done = TRUE;
		}
	      else if (key == SDLK_F4 && (event.key.keysym.mod & KMOD_ALT))
		{
		  /* ALT+F4 - Quit! */
		  
		  done = TRUE;
		  quit = TRUE;
		}
	      else if (key == SDLK_LEFT)
		{
		  /* Left is being held */
		  
		  key_left = TRUE;
		  key_right = FALSE;
		  follow_want = FALSE;
		}
	      else if (key == SDLK_RIGHT)
		{
		  /* Right is being held */
		  
		  key_right = TRUE;
		  key_left = FALSE;
		  follow_want = FALSE;
		}
	      else if (key == SDLK_SPACE)
		{
		  /* Space - Fire! */
		  
		  want_shot = TRUE;
		}
	    }
	  else if (event.type == SDL_KEYUP)
	    {
	      /* Key release! */
	      
	      key = event.key.keysym.sym;
	      
	      if (key == SDLK_LEFT)
		key_left = FALSE;
	      else if (key == SDLK_RIGHT)
		key_right = FALSE;
	    }
	  else if (event.type == SDL_MOUSEMOTION)
	    {
	      /* Mouse movement! */
	      
	      if (frame > 2)
		{
		  want_cannon_x = event.motion.x - 320;
		  follow_want = TRUE;
		}
	    }
	  else if (event.type == SDL_MOUSEBUTTONDOWN)
	    {
	      /* Mouse click: fire! */
	      
	      want_shot = TRUE;
	    }
	}
      
      
      if (player_exploding == 0)
	{
	  /* Fire a bullet? */
	  
	  if (want_shot && bullet_on == FALSE)
	    {

		  playsound(SHOOT);
	      bullet_on = TRUE;
	      bullet_x = cannon_x;
	      bullet_y = (teeter_height(cannon_x + 320) -
			  (images[IMG_WHEEL1]->h +
			   (images[IMG_CANNON]->h / 2)));
	    }
	  
	  
	  /* Move cannon: */
	  
	  cannon_x = cannon_x + cannon_xm;
	  if (cannon_x < -CANNON_LIMIT)
	    cannon_x = -CANNON_LIMIT;
	  if (cannon_x > CANNON_LIMIT)
	    cannon_x = CANNON_LIMIT;
	  
	  if (key_left)
	    cannon_xm--;
	  else if (key_right)
	    cannon_xm++;
	  else if (follow_want)
	    {
	      if (cannon_x > want_cannon_x)
		{
		  cannon_xm--;
		  
		  if (cannon_x + cannon_xm <= want_cannon_x)
		    {
		      cannon_x = want_cannon_x;
		      cannon_xm = 0;
		      follow_want = FALSE;
		    }
		}
	      else if (cannon_x < want_cannon_x)
		{
		  cannon_xm++;
		  
		  if (cannon_x + cannon_xm >= want_cannon_x)
		    {
		      cannon_x = want_cannon_x;
		      cannon_xm = 0;
		      follow_want = FALSE;
		    }
		}
	    }
	  
	  if (cannon_xm < -MAX_CANNON_SPEED)
	    cannon_xm = -MAX_CANNON_SPEED;
	  else if (cannon_xm > MAX_CANNON_SPEED)
	    cannon_xm = MAX_CANNON_SPEED;
	  
	  if ((frame % 2) == 0)
	    {
	      if (cannon_xm > 0)
		cannon_xm--;
	      else if (cannon_xm < 0)
		cannon_xm++;
	    }
	  
	  
	  /* Adjust position of teeter totter: */
	  
	  if (frame % (((MAX_LEVEL + 1) - level)) == 0)
	    {
	      /* (For cannon) */
	      
	      adjust_teeter(cannon_x, CANNON_WEIGHT);
	      
	      /* (For monsters) */
	      
	      for (i = 0; i < 18; i++)
		{
		  if (monsters[i].alive && monsters[i].attached)
		    adjust_teeter(monsters[i].x - 320 +
				  (images[IMG_BADDIE1]->w / 2),
				  MONSTER_WEIGHT);
		}
	    }
	  
	  //play ticking sound if it gets close! -Neobomb
	  if (teeter_height(0) > (480 - images[IMG_DETONATOR]->h) - 20 ||
	      teeter_height(639) > (480 - images[IMG_DETONATOR]->h) - 20 )
	    {
				 
				 if (!Mix_Playing(4))
			        Mix_PlayChannel(4, sounds[TICKTOCK], 0);
								
		}

	  /* Blow up TNT if teeter totter goes too far! */
	  
	  if (teeter_height(0) > 480 - images[IMG_DETONATOR]->h ||
	      teeter_height(639) > 480 - images[IMG_DETONATOR]->h)
	    {
	      cmd = huge_explosion();
	      
	      if (cmd == CMD_MENU)
		done = TRUE;
	      else if (cmd == CMD_QUIT)
		{
		  done = TRUE;
		  quit = TRUE;
		}
	      
	      bullet_on = FALSE;
	      bullet_x = 0;
	      bullet_y = 0;
	      cannon_x = 0;
	      cannon_xm = 0; 
	      follow_want = FALSE;
	      want_cannon_x = 0;
	      player_exploding = 0;
	      teeter = 0;
	      key_left = FALSE;
	      key_right = FALSE;
	      
	      init_monsters(TRUE);
	    }
	  
	  
	  /* Move bullet: */
	  
	  if (bullet_on)
	    {
	      bullet_y = bullet_y - 12;
	      
	      if (bullet_y <= 0)
		bullet_on = FALSE;
	    }
	  
      
	  /* Make a random monster move now and then: */
	  
	    if (frame % (MAX_LEVEL + 1 - level) == 0)
	    {
	      i = rand() % 18;
	      
	      if (monsters[i].alive == TRUE &&
		  monsters[i].attached == FALSE &&	      
		  monsters[i].falling == FALSE)
		{
		  monsters[i].falling = TRUE;
		}
	    }
	  
	  
	  /* Deal with monsters: */
      
	  non_attached_monsters = 0;
	  attached_monsters = 0;
	  
	  for (i = 0; i < 18; i++)
	    {
	      if (monsters[i].alive == TRUE)
		{
		  if (monsters[i].attached == FALSE)
		    {
		      /* Count the non-attached monsters: */
		      
		      non_attached_monsters++;
		      
		      
		      /* Move the falling ones: */
		      
		      if (monsters[i].falling == TRUE)
			{
			  monsters[i].y = monsters[i].y + ((level / 3) + 1);
			  
			  if (monsters[i].y >= teeter_height(monsters[i].x))
			    {
			      /* Stick to the teeter totter! */
			      
				  //playsound(ATTACH);
			      monsters[i].falling = FALSE;
			      monsters[i].attached = TRUE;
			    }
			}
		      
		      
		      /* Bullets kill monsters: */
		      
		      sx = bullet_x + 320 - (images[IMG_BULLET]->w / 2);
		      
		      if (bullet_on &&
			  sx + images[IMG_BULLET]->w >= monsters[i].x &&
			  sx <= monsters[i].x + images[IMG_BADDIE1]->w &&
			  bullet_y + images[IMG_BULLET]->h >= monsters[i].y &&
			  bullet_y <= monsters[i].y + images[IMG_BADDIE1]->h)
			{
			  bullet_on = FALSE;
			  monsters[i].alive = FALSE;
			  playsound(POP);
			  
			  for (j = 0; j < 10; j++)
			    {
			      add_explosion(monsters[i].x,
					    monsters[i].y,
					    (monsters[i].x +
					     images[IMG_BADDIE1]->w),
					    (monsters[i].y +
					     images[IMG_BADDIE1]->h),
					    EXPL_MONSTER);
			    }
			  
			  score = score + level;
			}
		      
		      
		      /* Monsters kill player: */
		      
		      sx = cannon_x + 320 - (images[IMG_CANNON]->w / 2);
		      sy = teeter_height(sx) - images[IMG_WHEEL1]->h;
		      
		      sx = sx - images[IMG_WHEEL1]->w;
		      w = (images[IMG_WHEEL1]->w * 2) + images[IMG_CANNON]->w;
			    
		      if (sx + w >= monsters[i].x &&
			  sx <= monsters[i].x + images[IMG_BADDIE1]->w &&
			  sy + images[IMG_WHEEL1]->h >= monsters[i].y &&
			  sy <= monsters[i].y + images[IMG_BADDIE1]->h)
			{
			  playsound(DEATH);
			  explode_player(cannon_x);
			}
		    }
		  else
		    {
			  
		      attached_monsters++;
		    }
		}
	    }
	  
	  
	  /* Advance to next level: */
	  
	  if (non_attached_monsters == 0 || attached_monsters == 18)
	    {
	      if (level < MAX_LEVEL)
				level++;
  
	      clear_screen(level_colors[(level - 1) / 2][0],
	      		   level_colors[(level - 1) / 2][1],
    	       		   level_colors[(level - 1) / 2][2]);

	      init_monsters((attached_monsters == 18));
	    }
	}
      
      
      /* Handle player explosion: */
      
      if (player_exploding)
	{
	  player_exploding--;
	  
	  if (player_exploding == 0)
	    init_monsters(FALSE);
	}
      
      
      /* Handle explosion bits: */
      
      for (i = 0; i < NUM_EXPLOSION_BITS; i++)
	{
	  if (explosion[i].active)
	    {
	      explosion[i].active--;
	      
	      explosion[i].x += explosion[i].xm;
	      explosion[i].y += explosion[i].ym;
	      
	      if ((frame % 3) == 0)
		explosion[i].ym++;
	      
	      if (explosion[i].y >= teeter_height(explosion[i].x))
		{
		  explosion[i].y = teeter_height(explosion[i].x);
		  explosion[i].ym = -explosion[i].ym;
		}
	    }
	}
   

      /* Score and level: */
     
      draw_score(TRUE);

      
      /* Draw everything: */
     
      draw_left_detonator(TRUE);
      draw_right_detonator(TRUE);
      
      
      if (player_exploding == 0)
	{
	  /* Draw bullet: */
	  
	  if (bullet_on)
	    {
	      dest.x = 320 + bullet_x - (images[IMG_BULLET]->w / 2);
	      dest.y = bullet_y;
	      dest.w = images[IMG_BULLET]->w;
	      dest.h = images[IMG_BULLET]->h;
	      
	      SDL_BlitSurface(images[IMG_BULLET], NULL, screen, &dest);

	      add_rect(dest);
	    }
	  
	 
	  draw_cannon(TRUE);
	}

      
      /* Explosion bits: */
     
      draw_explosions(TRUE);
      
      
      /* Teeter totter: */
     
      draw_teetertotter(TRUE);
      
      
      /* Monsters: */
    
      draw_monsters(TRUE);
      
      
      /* Show it all! */
      
      SDL_UpdateRects(screen, num_rects, rects);
      
      
      /* Throttle FPS: */
      
      cur_time = SDL_GetTicks();
      
      if (cur_time < prev_time + (1000 / FPS))
	SDL_Delay((prev_time + (1000 / FPS)) - cur_time);
    }
  while (!done);
  
  return(quit);
}


int title_loop(void)
{
  SDL_Event event;
  SDLKey key;
  int cmd, done;
  int xHat, xButton; 
  Uint32 prev_time, cur_time;
  int ticks, info_screen;
  //title music! -Neobomb
 
  
  

  done = FALSE;
  cmd = -1;
  ticks = 0;
  info_screen = SCREEN_TITLE;
 
  show_title();
  
  Mix_PlayMusic(introMusic, -1);
  do
  {
    prev_time = SDL_GetTicks();

    while (SDL_PollEvent(&event))
    {
      if (event.type == SDL_QUIT)
      {
	/* App quit event - Quit! */

	done = TRUE;
	cmd = CMD_QUIT;
      }
	  else if (event.type == SDL_JOYBUTTONDOWN)
	  {
		  xButton = event.jbutton.button;

		  if((xButton || !xButton) && (xButton != JOY_BTTN_BACK &&
			 xButton != JOY_BTTN_RSTICK && xButton != JOY_BTTN_RTRIG
			 && xButton != JOY_BTTN_LTRIG))
		  {
			done = TRUE;
			cmd = CMD_GAME;
		  }
		  if(xButton == JOY_BTTN_BACK)
		  {
			done = TRUE;
			cmd = CMD_QUIT;

		  }
		  if(xButton == JOY_BTTN_RSTICK)
		  {
			  resize_screen();
			  show_title();
		  }
		  if(xButton == JOY_BTTN_RTRIG)
		  {
			  if(VOLUME < MAX_VOLUME)
				  VOLUME = VOLUME + 8;

			  Mix_VolumeMusic(VOLUME); //max 128

		  }
		  if(xButton == JOY_BTTN_LTRIG)
		  {
			  if(VOLUME > 0)
				  VOLUME = VOLUME - 8;

			  Mix_VolumeMusic(VOLUME); //max 128

		  }
	  }
	  else if (event.type == SDL_JOYHATMOTION)
	  {
		  xHat = SDL_JoystickGetHat(xJoy, 0);

		  if (xHat == SDL_HAT_RIGHT)
		  {
			printf("Right\n");
			ticks = (FPS * 5) - 1;
		  }
		  else if (xHat == SDL_HAT_LEFT)
		  {
			printf("Left\n");
			info_screen = info_screen - 2;
		  if (info_screen < 0);
			info_screen = info_screen + NUM_SCREENS;

		  ticks = (FPS * 5) - 1;
	}
	  }
      else if (event.type == SDL_KEYDOWN)
      {
	key = event.key.keysym.sym;

	if (key == SDLK_ESCAPE ||
	    (key == SDLK_F4 && (event.key.keysym.mod & KMOD_ALT)))
	{
	  /* ESCAPE or ALT+F4 - Quit! */
	  
	  done = TRUE;
	  cmd = CMD_QUIT;
	}
	else if (key == SDLK_SPACE || key == SDLK_RETURN)
	{
	  /* SPACE or ENTER - Start the game! */
	
	  done = TRUE;
	  cmd = CMD_GAME;
	}
	else if (key == SDLK_RIGHT)
	{
	  printf("Right\n");
	  ticks = (FPS * 5) - 1;
	}
	else if (key == SDLK_LEFT)
	{
	  printf("Left\n");
	  info_screen = info_screen - 2;
	  if (info_screen < 0);
	    info_screen = info_screen + NUM_SCREENS;

	  ticks = (FPS * 5) - 1;
	}
      }
    }


    /* Every once in a while, show a different info/attract screen: */

    ticks++;

    if (ticks == FPS * 5)
    {
      /* Draw the next screen: */

      ticks = 0;

      info_screen = (info_screen + 1) % NUM_SCREENS;

      if (info_screen == SCREEN_TITLE)
	show_title();
      else if (info_screen == SCREEN_INSTRUCTIONS)
	show_instructions();
      else if (info_screen == SCREEN_HIGH_SCORE)
	show_high_score();
    }


    /* Throttle FPS: */
      
    cur_time = SDL_GetTicks();
      
    if (cur_time < prev_time + (1000 / FPS))
      SDL_Delay((prev_time + (1000 / FPS)) - cur_time);
  }
  while (!done);
  
  return(cmd);
}


void show_title(void)
{
  int ch;
  SDL_Rect rect;

  ch = images[IMG_LETTERS]->h;

  SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 0xFF, 0xFF, 0xFF));

  rect.x = (screen->w - images[IMG_TITLE]->w) / 2;
  rect.y = 0;
  SDL_BlitSurface(images[IMG_TITLE], NULL, screen, &rect);
  
  draw_text(TEXT_CENTERED, screen->h - (ch * 16), "BASED ON THE 1982", FALSE);
  draw_text(TEXT_CENTERED, screen->h - (ch * 14), "PROTOTYPE BY EXIDY", FALSE);
  
  draw_text(TEXT_CENTERED, screen->h - (ch * 10), "REIMPLEMENTED", FALSE);
  draw_text(TEXT_CENTERED, screen->h - (ch * 8), "BY BILL KENDRICK", FALSE);
  draw_text(TEXT_CENTERED, screen->h - (ch * 6), "NEW BREED SOFTWARE 2005", FALSE);
  draw_text(TEXT_CENTERED, screen->h - (ch * 4), "XBOX PORT BY NEOBOMB 2011", FALSE);
  

  SDL_Flip(screen);
}


void show_instructions(void)
{
  int ch;
  
  SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 0, 0, 0));

  ch = images[IMG_LETTERS]->h;

  draw_text(TEXT_CENTERED, ch *  4, "TEETER TORTURE", FALSE);
  draw_text(TEXT_CENTERED, ch *  5, "INSTRUCTIONS", FALSE);

  draw_text(TEXT_CENTERED, ch *  8, "BALANCE THE TEETER TOTTER", TRUE);
  draw_text(TEXT_CENTERED, ch * 10, "BY MOVING THE CANNON", TRUE);
  draw_text(TEXT_CENTERED, ch * 12, "OR BY CATCHING THE WEIGHTS", TRUE);

  draw_text(TEXT_CENTERED, ch * 16, "BONUS FOR WEIGHTS CAUGHT", TRUE);
  draw_text(TEXT_CENTERED, ch * 20, "WEIGHTS DESTROY CANNON", TRUE);
  draw_text(TEXT_CENTERED, ch * 24, "CANNON BALLS", TRUE);
  draw_text(TEXT_CENTERED, ch * 26, "DESTROY WEIGHTS", TRUE);

  
  SDL_Flip(screen);
}

char * score_labels[6] = {
  "INCREDIBLE",
  "FABULOUS",
  "GREAT",
  "SUPER",
  "GOOD"
};


void show_high_score(void)
{
  int cw, ch;
  int i, y;
  char str[10];
  
  SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 0, 0, 0));

  cw = images[IMG_LETTERS]->w / 26;
  ch = images[IMG_LETTERS]->h;

  draw_text(TEXT_CENTERED, ch * 1, "TEETER TORTURE", FALSE);
  draw_text(TEXT_CENTERED, ch * 2, "HALL OF FAME", FALSE);

  for (i = 0; i < 5; i++)
  {
    y = (((ch * i) * 150) / 100) + (ch * 4);

    draw_text(cw * 4, y, high_score_names[i], TRUE);

    _snprintf(str, sizeof(str), "%5d", high_score_scores[i]);
    draw_text(cw * 8, y, str, TRUE);

    draw_text(cw * 14, y, score_labels[i], TRUE);
  }

  draw_text(TEXT_CENTERED, ch * 16, "-------------------", TRUE);
  draw_text(TEXT_CENTERED, ch * 18, "- YOUR HIGH SCORE -", TRUE);
  draw_text(TEXT_CENTERED, ch * 16, "-------------------", TRUE);

  if(high_score_scores[5] > score)
	  _snprintf(str, sizeof(str), "%5d", high_score_scores[5]);
  else
	  _snprintf(str, sizeof(str), "%5d", score);

  draw_text(TEXT_CENTERED, ch * 22, str, TRUE);
  draw_text(TEXT_CENTERED, ch * 16, "-------------------", TRUE);
  
  SDL_Flip(screen);
}


boolean setup(int argc, char * argv[])
{
  int i;
  char filename[FILENAME_MAX];
//  char * home;
  SDL_Surface * tmp_surf;
  int depth;

  use_sound = TRUE;
  fullscreen = FALSE;
  
  for (i = 0; i < NUM_IMAGES; i++)
    images[i] = NULL;
  
  if (SDL_Init(SDL_INIT_VIDEO) < 0)
    {
      fprintf(stderr, "Error: Couldn't initialize video:\n%s\n",
	      SDL_GetError());
      return(FALSE);
    }
  
  if (use_sound)
    {
      if (SDL_Init(SDL_INIT_AUDIO) < 0)
	{
	  fprintf(stderr, "Wanring: Couldn't initialize audio:\n%s\n",
		  SDL_GetError());
	  use_sound = FALSE;
	}
    }
  
  if (use_sound)
    {
      if (Mix_OpenAudio(44100, AUDIO_S16SYS, 2, 1024) < 0)
	{
	  fprintf(stderr, "Warning: Could not set up 44100 Hz audio:\n%s\n",
		  SDL_GetError());
	  use_sound = FALSE;
	}

	
      for (i = 0; i < NUM_SOUNDS; i++)
	{
	  sounds[i] = Mix_LoadWAV(sound_filenames[i]);
          if (sounds[i] == NULL)
            {
              fprintf(stderr,
                      "\nError: I could not load the sound file:\n"
                      "%s\n"
                      "The Simple DirectMedia error that occured was:\n"
                      "%s\n\n", sound_filenames[i], SDL_GetError());
              //exit(1);
            }
	}
    }
  
  //SDL_WM_SetCaption("Teeter Torture", "Teeter Torture");
  
	  
	  depth = SDL_VideoModeOK(640, 480, 16, SDL_FULLSCREEN);
      screen = SDL_SetVideoMode(640, 480, depth, SDL_SWSURFACE | SDL_ANYFORMAT);
      
      if (screen == NULL)
	{
	  fprintf(stderr, "Warning: Could not switch to full screen:\n%s\n",
		  SDL_GetError());
	  fullscreen = FALSE;
	}
	  //Don't want a cursor on Xbox
      SDL_ShowCursor(0);
  /*if (fullscreen == FALSE)
    {
      screen = SDL_SetVideoMode(640, 480, 16, SDL_SWSURFACE | SDL_ANYFORMAT);
      
      if (screen == NULL)
	{
	  fprintf(stderr, "Warning: Could not open display:\n%s\n",
		  SDL_GetError());
	  return(FALSE);
	}
    }*/
  
  for (i = 0; i < NUM_IMAGES; i++)
    {
		_snprintf(filename, sizeof(filename), "D:\\images\\%s", img_filenames[i]);
      tmp_surf = IMG_Load(filename);
      
      if (tmp_surf == NULL)
	{
	  fprintf(stderr, "Error: Couldn't load image %s:\n%s\n",
		  filename, SDL_GetError());
	  return(FALSE);
	}
      
      images[i] = SDL_DisplayFormatAlpha(tmp_surf);
      SDL_FreeSurface(tmp_surf);
      
      if (images[i] == NULL)
	{
	  fprintf(stderr, "Error: Couldn't convert image %s:\n%s\n",
		  filename, SDL_GetError());
	  return(FALSE);
	}
    }


  /* Set default values: */

  strcpy(high_score_names[0], "SPT");
  high_score_scores[0] = 53940;

  strcpy(high_score_names[1], "LWH");
  high_score_scores[1] = 49720;

  strcpy(high_score_names[2], "DJS");
  high_score_scores[2] = 47340;

  strcpy(high_score_names[3], "VIC");
  high_score_scores[3] = 44130;

  strcpy(high_score_names[4], "MRK");
  high_score_scores[4] = 37840;

  strcpy(high_score_names[5], "YOU");
  high_score_scores[5] = 0;



  /* Try to load prefs and data: */

  //home = "D:\\";//getenv("HOME");

   // _snprintf(filename, sizeof(filename), "%s/teetertorturerc.dat", home);
	//filename = "D:\\teetertorture.dat";

    load_options();
	
	
    //free(home);
  
    //load music - Neobomb
    music = Mix_LoadMUS("D:\\music\\GAME_MUSIC.ogg");
    introMusic = Mix_LoadMUS("D:\\music\\INTRO_MUSIC.ogg");
    Mix_VolumeMusic(VOLUME); //max 128

	if(music == NULL || introMusic == NULL) 
	{
		fprintf(stderr, "Error: Could not load music\n");
		return(FALSE);
	}
	//Xbox Joystick - Neobomb
	if (SDL_Init(SDL_INIT_JOYSTICK) < 0)
    {
      return(FALSE);
    }
	else
	{
		xJoy = SDL_JoystickOpen(0);
		SDL_JoystickEventState(SDL_ENABLE);
	}


  
  
  return(TRUE);
}


void do_shutdown(void)
{
  int i;
//  char * home;
//  char filename[FILENAME_MAX];

  //my new save system -Neobomb
  save_options();

  //home = "D:\\"; //getenv("HOME");

    //_snprintf(filename, sizeof(filename), "%s/.teetertorturerc", home);

  //  if (fi != NULL)
  //  {
  //    for (i = 0; i < 6; i++)
  //    {
		//fprintf(fi, "name%d=%s\n", i + 1, high_score_names[i]);
		//fprintf(fi, "high%d=%d\n", i + 1, high_score_scores[i]);

		//if(i == 5)
		//{
		//	fprintf(fi, "name%d=%s\n", i + 1, high_score_names[i]);
		//	fprintf(fi, "high%d=%d\n", i + 1, score);

		//}

  //    }
  //    fclose(fi);
  //  }
  // // free(home);
  

  
  for (i = 0; i < NUM_IMAGES; i++)
    {
      if (images[i] != NULL)
	{
	  SDL_FreeSurface(images[i]);
	  images[i] = NULL;
	}
    }

  //Reboot to Xbox dash
  XLaunchNewImage( NULL, NULL);
}


int teeter_height(int x)
{
  int y;
  
  y = ((x - 320) * teeter) / (images[IMG_PANEL]->w) / 80;
  
  return(y + (480 - images[IMG_TNT]->h -
	      images[IMG_AXIS]->h - images[IMG_PANEL]->h));
}


void adjust_teeter(int position, int weight)
{
  teeter = teeter + ((position * weight) / 320);
}


void init_monsters(boolean complete_reset)
{
  int i;

  for (i = 0; i < 18; i++)
    {
      monsters[i].alive = TRUE;
      monsters[i].falling = FALSE;
      monsters[i].y = images[IMG_LETTERS]->h;
      
      if (complete_reset == TRUE)
	{
	  monsters[i].attached = FALSE;
	}
      
      if (i < 9)
	monsters[i].x = 54 + i * 24;
      else
	monsters[i].x = 640 - 54 - (i - 8) * 24;      
    }
}


void draw_text(int x, int y, char * str, boolean yellow)
{
  int i;
  int cw;
  SDL_Rect src, dest;
  
  cw = images[IMG_LETTERS]->w / 26;

  if (x == TEXT_CENTERED)
  {
    x = (screen->w - (cw * strlen(str))) / 2;
  }
  
  for (i = 0; i < (int)strlen(str); i++)
    {
      dest.x = x;
      dest.y = y;

      if (str[i] >= 'A' && str[i] <= 'Z')
	{
	  src.x = (str[i] - 'A') * cw;
	  src.y = 0;
	  src.w = cw;
	  src.h = images[IMG_LETTERS]->h;

	  if (yellow)
	    SDL_BlitSurface (images[IMG_LETTERS_YELLOW], &src, screen, &dest);
	  else
	    SDL_BlitSurface (images[IMG_LETTERS], &src, screen, &dest);
	}
      else if (str[i] >= '0' && str[i] <= '9')
	{
	  src.x = (str[i] - '0') * cw;
	  src.y = 0;
	  src.w = cw;
	  src.h = images[IMG_NUMBERS]->h;

	  if (yellow)
	    SDL_BlitSurface (images[IMG_NUMBERS_YELLOW], &src, screen, &dest);
	  else
	    SDL_BlitSurface (images[IMG_NUMBERS], &src, screen, &dest);
	}
      
      x = x + cw;
    }
}


void explode_player(int x)
{
  int i;
  int left_x, top_y, w, h;

  

  left_x = (x + 320 - (images[IMG_CANNON]->w / 2) - images[IMG_WHEEL1]->w);
  top_y = teeter_height(x + 320);
  
  w = images[IMG_CANNON]->w + (images[IMG_WHEEL1]->w * 2);
  h = images[IMG_WHEEL1]->h + (images[IMG_CANNON]->h / 2);
  
  for (i = 0; i < NUM_EXPLOSION_BITS; i++)
    {
      /* (Force at least half of them to come on, even if they're
	 being used for monsters!) */
      
      if ((i % 2) == 0)
	explosion[i].active = 0;
      
      add_explosion(left_x, top_y,
		    left_x + w, top_y + h,
		    EXPL_PLAYER);
    }
  
  player_exploding = 128;
}


void add_explosion(int x1, int y1, int x2, int y2, int expl_type)
{
  int i, found;
  
  found = -1;
  for (i = 0; i < NUM_EXPLOSION_BITS && found == -1; i++)
    {
      if (explosion[i].active == 0)
	found = i;
    }
  
  if (found != -1)
    {
      if (expl_type == EXPL_MONSTER)
	explosion[found].active = 32;
      else if (expl_type == EXPL_PLAYER)
	explosion[found].active = 128;

      explosion[found].expl_type = expl_type;
      
      explosion[found].x = x1 + (rand() % (x2 - x1));
      explosion[found].y = y1 + (rand() % (y2 - y1));
      
      explosion[found].xm = (rand() % 9) - 4;
      explosion[found].ym = (rand() % 15) - 10;
    }
}


int huge_explosion(void)
{
  int i;
  int scale;
  int x, y, xx, yy;
  int sx, sy;
  SDL_Surface * tmp_surf;
  SDL_Rect src, dest;
  Uint32 prev_time, cur_time;
  Uint32 amask;
  Uint8 grey;
  int cmd;
  SDL_Event event;
  SDLKey key;

  SDL_Flip(screen);

  amask = ~(screen->format->Rmask |
	    screen->format->Gmask |
	    screen->format->Bmask);
  
  tmp_surf = SDL_CreateRGBSurface(SDL_SWSURFACE, 640, 480,
				  16,
				  0,
				  0,
				  0,
				  0);
  
  
  if (tmp_surf == NULL)
    {
      /* Can't create surface for special effect!  Do a cheap one, instead: */

      SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 255, 255, 64));
      SDL_Flip(screen);
      SDL_Delay(1000);
      
      return(CMD_GAME);
    }
  
  
  cmd = CMD_GAME;
  playsound(BOOM);
  for (i = 0; i < 32 && cmd == CMD_GAME; i++)
    {
      prev_time = SDL_GetTicks();
      
      
      while (SDL_PollEvent(&event))
	{
	  if (event.type == SDL_QUIT)
	    cmd = CMD_QUIT;
	  else if (event.type == SDL_KEYDOWN)
	    {
	      key = event.key.keysym.sym;
	      
	      if (key == SDLK_ESCAPE)
		cmd = CMD_MENU;
	      else if (key == SDLK_F4 && (event.key.keysym.mod & KMOD_ALT))
		cmd = CMD_QUIT;
	    }
	}

      for (y = 0; y < 384; y = y + 6)
	{
	  for (x = -256; x < 256; x = x + 6)
	    {
	      src.x = (320 + x);
	      src.y = (480 - y);
	      src.w = 6;
	      src.h = 6;
	      
	      scale = 10 + ((64 - i) / 20);
	      
	      sx = 320 + ((x * scale) / 8);
	      sy = 480 - ((y * scale) / 8);

	      for (yy = 0; yy <= 6; yy = yy + 3)
		{
		  for (xx = 0; xx <= 6; xx = xx + 3)
		    {
		      dest.x = sx + xx + (rand() % 9) - 4;
		      dest.y = sy + yy - (rand() % 8);
		      
		      SDL_BlitSurface(screen, &src, tmp_surf, &dest);
		    }
		}
	      
	      
	      if ((rand() % 5) == 0)
		{
		  if (i + (rand() % 5) < 10)
		    {
		      dest.x = 320 - ((x * 2) / 8);
		      dest.y = 480 - ((y * 2) / 8);
		      dest.w = 2;
		      dest.h = 2;
		      
		      if ((rand() % 5) == 0)
			{
			  SDL_FillRect(tmp_surf, &dest,
				       SDL_MapRGBA(tmp_surf->format,
						   (rand() % 32) + 224,
						   (((rand() % 32) + 224) *
						    rand() % 2),
						   0,
						   128));
			}
		      else
			{
			  SDL_FillRect(tmp_surf, &dest,
				       SDL_MapRGBA(tmp_surf->format,
						   (rand() % 128) + 128,
						   (rand() % 96) + 32,
						   (rand() % 32),
						   (rand() % 64) + 64));
			}
		    }
		  else
		    {
		      dest.x = 320 - ((x * 2) / 8);
		      dest.y = 480 - ((y * 2) / 8);
		      dest.w = 2;
		      dest.h = 2;
		      
		      grey = (rand() % (128 - i));
		  
		      SDL_FillRect(tmp_surf, &dest,
				   SDL_MapRGBA(tmp_surf->format,
					       grey, grey, grey,
					       (rand() % 128) + 64));
		    }
		}
	    }
	}
      
      SDL_BlitSurface(tmp_surf, NULL, screen, NULL);
      SDL_Flip(screen);
      
      
      /* Throttle FPS: */
      
      cur_time = SDL_GetTicks();
      
      if (cur_time < prev_time + (1000 / FPS))
	SDL_Delay((prev_time + (1000 / FPS)) - cur_time);
    }
  
  SDL_FreeSurface(tmp_surf);
  
  clear_screen(level_colors[(level - 1) / 2][0],
	       level_colors[(level - 1) / 2][1],
    	       level_colors[(level - 1) / 2][2]);

  return(cmd);
}


void clear_screen(Uint8 r, Uint8 g, Uint8 b)
{
  SDL_Rect dest;


  SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, r, g, b));

  bkgd_r = r;
  bkgd_g = g;
  bkgd_b = b;
  

  /* Fuse: */
  
  dest.x = 320 - (images[IMG_FUSE]->w / 2);
  dest.y = 480 - images[IMG_FUSE]->h;
  
  SDL_BlitSurface(images[IMG_FUSE], NULL, screen, &dest);
  
  
  /* TNT: */
  
  dest.x = 320 - (images[IMG_TNT]->w / 2);
  dest.y = 480 - (images[IMG_TNT]->h);
  
  SDL_BlitSurface(images[IMG_TNT], NULL, screen, &dest);
  
  
  /* Axis */
  
  dest.x = 320 - (images[IMG_AXIS]->w / 2);
  dest.y = 480 - images[IMG_TNT]->h - images[IMG_AXIS]->h;
  
  SDL_BlitSurface(images[IMG_AXIS], NULL, screen, &dest);


  /* Flip screen and init rects: */

  SDL_Flip(screen);
  num_rects = 0;
}

void add_rect(SDL_Rect rect)
{
  if (num_rects < MAX_RECTS &&
      rect.w > 0 && rect.h > 0 &&
      rect.x < screen->w && rect.y < screen->h)
  {
    if (rect.x < 0)
    {
      rect.w = rect.w + rect.x;
      rect.x = 0;
    }

    if (rect.y < 0)
    {
      rect.h = rect.h + rect.y;
      rect.y = 0;
    }
    if (rect.x + rect.w >= screen->w)
      rect.w = screen->w - rect.x;

    if (rect.y + rect.h >= screen->h)
      rect.h = screen->h - rect.y;

    if (rect.w > 0 && rect.h > 0)
    {
      memcpy(&(rects[num_rects]), &rect, sizeof(SDL_Rect));
      num_rects++;
    }
  }
}


void draw_left_detonator(boolean draw)
{
  int min_y;
  int sx, sy;
  SDL_Rect dest;

  /* Plunger: */

  min_y = (480 - images[IMG_DETONATOR]->h - images[IMG_PLUNGER]->h);

  sx = (images[IMG_DETONATOR]->w / 2) - (images[IMG_PLUNGER]->w / 2);
  sy = teeter_height(sx) + images[IMG_PANEL]->h;
  if (sy < min_y)
    sy = min_y;
  
  dest.x = sx;
  dest.y = sy;
  dest.w = images[IMG_PLUNGER]->w;
  dest.h = images[IMG_PLUNGER]->h;
 
  if (draw)
    SDL_BlitSurface(images[IMG_PLUNGER], NULL, screen, &dest);
  else
    erase_rect(dest);

  add_rect(dest);

  
  /* Detonator: */

  if (draw)
  {
    dest.x = 0;
    dest.y = 480 - (images[IMG_DETONATOR]->h);
    dest.w = images[IMG_DETONATOR]->w;
    dest.h = images[IMG_DETONATOR]->h;
  
    SDL_BlitSurface(images[IMG_DETONATOR], NULL, screen, &dest);

    add_rect(dest);
  }
}


void draw_right_detonator(boolean draw)
{
  int min_y;
  int sx, sy;
  SDL_Rect dest;

  /* Plunger: */

  min_y = (480 - images[IMG_DETONATOR]->h - images[IMG_PLUNGER]->h);

  sx = 640 - (images[IMG_DETONATOR]->w / 2) - (images[IMG_PLUNGER]->w / 2);
  sy = teeter_height(sx) + images[IMG_PANEL]->h;
  if (sy < min_y)
    sy = min_y;
  
  dest.x = sx;
  dest.y = sy;
  dest.w = images[IMG_PLUNGER]->w;
  dest.h = images[IMG_PLUNGER]->h;
  
  if (draw)
    SDL_BlitSurface(images[IMG_PLUNGER], NULL, screen, &dest);
  else
    erase_rect(dest);

  add_rect(dest);

  
  /* Right detonator: */

  if (draw)
  {
    dest.x = 640 - (images[IMG_DETONATOR]->w);
    dest.y = 480 - (images[IMG_DETONATOR]->h);
    dest.w = images[IMG_DETONATOR]->w;
    dest.h = images[IMG_DETONATOR]->h;
  
    SDL_BlitSurface(images[IMG_DETONATOR], NULL, screen, &dest);

    add_rect(dest);
  }
}



void erase_rect(SDL_Rect rect)
{
  SDL_FillRect(screen, &rect,
	       SDL_MapRGB(screen->format, bkgd_r, bkgd_g, bkgd_b));
}


void draw_score(boolean draw)
{
  SDL_Rect dest;
  char str[32];

  sprintf(str, "SCORE %05d  LEVEL %2d", score, level);

  dest.x = 0;
  dest.y = 0;
  dest.w = (((images[IMG_LETTERS]->w) / 26) * strlen(str)) + 2;
  dest.h = images[IMG_LETTERS]->h + 2;

  if (draw)
  {
    draw_text(dest.x + 2, dest.y + 2, str, TRUE);
    draw_text(dest.x, dest.y, str, FALSE);
  }
  else
    erase_rect(dest);

  add_rect(dest);
}


void draw_cannon(boolean draw)
{
  int sx;
  SDL_Rect dest;

  /* Draw cannon: */
  
  sx = 320 + cannon_x - (images[IMG_CANNON]->w / 2);
  
  dest.x = sx;
  dest.y = teeter_height(sx) - (images[IMG_WHEEL1]->h +
				(images[IMG_CANNON]->h / 2));
  dest.w = images[IMG_CANNON]->w;
  dest.h = images[IMG_CANNON]->h;
 
  if (draw)
    SDL_BlitSurface(images[IMG_CANNON], NULL, screen, &dest);
  else
    erase_rect(dest);

  add_rect(dest);
  
  
  /* Wheels! */
  
  wheel_frame = ((cannon_x + 320) / 4) % 3;
  
  
  /* Left wheel: */
  
  dest.x = sx - (images[IMG_WHEEL1]->w);
  dest.y = teeter_height(dest.x) - (images[IMG_WHEEL1]->h);
  dest.w = images[IMG_WHEEL1]->w;
  dest.h = images[IMG_WHEEL2]->h;
 
  if (draw)
  {
    SDL_BlitSurface(images[IMG_WHEEL1 + (wheel_frame % 3)], NULL,
		    screen, &dest);
  }
  else
    erase_rect(dest);
  
  add_rect(dest);
 

  /* Right wheel: */
  
  dest.x = sx + (images[IMG_CANNON]->w);
  dest.y = teeter_height(dest.x) - (images[IMG_WHEEL1]->h);
  dest.w = images[IMG_WHEEL1]->w;
  dest.h = images[IMG_WHEEL2]->h;
  
  if (draw)
  {
    SDL_BlitSurface(images[IMG_WHEEL1 + ((wheel_frame + 1) % 3)], NULL,
		    screen, &dest);
  }
  else
    erase_rect(dest);

  add_rect(dest);
}


void draw_explosions(boolean draw)
{
  int i;
  SDL_Rect src, dest;

  for (i = 0; i < NUM_EXPLOSION_BITS; i++)
  {
    if (explosion[i].active)
    {
      dest.x = explosion[i].x;
      dest.y = explosion[i].y;
      
      if (explosion[i].expl_type == EXPL_PLAYER)
	{
	  dest.w = 3;
	  dest.h = 3;
	 
	  if (draw)
	  {
	    SDL_FillRect(screen, &dest,
		         SDL_MapRGB(screen->format,
				    (rand() % player_exploding),
				    (rand() % player_exploding) + 64,
				    (rand() % player_exploding) + 128));
	  }
	  else
	    erase_rect(dest);
	}
      else
	{
	  src.x = (rand() % (images[IMG_BADDIE1]->w - 1));
	  src.y = (rand() % (images[IMG_BADDIE1]->h - 1));
	  src.w = 2;
	  src.h = 2;
	  
	  dest.w = src.w;
	  dest.h = src.h;
	 
	  if (draw)
	    SDL_BlitSurface(images[IMG_BADDIE1], &src, screen, &dest);
	  else
	    erase_rect(dest);
	}

      add_rect(dest);
    }
  }
}

void draw_monsters(boolean draw)
{
  int i;
  SDL_Rect dest;


  for (i = 0; i < 18; i++)
  {
    if (monsters[i].alive)
    {
      dest.x = monsters[i].x;
    
      if (monsters[i].attached == FALSE)
		dest.y = monsters[i].y;
      else
		dest.y = teeter_height(dest.x);

      dest.w = images[IMG_BADDIE1]->w;
      dest.h = images[IMG_BADDIE1]->h;

      if (draw)
        SDL_BlitSurface(images[IMG_BADDIE1], NULL, screen, &dest);
	  else
	  erase_rect(dest);

	  add_rect(dest);}
    

  }
}



void draw_teetertotter(boolean draw)
{
  int sx, sy;
  SDL_Rect dest;

  for (sx = 0; sx < 640; sx = sx + images[IMG_PANEL]->w)
  {
    sy = teeter_height(sx);
	  
    dest.x = sx;
    dest.y = sy;
    dest.w = images[IMG_PANEL]->w;
    dest.h = images[IMG_PANEL]->h;

    if (draw)
      SDL_BlitSurface(images[IMG_PANEL], NULL, screen, &dest);
    else
      erase_rect(dest);

    add_rect(dest);
  }
}

//XBOX FUNCTIONS//

void playsound(int snd)
{
  int which, i;
  which = (rand() % 3) + 0;

  for (i = 0; i < 4; i++)
	{
	  if (!Mix_Playing(i))
	    which = i;
	}

	Mix_PlayChannel(which, sounds[snd], 0);
    

}
void save_options(void)
{
  int i;

  FILE * fi;

    fi = fopen("U:\\652B68593215\\game.xsv", "w");

	if(fi == NULL)
	{
		CreateDirectory("U:\\652B68593215", NULL);
		fi = fopen("U:\\savegame\\game.xsv", "w");
	}

	if(fi != NULL)
	{
		for (i = 0; i < 6; i++)
		{
		fprintf(fi, "name%d=%s\n", i + 1, high_score_names[i]);

		if(i == 5)
		{
			if(score > high_score_scores[i]) 
			fprintf(fi, "high%d=%d\n", i + 1, score);
			else
			fprintf(fi, "high%d=%d\n", i + 1, high_score_scores[i]);
		}
		else
			fprintf(fi, "high%d=%d\n", i + 1, high_score_scores[i]);
		}

		fprintf(fi, "XPOS=%d\n", xpos);
		fprintf(fi, "YPOS=%d\n", ypos);
		fprintf(fi, "XSTRETCH=%d\n", xstretch);
		fprintf(fi, "YSTRETCH=%d\n", ystretch);
		fprintf(fi, "VOLUME=%d\n", VOLUME);

	  fclose(fi);

	}

	fi = fopen("U:\\652B68593215\\SaveMeta.xbx", "r+");
	if(fi == NULL)
	{
		fi = fopen("U:\\652B68593215\\SaveMeta.xbx", "w");
		fprintf(fi, "Name=Game Data");
		fclose(fi);
		CopyFile("U:\\SaveImage.xbx", "U:\\652B68593215\\SaveImage.xbx", FALSE);
	}
	else
		fclose(fi);


	

}

void load_options(void)
{
  char str[500];
  FILE * fi;
  fi = fopen("U:\\652B68593215\\game.xsv", "r");
  

  if(fi != NULL) 
    {
      do
      {
		fgets(str, sizeof(str), fi);
		if (!feof(fi))
		{
			if (strstr(str, "name") == str &&
				str[4] >= '1' && str[4] <= '6' &&
				str[5] == '=' && strlen(str) >= 9)
			{
				high_score_names[str[4] - '1'][0] = str[6];
				high_score_names[str[4] - '1'][1] = str[7];
				high_score_names[str[4] - '1'][2] = str[8];
				high_score_names[str[4] - '1'][3] = '\0';
			}

			if (strstr(str, "high") == str &&
				str[4] >= '1' && str[4] <= '6' &&
				str[5] == '=')
			{
				high_score_scores[str[4] - '1'] = atoi(str + 6);
			}

			if (strstr(str, "XPOS=") == str){
				
				xpos = atoi(str + 5);

			}
			else if (strstr(str, "YPOS=") == str){
				
				ypos = atoi(str + 5);
	
			}
			else if (strstr(str, "XSTRETCH=") == str){
				
				xstretch = atoi(str + 9);

			}
			else if (strstr(str, "YSTRETCH=") == str){

				ystretch = atoi(str + 9);

			}
			else if (strstr(str, "VOLUME=") == str){

				VOLUME = atoi(str + 7);

			}
		}
      }
      while (!feof(fi));
      
      fclose(fi);
    }

	//If user has changed screen size
	if((xstretch + ystretch + xpos + ypos) != 0)
	{
		//Set Screen Size 
		SDL_XBOX_SetScreenStretch(xstretch, ystretch);
		SDL_XBOX_SetScreenPosition(xpos, ypos);
		
	}

	return;
}

void resize_screen()
{
  int done = 0, button;
  SDL_Rect src, dest;
  SDL_Event event;
  
	dest.x = 0;
    dest.y = 0;
    dest.w = 640;
    dest.h = 480;
				
	SDL_FillRect(screen, &dest,
	SDL_MapRGB(screen->format, 0x00, 0x00, 0x00));
	src.x = 0;
	src.y = 0;
	src.w = images[IMG_RESIZE]->w;
	src.h = images[IMG_RESIZE]->h;
	dest.x = 0;
	dest.y = 0;
	dest.w = images[IMG_RESIZE]->w;
	dest.h = images[IMG_RESIZE]->h;
	SDL_BlitSurface(images[IMG_RESIZE], NULL, screen, &dest);
	SDL_UpdateRect(screen, 0, 0, 640, 480);
				
				
	do{
							    

		axisL = SDL_JoystickGetAxis(xJoy, JOY_AXIS_LX);
		axisLY = SDL_JoystickGetAxis(xJoy, JOY_AXIS_LY);
		axisR = SDL_JoystickGetAxis(xJoy, JOY_AXIS_RX);
		axisRY = SDL_JoystickGetAxis(xJoy, JOY_AXIS_RY);               
        
		// y = top to bottom
		// x = left to right

		if (axisL < -JOY_DEADZONE)
		{
			xstretch = xstretch - 4;
				
				
				
		}
        else if (axisL > JOY_DEADZONE)
		{
			xstretch = xstretch + 4;
					
					
		}

		else if (axisLY < -JOY_DEADZONE)
		{
			ystretch = ystretch - 3;
					
		}
        else if (axisLY > JOY_DEADZONE)
		{
			ystretch = ystretch + 3;
		}


		//Screen RePosition
        if (axisR < -JOY_DEADZONE)
		{
			xpos = xpos - 1;
		}

		else if (axisR > JOY_DEADZONE)
		{
			xpos = xpos + 1;
		}

		else if (axisRY < -JOY_DEADZONE)
		{
			ypos = ypos - 1;
		}
        else if (axisRY > JOY_DEADZONE)
		{
			ypos = ypos + 1;
		}


				
		while (SDL_PollEvent(&event)){

			if (event.type == SDL_JOYBUTTONDOWN)
			{
				button = event.jbutton.button;
					if (button == JOY_BTTN_A)
					{
						done = 1;
					
					}
			}
			}

		SDL_XBOX_SetScreenPosition(xpos, ypos);
		SDL_XBOX_SetScreenStretch(xstretch, ystretch);



		}
	while (done == 0);
				


				save_options();
				clear_screen(level_colors[(level - 1) / 2][0],
				             level_colors[(level - 1) / 2][1],
    			             level_colors[(level - 1) / 2][2]);
				return;
				
}














