#include <assert.h>
//#include <d3dx9.h>
//#include "DXUtil.h"
#include <xtl.h>
#include "port.h"

#include "HeightMap.h"

#include ".\shadow.h"

#include "DxObjects.h"

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
LPDIRECT3DTEXTURE8		CShadow::m_pTexture = NULL;
//-----------------------------------------------------------------------------

CShadow::CShadow(void)
{
	m_pVB		= NULL;
	m_dwFVF		= D3DFVF_XYZ|D3DFVF_TEX1;
	m_bRender	= false;

	m_pIndexBuf	= NULL;

	m_nNumVertsX = (int)sqrt((float)NUM_VERTS);
	D3DXMatrixIdentity(&m_matWorld);
}

//-----------------------------------------------------------------------------

CShadow::~CShadow(void)
{
}

//-----------------------------------------------------------------------------

void CShadow::FrameMove(	const D3DXVECTOR3	&vecShipPos	,
							float				fShipBearing,
							const CHeightMap	&HeightMap)
{
	if(m_bRender)
	{
		int nHeightMapSizeX = HeightMap.GetSizeX();
		int nHeightMapSizeZ = HeightMap.GetSizeZ();

		// Calculate ship position in HeightMap grid
		//   (x,z) is grid square "near left" of ship (LeftHand co-ord system)
		float scaledX = vecShipPos.x * nHeightMapSizeX;
		float scaledZ = vecShipPos.z * nHeightMapSizeZ;
		int x = (int)floor(scaledX);
		int z = (int)floor(scaledZ);

		float fOffsetX = (scaledX - (float)x) / m_nNumVertsX;
		float fOffsetZ = (scaledZ - (float)z) / m_nNumVertsX;


		// Offset (x,z) by half shadow dimension
		//  Now (x,z) is "near left" corner of shadow
		int		nHalfShadowWidth = m_nNumVertsX / 2;
		x -= nHalfShadowWidth;
		z -= nHalfShadowWidth;

		float	fLandScapeUnitSize = 1.0f / float(nHeightMapSizeX);

		// Rebuild the vBuf with new position + texture data...
		CShadow::CUSTOMVERTEX* pVertices;
		//m_pVB->Lock( 0, 0, (VOID**)&pVertices, 0 );
		m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0);

		float tu, tv;
		float sinTheta = (float)sin(fShipBearing);
		float cosTheta = (float)cos(fShipBearing);

		int heightMapX, heightMapZ;
		int nCurrentVertex = 0;
		for(int iz = 0; iz < m_nNumVertsX; ++iz)
		{
			for(int ix = 0; ix < m_nNumVertsX; ++ix)
			{
				pVertices[nCurrentVertex].position.x = (float)(x+ix) * fLandScapeUnitSize;
				pVertices[nCurrentVertex].position.z = (float)(z+iz) * fLandScapeUnitSize;

				heightMapX = x + ix;
				heightMapZ = z + iz;

				pVertices[nCurrentVertex].position.y = 
						HeightMap.GetData(heightMapZ, heightMapX) + 0.002f;

				tu = m_pTexCoords[nCurrentVertex].tu - fOffsetX;
				tv = m_pTexCoords[nCurrentVertex].tv + fOffsetZ;

				pVertices[nCurrentVertex].tu = tu *  cosTheta + tv * sinTheta + 0.5f;
				pVertices[nCurrentVertex].tv = tu * -sinTheta + tv * cosTheta + 0.5f;
				++nCurrentVertex;
			}
		}

		//assert(ValidInXZ(pVertices));

		m_pVB->Unlock();
	}
}

//-----------------------------------------------------------------------------

HRESULT CShadow::InitDeviceObjects()
{
	HRESULT hr = S_OK;

	// Build array of untransformed texture co-ords
	for(int iz = 0; iz < m_nNumVertsX; ++iz)
	{
		for(int ix = 0; ix < m_nNumVertsX; ++ix)
		{
			m_pTexCoords[iz * m_nNumVertsX + ix].tu = (float)ix * 1.0f / (float)m_nNumVertsX - 0.5f;
			m_pTexCoords[iz * m_nNumVertsX + ix].tv = 1.0f - (float)iz * 1.0f / (float)m_nNumVertsX - 0.5f;
		}
	}

	// Create Index buffer
	if((hr = CreateIndexBuffer()) != S_OK)
	{
//        MessageBox(NULL, "Shadow IndBuf creation error", "FlyPhil.exe", MB_OK);
		return(hr);
	}

	// Create texture
	if(!m_pTexture)
	{
		LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();
		if((hr = D3DXCreateTextureFromFileEx(pDevice, "D:\\Shadow.bmp", D3DX_DEFAULT,
				D3DX_DEFAULT, D3DX_DEFAULT, 0, D3DFMT_A8R8G8B8, D3DPOOL_MANAGED,
				D3DX_DEFAULT, D3DX_DEFAULT, 0, NULL, NULL, &m_pTexture)) != S_OK)
		{
//			MessageBox(NULL, "Could not load shadow texture map", "FlyPhil.exe", MB_OK);
			return(hr);
		}
		
		D3DSURFACE_DESC pDDS;
		if((hr=m_pTexture->GetLevelDesc(0, &pDDS)) != S_OK)
		{
//			MessageBox(NULL, "Can't GetLevelDesc() shadow texture", "FlyPhil.exe", MB_OK);
			return(hr);
		}
		int nNumPixels = pDDS.Width * pDDS.Height;

		D3DLOCKED_RECT lockedRect;
		if((hr=m_pTexture->LockRect(0, &lockedRect, NULL, 0)) != S_OK)
		{
//			MessageBox(NULL, "Can't LockRect() shadow texture", "FlyPhil.exe", MB_OK);
			return(hr);
		}

		BYTE *ptr = (BYTE *)(lockedRect.pBits);
		BYTE b;
		for(int i=0; i < nNumPixels; ++i)
		{
			b = *(ptr + (i * 4));
			*(ptr + (i * 4)) = 8;
			*(ptr + (i * 4) + 1) = 8;
			*(ptr + (i * 4) + 2) = 8;
			*(ptr + (i * 4) + 3) = (BYTE)max(0, (196 - b));
		}

		m_pTexture->UnlockRect(0);
	}

	return(hr);
}

//-----------------------------------------------------------------------------

void CShadow::DeleteDeviceObjects()
{
	SAFE_RELEASE( m_pTexture );
	SAFE_RELEASE(m_pIndexBuf);

	m_bRender = false;
}

//-----------------------------------------------------------------------------

HRESULT CShadow::RestoreDeviceObjects()
{
	HRESULT hr = CreateNullVertexBuffer();
	if(hr == S_OK)
		m_bRender = true;

	return(hr);
}

//-----------------------------------------------------------------------------

void CShadow::InvalidateDeviceObjects()
{
	SAFE_RELEASE( m_pVB );

	m_bRender = false;
}

//-----------------------------------------------------------------------------

void CShadow::Render() const
{
	if(m_bRender)
	{
		LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();

		pDevice->SetTransform(D3DTS_WORLD, &m_matWorld);
		pDevice->SetStreamSource( 0, m_pVB, sizeof(CShadow::CUSTOMVERTEX) );
		pDevice->SetVertexShader(m_dwFVF);
		pDevice->SetTexture(0, m_pTexture);
		pDevice->SetIndices(m_pIndexBuf, 0);
		pDevice->DrawIndexedPrimitive(	D3DPT_TRIANGLESTRIP	,
										0, NUM_VERTS, 0,
										m_nNumPrims			);
	}
}

//-----------------------------------------------------------------------------

HRESULT CShadow::CreateNullVertexBuffer()
{
	HRESULT hr;

	LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();
    if((hr = pDevice->CreateVertexBuffer(	NUM_VERTS * sizeof(CShadow::CUSTOMVERTEX)	,
							D3DUSAGE_DYNAMIC, m_dwFVF, D3DPOOL_DEFAULT, &m_pVB)) != S_OK)
		return(hr);

	// Entire VBuf contents will be assigned every frame by FrameMove()

	return S_OK;
}

//-----------------------------------------------------------------------------

HRESULT CShadow::CreateIndexBuffer()
{
	if(!m_pIndexBuf)
	{
		// Create and Lock IndBuf
		int NumOfIndicesInIndexBuf = ((m_nNumVertsX*2)+2) * (m_nNumVertsX - 1);

		HRESULT hr;
		if((hr = CDxObjects::GetDevice()->CreateIndexBuffer(NumOfIndicesInIndexBuf * sizeof(WORD),
					D3DUSAGE_WRITEONLY, D3DFMT_INDEX16, D3DPOOL_MANAGED,
					&m_pIndexBuf)) != S_OK)
			return(hr);

		WORD *pIndices = NULL;
		if((hr = m_pIndexBuf->Lock(0, 0, (BYTE**)&pIndices, 0)) != S_OK)
			return(hr);

		// Fill in IndBuf
		int		nCurIndex = 0;
		for(int y=(m_nNumVertsX - 2); y >= 0 ; --y)
		{
			for(int x=(m_nNumVertsX - 1); x >= 0; --x)
			{
				pIndices[nCurIndex] = (WORD)(x + (y * m_nNumVertsX));
				++nCurIndex;
				pIndices[nCurIndex] = (WORD)(x + ((y+1) * m_nNumVertsX));
				++nCurIndex;
			}
			pIndices[nCurIndex] = pIndices[nCurIndex - 1];
			++nCurIndex;
			pIndices[nCurIndex] = (WORD)((y + 1) * m_nNumVertsX);
			++nCurIndex;
		}
		m_nNumPrims = (((m_nNumVertsX - 1) * 2 + 4) *
			(m_nNumVertsX - 1)) - 4;

		// Unlock IndBuf
		m_pIndexBuf->Unlock();
	}
	return S_OK;
}

//-----------------------------------------------------------------------------

//bool CShadow::ValidInXZ(CShadow::CUSTOMVERTEX* pVertices)
//{
//	bool bValid = true;
//
//	float fStep = 1.0f / 64.0f;
//	for(int z=0; z < m_nNumVertsX; ++z)
//	{
//		for(int x=0; x < m_nNumVertsX; ++x)
//		{
//			if(x)
//				bValid = (pVertices[z * m_nNumVertsX + x].position.x == pVertices[z * m_nNumVertsX + x - 1].position.x + fStep);
//			if(!bValid)
//				break;
//			if(z)
//				bValid = (pVertices[z * m_nNumVertsX + x].position.z == pVertices[(z-1) * m_nNumVertsX + x].position.z + fStep);
//			if(!bValid)
//				break;
//		}
//		if(!bValid)
//			break;
//	}
//	return(bValid);
//}

//-----------------------------------------------------------------------------
