

#ifndef FILESELECTOR_H
#define FILESELECTOR_H

#include <stdio.h>
#include <stdarg.h>
#include <xtl.h>
#include <xgraphics.h>
#include "debug.h"

#define		DIRENT_MAX_COUNT	512

const SHORT XINPUT_DEADZONE = (SHORT)( 0.24f * FLOAT(0x7FFF) );

#define		FILESEL_DEADZONE	0.2f
#define		FILESEL_OVERDRIVE	0.9f

// How long to hold for fastest scrolling
#define		FILESEL_SLOW_TIME	1000
#define		FILESEL_MED_TIME	2000
#define		FILESEL_FAST_TIME	4000

// How big the multipliers are
#define		FILESEL_NORM_MULT	0.2f
#define		FILESEL_SLOW_MULT	1.0f
#define		FILESEL_MED_MULT	2.0f
#define		FILESEL_HIGH_MULT	5.0f

#define		FILESEL_SELECT_CLAMP	3

class CXBoxFileSelector
{
public:

	CXBoxFileSelector(CXBoxDebug *debug,LPDIRECT3DTEXTURE8 backgroundtexture,CXBFont &font)
		:mFont(font)
	{
		int twiddle=0;

		mpDebug=debug;
		mpBackgroundTexture=backgroundtexture;
		mpDirectory=NULL;
		mDirEntCount=0;
		mDirEntStart=0;
		mDirEntSelect=0.0f;
		mDirColumns=1;

		// Setup default display area for listing
		mDisplayRect.left=30;
		mDisplayRect.top=30;
		mDisplayRect.right=610;
		mDisplayRect.bottom=450;

		// Default colours
		mHighlightColour=0xffffffff;
		mLowlightColour=0x50ffffff;

		if( FAILED( D3DXCreateSprite(g_pd3dDevice,&mpSpriteInterface ) ) )
		{
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - Failed to create sprite interface. Xbox Halted (Press OFF)");
			while(1) twiddle++;
		}

		// Texture size correction
		if(mpBackgroundTexture)
		{
			D3DSURFACE_DESC desc;
			mpBackgroundTexture->GetLevelDesc(0, &desc);
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - Size=%d, Width=%d, Height=%d",desc.Size,desc.Width,desc.Height);
			mBackgroundScaling.x=640.0f/desc.Width;
			mBackgroundScaling.y=480.0f/desc.Height;
		}
	}

	~CXBoxFileSelector()
	{
		if(mpDirectory) delete mpDirectory;
		for(int loop=0;loop<DIRENT_MAX_COUNT;loop++)
		{
			if(mpDirList[loop]) delete mpDirList[loop];
			mpDirList[loop]=NULL;
		}
	}

	void SetColours(DWORD highlight, DWORD lowlight)
	{
		mHighlightColour=highlight;
		mLowlightColour=lowlight;
		if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - SetColours accepted %08x %08x",highlight,lowlight);
	}

	void SetDisplayRect(RECT displayrect)
	{
		mDisplayRect=displayrect;
		if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - SetDisplayRect accepted %03d %03d %03d %03d",displayrect.left,displayrect.top,displayrect.right,displayrect.bottom);
	}

	void SetColumns(int columns)
	{
		if(columns>0 && columns<4)
		{
			mDirColumns=columns;
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - SetColumns accepted %d",columns);
		}
		else
		{
			if(mpDebug) mpDebug->MessageInstant(L"CXBoxFileSelector() - SetColumns rejected %d",columns);
		}
	}

	void SetDirectory(CHAR *path)
	{
	    WIN32_FIND_DATA wfd;
	    HANDLE hFind;

		// Save the directory path for later, remember it has a filter on it
		if(mpDirectory) delete mpDirectory;
		mpDirectory= new CHAR[strlen(path)+1];
		strcpy(mpDirectory,path);

		// Flush the directory cash
		for(int loop=0;loop<DIRENT_MAX_COUNT;loop++)
		{
			if(mpDirList[loop]) delete mpDirList[loop];
			mpDirList[loop]=NULL;
		}
		mDirEntCount=0;
		mDirEntStart=0;
		mDirEntSelect=0;

		// Now we must read in the whole directory into an array
		if(mpDebug) mpDebug->Message(L"Reading Directory: %s",path);

	    // Start the find and check for failure.
		hFind = FindFirstFile( mpDirectory, &wfd );

	    if( INVALID_HANDLE_VALUE == hFind )
		{
			mDirEntCount=0;
			if(mpDebug) mpDebug->Message(L"CXBoxFileSelector() SetDirectory FindFirstFile Failed on %s",mpDirectory);
			return;
	    }
		else
	    {
		    // Display each file and ask for the next.
			do
	        {
				if(mpDebug) mpDebug->Message(L"CXBoxFileSelector() SetDirectory Found: %hs",wfd.cFileName);
				mpDirList[mDirEntCount]= new CHAR[strlen(wfd.cFileName)+1];
				strcpy(mpDirList[mDirEntCount],wfd.cFileName);
				mDirEntCount++;
	        }
			while( FindNextFile( hFind, &wfd ) && mDirEntCount<DIRENT_MAX_COUNT);

			if(mpDebug && mDirEntCount==DIRENT_MAX_COUNT) mpDebug->Message(L"CXBoxFileSelector() SetDirectory exceeded DIRENT_MAX_COUNT on %s",mpDirectory);
	
		    // Close the find handle.
			FindClose( hFind );
	    }
	}

	void ProcessFileSelect(XBGAMEPAD &Gamepad)
	{
		static DWORD rampstart=0;
		DWORD holddown;
		float velocity,t1,t2,scale;
		int onscreen=HowManyOnScreen();

		scale=0.0;
		t1 = (float)(Gamepad.sThumbLY)/32768.0f;
		t2 = (float)(Gamepad.sThumbRY)/32768.0f;

		velocity=t1+t2;
		// Of both are out of the deadzone then we must rescale
		if(fabs(t1)>XINPUT_DEADZONE && fabs(t2)>XINPUT_DEADZONE ) velocity/=2.0;

		// For our purposes the axis are reversed
		velocity=-velocity;

		// do dead zone and gamma curve
		if(fabs(velocity) < FILESEL_DEADZONE)
		{
			// zero dead zone (for spring slack & noise)
			velocity = 0.0f;
			rampstart=0;
		}
		else
		{
			// If the activity has just started
			if(!rampstart) rampstart=GetTickCount();

			// How long has the active state been held
			holddown=GetTickCount()-rampstart;

			//start at 0.0f after deadzone
			if( velocity < 0.0f ) velocity += FILESEL_DEADZONE;	else velocity -= FILESEL_DEADZONE;

			//rescale to correct range bacause of deadzone removal
			velocity *= (1.0f/(1.0f-FILESEL_DEADZONE));
			velocity *= FILESEL_NORM_MULT;

			//increase max held count (for speed up bands)
			if( fabs(velocity) > FILESEL_OVERDRIVE )
			{
				//check for speed scale banding
				if( holddown > FILESEL_FAST_TIME)		velocity *= FILESEL_HIGH_MULT;
				else if( holddown > FILESEL_MED_TIME)	velocity *= FILESEL_MED_MULT;
				else velocity *= FILESEL_SLOW_MULT;
			}
		}

		// Select the new entry
		mDirEntSelect+=velocity;

		// We must clamp the selector within the list bounds
		if(mDirEntSelect<0) mDirEntSelect=0;
		if((int)mDirEntSelect>=mDirEntCount) mDirEntSelect=(float)(mDirEntCount-1);

		// Set the start position
		if(mDirEntCount<=onscreen)
		{
			mDirEntStart=0;
		}
		else
		{
			if(mDirColumns>1)
			{
				// Clamp to start/end +/-3
				if(mDirEntSelect<(mDirEntStart+FILESEL_SELECT_CLAMP)) mDirEntStart=(int)(mDirEntSelect-FILESEL_SELECT_CLAMP);
				if(mDirEntSelect>(mDirEntStart+(onscreen-FILESEL_SELECT_CLAMP))) mDirEntStart=(int)(mDirEntSelect-(onscreen-FILESEL_SELECT_CLAMP));
			}
			else
			{
				mDirEntStart=(int)mDirEntSelect-(onscreen/2);
			}
			// Fix the special cases at the start and end
			if(mDirEntStart<0) mDirEntStart=0;
			if(mDirEntStart>(mDirEntCount-onscreen)) mDirEntStart=mDirEntCount-onscreen;
		}
	}

	void RenderFileSelect(void)
	{
		float xpos,ypos,ylimit;
		int	dirent=mDirEntStart;

		// The Backdrop, its allowed to be NULL
		if(mpBackgroundTexture)
		{
			mpSpriteInterface->Begin();
			mpSpriteInterface->Draw(mpBackgroundTexture,NULL,&mBackgroundScaling,NULL,0.0f ,NULL,D3DCOLOR_ARGB(0xff, 0xff, 0xff, 0xff));
			mpSpriteInterface->End();
		}

		xpos=(float)mDisplayRect.left;
		ypos=(float)mDisplayRect.top;
		ylimit=(float)mDisplayRect.bottom;

		// Now the file list
		mFont.Begin();
		for(int loop=0;loop<mDirColumns;loop++)
		{
			while(ypos<ylimit && dirent<mDirEntCount)
			{
				WCHAR dirstr[1024];

				// Convert to unicode for display
				swprintf(dirstr,L"%hs",mpDirList[dirent]);
	
				// Print file entry
				if(dirent==(int)mDirEntSelect)
				{
					// Print 100% alpha
					mFont.DrawText( xpos,ypos,mHighlightColour,dirstr);
				}
				else
				{
					// Blend to background
					mFont.DrawText( xpos,ypos,mLowlightColour,dirstr);
				}
			
				dirent++;
				// Increment font position
				ypos+=mFont.m_dwFontHeight + 1;
			}

			// Increment the Xpos and reset the Y
			xpos+=(float)((mDisplayRect.right-mDisplayRect.left)/mDirColumns);
			ypos=(float)mDisplayRect.top;

		}
		mFont.End();	
	}

	int HowManyOnScreen(void)
	{
		int count=0;
		float ypos=(float)mDisplayRect.top;
		float ylimit=(float)mDisplayRect.bottom;

		// Count how many in a column
		while(ypos<ylimit && count<DIRENT_MAX_COUNT)
		{
			count++;
			// Increment font position
			ypos+=mFont.m_dwFontHeight + 1;
		}

		// Multiply by number of columns
		count*=mDirColumns;

		return count;
	}	

	const char* GetSelected(void)
	{
		return mpDirList[(int)mDirEntSelect];
	}
	
	char				*mpDirList[DIRENT_MAX_COUNT];
	int					mDirEntCount;
	int					mDirEntStart;
	float				mDirEntSelect;
	int					mDirColumns;

	LPD3DXSPRITE		mpSpriteInterface;
	CXBoxDebug			*mpDebug;
	CXBFont				mFont;
	LPDIRECT3DTEXTURE8	mpBackgroundTexture;
	D3DXVECTOR2			mBackgroundScaling;

	CHAR				*mpDirectory;

	RECT				mDisplayRect;

	DWORD				mHighlightColour;
	DWORD				mLowlightColour;

};

#endif