/* error.c: The error reporting widget
   Copyright (c) 2002-2004 Philip Kendall

   $Id: error.c,v 1.17 2004/03/11 14:14:18 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/

#include <config.h>

#ifdef USE_WIDGET

#include <stddef.h>
#include <stdio.h>
#include <string.h>

#include "fuse.h"
#include "widget_internals.h"

static int split_message( const char *message, char ***lines, size_t *count,
			  const size_t line_length );

widget_error_t *error_info;

int
ui_error_specific( ui_error_level severity, const char *message )
{
  widget_error_t error_info;

  error_info.severity = severity;
  error_info.message  = message;

  //fuse_emulation_pause();
  //widget_do( WIDGET_TYPE_ERROR, &error_info );
  //fuse_emulation_unpause();

  return 0;
}

int widget_error_draw( void *data )
{
  char **lines; size_t count;
  size_t i;

  error_info = (widget_error_t*)data;
  if( split_message( error_info->message, &lines, &count, 28 ) ) return 1;
  
  widget_dialog_with_border( 1, 2, 30, count+2 );

  switch( error_info->severity ) {
  case UI_ERROR_INFO:
    widget_printstring( 13, 2, WIDGET_COLOUR_FOREGROUND, "Info" );
    break;
  case UI_ERROR_WARNING:
    widget_printstring( 12, 2, WIDGET_COLOUR_FOREGROUND, "Warning" );
    break;
  case UI_ERROR_ERROR:
    widget_printstring( 13, 2, WIDGET_COLOUR_FOREGROUND, "Error" );
    break;
  default:
    widget_printstring(  7, 2, WIDGET_COLOUR_FOREGROUND, "(Unknown message)" );
    break;
  }

  for( i=0; i<count; i++ ) {
    widget_printstring( 2, i+4, WIDGET_COLOUR_FOREGROUND, lines[i] );
    free( lines[i] );
  }

  free( lines );

  widget_display_lines( 2, count + 4 );

  return 0;
}

static int
split_message( const char *message, char ***lines, size_t *count,
	       const size_t line_length )
{
  const char *ptr = message;
  int position;

  /* Setup so we'll allocate the first line as soon as we get the first
     word */
  *lines = NULL; *count = 0; position = line_length;

  while( *ptr ) {

    /* Skip any whitespace */
    while( *ptr == ' ' ) ptr++; message = ptr;

    /* Find end of word */
    while( *ptr && *ptr != ' ' ) ptr++;

    /* message now points to a word of length (ptr-message); if
       that's longer than an entire line (most likely filenames), just
       take the last bit */
    if( ptr - message > (ptrdiff_t)line_length ) message = ptr - line_length;

    /* Check we've got room for the word, plus the prefixing space */
    if( position + ( ptr - message + 1 ) > (ptrdiff_t)line_length ) {
      char **new_lines; size_t i;

      /* If we've filled the screen, stop */
      if( *count == 18 ) return 0;

      new_lines = (char**)realloc( (*lines), (*count + 1) * sizeof( char** ) );
      if( new_lines == NULL ) {
	for( i=0; i<*count; i++ ) free( (*lines)[i] );
	if(*lines) free( (*lines) );
	return 1;
      }
      (*lines) = new_lines;

      (*lines)[*count] = (char*)malloc( (line_length+1) * sizeof(char) );
      if( (*lines)[*count] == NULL ) {
	for( i=0; i<*count; i++ ) free( (*lines)[i] );
	free( (*lines) );
	return 1;
      }
      
      strncpy( (*lines)[*count], message, ptr - message );
      position = ptr - message;
      (*lines)[*count][position] = '\0';

      (*count)++;

    } else {		/* Enough room on this line */

      strcat( (*lines)[*count-1], " " );
      strncat( (*lines)[*count-1], message, ptr - message );
      position += ptr - message + 1;
      (*lines)[*count-1][position] = '\0';

    }

    message = ptr;
    
  }

  return 0;
}

void
widget_error_keyhandler( input_key key )
{
  switch( key ) {

#if 0
  case INPUT_KEY_Resize:	/* Fake keypress generated on window resize */
    widget_error_draw( error_info );
    break;
#endif
    
  case INPUT_KEY_Escape:
    widget_end_widget( WIDGET_FINISHED_CANCEL );
    return;

  case INPUT_KEY_Return:
    widget_end_widget( WIDGET_FINISHED_OK );
    return;

  default:	/* Keep gcc happy */
    break;

  }
}

#endif				/* #ifdef USE_WIDGET */
