/*
	WinSTon

	Key remapping functions
*/

#include "..\includes\winston.h"
#include "..\includes\debug.h"
#include "..\includes\keymap.h"
#include "..\includes\memory.h"
#include "..\includes\misc.h"

//-----------------------------------------------------------------------
/*
	Remap table of Windows keys to ST Scan codes, -ve is invalid key(ie doesn't occur on ST)

  PC Keyboard:-

    Esc  F1  F2  F3  F4    F5  F6  F7  F8    F9  F10 F11 F12       Print Scroll Pause

     1   59  60  61  62    63  64  65  66    67  68  87  88                70     69


  !   "      $   %   ^   &   *   (   )   _   +                                 Page
`  1   2   3   4   5   6   7   8   9   0   -   =   <-               Ins   Home    Up

41 2   3   4   5   6   7   8   9   10  11  12  13  14               82     71     73
                                                                    --     --     --
                                                       |
                                             {   }     |                         Page
Tab  Q   W   E   R   T   Y   U   I   O   P   [   ] <----            Del   End    Down
                                                  
15   16  17  18  19  20  21  22  23  24  25  26  27  28             83     79     81
                                                                    --     --     --

                                           :   @   ~                       ^
Caps   A   S   D   F   G   H   J   K   L   ;   '   #                       |

58     30  31  32  33  34  35  36  37  38  39  40  43                      72
                                                                           --
  
^   |                               <   >   ?   ^
|   \   Z   X   C   V   B   N   M   ,   .   /   |                     <-   |   ->

42  86  44  45  46  47  48  49  50  51  52  53  54                    75   80  77
                                                                      --   --  --

Ctrl  Alt          SPACE             Alt Gr      Ctrl

 29    56            57                56         29
                                                  --

  And:-

Num
Lock   /    *    -

69     53   55   74
--     --

 7    8     9    +
Home  ^   Pg Up

71    72    73   78


4     5     6
<-          ->

75    76    77


1     2     3
End   |   Pg Dn  Enter

79    70    81    28
                  --

0     .
Ins   Del

82    83


*/

char Default_WindowsKeyToSTScanCode[] = {	// 256 scan codes + 256 extended scan codes
// ST Code     PC Code
	-1,			// 0
	0x01,		// 1 ESC
	0x02,		// 2 1
	0x03,		// 3 2
	0x04,		// 4 3
	0x05,		// 5 4
	0x06,		// 6 5
	0x07,		// 7 6
	0x08,		// 8 7
	0x09,		// 9 8
	0x0A,		// 10 9
	0x0B,		// 11 0
	0x0C,		// 12 -
	0x0D,		// 13 =
	0x0E,		// 14 BACKSPACE
	0x0F,		// 15 TAB
	0x10,		// 16 Q
	0x11,		// 17 W
	0x12,		// 18 E
	0x13,		// 19 R
	0x14,		// 20 T
	0x15,		// 21 Y
	0x16,		// 22 U
	0x17,		// 23 I
	0x18,		// 24 O
	0x19,		// 25 P
	0x1A,		// 26 [
	0x1B,		// 27 ]
	0x1C,		// 28 RETURN
	0x1D,		// 29 LEFT CTRL
	0x1E,		// 30 A
	0x1F,		// 31 S
	0x20,		// 32 D
	0x21,		// 33 F
	0x22,		// 34 G
	0x23,		// 35 H
	0x24,		// 36 J
	0x25,		// 37 K
	0x26,		// 38 L
	0x27,		// 39 ;
	0x28,		// 40 '
	-1,			// 41
	0x2A,		// 42 LEFT SHIFT
	0x2B,		// 43 #
	0x2C,		// 44 Z
	0x2D,		// 45 X
	0x2E,		// 46 C
	0x2F,		// 47 V
	0x30,		// 48 B
	0x31,		// 49 N
	0x32,		// 50 M
	0x33,		// 51 ,
	0x34,		// 52 .
	0x35,		// 53 /
	0x36,		// 54 RIGHT SHIFT
	0x66,		// 55 *(numeric pad)
	0x38,		// 56 ALT
	0x39,		// 57 SPACE
	0x3A,		// 58 CAPS LOCK
	0x3B,		// 59 F1
	0x3C,		// 60 F2
	0x3D,		// 61 F3
	0x3E,		// 62 F4
	0x3F,		// 63 F5
	0x40,		// 64 F6
	0x41,		// 65 F7
	0x42,		// 66 F8
	0x43,		// 67 F9
	0x44,		// 68 F10
	-1,			// 69 PAUSE(not used)
	-1,			// 70 SCROLL LOCK(not used)
	0x67,		// 71 7(numeric pad)
	0x68,		// 72 8(numeric pad)
	0x69,		// 73 9(numeric pad)
	0x4A,		// 74 -(numeric pad)
	0x6A,		// 75 4(numeric pad)
	0x6B,		// 76 5(numeric pad)
	0x6C,		// 77 6(numeric pad)
	0x4E,		// 78 +(numeric pad)
	0x6D,		// 79 1(numeric pad)
	0x6E,		// 80 2(numeric pad)
	0x6F,		// 81 3(numeric pad)
	0x70,		// 82 0(numeric pad)
	0x71,		// 83 .(numeric pad)
	-1,			// 84
	-1,			// 85
	0x60,		// 86 |
	-1,			// 87 F11(not used)
	-1,			// 88 F12(not used)
	-1,			// 89
	-1,			// 90
	-1,			// 91
	-1,			// 92
	-1,			// 93
	-1,			// 94
	-1,			// 95
	-1,			// 96
	-1,			// 97
	-1,			// 98
	-1,			// 99
	-1,			// 100
	-1,			// 101
	-1,			// 102
	-1,			// 103
	-1,			// 104
	-1,			// 105
	-1,			// 106
	-1,			// 107
	-1,			// 108
	-1,			// 109
	-1,			// 110
	-1,			// 111
	-1,			// 112
	-1,			// 113
	-1,			// 114
	-1,			// 115
	-1,			// 116
	-1,			// 117
	-1,			// 118
	-1,			// 119
	-1,			// 120
	-1,			// 121
	-1,			// 122
	-1,			// 123
	-1,			// 124
	-1,			// 125
	-1,			// 126
	-1,			// 127

// ST Code     EXTENDED PC Code
	-1,			// 0
	-1,			// 1
	-1,			// 2
	-1,			// 3
	-1,			// 4
	-1,			// 5
	-1,			// 6
	-1,			// 7
	-1,			// 8
	-1,			// 9
	-1,			// 10
	-1,			// 11
	-1,			// 12
	-1,			// 13
	-1,			// 14
	-1,			// 15
	-1,			// 16
	-1,			// 17
	-1,			// 18
	-1,			// 19
	-1,			// 20
	-1,			// 21
	-1,			// 22
	-1,			// 23
	-1,			// 24
	-1,			// 25
	-1,			// 26
	-1,			// 27
	0x72,		// 28 ENTER(numeric pad)
	0x1D,		// 29 RIGHT CTRL
	-1,			// 30
	-1,			// 31
	-1,			// 32
	-1,			// 33
	-1,			// 34
	-1,			// 35
	-1,			// 36
	-1,			// 37
	-1,			// 38
	-1,			// 39
	-1,			// 40
	-1,			// 41
	-1,			// 42
	-1,			// 43
	-1,			// 44
	-1,			// 45
	-1,			// 46
	-1,			// 47
	-1,			// 48
	-1,			// 49
	-1,			// 50
	-1,			// 51
	-1,			// 52
	0x65,		// 53 /
	-1,			// 54
	-1,			// 55
	-1,			// 56
	-1,			// 57
	-1,			// 58
	-1,			// 59
	-1,			// 60
	-1,			// 61
	-1,			// 62
	-1,			// 63
	-1,			// 64
	-1,			// 65
	-1,			// 66
	-1,			// 67
	-1,			// 68
	-1,			// 69 NUM LOCK(not used)
	-1,			// 70
	0x47,		// 71 HOME
	0x48,		// 72 CURSOR UP
	0x62,		// 73 PAGE UP(Help)
	-1,			// 74
	0x4B,		// 75 CURSOR LEFT
	-1,			// 76
	0x4D,		// 77 CURSOR RIGHT
	-1,			// 78
	-1,			// 79 END
	0x50,		// 80 CURSOR DOWN
	0x61,		// 81 PAGE DOWN(Undo)
	0x52,		// 82 INSERT
	0x53,		// 83 DELETE
	-1,			// 84
	-1,			// 85
	-1,			// 86
	-1,			// 87
	-1,			// 88
	-1,			// 89
	-1,			// 90
	-1,			// 91
	-1,			// 92
	-1,			// 93
	-1,			// 94
	-1,			// 95
	-1,			// 96
	-1,			// 97
	-1,			// 98
	-1,			// 99
	-1,			// 100
	-1,			// 101
	-1,			// 102
	-1,			// 103
	-1,			// 104
	-1,			// 105
	-1,			// 106
	-1,			// 107
	-1,			// 108
	-1,			// 109
	-1,			// 110
	-1,			// 111
	-1,			// 112
	-1,			// 113
	-1,			// 114
	-1,			// 115
	-1,			// 116
	-1,			// 117
	-1,			// 118
	-1,			// 119
	-1,			// 120
	-1,			// 121
	-1,			// 122
	-1,			// 123
	-1,			// 124
	-1,			// 125
	-1,			// 126
	-1,			// 127
};

char Loaded_WindowsKeyToSTScanCode[256];
BOOL bRemapKeyLoaded=FALSE;

//-----------------------------------------------------------------------
/*
	Remap Windows Key(0..255) to ST Scan code
*/
char Keymap_RemapWindowsKeyToSTScanCode(unsigned int Key)
{
	// Use default or loaded?
	if (bRemapKeyLoaded)
		return(Loaded_WindowsKeyToSTScanCode[Key]);
	else
		return(Default_WindowsKeyToSTScanCode[Key]);
}

//-----------------------------------------------------------------------
/*
	Load keyboard remap file
*/
void Keymap_LoadRemapFile(char *pszFileName)
{
	char szString[1024];
	unsigned int STScanCode,PCScanCode;
	ifstream in;

	// Default to not loaded
	bRemapKeyLoaded = FALSE;
	Memory_Set(Loaded_WindowsKeyToSTScanCode,-1,sizeof(Loaded_WindowsKeyToSTScanCode));

	// Attempt to load file
	if (strlen(pszFileName)>0) {
		// Open file
		in.open(pszFileName);
		if (in) {
			while(!in.eof()) {
				// Read line from file
				in.getline(szString,sizeof(szString));
				// Remove white-space from start of line
				Misc_RemoveWhiteSpace(szString,sizeof(szString));
				if (strlen(szString)>0) {
					// Is a comment?
					if ( (szString[0]==';') || (szString[0]=='#') )
						continue;
					// Read values
					sscanf(szString,"%d,%d",&STScanCode,&PCScanCode);
					// Store into remap table, check both value within range
					if ( (PCScanCode>=0) && (PCScanCode<256) && (STScanCode>=0) && (STScanCode<256) )
						Loaded_WindowsKeyToSTScanCode[PCScanCode] = STScanCode;
				}
			}
			// Loaded OK
			bRemapKeyLoaded = TRUE;

			in.close();
		}
	}
}
