/*
 * kbd.h - Keyboard emulation.
 *
 * Written by
 *  Ettore Perazzoli <ettore@comm2000.it>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#ifndef _KBD_H
#define _KBD_H

#include "types.h"

extern void kbd_arch_init(void);

extern signed long kbd_arch_keyname_to_keynum(char *keyname);
extern const char *kbd_arch_keynum_to_keyname(signed long keynum);



#define VK_BACK           0x08
#define VK_TAB            0x09

 /*
 * 0x0A - 0x0B : reserved
 */

#define VK_CLEAR          0x0C
#define VK_RETURN         0x0D
#define VK_SHIFT          0x10
#define VK_CONTROL        0x11
#define VK_MENU           0x12
#define VK_PAUSE          0x13
#define VK_CAPITAL        0x14

#define VK_KANA           0x15
#define VK_HANGEUL        0x15  /* old name - should be here for compatibility */
#define VK_HANGUL         0x15
#define VK_JUNJA          0x17
#define VK_FINAL          0x18
#define VK_HANJA          0x19
#define VK_KANJI          0x19

#define VK_ESCAPE         0x1B

#define VK_CONVERT        0x1C
#define VK_NONCONVERT     0x1D
#define VK_ACCEPT         0x1E
#define VK_MODECHANGE     0x1F

#define VK_SPACE          0x20
#define VK_PRIOR          0x21
#define VK_NEXT           0x22
#define VK_END            0x23
#define VK_HOME           0x24
#define VK_LEFT           0x25
#define VK_UP             0x26
#define VK_RIGHT          0x27
#define VK_DOWN           0x28
#define VK_SELECT         0x29
#define VK_PRINT          0x2A
#define VK_EXECUTE        0x2B
#define VK_SNAPSHOT       0x2C
#define VK_INSERT         0x2D
#define VK_DELETE         0x2E
#define VK_HELP           0x2F

/*
 * VK_0 - VK_9 are the same as ASCII '0' - '9' (0x30 - 0x39)
 * 0x40 : unassigned
 * VK_A - VK_Z are the same as ASCII 'A' - 'Z' (0x41 - 0x5A)
 */

#define VK_LWIN           0x5B
#define VK_RWIN           0x5C
#define VK_APPS           0x5D

/*
 * 0x5E : reserved
 */

#define VK_SLEEP          0x5F

#define VK_NUMPAD0        0x60
#define VK_NUMPAD1        0x61
#define VK_NUMPAD2        0x62
#define VK_NUMPAD3        0x63
#define VK_NUMPAD4        0x64
#define VK_NUMPAD5        0x65
#define VK_NUMPAD6        0x66
#define VK_NUMPAD7        0x67
#define VK_NUMPAD8        0x68
#define VK_NUMPAD9        0x69
#define VK_MULTIPLY       0x6A
#define VK_ADD            0x6B
#define VK_SEPARATOR      0x6C
#define VK_SUBTRACT       0x6D
#define VK_DECIMAL        0x6E
#define VK_DIVIDE         0x6F
#define VK_F1             0x70
#define VK_F2             0x71
#define VK_F3             0x72
#define VK_F4             0x73
#define VK_F5             0x74
#define VK_F6             0x75
#define VK_F7             0x76
#define VK_F8             0x77
#define VK_F9             0x78
#define VK_F10            0x79
#define VK_F11            0x7A
#define VK_F12            0x7B
#define VK_F13            0x7C
#define VK_F14            0x7D
#define VK_F15            0x7E
#define VK_F16            0x7F
#define VK_F17            0x80
#define VK_F18            0x81
#define VK_F19            0x82
#define VK_F20            0x83
#define VK_F21            0x84
#define VK_F22            0x85
#define VK_F23            0x86
#define VK_F24            0x87

/*
 * 0x88 - 0x8F : unassigned
 */

#define VK_NUMLOCK        0x90
#define VK_SCROLL         0x91

/*
 * NEC PC-9800 kbd definitions
 */
#define VK_OEM_NEC_EQUAL  0x92   // '=' key on numpad

/*
 * Fujitsu/OASYS kbd definitions
 */
#define VK_OEM_FJ_JISHO   0x92   // 'Dictionary' key
#define VK_OEM_FJ_MASSHOU 0x93   // 'Unregister word' key
#define VK_OEM_FJ_TOUROKU 0x94   // 'Register word' key
#define VK_OEM_FJ_LOYA    0x95   // 'Left OYAYUBI' key
#define VK_OEM_FJ_ROYA    0x96   // 'Right OYAYUBI' key

/*
 * 0x97 - 0x9F : unassigned
 */

/*
 * VK_L* & VK_R* - left and right Alt, Ctrl and Shift virtual keys.
 * Used only as parameters to GetAsyncKeyState() and GetKeyState().
 * No other API or message will distinguish left and right keys in this way.
 */
#define VK_LSHIFT         0xA0
#define VK_RSHIFT         0xA1
#define VK_LCONTROL       0xA2
#define VK_RCONTROL       0xA3
#define VK_LMENU          0xA4
#define VK_RMENU          0xA5

#define VK_BROWSER_BACK        0xA6
#define VK_BROWSER_FORWARD     0xA7
#define VK_BROWSER_REFRESH     0xA8
#define VK_BROWSER_STOP        0xA9
#define VK_BROWSER_SEARCH      0xAA
#define VK_BROWSER_FAVORITES   0xAB
#define VK_BROWSER_HOME        0xAC

#define VK_VOLUME_MUTE         0xAD
#define VK_VOLUME_DOWN         0xAE
#define VK_VOLUME_UP           0xAF
#define VK_MEDIA_NEXT_TRACK    0xB0
#define VK_MEDIA_PREV_TRACK    0xB1
#define VK_MEDIA_STOP          0xB2
#define VK_MEDIA_PLAY_PAUSE    0xB3
#define VK_LAUNCH_MAIL         0xB4
#define VK_LAUNCH_MEDIA_SELECT 0xB5
#define VK_LAUNCH_APP1         0xB6
#define VK_LAUNCH_APP2         0xB7

/*
 * 0xB8 - 0xB9 : reserved
 */

#define VK_OEM_1          0xBA   // ';:' for US
#define VK_OEM_PLUS       0xBB   // '+' any country
#define VK_OEM_COMMA      0xBC   // ',' any country
#define VK_OEM_MINUS      0xBD   // '-' any country
#define VK_OEM_PERIOD     0xBE   // '.' any country
#define VK_OEM_2          0xBF   // '/?' for US
#define VK_OEM_3          0xC0   // '`~' for US

/*
 * 0xC1 - 0xD7 : reserved
 */

/*
 * 0xD8 - 0xDA : unassigned
 */

#define VK_OEM_4          0xDB  //  '[{' for US
#define VK_OEM_5          0xDC  //  '\|' for US
#define VK_OEM_6          0xDD  //  ']}' for US
#define VK_OEM_7          0xDE  //  ''"' for US
#define VK_OEM_8          0xDF





#define KBD_C64_SYM   "win_sym.vkm"
#define KBD_C64_POS   "Z:\\win_pos.vkm"
#define KBD_C128_SYM  "win_sym.vkm"
#define KBD_C128_POS  "win_pos.vkm"
#define KBD_VIC20_SYM "win_sym.vkm"
#define KBD_VIC20_POS "Z:\\win_pos.vkm"
#define KBD_PET_BUKS  "Z:\\win_buks.vkm"
#define KBD_PET_BUKP  "win_bukp.vkm"
#define KBD_PET_BDES  "win_bdes.vkm"
#define KBD_PET_BDEP  "win_bdep.vkm"
#define KBD_PET_BGRS  "Z:\\win_bgrs.vkm"
#define KBD_PET_BGRP  "win_bgrp.vkm"
#define KBD_PLUS4_SYM "win_sym.vkm"
#define KBD_PLUS4_POS "win_pos.vkm"
#define KBD_CBM2_BUKS "win_buks.vkm"
#define KBD_CBM2_BUKP "win_bukp.vkm"
#define KBD_CBM2_BDES "win_bdes.vkm"
#define KBD_CBM2_BDEP "win_bdep.vkm"
#define KBD_CBM2_BGRS "win_bgrs.vkm"
#define KBD_CBM2_BGRP "win_bgrp.vkm"

#define KBD_INDEX_C64_DEFAULT   KBD_INDEX_C64_POS
#define KBD_INDEX_C128_DEFAULT  KBD_INDEX_C128_POS
#define KBD_INDEX_VIC20_DEFAULT KBD_INDEX_VIC20_POS
#define KBD_INDEX_PET_DEFAULT   KBD_INDEX_PET_BUKP
#define KBD_INDEX_PLUS4_DEFAULT KBD_INDEX_PLUS4_POS
#define KBD_INDEX_CBM2_DEFAULT  KBD_INDEX_CBM2_BUKP

/* Keymap definition structure.  */
typedef struct {
    BYTE row;
    BYTE column;
    int vshift;
} keyconv;


#define C64K_Run_Stop 1
#define C64K_1 2
#define C64K_2 3
#define C64K_3 4
#define C64K_4 5
#define C64K_5 6
#define C64K_6 7
#define C64K_7 8
#define C64K_8 9
#define C64K_9 10
#define C64K_0 11
#define C64K_Plus 12
#define C64K_Minus 13
#define C64K_Del 14
#define C64K_Ctrl 15
#define C64K_Q 16
#define C64K_W 17
#define C64K_E 18
#define C64K_R 19
#define C64K_T 20
#define C64K_Y 21
#define C64K_U 22
#define C64K_I 23
#define C64K_O 24
#define C64K_P 25
#define C64K_AT_SIGN 26
#define C64K_STAR 27
#define C64K_Return 28
#define C64K_CBM 29
#define C64K_A 30
#define C64K_S 31
#define C64K_D 32
#define C64K_F 33
#define C64K_G 34
#define C64K_H 35
#define C64K_J 36
#define C64K_K 37
#define C64K_L 38
#define C64K_COLON 39
#define C64K_SEMICOLON 40
#define C64K_Left_Arrow 41
#define C64K_Left_Shift 42
#define C64K_EQUALS 43
#define C64K_Z 44
#define C64K_X 45
#define C64K_C 46
#define C64K_V 47
#define C64K_B 48
#define C64K_N 49
#define C64K_M 50
#define C64K_COMMA 51
#define C64K_PERIOD 52
#define C64K_SLASH 53
#define C64K_Right_Shift 54
//#define C64K_STAR 55
#define C64K_Space 57
//#define C64K_Run_Stop 58
#define C64K_F1 59
#define C64K_F2 60
#define C64K_F3 61
#define C64K_F4 62
#define C64K_F5 63
#define C64K_F6 64
#define C64K_F7 65
#define C64K_F8 66
#define C64K_CLR_HOME 89
#define C64K_CRSR_UP 90
#define C64K_RESTORE 91
#define C64K_CRSR_LEFT 92
#define C64K_CRSR_RIGHT 93
#define C64K_CRSR_DOWN 95
#define C64K_Pound 97
#define C64K_Up_Arrow 98



/* Warning: this might be not 100% correct.  */
typedef enum {
    K_NONE, K_ESC, K1, K_2, K_3, K_4, K_5, K_6, K_7, K_8, K_9, K_0, K_MINUS,
    K_EQUAL, K_BS, K_TAB, K_Q, K_W, K_E, K_R, K_T, K_Y, K_U, K_I, K_O, K_P,
    K_LEFTBR, K_RIGHTBR, K_ENTER, K_LEFTCTRL, K_A, K_S, K_D, K_F, K_G, K_H,
    K_J, K_K, K_L, K_SEMICOLON, K_GRAVE, K_NUMSGN, K_LEFTSHIFT, K_LTGT, K_Z,
    K_X, K_C, K_V, K_B, K_N, K_M, K_COMMA, K_PERIOD, K_SLASH, K_RIGHTSHIFT,
    K_KPMULT, K_LEFTALT, K_SPACE, K_CAPSLOCK, K_F1, K_F2, K_F3, K_F4, K_F5,
    K_F6, K_F7, K_F8, K_F9, K_F10, K_NUMLOCK, K_SCROLLOCK, K_KP7, K_KP8, K_KP9,
    K_KPMINUS, K_KP4, K_KP5, K_KP6, K_KPPLUS, K_KP1, K_KP2, K_KP3, K_KP0,
    K_KPDOT, K_SYSREQ, K_85, K_86, K_F11, K_F12, K_HOME, K_UP, K_PGUP, K_LEFT,
    K_RIGHT, K_END, K_DOWN, K_PGDOWN, K_INS, K_DEL, K_KPENTER, K_RIGHTCTRL,
    K_PAUSE, K_PRTSCR, K_KPDIV, K_RIGHTALT, K_BREAK, K_LEFTW95, K_RIGHTW95
} kbd_code_t;

extern BYTE _kbd_extended_key_tab[];

extern int kbd_init(int num, ...);
extern int kbd_handle_keydown(DWORD virtual_key, DWORD key_data);
extern int kbd_handle_keyup(DWORD virtual_key, DWORD key_data);
extern const char *kbd_code_to_string(kbd_code_t kcode);

#endif

