/*
 * lib.c - Library functions.
 *
 * Written by
 *  Andreas Boose <viceteam@t-online.de>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* #define LIB_DEBUG */


#ifdef LIB_DEBUG
#define LIB_DEBUG_SIZE 0x10000

static void *lib_debug_address[LIB_DEBUG_SIZE];
static unsigned int lib_debug_size[LIB_DEBUG_SIZE];
static void *lib_debug_caller[LIB_DEBUG_SIZE];
static unsigned int lib_debug_initialized = 0;

#ifdef __GNUC__
static void *lib_debug_func_level1(void)
{
    return __builtin_return_address(1 + 1);
}

static void *lib_debug_func_level2(void)
{
    return __builtin_return_address(2 + 1);
}

static void *lib_debug_func_level3(void)
{
    return __builtin_return_address(3 + 1);
}
#endif

static void lib_debug_alloc(void *ptr, size_t size, int level)
{
    unsigned int index;
    void *func = NULL;

    if (!lib_debug_initialized) {
        memset(lib_debug_address, 0, sizeof(lib_debug_address));
        memset(lib_debug_caller, 0, sizeof(lib_debug_caller));
        lib_debug_initialized = 1;
    }

    index = 0;

    while (index < LIB_DEBUG_SIZE && lib_debug_address[index] != NULL)
        index++;

    if (index == LIB_DEBUG_SIZE) {
        printf("lib_debug_alloc(): Out of debug address slots.");
        return;
    }    

#ifdef __GNUC__
    switch (level) {
      case 1:
        func = lib_debug_func_level1();
        break;
      case 2:
        func = lib_debug_func_level2();
        break;
      case 3:
        func = lib_debug_func_level3();
        break;
    }
    lib_debug_caller[index] = func;
#endif
#if 0
    printf("lib_debug_alloc(): Alloc address %p size %i slot %i from %p.\n",
           ptr, size, index, func);
#endif
    lib_debug_address[index] = ptr;
    lib_debug_size[index] = (unsigned int)size;
}

static void lib_debug_free(void *ptr, unsigned int level, unsigned int fill)
{
    unsigned int index;
    void *func = NULL;

    if (ptr == NULL)
        return;

    index = 0;

    while (index < LIB_DEBUG_SIZE && lib_debug_address[index] != ptr)
        index++;

    if (index == LIB_DEBUG_SIZE) {
#if 0
        printf("lib_debug_free(): Cannot find debug address!");
#endif
        return;
    }

#ifdef __GNUC__
    switch (level) {
      case 1:
        func = lib_debug_func_level1();
        break;
      case 2:
        func = lib_debug_func_level2();
        break;
      case 3:
        func = lib_debug_func_level3();
        break;
    }
#endif

#if 0
    printf("lib_debug_free(): Free address %p size %i slot %i from %p.\n",
           ptr, lib_debug_size[index], index, func);
#endif

    if (fill)
        memset(ptr, 0xdd, lib_debug_size[index]);

    lib_debug_address[index] = NULL;
}
#endif

/*-----------------------------------------------------------------------*/

void lib_debug_check(void)
{
#ifdef LIB_DEBUG
    unsigned int index, count;

    count = 0;

    for (index = 0; index < LIB_DEBUG_SIZE; index++) {
        if (lib_debug_address[index] != NULL)
            printf("Memory leak %i at %p with size %i from %p.\n", ++count,
                   lib_debug_address[index], lib_debug_size[index],
                   lib_debug_caller[index]);
    }
#endif
}

/*-----------------------------------------------------------------------*/

void *lib_malloc(size_t size)
{
    void *ptr = malloc(size);

#ifndef __OS2__
    if (ptr == NULL && size > 0)
        exit(-1);
#endif
#ifdef LIB_DEBUG
    lib_debug_alloc(ptr, size, 1);
#endif

    return ptr;
}

/* Like calloc, but abort if not enough memory is available.  */
void *lib_calloc(size_t nmemb, size_t size)
{
    void *ptr = calloc(nmemb, size);

#ifndef __OS2__
    if (ptr == NULL && (size * nmemb) > 0)
        exit(-1);
#endif
#ifdef LIB_DEBUG
    lib_debug_alloc(ptr, size * nmemb, 1);
#endif

    return ptr;
}

/* Like realloc, but abort if not enough memory is available.  */
void *lib_realloc(void *ptr, size_t size)
{
    void *new_ptr = realloc(ptr, size);

#ifndef __OS2__
    if (new_ptr == NULL)
        exit(-1);
#endif
#ifdef LIB_DEBUG
    lib_debug_free(ptr, 1, 0);
    lib_debug_alloc(new_ptr, size, 1);
#endif

    return new_ptr;
}

void lib_free(void *ptr)
{
#ifdef LIB_DEBUG
    lib_debug_free(ptr, 1, 1);
#endif

    free(ptr);
}

/*-----------------------------------------------------------------------*/

/* Malloc enough space for `str', copy `str' into it and return its
   address.  */
char *lib_stralloc(const char *str)
{
    size_t size;
    char *ptr;

    if (str == NULL)
        exit(-1);

    size = strlen(str) + 1;
    ptr = (char *)malloc(size);

#ifdef LIB_DEBUG
    lib_debug_alloc(ptr, size, 1);
#endif

    memcpy(ptr, str, size);
    return ptr;
}

/* xmsprintf() is like sprintf() but lib_malloc's the buffer by itself.  */

#define xmvsprintf_is_digit(c) ((c) >= '0' && (c) <= '9')

static int xmvsprintf_skip_atoi(const char **s)
{
    int i = 0;

    while (xmvsprintf_is_digit(**s))
        i = i * 10 + *((*s)++) - '0';
    return i;
}

#define ZEROPAD 1               /* pad with zero */
#define SIGN    2               /* unsigned/signed long */
#define PLUS    4               /* show plus */
#define SPACE   8               /* space if plus */
#define LEFT    16              /* left justified */
#define SPECIAL 32              /* 0x */
#define LARGE   64              /* use 'ABCDEF' instead of 'abcdef' */

static inline int xmvsprintf_do_div(long *n, unsigned int base)
{
    int res;

    res = ((unsigned long)*n) % (unsigned)base;
    *n = ((unsigned long)*n) / (unsigned)base;
    return res;
}

static size_t xmvsprintf_strnlen(const char * s, size_t count)
{
    const char *sc;

    for (sc = s; count-- && *sc != '\0'; ++sc)
        /* nothing */;
    return sc - s;
}

static void xmvsprintf_add(char **buf, unsigned int *bufsize,
                           unsigned int *position, char write)
{
    if (*position == *bufsize) {
        *bufsize *= 2;
        *buf = lib_realloc(*buf, *bufsize);
    }
    (*buf)[*position] = write;
    *position += 1;
}

static void xmvsprintf_number(char **buf, unsigned int *bufsize,
                              unsigned int *position, long num, int base,
                              int size, int precision, int type)
{
    char c, sign, tmp[66];
    const char *digits = "0123456789abcdefghijklmnopqrstuvwxyz";
    int i;

    if (type & LARGE)
        digits = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    if (type & LEFT)
        type &= ~ZEROPAD;
    if (base < 2 || base > 36)
        return;
    c = (type & ZEROPAD) ? '0' : ' ';
    sign = 0;
    if (type & SIGN) {
        if (num < 0) {
            sign = '-';
            num = -num;
            size--;
        } else if (type & PLUS) {
            sign = '+';
            size--;
        } else if (type & SPACE) {
            sign = ' ';
            size--;
        }
    }
    if (type & SPECIAL) {
        if (base == 16)
            size -= 2;
        else if (base == 8)
            size--;
    }

    i = 0;
    if (num == 0)
        tmp[i++] = '0';
    else while (num != 0)
        tmp[i++] = digits[xmvsprintf_do_div(&num, base)];
    if (i > precision)
        precision = i;
        size -= precision;
    if (!(type & (ZEROPAD + LEFT)))
        while(size-->0)
            xmvsprintf_add(buf, bufsize, position, ' ');
    if (sign)
        xmvsprintf_add(buf, bufsize, position, sign);
    if (type & SPECIAL) {
        if (base == 8)
            xmvsprintf_add(buf, bufsize, position, '0');
        else if (base == 16) {
            xmvsprintf_add(buf, bufsize, position, '0');
            xmvsprintf_add(buf, bufsize, position, digits[33]);
        }
    }
    if (!(type & LEFT))
        while (size-- > 0)
            xmvsprintf_add(buf, bufsize, position, c);
    while (i < precision--)
        xmvsprintf_add(buf, bufsize, position, '0');
    while (i-- > 0)
        xmvsprintf_add(buf, bufsize, position, tmp[i]);
    while (size-- > 0)
        xmvsprintf_add(buf, bufsize, position, ' ');
}

char *lib_mvsprintf(const char *fmt, va_list args)
{
    char *buf;
    unsigned int position, bufsize;

    int len, i, base;
    unsigned long num;
    const char *s;

    int flags;        /* flags to number() */
    int field_width;  /* width of output field */
    int precision;    /* min. # of digits for integers; max
                         number of chars for from string */
    int qualifier;    /* 'h', 'l', or 'L' for integer fields */

    /* Setup the initial buffer.  */
    buf = lib_malloc(10);
    position = 0;
    bufsize = 10;

    for ( ; *fmt ; ++fmt) {
        if (*fmt != '%') {
            xmvsprintf_add(&buf, &bufsize, &position, *fmt);
            continue;
        }

        /* process flags */
        flags = 0;
repeat:
        ++fmt;  /* this also skips first '%' */
        switch (*fmt) {
          case '-':
            flags |= LEFT;
            goto repeat;
          case '+':
            flags |= PLUS;
            goto repeat;
          case ' ':
            flags |= SPACE;
            goto repeat;
          case '#':
            flags |= SPECIAL;
            goto repeat;
          case '0':
            flags |= ZEROPAD;
            goto repeat;
        }

        /* get field width */
        field_width = -1;
        if (xmvsprintf_is_digit(*fmt))
            field_width = xmvsprintf_skip_atoi(&fmt);
        else if (*fmt == '*') {
            ++fmt;
            /* it's the next argument */
            field_width = va_arg(args, int);
            if (field_width < 0) {
                field_width = -field_width;
                flags |= LEFT;
            }
        }

        /* get the precision */
        precision = -1;
        if (*fmt == '.') {
            ++fmt;
            if (xmvsprintf_is_digit(*fmt))
                precision = xmvsprintf_skip_atoi(&fmt);
            else if (*fmt == '*') {
                ++fmt;
                 /* it's the next argument */
                 precision = va_arg(args, int);
            }
            if (precision < 0)
                precision = 0;
        }

        /* get the conversion qualifier */
        qualifier = -1;
        if (*fmt == 'h' || *fmt == 'l' || *fmt == 'L') {
            qualifier = *fmt;
            ++fmt;
        }

        /* default base */
        base = 10;

        switch (*fmt) {
          case 'c':
            if (!(flags & LEFT))
                while (--field_width > 0)
                    xmvsprintf_add(&buf, &bufsize, &position, ' ');
            xmvsprintf_add(&buf, &bufsize, &position,
                           (unsigned char) va_arg(args, int));
            while (--field_width > 0)
                xmvsprintf_add(&buf, &bufsize, &position, ' ');
            continue;

          case 's':
            s = va_arg(args, char *);
            if (!s)
                s = "<NULL>";

            len = xmvsprintf_strnlen(s, precision);

            if (!(flags & LEFT))
                while (len < field_width--)
                    xmvsprintf_add(&buf, &bufsize, &position, ' ');
            for (i = 0; i < len; ++i)
                xmvsprintf_add(&buf, &bufsize, &position, *s++);
            while (len < field_width--)
                xmvsprintf_add(&buf, &bufsize, &position, ' ');
            continue;

          case 'p':
            if (field_width == -1) {
                field_width = 2*sizeof(void *);
                flags |= ZEROPAD;
            }
            xmvsprintf_number(&buf, &bufsize, &position,
                              (unsigned long) va_arg(args, void *), 16,
                              field_width, precision, flags);
            continue;

          case '%':
            xmvsprintf_add(&buf, &bufsize, &position, '%');
            continue;

          /* integer number formats - set up the flags and "break" */
          case 'o':
            base = 8;
            break;
          case 'X':
            flags |= LARGE;
          case 'x':
            base = 16;
            break;
          case 'd':
          case 'i':
            flags |= SIGN;
          case 'u':
            break;

          default:
            xmvsprintf_add(&buf, &bufsize, &position, '%');
            if (*fmt)
                xmvsprintf_add(&buf, &bufsize, &position, *fmt);
            else
                --fmt;
            continue;
        }
        if (qualifier == 'l')
            num = va_arg(args, unsigned long);
        else if (qualifier == 'h') {
            num = (unsigned short) va_arg(args, int);
            if (flags & SIGN)
            num = (short) num;
        } else if (flags & SIGN)
            num = va_arg(args, int);
        else
            num = va_arg(args, unsigned int);
        xmvsprintf_number(&buf, &bufsize, &position, num, base, field_width,
                          precision, flags);
    }
    xmvsprintf_add(&buf, &bufsize, &position, '\0');

    /* Trim buffer to final size.  */
    buf = lib_realloc(buf, strlen(buf) + 1);

    return buf;
}

char *lib_msprintf(const char *fmt, ...)
{
    va_list args;
    char *buf;

    va_start(args, fmt);
    buf = lib_mvsprintf(fmt, args);
    va_end(args);

    return buf;
}

