/*

	psms.c - PSMS main source file. Contains PSMS core.

	                 (c) Nick Van Veen (aka Sjeep), 2002

	-------------------------------------------------------------------------

    This file is part of the PSMS.

    PSMS is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    PSMS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Foobar; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <tamtypes.h>
#include <string.h>
#include <kernel.h>
#include <sifrpc.h>
#include <stdlib.h>
#include <stdio.h>
#include "hw.h"
#include "gs.h"
#include "gfxpipe.h"
#include "shared.h"
#include "pad.h"
#include "psms.h"
#include "menu.h"
#include "sjpcm.h"

#define SND_RATE	48000

u8 bitmap_data[256 * 256] __attribute__((aligned(16))) __attribute__ ((section (".bss"))); // SMS display is only 256x192, extra data is padding
u16 clut[256] __attribute__((aligned(16))) __attribute__ ((section (".bss")));

char padBuf1[256] __attribute__((aligned(64))) __attribute__ ((section (".bss")));
char padBuf2[256] __attribute__((aligned(64))) __attribute__ ((section (".bss")));

u8 base_rom[1048576] __attribute__((aligned(64))) __attribute__ ((section (".bss")));

u8 *state;
int state_saved = 0;

int whichdrawbuf = 0;
int endflag = 0;
int sound = 1;
int snd_sample = 0;

int dispx, dispy;

int main()
{
	char rom_filename[256];

	init_machine();
//	display_error2("PS2 initialized.");
	system_init(SND_RATE);
//	display_error2("SMS initialized.");
	ProcessROMlist();
//	display_error2("ROM list parsed.");

	while(1) {

#ifdef CD_BUILD
		strcpy(rom_filename,"cdrom0:\\");
		strcat(rom_filename,menu_main());
		strcat(rom_filename,";1");
#else
		strcpy(rom_filename,"host:");
		strcat(rom_filename,menu_main());
#endif

		if(load_rom(rom_filename)) continue; // if error loading file, start loop again (ie, go back to menu)

		system_reset();

#ifdef DEVEL
		if(snd.enabled) printf("Sound enabled!\n");
		else printf("Failure initialising sound.\n");
#endif

  		if(sound) {
			SjPCM_Clearbuff();
			SjPCM_Play();
		}

		while(1)
    	{
        	/* Get current input */
			update_input();

			if(endflag) break;

        	/* Run the SMS emulation for a frame */
        	sms_frame(0);

			if(sound) SjPCM_Enqueue(snd.buffer[0], snd.buffer[1], snd_sample, 1);

        	/* Update the display if desired */
			update_video();
    	}

		if(sound) SjPCM_Pause();

		system_reset();
		endflag = 0;
		state_saved = 0;
	}
}

int init_machine()
{
	sif_rpc_init(0);
#ifdef CD_BUILD
	sif_reset_iop();
	sif_rpc_init(0);
#endif

	DmaReset();

	if(pal_ntsc() == 3) {
		GS_InitGraph(GS_PAL,GS_NONINTERLACE);
		dispx = 65;
		dispy = 40;
		GS_SetDispMode(dispx,dispy,WIDTH,HEIGHT);
		snd_sample = SND_RATE / 50;
	} else {
		GS_InitGraph(GS_NTSC,GS_NONINTERLACE);
		dispx = 65;
		dispy = 17;
		GS_SetDispMode(dispx,dispy,WIDTH,HEIGHT);
		snd_sample = SND_RATE / 60;
	}

	GS_SetEnv(WIDTH, HEIGHT, 0, 0x40000, GS_PSMCT32, 0x80000, GS_PSMZ16S);
    install_VRstart_handler();
	createGfxPipe(&thegp, (void *)0xF00000, 0x40000);

	// Upload texture data to VRAM
	gp_uploadTexture(&thegp, LOGO_TEX, 256, 0, 0, GS_PSMT8, psms_image, psms_width, psms_height);
	gp_uploadTexture(&thegp, LOGO_CLUT, 256, 0, 0, GS_PSMCT16, psms_clut, 256, 1);
	gp_uploadTexture(&thegp, FONT_TEX, 256, 0, 0, GS_PSMT8, vixar_image, vixar_width, vixar_height);
	gp_uploadTexture(&thegp, FONT_CLUT, 256, 0, 0, GS_PSMCT16, vixar_clut, 256, 1);
	gp_hardflush(&thegp);

//	display_error2("Texture data uploaded to VRAM.");

	LoadModules();

//	display_error2("Modules loaded.");

	padInit(0);

//	display_error2("Pad init 1.");

	InitPad(0, 0, padBuf1);
	InitPad(1, 0, padBuf2);

//	display_error2("Pad init 2.");

	/* Initialize display bitmap */
    memset(&bitmap, 0, sizeof(t_bitmap));
    bitmap.width  = 256;
    bitmap.height = 192;
    bitmap.depth  = 8;
    bitmap.pitch  = (bitmap.width * (bitmap.depth >> 3));
    bitmap.data   = (unsigned char *)&bitmap_data[0];

	memset(&base_rom,0,1048576);


	state = malloc(sizeof(t_vdp) + sizeof(t_sms) + sizeof(Z80_Regs) + sizeof(int) + sizeof(t_SN76496) + 0x40);
	if(state == NULL) {
		display_error("Failed to allocate memory!", 1);
	}

//	display_error2("SMS bitmap initialized.");

	if(SjPCM_Init() < 0) display_error("SjPCM Bind failed!!", 1);

//	display_error2("SjPCM Initialized.");

}

int load_rom(char* filename)
{

	int fd,fd_size;
	int i;
#ifdef CD_BUILD
	for(i = 0;i<strlen(filename);i++) {
		if(filename[i] == '/') filename[i] = '\\';
	}
#endif

#ifdef DEVEL
	printf("Filename: %s\n",filename);
#endif

	fd = fio_open(filename, O_RDONLY);
	if(fd <= 0) {
		display_error("Error opening file.",0);
		return 1;
	}

	fd_size = fio_lseek(fd,0,SEEK_END);
	fio_lseek(fd,0,SEEK_SET);

	if(fd_size%32768) { // skip 512k header if necessary
		fio_lseek(fd,512,SEEK_SET);
		fd_size -= 512;
	}

	if(fio_read(fd, &base_rom, fd_size) != fd_size) {
		display_error("Short file.",0);
		return 1;
	}

	fio_close(fd);

	if(strstr(filename, ".GG") != NULL)
		cart.type = TYPE_GG;
	else
		cart.type = TYPE_SMS;

	cart.rom = (char *)&base_rom;
	cart.pages = (fd_size / 0x4000);

	return 0;

}

void update_video()
{

	gp_uploadTexture(&thegp, SMS_TEX, 256, 0, 0, GS_PSMT8, &bitmap_data, 256, 256);
	gp_uploadTexture(&thegp, SMS_CLUT, 256, 0, 0, GS_PSMCT16, &clut, 256, 1);
	gp_setTex(&thegp, SMS_TEX, 256, 256, 256, GS_PSMT8, SMS_CLUT, 256, GS_PSMCT16);

	if(cart.type == TYPE_SMS)
		gp_texrect(&thegp, 0, 24<<4, 0, 0, 256<<4, 216<<4, 256<<4, 192<<4, 10, GS_SET_RGBA(255, 255, 255, 200));
	else
		gp_texrect(&thegp, 48<<4,48<<4,48<<4,24<<4,208<<4,168<<4,208<<4,168<<4,10,GS_SET_RGBA(255, 255, 255, 200));

	gp_hardflush(&thegp);

	WaitForNextVRstart(1);

	// Update drawing and display enviroments.
    GS_SetCrtFB(whichdrawbuf);
    whichdrawbuf ^= 1;
    GS_SetDrawFB(whichdrawbuf);

}


void system_load_sram() {}
void fwrite() {}
void fread() {}

void LoadModules(void)
{
    int ret;

	ret = sif_load_module("rom0:SIO2MAN", 0, NULL);
	if (ret < 0) {
			display_error("Failed to load module: SIO2MAN", 1);
    }

	ret = sif_load_module("rom0:PADMAN", 0, NULL);
	if (ret < 0) {
			display_error("Failed to load module: PADMAN", 1);
    }

	ret = sif_load_module("rom0:LIBSD", 0, NULL);
	if (ret < 0) {
			display_error("Failed to load module: LIBSD", 1);
    }

#ifdef CD_BUILD
	ret = sif_load_module("cdrom0:\\SJPCM.IRX;1", 0, NULL);
#else
	ret = sif_load_module("host:SJPCM.IRX", 0, NULL);
#endif
	if (ret < 0) {
			display_error("Failed to load module: SJPCM.IRX", 1);
    }

}

int InitPad(int port, int slot, char* buffer)
{
    int ret;

	if((ret = padPortOpen(port, slot, buffer)) == 0) {
		display_error("Failed to open pad port.", 1);
    }

	while((ret=padGetState(port, slot)) != PAD_STATE_STABLE) {
        if(ret==0) return 0; // no pad connected
        WaitForNextVRstart(1);
    }

	padSetMainMode(port, slot, PAD_MMODE_DUALSHOCK, PAD_MMODE_UNLOCK);

	WAIT_PAD_READY(port, slot);

    return 0;
}

void update_input()
{
	static struct padButtonStatus pad1; // just in case
	static struct padButtonStatus pad2;
	static int pad1_connected = 0, pad2_connected = 0;
	static int p1_1t=0,p1_2t=0,p2_1t=0,p2_2t=0;
	int pad1_data = 0;
	int pad2_data = 0;

	memset(&input, 0, sizeof(t_input));

	if(pad1_connected) {
		padRead(0, 0, &pad1); // port, slot, buttons
		pad1_data = 0xffff ^ ((pad1.btns[0] << 8) | pad1.btns[1]);

		if(pad1_data & PAD_L1) p1_1t ^= 1;
		else p1_1t = 0;
		if(pad1_data & PAD_R1) p1_2t ^= 1;
		else p1_2t = 0;
		if(pad1_data & PAD_LEFT)				input.pad[0] |= INPUT_LEFT;
		if(pad1_data & PAD_RIGHT)				input.pad[0] |= INPUT_RIGHT;
		if(pad1_data & PAD_UP)					input.pad[0] |= INPUT_UP;
		if(pad1_data & PAD_DOWN)				input.pad[0] |= INPUT_DOWN;
		if((pad1_data & PAD_CIRCLE) || p1_2t)	input.pad[0] |= INPUT_BUTTON2;
		if((pad1_data & PAD_SQUARE) || p1_1t)	input.pad[0] |= INPUT_BUTTON1;
		if(pad1_data & PAD_START)				input.system |= (IS_GG) ? INPUT_START : INPUT_PAUSE;

		if((pad1.mode >> 4) == 0x07) {
			if(pad1.ljoy_h < 64) input.pad[0] |= INPUT_LEFT;
			else if(pad1.ljoy_h > 192) input.pad[0] |= INPUT_RIGHT;

			if(pad1.ljoy_v < 64) input.pad[0] |= INPUT_UP;
			else if(pad1.ljoy_v > 192) input.pad[0] |= INPUT_DOWN;
		}
	}

	if(pad2_connected) {
		padRead(1, 0, &pad2); // port, slot, buttons
		pad2_data = 0xffff ^ ((pad2.btns[0] << 8) | pad2.btns[1]);

		if(pad2_data & PAD_L1) p2_1t ^= 1;
		else p2_1t = 0;
		if(pad2_data & PAD_R1) p2_2t ^= 1;
		else p2_2t = 0;
		if(pad2_data & PAD_LEFT)				input.pad[1] |= INPUT_LEFT;
		if(pad2_data & PAD_RIGHT)				input.pad[1] |= INPUT_RIGHT;
		if(pad2_data & PAD_UP)					input.pad[1] |= INPUT_UP;
		if(pad2_data & PAD_DOWN)				input.pad[1] |= INPUT_DOWN;
		if((pad2_data & PAD_CIRCLE) || p2_2t)	input.pad[1] |= INPUT_BUTTON2;
		if((pad2_data & PAD_SQUARE) || p2_1t)	input.pad[1] |= INPUT_BUTTON1;
//		if(pad2_data & PAD_START)	input.system |= (IS_GG) ? INPUT_START : INPUT_PAUSE;

		if((pad2.mode >> 4) == 0x07) {
			if(pad2.ljoy_h < 64) input.pad[1] |= INPUT_LEFT;
			else if(pad2.ljoy_h > 192) input.pad[1] |= INPUT_RIGHT;

			if(pad2.ljoy_v < 64) input.pad[1] |= INPUT_UP;
			else if(pad2.ljoy_v > 192) input.pad[1] |= INPUT_DOWN;
		}
	}

	//check controller status
	if((padGetState(0, 0)) == PAD_STATE_STABLE) {
		if(pad1_connected == 0) {
#ifdef DEVEL
			printf("Pad 1 inserted!\n");
#endif
			WaitForNextVRstart(1);
		}
		pad1_connected = 1;
	} else pad1_connected = 0;

	if((padGetState(1, 0)) == PAD_STATE_STABLE) {
		if(pad2_connected == 0) {
#ifdef DEVEL
			printf("Pad 2 inserted!\n");
#endif
			WaitForNextVRstart(1);
		}
		pad2_connected = 1;
	} else pad2_connected = 0;

	if(pad1_data & PAD_SELECT) IngameMenu();

}

// Quick & dirty font printing code.
void TextOut(int x, int y, char *string, int z)
{
#define CHAR_W		16
#define CHAR_H		16
#define CHAR_ROW	16
#define CHAR_COL	16
	int u,v;
	int oldx = x;
	int i;
	int ncharw;

	// Set texture/clut regs
	gp_setTex(&thegp, FONT_TEX, 256, vixar_width, vixar_height, GS_PSMT8, FONT_CLUT, 256, GS_PSMCT16);

	for(i=0;i<strlen(string);i++) {

		if(string[i] == '\n') {
			y+=(CHAR_H+2)<<4;
			x = oldx;
		}else{
			u = ((int)string[i]	% CHAR_ROW) * CHAR_W;
			v = ((int)string[i] / CHAR_COL) * CHAR_H;

			ncharw = vixarmet[(int)string[i]] + 1;

			gp_texrect(&thegp, x, y, u<<4, v<<4, x+(ncharw<<4), y+(CHAR_H<<4), (u+ncharw)<<4, (v+CHAR_H)<<4, z,GS_SET_RGBA(255, 255, 255, 128));

			x+=ncharw<<4;
		}
	}
}

void TextOutC(int x_start, int x_end, int y, char *string, int z)
{
	int u,v;
	int x;
	int i;
	int ncharw;
	int total_width = 0;

	gp_setTex(&thegp, FONT_TEX, 256, vixar_width, vixar_height, GS_PSMT8, FONT_CLUT, 256, GS_PSMCT16);

	for(i=0;i<strlen(string);i++) total_width += (vixarmet[(int)string[i]] + 1);

	if((total_width<<4) > (x_end - x_start)) x = x_start;
	else x = x_start + ((x_end-x_start)/2) - ((total_width/2)<<4);

	for(i=0;i<strlen(string);i++) {
		u = ((int)string[i]	% CHAR_ROW) * CHAR_W;
		v = ((int)string[i] / CHAR_COL) * CHAR_H;

		ncharw = vixarmet[(int)string[i]] + 1;

		if(((ncharw<<4) + x) > x_end) return;

		gp_texrect(&thegp, x, y, u<<4, v<<4, x+(ncharw<<4), y+(CHAR_H<<4), (u+ncharw)<<4, (v+CHAR_H)<<4, z,GS_SET_RGBA(255, 255, 255, 128));

		x+=ncharw<<4;
	}
}

void display_error(char* errmsg, int fatal)
{
	struct padButtonStatus pad1;
	int pad1_data = 0;

	while(1) {
		gp_gouradrect(&thegp, 24<<4, 72<<4, GS_SET_RGBA(140, 0, 255, 128), 232<<4, 152<<4, GS_SET_RGBA(13, 88, 174, 128), 1);
		gp_linerect(&thegp, 23<<4, 71<<4, 232<<4, 152<<4, 2, GS_SET_RGBA(255, 255, 255, 128));
		TextOutC(0<<4,256<<4,88<<4,errmsg,3);
		if(!fatal) TextOutC(0<<4,256<<4,120<<4,"Press START to continue",4);
		gp_hardflush(&thegp);
		WaitForNextVRstart(1);
    	GS_SetCrtFB(whichdrawbuf);
	    whichdrawbuf ^= 1;
	    GS_SetDrawFB(whichdrawbuf);

		if(!fatal) {
			if(padGetState(0, 0) == PAD_STATE_STABLE) {
				padRead(0, 0, &pad1); // port, slot, buttons
				pad1_data = 0xffff ^ ((pad1.btns[0] << 8) | pad1.btns[1]);
			}

			if(pad1_data & PAD_START) break;
		}
	}
}

void display_error2(char* errmsg)
{
	int cnt = 100;

	while(cnt--) {
		gp_gouradrect(&thegp, 24<<4, 72<<4, GS_SET_RGBA(140, 0, 255, 128), 232<<4, 152<<4, GS_SET_RGBA(13, 88, 174, 128), 1);
		gp_linerect(&thegp, 23<<4, 71<<4, 232<<4, 152<<4, 2, GS_SET_RGBA(255, 255, 255, 128));
		TextOutC(0<<4,256<<4,88<<4,errmsg,3);
		gp_hardflush(&thegp);
		WaitForNextVRstart(1);
    	GS_SetCrtFB(whichdrawbuf);
	    whichdrawbuf ^= 1;
	    GS_SetDrawFB(whichdrawbuf);

	}
}


void IngameMenu()
{
	struct padButtonStatus pad1;
	int pad1_data = 0;
	int old_pad = 0;
	int new_pad;
	int selection = 0;
	static int ypos[6] = {62<<4,90<<4,108<<4,126<<4,144<<4,172<<4};

	if(sound) SjPCM_Pause();

	while(1) {
		// All this probably isnt necessary.. eh..
		gp_uploadTexture(&thegp, SMS_TEX, 256, 0, 0, GS_PSMT8, &bitmap_data, 256, 256);
		gp_uploadTexture(&thegp, SMS_CLUT, 256, 0, 0, GS_PSMCT16, &clut, 256, 1);
		gp_setTex(&thegp, SMS_TEX, 256, 256, 256, GS_PSMT8, SMS_CLUT, 256, GS_PSMCT16);

		if(cart.type == TYPE_SMS)
			gp_texrect(&thegp, 0, 24<<4, 0, 0, 256<<4, 216<<4, 256<<4, 192<<4, 1, GS_SET_RGBA(255, 255, 255, 200));
		else
			gp_texrect(&thegp, 48<<4,48<<4,48<<4,24<<4,208<<4,168<<4,208<<4,168<<4,1,GS_SET_RGBA(255, 255, 255, 200));

		// Shade SMS display
		gp_frect(&thegp, 0, 24<<4, 256<<4, 216<<4, 2, GS_SET_RGBA(0, 0, 0, 64));

		gp_gouradrect(&thegp,56<<4,54<<4,GS_SET_RGBA(140, 0, 255, 128), 200<<4,198<<4, GS_SET_RGBA(13, 88, 174, 128), 3);
		gp_linerect(&thegp, 55<<4, 53<<4, 200<<4, 198<<4, 4, GS_SET_RGBA(255, 255, 255, 128));
		TextOutC(0<<4,256<<4,ypos[0],"Return to Game",5);
		TextOutC(0<<4,256<<4,ypos[1],"Save State",5);
		TextOutC(0<<4,256<<4,ypos[2],"Load State",5);
		if(sms.country == TYPE_OVERSEAS)
			TextOutC(0<<4,256<<4,ypos[3],"Region: US/EUR",5);
		else
			TextOutC(0<<4,256<<4,ypos[3],"Region: Japan",5);
		if(sound)
			TextOutC(0<<4,256<<4,ypos[4],"Sound: Enabled",5);
		else
			TextOutC(0<<4,256<<4,ypos[4],"Sound: Disabled",5);
		TextOutC(0<<4,256<<4,ypos[5],"Return to Main Menu",5);
		gp_frect(&thegp, 56<<4, ypos[selection], 200<<4, ypos[selection] + (16<<4), 6, GS_SET_RGBA(123, 255, 255, 40));

		gp_hardflush(&thegp);
		WaitForNextVRstart(1);
    	GS_SetCrtFB(whichdrawbuf);
	    whichdrawbuf ^= 1;
	    GS_SetDrawFB(whichdrawbuf);

		if(padGetState(0, 0) == PAD_STATE_STABLE) {
			padRead(0, 0, &pad1);
			pad1_data = 0xffff ^ ((pad1.btns[0] << 8) | pad1.btns[1]);

			if((pad1.mode >> 4) == 0x07) {
				if(pad1.ljoy_v < 64) pad1_data |= PAD_UP;
				else if(pad1.ljoy_v > 192) pad1_data |= PAD_DOWN;
			}
		}
		new_pad = pad1_data & ~old_pad;
  		old_pad = pad1_data;

		if(pad1_data & PAD_SELECT) {
			if((pad1_data & PAD_UP) && dispy) dispy--;
			if(pad1_data & PAD_DOWN) dispy++;
			if((pad1_data & PAD_LEFT) && dispx) dispx--;
			if(pad1_data & PAD_RIGHT) dispx++;

			GS_SetDispMode(dispx,dispy,WIDTH,HEIGHT);
			continue;
		}

		if((new_pad & PAD_UP) && (selection > 0)) selection--;
		if((new_pad & PAD_DOWN) && (selection < 5)) selection++;

		if(new_pad & PAD_CROSS) {
			if(selection == 0) break;
			if(selection == 1) {
				psms_save_state();
				state_saved = 1;
				break;
			}
			if(selection == 2) {
				if(state_saved) psms_load_state();
				else display_error("No state previously saved.",0);
				break;
			}
			if(selection == 3) sms.country ^= 1;
			if(selection == 4) sound ^= 1;
			if(selection == 5) {
				endflag = 1;
				break;
			}
		}
	}

	// Wait till X has stopped been pressed.
	while(1) {
		if(padGetState(0, 0) == PAD_STATE_STABLE) {
			padRead(0, 0, &pad1); // port, slot, buttons
			pad1_data = 0xffff ^ ((pad1.btns[0] << 8) | pad1.btns[1]);
		}
		if(!(pad1_data & PAD_CROSS)) break;
	}

	if(sound) SjPCM_Play();
}

void psms_save_state()
{
	int pos = 0;

    /* Save VDP context */
    //fwrite(&vdp, sizeof(t_vdp), 1, fd);
	memcpy(&state[pos],&vdp,sizeof(t_vdp));
	pos += sizeof(t_vdp);

    /* Save SMS context */
    //fwrite(&sms, sizeof(t_sms), 1, fd);
	memcpy(&state[pos],&sms,sizeof(t_sms));
	pos += sizeof(t_sms);

    /* Save Z80 context */
    //fwrite(Z80_Context, sizeof(Z80_Regs), 1, fd);
    //fwrite(&after_EI, sizeof(int), 1, fd);
	memcpy(&state[pos],Z80_Context,sizeof(Z80_Regs));
	pos += sizeof(Z80_Regs);
	memcpy(&state[pos],&after_EI,sizeof(int));
	pos += sizeof(int);

    /* Save YM2413 registers */
    //fwrite(&ym2413[0].reg[0], 0x40, 1, fd);
	memcpy(&state[pos],&ym2413[0].reg[0],0x40);
	pos += 0x40;

    /* Save SN76489 context */
    //fwrite(&sn[0], sizeof(t_SN76496), 1, fd);
	memcpy(&state[pos],&sn[0],sizeof(t_SN76496));
	pos += sizeof(t_SN76496);
}

void psms_load_state()
{
    int i;
    byte reg[0x40];
	int pos = 0;

    /* Initialize everything */
    cpu_reset();
    system_reset();

    /* Load VDP context */
    //fread(&vdp, sizeof(t_vdp), 1, fd);
	memcpy(&vdp,&state[pos],sizeof(t_vdp));
	pos += sizeof(t_vdp);

    /* Load SMS context */
    //fread(&sms, sizeof(t_sms), 1, fd);
	memcpy(&sms,&state[pos],sizeof(t_sms));
	pos += sizeof(t_sms);

    /* Load Z80 context */
    //fread(Z80_Context, sizeof(Z80_Regs), 1, fd);
    //fread(&after_EI, sizeof(int), 1, fd);
	memcpy(Z80_Context,&state[pos],sizeof(Z80_Regs));
	pos += sizeof(Z80_Regs);
	memcpy(&after_EI,&state[pos],sizeof(int));
	pos += sizeof(int);

    /* Load YM2413 registers */
    //fread(reg, 0x40, 1, fd);
	memcpy(reg,&state[pos],0x40);
	pos += 0x40;

    /* Load SN76489 context */
    //fread(&sn[0], sizeof(t_SN76496), 1, fd);
	memcpy(&sn[0],&state[pos],sizeof(t_SN76496));
	pos += sizeof(t_SN76496);

    /* Restore callbacks */
    z80_set_irq_callback(sms_irq_callback);

    cpu_readmap[0] = cart.rom + 0x0000; /* 0000-3FFF */
    cpu_readmap[1] = cart.rom + 0x2000;
    cpu_readmap[2] = cart.rom + 0x4000; /* 4000-7FFF */
    cpu_readmap[3] = cart.rom + 0x6000;
    cpu_readmap[4] = cart.rom + 0x0000; /* 0000-3FFF */
    cpu_readmap[5] = cart.rom + 0x2000;
    cpu_readmap[6] = sms.ram;
    cpu_readmap[7] = sms.ram;

    cpu_writemap[0] = sms.dummy;
    cpu_writemap[1] = sms.dummy;
    cpu_writemap[2] = sms.dummy;
    cpu_writemap[3] = sms.dummy;
    cpu_writemap[4] = sms.dummy;
    cpu_writemap[5] = sms.dummy;
    cpu_writemap[6] = sms.ram;
    cpu_writemap[7] = sms.ram;

    sms_mapper_w(3, sms.fcr[3]);
    sms_mapper_w(2, sms.fcr[2]);
    sms_mapper_w(1, sms.fcr[1]);
    sms_mapper_w(0, sms.fcr[0]);

    /* Force full pattern cache update */
    is_vram_dirty = 1;
    memset(vram_dirty, 1, 0x200);

    /* Restore palette */
    for(i = 0; i < PALETTE_SIZE; i += 1)
        palette_sync(i);

    /* Restore sound state */
    if(snd.enabled)
    {
        /* Restore YM2413 emulation */
        OPLResetChip(ym3812);

        /* Clear YM2413 context */
        ym2413_reset(0);

        /* Restore rhythm enable first */
        ym2413_write(0, 0, 0x0E);
        ym2413_write(0, 1, reg[0x0E]);

        /* User instrument settings */
        for(i = 0x00; i <= 0x07; i += 1)
        {
            ym2413_write(0, 0, i);
            ym2413_write(0, 1, reg[i]);
        }

        /* Channel frequency */
        for(i = 0x10; i <= 0x18; i += 1)
        {
            ym2413_write(0, 0, i);
            ym2413_write(0, 1, reg[i]);
        }

        /* Channel frequency + ctrl. */
        for(i = 0x20; i <= 0x28; i += 1)
        {
            ym2413_write(0, 0, i);
            ym2413_write(0, 1, reg[i]);
        }

        /* Instrument and volume settings  */
        for(i = 0x30; i <= 0x38; i += 1)
        {
            ym2413_write(0, 0, i);
            ym2413_write(0, 1, reg[i]);
        }
    }
}
