/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Minimon.h"

Executor::Executor( Mini *mini )
{
	m_Emulator = mini;
	m_Running = false;
	m_HonorBreakpoints = false;

	m_Breaks = NULL;
	m_Watchs = NULL;
	m_Symbols = NULL;

	m_BreakExplore = NULL;
	m_WatchExplore = NULL;

	m_SymbolCount = 0;
	m_BreakCount = 0;
	m_WatchCount = 0;

	for( int i = 0; i < 0x80; i+=8 )
		NewWatchpoint( 0x2000 + i, 8 );
}

Executor::~Executor()
{
	DeletePoint( m_Breaks );
	DeletePoint( m_Watchs );
	DeletePoint( m_Symbols );
}

void Executor::DeletePoint( CodePoint *point )
{
	if( point == NULL )
	{
		return ;
	}

	DeletePoint( point->m_Left );
	DeletePoint( point->m_Right );

	delete point;
}

bool Executor::Running()
{
	return m_Running;
}

bool Executor::HonoringBreaks()
{
	return m_HonorBreakpoints;
}

void Executor::DeleteSymbols()
{
	DeletePoint( m_Symbols );
	m_Symbols = NULL;
	m_SymbolCount = 0;
}

void Executor::DeleteBreaks()
{
	DeletePoint( m_Breaks );
	m_Breaks = NULL;
	m_BreakCount = 0;
}

void Executor::DeleteWatchs()
{
	DeletePoint( m_Watchs );
	m_Watchs = NULL;
	m_WatchCount = 0;
}

char* Executor::GetSymbol( long address )
{
	CodePoint *symbol = m_Symbols;

	while( symbol != NULL )
	{
		if( symbol->m_Address == address )
		{
			return symbol->m_Name;
		}
		else if( symbol->m_Address < address )
		{
			symbol = symbol->m_Left;
		}
		else if( symbol->m_Address > address )
		{
			symbol = symbol->m_Right;
		}
	}

	return NULL;
}

void Executor::LoadSymbols( char *file )
{
	FILE *fo = fopen( file, "r" );
	char line[256];
	char *name;
	long address;

	while( !feof(fo) )
	{
		if( fgets( line, sizeof(line), fo ) == NULL )
		{
			break ;
		}

		address = strtol( strtok( line, " \n\r" ), NULL, 16 );
		name = strtok( NULL, " \n\r" );

		NewSymbol( name, address );
	}

	fclose(fo);
}

void Executor::SaveSymbols( char *file )
{
	FILE *fo = fopen( file, "w" );
	char buffer[32] = "00000000", *value = &buffer[8];
	CodePoint *cp;

	if( fo == NULL )
	{
		return ;
	}

	cp = ListSymbol( true );

	while( cp != NULL )
	{
		itoa( cp->m_Address, value, 16 );
		fprintf(fo, "%s %s\n", &buffer[strlen(value)], cp->m_Name);
		cp = ListSymbol( false );
	}

	fclose(fo);
}

bool Executor::StuffCodePoint( CodePoint **tree, CodePoint *node )
{
	if( node == NULL )
	{
		return false ;
	}

	StuffCodePoint( tree, node->m_Left );
	StuffCodePoint( tree, node->m_Right );

	node->m_Left = NULL;
	node->m_Right = NULL;

	if( *tree == NULL )
	{
		*tree = node;
		node->m_Parent = NULL;
		return true;
	}

	CodePoint *cp = *tree;

	while( true )
	{
		if( cp->m_Address < node->m_Address )
		{
			if( cp->m_Left != NULL )
			{
				cp = cp->m_Left;
			}
			else
			{
				cp->m_Left = node;
				node->m_Parent = cp;
				return true;
			}
		}
		else if( cp->m_Address > node->m_Address )
		{
			if( cp->m_Right != NULL )
			{
				cp = cp->m_Right;
			}
			else
			{
				cp->m_Right = node;
				node->m_Parent = cp;
				return true;
			}
		}
		else
		{
			// We have a duplicate node, delete the bugger
			delete node;
			return false;
		}
	}
}

void Executor::DeletePoint( CodePoint **tree, CodePoint *node )
{
	if( node == NULL )
	{
		return ;
	}

	if( *tree == node )
	{
		*tree = NULL ;
	}
	else
	{
		if( node->m_Parent->m_Left == node )
		{
			node->m_Parent->m_Left = NULL ;
		}
		else
		{
			node->m_Parent->m_Right = NULL ;
		}
	}

	StuffCodePoint( tree, node->m_Left );
	StuffCodePoint( tree, node->m_Right );

	delete node;
}

void Executor::DeleteBreak( long address )
{
	CodePoint *cp;

	cp = ListBreakpoint( true );

	while( cp != NULL )
	{
		if( cp->m_Address == address )
		{
			DeletePoint( &m_Breaks, cp );
			return ;
		}

		cp = ListBreakpoint( false );
	}
}

void Executor::DeleteSymbol( long address )
{
	CodePoint *cp;

	cp = ListSymbol( true );

	while( cp != NULL )
	{
		if( cp->m_Address == address )
		{
			DeletePoint( &m_Symbols, cp );
			return ;
		}

		cp = ListSymbol( false );
	}
}

void Executor::DeleteWatch( long address )
{
	CodePoint *cp;

	cp = ListWatchpoint( true );

	while( cp != NULL )
	{
		if( cp->m_Address == address )
		{
			DeletePoint( &m_Watchs, cp );
			return ;
		}

		cp = ListWatchpoint( false );
	}
}

void Executor::NewSymbol( const char *name, long address )
{
	CodePoint *cp = new CodePoint;
	cp->m_Left = cp->m_Right = NULL;
	cp->m_Address = address;
	cp->m_Name = new char[strlen(name)+1];
	strcpy( cp->m_Name, name );
	
	if( StuffCodePoint( &m_Symbols, cp ) )
	{
		m_SymbolCount++;
	}
}

void Executor::NewBreakpoint( long address, bool temporary )
{
	CodePoint *cp = new CodePoint;
	cp->m_Left = cp->m_Right = NULL;
	cp->m_AutoRemove = temporary;
	cp->m_Address = address;

	if( StuffCodePoint( &m_Breaks, cp ) )
	{
		m_BreakCount++;
	}
}

void Executor::NewWatchpoint( long address, int datasize )
{
	CodePoint *cp = new CodePoint;
	cp->m_Left = cp->m_Right = NULL;

	if( datasize > sizeof( cp->m_Data ) )
		datasize = sizeof( cp->m_Data );

	if( datasize < 1 )
		datasize = 1;

	cp->m_Size = datasize;
	cp->m_Address = address;

	if( StuffCodePoint( &m_Watchs, cp ) )
	{
		m_WatchCount++;
	}
}

bool Executor::GetBreakpoint( long address, bool execution )
{
	CodePoint *cp = m_Breaks;

	while( cp != NULL )
	{
		if( cp->m_Address == address )
		{
			if( cp->m_AutoRemove && execution )
			{
				DeletePoint( &m_Breaks, cp );
				m_BreakCount--;
			}

			return true;
		}
		else if( cp->m_Address < address )
		{
			cp = cp->m_Left;
		}
		else if( cp->m_Address > address )
		{
			cp = cp->m_Right;
		}
	}

	return false;
}

int Executor::BreakCount()
{
	return m_BreakCount;
}

int Executor::WatchCount()
{
	return m_WatchCount;
}

int Executor::SymbolCount()
{
	return m_SymbolCount;
}

CodePoint *Executor::ListBreakpoint( bool reset )
{
	if( reset || m_BreakExplore == NULL )
	{
		 m_BreakExplore = m_Breaks;

		 while( m_BreakExplore && m_BreakExplore->m_Right )
		 {
			 m_BreakExplore = m_BreakExplore->m_Right;
		 }

		return m_BreakExplore;
	}

	return Explore( &m_BreakExplore );
}

CodePoint *Executor::ListWatchpoint( bool reset )
{
	if( reset || m_WatchExplore == NULL )
	{
		 m_WatchExplore = m_Watchs;

		 while( m_WatchExplore && m_WatchExplore->m_Right )
		 {
			 m_WatchExplore = m_WatchExplore->m_Right;
		 }

		return m_WatchExplore;
	}

	return Explore( &m_WatchExplore );
}

CodePoint *Executor::ListSymbol( bool reset )
{
	if( reset || m_SymbolExplore == NULL )
	{
		 m_SymbolExplore = m_Symbols;

		 while( m_SymbolExplore && m_SymbolExplore->m_Right )
		 {
			 m_SymbolExplore = m_SymbolExplore->m_Right;
		 }

		return m_SymbolExplore;
	}

	return Explore( &m_SymbolExplore );
}

CodePoint *Executor::Explore( CodePoint **ex )
{
	if( (*ex)->m_Left != NULL )
	{
		*ex = (*ex)->m_Left;

		while( (*ex)->m_Right )
		{
			*ex = (*ex)->m_Right;
		}
	}
	else
	{
		while( (*ex)->m_Parent != NULL && (*ex)->m_Parent->m_Left == *ex )
		{
			*ex = (*ex)->m_Parent;
		}

		*ex = (*ex)->m_Parent;
	}

	return *ex;
}

void Executor::HonorBreaks( bool enable )
{
	m_HonorBreakpoints = enable;
}

void Executor::Run()
{
	m_Running = true;
	m_StepCycles = FRAME_CYCLES;
}

void Executor::DebugRun()
{
	m_Running = true;
	m_StepCycles = 1;
}

void Executor::StepOperation()
{
	m_Emulator->ExecCode();
}

void Executor::StepCycles( int cycles )
{
	long clock = m_Emulator->m_Clock + cycles;

	while( clock > m_Emulator->m_Clock )
	{
		m_Emulator->ExecCode();
	}
}

void Executor::StepFrame()
{
	long clock = m_Emulator->m_Clock + FRAME_CYCLES;

	while( clock > m_Emulator->m_Clock )
	{
		m_Emulator->ExecCode();
	}
}

void Executor::Stop()
{
	m_Running = false;
}

bool Executor::RuntimeLoop()
{	
	if( m_Running == false )
	{
		return false;
	}

	long clock = m_StepCycles;

	m_Emulator->m_Clock = 0;
	m_Emulator->ExecCode();

	if( m_HonorBreakpoints )
	{
		while( clock > m_Emulator->m_Clock )
		{
			long address = m_Emulator->m_CpuReg.word.PC;

			if( GetBreakpoint( (address & 0x8000) ? ((address&0x7FFF)|(m_Emulator->m_CpuReg.byte.V<<15)): address, true ) )
			{
				m_Running = false;
				return true;
			}
			
			m_Emulator->ExecCode();
		}
	}
	else
	{
		while( clock > m_Emulator->m_Clock )
		{
			m_Emulator->ExecCode();
		}
	}


	return true;
}