/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXPORT
#include "Minimon.h"
#include "hqFilters.h"
#undef EXPORT

void StartUp( Mini *emu )
{
	time_t timer;
	tm *t;

	time( &timer );	
	t = localtime( &timer );

	emu->m_CpuReg.word.HL = 0;
	emu->m_CpuReg.byte.N = 0x20;

	emu->Poke( 0x2000, (unsigned char) 0x4B );
	emu->Poke( 0x2002, (unsigned char) 0xDE );

	emu->m_EEPROM.HardWrite( 8182, 0x10 );
	emu->m_EEPROM.HardWrite( 8183, 0x0 );
	emu->m_EEPROM.HardWrite( 8184, 0x0 );	
	emu->m_EEPROM.HardWrite( 8185, t->tm_year - 100 );
	emu->m_EEPROM.HardWrite( 8186, t->tm_mon + 1 );
	emu->m_EEPROM.HardWrite( 8187, t->tm_mday );
	emu->m_EEPROM.HardWrite( 8188, t->tm_hour );
	emu->m_EEPROM.HardWrite( 8189, t->tm_min );
	emu->m_EEPROM.HardWrite( 8190, t->tm_sec );

	unsigned char sum = 0;
	for( int i = 8182; i <= 8190; i++ )
		sum += emu->m_EEPROM.HardRead( i );
	emu->m_EEPROM.HardWrite( 8191, sum );

	emu->SetTime( 0x10 );

	emu->ForceIRQ( 2 );
}

void DrawLine( HDC dest, HDC font, HBRUSH brush, long &x, int y, char *data )
{
	if( data == NULL )
	{
		return ;
	}

	SelectObject( dest, brush );
	while( *data )
	{
		BitBlt( dest, x, y, 8, 16, font, (*data % 16) * 8, (*data / 16) * 16, 0x00FC008A );
		data ++;
		x+=8;
	}
}

long CreateColor( DDPIXELFORMAT &ddpf, char r, char g, char b )
{
	for( int rshift = 31; rshift > 0; rshift-- )
	{
		if( ddpf.dwRBitMask >> rshift & 1 )
			break ;
	}

	for( int gshift = 31; gshift > 0; gshift-- )
	{
		if( ddpf.dwGBitMask >> gshift & 1 )
			break ;
	}

	for( int bshift = 31; bshift > 0; bshift-- )
	{
		if( ddpf.dwBBitMask >> bshift & 1 )
			break ;
	}

	long color = 0;

	return (r << 24 >> (31-rshift) & ddpf.dwRBitMask) |
		(g << 24 >> (31-gshift) & ddpf.dwGBitMask) |
		(b << 24 >> (31-bshift) & ddpf.dwBBitMask);
}

LRESULT CALLBACK LCDViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	LCDViewData *lcdData = (LCDViewData*) GetWindowLongPtr( hWnd, GWLP_USERDATA );
	DDSURFACEDESC2			ddsd;	

	switch( message )
	{	
	case WM_CREATE:
		lcdData = new LCDViewData;
		ZeroMemory( lcdData, sizeof(LCDViewData) );

		InitLUTs();

		if( FAILED(DirectDrawCreateEx( NULL, (LPVOID*)&lcdData->lpDD, IID_IDirectDraw7, NULL )) )
		{
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}	

		if( FAILED( lcdData->lpDD->SetCooperativeLevel( hWnd, DDSCL_NORMAL ) ) )
		{
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false ;
		}

		ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
		ddsd.dwSize = sizeof(DDSURFACEDESC2);
		ddsd.dwFlags = DDSD_CAPS;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

		if( FAILED( lcdData->lpDD->CreateSurface( &ddsd, &lcdData->lpDDSPrimary, NULL ) ) )
		{
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}

		if( FAILED(lcdData->lpDD->CreateClipper( 0, &lcdData->lpClipper, NULL ) ) )
		{
			lcdData->lpDDSPrimary->Release();
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}

		if( FAILED( lcdData->lpClipper->SetHWnd( 0, hWnd ) ) )
		{
			lcdData->lpClipper->Release();
			lcdData->lpDDSPrimary->Release();
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}

		if( FAILED( lcdData->lpDDSPrimary->SetClipper( lcdData->lpClipper ) ))
		{
			lcdData->lpClipper->Release();
			lcdData->lpDDSPrimary->Release();
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}

		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
		ddsd.dwWidth = 96*4;
		ddsd.dwHeight = 64*4;

		if( FAILED( lcdData->lpDD->CreateSurface( &ddsd, &lcdData->lpDDSBack, NULL )))
		{
			lcdData->lpClipper->Release();
			lcdData->lpDDSPrimary->Release();
			lcdData->lpDD->Release();
			delete lcdData;
			DestroyWindow( hWnd );
			return false;
		}

		ZeroMemory( &lcdData->ddpf, sizeof(DDPIXELFORMAT) );
		lcdData->ddpf.dwSize = sizeof(DDPIXELFORMAT);
		lcdData->lpDDSBack->GetPixelFormat( &lcdData->ddpf );

		switch( lcdData->ddpf.dwRGBBitCount )
		{
		case 16:
		case 32:
			break ;
		default:
			MessageBox( hWnd, "You are running an unsupported video mode\n\rPlease switch to 32 or 16 bit color", g_AppTitle, MB_OK | MB_ICONERROR );
		}
		
		lcdData->m_16bppColors = new short[256];
		lcdData->m_32bppColors = new long[256];
		lcdData->m_Emulator = NULL;
		SetWindowLongPtr( hWnd, GWLP_USERDATA, (LONG_PTR) lcdData );

		break ;
	case WM_CONFIG:
		{
			lcdData->m_Cfg = (Configuration*) lParam;

			int result = lcdData->m_Cfg->Seek("config.palette.color");
			while( result != NOT_FOUND )
			{
				int color, index;

				lcdData->m_Cfg->Read( "index", index, NULL );
				lcdData->m_Cfg->Read( "value", color, NULL );
				lcdData->m_Palette[index].RGB = color;

				result = lcdData->m_Cfg->Seek( NULL );
			}
			
			result = lcdData->m_Cfg->Seek("config.palette.fill");
			while( result != NOT_FOUND )
			{
				int color, start, end;

				lcdData->m_Cfg->Read( "start", start, NULL );
				lcdData->m_Cfg->Read( "end",   end,   NULL );
				lcdData->m_Cfg->Read( "color", color, NULL );

				for( int i = start; i <= end; i++ )
				{
					lcdData->m_Palette[i].RGB = color;
				}

				result = lcdData->m_Cfg->Seek( NULL );
			}

			result = lcdData->m_Cfg->Seek("config.palette.gradient");
			while( result != NOT_FOUND )
			{
				int start, end;

				lcdData->m_Cfg->Read( "start", start, NULL );
				lcdData->m_Cfg->Read( "end", end, NULL );
				SendMessage( hWnd, WM_GRADIENT, start, end );

				result = lcdData->m_Cfg->Seek( NULL );
			}
			
			SendMessage( hWnd, WM_UPDATE_PALETTE, 0, 0 );
		}

		return true;
	case WM_GRADIENT:
		if (lcdData != NULL) {
			int fr, lr;
			int fg, lg;
			int fb, lb;

			fr = lcdData->m_Palette[wParam].color.R;
			lr = lcdData->m_Palette[lParam].color.R;
			fg = lcdData->m_Palette[wParam].color.G;
			lg = lcdData->m_Palette[lParam].color.G;
			fb = lcdData->m_Palette[wParam].color.B;
			lb = lcdData->m_Palette[lParam].color.B;

			int max = lParam - wParam;

			for( int i = 1; i < max; i++ )
			{
				lcdData->m_Palette[i+wParam].color.R = (unsigned char)((lr-fr) * i / max + fr);
				lcdData->m_Palette[i+wParam].color.G = (unsigned char)((lg-fg) * i / max + fg);
				lcdData->m_Palette[i+wParam].color.B = (unsigned char)((lb-fb) * i / max + fb);
			}
		}
		return false;
	case WM_UPDATE_PALETTE:
		if (lcdData != NULL) {
			switch( lcdData->ddpf.dwRGBBitCount )
			{
			case 16:
				{
					for( int i = 0; i < 0x100; i++ )
					{
						lcdData->m_16bppColors[i] = 
							(short) CreateColor( 
							lcdData->ddpf, 
							lcdData->m_Palette[i].color.R,
							lcdData->m_Palette[i].color.G,
							lcdData->m_Palette[i].color.B
							);
					}
				}
				break ;
			case 32:
				{
					DDPIXELFORMAT ddpf;

					ddpf.dwRBitMask = 0xF800;
					ddpf.dwGBitMask = 0x07E0;
					ddpf.dwBBitMask = 0x001F;

					for( int i = 0; i < 0x100; i++ )
					{
						lcdData->m_16bppColors[i] = 
							(short) CreateColor( 
							ddpf, 
							lcdData->m_Palette[i].color.R,
							lcdData->m_Palette[i].color.G,
							lcdData->m_Palette[i].color.B
							);

						lcdData->m_32bppColors[i] = 
							CreateColor( 
							lcdData->ddpf, 
							lcdData->m_Palette[i].color.R,
							lcdData->m_Palette[i].color.G,
							lcdData->m_Palette[i].color.B
							);
					}					
				}
				break ;
			default:
				MessageBox( hWnd, "You are running an unsupported video mode\n\rPlease switch to 32 or 16 bit color", g_AppTitle, MB_OK | MB_ICONERROR );
			}
		}
		return false;
	case WM_SET_VIDEO_FILTER:
		if( lcdData != NULL )
		{
			HMENU menu = GetMenu( GetParent( hWnd ) );			
		
			switch( wParam )
			{
			case FILTER_DOTMASK:
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_CHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_UNCHECKED );
				break ;
			case FILTER_HQ2X:
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_CHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_UNCHECKED );
				break ;
			case FILTER_HQ3X:
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_CHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_UNCHECKED );
				break ;
			case FILTER_HQ4X:
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_CHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_UNCHECKED );
				break ;
			case FILTER_INTERLEAVE:
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_CHECKED );
				break ;
			default:
				wParam = FILTER_NONE;
				CheckMenuItem( menu, ID_VIDEO_PLAIN, MF_CHECKED );
				CheckMenuItem( menu, ID_DOT_MASK, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ2X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ3X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_HQ4X, MF_UNCHECKED );
				CheckMenuItem( menu, ID_INTERLEAVE, MF_UNCHECKED );
			}

			int prev = lcdData->m_VideoFilter;
			lcdData->m_VideoFilter = wParam;
			return prev;
		}
		return false;
	case WM_GET_VIDEO_FILTER:
		if( lcdData != NULL )
		{
			return lcdData->m_VideoFilter;
		}
		return false;
	case WM_POINTER:
		if( lcdData != NULL )
		{
			lcdData->m_Emulator = (Mini*) lParam;
		}
		return false;
	case WM_DESTROY:
		if( lcdData != NULL )
		{
			delete lcdData->m_16bppColors;
			delete lcdData->m_32bppColors;
			lcdData->lpClipper->Release();
			lcdData->lpDDSBack->Release();
			lcdData->lpDDSPrimary->Release();
			lcdData->lpDD->Release();
			delete lcdData;
		}
		return false;
	case WM_REFRESH:
		if( lcdData != NULL && lcdData->m_Emulator != NULL )
		{
			RECT rectDest;
			RECT rectSrc;
			POINT p;

			p.x = p.y = 0;
			ClientToScreen( hWnd, &p );
			GetClientRect( hWnd, &rectDest );
			OffsetRect( &rectDest, p.x, p.y );

			ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
			ddsd.dwSize = sizeof(DDSURFACEDESC2);

			if( !FAILED(lcdData->lpDDSBack->Lock( NULL, &ddsd, DDLOCK_WAIT, NULL )) )
			{
				switch(lcdData->ddpf.dwRGBBitCount)
				{
				case 16:
					{
						switch( lcdData->m_VideoFilter )
						{
						case FILTER_DOTMASK:
							{
								SetRect( &rectSrc, 0, 0, 96*3, 64*3 );

								short *colors = (short*)lcdData->m_16bppColors;
								short *surfa = (short *)(ddsd.lpSurface);
								short *surfb = surfa + ddsd.lPitch/2;
								short *surfc = surfb + ddsd.lPitch/2;
								long yInc = (ddsd.lPitch / 2) * 3;

								for( int ys = 0, yb = 0; ys < 64*3; ys+=3, yb++ )
								{
									for( int xs = 0, xb = 0; xs < 96*3; xs+=3, xb++ )
									{
										unsigned char pixel = lcdData->m_Emulator->m_FrameBuffer[yb][xb];
										surfb[xs] = surfb[xs+1] = surfa[xs] = surfa[xs+1] = colors[pixel];
										surfc[xs] = surfc[xs+1] = surfc[xs+2] = surfa[xs+2] = surfb[xs+2] = colors[pixel>>1];
									}

									surfa += yInc;
									surfb += yInc;
									surfc += yInc;
								}
							}
							break ;
						default:
							{
								SetRect( &rectSrc, 0, 0, 96, 64 );

								short *colors = (short*)lcdData->m_16bppColors;
								unsigned char *pixel = lcdData->m_Emulator->m_FrameBuffer[0];
								short *surface = (short *)(ddsd.lpSurface);
								long yInc = (ddsd.lPitch / 2) - 96;

								for( int yb = 0; yb < 64; yb++ )
								{
									for( int xb = 0; xb < 96; xb++ )
										*(surface++) = colors[*(pixel++)];

									surface += yInc;
								}
							}
							break ;
						}
					}
					break ;
				case 32:
					{					
						switch( lcdData->m_VideoFilter )
						{
						case FILTER_HQ4X:
						case FILTER_HQ3X:
						case FILTER_HQ2X:
							{
								unsigned char *pixel = lcdData->m_Emulator->m_FrameBuffer[0];
								short bitmap[64][96];
								short *colorUp = bitmap[0];

								for( int i = 0; i < 96*64; i++ )
									*(colorUp++) = lcdData->m_16bppColors[*(pixel++)];

								switch( lcdData->m_VideoFilter )
								{
								case FILTER_HQ2X:
									SetRect( &rectSrc, 0, 0, 96*2, 64*2 );
									hq2x_32( 
										(unsigned char*)bitmap, 
										(unsigned char*)ddsd.lpSurface, 
										96, 64, ddsd.lPitch );	// BpL is not correct, because hq2x assumes the line be equal length
									break ;
								case FILTER_HQ3X:
									SetRect( &rectSrc, 0, 0, 96*3, 64*3 );
									hq3x_32( 
										(unsigned char*)bitmap, 
										(unsigned char*)ddsd.lpSurface, 
										96, 64, ddsd.lPitch );	// BpL is not correct, because hq2x assumes the line be equal length
									break ;
								case FILTER_HQ4X:
									SetRect( &rectSrc, 0, 0, 96*4, 64*4 );
									hq4x_32( 
										(unsigned char*)bitmap, 
										(unsigned char*)ddsd.lpSurface, 
										96, 64, ddsd.lPitch );	// BpL is not correct, because hq2x assumes the line be equal length
									break ;
								}
								
							}
							break ;
						case FILTER_INTERLEAVE:
							{
								SetRect( &rectSrc, 0, 0, 96*2, 64*2 );

								static long field;

								field = (field > 0) ? 0 : (ddsd.lPitch/4);

								long *colors = (long*)lcdData->m_32bppColors;
								unsigned char *pixel = lcdData->m_Emulator->m_FrameBuffer[0];
								long *surface = (long *)(ddsd.lpSurface) + field;
								long yInc = (ddsd.lPitch / 2) - 96*2;

								for( int yb = 0; yb < 64; yb++ )
								{
									for( int xb = 0; xb < 96; xb++ )
									{
										surface[0] = surface[1] = colors[*(pixel++)];
										surface += 2;
									}

									surface += yInc;
								}
							}
							break ;
						case FILTER_DOTMASK:
							{
								SetRect( &rectSrc, 0, 0, 96*3, 64*3 );

								long *colors = (long*)lcdData->m_32bppColors;
								long *surfa = (long *)(ddsd.lpSurface);
								long *surfb = surfa + ddsd.lPitch/4;
								long *surfc = surfb + ddsd.lPitch/4;
								long yInc = (ddsd.lPitch / 4) * 3 - (96*3);
								unsigned char *pixel = lcdData->m_Emulator->m_FrameBuffer[0];

								for( int yb = 0; yb < 64; yb++ )
								{
									for( int xb = 0; xb < 96; xb++ )
									{								
										*surfb = *(surfb+1) = *surfa = *(surfa+1) = colors[*pixel];
										*surfc = *(surfc+1) = *(surfc+2) = *(surfa+2) = *(surfb+2) = colors[*pixel>>1];

										surfa += 3;
										surfb += 3;
										surfc += 3;
										pixel++;
									}

									surfa += yInc;
									surfb += yInc;
									surfc += yInc;
								}
							}
							break ;
						default:
							{
								SetRect( &rectSrc, 0, 0, 96, 64 );

								long *colors = (long*)lcdData->m_32bppColors;
								unsigned char *pixel = lcdData->m_Emulator->m_FrameBuffer[0];
								long *surface = (long *)(ddsd.lpSurface);
								long yInc = (ddsd.lPitch / 4) - 96;

								for( int yb = 0; yb < 64; yb++ )
								{
									for( int xb = 0; xb < 96; xb++ )
										*(surface++) = colors[*(pixel++)];

									surface += yInc;
								}
							}
							break ;
						}
					}
					break ;
				}

				lcdData->lpDDSBack->Unlock( NULL );
				lcdData->lpDDSPrimary->Blt( &rectDest, lcdData->lpDDSBack, &rectSrc, DDBLT_WAIT, NULL);
			}
		}
		return false;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

LRESULT CALLBACK TileViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetWindow( hWnd, GW_OWNER ), GWLP_USERDATA );

	switch( message )
	{
	case WM_CLOSE:
		ShowWindow( hWnd, SW_HIDE );
		return false;
	case WM_LBUTTONDOWN:
		if ( wndData->m_TileViewMode == TILE_OAM ) {
			unsigned short x, y;

			x = (LOWORD(lParam) - 26) / 34;
			y = (HIWORD(lParam) - 8) / 34;		

			if ( x < 6 && y < 4 )
			{
				wndData->m_SpriteSelected = x + y * 6;
			}
		}
		else
		{
			return false ;
		}
	case WM_PAINT:
	case WM_REFRESH:		
		if( wndData->m_Emulator != NULL ) {
			Mini *mini = wndData->m_Emulator;

			int x,y,t,b;

			unsigned char buffer[256][256];			
			PAINTSTRUCT ps;
			HDC hDC;

			if( message == WM_PAINT )
			{
				hDC = BeginPaint( hWnd, &ps );
			}
			else
			{
				hDC = GetDC( hWnd );
			}

			switch( wndData->m_TileViewMode )
			{
			case TILE_TILES:
				{
					for( y = 0; y < 16; y++ )
						for( x = 0; x < 128; x++ )
						{
							unsigned char byte = mini->SafeRead(
								mini->m_TileBase + (y*128+x), 
								false
								);

							for( b = 0; b < 8; b++ )
							{
								buffer[y*8+b][x] = (byte >> b & 1);
							}
						}
				}
				StretchDIBits( 
					hDC, 
					0, 0, 256, 256, 
					0, 128,  128,   128, 
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );
				break ;
			case TILE_MAP:
				{
					memset( buffer, 0, sizeof(buffer) );

					for( y = 0; y < mini->m_MapHeight * 8 + 4; y ++ )
						memset( buffer[6+y] + 6, 4, mini->m_MapWidth * 8 + 4 );

					for( t = 0; t < mini->m_MapHeight*mini->m_MapWidth; t++ )
					{
						unsigned char tile = mini->SafeRead( 0x1360 + t, false );

						for( x = 0; x < 8; x++ )
						{
							unsigned char byte = mini->SafeRead(
								mini->m_TileBase + tile*8 + x, 
								false
								);

							for( y = 0; y < 8; y++ )
							{
								buffer[(t / mini->m_MapWidth * 8)+y + 8 ][(t % mini->m_MapWidth * 8)+ x + 8] = (byte >> y & 1);
							}
						}
					}

					if( mini->m_MapWidth && mini->m_MapHeight )
					{
						for( x = 0; x < 96; x ++ )
						{
							buffer[  mini->m_MapY     % (mini->m_MapHeight * 8) + 8 ][ (mini->m_MapX+x) % (mini->m_MapWidth*8) + 8] = 5;
							buffer[ (mini->m_MapY+63) % (mini->m_MapHeight * 8) + 8 ][ (mini->m_MapX+x) % (mini->m_MapWidth*8) + 8] = 5;
						}

						for( y = 0; y < 64; y ++ )
						{
							buffer[ (mini->m_MapY+y) % (mini->m_MapHeight * 8) + 8 ][ (mini->m_MapX   ) % (mini->m_MapWidth*8) + 7] = 5;
							buffer[ (mini->m_MapY+y) % (mini->m_MapHeight * 8) + 8 ][ (mini->m_MapX+95) % (mini->m_MapWidth*8) + 9] = 5;
						}
					}
				}

				StretchDIBits( 
					hDC, 
					0, 0, 128, 256, 
					0, 0, 128, 256,
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );

				StretchDIBits( 
					hDC, 
					128, 0, 128, 200, 
					128, 56, 128, 200,
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );

				{
					HDC fontDC = CreateCompatibleDC( hDC );
					SelectObject( fontDC, wndData->m_Font );
					RECT rect;
					
					char buffer[16];

					rect.right = 256;

					sprintf( buffer, "X: %li", wndData->m_Emulator->m_MapX );
					rect.bottom = (rect.top = 200) + 16;
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 128, rect.top, buffer );
					FillRect(hDC, &rect, wndData->m_Colors.m_brClear );
					
					sprintf( buffer, "Y: %li", wndData->m_Emulator->m_MapY );
					rect.bottom = (rect.top = 216) + 16;
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 128, rect.top, buffer );
					FillRect(hDC, &rect, wndData->m_Colors.m_brClear );

					rect.top = rect.bottom;
					rect.bottom = 256;
					rect.left = 128;
					rect.right = 256;
					FillRect(hDC, &rect, wndData->m_Colors.m_brClear );
					
					DeleteDC( fontDC );
				}
				break ;
			case TILE_REGACCESS:
				{
					char color[4] = {0,5,4,1};

					memset( buffer, 3, sizeof(buffer) );
					for( t = 0; t < 256; t++ )
					{
						for( x = 0; x < 3; x++ )
						for( y = 0; y < 3; y++ )
						buffer[ ((t >> 4) << 2) +x][ ((t & 0xF) << 2) + y] = color[wndData->m_Emulator->m_RegAccessMeter[t] >> 30];
					}
				}

				StretchDIBits( 
					hDC, 
					0,   0,  256,   256,
					0, 256-64,   64,    64, 
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );

				break ;
			case TILE_SPRITES:
				{
					for( t = 0; t < 256; t++ )
					{
						for( x = 0; x < 64; x++ )
						{
							unsigned char byte = mini->SafeRead(
								mini->m_SpriteBase + t*64 + x, 
								false
								);

							for( y = 0; y < 8; y++ )
							{
								if( x & 16 )
								{
									buffer[ (t / 16 * 16) + y + (x & 0x08)][(t % 16 * 16) + (x%8) + ((x&0x20)>>2)] |= 
										(byte >> y) & 1;
								}
								else
								{
									buffer[ (t / 16 * 16) + y + (x & 0x08)][(t % 16 * 16) + (x%8) + ((x&0x20)>>2)] = 
										((byte >> y) & 1) ? 2 : 0;
								}
							}
						}
					}
				}

				StretchDIBits( 
					hDC, 
					0, 0, 256, 256, 
					0, 0, 256, 256, 
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );

				break ;
			case TILE_OAM:
				{
					memset( buffer, 0, sizeof(buffer) );

					for( y = 0; y < 4 * 17 + 3; y++ )
					{
						memset(buffer[ 2 + y ] + 11, 4, 6 * 17 + 3 );
					}

					for( y = 0; y < 18; y++ )
						for( x = 0; x < 18; x++ )
						{
							buffer[3+y+(wndData->m_SpriteSelected/6)*17][12+x+(wndData->m_SpriteSelected%6)*17] = 5;
						}

					for( int s = 0; s < 24; s++ )
					{
						int st = mini->m_Ram[0x302 + s*4];

						for( int x = 0; x < 64; x++ )
						{
							unsigned char byte = mini->SafeRead(
								mini->m_SpriteBase + st*64 + x, 
								false
								);

							for( int y = 0; y < 8; y++ )
							{
								if( x & 16 )
								{
									buffer[ 4 + y + (x & 0x08)+ (s/6)*17][13 + (x%8) + ((x&0x20)>>2)+ (s%6)*17] |= (byte >> y) & 1;
								}
								else
								{
									buffer[ 4 + y + (x & 0x08)+ (s/6)*17][13 + (x%8) + ((x&0x20)>>2)+ (s%6)*17] = ((byte >> y) & 1) ? 2 : 0;
								}
							}
						}
					}
				}
				
				StretchDIBits( 
					hDC, 
					0, 0, 256, 160, 
					0, 176,  128,   80, 
					buffer,
					(BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS,
					SRCCOPY );

				{
					int sx = mini->m_Ram[0x300 + wndData->m_SpriteSelected*4];
					int sy = mini->m_Ram[0x301 + wndData->m_SpriteSelected*4];
					int st = mini->m_Ram[0x302 + wndData->m_SpriteSelected*4];
					int sf = mini->m_Ram[0x303 + wndData->m_SpriteSelected*4];

					char *flags[8] = {" FlipX", " FlipY", " Invert", " Enable", " 4", " 5", " 6", " 7"};
					char buffer[64];
					RECT rect;
					int i;

					HDC fontDC = CreateCompatibleDC( hDC );
					SelectObject( fontDC, wndData->m_Font );
					
					rect.bottom = (rect.top = 160) + 16;

					rect.left = 0;
					rect.right = 16;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					sprintf( buffer, "X: %li", sx );
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 16, rect.top, buffer );
					rect.right = 90;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					sprintf( buffer, "Y: %li", sy );
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 90, rect.top, buffer );
					rect.right = 164;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
					
					sprintf( buffer, "Tile: %li", st );					
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 156, rect.top, buffer );
					rect.right = 256;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
					
					strcpy( buffer, "Flags:" );

					for( i = 3; i < 8; i++ )
					{
						if( sf & (1<<i) )
						{
							strcat( buffer, flags[i] );
						}
					}

					rect.bottom = (rect.top = 176) + 16;
					rect.left = 0;
					rect.right = 16;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 16, rect.top, buffer );					
					rect.right = 256;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					buffer[0] = 0;
					for( i = 0; i < 3; i++ )
					{
						if( sf & (1<<i) )
						{
							strcat( buffer, flags[i] );
						}
					}
					rect.bottom = (rect.top = 192) + 16;
					rect.left = 0;
					rect.right = 64;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 64, rect.top, buffer );
					rect.right = 256;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

					rect.bottom = 256;
					rect.top = 208;
					rect.left = 0;
					rect.right = 256;
					FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
					
					DeleteDC( fontDC );
				}
			}

			if( message == WM_PAINT )
			{
				EndPaint( hWnd, &ps );
			}
			else
			{
				DeleteDC( hDC );
			}
		}
		return false;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

BOOL CALLBACK SymbolProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetWindow( hDlg, GW_OWNER ), GWLP_USERDATA );
	HWND list = GetDlgItem( hDlg, IDC_SYMBOL );

	switch( message )
	{
	case WM_INITDIALOG:
		{
			CodePoint *cp;
			char *lable, buffer[16];
			bool enable;
			int i;

			switch( lParam & 0xF0 )
			{
			case SYMPROC_NO_VAL:
				lable = "";
				enable = false;
				break ;
			case SYMPROC_DATASIZE:
				lable = "Data Size";
				enable = true;
				break ;
			}

			EnableWindow( GetDlgItem( hDlg, IDC_DATASIZE ), enable );
			SetDlgItemText( hDlg, IDC_DATA, lable );


			switch( lParam & 0x0F )
			{
			case SYMPROC_SYMBOL:
				lable = "Symbol";
				cp = wndData->m_Executor->ListSymbol( true );
				break ;
			case SYMPROC_WATCH:
				lable = "Watch";
				cp = wndData->m_Executor->ListWatchpoint( true );
				break ;
			case SYMPROC_BREAK:
				lable = "Break";
				cp = wndData->m_Executor->ListBreakpoint( true );
				break ;
			}
			SetDlgItemText( hDlg, IDC_TYPE, lable );

			while( cp != NULL )
			{
				switch( lParam & 0x0F )
				{
				case SYMPROC_SYMBOL:
					i = SendMessage( list, CB_ADDSTRING, 0, (LPARAM) cp->m_Name );
					SendMessage( list, CB_SETITEMDATA, i, (LPARAM) cp->m_Address );
					cp = wndData->m_Executor->ListSymbol( false );
					break ;
				case SYMPROC_WATCH:
					i = SendMessage( list, CB_ADDSTRING, 0, (LPARAM) itoa( cp->m_Address, buffer, 16 ) );
					SendMessage( list, CB_SETITEMDATA, i, (LPARAM) cp->m_Address );
					cp = wndData->m_Executor->ListWatchpoint( false );
					break ;
				case SYMPROC_BREAK:
					i = SendMessage( list, CB_ADDSTRING, 0, (LPARAM) itoa( cp->m_Address, buffer, 16 ) );
					SendMessage( list, CB_SETITEMDATA, i, (LPARAM) cp->m_Address );
					cp = wndData->m_Executor->ListBreakpoint( false );
					break ;
				}
			}

			SendMessage( list, CB_SETCURSEL, 0, 0 );
		}
		return true ;
	case WM_SHOWWINDOW:
		SetFocus( GetDlgItem( hDlg, IDC_SYMBOL ) );
		return false ;
	case WM_COMMAND:
		switch( LOWORD(wParam) )
		{		
		case IDOK:
			{
				int i = SendMessage( list, CB_GETCURSEL, 0, 0 );
				char buffer[16];

				GetDlgItemText( hDlg, IDC_DATASIZE, buffer, sizeof(buffer) );
				wndData->m_DataSize = atoi( buffer );
				EndDialog( hDlg, SendMessage( list, CB_GETITEMDATA, i, 0 ) );
			}
			return true ;
		case IDCANCEL:
			EndDialog( hDlg, -1 );
			return true ;
		}
	}

	return false ;
}

BOOL CALLBACK ValueProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetWindow( hDlg, GW_OWNER ), GWLP_USERDATA );
	char buffer[256];
	long value;

	switch( message )
	{
	case WM_INITDIALOG:
		LoadString( g_hInstance, lParam, buffer, sizeof(buffer) );

		EnableWindow( GetDlgItem( hDlg, IDC_DATASIZE), lParam == IDS_WATCH );
		SetDlgItemText( hDlg, IDC_PROMPT, buffer );
		return true ;
	case WM_SHOWWINDOW:
		SetFocus( GetDlgItem( hDlg, IDC_VALUE ) );
		return false ;
	case WM_COMMAND:
		switch( LOWORD(wParam) )
		{
		case IDOK:
			{
				GetDlgItemText( hDlg, IDC_DATASIZE, buffer, sizeof(buffer) );
				wndData->m_DataSize = atoi( buffer );

				GetDlgItemText( hDlg, IDC_VALUE, buffer, sizeof(buffer) );
				if( buffer[0] == '0' )
				{
					switch( buffer[1] )
					{
					case 'b': case 'B':	// Binary value
						value = strtol( &buffer[2], NULL, 2 );
						break ;
					case 'x': case 'X': case 'h': case 'H':
						value = strtol( &buffer[2], NULL, 16 );
						break ;
					default:
						value = strtol( buffer, NULL, 10 );
					}
				}
				else
				{
					value = strtol( buffer, NULL, 10 );
				}

				EndDialog( hDlg, value );
			}
			return true ;
		case IDCANCEL:
			EndDialog( hDlg, -1 );
			return true ;
		}
		break ;
	case WM_CLOSE:
		EndDialog( hDlg, -1 );
		return true;
	}
	return false;
}

BOOL CALLBACK AddSymProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
	char *buffer = (char*) GetWindowLongPtr( hDlg, GWLP_USERDATA );

	switch( message )
	{
	case WM_INITDIALOG:
		SetWindowLongPtr( hDlg, GWLP_USERDATA, (LONG_PTR) lParam );
		return true ;
	case WM_SHOWWINDOW:
		SetFocus( GetDlgItem( hDlg, IDC_NAME ) );
		return false ;
	case WM_COMMAND:
		switch( LOWORD(wParam) )
		{
		case IDOK:
			{
				long value;

				GetDlgItemText( hDlg, IDC_ADDRESS, buffer, MAX_PATH );
				if( buffer[0] == '0' )
				{
					switch( buffer[1] )
					{
					case 'b': case 'B':	// Binary value
						value = strtol( &buffer[2], NULL, 2 );
						break ;
					case 'x': case 'X': case 'h': case 'H':
						value = strtol( &buffer[2], NULL, 16 );
						break ;
					default:
						value = strtol( buffer, NULL, 10 );
					}
				}
				else
				{
					value = strtol( buffer, NULL, 10 );
				}
				GetDlgItemText( hDlg, IDC_NAME, buffer, MAX_PATH );

				EndDialog( hDlg, value );
			}
			return true ;
		case IDCANCEL:
			EndDialog( hDlg, -1 );
			return true ;
		}
		break ;
	case WM_CLOSE:
		EndDialog( hDlg, -1 );
		return true;
	}
	return false;
}

BOOL CALLBACK CpuStatProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetParent( hDlg ), GWLP_USERDATA );
	char buffer[17];
	RegSet *rs;
	long value;

	switch( message )
	{
	case WM_COMMAND:
		rs = &(wndData->m_Emulator->m_CpuReg);		

		if( HIWORD(wParam) == EN_KILLFOCUS )
		{
			GetDlgItemText( hDlg, LOWORD(wParam), buffer, 16 );
			value = strtol( buffer, NULL, 16 );
			switch(LOWORD(wParam)) {
			case ID_PC:
				rs->word.PC = (unsigned short) value;
				break ;
			case ID_SP:
				rs->word.SP = (unsigned short) value;
				break ;
			case ID_BA:
				rs->word.BA = (unsigned short) value;
				break ;
			case ID_HL:
				rs->word.HL = (unsigned short) value;
				break ;
			case ID_X:
				rs->word.X = (unsigned short) value;
				break ;
			case ID_Y:
				rs->word.Y = (unsigned short) value;
				break ;
			case ID_A:
				rs->byte.A = (unsigned char) value;
				break ;
			case ID_B:
				rs->byte.B = (unsigned char) value;
				break ;
			case ID_L:
				rs->byte.L = (unsigned char) value;
				break ;
			case ID_H:
				rs->byte.H = (unsigned char) value;
				break ;
			case ID_F:
				rs->byte.F = (unsigned char) value;
				break ;
			case ID_E:
				rs->byte.E = (unsigned char) value;
				break ;
			case ID_U:
				rs->byte.U = (unsigned char) value;
				break ;
			case ID_V:
				rs->byte.V = (unsigned char) value;
				break ;
			case ID_XI:
				rs->byte.XI = (unsigned char) value;
				break ;
			case ID_YI:
				rs->byte.YI = (unsigned char) value;
				break ;
			case ID_I:
				rs->byte.I = (unsigned char) value;
				break ;
			case ID_N:
				rs->byte.N = (unsigned char) value;
				break ;
			default:
				return false;
			}
		}
		else
		{
			switch(LOWORD(wParam)) {
			case ID_F1:			
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F1 ) == BST_CHECKED) ? (rs->byte.F | 0x01) : (rs->byte.F & ~0x01);
				break ;
			case ID_F2:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F2 ) == BST_CHECKED) ? (rs->byte.F | 0x02) : (rs->byte.F & ~0x02);
				break ;
			case ID_F3:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F3 ) == BST_CHECKED) ? (rs->byte.F | 0x04) : (rs->byte.F & ~0x04);
				break ;
			case ID_F4:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F4 ) == BST_CHECKED) ? (rs->byte.F | 0x08) : (rs->byte.F & ~0x08);
				break ;
			case ID_F5:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F5 ) == BST_CHECKED) ? (rs->byte.F | 0x10) : (rs->byte.F & ~0x10);
				break ;
			case ID_F6:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F6 ) == BST_CHECKED) ? (rs->byte.F | 0x20) : (rs->byte.F & ~0x20);
				break ;
			case ID_F7:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F7 ) == BST_CHECKED) ? (rs->byte.F | 0x40) : (rs->byte.F & ~0x40);
				break ;
			case ID_F8:
				rs->byte.F = (IsDlgButtonChecked( hDlg, ID_F8 ) == BST_CHECKED) ? (rs->byte.F | 0x80) : (rs->byte.F & ~0x80);
				break ;
			case ID_E1:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E1 ) == BST_CHECKED) ? (rs->byte.E | 0x01) : (rs->byte.E & ~0x01);
				break ;
			case ID_E2:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E2 ) == BST_CHECKED) ? (rs->byte.E | 0x02) : (rs->byte.E & ~0x02);
				break ;
			case ID_E3:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E3 ) == BST_CHECKED) ? (rs->byte.E | 0x04) : (rs->byte.E & ~0x04);
				break ;
			case ID_E4:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E4 ) == BST_CHECKED) ? (rs->byte.E | 0x08) : (rs->byte.E & ~0x08);
				break ;
			case ID_E5:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E5 ) == BST_CHECKED) ? (rs->byte.E | 0x10) : (rs->byte.E & ~0x10);
				break ;
			case ID_E6:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E6 ) == BST_CHECKED) ? (rs->byte.E | 0x20) : (rs->byte.E & ~0x20);
				break ;
			case ID_E7:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E7 ) == BST_CHECKED) ? (rs->byte.E | 0x40) : (rs->byte.E & ~0x40);
				break ;
			case ID_E8:
				rs->byte.E = (IsDlgButtonChecked( hDlg, ID_E8 ) == BST_CHECKED) ? (rs->byte.E | 0x80) : (rs->byte.E & ~0x80);
				break ;
			default:
				return false;
			}
		}
		SetFocus( GetParent( hDlg ) );
		SendMessage( hDlg, WM_REFRESH, 0, 0 );
		return true;
	case WM_REFRESH:
		if( wndData != NULL )
		{
			rs = &(wndData->m_Emulator->m_CpuReg);
			SetDlgItemText( hDlg, ID_PC, itoa( rs->word.PC, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_SP, itoa( rs->word.SP, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_BA, itoa( rs->word.BA, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_HL, itoa( rs->word.HL, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_X, itoa( rs->word.X, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_Y, itoa( rs->word.Y, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_A, itoa( rs->byte.A, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_B, itoa( rs->byte.B, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_L, itoa( rs->byte.L, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_H, itoa( rs->byte.H, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_U, itoa( rs->byte.U, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_V, itoa( rs->byte.V, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_F, itoa( rs->byte.F, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_E, itoa( rs->byte.E, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_XI, itoa( rs->byte.XI, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_YI, itoa( rs->byte.YI, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_I, itoa( rs->byte.I, buffer, 16 ) );
			SetDlgItemText( hDlg, ID_N, itoa( rs->byte.N, buffer, 16 ) );

			SetDlgItemText( hDlg, ID_CLOCK, itoa( wndData->m_Emulator->m_Clock, buffer, 10 ) );			

			CheckDlgButton( hDlg, ID_F1, (rs->byte.F & 0x01) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F2, (rs->byte.F & 0x02) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F3, (rs->byte.F & 0x04) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F4, (rs->byte.F & 0x08) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F5, (rs->byte.F & 0x10) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F6, (rs->byte.F & 0x20) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F7, (rs->byte.F & 0x40) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_F8, (rs->byte.F & 0x80) ? BST_CHECKED : BST_UNCHECKED );

			CheckDlgButton( hDlg, ID_E1, (rs->byte.E & 0x01) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E2, (rs->byte.E & 0x02) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E3, (rs->byte.E & 0x04) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E4, (rs->byte.E & 0x08) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E5, (rs->byte.E & 0x10) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E6, (rs->byte.E & 0x20) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E7, (rs->byte.E & 0x40) ? BST_CHECKED : BST_UNCHECKED );
			CheckDlgButton( hDlg, ID_E8, (rs->byte.E & 0x80) ? BST_CHECKED : BST_UNCHECKED );
		}
		return true;
	}

	return false;
};

BOOL CALLBACK AboutProc( HWND hwndDlg, UINT message, WPARAM wParam, LPARAM lParam )
{
	if( message == WM_COMMAND && LOWORD(wParam) == IDOK )
	{
		EndDialog( hwndDlg, 0 );
		return true;
	}

	return false;
};

LRESULT CALLBACK EmulatorViewProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( hWnd, GWL_USERDATA );
	OPENFILENAME ofn;

	switch( message )
	{
	case WM_CREATE:
		{
			CREATESTRUCT *cs = (CREATESTRUCT*) lParam;
			RECT rect;
			
			wndData = new EmuWindowData;
			ZeroMemory( wndData, sizeof(EmuWindowData) );

			wndData->m_CpuStat = CreateDialog( g_hInstance, MAKEINTRESOURCE(IDD_CPUSTAT), hWnd, CpuStatProc );

			wndData->m_Memory = CreateWindow( 
				"MemoryViewer", NULL, WS_CHILD | WS_VSCROLL, 0, 0, 0, 0, hWnd, NULL, NULL, 0 
				);
			wndData->m_Disassembly = CreateWindow( 
				"AssemblyViewer", NULL, WS_CHILD | WS_VSCROLL, 0, 0, 0, 0, hWnd, NULL, NULL, 0 
				);

			wndData->m_CodePoints = CreateWindow(
				"CodePointViewer", NULL, WS_CHILD | WS_VSCROLL, 0, 0, 0, 0, hWnd, NULL, NULL, 0 
				);

			wndData->m_ErrorConsole = CreateWindow(
				"ErrorConsole", NULL, WS_TILEDWINDOW, 0, 0, 0, 0, hWnd, NULL, NULL, 0 
				);

			wndData->m_LCD = CreateWindow( "LCDViewer", NULL, WS_CHILD, 0, 0, 190, 100, hWnd, NULL, NULL, 0 );

			rect.top = rect.left = 0;
			rect.bottom = rect.right = 255;

			AdjustWindowRectEx( &rect, WS_CAPTION | WS_THICKFRAME, false, 0 );

			wndData->m_TileViewer = CreateWindow( "TileViewer",
				"Graphics Memory", 
				WS_OVERLAPPED | WS_SYSMENU,
				CW_USEDEFAULT,
				0,
				rect.right - rect.left - 1,
				rect.bottom - rect.top - 1,
				hWnd,
				NULL,
				g_hInstance,
				0 );
			
			ShowWindow( wndData->m_LCD, SW_SHOW );
			ShowWindow( wndData->m_Memory, SW_SHOW );
			ShowWindow( wndData->m_Disassembly, SW_SHOW );
			ShowWindow( wndData->m_CodePoints, SW_SHOW );
			ShowWindow( wndData->m_CpuStat, SW_SHOW );

			wndData->m_Cfg = new Configuration();
			wndData->m_Input = new InputHandle( hWnd );
			wndData->m_Emulator = new Mini();
			wndData->m_Executor = new Executor( wndData->m_Emulator );
			wndData->m_Audio = new AudioDriver( hWnd, wndData->m_Emulator );
			wndData->m_Emulator->LoadBios("bios.min");

			wndData->m_Emulator->ForceIRQ( 0 );
			wndData->m_EnableAudio = true;
			wndData->m_EnableDebugger = true;
			wndData->m_Audio->Enable( true );

			wndData->m_MemoryAddress = 0;
			wndData->m_WatchIndex = 0;

			SendMessage( wndData->m_LCD, WM_CONFIG, 0, (LPARAM) wndData->m_Cfg );
			wndData->m_Input->Configure( wndData->m_Cfg );
			wndData->m_Emulator->SetConsole( wndData->m_ErrorConsole );

			// Load the bitmap
			wndData->m_Font = LoadBitmap( g_hInstance, MAKEINTRESOURCE( IDB_SMALLFONT ) );
			
			// Todo: Load these from configuration
			wndData->m_Colors.m_brComment  = CreateSolidBrush( RGB( 127,   0,   0 ) );
			wndData->m_Colors.m_brLable    = CreateSolidBrush( RGB(   0,   0, 255 ) );
			wndData->m_Colors.m_brMnemonic = CreateSolidBrush( RGB( 127, 127,   0 ) );
			wndData->m_Colors.m_brRead     = CreateSolidBrush( RGB(   0,   0, 255 ) );
			wndData->m_Colors.m_brWrite    = CreateSolidBrush( RGB( 255,   0,   0 ) );
			wndData->m_Colors.m_brText     = CreateSolidBrush( RGB(   0,   0,   0 ) );
			wndData->m_Colors.m_brRegister = CreateSolidBrush( RGB(   0, 127,   0 ) );
			wndData->m_Colors.m_brSymbol   = CreateSolidBrush( RGB(   0,   0,   0 ) );
			wndData->m_Colors.m_brOperator = CreateSolidBrush( RGB( 127,   0, 127 ) );
			wndData->m_Colors.m_brLiteral  = CreateSolidBrush( RGB(   0, 127, 255 ) );
			wndData->m_Colors.m_brClear    = CreateSolidBrush( RGB( 255, 255, 255 ) );

			// DIB info for the tile viewer
			memset( &(wndData->m_TileBitmapInfo), 0, sizeof( wndData->m_TileBitmapInfo ) );
			wndData->m_TileBitmapInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
			wndData->m_TileBitmapInfo.bmiHeader.biWidth = 256;
			wndData->m_TileBitmapInfo.bmiHeader.biHeight = -256;
			wndData->m_TileBitmapInfo.bmiHeader.biPlanes = 1;
			wndData->m_TileBitmapInfo.bmiHeader.biBitCount = 8;
			wndData->m_TileBitmapInfo.bmiHeader.biSizeImage = 256*256;
			wndData->m_TileBitmapInfo.bmiHeader.biClrUsed = 6;
			wndData->m_TileBitmapInfo.bmiColors[0] = 0x00FFFFFF;
			wndData->m_TileBitmapInfo.bmiColors[1] = 0x00000000;
			wndData->m_TileBitmapInfo.bmiColors[2] = 0x0040CFFF;
			wndData->m_TileBitmapInfo.bmiColors[3] = 0x0040CFFF;
			wndData->m_TileBitmapInfo.bmiColors[4] = 0x00007FFF;
			wndData->m_TileBitmapInfo.bmiColors[5] = 0x00FF7F00;

			wndData->m_Rumble = false;

			SetWindowLongPtr( hWnd, GWL_USERDATA, (LONG_PTR) wndData );
			SendMessage( wndData->m_LCD, WM_POINTER, 0, (LPARAM) wndData->m_Emulator );
			SendMessage( wndData->m_Disassembly, WM_POINTER, 0, 0 );

			// Load Configuration
			int result = wndData->m_Cfg->Seek( "config.gui.size" );

			if( result != NOT_FOUND )
			{
				int width, height;
				
				wndData->m_Cfg->Read( "width", width, NULL );
				wndData->m_Cfg->Read( "height", height, NULL );

				wndData->m_LCDSize.right = width;
				wndData->m_LCDSize.bottom = height;
			}
			else
			{
				SendMessage( hWnd, WM_COMMAND, ID_LCD_1, 0 );
			}

			result = wndData->m_Cfg->Seek( "config.gui.video" );
			if( result != NOT_FOUND )
			{
				int filter, gray;

				wndData->m_Cfg->Read( "gray", gray, NULL );
				wndData->m_Cfg->Read( "filter", filter, NULL );

				wndData->m_Emulator->EmulateGray( (gray != 0) ? true : false );
				SendMessage( wndData->m_LCD, WM_SET_VIDEO_FILTER, filter, 0 );	
			}
			else
			{
				SendMessage( wndData->m_LCD, WM_SET_VIDEO_FILTER, 0, 0 );
				wndData->m_Emulator->EmulateGray( false );
			}

			{
				HMENU menu;
				menu = GetMenu(hWnd);
				CheckMenuItem( menu, ID_EMULATE_GRAY, wndData->m_Emulator->m_GrayEmulation ? MF_CHECKED : MF_UNCHECKED );
			}

			result = wndData->m_Cfg->Seek( "config.gui.enable" );

			if( result != NOT_FOUND )
			{
				int enable;

				wndData->m_Cfg->Read( "show", enable, NULL );
				GetWindowRect( hWnd, &wndData->m_DebugSize );

				SendMessage( hWnd, WM_GUI_MODE_CHANGE, enable, 0 );
			}

			result = wndData->m_Cfg->Seek( "config.gui.lastrom" );

			if( result != NOT_FOUND )
			{				
				wndData->m_Cfg->Read( "file", result, wndData->m_RomFile );
			}

		}
		return false ;
	case WM_REFRESH:
		SendMessage( wndData->m_LCD, WM_REFRESH, 0, 0 );
		SendMessage( wndData->m_ErrorConsole, WM_REFRESH, 0, 0 );

		if( wndData->m_Active && wndData->m_Rumble != wndData->m_Emulator->m_Rumble )
		{
			wndData->m_Rumble = wndData->m_Emulator->m_Rumble;
			wndData->m_Input->Rumble( wndData->m_Rumble );
		}

		if( wndData->m_EnableDebugger )
		{
			SendMessage( wndData->m_CpuStat, WM_REFRESH, 0, 0 );
			SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
			SendMessage( wndData->m_Memory, WM_REFRESH, 0, 0 );
			SendMessage( wndData->m_CodePoints, WM_REFRESH, 0, 0 );
			SendMessage( wndData->m_TileViewer, WM_REFRESH, 0, 0 );
		}
		return false ;
	case WM_ACTIVATE:
		if( wndData != NULL )
		{
			wndData->m_Active = !(wParam == WA_INACTIVE);

			if( !wndData->m_Active )
			{
				SetWindowText( hWnd, g_AppTitle );
				wndData->m_Audio->Enable( false );
				wndData->m_Input->Rumble( false );
				wndData->m_Rumble = false;
			}
			else
			{			
				wndData->m_Audio->Enable( wndData->m_EnableAudio && wndData->m_Executor->Running() );
			}
		}
		return true ;
	case WM_GUI_MODE_CHANGE:
		{
			RECT *oldrect = wndData->m_EnableDebugger ? &wndData->m_DebugSize : &wndData->m_LCDSize ;

			GetWindowRect( hWnd, oldrect );

			wndData->m_EnableDebugger = (wParam ? true : false);

			int show = wndData->m_EnableDebugger ? SW_SHOW : SW_HIDE;

			ShowWindow( wndData->m_Disassembly, show );				
			ShowWindow( wndData->m_CodePoints, show );
			ShowWindow( wndData->m_Memory, show );
			ShowWindow( wndData->m_CpuStat, show );

			if( !wndData->m_EnableDebugger )
			{
				ShowWindow( wndData->m_TileViewer, SW_HIDE );
			}

			CheckMenuItem( GetMenu( hWnd ), ID_ENABLE_DEBUGGER, wndData->m_EnableDebugger ? MF_CHECKED : MF_UNCHECKED );

			RECT *newrect = wndData->m_EnableDebugger ? &wndData->m_DebugSize : &wndData->m_LCDSize ;

			SetWindowPos( hWnd, NULL, 
				0, 0, 
				newrect->right-newrect->left, 
				newrect->bottom-newrect->top, 
				SWP_NOMOVE );			
		}
		return true;
	case WM_COMMAND:
		switch( LOWORD( wParam ) )
		{
		case ID_EMULATE_GRAY:
			{
				HMENU menu = GetMenu(hWnd);
				CheckMenuItem( menu, ID_EMULATE_GRAY, wndData->m_Emulator->m_GrayEmulation ? MF_UNCHECKED : MF_CHECKED );
				wndData->m_Emulator->EmulateGray( !wndData->m_Emulator->m_GrayEmulation );
			}
			break ;
		case ID_VIDEO_PLAIN:
		case ID_DOT_MASK:
		case ID_HQ2X:
		case ID_HQ3X:
		case ID_HQ4X:
		case ID_INTERLEAVE:
			{
				int filter;
				switch( LOWORD( wParam ) )
				{
				case ID_VIDEO_PLAIN:
					filter = FILTER_NONE;
					break ;
				case ID_DOT_MASK:
					filter = FILTER_DOTMASK;
					break ;
				case ID_HQ2X:
					filter = FILTER_HQ2X;
					break ;
				case ID_HQ3X:
					filter = FILTER_HQ3X;
					break ;
				case ID_HQ4X:
					filter = FILTER_HQ4X;
					break ;
				case ID_INTERLEAVE:
					filter = FILTER_INTERLEAVE;
					break ;
				}
				SendMessage( wndData->m_LCD, WM_SET_VIDEO_FILTER, filter, 0 );
			}
			return false;
		case ID_ENABLE_DEBUGGER:
			{
				SendMessage( hWnd, WM_GUI_MODE_CHANGE, !wndData->m_EnableDebugger, 0 );
			}
			return false;
		case ID_LCD_1:
		case ID_LCD_2:
		case ID_LCD_3:
		case ID_LCD_4:
			{				
				wndData->m_LCDSize.top = wndData->m_LCDSize.left = 0;

				switch(LOWORD( wParam ))
				{
					case ID_LCD_1:
						wndData->m_LCDSize.right = 96 * 3;
						wndData->m_LCDSize.bottom = 64 * 3;
						break ;
					case ID_LCD_2:
						wndData->m_LCDSize.right = 96 * 3 * 2;
						wndData->m_LCDSize.bottom = 64 * 3 * 2;
						break ;
					case ID_LCD_3:
						wndData->m_LCDSize.right = 96 * 3 * 3;
						wndData->m_LCDSize.bottom = 64 * 3 * 3;
						break ;
					case ID_LCD_4:
						wndData->m_LCDSize.right = 96 * 3 * 4;
						wndData->m_LCDSize.bottom = 64 * 3 * 4;
						break ;
				}


				AdjustWindowRectEx( &wndData->m_LCDSize, WS_CAPTION | WS_THICKFRAME, true, 0 );
				
				if( !wndData->m_EnableDebugger )
				{
					SetWindowPos( hWnd, NULL, 
						0, 0, 
						wndData->m_LCDSize.right-wndData->m_LCDSize.left, 
						wndData->m_LCDSize.bottom-wndData->m_LCDSize.top, 
						SWP_NOMOVE );
				}
			}
			return false;
		case ID_CONFIG_INPUT:
			wndData->m_Input->Configure();
			return false;
		case ID_RESET:
			wndData->m_Emulator->ForceIRQ(2);
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_ABOUT:
			DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_ABOUT), hWnd, AboutProc );
			return false ;
		case ID_OPEN:
			ZeroMemory(&ofn, sizeof(OPENFILENAME));
			ofn.lStructSize = sizeof(OPENFILENAME);
			ofn.hwndOwner = hWnd;
			ofn.lpstrFile = wndData->m_RomFile;
			ofn.nMaxFile = sizeof(wndData->m_RomFile);
			ofn.lpstrFilter = "Pokemon Mini ROM image\0*.min\0All files\0*.*\0";
			ofn.nFilterIndex = 1;
			ofn.lpstrFileTitle = NULL;
			ofn.nMaxFileTitle = 0;
			ofn.lpstrInitialDir = NULL;
			ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

			if( GetOpenFileName( &ofn ) == 0 )
			{
				break ;
			}

		case ID_RELOAD:
			if( wndData->m_RomFile[0] == 0 )
			{
				SendMessage( hWnd, WM_COMMAND, ID_OPEN, 0 );
				break ;
			}

			wndData->m_Emulator->LoadRom(wndData->m_RomFile);
			StartUp(wndData->m_Emulator);
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			break ;
		case ID_CLOSE:
			wndData->m_RomFile[0] = 0;
			wndData->m_Emulator->FreeRom();
			wndData->m_Emulator->ForceIRQ(0);
			wndData->m_Emulator->m_EEPROM.Load(NULL);
			break ;
		case ID_SAVE_AS:
			ZeroMemory(&ofn, sizeof(OPENFILENAME));
			ofn.lStructSize = sizeof(OPENFILENAME);
			ofn.hwndOwner = hWnd;
			ofn.lpstrFile = wndData->m_RomFile;
			ofn.nMaxFile = sizeof(wndData->m_RomFile);
			ofn.lpstrFilter = "Pokemon Mini ROM image\0*.min\0All files\0*.*\0";
			ofn.nFilterIndex = 1;
			ofn.lpstrFileTitle = NULL;
			ofn.nMaxFileTitle = 0;
			ofn.lpstrInitialDir = NULL;
			ofn.Flags = OFN_PATHMUSTEXIST;

			if( GetSaveFileName( &ofn ) == 0 )
			{
				break ;
			}

		case ID_SAVE:
			if( wndData->m_RomFile[0] == 0 )
			{
				SendMessage( hWnd, WM_COMMAND, ID_SAVE_AS, 0 );
				break ;
			}

			{
				FILE *str = fopen( wndData->m_RomFile, "r" );

				if( str != NULL )
				{
					fclose(str);
					
					if( MessageBox( hWnd, "Are you sure you want to override the rom?", g_AppTitle, MB_YESNO | MB_ICONWARNING ) == IDNO )
					{
						break ;
					}
				}
					
				str = fopen( wndData->m_RomFile, "wb" );
				
				if( str != NULL )
				{
					fwrite( wndData->m_Emulator->m_Rom, wndData->m_Emulator->m_RomSize, 1, str );
					fclose( str );
				}
				else
				{
					wndData->m_Emulator->Error("Failed to write rom", 0, NULL);
				}
			}
			break ;
		case ID_TILE_VIEWER:
		case ID_MAP_VIEWER:
		case ID_SPRITE_VIEWER:
		case ID_OAM_VIEWER:
		case ID_REGACCESS_VIEWER:
			switch( LOWORD( wParam ) )
			{
			case ID_TILE_VIEWER: 
				wndData->m_TileViewMode = TILE_TILES;
				break ;
			case ID_MAP_VIEWER:
				wndData->m_TileViewMode = TILE_MAP;
				break ;
			case ID_SPRITE_VIEWER:
				wndData->m_TileViewMode = TILE_SPRITES;
				break ;
			case ID_OAM_VIEWER:
				wndData->m_TileViewMode = TILE_OAM;
				break ;
			case ID_REGACCESS_VIEWER:
				wndData->m_TileViewMode = TILE_REGACCESS;
				break ;
			};
			
			if( wndData->m_EnableDebugger )
			{
				SendMessage( wndData->m_TileViewer, WM_REFRESH, 0, 0 );
				ShowWindow( wndData->m_TileViewer, SW_SHOW );
			}

			return false;
		case ID_DISASSEMBLE_USER:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	
				CheckMenuItem( menu, ID_DISASSEMBLE_USER, MF_CHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_PC, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_BOTH, MF_UNCHECKED );
				if( wndData->m_EnableDebugger )
				{
					SendMessage( wndData->m_Disassembly, WM_COMMAND, ID_DISASSEMBLE_USER, 0 );
					SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
				}
			}
			return false;
		case ID_DISASSEMBLE_PC:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	
				CheckMenuItem( menu, ID_DISASSEMBLE_USER, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_PC, MF_CHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_BOTH, MF_UNCHECKED );
				if( wndData->m_EnableDebugger )
				{
					SendMessage( wndData->m_Disassembly, WM_COMMAND, ID_DISASSEMBLE_PC, 0 );
					SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
				}
			}
			return false;
		case ID_DISASSEMBLE_BOTH:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	
				CheckMenuItem( menu, ID_DISASSEMBLE_USER, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_PC, MF_UNCHECKED );
				CheckMenuItem( menu, ID_DISASSEMBLE_BOTH, MF_CHECKED );
				if( wndData->m_EnableDebugger )
				{
					SendMessage( wndData->m_Disassembly, WM_COMMAND, ID_DISASSEMBLE_BOTH, 0 );
					SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
				}
			}
			InvalidateRect( wndData->m_Disassembly, NULL, true );
			return false ;
		case ID_DISASSEMBLER_JUMP_ADDR:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_ADDRESS );

				if( callback > 0x8000 )
				{
					callback = callback & 0x7FFF | 0x8000;
				}

				if( callback > 0 )
				{
					SendMessage( wndData->m_Disassembly, WM_VALUE, 0, callback );
					SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
				}
			}
			return false ;
		case ID_DISASSEMBLER_JUMP_PC:
			{
				SendMessage( wndData->m_Disassembly, WM_VALUE, 0, wndData->m_Emulator->m_CpuReg.word.PC );
				SendMessage( wndData->m_Disassembly, WM_REFRESH, 0, 0 );
			}
			return false ;
		case ID_MEMORY_BIOS:
			wndData->m_MemoryAddress = 0;
			InvalidateRect( wndData->m_Memory, NULL, true );
			return false ;
		case ID_MEMORY_RAM:
			wndData->m_MemoryAddress = 0x1000;
			InvalidateRect( wndData->m_Memory, NULL, true );
			return false ;
		case ID_MEMORY_REGISTERS:
			wndData->m_MemoryAddress = 0x2000;
			InvalidateRect( wndData->m_Memory, NULL, true );
			return false ;
		case ID_MEMORY_ROM:
			wndData->m_MemoryAddress = 0x2100;
			InvalidateRect( wndData->m_Memory, NULL, true );
			return false ;
		case ID_MEMORY_ADDRESS:
			{				
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_ADDRESS );

				if( callback > 0 )
				{
					wndData->m_MemoryAddress = callback ;
				}
				InvalidateRect( wndData->m_Memory, NULL, true );
			}
			return false ;
		case ID_MEMORY_SYMBOL:
			{				
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_SYMBOL );

				if( callback > 0 )
				{
					wndData->m_MemoryAddress = callback ;
				}
				InvalidateRect( wndData->m_Memory, NULL, true );
			}
			return false ;
		case ID_SHOW_CONSOLE:
			ShowWindow( wndData->m_ErrorConsole, SW_SHOW );
			return false;
		case ID_QUIT:
			DestroyWindow(hWnd);
			return false ;
		case ID_STOP:
			wndData->m_Audio->Enable( false );
			wndData->m_Executor->Stop();
			return false;
		case ID_RUN_FULL:
			wndData->m_Audio->Enable( wndData->m_EnableAudio );
			wndData->m_Executor->Run();
			return false;
		case ID_RUN_DEBUG:
			wndData->m_Audio->Enable( wndData->m_EnableAudio );
			wndData->m_Executor->DebugRun();
			return false;
		case ID_RUN_ADDRESS:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_ADDRESS );
				
				if( callback >= 0 )
				{
					wndData->m_Executor->NewBreakpoint( callback, true );
					wndData->m_Executor->HonorBreaks( true );
					wndData->m_Executor->Run();
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			return false;
		case ID_RUN_SYMBOL:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_SYMBOL );
				
				if( callback >= 0 )
				{
					wndData->m_Executor->NewBreakpoint( callback, true );
					wndData->m_Executor->HonorBreaks( true );
					wndData->m_Executor->Run();
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			return false;
		case ID_BREAK_ADDRESS:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_ADDRESS );
				
				if( callback >= 0 )
				{
					wndData->m_Executor->NewBreakpoint( callback, false );
					wndData->m_Executor->HonorBreaks( true );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_BREAK_SYMBOL:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_SYMBOL );
				
				if( callback >= 0 )
				{
					wndData->m_Executor->NewBreakpoint( callback, false );
					wndData->m_Executor->HonorBreaks( true );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_BREAK_IRQ_BIOS:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_IRQ );
				
				if( callback >= 0 && callback < 0x80 )
				{
					long address;
					address = wndData->m_Emulator->SafeRead( callback * 2, false );
					address |= wndData->m_Emulator->SafeRead( address * 2 + 1, false ) << 8;
					wndData->m_Executor->NewBreakpoint( callback, false );
					wndData->m_Executor->HonorBreaks( true );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_BREAK_IRQ_ROM:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_IRQ );
				
				if( callback >= 0 && callback < 26 )
				{
					wndData->m_Executor->NewBreakpoint( 0x2102 + callback*6, false );
					wndData->m_Executor->HonorBreaks( true );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_WATCH_ADDRESS:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_WATCH );
				
				if( callback >= 1 )
				{
					wndData->m_Executor->NewWatchpoint( callback, wndData->m_DataSize );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			if( wndData->m_EnableDebugger )
				SendMessage( wndData->m_CodePoints, WM_REFRESH, 0, 0 );
			return false;
		case ID_WATCH_SYMBOL:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_SYMBOL|SYMPROC_DATASIZE );
				
				if( callback >= 1 )
				{
					wndData->m_Executor->NewWatchpoint( callback, wndData->m_DataSize );
				}
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, MF_CHECKED );
			}
			if( wndData->m_EnableDebugger )
				SendMessage( wndData->m_CodePoints, WM_REFRESH, 0, 0 );
			return false;
		case ID_FORCE_IRQ:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_IRQ );
					
				if( callback >= 0 )
				{
					wndData->m_Emulator->ForceIRQ((unsigned char)(callback*2));
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_STEP_OPCODE:
			wndData->m_Executor->StepOperation();
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false ;
		case ID_STEP_CYCLES:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_INT), hWnd, ValueProc, IDS_CYCLES );

				if( callback > 0 )
				{
					wndData->m_Executor->StepCycles(callback);
				}

				SendMessage( hWnd, WM_REFRESH, 0, 0 );
			}
			return false ;
		case ID_STEP_FRAME:			
			wndData->m_Executor->StepFrame();
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_LOAD_BIOS:
			{
				char fileName[1024] = "";

				ZeroMemory(&ofn, sizeof(OPENFILENAME));
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hWnd;
				ofn.lpstrFile = fileName;
				ofn.nMaxFile = sizeof(fileName);
				ofn.lpstrFilter = "Pokemon mini rom\0*.min\0All files\0*.*\0";
				ofn.nFilterIndex = 1;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

				if( GetOpenFileName( &ofn ) )
				{
					wndData->m_Emulator->LoadBios( fileName );
					break ;
				}
			}
			wndData->m_Emulator->ForceIRQ( 0 );
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_SYMBOLS_LOAD:
			{
				ZeroMemory(&ofn, sizeof(OPENFILENAME));
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hWnd;
				ofn.lpstrFile = wndData->m_SymFile;
				ofn.nMaxFile = sizeof(wndData->m_SymFile);
				ofn.lpstrFilter = "PMAS Symbol File\0*.map;*.sym\0All files\0*.*\0";
				ofn.nFilterIndex = 1;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

				if( GetOpenFileName( &ofn ) )
				{
					wndData->m_Executor->LoadSymbols( wndData->m_SymFile );
					break ;
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;		
		case ID_SYMBOLS_SAVE_AS:
			{
				ZeroMemory(&ofn, sizeof(OPENFILENAME));
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hWnd;
				ofn.lpstrFile = wndData->m_SymFile;
				ofn.nMaxFile = sizeof(wndData->m_SymFile);
				ofn.lpstrFilter = "PMAS Symbol File\0*.map;*.sym\0All files\0*.*\0";
				ofn.nFilterIndex = 1;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.Flags = OFN_PATHMUSTEXIST;

				if( GetSaveFileName( &ofn ) == 0 )
				{
					break ;
				}
			}
		case ID_SYMBOLS_SAVE:
			if( wndData->m_SymFile[0] == 0 )
			{
				SendMessage( hWnd, WM_COMMAND, ID_SYMBOLS_SAVE_AS, 0 );
				break ;
			}
			
			wndData->m_Executor->SaveSymbols( wndData->m_SymFile );
			break ;
		case ID_SYMBOLS_ADD:
			{
				char buffer[MAX_PATH];

				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_ADD_SYMBOL), hWnd, AddSymProc, (LPARAM)buffer );

				if( callback >= 0 )
				{
					wndData->m_Executor->NewSymbol( buffer, callback );
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_SYMBOLS_DELETE:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_SYMBOL );

				if( callback >= 0 )
				{
					wndData->m_Executor->DeleteSymbol( callback );
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_WATCH_DELETE:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_WATCH );

				if( callback >= 0 )
				{
					wndData->m_Executor->DeleteWatch( callback );
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_BREAK_DELETE:
			{
				long callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE(IDD_REQUEST_SYM), hWnd, SymbolProc, SYMPROC_BREAK );

				if( callback >= 0 )
				{
					wndData->m_Executor->DeleteBreak( callback );
				}
			}
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_BREAK_DELALL:
			wndData->m_Executor->DeleteBreaks();
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_WATCH_DELALL:
			wndData->m_Executor->DeleteWatchs();
			
			if( wndData->m_EnableDebugger )
				SendMessage( wndData->m_CodePoints, WM_REFRESH, 0, 0 );
			return false;
		case ID_SYMBOLS_CLEAR:
			wndData->m_Executor->DeleteSymbols();
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
			return false;
		case ID_AUDIO_ENABLE:
			{
				HMENU menu;
				menu = GetMenu( hWnd );	

				wndData->m_EnableAudio = !wndData->m_EnableAudio;
				wndData->m_Audio->Enable( wndData->m_EnableAudio && wndData->m_Executor->Running() );

				// Flush audio
				if( wndData->m_EnableAudio )
				{
					wndData->m_Emulator->m_StreamSample = 0;
				}

				CheckMenuItem( menu, ID_AUDIO_ENABLE, wndData->m_EnableAudio ? MF_CHECKED : MF_UNCHECKED );
			}
			return false;
		case ID_BREAK_ENABLE:
			{
				HMENU menu;
				menu = GetMenu( hWnd );

				bool breaks = !wndData->m_Executor->HonoringBreaks();				
				
				CheckMenuItem( menu, ID_BREAK_ENABLE, breaks ? MF_CHECKED : MF_UNCHECKED );				
				wndData->m_Executor->HonorBreaks( breaks );
			}
			return false;
		case WM_INPUTPRESS:
			switch( lParam )
			{
			case PAD_THROTTLE:
				wndData->m_Throttle = true;
				break ;
			case PAD_SHAKE:
				wndData->m_Emulator->Shake();
				break ;
			default:
				wndData->m_Emulator->PressPad( (unsigned char)lParam );
			}
			break;
		case WM_INPUTRELEASE:
			switch( lParam )
			{
			case PAD_THROTTLE:
				wndData->m_Throttle = false;
				break ;
			case PAD_SHAKE:
				break ;
			default:
				wndData->m_Emulator->ReleasePad( (unsigned char)lParam );
				break;
			}
			break ;
		case ID_SCREENSHOT:
			{
				FILE *screen;
				char filename[255], *number;

				strcpy( filename, wndData->m_RomFile );
				number = filename;

				while( *number )
					number++;

				while( *number != '.'  && number != filename )
					number--;

				for( int i = 0; i < 65536; i++ )
				{
					itoa( i, number, 16 );
					strcat(number, ".raw" );

					screen = fopen( filename, "rb" );

					if( screen == NULL )
						break ;
				}				

				screen = fopen( filename, "wb" );
				fwrite( wndData->m_Emulator->m_FrameBuffer, 96, 64, screen );
				fclose(screen);
			}
			return false;
		}
		break ;
	case WM_SIZE:
		if ( wndData->m_EnableDebugger ) 
		{
			int width, height;
			RECT rect;

			width = LOWORD( lParam );
			height = HIWORD( lParam );

			GetWindowRect( wndData->m_CpuStat, &rect );

			rect.bottom -= rect.top;
			rect.right -= rect.left;

			SetWindowPos( 
				wndData->m_CpuStat, NULL,
				width-rect.right, 4, 
				rect.right, rect.bottom,
				0
				);

			SetWindowPos( 
				wndData->m_LCD, 
				NULL, 
				width-rect.right-(96*3-rect.right)/2, 
				8, 
				96*3, 
				64 * 3, 
				0 
				);

			if( rect.bottom < (height - 8) )
			{
				SetWindowPos( 
					wndData->m_CodePoints, NULL,
					width-rect.right, rect.bottom + 8, 
					rect.right, height - rect.bottom - 8,
					0
					);

				ShowWindow( wndData->m_CodePoints, SW_SHOW ) ;
			}
			else
			{
				ShowWindow( wndData->m_CodePoints, SW_HIDE ) ;
			}

			SetWindowPos( 
				wndData->m_Disassembly, NULL, 
				0, 0,
				width-rect.right, height-256-2,
				0
			);

			SetWindowPos( 
				wndData->m_Memory, NULL, 
				0, (height-256),
				width-rect.right, 256,
				0
			);

		}
		else
		{
			SetWindowPos( wndData->m_LCD, NULL, 0, 0, LOWORD( lParam ), HIWORD( lParam ), 0 );
		}
		return false;
	case WM_CLOSE:
		DestroyWindow(hWnd);
		return false ;
	case WM_DESTROY:
		// Save Configuration

		int result = wndData->m_Cfg->Seek( "config.gui.size" );

		if( result == NOT_FOUND )
		{
			wndData->m_Cfg->Create( "config.gui.size" );
		}

		wndData->m_Cfg->Write( "width",  wndData->m_LCDSize.right - wndData->m_LCDSize.left );
		wndData->m_Cfg->Write( "height", wndData->m_LCDSize.bottom - wndData->m_LCDSize.top );

		result = wndData->m_Cfg->Seek( "config.gui.enable" );

		if( result == NOT_FOUND )
		{
			wndData->m_Cfg->Create( "config.gui.enable" );
		}

		wndData->m_Cfg->Write( "show", wndData->m_EnableDebugger ? 1 : 0 );

		result = wndData->m_Cfg->Seek( "config.gui.video" );

		if( result == NOT_FOUND )
		{
			wndData->m_Cfg->Create( "config.gui.video" );
		}

		wndData->m_Cfg->Write( "gray", wndData->m_Emulator->m_GrayEmulation ? 1 : 0 );
		wndData->m_Cfg->Write( "filter", SendMessage( wndData->m_LCD, WM_GET_VIDEO_FILTER, 0, 0 ) );

		result = wndData->m_Cfg->Seek( "config.gui.lastrom" );

		if( result == NOT_FOUND )
		{
			wndData->m_Cfg->Create( "config.gui.lastrom" );
		}

		wndData->m_Cfg->Write( "file", wndData->m_RomFile );

		// Clean up 
		if( wndData->m_Emulator != NULL )
		{
			delete wndData->m_Emulator;
		}

		if( wndData->m_Executor != NULL )
		{
			delete wndData->m_Executor;
		}

		DeleteObject( wndData->m_Colors.m_brClear );
		DeleteObject( wndData->m_Colors.m_brRegister );
		DeleteObject( wndData->m_Colors.m_brSymbol );
		DeleteObject( wndData->m_Colors.m_brOperator );
		DeleteObject( wndData->m_Font );
		DeleteObject( wndData->m_Colors.m_brComment );
		DeleteObject( wndData->m_Colors.m_brLable );
		DeleteObject( wndData->m_Colors.m_brMnemonic );
		DeleteObject( wndData->m_Colors.m_brRead );
		DeleteObject( wndData->m_Colors.m_brText );
		DeleteObject( wndData->m_Colors.m_brWrite );
		DeleteObject( wndData->m_Colors.m_brLiteral );

		delete wndData->m_Audio;
		delete wndData->m_Input;
		delete wndData->m_Cfg;
		delete wndData;
		
		PostQuitMessage(0);
		return false;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

HWND SpawnGUI( HINSTANCE hInstance )
{
	WNDCLASSEX wcex;
	HWND hWnd;

	memset( &wcex, 0, sizeof(WNDCLASSEX) );

	wcex.cbSize = sizeof(WNDCLASSEX); 
	wcex.hInstance		= hInstance;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);

	wcex.style			= CS_DBLCLKS;
	wcex.lpszClassName	= (LPTSTR) "MemoryViewer";
	wcex.lpfnWndProc	= (WNDPROC)MemoryViewerProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	wcex.lpszClassName	= (LPTSTR) "AssemblyViewer";
	wcex.lpfnWndProc	= (WNDPROC)AssemblyViewerProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	wcex.lpszClassName	= (LPTSTR) "CodePointViewer";
	wcex.lpfnWndProc	= (WNDPROC)CodePointViewerProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}


	wcex.style			= 0;
	wcex.lpszClassName	= (LPTSTR) "ErrorConsole";
	wcex.lpfnWndProc	= (WNDPROC)ErrorProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	wcex.lpszClassName	= (LPTSTR) "TileViewer";
	wcex.lpfnWndProc	= (WNDPROC)TileViewerProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	wcex.hbrBackground	= (HBRUSH)(COLOR_BTNFACE+1);
	wcex.lpszClassName	= (LPTSTR) "LCDViewer";
	wcex.lpfnWndProc	= (WNDPROC)LCDViewerProc;
	
	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	wcex.hIcon			= LoadIcon(hInstance, (LPCTSTR)IDI_MAIN);
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hIconSm		= LoadIcon(hInstance, (LPCTSTR)IDI_SMALLICO);

	wcex.lpszMenuName	= (LPCSTR)IDI_MENU;
	wcex.lpszClassName	= (LPTSTR) "EmulatorView";
	wcex.lpfnWndProc	= (WNDPROC)EmulatorViewProc;	

	if(!RegisterClassEx(&wcex)) 
	{
		return NULL;
	}

	hWnd = CreateWindow(
		"EmulatorView", 
		g_AppTitle,
		WS_TILEDWINDOW,
		CW_USEDEFAULT,0,
		CW_USEDEFAULT,0,
		NULL, NULL, hInstance, 0 );

	if( hWnd == NULL )
	{
		return NULL;
	}

	return hWnd;
}

void CommandLine( LPSTR lpCmdLine, HWND wnd, EmuWindowData *wndData )
{
	char *token = lpCmdLine;

	while( *token )
	{
		if( *token == '-' )
		{
			switch( token[1] )
			{
			case 'd': case 'D': // Disable debugger
				SendMessage( wnd, WM_GUI_MODE_CHANGE, 0, 0 );
				break ;
			case 'r': case 'R': // Run at start
				wndData->m_Executor->Run();
				break ;
			case '1':
				SendMessage( wnd, WM_COMMAND, ID_LCD_1, 0 );
				break ;
			case '2':
				SendMessage( wnd, WM_COMMAND, ID_LCD_2, 0 );
				break ;
			case '3':
				SendMessage( wnd, WM_COMMAND, ID_LCD_3, 0 );
				break ;
			case '4':
				SendMessage( wnd, WM_COMMAND, ID_LCD_4, 0 );
				break ;
			}
			
			token += 2;
		}
		else if( *token == ' ' )
		{
			token++;
		}
		else if( *token )
		{
			wndData->m_Emulator->LoadRom( token );
			StartUp(wndData->m_Emulator);
			break ;
		}
	}
}

int WINAPI WinMain(
  HINSTANCE hInstance,      // handle to current instance
  HINSTANCE hPrevInstance,  // handle to previous instance
  LPSTR lpCmdLine,          // command line
  int nCmdShow              // show state
)
{	
	HACCEL hAccelTable;
	HWND hWnd;
	MSG msg;
	
	g_hInstance = hInstance;

	hWnd = SpawnGUI( hInstance );	

	if( hWnd == NULL )
	{
		return -1;
	}
	
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( hWnd, GWLP_USERDATA );

	CommandLine( lpCmdLine, hWnd, wndData );

	ShowWindow( hWnd, nCmdShow );
	hAccelTable = LoadAccelerators(hInstance, (LPCTSTR)IDR_ACCELERATOR);

	// Updater code
	char appPath[MAX_PATH], *appName = appPath;

	strcpy( appPath, GetCommandLine() );
	
	if( appPath[0] == '"' )
	{
		char *seek = appPath+1;
		while( *seek != '"' ) seek++;
		*seek = 0;
		appName++;
	}
	else
	{
		char *seek = appPath;
		while( *seek != ' ' ) seek++;
		*seek = 0;
	}

	AutoUpdater updater(appName);
	updater.Update();

	int nextFrame = GetTickCount() + FRAME_TICKS;

	int cpuFrames = 0;	// Counts up to 40
	int firstTick = 0;	// Should be 40000 for 100%

	while( true )
	{
		while( PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) )
		{
			if( msg.message == WM_QUIT )
			{
				DestroyWindow( hWnd );
				return (int) msg.wParam;
			}

			if (!TranslateAccelerator(msg.hwnd, hAccelTable, &msg)) 
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		wndData->m_Input->Process(wndData->m_Active);

		if( wndData->m_Active && wndData->m_Executor->RuntimeLoop() )
		{
			SendMessage( hWnd, WM_REFRESH, 0, 0 );

			int time = GetTickCount();

			cpuFrames++;
			// EST. CPU Time feature
			if( cpuFrames == 40 )
			{
			 	float relativeSpeed = 100000.0f / ( time - firstTick );
				char buffer[256];

				sprintf( buffer, "%.2f%% - %s", relativeSpeed, g_AppTitle );
				SetWindowText( hWnd, buffer );

				firstTick = time;
				cpuFrames = 0;
			}

			if( time < nextFrame && !wndData->m_Throttle )
			{
				Sleep( nextFrame - time );
				nextFrame += FRAME_TICKS;
			}
			else
			{
				nextFrame = time + FRAME_TICKS;
			}
		}
		else
		{
			Sleep( 100 );			
			SendMessage( wndData->m_LCD, WM_REFRESH, 0, 0 );
		}
	}

	return -1; // We should never reach here	
}