// Minimon Copyright (c) 2004-2005 by Robert Bryon Vandiver
//
// This is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
// 

#include "Minimon.h"

Mini::Mini() : MinxCore()
{
	memset( m_FrameBuffer, 0, sizeof(m_FrameBuffer) );

	m_TileBase = 0x1000;
	m_SpriteBase = 0xC000;
	m_Contrast = 0x1F;
	m_IRQFire = 0;
	m_core = 0 ;
	
	m_LCDEnable1 = false;
	m_LCDEnable2 = false;
	m_LCDInverted = false;
	m_MAPActive = false;
	m_SPRActive = false;
	m_Rumble = false;

	m_LCDClock = 0;

	m_SecondCount = 0;
	m_FractSecondCount = 0;	
	
	m_StreamSample = 0;
	m_Volume = 0;

	m_LastBranch = 0;

	m_MapX = 0;
	m_MapY = 0;
	
	m_MapWidth = 12;
	m_MapHeight = 20;

	m_GrayEmulation = false;

	memset( m_Frame, 0, sizeof(m_Frame) );
	memset( m_HardReg, 0, sizeof( m_HardReg ) );
	memset( m_RegAccessMeter, 0, sizeof( m_RegAccessMeter ) );
	memset( m_AudioBuffer, 0, sizeof (m_AudioBuffer) );

	m_AudReadCursor = 0;
	m_AudWriteCursor = sizeof(m_AudioBuffer)/2;
	
	m_HardReg[0x52] = 0xFF;			// Start with no buttons pressed
	m_IOCommand = 0;

	m_ErrorConsole = NULL;
}

Mini::~Mini()
{
	//DestroyWindow( m_ErrorConsole );
}

void Mini::Stream( char* buffer, int length )
{
	

	short duty = *(short*) &m_HardReg[0x4C], 
		freq = *(short*) &m_HardReg[0x4A];

	if( freq == 0 || ~m_HardReg[0x48] & 0x84)
		memset( buffer, 0, length );

	else for( int i = 0; i < length; i++ )
	{		
		m_StreamSample = (m_StreamSample + CPU_FREQ / SAMPLE_RATE) % freq;
		buffer[i] = m_StreamSample < duty ? 0 : m_Volume;
	}
}

void Mini::LoadRom( const char *filename )
{
	MinxCore::LoadRom( filename );
	m_EEPROM.Load( filename );
}

void Mini::SetTime( int seconds )
{
	long *clock = (long*) &m_HardReg[0x8];

	*clock = (seconds << 8) | 0x01;
}

void Mini::FireIRQ( long bit )
{
	m_IRQFire |= bit;
	*(long*) &m_HardReg[0x27] |= bit;
}

void Mini::Shake()
{
	FireIRQ( 0x40000000 );
}

void Mini::PressPad( unsigned char buttons )
{
	FireIRQ( buttons << 16 );
	m_HardReg[0x52] &= ~buttons;
}

void Mini::ReleasePad( unsigned char buttons )
{
	m_HardReg[0x52] |= buttons;
}

void Mini::LogBranch( long addr )
{
	m_LastBranch = addr;
}

int Mini::PredictIRQ( int maxCycles )
{
	int predCycles;

	if( m_HardReg[0x23] & 0x40 )	// Predict time to VDRAW
	{
		predCycles = FRAME_CYCLES - m_LCDClock;

		if( predCycles < maxCycles )
		{
			maxCycles = predCycles;
		}
	}

	if( m_HardReg[0x23] & 0x80 )	// Predict time to VBLANK
	{

		if( m_LCDClock > FRAME_DRAW_TIME )			
			predCycles = FRAME_CYCLES + FRAME_DRAW_TIME - m_LCDClock;
		else
			predCycles = FRAME_DRAW_TIME - m_LCDClock;

		if( predCycles < maxCycles )
		{
			maxCycles = predCycles;
		}
	}

	if( m_HardReg[0x19] & 0x20 )	// Timer master enable
	{
		if( (~m_HardReg[0x30] & 0x01) )
		{
			unsigned short *reset		= (unsigned short*) &m_HardReg[0x32];
			unsigned short *end			= (unsigned short*) &m_HardReg[0x34];
			unsigned short *count		= (unsigned short*) &m_HardReg[0x36];

			if( *end < *reset )
			{
				// Predict trigger
				if( m_HardReg[0x23] & 0x04 )
				{
					if ( m_core )
						predCycles = (m_Timer0Clock - (*end << 16)) >> m_ScalarTicks[ m_HardReg[0x18] & 7 ];
					else
						predCycles = (m_Timer0Clock - (*end << 16)) >> m_ScalarMultiplier[ m_HardReg[0x18] & 7 ];

					if( predCycles < maxCycles )
					{
						maxCycles = predCycles;
					}
				}

				// Predict overflow
				if( m_HardReg[0x23] & 0x08 )
				{
				}
			}
		}
		if( (~m_HardReg[0x38] & 0x01) )
		{
			unsigned short *reset	 = (unsigned short*) &m_HardReg[0x3A];
			unsigned short *end		 = (unsigned short*) &m_HardReg[0x3C];
			unsigned short *count    = (unsigned short*) &m_HardReg[0x3E];

			if( *end < *reset )
			{
				// Predict trigger
				if( m_HardReg[0x23] & 0x10 )
				{
					if ( m_core )
						predCycles = (m_Timer1Clock - (*end << 16)) >> m_ScalarTicks[ m_HardReg[0x1A] & 7 ];
					else
						predCycles = (m_Timer1Clock - (*end << 16)) >> m_ScalarMultiplier[ m_HardReg[0x1A] & 7 ];

					if( predCycles < maxCycles )
					{
						maxCycles = predCycles;
					}
				}
				
				// Predict overflow
				if( m_HardReg[0x23] & 0x20 )
				{
				}
			}
		}
		if( (~m_HardReg[0x48] & 0x01) )
		{
			unsigned short *reset	= (unsigned short*) &m_HardReg[0x4A];
			unsigned short *end		= (unsigned short*) &m_HardReg[0x4C];
			unsigned short *count	= (unsigned short*) &m_HardReg[0x4E];

			if( *end < *reset )
			{
				// Predict trigger
				if( m_HardReg[0x23] & 0x01 )
				{
					if ( m_core )
						predCycles = (m_Timer2Clock - (*end << 16)) >> m_ScalarTicks[ m_HardReg[0x1C] & 7 ];
					else
						predCycles = (m_Timer2Clock - (*end << 16)) >> m_ScalarMultiplier[ m_HardReg[0x1C] & 7 ];

					if( predCycles < maxCycles )
					{
						maxCycles = predCycles;
					}
				}

				// Predict overflow
				if( m_HardReg[0x23] & 0x02 )
				{
				}
			}
		}
	}

	if( maxCycles == 0 )
	{
		maxCycles = 1;
	}

	return maxCycles;
}

void Mini::Clock( int cycles )
{
	// Draw IRQ
	if( (m_LCDClock+cycles) >= FRAME_CYCLES && m_LCDClock < FRAME_CYCLES )
	{
		VPURender();
		FireIRQ( 0x2040 );
	}
	
	// VSYNC IRQ
	if ( m_LCDClock < FRAME_DRAW_TIME )
	{
		if( (m_LCDClock+cycles) >= FRAME_DRAW_TIME )
		{
			UpdateFrame();
			FireIRQ( 0x80 );
		}
	}
	else
	{
		if( (m_LCDClock+cycles) >= FRAME_DRAW_TIME+FRAME_CYCLES )
		{
			UpdateFrame();
			FireIRQ( 0x80 );
		}
	}

	m_LCDClock = (m_LCDClock+cycles);
	
	if( m_LCDClock > FRAME_CYCLES )
	{
		m_LCDClock -= FRAME_CYCLES;
	}

	// Fast timer
	if( m_HardReg[0x40] & 0x1 )
	{
		m_FractSecondCount += cycles;

		while( m_FractSecondCount > (CPU_FREQ / 256) )
		{
			m_FractSecondCount -= (CPU_FREQ / 256);
			m_HardReg[0x41]++;

			if( m_HardReg[0x41] == 0 )
			{
			}
		}
	}

	// Second timer
	if( m_HardReg[0x08] & 0x1 )
	{
		m_SecondCount += cycles;

		while( m_SecondCount > CPU_FREQ )
		{
			m_SecondCount -= CPU_FREQ;

			if( m_HardReg[0x09] == 0xFF )
			{
				if( m_HardReg[0x0A] == 0xFF )
				{
					m_HardReg[0x0B]++;
				}
				m_HardReg[0x0A]++;
			}
			m_HardReg[0x09]++;
		}
	}

	if( m_HardReg[0x19] & 0x20 )
	{
		// Timer 0
		if( ~m_HardReg[0x30] & 0x01 )
		{
			unsigned short *reset		= (unsigned short*) &m_HardReg[0x32];
			unsigned short *end			= (unsigned short*) &m_HardReg[0x34];
			unsigned short *count		= (unsigned short*) &m_HardReg[0x36];

			if( m_Timer0Clock < (*end << 16) )
			{
				m_Timer0Clock = (*reset << 16);
			}

			if( *reset > *end )
			{
				unsigned long  ticks		;
				
				if ( m_core )
					ticks = cycles << m_ScalarTicks[m_HardReg[0x18]&7];
				else
					ticks = cycles << m_ScalarTicksSound[m_HardReg[0x18]&7];

				unsigned long  range		= (*reset - *end)<<16;
				unsigned long  remaining    = m_Timer0Clock - (*end << 16);
				
				if( ticks > range )
				{
//					FireIRQ( 0x04 );
					FireIRQ( 0x08 );

					do
					{
						ticks -= range;
					} while( ticks > range );
				}
				
				if( ticks > remaining )
				{
					if ( m_core )
						m_Timer0Clock += range - ticks;
					else
						m_Timer0Clock = (*reset << 16) + (range - ticks);
//					FireIRQ( 0x04 );
					FireIRQ( 0x08 );
				}
				else
				{
					m_Timer0Clock -= ticks;
				}

				*count = m_Timer0Clock >> 16;
			}
			else
			{
				*count = *reset;
			}
		}

		// Timer 1
		if( ~m_HardReg[0x38] & 0x01 )
		{
			unsigned short *reset		= (unsigned short*) &m_HardReg[0x3A];
			unsigned short *end			= (unsigned short*) &m_HardReg[0x3C];
			unsigned short *count		= (unsigned short*) &m_HardReg[0x3E];
			
			if( m_Timer1Clock < (*end << 16) )
			{
				m_Timer1Clock = (*reset << 16);
			}

			if( *reset > *end )
			{
				unsigned long  ticks		;
				
				if ( m_core )
					ticks = cycles << m_ScalarTicks[m_HardReg[0x18]&7];
				else
					ticks = cycles << m_ScalarTicksSound[m_HardReg[0x18]&7];

				unsigned long  range		= (*reset - *end)<<16;
				unsigned long  remaining    = m_Timer1Clock - (*end << 16);
				
				if( ticks > range )
				{
//					FireIRQ( 0x10 );
					FireIRQ( 0x20 );

					do
					{
						ticks -= range;
					} while( ticks > range );
				}
				
				if( ticks > remaining )
				{
					if ( m_core )
						m_Timer1Clock += range - ticks;
					else
						m_Timer1Clock = (*reset << 16) + (range - ticks);
//					FireIRQ( 0x10 );
					FireIRQ( 0x20 );
				}
				else
				{
					m_Timer1Clock -= ticks;
				}

				*count = m_Timer1Clock >> 16;
			}
			else
			{
				*count = *reset;
			}
		}

		// Timer 2 (Audio)
		if( ~m_HardReg[0x38] & 0x01 )
		{
			unsigned short *reset		= (unsigned short*) &m_HardReg[0x4A];
			unsigned short *end			= (unsigned short*) &m_HardReg[0x4C];
			unsigned short *count		= (unsigned short*) &m_HardReg[0x4E];
			
			if( m_Timer2Clock < (*end << 16) )
			{
				m_Timer2Clock = (*reset << 16);
			}

			if( *reset > *end )
			{
				unsigned long  ticks		;
				
				if ( m_core )
					ticks = cycles << m_ScalarTicks[m_HardReg[0x1C]&7];
				else
					ticks = cycles << m_ScalarTicksSound[m_HardReg[0x1C]&7];

				unsigned long  range		= (*reset - *end)<<16;
				unsigned long  remaining    = m_Timer2Clock - (*end << 16);
				
				if( ticks > range )
				{
//					FireIRQ( 0x01 );
//					FireIRQ( 0x02 );

					do
					{
						ticks -= range;
					} while( ticks > range );
				}
				
				if( ticks > remaining )
				{
					if ( m_core )
						m_Timer2Clock += range - ticks;
					else
						m_Timer2Clock = (*reset << 16) + (range - ticks);
//					FireIRQ( 0x01 );
//					FireIRQ( 0x02 );
				}
				else
				{
					m_Timer2Clock -= ticks;
				}

				*count = m_Timer2Clock >> 16;
			}
			else
			{
				*count = *reset;
			}
		}
	}

	if( m_CpuReg.byte.U == m_CpuReg.byte.V )
		DoIRQs();
}

#define IRQSet( b, i )  case b: ForceIRQ(i<<1); break;

void Mini::DoIRQs()
{
	// Strobe the registers

	if( m_CpuReg.byte.F & FLAG_IE )
	{
		return ;
	}

	// NOTE: There are more ways to mask IRQs that are not currently being used.

	long irqSet = (*(long*) &m_HardReg[0x23]) & m_IRQFire;

	if( irqSet )
	{
		// Locate IRQ
		for( int irq = 0; irq < 32; irq++ )
		{
			if( irqSet & ( 1 << irq ) )
			{
				// Clear the fired IRQ
				m_IRQFire ^= (1 << irq);

				switch( irq )
				{
					IRQSet(  7,  3 );
					IRQSet(  6,  4 );
					IRQSet(  5,  5 );
					IRQSet(  4,  6 );
					IRQSet(  3,  7 );
					IRQSet(  2,  8 );
					IRQSet(  1,  9 );
					IRQSet(  0, 10 );

		//			IRQSet( 15,  3 );
		//			IRQSet( 14,  3 );
					IRQSet( 13, 11 );
					IRQSet( 12, 12 );
					IRQSet( 11, 13 );
					IRQSet( 10, 14 );
		//			IRQSet(  9,  3 );
		//			IRQSet(  8,  3 );

					IRQSet( 23, 21 );
					IRQSet( 22, 22 );
					IRQSet( 21, 23 );
					IRQSet( 20, 24 );
					IRQSet( 19, 25 );
					IRQSet( 18, 26 );
					IRQSet( 17, 27 );
					IRQSet( 16, 28 );

					IRQSet( 31, 15 );
					IRQSet( 30, 16 );
		//			IRQSet( 29, 00 );
		//			IRQSet( 28, 00 );
		//			IRQSet( 27, 00 );
		//			IRQSet( 26, 00 );
		//			IRQSet( 25, 00 );
		//			IRQSet( 24, 00 );
				default:
					continue ;
				}

				return ;
			}
		}
	}
}

void Mini::PeekReg( unsigned char reg, unsigned char &value )
{
	value = m_HardReg[reg] & RegReadMask[reg] | RegOrMask[reg];

	m_RegAccessMeter[reg] = ((m_RegAccessMeter[reg]+1) & 0x3FFFFFFF) | (m_RegAccessMeter[reg] & 0x40000000) | 0x80000000;

	switch( reg )
	{
	case 0x8A:	// VPU Status register
		if( m_LCDClock > FRAME_DRAW_TIME )
		{
			value = 0xFF;	// Should be 0x10
		}
		else
		{
			value = 0;
		}
		// NOTE: Need to figure out other bits

		break ;

	case 0x00: case 0x01: case 0x02:			// CPU control registers (unhandled)
	case 0x60:									// EXT Hardware direction
	case 0x30: case 0x31: case 0x32: case 0x33: // Timers
	case 0x34: case 0x35: case 0x36: case 0x37: 
	case 0x38: case 0x39: case 0x3A: case 0x3B: 
	case 0x3C: case 0x3D: case 0x3E: case 0x3F: 
	case 0x48: case 0x49: case 0x4A: case 0x4B: 
	case 0x4C: case 0x4D: case 0x4E: case 0x4F: 
	case 0x70: case 0x71:						// Volume controls
	case 0x20: case 0x21: case 0x22:			// IRQ Group settings
	case 0x23: case 0x24: case 0x25: case 0x26:	// IRQ Enables
	case 0x27: case 0x28: case 0x29: case 0x2A:	// IRQ Occurance strobes
//	case 0x08: case 0x09: case 0x0A: case 0x0B: // Second timer
	case 0x40: case 0x41:						// Fast timer
	case 0x52:									// Keypad
	case 0x80: case 0x81:						// LCD Control
	case 0x82: case 0x83: case 0x84:			// BG Tile offset
	case 0x85: case 0x86:						// Scrolling registers
	case 0x87: case 0x88: case 0x89:			// OBJ Tile offset
		break ;
	case 0x61:
		if( !(m_HardReg[0x60] & 0x04) )
		{
			value = (value & ~0x04) | (m_EEPROM.Read()?4:0);
		}
		break ;
	default:
		{
			int errorCode[] = {m_CpuReg.word.PC, reg, value};
			Error( "Unhandled register read", 3, errorCode );
		}
		break ;
	}
} 

void Mini::PokeReg( unsigned char reg, unsigned char value )
{
	int old = m_HardReg[reg];
	m_HardReg[reg] = (value & RegWriteMask[reg]) | ( m_HardReg[reg] & ~RegWriteMask[reg] );

	m_RegAccessMeter[reg] = ((m_RegAccessMeter[reg]+1) & 0x3FFFFFFF) | (m_RegAccessMeter[reg] & 0x80000000) | 0x40000000;

	switch( reg )
	{
	case 0x08:		// Second timer control
		if( value & 0x02 )
		{
			m_SecondCount = 0;
			m_HardReg[0x09] = m_HardReg[0x0A] = m_HardReg[0x0B] = 0;
		}
		break ;
	case 0x40:		// 1/256 second control
		if( value & 0x02 )
		{
			m_FractSecondCount = 0;
			m_HardReg[0x41] = 0;
		}
		break ;		
	case 0x30: 
		if( value & 0x02 )
		{
			m_Timer0Clock = 0;
		}
		break ;
	case 0x38: // Timer 0-1
		if( value & 0x02 )
		{
			m_Timer1Clock = 0;
		}
		break ;
	case 0x48: // Timer 2
		if( value & 0x02 )
		{
			m_Timer2Clock = 0;
		}
		
		break ;
	case 0x80:		// LCD Control 0
		m_LCDInverted = (( value & 0x01 ) != 0);
		m_MAPActive = (( value & 0x02 ) != 0);
		m_SPRActive = (( value & 0x04 ) != 0);
		m_LCDEnable1 = (( value & 0x08 ) != 0);

		switch ( value & 0x30 )
		{
		case 0x00:
			m_MapWidth  = 12;
			m_MapHeight = 16;
			break ;
		case 0x10:
			m_MapWidth  = 16;
			m_MapHeight = 12;
			break ;
		case 0x20:
			m_MapWidth  = 24;
			m_MapHeight = 8;
			break ;
		case 0x30:
			m_MapWidth = 24;
			m_MapHeight = 16;
			break ;
		}
		break ;
	case 0x81:		// LCD Control 1
		m_LCDEnable2 = (( value & 0x01 ) != 0 );
		// TODO: Process other bits
		break ;
	case 0x82: case 0x83: case 0x84:	// BG Tile offset
		m_TileBase = (m_HardReg[0x84] << 16) | (m_HardReg[0x83] << 8) |  m_HardReg[0x82];
		break ;
	case 0x85:		// Map X offset
		m_MapY = m_HardReg[0x85];
		break ;
	case 0x86:		// Map Y offset
		m_MapX = m_HardReg[0x86];
		break ;
	case 0x87: case 0x88: case 0x89:	// OBJ Tile offset
		m_SpriteBase = (m_HardReg[0x89] << 16) | (m_HardReg[0x88] << 8) |  m_HardReg[0x87];
		break ;
	case 0xFE:
		m_IOCommand = old;
	case 0xFF:
		if( reg == 0xFF )
		{
			m_IOCommand = m_HardReg[0xFE];
		}

		switch( m_IOCommand )
		{
		case 0x81:
			m_Contrast = value;
		}

		m_IOCommand = 0;

		break ;

	case 0x27: case 0x28: case 0x29: case 0x2A:		// IRQ Occurance strobes
		m_HardReg[reg] &= ~value;
		return ;

	case 0x71:
		switch(value & 0x3)
		{
		case 0:
			m_Volume = 0;
			break ;
		case 1: case 2:
			m_Volume = 0x40;
			break ;
		case 3:
			m_Volume = 0x80;
			break ;
		}

		break ;

	case 0x00: case 0x01: case 0x02:				// CPU control registers (unhandled)
	case 0x20: case 0x21: case 0x22:				// IRQ Group settings
	case 0x23: case 0x24: case 0x25: case 0x26:		// IRQ Enables
	case 0x32: case 0x33:							// Timers
	case 0x34: case 0x35: case 0x36: case 0x37: 
	case 0x3A: case 0x3B: case 0x3C: case 0x3D: 
	case 0x3E: case 0x3F: 
	case 0x4A: case 0x4B: case 0x4C: case 0x4D: 	// Audio settings
	case 0x4E: case 0x4F: 
//	case 0x09: case 0x0A: case 0x0B:				// Second counter is read only
	case 0x41:										// Fast counter is read only
		break ;
	case 0x60:										// Ext Hardware
	case 0x61:
		m_Rumble = ( m_HardReg[0x60] & m_HardReg[0x61] & 0x10 ) ? true : false;

		m_EEPROM.Write( (m_HardReg[0x61] & m_HardReg[0x60] & 0x04) != 0 );
		m_EEPROM.Clock( (m_HardReg[0x61] & m_HardReg[0x60] & 0x08) != 0 );
		break ;
	default:
		{
			int errorCode[] = {m_CpuReg.word.PC, reg, value };
			
			Error( "Unhandled register write", 3, errorCode );
		}
		break ;
	}
}

unsigned char Mini::SafeRead( long addr, bool paged )
{
	if( (addr & 0x8000) && paged )
	{
		addr = (addr & 0x7FFF) | (m_CpuReg.word.V << 15);
	}
	else if( paged )
	{
		addr &= 0x7FFF;
	}
	
	if( addr < 0x1000 )
	{
		return m_Bios[addr];
	}
	else if( addr < 0x2000 )
	{
		return m_Ram[addr & 0xFFF];
	}
	else if( addr < 0x2100 )
	{
		return m_HardReg[addr & 0xFF];
	}
	else if( m_Rom != NULL )
	{
		return m_Rom[ addr & m_RomMask ];
	}

	return 0;
}

unsigned char *Mini::SafePointer( long addr )
{
	if( addr < 0x1000 )
	{
		return &m_Bios[addr];
	}
	else if( addr < 0x2000 )
	{
		return &m_Ram[addr & 0xFFF];
	}
	else if( addr < 0x2100 )
	{
		return &m_HardReg[addr & 0xFF];
	}
	else if( m_Rom != NULL )
	{
		return &m_Rom[ addr & m_RomMask ];
	}

	return 0;
}

void Mini::DebugWrite( long addr, unsigned char data, bool paged )
{
	if( (addr & 0x8000) && paged )
	{
		addr = (addr & 0x7FFF) | (m_CpuReg.word.V << 15);
	}
	else if( paged )
	{
		addr &= 0x7FFF;
	}

	if( addr < 0x1000 )
	{
		m_Bios[addr] = data;
	}
	else if( addr < 0x2000 )
	{
		m_Ram[addr & 0xFFF] = data;
	}
	else if( addr < 0x2100 )
	{
		m_HardReg[addr & 0xFF] = data;
	}
	else if( m_Rom != NULL )
	{
		m_Rom[ addr & m_RomMask ] = data;
	}
}

void Mini::VPURender()
{
	unsigned char map[256][256];
	unsigned char *decodedBuffer = &map[m_MapY][m_MapX];
	unsigned char invert;
	int x,y,tx,ty,b;
	
	// VPU Inactive
	if( !m_MAPActive && !m_SPRActive )
		return ;

	unsigned char *frame = decodedBuffer;
	unsigned char *serialRam = m_Ram;

	for( y = 0; y < 8; y++ )
	{
		for( x = 0; x < 96; x++ )
		{
			for( b = 0; b < 8; b++ )
				frame[b*256] = (*serialRam >> b) & 1;
			
			serialRam++;
			frame++;
		}

		frame += 256 * 7 + (256-96);
	}

	// Unpack map without wrapping
	if( m_MAPActive )
	{
		invert = m_LCDInverted ? 0xFF : 0;

		unsigned char *mapData = m_Ram + 0x360;
		frame = map[0];

		for( ty = 0; ty < m_MapHeight; ty++ )
		{
			for( tx = 0; tx < m_MapWidth; tx++ )
			{
				unsigned char *tileData = SafePointer( *(mapData++) * 8 + m_TileBase );

				for( x = 0; x < 8; x++ )
				{
					// Potential chance for optomization: 
					unsigned char byte = *(tileData++) ^ invert;

					for( b = 0; b < 8; b++ )
					{
						frame[b*256] = (byte >> b) & 1;
					}
					frame++;
				}
			}

			frame += (256 - m_MapWidth*8) + (256 * 7);
		}
	}
	if( m_SPRActive )
	{
		serialRam = m_Ram + 0x300 + 24 * 4 - 1;

		for( int s = 0; s < 24; s++ )
		{
			int sf = *(serialRam--);

			// Preabort on a disabled sprite
			if( !(sf & 0x08) )
			{
				serialRam -= 3;
				continue ;
			}

			invert = (sf & 0x04) ? 0xFF : 0;

			// Coutners for masks
			unsigned char *mask1addr = SafePointer(*(serialRam--) * 64 + m_SpriteBase);
			unsigned char *mask2addr =  mask1addr + 32;
			unsigned char *pixel1addr = mask1addr + 16;
			unsigned char *pixel2addr = mask1addr + 48;

			int sy = *(serialRam--) - 16 + m_MapY;
			int sx = *(serialRam--) - 16 + m_MapX;

			// Set the pointer to the map
			frame = &map[sy][sx];

			// Draw tops
			for( x = 0; x < 8; x++ )
			{
				unsigned char mask_a  = ~*(mask1addr++);
				unsigned char mask_b  = ~*(mask2addr++);
				unsigned char pixel_a = *(pixel1addr++) ^ invert;
				unsigned char pixel_b = *(pixel2addr++) ^ invert;

				unsigned char dx, dy;

				for( y = 0; y < 8; y++ )
				{
					dx = ((sf & 0x1) ? (15-x) : (  x)) + sx;
					dy = ((sf & 0x2) ? (15-y) : (  y)) + sy;

					if( (mask_a >> y) & 1 )
					{
						map[dy][dx] = (pixel_a >> y) & 1;
					}

					dx = ((sf & 0x1) ? ( 7-x) : (x+8)) + sx;
					dy = ((sf & 0x2) ? (15-y) : (  y)) + sy;

					if( (mask_b >> y) & 1 )
					{
						map[dy][dx] = (pixel_b >> y) & 1;
					}
				}
			}

			// Draw bottoms
			for( x = 0; x < 8; x++ )
			{
				unsigned char mask_a  = ~*(mask1addr++);
				unsigned char mask_b  = ~*(mask2addr++);
				unsigned char pixel_a = *(pixel1addr++) ^ invert;
				unsigned char pixel_b = *(pixel2addr++) ^ invert;

				unsigned char dx, dy;

				for( y = 0; y < 8; y++ )
				{
					dx = ((sf & 0x1) ? (15-x) : (  x)) + sx;
					dy = ((sf & 0x2) ? ( 7-y) : (y+8)) + sy;

					if( (mask_a >> y) & 1 )
					{
						map[dy][dx] = (pixel_a >> y) & 1;
					}

					dx = ((sf & 0x1) ? ( 7-x) : (x+8)) + sx;
					dy = ((sf & 0x2) ? ( 7-y) : (y+8)) + sy;

					if( (mask_b >> y) & 1 )
					{
						map[dy][dx] = (pixel_b >> y) & 1;
					}
				}
			}
		}
	}

	// Rencode the screen, With proper addesses and all
	frame = decodedBuffer;
	serialRam = m_Ram;

	for( int ty = 0; ty < 8; ty++ )
	{
		for( int tx = 0; tx < 96; tx++ )
		{
			char byte = *frame;

			for( int b = 1; b < 8; b++ )
			{
				byte |= frame[b*256] << b;
			}
			
			*serialRam = byte;

			serialRam++;
			frame++;
		}

		frame += 256 * 7 + (256-96);
	}
}

#define __JUSTBURN_GRAY

void Mini::UpdateFrame()
{
	unsigned char loColor, hiColor;
	int x,y,b;

	if( m_LCDEnable1 )
	{
		memcpy( m_Frame, m_Ram, sizeof(m_Frame) );
	}

	if( !m_LCDEnable2 )
	{
		loColor = hiColor = 0;
	}
	else  if( m_Contrast <= 0x1F )
	{
		loColor = 0;
		hiColor = (short)0xFF * m_Contrast / 0x1F;
	}
	else if( m_Contrast <= 0x3F )
	{
		loColor = (short)0xFF * (m_Contrast - 0x1F) / 0x1F;
		hiColor = 0xFF;
	}
	else
	{
		loColor = 0xFF;
		hiColor = 0xFF;
	}	
	
	unsigned char gray, color;

	if( m_GrayEmulation )
	{
		for( y = 0; y < 8; y++ )
			for( x = 0; x < 96; x++ )
			{
				gray  = m_Frame[y * 96 + x] ^ m_PrevFrame[y * 96 + x];
				color = m_Frame[y * 96 + x] & m_PrevFrame[y * 96 + x];

				for( b = 0; b < 8; b++ )
				{
					if( (gray >> b) & 1 )
						m_FrameBuffer[y*8+b][x] = ((hiColor - loColor) >> 1) + loColor;
					else
						m_FrameBuffer[y*8+b][x] = ((color>>b)&1) ? hiColor : loColor;
				}
			}
		memcpy( m_PrevFrame, m_Frame, 96 * 8 );
	}
	else
	{
		for( y = 0; y < 8; y++ )
			for( x = 0; x < 96; x++ )
			{
				color = m_Frame[y * 96 + x];

				for( b = 0; b < 8; b++ )
				{
					m_FrameBuffer[y*8+b][x] = (((color>>b)&1) ? hiColor : loColor)*1/3 + (m_FrameBuffer[y*8+b][x])*2/3;
//					m_FrameBuffer[y*8+b][x] = ((color>>b)&1) ? hiColor : loColor;
				}
			}
	}

}

void Mini::EmulateGray( bool emulate )
{
	m_GrayEmulation = emulate;
}

void Mini::SetConsole( HWND errorConsole )
{
	m_ErrorConsole = errorConsole;
}

void Mini::Error( const char *error, int errc, int *errv )
{
	char buffer[256],num[16];

	strcpy( buffer, "Execution error (" );
	for( int i = 0; i < errc; i++ )
	{
		strcat( buffer, strupr(itoa( errv[i], num, (i==2)?2:16 )) );

		if( i != (errc - 1) )
			strcat( buffer, " " );
	}
	strcat( buffer, "): " );
	strcat( buffer, error );

//	fprintf( logfo, "%s\n", buffer );

	//SetWindowText( m_ErrorConsole, buffer );
}
