/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Minimon.h"

int parseNum( char *arg )
{
	if( *arg == '0' )
	{
		switch( arg[1] )
		{
		case 'B':
			return strtol( arg, NULL, 2 );
		case 'X':
			return strtol( arg, NULL, 16 );
		default:
			return strtol( arg, NULL, 10 );
		}
	}
	else
	{
		return atoi( arg );
	}
}

int parseArg( char *arg, int &data )
{
	char *copy = arg, *src = arg;

	if( arg == NULL )
	{
		return NO_IMMEDIATE;
	}

	while( *src )
	{
		if( *src == ' ' )
		{
			src++;
			continue ;
		}

		*(copy++) = toupper(*(src++));
	}

	*copy = 0;

	// Data processing here

	if( *arg == '[' )
	{
		copy = strstr( ++arg, "]" );

		if( copy == NULL )
		{
			return NO_IMMEDIATE;
		}
		
		*copy = 0;

		if( *arg >= '0' && *arg <= '9' )
		{
			data = parseNum( arg );
		}		
		else if( strcmp( arg, "HL" ) == 0 )
		{
			return ADDR_HL;
		}
		else if( strcmp( arg, "X" ) == 0 )
		{
			return ADDR_X;
		}
		else if( strcmp( arg, "Y" ) == 0 )
		{
			return ADDR_Y;
		}
		else if( strcmp( arg, "X+L" ) == 0 )
		{
			return ADDR_X_L;
		}
		else if( strcmp( arg, "Y+L" ) == 0 )
		{
			return ADDR_Y_L;
		}
		else if( strstr( arg, "SP+" ) == arg )
		{
			data = parseNum( arg+3 );
			return ADDR_SP_I8;
		}		
		else if( strstr( arg, "X+" ) == arg )
		{
			data = parseNum( arg+2 );
			return ADDR_X_I8;
		}		
		else if( strstr( arg, "Y+" ) == arg )
		{
			data = parseNum( arg+2 );
			return ADDR_Y_I8;
		}		
		else if( strstr( arg, "I+N+" ) == arg )
		{
			data = parseNum( arg+4 );
			return ADDR_N_I8;
		}		
		else if( strstr( arg, "I+" ) == arg )
		{
			data = parseNum( arg+2 );
			return ADDR_I_I16;
		}		

		// process address
	}
	else if( *arg >= '0' && *arg <= '9' )
	{
		data = parseNum( arg );
		return IMM_8;
	}
	else if( strcmp( arg, "A" ) == 0 )
		return REG_A;
	else if( strcmp( arg, "B" ) == 0 )
		return REG_B;
	else if( strcmp( arg, "H" ) == 0 )
		return REG_H;
	else if( strcmp( arg, "L" ) == 0 )
		return REG_L;
	else if( strcmp( arg, "U" ) == 0 )
		return REG_U;
	else if( strcmp( arg, "V" ) == 0 )
		return REG_V;
	else if( strcmp( arg, "F" ) == 0 )
		return REG_F;
	else if( strcmp( arg, "E" ) == 0 )
		return REG_E;
	else if( strcmp( arg, "I" ) == 0 )
		return REG_I;
	else if( strcmp( arg, "N" ) == 0 )
		return REG_N;
	else if( strcmp( arg, "XI" ) == 0 )
		return REG_XI;
	else if( strcmp( arg, "YI" ) == 0 )
		return REG_YI;
	else if( strcmp( arg, "X" ) == 0 )
		return REG_X;
	else if( strcmp( arg, "Y" ) == 0 )
		return REG_Y;
	else if( strcmp( arg, "BA" ) == 0 )
		return REG_BA;
	else if( strcmp( arg, "HL" ) == 0 )
		return REG_HL;
	else if( strcmp( arg, "SP" ) == 0 )
		return REG_SP;
	else if( strcmp( arg, "PC" ) == 0 )
		return REG_PC;

	return NO_IMMEDIATE;
}
void MessageBox( HWND h, const char *c, const char *c2, int ii ) 
{
}

#define MB_OK 0

void Assemble( char *code, long address, Mini *mini )
{
	char *opcode = NULL,
		*size = NULL,
		*imm_a = NULL,
		*imm_b = NULL;

	while( true )
	{
		while( *code == ' ' ) code++;

		opcode = code;

		while( *code != ' ' && *code != 0 ) code++;

		if( *code != 0 )
			*(code++) = 0;
		else
			break ;

		while( *code == ' ' ) code++;

		if( strstr( code, "byte" ) == code ||
			strstr( code, "word" ) == code ||
			strstr( code, "short" ) == code ||
			strstr( code, "long" ) == code
			)
		{
			size = code;

			while( *code != ' ' && *code != 0 ) code++;
			if( *code != 0 )
				*(code++) = 0;
			else
				break ;
			while( *code == ' ' ) code++;
		}

		if( *code != 0 )
			imm_a = code;
		else
			break ;

		while( *code != ',' && *code != 0 ) code++;
		
		if( *code != 0 )
			*(code++) = 0;
		else
			break ;

		while( *code == ' ' ) code++;

		if( *code != 0 )
			imm_b = code;

		break ;
	}	

	int arg_a, arg_b;
	int type_a, type_b;

	type_a = parseArg( imm_a, arg_a );
	type_b = parseArg( imm_b, arg_b );

	for( int i = 0; i < 0x300; i++ )
	{
		if( Mnemonic[i].m_Mnemonic == NULL )
			continue ;

		if( stricmp( opcode, Mnemonic[i].m_Mnemonic ) == 0 )
		{
			if( Mnemonic[i].m_Extender != NULL && (size == NULL || stricmp( Mnemonic[i].m_Extender, size ) != 0) )
				continue;

			switch( Mnemonic[i].m_Op[0] )
			{				
				case ADDR_PC_I8:
				case ADDR_PC_I16:
				case IMM_8:
				case IMM_16:
					if( type_a != IMM_8 )
						continue ;
					break ;
				default:
					if( Mnemonic[i].m_Op[0] != type_a )
						continue ;
			}
			
			switch( Mnemonic[i].m_Op[1] )
			{				
				case ADDR_PC_I8:
				case ADDR_PC_I16:
				case IMM_8:
				case IMM_16:
					if( type_b != IMM_8 )
						continue ;
					break ;
				default:
					if( Mnemonic[i].m_Op[1] != type_b )
						continue ;
			}

			if( i > 0x200 )
				mini->DebugWrite( address++, 0xCF, false );
			else if( i > 0x100 )
				mini->DebugWrite( address++, 0xCE, false );

			mini->DebugWrite( address++, i & 0xFF, false );

			switch( Mnemonic[i].m_Op[0] )
			{
				case IMM_16:
				case ADDR_I16:
				case ADDR_I_I16:
					mini->DebugWrite( address++, arg_a & 0xFF, false );
					mini->DebugWrite( address++, (arg_a >> 8) & 0xFF, false );
					return ;
				case ADDR_PC_I16:
					{
						int offset = arg_a - address - 1;
						mini->DebugWrite( address++, offset & 0xFF, false );
						mini->DebugWrite( address++, (offset >> 8) & 0xFF, false );
					}
					return ;
				case IMM_8:
				case ADDR_N_I8:
				case ADDR_SP_I8:
				case ADDR_X_I8:
				case ADDR_Y_I8:
					mini->DebugWrite( address++, arg_a & 0xFF, false );
					break ;
				case ADDR_PC_I8:
					{
						int offset = arg_a - address;
						mini->DebugWrite( address++, offset & 0xFF, false );
					}
					break ;
			}
			
			switch( Mnemonic[i].m_Op[1] )
			{
				case IMM_8:
				case ADDR_N_I8:
				case ADDR_SP_I8:
				case ADDR_X_I8:
				case ADDR_Y_I8:
					mini->DebugWrite( address++, arg_b & 0xFF, false );
					break ;
			}
			
			return ;
		}
	}

	MessageBox( NULL, "Invalid instruction", g_AppTitle, MB_OK );
}

char *paditoa( int value, char *buffer, int radix, int length )
{
	char ascii[16];
	char *source = ascii, *dest = buffer;

	itoa( value, ascii, radix );
	length = length - strlen(ascii);

	while( length > 0 )
	{
		*dest = '0';
		dest++;
		length --;
	}
	while( *source )
	{
		*dest = toupper(*source);
		source++;
		dest++;
	}
	*dest = 0;

	return buffer;
}

DasmValue *DrawAssembly( DisasmData *wndData, DasmValue *valueBlock, HDC hDC, HDC fontDC, RECT *rect, long address )
{
#if 0
	Mini *mini = wndData->m_Emulator;
	RegSet *rs = &(wndData->m_Emulator->m_CpuReg);

	unsigned char imm_a,imm_b;
	unsigned short imm_w;
	char buffer[256];
	int op;

	int lines = (rect->bottom - rect->top) / 16;

	for( int line = 0; line <= lines; line++ ) {
		long bankedAddress = (address & 0x8000) ? ((address&0x7FFF)|(rs->byte.V<<15)):(address&0x7FFF);

		op = mini->SafeRead( address, true );

		rect->bottom = rect->top + 16;
		
		valueBlock->rect.left   = 152+8*18;
		valueBlock->rect.right  = valueBlock->rect.left + 8;
		valueBlock->rect.top    = rect->top;
		valueBlock->rect.bottom = rect->bottom;
		valueBlock->m_Mode      = VALUE_LINE_ADDRESS;
		valueBlock->m_Value     = bankedAddress;
		valueBlock++;

		DrawLine( hDC, fontDC, wndData->m_Colors->m_brText, rect->left = 4, rect->top, 
			paditoa( bankedAddress, buffer, 16, 6) );

		if( op == 0xCE || op == 0xCF)
		{
			op = mini->SafeRead( address + 1, true ) | ((op == 0xCE) ? 0x100 : 0x200);

			imm_a = mini->SafeRead( address + 2, true );
			imm_b = mini->SafeRead( address + 3, true );
			imm_w = imm_b << 8 | imm_a;
		}
		else
		{					
			imm_a = mini->SafeRead( address + 1, true );
			imm_b = mini->SafeRead( address + 2, true );
			imm_w = imm_b << 8 | imm_a;
		}
		
		rect->left = 64;
		for( int i = 0; i < Mnemonic[op].m_Length; i++ )
		{
			DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left+=4, rect->top, 
				paditoa( mini->SafeRead( address+i, true ), buffer, 16, 2 ) );
		}

		rect->right = 164;
		FillRect( hDC, rect, wndData->m_Colors->m_brClear );

		char *lable = wndData->m_Executor->GetSymbol( bankedAddress );

		if( lable != NULL )
		{
			DrawLine( hDC, fontDC, wndData->m_Colors->m_brLable, rect->left=156, rect->top, lable);
			DrawLine( hDC, fontDC, wndData->m_Colors->m_brText, rect->left, rect->top, ":");
		}

		rect->right = 164+8*18;
		FillRect( hDC, rect, wndData->m_Colors->m_brClear );

		if( address == rs->word.PC )
		{
			BitBlt( hDC, 152+8*18, rect->top, 8, 16, fontDC, 0, 128, SRCAND );
		}

		if( wndData->m_Executor->GetBreakpoint( address ) )
		{
			BitBlt( hDC, 152+8*18, rect->top, 8, 16, fontDC, 8, 128, SRCAND );
		}

		valueBlock->rect.left   = rect->left;
		valueBlock->rect.top    = rect->top;
		valueBlock->rect.bottom = rect->bottom;
		valueBlock->m_Mode      = VALUE_OP;
		valueBlock->m_Value     = bankedAddress;

		DrawLine( hDC, fontDC, wndData->m_Colors->m_brMnemonic, rect->left=164+8*18, rect->top, Mnemonic[op].m_Mnemonic);
		rect->right = rect->left+8;
		FillRect( hDC, rect, wndData->m_Colors->m_brClear );
		DrawLine( hDC, fontDC, wndData->m_Colors->m_brMnemonic, rect->left+=8, rect->top, Mnemonic[op].m_Extender);

		valueBlock->rect.right  = rect->left;
		valueBlock++;

		rect->right = 164 + 8 * (32);
		FillRect( hDC, rect, wndData->m_Colors->m_brClear );

		rect->left = 164 + 8 * (32);

		address += Mnemonic[op].m_Length;

		for( int i = 0; i < 2; i++ )
		{
			if( i != 0 && Mnemonic[op].m_Op[i] != NO_IMMEDIATE )
			{
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, ", " );
			}

			valueBlock->rect.left   = rect->left;
			valueBlock->rect.top    = rect->top;
			valueBlock->rect.bottom = rect->bottom; 

			switch( Mnemonic[op].m_Op[i] )
			{
			case NO_IMMEDIATE:
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case IMM_8:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( imm_a, buffer, 16,2 ) );

				valueBlock->m_Mode = VALUE_NONE;

				imm_a = imm_b;
				break ;
			case IMM_16:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( imm_w, buffer, 16,4 ) );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_PC:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "PC" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_SP:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "SP" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_BA:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "BA" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_HL:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "HL" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_X:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "X" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_Y:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "Y" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_A:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "A" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_B:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "B" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_H:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "H" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_L:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "L" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_U:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "U" );

				valueBlock->m_Mode = VALUE_NONE;

				break ;
			case REG_V:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "V" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_F:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "F" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_E:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "E" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_XI:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "XI" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_YI:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "YI" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_N:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "N" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case REG_I:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "I" );
				valueBlock->m_Mode = VALUE_NONE;
				break ;
			case ADDR_I16:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( imm_w, buffer, 16,4 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = imm_w;
				break ;
			case ADDR_I_I16:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "I" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( imm_w, buffer, 16,4 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );

				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = imm_w | (wndData->m_Emulator->m_CpuReg.byte.I << 16);

				break ;
			case ADDR_N_I8:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "I" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "N" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( imm_a, buffer, 16,2 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );

				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = imm_a | 
					(wndData->m_Emulator->m_CpuReg.byte.N << 8) | 
					(wndData->m_Emulator->m_CpuReg.byte.I << 16);

				imm_a = imm_b;
				break ;
			case ADDR_HL:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "HL" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );

				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = wndData->m_Emulator->m_CpuReg.addr.HLA;

				break ;
			case ADDR_X:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "X" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );

				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = wndData->m_Emulator->m_CpuReg.addr.XA;

				break ;
			case ADDR_Y:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "Y" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );

				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = wndData->m_Emulator->m_CpuReg.addr.YA;

				break ;
			case ADDR_PC_I8:
				{
					long jump = (address + (signed char)(imm_a) - 1) & 0xFFFF;

					if( jump & 0x8000 )
					{
						jump = (jump & 0x7FFF) | ( rs->byte.V << 15 );
					}

					valueBlock->m_Mode = VALUE_ADDRESS;
					valueBlock->m_Value = jump;

					char *symbol = wndData->m_Executor->GetSymbol( jump );

					if( symbol != NULL )
					{
						DrawLine( hDC, fontDC, wndData->m_Colors->m_brLable, rect->left, rect->top, symbol );
					}
					else
					{
						DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( jump, buffer, 16,6 ) );
					}
				}
				imm_a = imm_b;
				break ;
			case ADDR_PC_I16:
				{
					long jump = (address + (signed short)(imm_w) - 1) & 0xFFFF;

					if( jump & 0x8000 )
					{
						jump = (jump & 0x7FFF) | ( rs->byte.V << 15 );
					}

					valueBlock->m_Mode = VALUE_ADDRESS;
					valueBlock->m_Value = jump;

					char *symbol = wndData->m_Executor->GetSymbol( jump );

					if( symbol != NULL )
					{
						DrawLine( hDC, fontDC, wndData->m_Colors->m_brLable, rect->left, rect->top, symbol );
					}
					else
					{
						DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, paditoa( jump, buffer, 16,6 ) );
					}
				}
				break ;
			case ADDR_SP_I8:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "I" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "SP" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, itoa( (signed)imm_a, buffer, 10 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = (wndData->m_Emulator->m_CpuReg.word.SP + (signed) imm_a & 0xFFFF);
				imm_a = imm_b;
				break ;
			case ADDR_X_I8:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "X" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, itoa( imm_a, buffer, 16 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = (wndData->m_Emulator->m_CpuReg.word.X + (signed) imm_a & 0xFFFF) |
					(wndData->m_Emulator->m_CpuReg.byte.XI << 16);
				imm_a = imm_b;
				break ;
			case ADDR_Y_I8:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "Y" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brLiteral, rect->left, rect->top, itoa( imm_a, buffer, 16 ) );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = (wndData->m_Emulator->m_CpuReg.word.Y + (signed) imm_a & 0xFFFF) |
					(wndData->m_Emulator->m_CpuReg.byte.YI << 16);
				imm_a = imm_b;
				break ;
			case ADDR_X_L:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "X" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "L" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = (wndData->m_Emulator->m_CpuReg.word.X + (signed) wndData->m_Emulator->m_CpuReg.byte.L & 0xFFFF) |
					(wndData->m_Emulator->m_CpuReg.byte.XI << 16);
				break ;
			case ADDR_Y_L:
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "[" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "Y" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brOperator, rect->left, rect->top, "+" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brRegister, rect->left, rect->top, "L" );
				DrawLine( hDC, fontDC, wndData->m_Colors->m_brSymbol, rect->left, rect->top, "]" );
				valueBlock->m_Mode = VALUE_MEMORY;
				valueBlock->m_Value = (wndData->m_Emulator->m_CpuReg.word.Y + (signed) wndData->m_Emulator->m_CpuReg.byte.L & 0xFFFF) |
					(wndData->m_Emulator->m_CpuReg.byte.YI << 16);
				break ;
			}

			valueBlock->rect.right = rect->left;
			valueBlock++;
		}

		rect->right = 164 + 8 * (51);
		FillRect( hDC, rect, wndData->m_Colors->m_brClear );

//		rect->left = 164 + 8 * (51);
//		DrawLine( hDC, fontDC, wndData->m_Colors->m_brComment, rect->left, rect->top, "; --- COMMENTARY HERE ---" );
		rect->top += 16;
	}	

	valueBlock->m_Mode = VALUE_TERM;
#endif
	return valueBlock;
}

char *disassemble( Mini *mini, long address, char *output )
{
	RegSet *rs = &(mini->m_CpuReg);

	unsigned char imm_a,imm_b;
	unsigned short imm_w;
	char buffer[256];
	int op;

	long bankedAddress = (address & 0x8000) ? ((address&0x7FFF)|(rs->byte.V<<15)):(address&0x7FFF);

	op = mini->SafeRead( address, true );

	strcpy( output, paditoa( bankedAddress, buffer, 16, 6) );
	strcat( output, " " );

	if( op == 0xCE || op == 0xCF)
	{
		op = mini->SafeRead( address + 1, true ) | ((op == 0xCE) ? 0x100 : 0x200);

		imm_a = mini->SafeRead( address + 2, true );
		imm_b = mini->SafeRead( address + 3, true );
		imm_w = imm_b << 8 | imm_a;
	}
	else
	{					
		imm_a = mini->SafeRead( address + 1, true );
		imm_b = mini->SafeRead( address + 2, true );
		imm_w = imm_b << 8 | imm_a;
	}
	
//	for( int i = 0; i < Mnemonic[op].m_Length; i++ )
//	{
//		strcat( output, paditoa( mini->SafeRead( address+i, true ), buffer, 16, 2 ) );
//		strcat( output, " " );
//	}

	strcat( output, Mnemonic[op].m_Mnemonic );
	
	if( Mnemonic[op].m_Extender != NULL )
	{
		strcat( output, " " );
		strcat( output, Mnemonic[op].m_Extender );
	}

	address += Mnemonic[op].m_Length;

	for( int i = 0; i < 2; i++ )
	{
		if( i != 0 && Mnemonic[op].m_Op[i] != NO_IMMEDIATE )
		{
			strcat( output, ", " );
		}
		else
		{
			strcat( output, " " );
		}

		switch( Mnemonic[op].m_Op[i] )
		{
		case NO_IMMEDIATE:
			break ;
		case IMM_8:
			strcat( output, paditoa( imm_a, buffer, 16,2 ) );
			imm_a = imm_b;
			break ;
		case IMM_16:
			strcat( output, paditoa( imm_w, buffer, 16,4 ) );
			break ;
		case REG_PC:
			strcat( output, "PC" );
			break ;
		case REG_SP:
			strcat( output, "SP" );
			break ;
		case REG_BA:
			strcat( output, "BA" );
			break ;
		case REG_HL:
			strcat( output, "HL" );
			break ;
		case REG_X:
			strcat( output, "X" );
			break ;
		case REG_Y:
			strcat( output, "Y" );
			break ;
		case REG_A:
			strcat( output, "A" );
			break ;
		case REG_B:
			strcat( output, "B" );
			break ;
		case REG_H:
			strcat( output, "H" );
			break ;
		case REG_L:
			strcat( output, "L" );
			break ;
		case REG_U:
			strcat( output, "U" );
			break ;
		case REG_V:
			strcat( output, "V" );
			break ;
		case REG_F:
			strcat( output, "F" );
			break ;
		case REG_E:
			strcat( output, "E" );
			break ;
		case REG_XI:
			strcat( output, "XI" );
			break ;
		case REG_YI:
			strcat( output, "YI" );
			break ;
		case REG_N:
			strcat( output, "N" );
			break ;
		case REG_I:
			strcat( output, "I" );
			break ;
		case ADDR_I16:
			strcat( output, "[" );
			strcat( output, paditoa( imm_w, buffer, 16,4 ) );
			strcat( output, "]" );
			break ;
		case ADDR_I_I16:
			strcat( output, "[I+" );
			strcat( output, paditoa( imm_w, buffer, 16,4 ) );
			strcat( output, "]" );
			break ;
		case ADDR_N_I8:
			strcat( output, "[I+N+" );
			strcat( output, paditoa( imm_a, buffer, 16,2 ) );
			strcat( output, "]" );
			imm_a = imm_b;
			break ;
		case ADDR_HL:
			strcat( output, "[HL]" );
			break ;
		case ADDR_X:
			strcat( output, "[X]" );
			break ;
		case ADDR_Y:
			strcat( output, "[Y]" );
			break ;
		case ADDR_PC_I8:
			{
				long jump = (address + (signed char)(imm_a) - 1) & 0xFFFF;

				if( jump & 0x8000 )
				{
					jump = (jump & 0x7FFF) | ( rs->byte.V << 15 );
				}

				strcat( output, paditoa( jump, buffer, 16,6 ) );
			}
			imm_a = imm_b;
			break ;
		case ADDR_PC_I16:
			{
				long jump = (address + (signed short)(imm_w) - 1) & 0xFFFF;

				if( jump & 0x8000 )
				{
					jump = (jump & 0x7FFF) | ( rs->byte.V << 15 );
				}

				strcat( output, paditoa( jump, buffer, 16,6 ) );
			}
			break ;
		case ADDR_SP_I8:
			strcat( output, "[I+SP+" );
			strcat( output, itoa( (signed)imm_a, buffer, 10 ) );
			strcat( output, "]" );
			imm_a = imm_b;
			break ;
		case ADDR_X_I8:
			strcat( output, "[X+" );
			strcat( output, itoa( imm_a, buffer, 16 ) );
			strcat( output, "]" );
			imm_a = imm_b;
			break ;
		case ADDR_Y_I8:
			strcat( output, "[Y+" );
			strcat( output, itoa( imm_a, buffer, 16 ) );
			strcat( output, "]" );
			imm_a = imm_b;
			break ;
		case ADDR_X_L:
			strcat( output, "[X+L]" );
			break ;
		case ADDR_Y_L:
			strcat( output, "[Y+L]" );
			break ;
		}
	}

	return output;
}

BOOL CALLBACK OpcodeProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
#if 0
	static char op[MAX_PATH];

	switch( message )
	{
	case WM_SHOWWINDOW:
		SetFocus( GetDlgItem( hDlg, IDC_OP ) );
		return true;
	case WM_COMMAND:
		switch( wParam )
		{
		case IDOK:
			GetDlgItemText( hDlg, IDC_OP, op, sizeof(op) );
			EndDialog( hDlg, (INT_PTR)op );
			break ;
		case IDCANCEL:
			EndDialog( hDlg, NULL );
			break ;
		}
		return true;
	case WM_CLOSE:
		EndDialog( hDlg, NULL );
		return true;
	}
#endif
	return false;
}

BOOL CALLBACK MemoryProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam ) 
{
#if 0
	int *values = (int*) GetWindowLongPtr( hDlg, GWLP_USERDATA );

	switch( message )
	{
	case WM_INITDIALOG:
		{
			SetWindowLongPtr( hDlg, GWLP_USERDATA, (LONG_PTR) lParam );
			values = (int*) lParam;
			
			char buffer[256], value[16];

			sprintf(buffer, "0x%s", itoa( values[0], value, 16 ) );

			SetDlgItemText( hDlg, IDC_ADDRESS, buffer );
			SetDlgItemText( hDlg, IDC_DATA, itoa( values[1], value, 10 ) );
			CheckDlgButton( hDlg, IDC_EDIT, BST_CHECKED );
		}
		return true ;
	case WM_SHOWWINDOW:
		SetFocus( GetDlgItem( hDlg, IDC_NAME ) );
		return false ;
	case WM_COMMAND:
		switch( LOWORD(wParam) )
		{
		case IDC_EDIT:
			if( HIWORD(wParam) == BN_CLICKED )
			{
				SetDlgItemText( hDlg, IDC_TYPE, "Value" );
			}
			return true ;
		case IDC_WATCH:
			if( HIWORD(wParam) == BN_CLICKED )
			{
				SetDlgItemText( hDlg, IDC_TYPE, "Size" );
			}
			return true ;
		case IDOK:
			{
				char buffer[16];

				GetDlgItemText( hDlg, IDC_ADDRESS, buffer, sizeof(buffer) );
				if( buffer[0] == '0' )
				{
					switch( buffer[1] )
					{
					case 'b': case 'B':	// Binary value
						values[0] = strtol( &buffer[2], NULL, 2 );
						break ;
					case 'x': case 'X': case 'h': case 'H':
						values[0] = strtol( &buffer[2], NULL, 16 );
						break ;
					default:
						values[0] = strtol( buffer, NULL, 10 );
					}
				}
				else
				{
					values[0] = strtol( buffer, NULL, 10 );
				}

				GetDlgItemText( hDlg, IDC_DATA, buffer, sizeof(buffer) );
				if( buffer[0] == '0' )
				{
					switch( buffer[1] )
					{
					case 'b': case 'B':	// Binary value
						values[1] = strtol( &buffer[2], NULL, 2 );
						break ;
					case 'x': case 'X': case 'h': case 'H':
						values[1] = strtol( &buffer[2], NULL, 16 );
						break ;
					default:
						values[1] = strtol( buffer, NULL, 10 );
					}
				}
				else
				{
					values[1] = strtol( buffer, NULL, 10 );
				}

				EndDialog( hDlg, IsDlgButtonChecked( hDlg, IDC_WATCH ) ? MEMPROC_WATCH : MEMPROC_EDIT );
			}
			return true ;
		case IDCANCEL:
			EndDialog( hDlg, -1 );
			return true ;
		}
		break ;
	case WM_CLOSE:
		EndDialog( hDlg, -1 );
		return true;
	}
#endif
	return false;
}

LRESULT CALLBACK DefWindowProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	return 0 ;
}

LRESULT CALLBACK MemoryViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
#if 0
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetParent( hWnd ), GWLP_USERDATA );

	switch( message )
	{
	case WM_REFRESH:
	case WM_PAINT:
		if ( wndData != NULL )
		{
			PAINTSTRUCT ps;
			HDC hDC, fontDC;
			RECT rect;
			char buffer[16];

			if( message == WM_PAINT )
			{
				hDC = BeginPaint( hWnd, &ps );
			}
			else
			{
				hDC = GetDC( hWnd );
			}

			fontDC = CreateCompatibleDC( hDC );
			SelectObject( fontDC, wndData->m_Font );

			GetWindowRect( hWnd, &rect );

			int width = rect.right -= rect.left;
			int height = rect.bottom -= rect.top;			
			int memaddr = wndData->m_MemoryAddress;
			long bytes = (rect.right - 7 * 8) / 26 - 1;
			char bitmap[256][256];
			long byte;

			if( bytes <= 0 )
			{
				rect.top = rect.left = 0;
				FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
			}
			else for( rect.top = 0; rect.top < height; rect.top += 16 )
			{
				rect.bottom = rect.top + 16;
				rect.left = 4;
								
				DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 4, rect.top, paditoa( memaddr, buffer, 16, 6 ) );

				rect.right = 7 * 8 + 4;
				FillRect(hDC, &rect, wndData->m_Colors.m_brClear );

				rect.left = 7 * 8 + 3 + bytes*2;
				rect.right = 7 * 8 + 16 + bytes*2;
				FillRect(hDC, &rect, wndData->m_Colors.m_brClear );

				rect.left = 7 * 8 + 8 + bytes*2;

				for( byte = 0; byte < bytes; byte++ )
				{
					unsigned char data = wndData->m_Emulator->SafeRead( memaddr+byte, false );
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brLiteral, rect.left += 8, rect.top, paditoa( data, buffer, 16, 2 ) );

					rect.right = rect.left + 8;
					FillRect(hDC, &rect, wndData->m_Colors.m_brClear );

					for( int bit = 0; bit < 8; bit++ )
					{
						bitmap[bit][byte] = (data >> bit) & 1;
					}
				}

				rect.right = width;
				FillRect(hDC, &rect, wndData->m_Colors.m_brClear );

				memaddr += byte;

				StretchDIBits( hDC, 
					7 * 8 + 4, rect.top, bytes*2, 16, 
					0, 256-8, bytes, 8, 
					bitmap, (BITMAPINFO*) &wndData->m_TileBitmapInfo,
					DIB_RGB_COLORS, SRCCOPY );
			}
			
			DeleteObject( fontDC );
			if( message == WM_PAINT )
			{
				EndPaint( hWnd, &ps );
			}
			else
			{
				DeleteDC( hDC );
			}
		}
		return false;
	case WM_LBUTTONDBLCLK:
		{
			RECT rect;

			GetWindowRect( hWnd, &rect );

			long bytes = ((rect.right - rect.left) - 7 * 8) / 26 - 1;
			int memaddr = wndData->m_MemoryAddress;
			memaddr += (HIWORD(lParam) / 16) * bytes;
			memaddr += (LOWORD(lParam) - 8 * 8 - bytes*2) / 24;

			int values[2] = {memaddr, wndData->m_Emulator->SafeRead( memaddr, false ) };

			int callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE( IDD_MEMORY ), hWnd, MemoryProc, (LPARAM)&values );

			switch( callback )
			{
			case MEMPROC_WATCH:
				wndData->m_Executor->NewWatchpoint( values[0], values[1] );
				break ;
			case MEMPROC_EDIT:
				wndData->m_Emulator->DebugWrite( values[0], (unsigned char) values[1], false );
				break ;
			default:
				return true;
			}

			SendMessage( GetParent(hWnd), WM_REFRESH, 0, 0 );
		}
		return false;
	case WM_SIZE:
		{
			int width, height;
			
			width = LOWORD(lParam);
			height = HIWORD(lParam);

			int max = 0x200000 - ((width - 7 * 8) / 26 - 1) * (height / 16);

			SetScrollRange( hWnd, SB_VERT, 0, max, false );

			if( wndData != NULL && GetScrollPos( hWnd, SB_VERT ) > max )
			{
				SetScrollPos( hWnd, SB_VERT, max, false );
				wndData->m_MemoryAddress = max;
			}
		}
		SendMessage( hWnd, WM_REFRESH, 0, 0 );
		return false;
	case WM_VSCROLL:
		{
			SCROLLINFO si;
			RECT rect;

			GetWindowRect( hWnd, &rect );

			rect.right -= rect.left;
			rect.bottom -= rect.top;

			si.fMask = SIF_TRACKPOS;
			GetScrollInfo( hWnd, SB_VERT, &si );

			switch ( LOWORD(wParam) ) {
			case SB_LINEDOWN:
				wndData->m_MemoryAddress += (rect.right - 7 * 8) / 26 - 1;
				break ;
			case SB_LINEUP:
				wndData->m_MemoryAddress -= (rect.right - 7 * 8) / 26 - 1;
				break ;
			case SB_PAGEDOWN:
				wndData->m_MemoryAddress += ((rect.right - 7 * 8) / 26 - 1) * (rect.bottom / 16);
				break ;
			case SB_PAGEUP:
				wndData->m_MemoryAddress -= ((rect.right - 7 * 8) / 26 - 1) * (rect.bottom / 16);
				break ;
			case SB_THUMBTRACK:
				wndData->m_MemoryAddress = si.nTrackPos;
				break ;
			case SB_TOP:
				wndData->m_MemoryAddress = 0;
				break ;
			}

			if( wndData->m_MemoryAddress < 0 )
				wndData->m_MemoryAddress = 0;
			else if( wndData->m_MemoryAddress > 0x200000 - ((rect.right - 7 * 8) / 26 - 1) * (rect.bottom / 16) )
				wndData->m_MemoryAddress = 0x200000 - ((rect.right - 7 * 8) / 26 - 1) * (rect.bottom / 16);

			SetScrollPos( hWnd, SB_VERT, wndData->m_MemoryAddress, true );
			InvalidateRect( hWnd, NULL, false );
			return false;
		}
	}
#endif
	return DefWindowProc(hWnd, message, wParam, lParam);
}

LRESULT CALLBACK CodePointViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
#if 0
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetParent( hWnd ), GWLP_USERDATA );

	switch( message )
	{
	case WM_SIZE:
	case WM_REFRESH:
		if( wndData != NULL )
		{
			RECT rect;
			GetWindowRect( hWnd, &rect );

			rect.right -= rect.left;
			rect.bottom -= rect.top;

			int max = wndData->m_Executor->WatchCount() - (rect.bottom/16);

			if( max < 0 )
			{
				max = 0;
			}

			if( max < wndData->m_WatchIndex )
			{
				wndData->m_WatchIndex = max;
				SetScrollPos( hWnd, SB_VERT, wndData->m_WatchIndex, true );
			}

			SetScrollRange( hWnd, SB_VERT, 0, max, false );
		}
	case WM_PAINT:
		if( wndData != NULL )
		{
			PAINTSTRUCT ps;
			HDC hDC, fontDC;
			RECT rect;
			char buffer[16];

			if( message == WM_PAINT )
			{
				hDC = BeginPaint( hWnd, &ps );
			}
			else
			{
				hDC = GetDC( hWnd );
			}

			fontDC = CreateCompatibleDC( hDC );
			SelectObject( fontDC, wndData->m_Font );

			GetWindowRect( hWnd, &rect );

			int width = rect.right -= rect.left;
			int height = rect.bottom -= rect.top;
			rect.top = rect.left = 0;

			CodePoint *cp = wndData->m_Executor->ListWatchpoint( true );

			for( int i = 0; i < wndData->m_WatchIndex; i++ )
			{
				cp = wndData->m_Executor->ListWatchpoint();
			}

			for( rect.top = 0; cp != NULL && rect.top < height; rect.top+=16 )
			{
				rect.left = 4;

				char *symbol = wndData->m_Executor->GetSymbol( cp->m_Address );

				rect.bottom = rect.top + 16;

				if( symbol )
				{
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brLable, rect.left, rect.top, symbol );
				}
				else
				{
					DrawLine( hDC, fontDC, wndData->m_Colors.m_brLiteral, rect.left, rect.top, paditoa( cp->m_Address, buffer, 16, 6 ) );
				}

				rect.right = 16*8+4;

				FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

				rect.left = 16*8+4;

				for( int v = cp->m_Size-1; v >= 0; v-- )
				{
					HBRUSH brColor;
					int newData;
					
					newData = wndData->m_Emulator->SafeRead( cp->m_Address + v, false );
					
					if( cp->m_Data[v] != newData )
					{
						brColor = wndData->m_Colors.m_brWrite;
					}
					else
					{
						brColor = wndData->m_Colors.m_brText;
					}

					cp->m_Data[v] = newData;
					
					DrawLine( hDC, fontDC, brColor, rect.left, rect.top, paditoa( newData, buffer, 16, 2 ) );
				}

				rect.right = width;
				FillRect( hDC, &rect, wndData->m_Colors.m_brClear );

				cp = wndData->m_Executor->ListWatchpoint();
			}

			rect.left = 0;
			rect.bottom = height;

			FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
			
			DeleteObject( fontDC );
			if( message == WM_PAINT )
			{
				EndPaint( hWnd, &ps );
			}
			else
			{
				DeleteDC( hDC );
			}
		}
		return false;
	case WM_LBUTTONDBLCLK:
		if( LOWORD(lParam) < 16*8+4 )
		{
			return false;
		}
	case WM_RBUTTONDBLCLK:
		{
			int index = HIWORD(lParam) / 16 + wndData->m_WatchIndex;

			CodePoint *cp = wndData->m_Executor->ListWatchpoint( true );

			if ( cp == NULL )
			{
				return false;
			}

			for( int i = 0; i < index; i++ )
			{
				cp = wndData->m_Executor->ListWatchpoint();

				if ( cp == NULL )
				{
					return false;
				}
			}

			switch( message )
			{
			case WM_LBUTTONDBLCLK:
				{
					int callback = DialogBoxParam( g_hInstance, MAKEINTRESOURCE( IDD_REQUEST_INT ), hWnd, ValueProc, IDS_VALUE );

					if( callback >= 0 )
					{
						for( int i = 0; i < cp->m_Size; i++ )
						{
							unsigned char value = (callback>>(i*8)) & 0xFF;
							wndData->m_Emulator->DebugWrite( cp->m_Address + i, value, false );
						}
					}
				}
				break ;
			case WM_RBUTTONDBLCLK:
				wndData->m_Executor->DeleteWatch( cp->m_Address );
				break ;
			}

			SendMessage( GetParent(hWnd), WM_REFRESH, 0, 0 );
		}
		return false;
	case WM_VSCROLL:
		if ( wndData != NULL ) {
			RECT rect;

			GetWindowRect( hWnd, &rect );

			rect.right -= rect.left;
			rect.bottom -= rect.top;

			switch( LOWORD(wParam) )
			{
			case SB_LINEDOWN:
				wndData->m_WatchIndex++;
				break ;
			case SB_LINEUP:
				wndData->m_WatchIndex--;
				break ;
			case SB_PAGEDOWN:
				wndData->m_WatchIndex += rect.bottom / 16;
				break ;
			case SB_PAGEUP:
				wndData->m_WatchIndex -= rect.bottom / 16;
				break ;
			case SB_THUMBTRACK:
				wndData->m_WatchIndex = HIWORD(wParam);
				break ;
			}

			if( wndData->m_Executor->WatchCount() - (rect.bottom/16) < wndData->m_WatchIndex )
			{
				wndData->m_WatchIndex = wndData->m_Executor->WatchCount() - (rect.bottom/16);
			}

			if( wndData->m_WatchIndex < 0)
			{
				wndData->m_WatchIndex = 0;
			}
			
			SetScrollPos( hWnd, SB_VERT, wndData->m_WatchIndex, true );
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
		}
	}
#endif
	return DefWindowProc(hWnd, message, wParam, lParam);
}

LRESULT CALLBACK AssemblyViewerProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
#if 0
	DisasmData *wndData = (DisasmData*) GetWindowLongPtr( hWnd, GWLP_USERDATA );

	switch( message )
	{
	case WM_POINTER:
		{
			EmuWindowData *emuData = (EmuWindowData*) GetWindowLongPtr( GetParent(hWnd), GWLP_USERDATA );

			wndData = new DisasmData;
			SetWindowLongPtr( hWnd, GWLP_USERDATA, (LONG_PTR) wndData );

			wndData->m_Font = emuData->m_Font;
			wndData->m_Colors = &emuData->m_Colors;
			wndData->m_Emulator = emuData->m_Emulator;
			wndData->m_Executor = emuData->m_Executor;
			wndData->m_UserAddress  = 0x9A;
			wndData->m_DasmViewMode = DASM_PC;

			for( int n = 0; n < 0x300; n++ )
			{
				Mnemonic[n].m_Length = (n >= 0x100 ? 2 : 1);

				for( int x = 0; x < 2; x++ )
				{
					switch( Mnemonic[n].m_Op[x] )
					{
					case IMM_8:
					case ADDR_N_I8:
					case ADDR_PC_I8:
					case ADDR_SP_I8:
					case ADDR_X_I8:
					case ADDR_Y_I8:
						Mnemonic[n].m_Length ++;
						break ;
					case IMM_16:
					case ADDR_I16:
					case ADDR_I_I16:
					case ADDR_PC_I16:
						Mnemonic[n].m_Length += 2;
						break ;
					};
				}
			}
		}
		break ;
	case WM_VALUE:
		wndData->m_UserAddress = (unsigned short) lParam;
		return false;
	case WM_DESTROY:
		delete wndData;
		return false;
	case WM_COMMAND:
		{
			switch( wParam )
			{
			case ID_DISASSEMBLE_USER:
				wndData->m_DasmViewMode = DASM_USER;
				break ;
			case ID_DISASSEMBLE_PC:
				wndData->m_DasmViewMode = DASM_PC;
				break ;
			case ID_DISASSEMBLE_BOTH:
				wndData->m_DasmViewMode = DASM_SPLIT;
				break ;
			default:
				return false;
			}
		}
	case WM_RBUTTONDBLCLK:
	case WM_LBUTTONDBLCLK:
		{
			DasmValue *valBlock;
			int x, y;

			x = LOWORD(lParam);
			y = HIWORD(lParam);

			valBlock = wndData->m_ValueBlock;

			while( valBlock->m_Mode != VALUE_TERM )
			{
				if( x > valBlock->rect.right || x < valBlock->rect.left ||
					y > valBlock->rect.bottom || y < valBlock->rect.top )
				{
					valBlock++;
					continue ;
				}

				switch ( valBlock->m_Mode )
				{
				case VALUE_OP:
					{
						char *op;

						op = (char*) DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_OPCODE), hWnd, OpcodeProc );

						if( op != NULL )
						{
							Assemble( op, valBlock->m_Value, wndData->m_Emulator );
						}
					}
					break ;
				case VALUE_MEMORY:
					{
						long values[] = { valBlock->m_Value, wndData->m_Emulator->SafeRead( valBlock->m_Value, false ) };

						int mode = DialogBoxParam( g_hInstance, 
							MAKEINTRESOURCE( IDD_MEMORY ), 
							hWnd, 
							MemoryProc, (LPARAM) &values );
						
						switch( mode )
						{
						case MEMPROC_WATCH:
							wndData->m_Executor->NewWatchpoint( values[0], values[1] );
							break ;
						case MEMPROC_EDIT:
							wndData->m_Emulator->DebugWrite( values[0], (unsigned char) values[1], false );
							break ;
						}
					}
					break ;
				case VALUE_LINE_ADDRESS:
					if( wndData->m_Executor->GetBreakpoint( valBlock->m_Value, false ) )
					{
						wndData->m_Executor->DeleteBreak( valBlock->m_Value );
					}
					else
					{
						wndData->m_Executor->NewBreakpoint( valBlock->m_Value, false );
					}
					break ;
				case VALUE_ADDRESS:
					switch( message )
					{
					case WM_RBUTTONDBLCLK:
						wndData->m_Executor->NewBreakpoint( valBlock->m_Value, false );
						break ;
					case WM_LBUTTONDBLCLK:
						wndData->m_UserAddress = (valBlock->m_Value & 0x7FFF) | ((valBlock->m_Value > 0x8000) ? 0x8000 : 0);
						break ;
					}
					break ;
				}

				SendMessage( GetParent(hWnd), WM_REFRESH, 0, 0 );
				return false;
			}
			valBlock++;
		}
		return false;
	case WM_REFRESH:
	case WM_PAINT:
		if( wndData != NULL )
		{
			PAINTSTRUCT ps;
			HDC hDC, fontDC;
			RECT rect;

			long address, seekAddress;
			int line, op, followWindowSize;

			Mini *mini = wndData->m_Emulator;
			RegSet *rs = &(wndData->m_Emulator->m_CpuReg);

			if( message == WM_PAINT )
			{
				hDC = BeginPaint( hWnd, &ps );
			}
			else
			{
				hDC = GetDC( hWnd );
			}
			
			fontDC = CreateCompatibleDC( hDC );
			SelectObject( fontDC, wndData->m_Font );

			GetWindowRect( hWnd, &rect );

			rect.right -= rect.left;
			rect.bottom -= rect.top;
			rect.left = rect.top = 0;

			if( wndData->m_DasmViewMode == DASM_SPLIT && (rect.bottom) >= SPLIT_HEIGHT )
			{
				followWindowSize = 4;
			}
			else
			{
				followWindowSize = (rect.bottom/32);

				if( followWindowSize <= 0 )
				{
					followWindowSize = 1;
				}
			}


			seekAddress = address = wndData->m_FollowAddress;			

			for( line = 0; line < followWindowSize; line++ )
			{
				if( seekAddress == wndData->m_Emulator->m_LastBranch )
				{
					 wndData->m_LastBranched = wndData->m_Emulator->m_LastBranch;
				}
				if( seekAddress == rs->word.PC )
				{
					break ;
				}

				op = mini->SafeRead( seekAddress, true );

				if( op == 0xCE )
				{
					op = mini->SafeRead( seekAddress+1, true ) | 0x100;
				}
				else if( op == 0xCF )
				{
					op = mini->SafeRead( seekAddress+1, true ) | 0x200;
				}

				seekAddress += Mnemonic[op].m_Length;
			}
				
			if( wndData->m_LastBranched != wndData->m_Emulator->m_LastBranch )
			{
				wndData->m_LastBranched = wndData->m_Emulator->m_LastBranch;
				wndData->m_FollowAddress = wndData->m_LastBranched;
				seekAddress = address = wndData->m_FollowAddress;
				line = 0;
			}

			for( ; seekAddress != rs->word.PC && line < 1000; line++ ) {	
				op = mini->SafeRead( seekAddress, true );

				if( op == 0xCE )
				{
					op = mini->SafeRead( seekAddress+1, true ) | 0x100;
				}
				else if( op == 0xCF )
				{
					op = mini->SafeRead( seekAddress+1, true ) | 0x200;
				}

				seekAddress += Mnemonic[op].m_Length;
			}
			
			if( line < 0 || line == 1000 )
			{
				seekAddress = address = wndData->m_FollowAddress = rs->word.PC;
				line = 0;
			}

			for( ; line >= followWindowSize; line-- )
			{
				op = mini->SafeRead( address, true );

				if( op == 0xCE )
				{
					op = mini->SafeRead( address+1, true ) | 0x100;
				}
				else if( op == 0xCF )
				{
					op = mini->SafeRead( address+1, true ) | 0x200;
				}

				address = (wndData->m_FollowAddress += Mnemonic[op].m_Length);
			}

			
			int height = rect.bottom - SPLIT_HEIGHT;
			int width = rect.right;

			RECT rule;

			if( wndData->m_DasmViewMode != wndData->m_LastView )
			{
				rule.left = 0;
				rule.top = height - 4;
				rule.right = width;
				rule.bottom = height - 1;
				FillRect( hDC, &rule, wndData->m_Colors->m_brClear );
			}
			wndData->m_LastView = wndData->m_DasmViewMode;

			switch( wndData->m_DasmViewMode )
			{
			case DASM_USER:
				DrawAssembly( wndData, wndData->m_ValueBlock, hDC, fontDC, &rect, wndData->m_UserAddress );
				break ;
			case DASM_PC:
				DrawAssembly( wndData, wndData->m_ValueBlock, hDC, fontDC, &rect, address );
				break ;
			case DASM_SPLIT:
				{
					if( height > 5 )
					{
						DasmValue *valBlock;
						HRGN region;

						rect.top = 0;
						rect.bottom = height - 5;

						Rectangle( hDC, 0, height-4, width, height - 1 );

						region = CreateRectRgn( 0, 0, width, rect.bottom );
						SelectObject( hDC, region );
						valBlock = DrawAssembly( wndData, wndData->m_ValueBlock, hDC, fontDC, &rect, wndData->m_UserAddress );
						DeleteObject( region );

						rect.top = height;
						rect.bottom = height + SPLIT_HEIGHT;

						region = CreateRectRgn( 0, 0, width, rect.bottom );
						SelectObject( hDC, region );
						DrawAssembly( wndData, valBlock, hDC, fontDC, &rect, address );
						DeleteObject( region );

						SetWindowRgn( hWnd, NULL, false );						
					}
					else
					{
						DrawAssembly( wndData, wndData->m_ValueBlock, hDC, fontDC, &rect, address );
					}
				}
				break ;
			default:
				TextOut( hDC, 0, 0, "Incomplete view mode", 20 );
			};

			DeleteDC( fontDC );

			if( message == WM_PAINT )
			{
				EndPaint( hWnd, &ps );
			}
			else
			{
				DeleteDC( hDC );
			}
		}
		return false;
	case WM_SIZE:
		if( wndData != NULL && wndData->m_DasmViewMode == DASM_SPLIT )
		{
			InvalidateRect( hWnd, NULL, true );
		}
		SetScrollRange( hWnd, SB_VERT, 0, 0xFFFF, false );
		return false ;
	case WM_VSCROLL:
		if ( wndData->m_DasmViewMode != DASM_PC ) {
			RECT rect;
			int i;

			GetWindowRect( hWnd, &rect );

			rect.right -= rect.left;
			rect.bottom -= rect.top;

			switch( LOWORD(wParam) )
			{
			case SB_LINEDOWN:
				{
					int op = wndData->m_Emulator->SafeRead( wndData->m_UserAddress, true );

					if( op == 0xCE )
					{
						op = 0x100 | wndData->m_Emulator->SafeRead( wndData->m_UserAddress + 1, true );
					}
					else if ( op == 0xCF )
					{
						op = 0x200 | wndData->m_Emulator->SafeRead( wndData->m_UserAddress + 1, true );
					}

					wndData->m_UserAddress += Mnemonic[op].m_Length;
				}
				break ;
			case SB_LINEUP:
				wndData->m_UserAddress--;
				break ;
			case SB_PAGEDOWN:
				for( i = 0; i < (rect.bottom/16); i++ ) {
					int op = wndData->m_Emulator->SafeRead( wndData->m_UserAddress, true );

					if( op == 0xCE )
					{
						op = 0x100 | wndData->m_Emulator->SafeRead( wndData->m_UserAddress + 1, true );
					}
					else if ( op == 0xCF )
					{
						op = 0x200 | wndData->m_Emulator->SafeRead( wndData->m_UserAddress + 1, true );
					}

					wndData->m_UserAddress += Mnemonic[op].m_Length;
				}
				break ;
			case SB_PAGEUP:
				wndData->m_UserAddress -= (unsigned short) (rect.bottom/16);
				break ;
			case SB_THUMBTRACK:
				wndData->m_UserAddress = HIWORD(wParam);
				break ;
			}
			SetScrollPos( hWnd, SB_VERT, wndData->m_UserAddress, false );
			SendMessage( hWnd, WM_REFRESH, 0, 0 );
		}
		return false;
	}
#endif
	return DefWindowProc(hWnd, message, wParam, lParam);
}

#define ERROR_INDEXES 256
#define MAX_STRING    256

struct ErrorWindow
{
	char *m_Data[ERROR_INDEXES];
	bool m_Dirty;
	int  m_Index;
};

LRESULT CALLBACK ErrorProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
#if 0
	EmuWindowData *wndData = (EmuWindowData*) GetWindowLongPtr( GetWindow( hWnd, GW_OWNER ), GWLP_USERDATA );
	ErrorWindow *errData = (ErrorWindow*) GetWindowLongPtr( hWnd, GWLP_USERDATA );

	switch ( message )
	{
	case WM_CREATE:
		errData = new ErrorWindow;
		SetWindowLongPtr( hWnd, GWLP_USERDATA, (LONG_PTR) errData );
		
		ZeroMemory( errData, sizeof(ErrorWindow) );
		return false;
	case WM_CLOSE:
		ShowWindow( hWnd, SW_HIDE );
		return false;
	case WM_DESTROY:
		{
			for( int i = 0; i < ERROR_INDEXES; i++ )
			{
				if( errData->m_Data[i] != NULL )
					delete errData->m_Data[i];
			}

			delete errData;

			return false;
		}
	case WM_SETTEXT:
		{
			errData->m_Index = (errData->m_Index + 255) % ERROR_INDEXES;

			char *string = errData->m_Data[errData->m_Index];

			if( string == NULL )
			{
				string = errData->m_Data[errData->m_Index] = new char[MAX_STRING];
			}

			strcpy( string, (char*)lParam );

			errData->m_Dirty = true;
			return false;
		}
	case WM_REFRESH:
		if( errData->m_Dirty == false )
			break ;

		SendMessage( hWnd, WM_SETTEXT, 0, (LPARAM) "-------------" );

		errData->m_Dirty = false;
	case WM_PAINT:
		{
			PAINTSTRUCT ps;
			HDC hDC, fontDC;
			RECT rect;
			if( message == WM_PAINT )
			{
				hDC = BeginPaint( hWnd, &ps );
			}
			else
			{
				hDC = GetDC( hWnd );
			}
			
			fontDC = CreateCompatibleDC( hDC );
			SelectObject( fontDC, wndData->m_Font );

			GetWindowRect( hWnd, &rect );
			rect.right -= rect.left;
			int height = rect.bottom -= rect.top;

			for( rect.top = 0; rect.top < height; rect.top+=16 )
			{
				if( rect.top / 16 > ERROR_INDEXES )
					break ;

				char *err = errData->m_Data[(rect.top / 16 + errData->m_Index) % ERROR_INDEXES];

				if( err == NULL )
					break ;

				DrawLine( hDC, fontDC, wndData->m_Colors.m_brText, rect.left = 4, rect.top, err );

				rect.bottom = rect.top + 16;	
				FillRect( hDC, &rect, wndData->m_Colors.m_brClear );
			}

			rect.left = 0;
			rect.bottom = height;
			FillRect( hDC, &rect, wndData->m_Colors.m_brClear );


			DeleteDC( fontDC );

			if( message == WM_PAINT )
			{
				EndPaint( hWnd, &ps );
			}
			else
			{
				DeleteDC( hDC );
			}

			return false;
		}
	}
#endif
	return DefWindowProc( hWnd, message, wParam, lParam );
}