/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Minimon.h"

MinxCore::MinxCore()
{
	m_Rom = NULL;
	m_RomSize = 0;

	m_Clock = 0;
	m_Halted = false;

	memset( m_Bios, 0xCE, sizeof(m_Bios) );
	memset( &m_CpuReg, 0, sizeof( RegSet ) );
	memset( m_Ram, 0, sizeof( m_Ram ) );
}

MinxCore::~MinxCore()
{
	if( m_Rom != NULL )
	{
		delete m_Rom;
	}
}

void MinxCore::Error( const char *error, int errc, int *errv )
{
	printf( "Execution error " );

	for( int i = 0; i < errc; i++ )
	{
		printf( "%li", errv[i] );
	}

	printf(": %s", error );
}

#include "pokebios.h"

void MinxCore::LoadBios( const char *bios )
{
	memcpy( m_Bios, pokeminibios, 0x1000 ) ;
#if 0
	FILE *rom = fopen( bios, "rb" );

	if( rom != NULL )
	{
		fread( m_Bios, 0x1000, 1, rom );

		fclose(rom);
	}
	else
	{		
		this->Error( "Could not locate bios", 0, NULL );
	}	
#endif
}

void MinxCore::LoadRom( const char *filename )
{
	FILE *rom = fopen( filename, "rb" );
	long fileSize;	

	if( rom == NULL )
	{
		this->Error( "Could not locate rom", 0, NULL );
		return ;
	}

	if( m_Rom != NULL )
	{
		delete m_Rom; 
	}

	fseek( rom, 0, SEEK_END );
	fileSize = ftell( rom );
	fseek( rom, 0, SEEK_SET );

	for( m_RomSize = 1; m_RomSize < fileSize; m_RomSize <<= 1 ) ;
	
	if( m_RomSize > 1 )
	{
		m_RomMask = m_RomSize - 1;
	}
	else
	{
		m_RomMask = 1;
	}

	m_Rom = new unsigned char[ m_RomSize ];

	if( m_Rom != NULL )
	{
		memset( &m_Rom[fileSize], 0xCD, m_RomSize - fileSize );
		fread( m_Rom, fileSize, sizeof( unsigned char ), rom );
	}
	else
	{
		this->Error( "Failed to allocate memory for rom", 0, NULL );
	}

	fclose(rom);
}

void MinxCore::FreeRom()
{
	if( m_Rom != NULL )
	{
		delete m_Rom;
	}

	m_Rom = NULL;
}

void MinxCore::Peek( long addr, unsigned char &value )
{
	LogRead( addr );

	if( addr < 0x1000 )
	{
		value = m_Bios[addr];
	}
	else if( addr < 0x2000 )
	{
		value = m_Ram[ addr & 0xFFF ];
	}
	else if( addr < 0x2100 )
	{
		PeekReg( (unsigned char)(addr & 0xFF), value );
	}
	else if ( m_Rom != NULL )
	{
		value = m_Rom[ addr & m_RomMask ];
	}
	else
	{
		value = 0;
	}
}

void MinxCore::Peek( long addr, unsigned short &value )
{
	unsigned char va, vb;

	Peek( addr, va );
	Peek( (addr + 1 & 0xFFFF) | (addr & 0xFF0000), vb );

	value = vb << 8 | va;
}

void MinxCore::Poke( long addr, unsigned char value )
{
	LogWrite( addr );
	
	if( addr >= 0x1000 && addr < 0x2000 )
	{
		m_Ram[ addr & 0xFFF ] = value;
	}
	else if( addr < 0x2100 )
	{
		PokeReg( (unsigned char)(addr & 0xFF), value );
	}
}

void MinxCore::Poke( long addr, unsigned short value )
{
	Poke( addr, (unsigned char) (value & 0xFF) );
	Poke( ((addr + 1) & 0xFFFF) | (addr & 0xFF0000), (unsigned char) (value >> 8) );
}

void MinxCore::Push( unsigned char value )
{
	m_CpuReg.word.SP--;
	Poke( m_CpuReg.word.SP, value );
}

void MinxCore::Push( unsigned short value )
{
	m_CpuReg.word.SP -= 2;
	Poke( m_CpuReg.word.SP, value );
}

void MinxCore::Pop( unsigned char &value )
{
	Peek( m_CpuReg.word.SP, value );
	m_CpuReg.word.SP++;
}

void MinxCore::Pop( unsigned short &value )
{
	Peek( m_CpuReg.word.SP, value );
	m_CpuReg.word.SP += 2;
}

void MinxCore::Fetch( unsigned char &value )
{
	long addr;

	if( m_CpuReg.word.PC & 0x8000 )
	{
		addr = (m_CpuReg.word.PC++ & 0x7FFF) | (m_CpuReg.word.V << 15);
	}
	else
	{
		addr = m_CpuReg.word.PC++;
	}

	if( addr < 0x1000 )
	{
		value = m_Bios[addr];
	}
	else if( addr < 0x2000 )
	{
		value = m_Ram[ addr & 0xFFF ];
	}
	else if( addr < 0x2100 )
	{
		PeekReg( (unsigned char)(addr & 0xFF), value );
	}
	else if ( m_Rom != NULL )
	{
		value = m_Rom[ addr & m_RomMask ];
	}
	else
	{
		value = 0;
	}
}

void MinxCore::Fetch( unsigned short &value )
{
	unsigned char byte1, byte2;
	Fetch( byte1 );
	Fetch( byte2 );

	value = (byte2 << 8) | byte1;
}

void MinxCore::ForceIRQ( unsigned char IRQ )
{
	Push( m_CpuReg.byte.V );
	Push( m_CpuReg.word.PC );
	Push( m_CpuReg.byte.F );

	m_CpuReg.byte.F |= 0xC0;	// IRQ masking

	Peek( IRQ & 0xFE, m_CpuReg.word.PC );
	LogBranch( m_CpuReg.word.PC );

	m_Halted = false;
}

void MinxCore::Add( unsigned char &left, unsigned char right, bool carry )
{
	int result;
	
	if( m_CpuReg.byte.F & FLAG_MASKLOW )
	{
		right &= 0xF;
		left &= 0xF;
	}
	
	// NEED TO TEST TO SEE IF BCD VALIDIDITY CHECKS EXIST!!!

	if( (m_CpuReg.byte.F & FLAG_BCD) &&
		    ((left & 0xF) < 0xA) && ((left & 0xF0) < 0xA0) &&
			((right & 0xF) < 0xA) && ((right & 0xF0) < 0xA0) )
	{
 		left = (left & 0xF) + ((left >> 4) * 10);
		right = (right & 0xF) + ((right >> 4) * 10);

		result = left + right + (carry?1:0);

		Flag( FLAG_OVERFLOW, 0 );
		Flag( FLAG_SIGN, 0 );
		Flag( FLAG_CARRY, result >= 100 );

		result = result % 100;
		left  = (result / 10) * 0x10;
		left += (result % 10);

		Flag( FLAG_ZERO, result == 0 );
	}
	else
	{
		int result = left + right + (carry ? 1 : 0);

		Flag( FLAG_ZERO, (result & 0xFF) == 0 );
		Flag( FLAG_OVERFLOW, (((left ^ right) & 0x80) == 0) && (((left ^ result) & 0x80) != 0) );
		Flag( FLAG_SIGN, (result & 0x80) != 0 );
		Flag( FLAG_CARRY, (result & ~0xFF) != 0 );

		left = result & 0xFF;
	}
}

void MinxCore::Add( unsigned short &left, unsigned short right, bool carry )
{
	int result = left + right + (carry ? 1 : 0);

	Flag( FLAG_OVERFLOW, (((left ^ right) & 0x8000) == 0) && (((left ^ result) & 0x8000) != 0) );
	Flag( FLAG_SIGN, (result & 0x8000) != 0 );
	Flag( FLAG_ZERO, (result & 0xFFFF) == 0 );
	Flag( FLAG_CARRY, (result & ~0xFFFF) != 0 );

	left = result & 0xFFFF;
}

void MinxCore::Sub( unsigned char &left, unsigned char right, bool carry )
{
	int result;
	
	if( m_CpuReg.byte.F & FLAG_MASKLOW )
	{
		right &= 0xF;
		left &= 0xF;
	}
	
	if( (m_CpuReg.byte.F & FLAG_BCD) &&
		    ((left & 0xF) < 0xA) && ((left & 0xF0) < 0xA0) &&
			((right & 0xF) < 0xA) && ((right & 0xF0) < 0xA0) )
	{
		left = (left & 0xF) + ((left >> 4) * 10);
		right = (right & 0xF) + ((right >> 4) * 10);

		result = left - right - (carry?1:0);

		Flag( FLAG_OVERFLOW, 0 );
		Flag( FLAG_SIGN, 0 );
		Flag( FLAG_CARRY, result < 0 );
		
		result = (result + 200) % 100;	// Fix for positive
		left  = (result / 10) * 0x10;
		left += (result % 10);
		
		Flag( FLAG_ZERO, left == 0 );
	}
	else
	{
		result = left - right - (carry ? 1 : 0);

		Flag( FLAG_ZERO,  (result & 0xFF) == 0 );
		Flag( FLAG_CARRY, (result & ~0xFF) != 0 );
		Flag( FLAG_OVERFLOW, (((left ^ right) & 0x80) != 0) && (((left ^ result) & 0x80) != 0) );
		Flag( FLAG_SIGN, (result & 0x80) != 0 );

		left = result & 0xFF;
	}
}

void MinxCore::Sub( unsigned short &left, unsigned short right, bool carry )
{
	int result = left - right - (carry ? 1 : 0);

	Flag( FLAG_OVERFLOW, (((left ^ right) & 0x8000) != 0) && (((left ^ result) & 0x8000) != 0) );
	Flag( FLAG_SIGN, (result & 0x8000) != 0 );
	Flag( FLAG_ZERO, (result & 0xFFFF) == 0 );
	Flag( FLAG_CARRY, (result & ~0xFFFF) != 0 );

	left = result & 0xFFFF;
}

void MinxCore::Flag( int flag, bool set )
{
	if( set )
	{
		m_CpuReg.byte.F |= flag;
	}
	else
	{
		m_CpuReg.byte.F &= ~flag;
	}
}

void MinxCore::Flags8( int flagMask, int value )
{
	if( flagMask & FLAG_ZERO )
	{
		Flag( FLAG_ZERO, value == 0 );
	}

	if( flagMask & FLAG_CARRY )
	{
		Flag( FLAG_CARRY, (value & 0x100) != 0 );
	}

	if( flagMask & FLAG_SIGN )
	{
		Flag( FLAG_SIGN, (value & 0x80) != 0 );
	}
}

void MinxCore::Flags16( int flagMask, int value )
{
	if( flagMask & FLAG_ZERO )
	{
		Flag( FLAG_ZERO, value == 0 );
	}

	if( flagMask & FLAG_CARRY )
	{
		Flag( FLAG_CARRY, (value & 0x10000) != 0 );
	}

	if( flagMask & FLAG_SIGN )
	{
		Flag( FLAG_SIGN, (value & 0x8000) != 0 );
	}
}
