/*
 *
 * Copyright (C) 2004-2005 Robert Bryon Vandiver (asterick@buxx.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "minimon.h"

EEPROM::EEPROM() {
	m_ClockBit   = false;
	m_DataBit    = false;
	m_Scanning   = false;
	m_DataLoaded = false;

	m_Mode     = EEPROM_IDLE;
	m_Bit      = 0;
	m_Address  = 0;
}

EEPROM::~EEPROM() {
	if ( m_DataLoaded )
	{
		FILE *fo = fopen(m_DataStore,"wb");
		fwrite( &m_Rom, sizeof(m_Rom), 1, fo );
		fclose(fo);
	}
}

unsigned char EEPROM::HardRead( unsigned short address )
{
	return m_Rom[address];
}

void EEPROM::HardWrite( unsigned short address, unsigned char byte )
{
	m_Rom[address] = byte;
}

void EEPROM::Load( const char *romName )
{
	FILE *fo;

	if ( m_DataLoaded )
	{
		fo = fopen(m_DataStore,"wb");
		fwrite( &m_Rom, sizeof(m_Rom), 1, fo );
		fclose(fo);
	}

	if( romName == NULL )
	{
		m_DataLoaded = false;
		*m_DataStore = 0;
		return ;
	}

	m_DataLoaded = true;

	strcpy( m_DataStore, romName );
	strcat( m_DataStore, ".eep" );

	fo = fopen(m_DataStore,"rb");

	if( fo != NULL )
	{
		// Quick fix for the name problem
		fseek(fo, 0, SEEK_END);
		fseek(fo, -(signed)sizeof(m_Rom), SEEK_CUR);
		long tell = ftell(fo);

		fread( &m_Rom, sizeof(m_Rom), 1, fo );
		fclose(fo);
	}
	else
	{
		memset( &m_Rom, 0xCD, sizeof(m_Rom) );
	}
}

void EEPROM::Clock( bool clock )
{
	// Latch data on rising edge of clock	
	if( clock && !m_ClockBit && m_Scanning )
	{		
		if( m_Bit >= 0 )
		{
			m_ReadData = (m_ReadData<<1) | (m_DataBit ? 1 : 0);
		}
		else
		{
			// Process data 
			switch( m_Mode )
			{
			case EEPROM_WRITE_CONTROL:
				{
					unsigned char code = m_ReadData & 0xF0;
					unsigned char bank = m_ReadData & 0x0E;
					unsigned char read = m_ReadData & 0x01;

					if( bank != 0 )
					{
						read = read;
					}

					switch( code )
					{
					case 0xA0:	// Memory access
						m_Mode  = read ? EEPROM_READ_BYTE : EEPROM_WRITE_ADDRESS_HI;
						m_Bit = 8;
						break ;
					case 0x00:
					case 0x10:
					case 0x20:
					case 0x30:
					case 0x40:
					case 0x50:
					case 0x60:
					case 0x70:
					case 0x80:
					case 0x90:
					case 0xB0:
					case 0xC0:
					case 0xD0:
					case 0xE0:
					case 0xF0:
						MessageBox( NULL, "Unknown EEPROM control command", g_AppTitle, MB_OK );
						read = read;
						break ;
					}
				}
				break ;
			case EEPROM_WRITE_ADDRESS_HI:
				m_Address	= (m_Address & 0x00FF) | ((m_ReadData<<8) & 0x1F00);
				m_Mode		= EEPROM_WRITE_ADDRESS_LO;
				m_Bit		= 8;
				break ;
			case EEPROM_WRITE_ADDRESS_LO:
				m_Address	= (m_Address & 0xFF00) | (m_ReadData);
				m_Mode		= EEPROM_WRITE_BYTE;
				m_Bit		= 8;
				break ;
			case EEPROM_WRITE_BYTE:
				m_Rom[m_Address] = m_ReadData;
				m_Address = (m_Address + 1) & 0x1FFF;

				m_Bit		= 8;
				break ;
			case EEPROM_READ_BYTE:
				m_Scanning = !m_DataBit;	// No Acknowledgement
				m_Address = (m_Address + 1 & 0x1FFF);
				m_Bit		= 8;
				break ;
			}
		}
	}
	// Decrement bit at falling edge of clock
	else if( !clock && m_ClockBit && m_Scanning )
	{
		m_Bit--;
	}

	m_ClockBit = clock;
}

void EEPROM::Write( bool din ) {
	// Start / Stop condition (change of data bit while clock is high)
	if( m_ClockBit && (din != m_DataBit))
	{
		switch( din )
		{
		case 0: // Start
			m_Scanning = true;
			m_Mode     = EEPROM_WRITE_CONTROL;
			m_Bit      = 8;
			m_ReadData = 0;
			break ;
		case 1:	// Stop 
			m_Scanning = false;
			break ;
		}
	}

	m_DataBit = din;
}

bool EEPROM::Read()
{
	if( !m_Scanning )
	{
		return true ;
	}

	switch( m_Mode )
	{
	case EEPROM_WRITE_CONTROL:
	case EEPROM_WRITE_ADDRESS_HI:
	case EEPROM_WRITE_ADDRESS_LO:
	case EEPROM_WRITE_BYTE:
		return ( m_Bit >= 0 && m_Bit <= 7 );
	case EEPROM_READ_BYTE:
		if( m_Bit >= 0 && m_Bit <= 7 )
		{
			return (m_Rom[ m_Address ] >> m_Bit & 1) != 0;
		}
		else
		{
			return false;
		}
	default:
		return true;
	}
}
