
/*
 *   O2EM Freeware Odyssey2 / Videopac+ Emulator
 *
 *   Created by Daniel Boris <dboris@comcast.net>  (c) 1997,1998
 *
 *   Developed by Andre de la Rocha <adlroc@users.sourceforge.net>
 *
 *   http://o2em.sourceforge.net
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "crc32.h"
#include "audio.h"
#include "vmachine.h"
#include "config.h"
#include "vdc.h"
#include "cpu.h"
#include "debug.h"
#include "keyboard.h"
#include "voice.h"
#include "allegro.h"
#ifdef ALLEGRO_WINDOWS
#include "winalleg.h"
#endif


#define MAXC 1024
extern int voice_count ;

unsigned char VIDEOPAC_ROM [] = {
 0x44,0x6a,0x00,0x84,0x02,0x00,0x00,0x84,
 0x04,0xc5,0xad,0x09,0xae,0x14,0xe7,0xb8,
 0xa1,0x80,0x72,0x18,0xfe,0x39,0xfd,0x93,
 0x84,0x06,0xa5,0xb5,0xb9,0xa2,0xb8,0x3d,
 0x81,0xa0,0x18,0x10,0xf0,0x53,0x3f,0xd3,
 0x3c,0x96,0x2f,0xf0,0x53,0xc0,0xa0,0x18,
 0xf0,0x37,0xf2,0x3a,0x37,0x53,0x7f,0xa0,
 0x14,0x89,0xb9,0x3f,0xf1,0x37,0xd2,0x14,
 0xeb,0x14,0x84,0x0a,0xfc,0xe3,0xa9,0x1c,
 0xfc,0xe3,0xaa,0xf9,0xf2,0x5c,0xd2,0x73,
 0xb2,0x7d,0x92,0x84,0xb9,0x3f,0xf1,0x53,
 0xbf,0xa1,0x04,0x14,0x53,0x7f,0xab,0xb8,
 0xaa,0x27,0x90,0xb8,0xa7,0xb9,0x03,0xfa,
 0xe3,0x90,0x18,0x1a,0xe9,0x67,0xfa,0xe3,
 0xaa,0x04,0x76,0x53,0x3f,0xab,0x1c,0xb8,
 0xaa,0xfa,0x90,0x04,0x14,0x53,0x1f,0xab,
 0xba,0x00,0x04,0x77,0xfa,0xac,0x84,0x0a,
 0x00,0xb8,0x7f,0xb9,0xa0,0x81,0x53,0xd6,
 0x91,0x89,0x7c,0x99,0xe7,0x80,0x96,0xa3,
 0x89,0xbc,0x99,0xb7,0xb9,0xa0,0x81,0x43,
 0x28,0x91,0x83,0xaa,0xc8,0x80,0xa9,0xc8,
 0x80,0x91,0xc8,0x19,0xea,0xa8,0x04,0x95,
 0x15,0xc5,0x09,0xae,0x89,0xbc,0x99,0xbb,
 0xb8,0xf0,0xba,0x06,0xf8,0x3a,0x0a,0x92,
 0xda,0xba,0x30,0xa8,0x0a,0xd8,0x96,0xdf,
 0xea,0xc4,0xf8,0x47,0x77,0xd3,0x07,0x53,
 0x3f,0xa9,0xff,0x53,0x3f,0xd9,0x96,0xe5,
 0x04,0xdf,0x18,0xea,0xbc,0xbf,0xff,0xff,
 0x43,0xc0,0xaf,0x64,0x78,0x64,0x76,0x89,
 0xbc,0x99,0xb7,0x83,0x89,0xbc,0x99,0xaf,
 0x83,0x89,0xbc,0x99,0xb7,0x23,0x20,0xa8,
 0xa9,0x27,0xa0,0x18,0xe9,0xfa,0x14,0xec,
 0xb9,0xff,0x91,0xe9,0x02,0x91,0x14,0xe7,
 0x34,0x1c,0xb8,0xff,0xb9,0x80,0x27,0x90,
 0xc8,0xe9,0x0f,0x23,0xf8,0x90,0xe8,0x15,
 0x90,0x34,0x27,0x83,0x15,0xc5,0xb8,0xa0,
 0x80,0x53,0xd6,0x90,0xd5,0x05,0x83,0x15,
 0xc5,0xb8,0xa0,0x80,0x43,0x28,0x90,0xd5,
 0x05,0x83,0x27,0x90,0xb8,0x3f,0xf0,0x43,
 0x80,0xa0,0x14,0xe7,0x83,0x34,0x76,0x14,
 0xb0,0xf2,0x3d,0x23,0x56,0x34,0xa2,0xc5,
 0xff,0xd5,0x83,0xfe,0x53,0xfe,0xae,0xfd,
 0x97,0xf7,0xf7,0xf7,0xad,0x27,0xf7,0x4e,
 0xae,0xfc,0x67,0x37,0x17,0x6d,0xad,0xf6,
 0x6a,0xfe,0x37,0x53,0x01,0x2e,0x53,0xfe,
 0x4e,0xae,0x83,0x23,0xf8,0xb8,0x10,0xba,
 0x30,0x90,0x18,0xea,0x71,0x83,0xa5,0x05,
 0x76,0x75,0x24,0x78,0x14,0xec,0xaa,0xb8,
 0x7f,0x23,0x08,0x90,0xc8,0xf9,0x90,0xc8,
 0xfa,0x47,0x53,0x0f,0xad,0x34,0x97,0xfa,
 0x53,0x0f,0xad,0x34,0x97,0x24,0x32,0xfc,
 0x90,0xc8,0xfb,0x90,0xc8,0x03,0x08,0xab,
 0x44,0x2c,0x15,0xc5,0xac,0xbb,0x01,0xb8,
 0x3f,0xf0,0x43,0x40,0xa0,0xd5,0x05,0x83,
 0xb8,0x3e,0xf0,0xf2,0x3a,0x53,0x3f,0xd3,
 0x3b,0x96,0x3a,0x14,0xec,0xf0,0xb8,0x02,
 0xb9,0x01,0xd2,0xe2,0xba,0x99,0x80,0xc6,
 0xce,0x6a,0x57,0x90,0x24,0xfc,0x81,0xc6,
 0xda,0x23,0x59,0x90,0x81,0x6a,0x57,0x91,
 0x24,0xfc,0xb8,0x3e,0xf0,0x43,0x80,0xa0,
 0x24,0x3a,0xba,0x01,0x80,0xd3,0x59,0xc6,
 0xef,0x80,0x6a,0x57,0x90,0x24,0xfc,0x90,
 0x81,0xd3,0x59,0x96,0xf8,0x91,0x24,0xfc,
 0x81,0x6a,0x57,0x91,0xb8,0x7f,0x54,0x35,
 0x23,0x42,0x90,0xc8,0xb9,0x01,0x81,0x47,
 0x54,0x29,0x54,0x35,0x23,0x52,0x90,0xc8,
 0x81,0x54,0x29,0x54,0x35,0x23,0x56,0x90,
 0xc8,0x19,0x81,0x47,0x54,0x29,0x54,0x35,
 0x23,0x4a,0x90,0xc8,0x81,0x54,0x29,0x24,
 0x32,0x53,0x0f,0xad,0x34,0x4b,0xfd,0x90,
 0xc8,0xfe,0x90,0xc8,0x83,0x23,0x02,0x90,
 0xc8,0x83,0xb8,0x40,0xb9,0x50,0xfc,0x90,
 0x91,0x18,0x19,0xfb,0x90,0x03,0x08,0x91,
 0xbd,0x0a,0xb8,0x46,0x54,0x61,0xbd,0x0c,
 0xb8,0x4e,0x54,0x61,0xbd,0x0c,0xb8,0x5a,
 0x54,0x61,0xbd,0x0c,0xb8,0x5e,0x54,0x61,
 0x83,0x34,0x4b,0xfd,0x90,0x18,0xfe,0x90,
 0x18,0x83,0xba,0x00,0x00,0xea,0x6c,0x99,
 0x9f,0xb9,0x02,0xbe,0x49,0xbf,0xa0,0x54,
 0x90,0x54,0x96,0x84,0x00,0xb8,0x06,0x80,
 0xf2,0x7f,0x83,0xbf,0xa0,0x43,0x40,0xae,
 0xb9,0x02,0x44,0x8e,0xb9,0x00,0x54,0x7d,
 0xfe,0x91,0x19,0xff,0x91,0x83,0xbf,0xff,
 0xff,0x3d,0x47,0x3c,0xff,0x3f,0x47,0x3e,
 0x83,0x89,0xbc,0x99,0x9f,0x83,0x27,0x64,
 0x83,0x54,0x96,0x14,0xe7,0xc5,0xff,0xd5,
 0xa9,0x85,0x64,0x8b,0x53,0x45,0x4c,0x45,
 0x43,0x54,0x20,0x47,0x41,0x4d,0x45,0x20,
 0x20,0x20,0x95,0x54,0xa1,0xbe,0x00,0xbf,
 0xc0,0x54,0x88,0xbe,0x1f,0xbf,0x00,0x54,
 0x88,0xbe,0x00,0xbf,0x80,0x54,0x88,0xbd,
 0x45,0x1e,0xfe,0x53,0x07,0xc6,0xd9,0xae,
 0xbb,0xb4,0xbc,0x0e,0xfb,0xa3,0xaf,0x54,
 0x8c,0x1b,0xec,0xe4,0xed,0xd9,0x23,0x01,
 0xbf,0x00,0x54,0x99,0x14,0xf1,0x23,0x4a,
 0x34,0xa2,0x34,0x3d,0xb6,0xa6,0x44,0xa9,
 0x00,0x0f,0xff,0xcf,0x03,0xf0,0x3f,0xcf,
 0x0f,0x0f,0x0f,0xcf,0x1c,0x71,0xc7,0xcf,
 0x33,0x33,0x33,0xcf,0x03,0xf0,0x3f,0xef,
 0x0f,0x0f,0x0f,0xef,0x1c,0x71,0xc7,0xef,
 0x33,0x33,0x33,0xef,0x55,0x55,0x55,0xef,
 0x90,0x10,0x94,0x00,0x21,0x00,0x56,0xdf,
 0x53,0xdd,0x50,0xda,0x4d,0xd8,0x4a,0xd5,
 0x47,0xd2,0x21,0x00,0x92,0x20,0x4a,0xcd,
 0x52,0xeb,0x4a,0xc9,0x52,0xe7,0x4a,0xc5,
 0x21,0x00,0x85,0x04,0x85,0x0c,0x85,0x14,
 0x85,0x1c,0x85,0x24,0x21,0x00,0x82,0x1c,
 0x21,0x00,0x8a,0x00,0x21,0x00,0x85,0x24,
 0x85,0x1c,0x85,0x14,0x85,0x0c,0x85,0x04,
 0x21,0x00,0x44,0xff,0x4d,0xdd,0x48,0xda,
 0x44,0xd5,0x42,0xd3,0x21,0x00,0xf9,0xaf,
 0xfe,0x39,0xff,0xd5,0x05,0x83,0x39,0x99,
 0xfe,0x84,0x08,0x99,0xfd,0x84,0x08,0x99,
 0xfc,0x84,0x08,0x89,0x03,0x84,0x08,0x09,
 0xa8,0x89,0xbc,0x99,0xbb,0xf9,0x43,0xf0,
 0x3a,0x85,0x27,0xaa,0xab,0x08,0xa9,0x12,
 0xa2,0xcb,0x32,0xa5,0x1a,0x52,0xa8,0x1b,
 0x72,0xab,0xca,0x92,0xae,0x95,0xf8,0x39,
 0x83,0xfb,0x53,0x0f,0xa8,0xfa,0x53,0x0f,
 0x47,0x48,0xa8,0xb9,0x07,0x23,0xc7,0x69,
 0xa3,0xd8,0xc6,0xc6,0xe9,0xbd,0x83,0x10,
 0x1f,0x0f,0xff,0xf0,0xf1,0x01,0x11,0xbc,
 0xff,0xfb,0x37,0x17,0xae,0xfa,0x6e,0x1c,
 0xf6,0xd6,0x6b,0xad,0x83,0xfa,0x96,0xe1,
 0x83,0xfb,0x96,0xe5,0x83,0x27,0x6b,0xea,
 0xe6,0x83,0xfc,0x90,0x18,0xf8,0xd2,0xf9,
 0xfb,0x90,0x03,0x08,0xab,0x18,0x54,0x61,
 0x83,0x53,0x0f,0x07,0xc6,0xf0,0x64,0xf5
};

unsigned char O2ROM[] = {
 0x84,0x00,0x00,0x84,0x02,0x00,0x00,0x84,
 0x04,0xc5,0xad,0x09,0xae,0x14,0xe7,0xb8,
 0xa1,0x80,0x72,0x18,0xfe,0x39,0xfd,0x93,
 0x84,0x06,0xa5,0xb5,0xb9,0xa2,0xb8,0x3d,
 0x81,0xa0,0x18,0x10,0xf0,0x53,0x3f,0xd3,
 0x3c,0x96,0x2f,0xf0,0x53,0xc0,0xa0,0x18,
 0xf0,0x37,0xf2,0x3a,0x37,0x53,0x7f,0xa0,
 0x14,0x89,0xb9,0x3f,0xf1,0x37,0xd2,0x14,
 0xeb,0x14,0x84,0x0a,0xfc,0xe3,0xa9,0x1c,
 0xfc,0xe3,0xaa,0xf9,0xf2,0x5c,0xd2,0x73,
 0xb2,0x7d,0x92,0x84,0xb9,0x3f,0xf1,0x53,
 0xbf,0xa1,0x04,0x14,0x53,0x7f,0xab,0xb8,
 0xaa,0x27,0x90,0xb8,0xa7,0xb9,0x03,0xfa,
 0xe3,0x90,0x18,0x1a,0xe9,0x67,0xfa,0xe3,
 0xaa,0x04,0x76,0x53,0x3f,0xab,0x1c,0xb8,
 0xaa,0xfa,0x90,0x04,0x14,0x53,0x1f,0xab,
 0xba,0x00,0x04,0x77,0xfa,0xac,0x84,0x0a,
 0x00,0xb8,0x7f,0xb9,0xa0,0x81,0x53,0xd6,
 0x91,0x89,0x7c,0x99,0xe7,0x80,0x96,0xa3,
 0x89,0xbc,0x99,0xb7,0xb9,0xa0,0x81,0x43,
 0x28,0x91,0x83,0xaa,0xc8,0x80,0xa9,0xc8,
 0x80,0x91,0xc8,0x19,0xea,0xa8,0x04,0x95,
 0x15,0xc5,0x09,0xae,0x89,0xbc,0x99,0xbb,
 0xb8,0xf0,0xba,0x06,0xf8,0x3a,0x0a,0x92,
 0xda,0xba,0x30,0xa8,0x0a,0xd8,0x96,0xdf,
 0xea,0xc4,0xf8,0x47,0x77,0xd3,0x07,0x53,
 0x3f,0xa9,0xff,0x53,0x3f,0xd9,0x96,0xe5,
 0x04,0xdf,0x18,0xea,0xbc,0xbf,0xff,0xff,
 0x43,0xc0,0xaf,0x64,0x78,0x64,0x76,0x89,
 0xbc,0x99,0xb7,0x83,0x89,0xbc,0x99,0xaf,
 0x83,0x89,0xbc,0x99,0xb7,0x23,0x20,0xa8,
 0xa9,0x27,0xa0,0x18,0xe9,0xfa,0x14,0xec,
 0xb9,0xff,0x91,0xe9,0x02,0x91,0x14,0xe7,
 0x34,0x1c,0xb8,0xff,0xb9,0x80,0x27,0x90,
 0xc8,0xe9,0x0f,0x23,0xf8,0x90,0xe8,0x15,
 0x90,0x34,0x27,0x83,0x15,0xc5,0xb8,0xa0,
 0x80,0x53,0xd6,0x90,0xd5,0x05,0x83,0x15,
 0xc5,0xb8,0xa0,0x80,0x43,0x28,0x90,0xd5,
 0x05,0x83,0x27,0x90,0xb8,0x3f,0xf0,0x43,
 0x80,0xa0,0x14,0xe7,0x83,0x34,0x76,0x14,
 0xb0,0xf2,0x3d,0x23,0x56,0x34,0xa2,0xc5,
 0xff,0xd5,0x83,0xfe,0x53,0xfe,0xae,0xfd,
 0x97,0xf7,0xf7,0xf7,0xad,0x27,0xf7,0x4e,
 0xae,0xfc,0x67,0x37,0x17,0x6d,0xad,0xf6,
 0x6a,0xfe,0x37,0x53,0x01,0x2e,0x53,0xfe,
 0x4e,0xae,0x83,0x23,0xf8,0xb8,0x10,0xba,
 0x30,0x90,0x18,0xea,0x71,0x83,0xa5,0x05,
 0x76,0x75,0x24,0x78,0x14,0xec,0xaa,0xb8,
 0x7f,0x23,0x08,0x90,0xc8,0xf9,0x90,0xc8,
 0xfa,0x47,0x53,0x0f,0xad,0x34,0x97,0xfa,
 0x53,0x0f,0xad,0x34,0x97,0x24,0x32,0xfc,
 0x90,0xc8,0xfb,0x90,0xc8,0x03,0x08,0xab,
 0x44,0x2c,0x15,0xc5,0xac,0xbb,0x01,0xb8,
 0x3f,0xf0,0x43,0x40,0xa0,0xd5,0x05,0x83,
 0xb8,0x3e,0xf0,0xf2,0x3a,0x53,0x3f,0xd3,
 0x3b,0x96,0x3a,0x14,0xec,0xf0,0xb8,0x02,
 0xb9,0x01,0xd2,0xe2,0xba,0x99,0x80,0xc6,
 0xce,0x6a,0x57,0x90,0x24,0xfc,0x81,0xc6,
 0xda,0x23,0x59,0x90,0x81,0x6a,0x57,0x91,
 0x24,0xfc,0xb8,0x3e,0xf0,0x43,0x80,0xa0,
 0x24,0x3a,0xba,0x01,0x80,0xd3,0x59,0xc6,
 0xef,0x80,0x6a,0x57,0x90,0x24,0xfc,0x90,
 0x81,0xd3,0x59,0x96,0xf8,0x91,0x24,0xfc,
 0x81,0x6a,0x57,0x91,0xb8,0x7f,0x54,0x35,
 0x23,0x42,0x90,0xc8,0xb9,0x01,0x81,0x47,
 0x54,0x29,0x54,0x35,0x23,0x52,0x90,0xc8,
 0x81,0x54,0x29,0x54,0x35,0x23,0x56,0x90,
 0xc8,0x19,0x81,0x47,0x54,0x29,0x54,0x35,
 0x23,0x4a,0x90,0xc8,0x81,0x54,0x29,0x24,
 0x32,0x53,0x0f,0xad,0x34,0x4b,0xfd,0x90,
 0xc8,0xfe,0x90,0xc8,0x83,0x23,0x02,0x90,
 0xc8,0x83,0xb8,0x40,0xb9,0x50,0xfc,0x90,
 0x91,0x18,0x19,0xfb,0x90,0x03,0x08,0x91,
 0xbd,0x0a,0xb8,0x46,0x54,0x61,0xbd,0x0c,
 0xb8,0x4e,0x54,0x61,0xbd,0x0c,0xb8,0x5a,
 0x54,0x61,0xbd,0x0c,0xb8,0x5e,0x54,0x61,
 0x83,0x34,0x4b,0xfd,0x90,0x18,0xfe,0x90,
 0x18,0x83,0x07,0xba,0xf8,0x6a,0xc9,0xf6,
 0x6d,0x6a,0x53,0x07,0x17,0xaa,0x27,0x97,
 0xa7,0xf7,0xea,0x79,0xaa,0x81,0x5a,0x83,
 0x54,0x6a,0xc6,0x89,0x81,0xda,0x91,0x27,
 0x17,0x83,0x54,0x6a,0x96,0x92,0x81,0xda,
 0x91,0x27,0x83,0x42,0xac,0x53,0x0f,0xad,
 0x54,0xa4,0x2a,0xaf,0xfc,0x53,0xf0,0x47,
 0xad,0x54,0xa4,0x83,0xfd,0x03,0xb3,0xa3,
 0x96,0xac,0xad,0xcd,0x1d,0xab,0x6f,0xf6,
 0xa4,0xfb,0x83,0x0f,0x0e,0x0d,0x02,0x0a,
 0x05,0x0b,0x03,0x06,0x0c,0x09,0x01,0x04,
 0x08,0x07,0x00,0xbf,0xff,0xd5,0x14,0xf1,
 0xb9,0xf2,0xb8,0x10,0xba,0x0b,0xbb,0x28,
 0xbc,0x70,0xbe,0x04,0x34,0x1c,0xf9,0xa3,
 0xad,0x74,0xea,0x19,0x1e,0x1e,0xea,0xd6,
 0x34,0x27,0x23,0x4a,0x34,0xa2,0x34,0x3d,
 0xa9,0x34,0x1c,0x34,0x6b,0x34,0x27,0xf9,
 0x84,0x08,0x19,0x12,0x0e,0x12,0x23,0x14,
 0x0c,0x1c,0x20,0x26,0x12,0x20,0x20,0x20,
 0x00,0x0f,0xff,0xcf,0x03,0xf0,0x3f,0xcf,
 0x0f,0x0f,0x0f,0xcf,0x1c,0x71,0xc7,0xcf,
 0x33,0x33,0x33,0xcf,0x03,0xf0,0x3f,0xef,
 0x0f,0x0f,0x0f,0xef,0x1c,0x71,0xc7,0xef,
 0x33,0x33,0x33,0xef,0x55,0x55,0x55,0xef,
 0x90,0x10,0x94,0x00,0x21,0x00,0x56,0xdf,
 0x53,0xdd,0x50,0xda,0x4d,0xd8,0x4a,0xd5,
 0x47,0xd2,0x21,0x00,0x92,0x20,0x4a,0xcd,
 0x52,0xeb,0x4a,0xc9,0x52,0xe7,0x4a,0xc5,
 0x21,0x00,0x85,0x04,0x85,0x0c,0x85,0x14,
 0x85,0x1c,0x85,0x24,0x21,0x00,0x82,0x1c,
 0x21,0x00,0x8a,0x00,0x21,0x00,0x85,0x24,
 0x85,0x1c,0x85,0x14,0x85,0x0c,0x85,0x04,
 0x21,0x00,0x44,0xff,0x4d,0xdd,0x48,0xda,
 0x44,0xd5,0x42,0xd3,0x21,0x00,0xf9,0xaf,
 0xfe,0x39,0xff,0xd5,0x05,0x83,0x20,0x99,
 0xfe,0x84,0x08,0x99,0xfd,0x84,0x08,0x99,
 0xfc,0x84,0x08,0x89,0x03,0x84,0x08,0x09,
 0xa8,0x89,0xbc,0x99,0x9b,0xf9,0x43,0xf0,
 0x3a,0x85,0x27,0xaa,0xab,0x08,0xa9,0x12,
 0xa2,0xcb,0x32,0xa5,0x1a,0x52,0xa8,0x1b,
 0x72,0xab,0xca,0x92,0xae,0x95,0xf8,0x39,
 0x83,0xfb,0x53,0x0f,0xa8,0xfa,0x53,0x0f,
 0x47,0x48,0xa8,0xb9,0x07,0x23,0xc7,0x69,
 0xa3,0xd8,0xc6,0xc6,0xe9,0xbd,0x83,0x10,
 0x1f,0x0f,0xff,0xf0,0xf1,0x01,0x11,0xbc,
 0xff,0xfb,0x37,0x17,0xae,0xfa,0x6e,0x1c,
 0xf6,0xd6,0x6b,0xad,0x83,0xfa,0x96,0xe1,
 0x83,0xfb,0x96,0xe5,0x83,0x27,0x6b,0xea,
 0xe6,0x83,0xfc,0x90,0x18,0xf8,0xd2,0xf9,
 0xfb,0x90,0x03,0x08,0xab,0x18,0x54,0x61,
 0x83,0x53,0x0f,0x07,0xc6,0xf0,0x64,0xf5
};

static char bios[MAXC], scshot[MAXC];


static long filesize(FILE *stream);
static void load_bios(const char *biosname);
static void load_cart(char *file);
int parse_option(char *attr, char *val);
void read_default_config(void);


int o2main(int argc, char *argv[]){
	int i;
	static char file[MAXC], attr[MAXC], val[MAXC], *p, *binver;

	#if defined(ALLEGRO_WINDOWS)
	binver = "Windows binary";
	#elif defined(ALLEGRO_DOS)
	binver = "DOS binary";
	#elif defined(ALLEGRO_LINUX)
	binver = "Linux binary";
	#elif defined(ALLEGRO_BEOS)
	binver = "BEOS binary";
	#elif defined(ALLEGRO_QNX)
	binver = "QNX binary";
	#elif defined(ALLEGRO_UNIX)
	binver = "UNIX binary";
	#elif defined(ALLEGRO_MPW)
	binver = "MacOS binary";
	#else
	binver = "Unknown binary";
	#endif

	printf("%s %s\n","\nO2EM v" O2EM_VERSION " " RELEASE_DATE "  - ", binver);
	printf("Freeware Odyssey2 / Videopac+ Emulator - http://o2em.sourceforge.net\n");
	printf("Developed by Andre de la Rocha\n");
	printf("Created by Daniel Boris (c)1996/1998\n");
	printf("\n");
	
	if (argc < 2) {
		printf("Use: o2em <file> [options]\n");
		printf("<file> = file to load with extension\n");
		#ifndef ALLEGRO_DOS
		printf("-wsize=n         Window size (1-4)\n");
		printf("-fullscreen      Full screen mode\n");
		#endif
		printf("-scanlines       Enable scanlines\n");
		printf("-nosound         Turn off sound emulation\n");
		printf("-novoice         Turn off voice emulation\n");
		printf("-svolume=n       Set sound volume (0-100)\n");
		printf("-vvolume=n       Set voice volume (0-100)\n");
		printf("-filter          Enable low-pass audio filter\n");
		printf("-debug	         Start the emulator in debug mode\n");
		printf("-speed=n         Relative speed (100 = original)\n");
		printf("-nolimit         Turn off speed limiter\n");
		printf("-bios=file       Set the O2 bios file name/dir (default=o2rom.bin)\n");
		printf("-scshot=file     Set the screenshot file name/template\n");
		printf("-videopac        Use videopac ROM\n");
		printf("-euro            Use European timing / 50Hz mode\n");
		printf("-exrom           Use special 3K program/1K data ROM mode\n");
		printf("-s<n>=mode/keys  Define stick n mode/keys (n=1-2)\n");
		printf("\nPress Enter...");
		return 0 ;
	}

	app_data.debug = 0;
	app_data.stick[0] = app_data.stick[1] = 1;
	set_defjoykeys(0,0);
	set_defjoykeys(1,1);
	app_data.bank = 0;
    app_data.limit = 1;
    app_data.sound_en = 1;
	app_data.speed = 100;
	app_data.wsize = 2;
	#ifdef ALLEGRO_DOS
	app_data.fullscreen = 1;
	#else
	app_data.fullscreen = 0;
	#endif
	app_data.scanlines = 0;
	app_data.voice = 1;
	app_data.window_title = "O2EM v" O2EM_VERSION;
	app_data.svolume = 100;
	app_data.vvolume = 100;
	app_data.filter = 0;
	app_data.exrom = 0;
	app_data.crc = 0;
	app_data.scshot = scshot;
	app_data.euro = 0;
	app_data.openb = 0;
	app_data.videopac = 0 ;

	strcpy(file,"");
	strcpy(bios,"");
	strcpy(scshot,"");

	read_default_config();

	for(i=1; i<argc; i++) {
		if (argv[i][0] != '-') 	{
			strncpy(file,argv[i],MAXC-1);
			file[MAXC-1]=0;
		} else {
			p=strtok(argv[i],"=");
			if (p){
				strncpy(attr,p+1,MAXC-1);
				attr[MAXC-1]=0;
			} else
				strcpy(attr,"");
			p=strtok(NULL,"=");
			if (p){
				strncpy(val,p,MAXC-1);
				val[MAXC-1]=0;
			} else
				strcpy(val,"");
			strlwr(attr);

			if (!parse_option(attr, val)) exit(EXIT_FAILURE);

		}
	}

	if (strlen(file)==0) {
		fprintf(stderr,"Error: file name missing\n");
		exit(EXIT_FAILURE);
	}

	printf("Starting emulation ...\n");

	//allegro_init();
	//install_timer();
    init_audio();
//	printf("Using Allegro %s\n",allegro_id);
	load_bios(bios);
	load_cart(file);
	
	if (app_data.voice) load_voice_samples();

	init_display();
	init_cpu();
	init_system();
	voice_count = 0 ;
	if (app_data.debug) key_debug=1;
	#ifndef _DEBUG
	#ifdef ALLEGRO_WINDOWS
	FreeConsole();
	#endif
	#endif
	run();
	//exit(EXIT_SUCCESS);
}

END_OF_MAIN();


int parse_option(char *attr, char *val){
	int t;

	if (!strcmp(attr,"nolimit")) {
		app_data.limit = !(val[0]!='0');
	} else if (!strcmp(attr,"nosound")) {
		app_data.sound_en = !(val[0]!='0');
	} else if (!strcmp(attr,"novoice")) {
		app_data.voice = !(val[0]!='0');
	} else if (!strcmp(attr,"filter")) {
		app_data.filter = (val[0]!='0');
	} else if (!strcmp(attr,"debug")) {
		app_data.debug = (val[0]!='0');
	} else if ((!strcmp(attr,"s1")) || (!strcmp(attr,"s2"))) {
		int sn;
		sn = (!strcmp(attr,"s1"))? 0 : 1;
		if (strlen(val)<2){
			t = -1;
			sscanf(val,"%d",&t);
			if ((t>=0) && (t<=3)) {
				if ((t==1)||(t==2)){
					app_data.stick[sn] = 1;
					set_defjoykeys(sn,t-1);
				} else {
					app_data.stick[sn] = (t==0) ? 0 : 2;
					set_joykeys(sn,0,0,0,0,0);
				}
			} else {
				fprintf(stderr,"Invalid value for option %s\n",attr);
				return 0;
			}
		} else {
			char *p,*s;
			int i,k,code,nk,codes[5];
			strupr(val);
			nk = 0;
			p = strtok(val,",");
			while (p) {
				i = code = 0;
				k = keybtab[i].keybcode;
				s = keybtab[i].keybname;
				while (k && (code==0)) {
					if (strcmp(s,p)==0) code = k;
					i++;
					k = keybtab[i].keybcode;
					s = keybtab[i].keybname;
				}
				if (!code) {
					fprintf(stderr,"Invalid value for option %s : key %s unknown\n",attr,p);
					return 0;
				}					
				codes[nk] = code;
				p = strtok(NULL,",");
				nk++;
				if (nk>5) {
					fprintf(stderr,"Invalid value for option %s : invalid number of keys\n",attr);
					return 0;
				}
			}
			if (nk != 5) {
				fprintf(stderr,"Invalid value for option %s : invalid number of keys\n",attr);
				return 0;
			}
			app_data.stick[sn] = 1;
			set_joykeys(sn,codes[0],codes[1],codes[2],codes[3],codes[4]);
		}
	} else if (!strcmp(attr,"speed")) {
		t = -1;
		sscanf(val,"%d",&t);
		if ((t>0) && (t<=10000))
			app_data.speed = t;
		else {
			fprintf(stderr,"Invalid value for option %s\n",attr);
			return 0;
		}
	} else if (!strcmp(attr,"svolume")) {
		t = -1;
		sscanf(val,"%d",&t);
		if ((t>=0) && (t<=100))
			app_data.svolume = t;
		else {
			fprintf(stderr,"Invalid value for option %s\n",attr);
			return 0;
		}
		if (t==0) app_data.sound_en=0;
	} else if (!strcmp(attr,"vvolume")) {
		t = -1;
		sscanf(val,"%d",&t);
		if ((t>=0) && (t<=100))
			app_data.vvolume = t;
		else {
			fprintf(stderr,"Invalid value for option %s\n",attr);
			return 0;
		}
		if (t==0) app_data.voice=0;
	} else if (!strcmp(attr,"wsize")) {
		t = -1;
		sscanf(val,"%d",&t);
		if ((t>0) && (t<5)) {
			app_data.wsize = t;
			app_data.fullscreen = 0;
		} else {
			fprintf(stderr,"Invalid value for option %s\n",attr);
			return 0;
		}
	} else if (!strcmp(attr,"fullscreen")) {
		app_data.fullscreen = (val[0]!='0');
	} else if (!strcmp(attr,"scanlines")) {
		app_data.scanlines = (val[0]!='0');
	} else if (!strcmp(attr,"scshot")) {
		strcpy(scshot,val);
	} else if (!strcmp(attr,"euro")) {
		app_data.euro = (val[0]!='0');
	} else if (!strcmp(attr,"exrom")) {
		app_data.exrom = (val[0]!='0');
	} else if (!strcmp(attr,"videopac")) {
		app_data.videopac = 1 ;
	} else if (!strcmp(attr,"bios")) {
		strcpy(bios,val);
	} else {
		fprintf(stderr,"Invalid option : %s\n",attr);
		return 0;
	}
	return 1;
}


void read_default_config(void){
	FILE *f;
	static char attr[MAXC], val[MAXC], s[MAXC];
	char *p, *fn;
	int i,l;
	
	fn = "o2em_def.cfg";
	f = fopen(fn,"r");
	if (!f) {
		fn = "O2EM_DEF.CFG";
		f = fopen(fn,"r");
	}
	if (!f) return;

	l=0;
	while (fgets(s,MAXC-1,f)){
		l++;
		p=s;
		while (*p && (isspace(*p))) p++;
		if (*p && (*p != '#')) {
			i=0;
			while (*p && (!isspace(*p)) && (*p != '=')) attr[i++] = *p++;
			attr[i]=0;
			while (*p && (isspace(*p))) p++;
			i=0;
			if (*p == '='){
				p++;
				while (*p && (isspace(*p))) p++;
				if (*p == '"'){
					p++;
					while (*p && (*p != '"') && (*p != '\n') && (*p != '\r')) val[i++] = *p++;
				} else {
					while (*p && (!isspace(*p))) val[i++] = *p++;
				}
			}
			val[i]=0;
			if (strlen(attr)>0) {
				strlwr(attr);
				if (!parse_option(attr,val)) {
					printf("Error in the %s file at line number %d !\n\n",fn,l);
				}
			}
		}
	}
	fclose(f);
}


static long filesize(FILE *stream){
   long curpos, length;

   curpos = ftell(stream);
   fseek(stream, 0L, SEEK_END);
   length = ftell(stream);
   fseek(stream, curpos, SEEK_SET);
   return length;
}


static void load_bios(const char *biosname){
	FILE *fn;
	static char s[MAXC+10];
	unsigned long crc;
#if 0
	if ((!biosname) || (strlen(biosname)==0)) {
		strcpy(s,"o2rom.bin");
		fn = fopen("o2rom.bin","rb");
		if (!fn) fn = fopen("O2ROM.BIN","rb");
	} else if ((biosname[strlen(biosname)-1]=='/') || (biosname[strlen(biosname)-1]=='\\') || (biosname[strlen(biosname)-1]==':')) {
		strcpy(s,biosname);
		strcat(s,"O2ROM.BIN");
		fn = fopen(s,"rb");
		if (!fn) {
			strcpy(s,biosname);
			strcat(s,"o2rom.bin");
			fn = fopen(s,"rb");
		}
	} else {
		strcpy(s,biosname);
		fn = fopen(biosname,"rb");
	}

	if (!fn) {
		fprintf(stderr,"Error loading bios ROM (%s)\n",s);
		exit(EXIT_FAILURE);
	}
	
 	if (fread(&rom1[0],1024,1,fn) != 1) {
 		fprintf(stderr,"Error loading bios ROM o2rom.bin\n");
 		exit(EXIT_FAILURE);
 	}
	fclose(fn);
#endif

	if ( app_data.videopac )
		memcpy( &rom1[0], VIDEOPAC_ROM, 1024 ) ;
	else
		memcpy( &rom1[0], O2ROM, 1024 ) ;

	memcpy(&rom2[0],&rom1[0],1024);
	memcpy(&rom3[0],&rom1[0],1024);
	memcpy(&rom4[0],&rom1[0],1024);

	crc = crc32_buf(rom1,1024);

	if (crc==0x8016A315)
		printf("Odyssey2 bios ROM loaded\n");
	else if (crc==0xE20A9F41)
		printf("Videopac+ G7400 bios ROM loaded\n");
	else
		printf("Bios ROM loaded (unknown version)\n");
}


static void load_cart(char *file){
	FILE *fn;
	long l;

	app_data.crc = crc32_file(file);

	if (app_data.crc == 0xAFB23F89) app_data.exrom = 1;  /* Musician */
	if (app_data.crc == 0x3BFEF56B) app_data.exrom = 1;  /* Four in 1 Row! */

	if (((app_data.crc == 0x975AB8DA) || (app_data.crc == 0xE246A812)) && (!app_data.debug)) {
		fprintf(stderr,"Error: file %s is an incomplete ROM dump\n",file);
		exit(EXIT_FAILURE);
	}
	
	fn=fopen(file,"rb");
	if (!fn) {
		fprintf(stderr,"Error loading %s\n",file);
		exit(EXIT_FAILURE);
	}
	printf("Loading: \"%s\"  Size: ",file);
	l = filesize(fn);
	if (l < 2049) {
		app_data.bank=1;
		if (fread(&rom1[1024],l,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		memcpy(&rom1[3072],&rom1[2048],1024); /* simulate missing A10 */
		rom=rom1;
		printf("2K");
	} else if (l == 3072) {
		app_data.bank=1;
		if (fread(&rom1[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		rom=rom1;
		printf("3K");
	} else if (l == 4096) {
		if (app_data.exrom) {
			app_data.bank=1;
			if (fread(&extROM[0], 1024,1,fn) != 1) {
				fprintf(stderr,"Error loading %s\n",file);
				exit(EXIT_FAILURE);
			}
			if (fread(&rom1[1024],3072,1,fn) != 1) {
				fprintf(stderr,"Error loading %s\n",file);
				exit(EXIT_FAILURE);
			}
			rom=rom1;
			printf("3K EXROM");
		} else {
			app_data.bank=2;
			if (fread(&rom1[1024],2048,1,fn) != 1) {
				fprintf(stderr,"Error loading %s\n",file);
				exit(EXIT_FAILURE);
			}
			if (fread(&rom2[1024],2048,1,fn) != 1) {
				fprintf(stderr,"Error loading %s\n",file);
				exit(EXIT_FAILURE);
			}
			memcpy(&rom1[3072],&rom1[2048],1024); /* simulate missing A10 */
			memcpy(&rom2[3072],&rom2[2048],1024); /* simulate missing A10 */
			rom=rom2;
			printf("4K");
		}
	} else if (l == 6144) {
		app_data.bank=2;
		if (fread(&rom1[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom2[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		rom=rom2;
		printf("6K");
	} else if (l == 8192) {
		app_data.bank=3;
		if (fread(&rom1[1024],2048,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom2[1024],2048,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom3[1024],2048,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom4[1024],2048,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		memcpy(&rom1[3072],&rom1[2048],1024); /* simulate missing A10 */
		memcpy(&rom2[3072],&rom2[2048],1024); /* simulate missing A10 */
		memcpy(&rom3[3072],&rom3[2048],1024); /* simulate missing A10 */
		memcpy(&rom4[3072],&rom4[2048],1024); /* simulate missing A10 */
		rom=rom4;
		printf("8K");
	} else if (l == 12288) {
		app_data.bank=3;
		if (fread(&rom1[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom2[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom3[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		if (fread(&rom4[1024],3072,1,fn) != 1) {
			fprintf(stderr,"Error loading %s\n",file);
			exit(EXIT_FAILURE);
		}
		rom=rom4;
		printf("12K");
	} else {
		fclose(fn);
		fprintf(stderr,"Invalid ROM size\n");
		exit(EXIT_FAILURE);
	}
	fclose(fn);
	if ((rom1[1024+12]=='O') && (rom1[1024+13]=='P') && (rom1[1024+14]=='N') && (rom1[1024+15]=='B')) app_data.openb=1;
	orom=rom;
	printf("  CRC: %08lX\n",app_data.crc);
}


