/* Mednafen - Multi-system Emulator
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "video-common.h"
#include "../ConvertUTF.h"
#include "font-data.h"

static uint8 FontDataCache5x7[65536][7];
static uint8 FontDataCache9x18[65536][18 * 2];

static uint32 FontDataHave5x7[65536 / 32];
static uint32 FontDataHave9x18[65536 / 32];

#ifdef WANT_INTERNAL_CJK
static uint8 FontDataCache18x18[65536][18 * 3];
static uint32 FontDataHave18x18[65536 / 32];
#endif

void MDFN_InitFontData(void)
{
 unsigned int x;
 unsigned int inx;

 memset(FontDataHave5x7, 0, sizeof(FontDataHave5x7));
 memset(FontDataHave9x18, 0, sizeof(FontDataHave9x18));

 for(inx=x=0;x<65536;x++)
 {
  if(inx < (FontData5x7_Size / sizeof(font5x7)) && FontData5x7[inx].glyph_num == x)
  {
   memcpy(FontDataCache5x7[x], FontData5x7[inx].data, 7);
   FontDataHave5x7[x >> 5] |= 1 << (x & 0x1F);
   inx++;
  }
  else
   memset(FontDataCache5x7[x], 0, 7);
 }

 for(inx=x=0;x<65536;x++)
 {
  if(inx < (FontData9x18_Size / sizeof(font9x18)) && FontData9x18[inx].glyph_num == x)
  {
   memcpy(FontDataCache9x18[x], FontData9x18[inx].data, 18 * 2);
   FontDataHave9x18[x >> 5] |= 1 << (x & 0x1F);
   inx++;
  }
  else
   memset(FontDataCache9x18[x], 0, 18 * 2);
 }

 #ifdef WANT_INTERNAL_CJK
 memset(FontDataHave18x18, 0, sizeof(FontDataHave18x18));
 for(inx=x=0;x<65536;x++)
 {
  if(inx < (FontData18x18_Size / sizeof(font18x18)) && FontData18x18[inx].glyph_num == x)
  {
   memcpy(FontDataCache18x18[x], FontData18x18[inx].data, 18 * 3);
   FontDataHave18x18[x >> 5] |= 1 << (x & 0x1F);
   inx++;
  }
  else
   memset(FontDataCache18x18[x], 0, 18 * 3);
 }
 #endif
}

static long min(long a1, long a2)
{
 if(a1 < a2)
  return(a1);
 return(a2);
}

uint32 GetTextPixLength(uint8 *msg)
{
 size_t msg_utf8_len = strlen((char*)msg);

 UTF32 utf32_buf[msg_utf8_len + 1];
 UTF32 *tstart = utf32_buf;
 uint32 slen;
 uint32 pixwidth;
 uint32 x;
 uint8 glyph_width[msg_utf8_len + 1];
 uint8 glyph_ov_width[msg_utf8_len + 1];

 ConvertUTF8toUTF32((const UTF8**)&msg, &msg[strlen((char *)msg)], &tstart, &tstart[msg_utf8_len + 1], lenientConversion);

 slen = (tstart - utf32_buf);

 pixwidth = 0;

 for(x=0;x<slen;x++)
 {
  uint32 thisglyph = utf32_buf[x];
  if(thisglyph > 0xFFFF) thisglyph = ' ';

  if(FontDataHave9x18[thisglyph >> 5] & (1 << (thisglyph & 0x1F)))
  {
   glyph_width[x] = 9;
   pixwidth += 9;
  }
  #ifdef WANT_INTERNAL_CJK
  else if(FontDataHave18x18[thisglyph >> 5] & (1 << (thisglyph & 0x1F)))
  {
   glyph_width[x] = 18;
   pixwidth += 18;
  }
  #endif
  else
  {
   glyph_width[x] = 9;
   pixwidth += 9;
  }
  if(thisglyph >= 0x0300 && thisglyph <= 0x036F)
  {
   if(x != (slen-1))
   {
    pixwidth -= min(glyph_width[x], glyph_width[x + 1]);
   }
   glyph_ov_width[x] = 0;
  }
  else
   glyph_ov_width[x] = glyph_width[x];
 }
 return(pixwidth);
}

static void DrawTextSub(uint8 *msg, uint8 **glyph_ptrs, uint8 *glyph_width, uint8 *glyph_ov_width, uint32 width, uint32 &pixwidth, uint32 &slen, bool want5x7 = 0)
{
 size_t msg_utf8_len = strlen((char*)msg);

 UTF32 utf32_buf[msg_utf8_len+1];
 UTF32 *tstart = utf32_buf;
 unsigned int x;

 ConvertUTF8toUTF32((const UTF8**)&msg, &msg[strlen((char *)msg)], &tstart, &tstart[msg_utf8_len+1], lenientConversion);

 slen = (tstart - utf32_buf);

 pixwidth = 0;

 for(x=0;x<slen;x++)
 {
  uint32 thisglyph = utf32_buf[x];

  if(want5x7)
  {
   if(FontDataHave5x7[thisglyph >> 5] & (1 << (thisglyph & 0x1F)))
   {
    glyph_ptrs[x] = FontDataCache5x7[thisglyph];
    glyph_width[x] = 5;
    pixwidth += 5;
   }
   else
   {
    glyph_ptrs[x] = FontDataCache5x7[(unsigned char)'?'];
    glyph_width[x] = 5;
    pixwidth += 5;
   }
  }
  else if(FontDataHave9x18[thisglyph >> 5] & (1 << (thisglyph & 0x1F)))
  {
   glyph_ptrs[x] = FontDataCache9x18[thisglyph];
   glyph_width[x] = 9;
   pixwidth += 9;
  }
  #ifdef WANT_INTERNAL_CJK
  else if(FontDataHave18x18[thisglyph >> 5] & (1 << (thisglyph & 0x1F)))
  {
   glyph_ptrs[x] = FontDataCache18x18[thisglyph];
   glyph_width[x] = 18;
   pixwidth += 18;
  }
  #endif
  else
  {
   glyph_ptrs[x] = FontDataCache9x18[(unsigned char)'?'];
   glyph_width[x] = 9;
   pixwidth += 9;
  }

  if(thisglyph >= 0x0300 && thisglyph <= 0x036F)
  {
   if(x != (slen-1))
   {
    pixwidth -= min(glyph_width[x], glyph_width[x + 1]);
   }
   glyph_ov_width[x] = 0;
  }
  else
   glyph_ov_width[x] = glyph_width[x];
  if(pixwidth > width) // Oopsies, it's too big for the screen!  Just truncate it for now.
  {
   slen = x;
   pixwidth -= glyph_width[x];
   break;
  }
 }

}

uint32 DrawTextTrans(uint8 *dest, int pitch, uint32 width, uint8 *msg, uint32 fgcolor, int centered, bool want5x7)
{
 size_t msg_utf8_len = strlen((char*)msg);

 uint32 slen;
 uint32 pixwidth;
 uint32 x;
 uint8 *glyph_ptrs[msg_utf8_len+1];
 uint8 glyph_width[msg_utf8_len+1];
 uint8 glyph_ov_width[msg_utf8_len+1];

 pitch /= sizeof(uint8);

 DrawTextSub(msg, glyph_ptrs, glyph_width, glyph_ov_width, width, pixwidth, slen, want5x7);

 if(centered)
 {
  int32 poot = width - pixwidth;

  dest += poot / 2;
 }

 if(want5x7)
  for(x=0;x<slen;x++)
  {
   int gx, gy;
   uint8 *src_glyph = glyph_ptrs[x];

   for(gy=0;gy<7;gy++)
   {
    for(gx=0;gx<glyph_width[x];gx++)
    {
     if((src_glyph[gy] << gx) & 0x80) dest[gy * pitch + gx] = fgcolor;
    }
   }
   dest += glyph_ov_width[x];
  }
 else
  for(x=0;x<slen;x++)
  {
   int gx, gy;
   uint8 *src_glyph = glyph_ptrs[x];

   if(glyph_width[x] > 16)
   {
    for(gy=0;gy<18;gy++)
    {
     for(gx=0;gx<glyph_width[x];gx++)
     {
       if((src_glyph[gy * 3 + (gx >> 3)] << (gx & 0x7)) & 0x80) dest[gy * pitch + gx] = fgcolor;
     }
    }
   }
   else
   {
    for(gy=0;gy<18;gy++)
    {
     for(gx=0;gx<glyph_width[x];gx++)
     {
       if((src_glyph[gy * 2 + (gx >> 3)] << (gx & 0x7)) & 0x80) dest[gy * pitch + gx] = fgcolor;
     }
    }
   }
   dest += glyph_ov_width[x];
  }

 return(pixwidth);
}

uint32 DrawTextTrans(uint32 *dest, int pitch, uint32 width, uint8 *msg, uint32 fgcolor, int centered, bool want5x7)
{
 size_t msg_utf8_len = strlen((char*)msg);
 uint32 slen;
 uint32 pixwidth;
 uint32 x;
 uint8 *glyph_ptrs[msg_utf8_len+1];
 uint8 glyph_width[msg_utf8_len+1];
 uint8 glyph_ov_width[msg_utf8_len+1];

 pitch /= sizeof(uint32);

 DrawTextSub(msg, glyph_ptrs, glyph_width, glyph_ov_width, width, pixwidth, slen, want5x7);

 if(centered)
 {
  int32 poot = width - pixwidth;

  dest += poot / 2;
 }

 if(want5x7)
  for(x=0;x<slen;x++)
  {
   int gx, gy;
   uint8 *src_glyph = glyph_ptrs[x];

   for(gy=0;gy<7;gy++)
   {
    for(gx=0;gx<glyph_width[x];gx++)
    {
     if((src_glyph[gy] << gx) & 0x80) dest[gy * pitch + gx] = fgcolor;
    }
   }
   dest += glyph_ov_width[x];
  }
 else
  for(x=0;x<slen;x++)
  {
   int gx, gy;
   uint8 *src_glyph = glyph_ptrs[x];

   if(glyph_width[x] > 16)
   {
    for(gy=0;gy<18;gy++)
    {
     for(gx=0;gx<glyph_width[x];gx++)
     {
       if((src_glyph[gy * 3 + (gx >> 3)] << (gx & 0x7)) & 0x80) dest[gy * pitch + gx] = fgcolor;
     }
    }
   }
   else
   {
    for(gy=0;gy<18;gy++)
    {
     for(gx=0;gx<glyph_width[x];gx++)
     {
       if((src_glyph[gy * 2 + (gx >> 3)] << (gx & 0x7)) & 0x80) dest[gy * pitch + gx] = fgcolor;
     }
    }
   }
   dest += glyph_ov_width[x];
  }
 return(pixwidth);
}

uint32 DrawTextTransShadow(uint32 *dest, int pitch, uint32 width, uint8 *textmsg, uint32 fgcolor, uint32 shadcolor, int centered, bool want5x7)
{
 DrawTextTrans(dest + 1 + (pitch >> 2), pitch, width, textmsg, shadcolor, centered, want5x7);
 return(DrawTextTrans(dest, pitch, width, textmsg, fgcolor, centered, want5x7));
}
